import { Pool, PoolClient } from "pg";
import type { ProductionPlan, ProductionPlanLine, MergePointProgress } from "@shared/schema";
import { findFormatkaInWarehouse } from "../warehouse/formatka-search";
import { createReservation } from "./buffer-reservations";

type PoolOrClient = Pool | PoolClient;

interface InsertProductionPlan {
  planNumber?: string;
  name: string;
  description?: string | null;
  plannedStartDate?: Date | null;
  plannedEndDate?: Date | null;
  status?: string;
  priority?: string;
  notes?: string | null;
  metadata?: any;
  createdBy?: number | null;
  autoAssignRoutings?: boolean;
}

interface InsertProductionPlanLine {
  planId: number;
  productId: number;
  quantity: number;
  sourceType?: string | null;
  sourceId?: number | null;
  sourceReference?: string | null;
  productionOrderId?: number | null;
  routingId?: number | null;
  routingOverride?: any;
  bomId?: number | null;
  plannedStartDate?: Date | null;
  plannedEndDate?: Date | null;
  status?: string;
  sequence?: number | null;
  notes?: string | null;
  metadata?: any;
}

function mapRowToPlan(row: any): ProductionPlan {
  return {
    id: row.id,
    planNumber: row.plan_number,
    shortName: row.short_name,
    name: row.name,
    description: row.description,
    plannedStartDate: row.planned_start_date,
    plannedEndDate: row.planned_end_date,
    actualStartDate: row.actual_start_date,
    actualEndDate: row.actual_end_date,
    status: row.status,
    priority: row.priority,
    notes: row.notes,
    metadata: row.metadata,
    createdBy: row.created_by,
    approvedBy: row.approved_by,
    approvedAt: row.approved_at,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
    nameTemplateId: row.name_template_id,
    nameSequence: row.name_sequence,
    nameGeneratedAt: row.name_generated_at,
    nameGenerationMeta: row.name_generation_meta,
    zlpLocked: row.zlp_locked ?? false,
    zlpLockedBy: row.zlp_locked_by ?? null,
    zlpLockedAt: row.zlp_locked_at ?? null,
    autoAssignRoutings: row.auto_assign_routings ?? false,
  };
}

function mapRowToPlanLine(row: any): ProductionPlanLine & {
  reservedQuantity?: number;
  warehouseTotalQty?: number | null;
  warehouseReservedQty?: number | null;
  packedProductId?: number | null;
  packedProductSku?: string | null;
} {
  return {
    id: row.id,
    planId: row.plan_id,
    productId: row.product_id,
    quantity: row.quantity,
    reservedQuantity: row.reserved_quantity ?? 0, // Per-line reservation
    colorCode: row.color_code ?? null,
    sourceType: row.source_type,
    sourceId: row.source_id,
    sourceReference: row.source_reference,
    productionOrderId: row.production_order_id,
    routingId: row.routing_id,
    routingOverride: row.routing_override,
    locationId: row.location_id ?? null,
    bomId: row.bom_id,
    plannedStartDate: row.planned_start_date,
    plannedEndDate: row.planned_end_date,
    status: row.status,
    sequence: row.sequence,
    notes: row.notes,
    metadata: row.metadata,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
    // Warehouse data from LEFT JOIN
    warehouseTotalQty: row.warehouse_total_qty ?? null,
    warehouseReservedQty: row.warehouse_reserved_qty ?? null,
    packedProductId: row.packed_product_id ?? null,
    packedProductSku: row.packed_product_sku ?? null,
  };
}

interface PlanFilters {
  status?: string;
  priority?: string;
  createdBy?: number;
  startDate?: Date;
  endDate?: Date;
  search?: string;
  limit?: number;
  offset?: number;
}

export interface PlanWithStats extends ProductionPlan {
  productsCount: number;
  ordersCount: number;
  packedCount: number;
  totalQuantity: number;
  reservedFormatki: number;
  totalFormatki: number;
}

export async function getPlans(pool: Pool, filters?: PlanFilters): Promise<PlanWithStats[]> {
  // Build WHERE clause for filters
  let whereClause = `WHERE 1=1`;
  const params: any[] = [];
  let paramIndex = 1;

  if (filters?.status) {
    whereClause += ` AND pp.status = $${paramIndex++}`;
    params.push(filters.status);
  }

  if (filters?.priority) {
    whereClause += ` AND pp.priority = $${paramIndex++}`;
    params.push(filters.priority);
  }

  if (filters?.createdBy) {
    whereClause += ` AND pp.created_by = $${paramIndex++}`;
    params.push(filters.createdBy);
  }

  if (filters?.startDate) {
    whereClause += ` AND pp.planned_start_date >= $${paramIndex++}`;
    params.push(filters.startDate);
  }

  if (filters?.endDate) {
    whereClause += ` AND pp.planned_end_date <= $${paramIndex++}`;
    params.push(filters.endDate);
  }

  if (filters?.search) {
    whereClause += ` AND (pp.plan_number ILIKE $${paramIndex} OR pp.name ILIKE $${paramIndex})`;
    params.push(`%${filters.search}%`);
    paramIndex++;
  }

  // Complex query with aggregations
  const query = `
    SELECT 
      pp.*,
      COALESCE(stats.products_count, 0) as products_count,
      COALESCE(stats.orders_count, 0) as orders_count,
      COALESCE(stats.total_quantity, 0) as total_quantity,
      COALESCE(packed_stats.packed_count, 0) as packed_count,
      COALESCE(formatki_stats.reserved_formatki, 0) as reserved_formatki,
      COALESCE(formatki_stats.total_formatki, 0) as total_formatki
    FROM production.production_plans pp
    LEFT JOIN LATERAL (
      SELECT 
        COUNT(DISTINCT ppl.id) as products_count,
        COUNT(DISTINCT ppl.source_reference) FILTER (WHERE ppl.source_reference IS NOT NULL) as orders_count,
        SUM(ppl.quantity) as total_quantity
      FROM production.production_plan_lines ppl
      WHERE ppl.plan_id = pp.id AND ppl.deleted_at IS NULL
    ) stats ON true
    LEFT JOIN LATERAL (
      SELECT COUNT(*)::int as packed_count
      FROM production.production_plan_lines ppl2
      WHERE ppl2.plan_id = pp.id 
        AND ppl2.deleted_at IS NULL
        AND ppl2.reserved_quantity > 0
    ) packed_stats ON true
    LEFT JOIN LATERAL (
      SELECT 
        COALESCE((SELECT COUNT(*)::int FROM production.production_buffer_reservations pbr 
                  WHERE pbr.zlp_id = pp.id AND pbr.status = 'ACTIVE'), 0) as reserved_formatki,
        COALESCE((SELECT SUM(sub.cnt)::int FROM (
          SELECT (SELECT COUNT(*) FROM bom.product_components pc 
                  INNER JOIN bom.product_boms pb ON pc.product_bom_id = pb.id 
                  WHERE pb.product_id = ppl3.product_id 
                    AND pc.calculated_length IS NOT NULL 
                    AND pc.calculated_width IS NOT NULL) as cnt
          FROM production.production_plan_lines ppl3 
          WHERE ppl3.plan_id = pp.id AND ppl3.deleted_at IS NULL
        ) sub), 0) as total_formatki
    ) formatki_stats ON true
    ${whereClause}
    ORDER BY pp.created_at DESC
  `;

  if (filters?.limit) {
    params.push(filters.limit);
  }
  if (filters?.offset) {
    params.push(filters.offset);
  }

  let finalQuery = query;
  if (filters?.limit) {
    finalQuery += ` LIMIT $${paramIndex++}`;
  }
  if (filters?.offset) {
    finalQuery += ` OFFSET $${paramIndex++}`;
  }

  const result = await pool.query(finalQuery, params);
  return result.rows.map(row => ({
    ...mapRowToPlan(row),
    productsCount: parseInt(row.products_count) || 0,
    ordersCount: parseInt(row.orders_count) || 0,
    packedCount: parseInt(row.packed_count) || 0,
    totalQuantity: parseInt(row.total_quantity) || 0,
    reservedFormatki: parseInt(row.reserved_formatki) || 0,
    totalFormatki: parseInt(row.total_formatki) || 0,
  }));
}

export async function getPlanById(pool: Pool, id: number): Promise<ProductionPlan | null> {
  const result = await pool.query(
    'SELECT * FROM production.production_plans WHERE id = $1',
    [id]
  );
  
  if (result.rows.length === 0) {
    return null;
  }
  
  return mapRowToPlan(result.rows[0]);
}

export function generatePlanNumberFromId(id: number): string {
  return `PLAN-${String(id).padStart(4, '0')}`;
}

export async function createPlan(pool: Pool, data: InsertProductionPlan): Promise<ProductionPlan> {
  // First insert with temporary plan_number, then update with ID-based number
  const result = await pool.query(
    `INSERT INTO production.production_plans 
      (plan_number, name, description, planned_start_date, planned_end_date, 
       status, priority, notes, metadata, created_by, auto_assign_routings, created_at, updated_at)
     VALUES ('TEMP', $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
     RETURNING *`,
    [
      data.name,
      data.description,
      data.plannedStartDate,
      data.plannedEndDate,
      data.status || 'draft',
      data.priority || 'normal',
      data.notes,
      data.metadata ? JSON.stringify(data.metadata) : null,
      data.createdBy,
      data.autoAssignRoutings ?? false,
    ]
  );

  const createdPlan = result.rows[0];
  
  // Generate plan_number based on ID (PLAN-0022 format)
  const planNumber = `PLAN-${String(createdPlan.id).padStart(4, '0')}`;
  
  // Generate short_name in format PLAN-21 (just the ID number, no padding)
  const shortName = `PLAN-${createdPlan.id}`;
  
  // Update the plan with plan_number, short_name and append it to the name
  const updatedName = `${createdPlan.name} | ${shortName}`;
  await pool.query(
    `UPDATE production.production_plans SET plan_number = $1, short_name = $2, name = $3 WHERE id = $4`,
    [planNumber, shortName, updatedName, createdPlan.id]
  );
  
  createdPlan.plan_number = planNumber;
  createdPlan.short_name = shortName;
  createdPlan.name = updatedName;

  return mapRowToPlan(createdPlan);
}

export async function updatePlan(
  pool: Pool,
  id: number,
  data: Partial<InsertProductionPlan>
): Promise<ProductionPlan | null> {
  const fields: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.name !== undefined) {
    fields.push(`name = $${paramIndex++}`);
    values.push(data.name);
  }

  if (data.description !== undefined) {
    fields.push(`description = $${paramIndex++}`);
    values.push(data.description);
  }

  if (data.plannedStartDate !== undefined) {
    fields.push(`planned_start_date = $${paramIndex++}`);
    values.push(data.plannedStartDate);
  }

  if (data.plannedEndDate !== undefined) {
    fields.push(`planned_end_date = $${paramIndex++}`);
    values.push(data.plannedEndDate);
  }

  if (data.status !== undefined) {
    fields.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }

  if (data.priority !== undefined) {
    fields.push(`priority = $${paramIndex++}`);
    values.push(data.priority);
  }

  if (data.notes !== undefined) {
    fields.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (data.metadata !== undefined) {
    fields.push(`metadata = $${paramIndex++}`);
    values.push(JSON.stringify(data.metadata));
  }

  if (data.autoAssignRoutings !== undefined) {
    fields.push(`auto_assign_routings = $${paramIndex++}`);
    values.push(data.autoAssignRoutings);
  }

  if (fields.length === 0) {
    return getPlanById(pool, id);
  }

  fields.push(`updated_at = CURRENT_TIMESTAMP`);
  values.push(id);

  const result = await pool.query(
    `UPDATE production.production_plans 
     SET ${fields.join(', ')} 
     WHERE id = $${paramIndex}
     RETURNING *`,
    values
  );

  if (result.rows.length === 0) {
    return null;
  }

  return mapRowToPlan(result.rows[0]);
}

export async function deletePlan(pool: Pool, id: number): Promise<boolean> {
  const result = await pool.query(
    'DELETE FROM production.production_plans WHERE id = $1',
    [id]
  );
  
  return result.rowCount !== null && result.rowCount > 0;
}

export async function getPlanLines(pool: Pool, planId: number): Promise<ProductionPlanLine[]> {
  const result = await pool.query(
    `SELECT 
       ppl.*,
       pp.id as packed_product_id,
       pp.product_sku as packed_product_sku,
       pp.reserved_quantity as warehouse_reserved_qty,
       pp.quantity as warehouse_total_qty
     FROM production.production_plan_lines ppl
     LEFT JOIN warehouse.packed_products pp ON pp.catalog_product_id = ppl.product_id
     WHERE ppl.plan_id = $1 
       AND ppl.deleted_at IS NULL
     ORDER BY ppl.sequence ASC, ppl.created_at ASC`,
    [planId]
  );
  
  return result.rows.map(mapRowToPlanLine);
}

export async function getPlanLineById(pool: Pool, id: number): Promise<ProductionPlanLine | null> {
  const result = await pool.query(
    'SELECT * FROM production.production_plan_lines WHERE id = $1',
    [id]
  );
  
  if (result.rows.length === 0) {
    return null;
  }
  
  return mapRowToPlanLine(result.rows[0]);
}

export interface CreatePlanLineResult {
  planLine: ProductionPlanLine;
  reservationInfo: {
    packedProductReserved: boolean;
    packedProductSku?: string;
    packedQtyReserved?: number;
    formatkaSearched: boolean;
    formatkiFound: string[];
    formatkiReserved: string[];
  };
}

export async function createPlanLine(pool: Pool, data: InsertProductionPlanLine): Promise<CreatePlanLineResult> {
  const client = await pool.connect();
  
  // Tracking reservation info
  const reservationInfo = {
    packedProductReserved: false,
    packedProductSku: undefined as string | undefined,
    packedQtyReserved: undefined as number | undefined,
    formatkaSearched: false,
    formatkiFound: [] as string[],
    formatkiReserved: [] as string[],
  };
  
  try {
    await client.query('BEGIN');
    
    // KROK 0a: Automatycznie pobierz BOM dla produktu (jeśli nie podano)
    let bomIdToUse = data.bomId;
    if (!bomIdToUse && data.productId) {
      const bomResult = await client.query(`
        SELECT id FROM bom.product_boms 
        WHERE product_id = $1 AND is_active = true
        ORDER BY version DESC
        LIMIT 1
      `, [data.productId]);
      
      if (bomResult.rows.length > 0) {
        bomIdToUse = bomResult.rows[0].id;
        console.log(`📋 [BOM] Auto-przypisano BOM #${bomIdToUse} dla produktu #${data.productId}`);
      }
    }
    
    // KROK 0b: Sprawdź czy dokładnie to samo źródło już istnieje W JAKIMKOLWIEK planie (zapobiegaj duplikacji globalnie)
    // NIE pozwalamy na ten sam produkt z tego samego zamówienia w żadnym planie
    // Sprawdzamy po source_reference (numer zamówienia) zamiast source_id
    if (data.sourceType === 'order_demand' && data.sourceReference) {
      const existingCheck = await client.query(
        `SELECT ppl.id, ppl.plan_id, pp.name as plan_name, pp.short_name
         FROM production.production_plan_lines ppl
         JOIN production.production_plans pp ON pp.id = ppl.plan_id
         WHERE ppl.product_id = $1 
           AND ppl.source_type = $2 
           AND ppl.source_reference = $3 
           AND ppl.deleted_at IS NULL
         LIMIT 1`,
        [data.productId, data.sourceType, data.sourceReference]
      );
      
      if (existingCheck.rows.length > 0) {
        const existing = existingCheck.rows[0];
        const planInfo = existing.short_name || `Plan #${existing.plan_id}`;
        await client.query('ROLLBACK');
        throw new Error(`Ten produkt z zamówienia #${data.sourceReference} już istnieje w planie "${planInfo}". Nie można dodać duplikatu.`);
      }
    }
    
    // KROK 1: Utwórz plan line
    const result = await client.query(
      `INSERT INTO production.production_plan_lines 
        (plan_id, product_id, quantity, reserved_quantity, source_type, source_id, source_reference,
         production_order_id, routing_id, routing_override, bom_id,
         planned_start_date, planned_end_date, status, sequence, notes, metadata,
         created_at, updated_at)
       VALUES ($1, $2, $3, 0, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, 
               CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
       RETURNING *`,
      [
        data.planId,
        data.productId,
        data.quantity,
        data.sourceType,
        data.sourceId,
        data.sourceReference,
        data.productionOrderId,
        data.routingId,
        data.routingOverride ? JSON.stringify(data.routingOverride) : null,
        bomIdToUse,
        data.plannedStartDate,
        data.plannedEndDate,
        data.status || 'pending',
        data.sequence,
        data.notes,
        data.metadata ? JSON.stringify(data.metadata) : null,
      ]
    );

    const planLineId = result.rows[0].id;

    // KROK 1.5: Automatycznie zarezerwuj produkt spakowany w magazynie (jeśli istnieje i jest dostępność)
    // Używamy tabeli packed_product_items z FIFO (najstarsze dostępne sztuki pierwsze)
    // Jeśli produkt spakowany zostanie zarezerwowany - NIE rezerwujemy formatek (produkt gotowy)
    let packedProductReserved = false;
    
    const packedProductCheck = await client.query(`
      SELECT id, product_sku, product_name
      FROM warehouse.packed_products
      WHERE catalog_product_id = $1
    `, [data.productId]);

    if (packedProductCheck.rows.length > 0) {
      const pp = packedProductCheck.rows[0];
      
      // Znajdź dostępne sztuki (FIFO - najstarsze pierwsze) i zablokuj je
      const availableItemsResult = await client.query(`
        SELECT id, serial_number
        FROM warehouse.packed_product_items
        WHERE packed_product_id = $1 
          AND status = 'available'
        ORDER BY packed_at ASC
        LIMIT $2
        FOR UPDATE SKIP LOCKED
      `, [pp.id, data.quantity]);
      
      const availableQty = availableItemsResult.rows.length;
      
      if (availableQty > 0) {
        const qtyToReserve = Math.min(data.quantity, availableQty);
        const itemIds = availableItemsResult.rows.slice(0, qtyToReserve).map((r: any) => r.id);
        const serialNumbers = availableItemsResult.rows.slice(0, qtyToReserve).map((r: any) => r.serial_number);
        
        // Zarezerwuj konkretne sztuki
        await client.query(`
          UPDATE warehouse.packed_product_items
          SET status = 'reserved',
              reserved_for_plan_line_id = $1,
              reserved_at = CURRENT_TIMESTAMP,
              updated_at = CURRENT_TIMESTAMP
          WHERE id = ANY($2::int[])
        `, [planLineId, itemIds]);
        
        // Aktualizuj również licznik w tabeli agregacyjnej (dla kompatybilności wstecznej)
        await client.query(`
          UPDATE warehouse.packed_products
          SET reserved_quantity = reserved_quantity + $1,
              updated_at = CURRENT_TIMESTAMP
          WHERE id = $2
        `, [qtyToReserve, pp.id]);
        
        // Update plan line reserved_quantity to match actual reservation
        await client.query(`
          UPDATE production.production_plan_lines
          SET reserved_quantity = $1, updated_at = CURRENT_TIMESTAMP
          WHERE id = $2
        `, [qtyToReserve, planLineId]);
        
        packedProductReserved = true;
        reservationInfo.packedProductReserved = true;
        reservationInfo.packedProductSku = pp.product_sku;
        reservationInfo.packedQtyReserved = qtyToReserve;
        console.log(`✅ [WAREHOUSE] Auto-zarezerwowano ${qtyToReserve}x SPAKOWANY ${pp.product_name} (${pp.product_sku})`);
        console.log(`📦 [WAREHOUSE] Zarezerwowane sztuki (FIFO): ${serialNumbers.join(', ')}`);
        console.log(`📦 [WAREHOUSE] Produkt spakowany - pomijam rezerwację formatek z BOM`);
      } else {
        console.log(`⚠️ [WAREHOUSE] Brak dostępności SPAKOWANEGO dla ${pp.product_name} (${pp.product_sku}). Brak dostępnych sztuk.`);
        console.log(`🔍 [WAREHOUSE] Szukam formatek z BOM do rezerwacji...`);
      }
    }

    // Fetch updated plan line
    const updatedLineResult = await client.query(
      'SELECT * FROM production.production_plan_lines WHERE id = $1',
      [planLineId]
    );
    const planLine = mapRowToPlanLine(updatedLineResult.rows[0]);

    // KROK 2: Jeśli NIE zarezerwowano spakowanego produktu - szukaj formatek z BOM
    // Pomijamy ten krok jeśli:
    // - produkt spakowany został zarezerwowany (jest gotowy, nie trzeba produkować)
    // - źródło to 'catalog_internal' lub 'catalog_product' (produkty katalogowe są produkowane w całości, formatki nie są rezerwowane z magazynu)
    const isCatalogProduct = data.sourceType === 'catalog_internal' || data.sourceType === 'catalog_product';
    const shouldReserveFormatki = !packedProductReserved && !isCatalogProduct;
    
    if (isCatalogProduct) {
      console.log(`📋 [CATALOG] Produkt katalogowy (${data.sourceType}) - formatki będą produkowane w całości, nie rezerwowane z magazynu`);
    }
    
    if (shouldReserveFormatki) {
      reservationInfo.formatkaSearched = true;
      const bomResult = await client.query(`
        SELECT bom.id as bom_id
        FROM bom.product_boms bom
        WHERE bom.product_id = $1 
          AND bom.is_active = true
        ORDER BY bom.version DESC
        LIMIT 1
      `, [data.productId]);

      if (bomResult.rows.length > 0) {
      const bomId = bomResult.rows[0].bom_id;
      
      // Pobierz komponenty BOM
      const componentsResult = await client.query(`
        SELECT 
          id,
          generated_name,
          quantity,
          calculated_length,
          calculated_width,
          thickness,
          component_type
        FROM bom.product_components
        WHERE product_bom_id = $1
      `, [bomId]);

      const reservationLogs: string[] = [];
      
      for (const component of componentsResult.rows) {
        // Identyfikuj formatki przez component_type lub konwencję nazewniczą
        const isFormatka = component.component_type === 'panel' || 
                          component.generated_name?.includes('-') ||
                          (component.calculated_length && component.calculated_width);

        if (isFormatka && component.generated_name) {
          // Wyodrębnij kolor z nazwy (ostatnia część po ostatnim myślniku)
          const nameParts = component.generated_name.split('-');
          const color = nameParts[nameParts.length - 1]?.toUpperCase() || null;
          
          const length = parseFloat(component.calculated_length) || 0;
          const width = parseFloat(component.calculated_width) || 0;
          const thickness = parseFloat(component.thickness) || null;
          
          if (length > 0 && width > 0) {
            // Wyszukaj formatkę w magazynie
            const formatka = await findFormatkaInWarehouse(
              client,
              component.generated_name,
              length,
              width,
              thickness,
              color
            );

            if (formatka) {
              reservationInfo.formatkiFound.push(formatka.name || formatka.internalCode);
              const quantityNeeded = (component.quantity || 1) * data.quantity;
              
              if (formatka.quantityAvailable >= quantityNeeded) {
                // Utwórz rezerwację
                try {
                  await createReservation(client, {
                    zlpId: null as any, // Plan line nie ma jeszcze ZLP - wypełnimy później
                    zlpItemId: planLine.id,
                    productSku: formatka.internalCode,
                    quantityReserved: quantityNeeded.toString(),
                    unitOfMeasure: formatka.unitOfMeasure,
                    locationId: null,
                    reservedBy: data.metadata?.created_by || null,
                    notes: `Rezerwacja dla planu #${data.planId}, produkt ${data.productId}: ${component.generated_name}`,
                  });
                  
                  reservationInfo.formatkiReserved.push(formatka.name || formatka.internalCode);
                  reservationLogs.push(
                    `✅ Zarezerwowano ${quantityNeeded}x ${formatka.name} (${formatka.internalCode})`
                  );
                } catch (err) {
                  console.error(`❌ Błąd rezerwacji dla ${component.generated_name}:`, err);
                  reservationLogs.push(
                    `❌ Nie udało się zarezerwować ${component.generated_name}: ${err instanceof Error ? err.message : 'Unknown error'}`
                  );
                }
              } else {
                reservationLogs.push(
                  `⚠️ Niewystarczająca ilość ${formatka.name}: potrzeba ${quantityNeeded}, dostępne ${formatka.quantityAvailable}`
                );
              }
            } else {
              reservationLogs.push(
                `⚠️ Nie znaleziono formatki: ${component.generated_name} (${length}x${width}mm, kolor: ${color || 'brak'})`
              );
            }
          }
        }
      }

      if (reservationLogs.length > 0) {
        console.log(`\n📦 [REZERWACJE FORMATEK] Plan Line #${planLine.id}:`);
        reservationLogs.forEach(log => console.log(`  ${log}`));
      }
      }
    } // Koniec bloku if (!packedProductReserved)
    
    await client.query('COMMIT');
    return { planLine, reservationInfo };
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

/**
 * Creates multiple plan lines, one for each unit of quantity.
 * This allows individual tracking of each product unit.
 * Each unit gets quantity=1 and a unique source_reference with unit index.
 */
export async function createPlanLinesWithSplit(
  pool: Pool, 
  data: InsertProductionPlanLine
): Promise<{ planLines: ProductionPlanLine[]; totalReservations: { packed: number; formatki: number } }> {
  const quantity = data.quantity || 1;
  const planLines: ProductionPlanLine[] = [];
  const totalReservations = { packed: 0, formatki: 0 };
  
  console.log(`📦 [SPLIT] Tworzenie ${quantity} osobnych rekordów dla produktu #${data.productId}`);
  
  for (let i = 1; i <= quantity; i++) {
    // Modify source_reference to include unit index for uniqueness
    const unitSourceReference = data.sourceReference 
      ? `${data.sourceReference} #${i}/${quantity}` 
      : `#${i}/${quantity}`;
    
    try {
      const result = await createPlanLine(pool, {
        ...data,
        quantity: 1,
        sourceReference: unitSourceReference,
        // Increment sequence for each unit
        sequence: data.sequence ? data.sequence + i - 1 : i,
      });
      
      planLines.push(result.planLine);
      
      if (result.reservationInfo.packedProductReserved) {
        totalReservations.packed += result.reservationInfo.packedQtyReserved || 0;
      }
      totalReservations.formatki += result.reservationInfo.formatkiReserved.length;
      
      console.log(`📦 [SPLIT] Utworzono rekord ${i}/${quantity} (ID: ${result.planLine.id})`);
    } catch (error: any) {
      // If duplicate error for this unit, skip it
      if (error.message?.includes('już istnieje')) {
        console.log(`⚠️ [SPLIT] Pominięto jednostkę ${i}/${quantity} - już istnieje`);
        continue;
      }
      throw error;
    }
  }
  
  console.log(`✅ [SPLIT] Utworzono ${planLines.length} rekordów dla produktu #${data.productId}`);
  
  return { planLines, totalReservations };
}

export async function updatePlanLine(
  pool: Pool,
  id: number,
  data: Partial<InsertProductionPlanLine>
): Promise<ProductionPlanLine | null> {
  const fields: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.productId !== undefined) {
    fields.push(`product_id = $${paramIndex++}`);
    values.push(data.productId);
  }

  if (data.quantity !== undefined) {
    fields.push(`quantity = $${paramIndex++}`);
    values.push(data.quantity);
  }

  if (data.sourceType !== undefined) {
    fields.push(`source_type = $${paramIndex++}`);
    values.push(data.sourceType);
  }

  if (data.sourceId !== undefined) {
    fields.push(`source_id = $${paramIndex++}`);
    values.push(data.sourceId);
  }

  if (data.sourceReference !== undefined) {
    fields.push(`source_reference = $${paramIndex++}`);
    values.push(data.sourceReference);
  }

  if (data.productionOrderId !== undefined) {
    fields.push(`production_order_id = $${paramIndex++}`);
    values.push(data.productionOrderId);
  }

  if (data.routingId !== undefined) {
    fields.push(`routing_id = $${paramIndex++}`);
    values.push(data.routingId);
  }

  if (data.routingOverride !== undefined) {
    fields.push(`routing_override = $${paramIndex++}`);
    values.push(data.routingOverride ? JSON.stringify(data.routingOverride) : null);
  }

  if (data.bomId !== undefined) {
    fields.push(`bom_id = $${paramIndex++}`);
    values.push(data.bomId);
  }

  if (data.plannedStartDate !== undefined) {
    fields.push(`planned_start_date = $${paramIndex++}`);
    values.push(data.plannedStartDate);
  }

  if (data.plannedEndDate !== undefined) {
    fields.push(`planned_end_date = $${paramIndex++}`);
    values.push(data.plannedEndDate);
  }

  if (data.status !== undefined) {
    fields.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }

  if (data.sequence !== undefined) {
    fields.push(`sequence = $${paramIndex++}`);
    values.push(data.sequence);
  }

  if (data.notes !== undefined) {
    fields.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (data.metadata !== undefined) {
    fields.push(`metadata = $${paramIndex++}`);
    values.push(data.metadata ? JSON.stringify(data.metadata) : null);
  }

  if (fields.length === 0) {
    return getPlanLineById(pool, id);
  }

  fields.push(`updated_at = CURRENT_TIMESTAMP`);
  values.push(id);

  const result = await pool.query(
    `UPDATE production.production_plan_lines 
     SET ${fields.join(', ')} 
     WHERE id = $${paramIndex}
     RETURNING *`,
    values
  );

  if (result.rows.length === 0) {
    return null;
  }

  return mapRowToPlanLine(result.rows[0]);
}

export async function deletePlanLine(pool: Pool, id: number): Promise<boolean> {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    
    // KROK 1: Pobierz informacje o linii przed usunięciem
    const lineInfo = await client.query(
      `SELECT product_id, quantity FROM production.production_plan_lines WHERE id = $1`,
      [id]
    );
    
    if (lineInfo.rows.length === 0) {
      await client.query('ROLLBACK');
      return false;
    }
    
    const { product_id, quantity } = lineInfo.rows[0];
    
    // KROK 2: Sprawdź czy to packed product i zwolnij rezerwację
    const packedProductCheck = await client.query(
      `SELECT id, product_sku FROM warehouse.packed_products WHERE catalog_product_id = $1 LIMIT 1`,
      [product_id]
    );
    
    if (packedProductCheck.rows.length > 0) {
      const packedProduct = packedProductCheck.rows[0];
      
      // Zwolnij pojedyncze sztuki zarezerwowane dla tej linii planu
      const releasedItemsResult = await client.query(
        `UPDATE warehouse.packed_product_items
         SET status = 'available',
             reserved_for_plan_line_id = NULL,
             reserved_at = NULL,
             updated_at = CURRENT_TIMESTAMP
         WHERE reserved_for_plan_line_id = $1
           AND status = 'reserved'
         RETURNING id, serial_number`,
        [id]
      );
      
      const releasedCount = releasedItemsResult.rows.length;
      
      // Aktualizuj licznik w tabeli agregacyjnej (dla kompatybilności wstecznej)
      await client.query(
        `UPDATE warehouse.packed_products 
         SET reserved_quantity = GREATEST(0, reserved_quantity - $1) 
         WHERE id = $2`,
        [releasedCount, packedProduct.id]
      );
      
      const serialNumbers = releasedItemsResult.rows.map((r: any) => r.serial_number);
      console.log(`📦 [ZWOLNIENIE] Zwolniono ${releasedCount} szt. rezerwacji produktu spakowanego ${packedProduct.product_sku} (ZLP line #${id} usunięta)`);
      if (serialNumbers.length > 0) {
        console.log(`📦 [ZWOLNIONE SZTUKI]: ${serialNumbers.join(', ')}`);
      }
    }
    
    // KROK 2.5: Zwolnij rezerwacje formatek BOM (production_buffer_reservations)
    const formatkiReservationsResult = await client.query(`
      SELECT id, product_sku, quantity_reserved
      FROM production.production_buffer_reservations
      WHERE zlp_item_id = $1 AND status = 'ACTIVE'
    `, [id]);
    
    if (formatkiReservationsResult.rows.length > 0) {
      // Anuluj rezerwacje formatek
      await client.query(`
        UPDATE production.production_buffer_reservations
        SET status = 'CANCELLED',
            cancelled_at = CURRENT_TIMESTAMP,
            notes = COALESCE(notes, '') || ' [Auto-anulowano - linia planu usunięta]'
        WHERE zlp_item_id = $1 AND status = 'ACTIVE'
      `, [id]);
      
      console.log(`📦 [ZWOLNIENIE FORMATEK] Zwolniono ${formatkiReservationsResult.rows.length} rezerwacji formatek (ZLP line #${id} usunięta):`);
      for (const res of formatkiReservationsResult.rows) {
        console.log(`  🔓 ${res.product_sku} (${res.quantity_reserved} szt.)`);
      }
    }
    
    // KROK 2.6: Usuń pozycje z dokumentów WZ w statusie draft powiązane z tą linią planu
    const deletedDocLinesResult = await client.query(`
      DELETE FROM warehouse.document_lines dl
      USING warehouse.documents d
      WHERE dl.document_id = d.id
        AND dl.plan_line_id = $1
        AND d.status = 'draft'
      RETURNING dl.id, d.id as doc_id, d.doc_number
    `, [id]);
    
    if (deletedDocLinesResult.rows.length > 0) {
      const affectedDocs = Array.from(new Set(deletedDocLinesResult.rows.map((r: any) => r.doc_number)));
      console.log(`📄 [DOKUMENT WZ] Usunięto ${deletedDocLinesResult.rows.length} pozycji z dokumentów: ${affectedDocs.join(', ')}`);
      
      // Sprawdź czy któryś dokument jest teraz pusty i usuń go
      const affectedDocIds = Array.from(new Set(deletedDocLinesResult.rows.map((r: any) => r.doc_id)));
      for (const docId of affectedDocIds) {
        const remainingLines = await client.query(
          'SELECT COUNT(*) as count FROM warehouse.document_lines WHERE document_id = $1',
          [docId]
        );
        if (parseInt(remainingLines.rows[0].count) === 0) {
          await client.query('DELETE FROM warehouse.documents WHERE id = $1', [docId]);
          console.log(`📄 [DOKUMENT WZ] Usunięto pusty dokument ID ${docId}`);
        }
      }
    }
    
    // KROK 3: Soft delete linii planu (zachowaj historię rezerwacji)
    const deleteResult = await client.query(
      `UPDATE production.production_plan_lines 
       SET status = 'cancelled', 
           deleted_at = CURRENT_TIMESTAMP,
           updated_at = CURRENT_TIMESTAMP
       WHERE id = $1 AND deleted_at IS NULL
       RETURNING id`,
      [id]
    );
    
    await client.query('COMMIT');
    return deleteResult.rowCount !== null && deleteResult.rowCount > 0;
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

export interface DemandFilters {
  startDate?: Date;
  endDate?: Date;
  marketplace?: string;
  orderStatus?: string[];
  paymentStatus?: string[];
}

export interface AggregatedDemand {
  productId: number;
  sku: string;
  title: string;
  totalQuantity: number;
  orderCount: number;
  orderReferences: Array<{
    orderId: number;
    orderNumber: string;
    marketplace: string;
    quantity: number;
  }>;
}

export async function aggregateDemand(
  pool: Pool,
  filters?: DemandFilters
): Promise<AggregatedDemand[]> {
  let query = `
    WITH order_product_mapping AS (
      SELECT 
        oi.id as order_item_id,
        oi.order_id,
        oi.quantity,
        oi.name as product_name,
        oi.raw_data->'offer'->'external'->>'id' as external_sku,
        o.source as marketplace,
        o.order_number,
        o.status as order_status,
        o.payment_status,
        o.order_date
      FROM commerce.order_items oi
      JOIN commerce.orders o ON oi.order_id = o.id
      WHERE oi.raw_data->'offer'->'external'->>'id' IS NOT NULL
    ),
    catalog_matched AS (
      SELECT 
        opm.*,
        cp.id as catalog_product_id,
        cp.sku as catalog_sku,
        cp.title as catalog_title
      FROM order_product_mapping opm
      LEFT JOIN catalog.products cp ON cp.sku = opm.external_sku
      WHERE cp.id IS NOT NULL
    )
    SELECT 
      cm.catalog_product_id,
      cm.catalog_sku,
      cm.catalog_title,
      SUM(cm.quantity) as total_quantity,
      COUNT(DISTINCT cm.order_id) as order_count,
      jsonb_agg(
        jsonb_build_object(
          'orderId', cm.order_id,
          'orderNumber', cm.order_number,
          'marketplace', cm.marketplace,
          'quantity', cm.quantity
        ) ORDER BY cm.order_date DESC
      ) as order_references
    FROM catalog_matched cm
    WHERE 1=1
  `;

  const params: any[] = [];
  let paramIndex = 1;

  if (filters?.startDate) {
    query += ` AND cm.order_date >= $${paramIndex++}`;
    params.push(filters.startDate);
  }

  if (filters?.endDate) {
    query += ` AND cm.order_date <= $${paramIndex++}`;
    params.push(filters.endDate);
  }

  if (filters?.marketplace) {
    query += ` AND cm.marketplace = $${paramIndex++}`;
    params.push(filters.marketplace);
  }

  if (filters?.orderStatus && filters.orderStatus.length > 0) {
    query += ` AND cm.order_status = ANY($${paramIndex++})`;
    params.push(filters.orderStatus);
  }

  if (filters?.paymentStatus && filters.paymentStatus.length > 0) {
    query += ` AND cm.payment_status = ANY($${paramIndex++})`;
    params.push(filters.paymentStatus);
  }

  query += `
    GROUP BY cm.catalog_product_id, cm.catalog_sku, cm.catalog_title
    ORDER BY total_quantity DESC
  `;

  const result = await pool.query(query, params);

  return result.rows.map(row => ({
    productId: row.catalog_product_id,
    sku: row.catalog_sku,
    title: row.catalog_title,
    totalQuantity: parseInt(row.total_quantity),
    orderCount: parseInt(row.order_count),
    orderReferences: row.order_references || []
  }));
}

export async function getPlanLinesWithDetails(pool: Pool, planId: number) {
  const result = await pool.query(`
    SELECT 
      ppl.id,
      ppl.plan_id,
      ppl.product_id,
      ppl.quantity,
      ppl.source_type,
      ppl.source_id,
      ppl.source_reference,
      ppl.production_order_id,
      ppl.routing_id,
      ppl.routing_override,
      ppl.bom_id,
      ppl.planned_start_date,
      ppl.planned_end_date,
      ppl.status,
      ppl.sequence,
      ppl.notes,
      ppl.metadata,
      ppl.created_at,
      ppl.updated_at,
      ppl.reserved_quantity as "reservedQuantity",
      
      -- Product data from catalog
      cp.sku as product_sku,
      cp.title as product_title,
      cp.color as product_color,
      cp.color_options as product_color_options,
      cp.length as product_length,
      cp.width as product_width,
      cp.height as product_height,
      cp.product_type,
      cp.product_group,
      cp.doors,
      cp.legs,
      cp.material,
      cp.base_price,
      cp.currency,
      COALESCE(
        ppl.metadata->>'image_url',
        (
          SELECT thumbnail_url
          FROM catalog.product_images
          WHERE product_id = cp.id
          ORDER BY is_primary DESC, sort_order ASC, id ASC
          LIMIT 1
        )
      ) as product_image,
      
      -- Order data from commerce.orders (if source_type = 'order_demand')
      o.order_number,
      o.order_date,
      o.source as marketplace,
      o.buyer_first_name,
      o.buyer_last_name,
      o.buyer_email,
      o.payment_status as order_payment_status,
      o.total_to_pay_amount as order_total_amount,
      
      -- BOM component count
      (SELECT COUNT(*) 
       FROM bom.product_components pc 
       INNER JOIN bom.product_boms pb ON pc.product_bom_id = pb.id 
       WHERE pb.product_id = cp.id) as bom_count,
      
      -- BOM formatki count (only components with dimensions = formatki)
      (SELECT COUNT(*) 
       FROM bom.product_components pc 
       INNER JOIN bom.product_boms pb ON pc.product_bom_id = pb.id 
       WHERE pb.product_id = cp.id 
         AND pc.calculated_length IS NOT NULL 
         AND pc.calculated_width IS NOT NULL) as bom_formatki_count,
      
      -- Reserved formatki count for this plan line from buffer reservations
      (SELECT COUNT(*) 
       FROM production.production_buffer_reservations pbr
       WHERE pbr.zlp_item_id = ppl.id 
         AND pbr.status = 'ACTIVE') as reserved_formatki_count,
      
      -- Order item ID from metadata (for duplicate prevention)
      (ppl.metadata->>'orderItemId')::integer as order_item_id,
      
      -- Warehouse packed products data
      wpp.id as packed_product_id,
      wpp.product_sku as packed_product_sku,
      wpp.quantity as warehouse_total_qty,
      wpp.reserved_quantity as warehouse_reserved_qty
      
    FROM production.production_plan_lines ppl
    INNER JOIN catalog.products cp ON ppl.product_id = cp.id
    LEFT JOIN warehouse.packed_products wpp ON wpp.catalog_product_id = ppl.product_id
    LEFT JOIN commerce.orders o ON (
      ppl.source_type = 'order_demand' 
      AND (
        ppl.source_reference = o.order_number::text
        OR ppl.source_id = o.id
        OR (ppl.metadata->>'order_number') = o.order_number::text
      )
    )
    WHERE ppl.plan_id = $1
      AND ppl.deleted_at IS NULL
    ORDER BY ppl.sequence ASC NULLS LAST, ppl.created_at ASC
  `, [planId]);

  return result.rows;
}

export async function transferPlanLine(
  pool: Pool, 
  lineId: number, 
  targetPlanId: number
): Promise<ProductionPlanLine> {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    
    // Sprawdź czy linia istnieje i pobierz jej dane
    const lineResult = await client.query(
      `SELECT * FROM production.production_plan_lines WHERE id = $1 AND deleted_at IS NULL`,
      [lineId]
    );
    
    if (lineResult.rows.length === 0) {
      throw new Error(`Linia planu #${lineId} nie istnieje`);
    }
    
    const line = lineResult.rows[0];
    const sourcePlanId = line.plan_id;
    
    if (sourcePlanId === targetPlanId) {
      throw new Error('Linia jest już w tym planie');
    }
    
    // Sprawdź czy plan docelowy istnieje
    const targetPlanResult = await client.query(
      `SELECT id, short_name, name FROM production.production_plans WHERE id = $1`,
      [targetPlanId]
    );
    
    if (targetPlanResult.rows.length === 0) {
      throw new Error(`Plan docelowy #${targetPlanId} nie istnieje`);
    }
    
    const targetPlan = targetPlanResult.rows[0];
    
    // Sprawdź czy ten sam produkt z tego samego zamówienia już istnieje w planie docelowym
    if (line.source_type === 'order_demand' && line.source_reference) {
      const duplicateCheck = await client.query(
        `SELECT id FROM production.production_plan_lines 
         WHERE plan_id = $1 
           AND product_id = $2 
           AND source_type = $3 
           AND source_reference = $4 
           AND deleted_at IS NULL
           AND id != $5
         LIMIT 1`,
        [targetPlanId, line.product_id, line.source_type, line.source_reference, lineId]
      );
      
      if (duplicateCheck.rows.length > 0) {
        throw new Error(`Ten produkt z zamówienia #${line.source_reference} już istnieje w planie "${targetPlan.short_name || targetPlan.name}"`);
      }
    }
    
    // Pobierz następny numer sekwencji w planie docelowym
    const seqResult = await client.query(
      `SELECT COALESCE(MAX(sequence), 0) + 1 as next_seq 
       FROM production.production_plan_lines 
       WHERE plan_id = $1 AND deleted_at IS NULL`,
      [targetPlanId]
    );
    const nextSequence = seqResult.rows[0].next_seq;
    
    // Przenieś linię do nowego planu (rezerwacje pozostają nietknięte!)
    const updateResult = await client.query(
      `UPDATE production.production_plan_lines 
       SET plan_id = $1, 
           sequence = $2,
           updated_at = CURRENT_TIMESTAMP
       WHERE id = $3
       RETURNING *`,
      [targetPlanId, nextSequence, lineId]
    );
    
    // Pobierz info o rezerwacjach (dla logowania)
    const reservationsResult = await client.query(
      `SELECT COUNT(*) as count, string_agg(serial_number, ', ') as serials
       FROM warehouse.packed_product_items 
       WHERE reserved_for_plan_line_id = $1 AND status = 'reserved'`,
      [lineId]
    );
    
    const reservations = reservationsResult.rows[0];
    
    await client.query('COMMIT');
    
    console.log(`🔄 [TRANSFER] Przeniesiono linię #${lineId} z planu #${sourcePlanId} do planu #${targetPlanId}`);
    if (parseInt(reservations.count) > 0) {
      console.log(`📦 [TRANSFER] Zachowano ${reservations.count} rezerwacji: ${reservations.serials}`);
    }
    
    return updateResult.rows[0];
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

// ==========================================
// AUTOMATYCZNE PRZEŁĄCZANIE REZERWACJI
// ==========================================

/**
 * Zarezerwuj formatki z magazynu dla linii planu (na podstawie BOM)
 * Wywoływane gdy odrezerwowujemy produkt spakowany
 * 
 * @param poolOrClient - Pool lub PoolClient (jeśli przekazany PoolClient, działa w istniejącej transakcji)
 * @throws Error jeśli wystąpi błąd krytyczny (DB error)
 */
export async function reserveFormatkiForPlanLine(
  poolOrClient: PoolOrClient,
  lineId: number,
  userId?: number
): Promise<{ success: boolean; logs: string[] }> {
  const isClient = 'release' in poolOrClient && typeof poolOrClient.release === 'function';
  const client = isClient ? (poolOrClient as PoolClient) : await (poolOrClient as Pool).connect();
  const logs: string[] = [];
  
  try {
    if (!isClient) await client.query('BEGIN');
    
    // Pobierz informacje o linii planu
    const lineResult = await client.query(`
      SELECT ppl.id, ppl.product_id, ppl.quantity, ppl.plan_id, ppl.bom_id
      FROM production.production_plan_lines ppl
      WHERE ppl.id = $1 AND ppl.deleted_at IS NULL
    `, [lineId]);
    
    if (lineResult.rows.length === 0) {
      // Soft error - przy własnej transakcji robimy rollback, przy przekazanym kliencie caller decyduje
      if (!isClient) await client.query('ROLLBACK');
      return { success: false, logs: ['Nie znaleziono linii planu'] };
    }
    
    const line = lineResult.rows[0];
    
    // Jeśli nie ma przypisanego BOM, pobierz aktywny BOM dla produktu
    let bomId = line.bom_id;
    if (!bomId) {
      const bomResult = await client.query(`
        SELECT id FROM bom.product_boms
        WHERE product_id = $1 AND is_active = true
        ORDER BY version DESC LIMIT 1
      `, [line.product_id]);
      
      if (bomResult.rows.length === 0) {
        // Soft success - produkt nie ma BOM, to OK (np. prosty produkt bez formatek)
        if (!isClient) await client.query('COMMIT');
        logs.push('ℹ️ Produkt nie ma przypisanego BOM - pomijam rezerwację formatek');
        return { success: true, logs };
      }
      bomId = bomResult.rows[0].id;
    }
    
    // Pobierz komponenty BOM (formatki)
    const componentsResult = await client.query(`
      SELECT 
        id,
        generated_name,
        quantity,
        calculated_length,
        calculated_width,
        thickness,
        component_type,
        color
      FROM bom.product_components
      WHERE product_bom_id = $1
    `, [bomId]);
    
    for (const component of componentsResult.rows) {
      const isFormatka = component.component_type === 'panel' || 
                        component.generated_name?.includes('-') ||
                        (component.calculated_length && component.calculated_width);

      if (isFormatka && component.generated_name) {
        const nameParts = component.generated_name.split('-');
        const color = component.color || nameParts[nameParts.length - 1]?.toUpperCase() || null;
        
        const length = parseFloat(component.calculated_length) || 0;
        const width = parseFloat(component.calculated_width) || 0;
        const thickness = parseFloat(component.thickness) || null;
        
        if (length > 0 && width > 0) {
          const formatka = await findFormatkaInWarehouse(
            client,
            component.generated_name,
            length,
            width,
            thickness,
            color
          );

          if (formatka) {
            const quantityNeeded = (component.quantity || 1) * line.quantity;
            
            if (formatka.quantityAvailable >= quantityNeeded) {
              // createReservation rzuci wyjątek przy błędzie DB - propagujemy go
              await createReservation(client, {
                zlpId: null as any,
                zlpItemId: lineId,
                productSku: formatka.internalCode,
                quantityReserved: quantityNeeded.toString(),
                unitOfMeasure: formatka.unitOfMeasure,
                locationId: null,
                reservedBy: userId || null,
                notes: `Rezerwacja dla planu #${line.plan_id}, linia #${lineId}: ${component.generated_name}`,
              });
              
              logs.push(`✅ Zarezerwowano ${quantityNeeded}x ${formatka.name}`);
            } else {
              logs.push(`⚠️ Niewystarczająca ilość ${formatka.name}: potrzeba ${quantityNeeded}, dostępne ${formatka.quantityAvailable}`);
            }
          } else {
            logs.push(`⚠️ Nie znaleziono formatki: ${component.generated_name}`);
          }
        }
      }
    }
    
    if (!isClient) await client.query('COMMIT');
    console.log(`\n📦 [AUTO-REZERWACJA FORMATEK] Linia #${lineId}:`);
    logs.forEach(log => console.log(`  ${log}`));
    
    return { success: true, logs };
  } catch (error) {
    if (!isClient) await client.query('ROLLBACK');
    console.error('Błąd rezerwacji formatek:', error);
    // Rzucamy wyjątek by główna transakcja też się rollback
    throw error;
  } finally {
    if (!isClient) client.release();
  }
}

/**
 * Zwolnij rezerwacje formatek dla linii planu
 * Wywoływane gdy rezerwujemy produkt spakowany
 * 
 * @param poolOrClient - Pool lub PoolClient (jeśli przekazany PoolClient, działa w istniejącej transakcji)
 * @throws Error jeśli wystąpi błąd krytyczny (DB error)
 */
export async function releaseFormatkiReservationsForPlanLine(
  poolOrClient: PoolOrClient,
  lineId: number
): Promise<{ success: boolean; releasedCount: number; logs: string[] }> {
  const isClient = 'release' in poolOrClient && typeof poolOrClient.release === 'function';
  const client = isClient ? (poolOrClient as PoolClient) : await (poolOrClient as Pool).connect();
  const logs: string[] = [];
  
  try {
    if (!isClient) await client.query('BEGIN');
    
    // Znajdź wszystkie aktywne rezerwacje formatek dla tej linii planu
    const reservationsResult = await client.query(`
      SELECT id, product_sku, quantity_reserved
      FROM production.production_buffer_reservations
      WHERE zlp_item_id = $1 AND status = 'ACTIVE'
    `, [lineId]);
    
    const releasedCount = reservationsResult.rows.length;
    
    if (releasedCount === 0) {
      // Soft success - brak rezerwacji do zwolnienia to OK
      if (!isClient) await client.query('COMMIT');
      logs.push('Brak rezerwacji formatek do zwolnienia');
      return { success: true, releasedCount: 0, logs };
    }
    
    // Anuluj rezerwacje - błąd DB propagujemy do głównej transakcji
    await client.query(`
      UPDATE production.production_buffer_reservations
      SET status = 'CANCELLED',
          cancelled_at = CURRENT_TIMESTAMP,
          notes = COALESCE(notes, '') || ' [Auto-anulowano - produkt spakowany zarezerwowany]'
      WHERE zlp_item_id = $1 AND status = 'ACTIVE'
    `, [lineId]);
    
    for (const res of reservationsResult.rows) {
      logs.push(`🔓 Zwolniono rezerwację: ${res.product_sku} (${res.quantity_reserved} szt.)`);
    }
    
    if (!isClient) await client.query('COMMIT');
    console.log(`\n📦 [AUTO-ZWOLNIENIE FORMATEK] Linia #${lineId}:`);
    logs.forEach(log => console.log(`  ${log}`));
    
    return { success: true, releasedCount, logs };
  } catch (error) {
    if (!isClient) await client.query('ROLLBACK');
    console.error('Błąd zwalniania rezerwacji formatek:', error);
    // Rzucamy wyjątek by główna transakcja też się rollback
    throw error;
  } finally {
    if (!isClient) client.release();
  }
}

// ===== ZLP DASHBOARD =====

export interface ZlpWorkOrder {
  id: number;
  workOrderNumber: string;
  sequence: number;
  status: string;
  operationName: string | null;
  operationCode: string | null;
  operationIcon: string | null;
  operationColor: string | null;
  workCenterName: string | null;
  bufferBeforeName: string | null;
  bufferAfterName: string | null;
  quantityPlanned: number;
  quantityProduced: number;
  quantityScrap: number;
  actualStartTime: string | null;
  actualEndTime: string | null;
  operatorUserId?: number | null;
  operatorName?: string | null;
  routingOperationId?: number | null;
}

export interface ZlpSummary {
  id: number;
  orderNumber: string;
  status: string;
  colorCode: string | null;
  routingId: number | null;
  routingName: string | null;
  quantityPlanned: number;
  quantityProduced: number;
  quantityScrap: number;
  progress: number; // Percentage 0-100
  workOrders: ZlpWorkOrder[];
  activeOperationIndex: number | null;
  createdAt: string;
  updatedAt: string;
}

export interface ZlpDashboardComponent {
  id: number;
  componentType: string;
  name: string;
  sku: string | null;
  quantity: number;
  reservedQuantity: number;
  zlpCount: number;
}

export interface ZlpDashboardData {
  planId: number;
  planNumber: string;
  planName: string;
  totalZlps: number;
  completedZlps: number;
  inProgressZlps: number;
  pendingZlps: number;
  overallProgress: number;
  zlps: ZlpSummary[];
  components: ZlpDashboardComponent[];
}

export async function getZlpDashboardData(pool: Pool, planId: number): Promise<ZlpDashboardData | null> {
  // Get plan info
  const planResult = await pool.query(`
    SELECT id, plan_number, name
    FROM production.production_plans
    WHERE id = $1
  `, [planId]);
  
  if (planResult.rows.length === 0) return null;
  
  const plan = planResult.rows[0];
  
  // Get all ZLPs (production orders) for this plan through source_order_number OR production_plan_lines
  const zlpsResult = await pool.query(`
    SELECT DISTINCT ON (po.id)
      po.id,
      po.order_number,
      po.status,
      po.color_code,
      po.routing_id,
      r.name as routing_name,
      po.quantity_planned,
      po.quantity_produced,
      po.created_at,
      po.updated_at,
      COALESCE(scrap.total_scrap, 0) as total_scrap
    FROM production.production_orders po
    LEFT JOIN production.production_routings r ON r.id = po.routing_id
    LEFT JOIN LATERAL (
      SELECT SUM(COALESCE(quantity_scrap, 0)) as total_scrap
      FROM production.production_work_orders
      WHERE production_order_id = po.id
    ) scrap ON true
    WHERE po.source_order_number = $2
       OR po.id IN (
         SELECT ppl.production_order_id 
         FROM production.production_plan_lines ppl 
         WHERE ppl.plan_id = $1 AND ppl.deleted_at IS NULL AND ppl.production_order_id IS NOT NULL
       )
    ORDER BY po.id, po.order_number
  `, [planId, plan.plan_number]);
  
  // Get work orders for all ZLPs (through source_order_number OR plan_lines)
  // Use dictionary icons/colors as fallback when routing_operations don't have them set
  const workOrdersResult = await pool.query(`
    SELECT 
      wo.id,
      wo.production_order_id,
      wo.work_order_number,
      wo.sequence,
      wo.status,
      wo.quantity_planned,
      wo.quantity_produced,
      wo.quantity_scrap,
      wo.actual_start_time,
      wo.actual_end_time,
      wo.operator_user_id,
      wo.operator_id,
      COALESCE(wo.operator_name, op.full_name, u.first_name || ' ' || u.last_name, u.username) as operator_name,
      ro.id as routing_operation_id,
      ro.name as operation_name,
      ro.code as operation_code,
      COALESCE(ro.icon, dict.icon) as operation_icon,
      COALESCE(ro.color, dict.color) as operation_color,
      wc.name as work_center_name,
      buf_before.name as buffer_before_name,
      buf_after.name as buffer_after_name
    FROM production.production_work_orders wo
    LEFT JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
    LEFT JOIN production.production_work_centers wc ON wc.id = wo.work_center_id
    LEFT JOIN production.production_locations buf_before ON buf_before.id = wo.buffer_before_id
    LEFT JOIN production.production_locations buf_after ON buf_after.id = wo.buffer_after_id
    LEFT JOIN product_creator.dictionaries dict ON dict.dictionary_type = 'production_operation' AND LOWER(TRIM(dict.name)) = LOWER(TRIM(ro.name))
    LEFT JOIN production.production_operators op ON op.id = wo.operator_id
    LEFT JOIN users u ON u.id = wo.operator_user_id
    WHERE wo.production_order_id IN (
      SELECT po.id FROM production.production_orders po
      WHERE po.source_order_number = $2
         OR po.id IN (
           SELECT ppl.production_order_id 
           FROM production.production_plan_lines ppl 
           WHERE ppl.plan_id = $1 AND ppl.deleted_at IS NULL AND ppl.production_order_id IS NOT NULL
         )
    )
    ORDER BY wo.production_order_id, wo.sequence
  `, [planId, plan.plan_number]);
  
  // Group work orders by ZLP
  const workOrdersByZlp = new Map<number, ZlpWorkOrder[]>();
  for (const wo of workOrdersResult.rows) {
    const zlpId = wo.production_order_id;
    if (!workOrdersByZlp.has(zlpId)) {
      workOrdersByZlp.set(zlpId, []);
    }
    workOrdersByZlp.get(zlpId)!.push({
      id: wo.id,
      workOrderNumber: wo.work_order_number,
      sequence: wo.sequence,
      status: wo.status,
      operationName: wo.operation_name,
      operationCode: wo.operation_code,
      operationIcon: wo.operation_icon,
      operationColor: wo.operation_color,
      workCenterName: wo.work_center_name,
      bufferBeforeName: wo.buffer_before_name,
      bufferAfterName: wo.buffer_after_name,
      quantityPlanned: parseFloat(wo.quantity_planned || 0),
      quantityProduced: parseFloat(wo.quantity_produced || 0),
      quantityScrap: parseFloat(wo.quantity_scrap || 0),
      actualStartTime: wo.actual_start_time,
      actualEndTime: wo.actual_end_time,
      operatorUserId: wo.operator_user_id,
      operatorName: wo.operator_name,
      routingOperationId: wo.routing_operation_id,
    });
  }
  
  // Build ZLP summaries
  const zlps: ZlpSummary[] = zlpsResult.rows.map(row => {
    const workOrders = workOrdersByZlp.get(row.id) || [];
    const completedCount = workOrders.filter(wo => wo.status === 'done').length;
    const progress = workOrders.length > 0 ? Math.round((completedCount / workOrders.length) * 100) : 0;
    const activeIndex = workOrders.findIndex(wo => wo.status === 'in_progress');
    
    return {
      id: row.id,
      orderNumber: row.order_number,
      status: row.status,
      colorCode: row.color_code,
      routingId: row.routing_id,
      routingName: row.routing_name,
      quantityPlanned: parseFloat(row.quantity_planned || 0),
      quantityProduced: parseFloat(row.quantity_produced || 0),
      quantityScrap: parseFloat(row.total_scrap || 0),
      progress,
      workOrders,
      activeOperationIndex: activeIndex >= 0 ? activeIndex : null,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  });
  
  // Get aggregated components from BOM items for ZLPs of this plan
  // Note: production_order_bom_items uses component_name, component_type, color_code
  // There is no warehouse_material_id, packed_product_id, or component_sku column
  const componentsResult = await pool.query(`
    SELECT 
      bi.component_type,
      COALESCE(bi.component_name, 'Nieznany') as name,
      bi.color_code as sku,
      SUM(bi.quantity) as quantity,
      0 as reserved_quantity,
      COUNT(DISTINCT bi.production_order_bom_id) as zlp_count
    FROM production.production_order_bom_items bi
    JOIN production.production_order_boms bob ON bob.id = bi.production_order_bom_id
    JOIN production.production_orders po ON po.id = bob.production_order_id
    WHERE po.source_order_number = $2
       OR po.id IN (
         SELECT ppl.production_order_id 
         FROM production.production_plan_lines ppl 
         WHERE ppl.plan_id = $1 AND ppl.deleted_at IS NULL AND ppl.production_order_id IS NOT NULL
       )
    GROUP BY bi.component_type, COALESCE(bi.component_name, 'Nieznany'), bi.color_code
    ORDER BY bi.component_type, name
  `, [planId, plan.plan_number]);
  
  const components: ZlpDashboardComponent[] = componentsResult.rows.map((row, index) => ({
    id: index + 1,
    componentType: row.component_type || 'unknown',
    name: row.name,
    sku: row.sku,
    quantity: parseFloat(row.quantity || 0),
    reservedQuantity: parseFloat(row.reserved_quantity || 0),
    zlpCount: parseInt(row.zlp_count || 0),
  }));
  
  // Calculate stats
  const completedZlps = zlps.filter(z => z.status === 'done').length;
  const inProgressZlps = zlps.filter(z => z.status === 'in_progress').length;
  const pendingZlps = zlps.filter(z => ['draft', 'confirmed', 'pending'].includes(z.status)).length;
  const totalProgress = zlps.length > 0 
    ? Math.round(zlps.reduce((sum, z) => sum + z.progress, 0) / zlps.length) 
    : 0;
  
  return {
    planId: plan.id,
    planNumber: plan.plan_number,
    planName: plan.name,
    totalZlps: zlps.length,
    completedZlps,
    inProgressZlps,
    pendingZlps,
    overallProgress: totalProgress,
    zlps,
    components,
  };
}

// ============================================================================
// Flow Graph Generation for Production Plan Visualization
// ============================================================================

import type { 
  FlowNode, 
  FlowEdge, 
  FlowNodeType, 
  FlowNodeStatus, 
  ComponentFamily,
  ProductionPlanFlowGraph 
} from "@shared/schema";

interface AssignedOperator {
  operatorId: number;
  operatorName: string;
  isPrimary: boolean;
}

interface OperationAggregation {
  operationCode: string;
  operationName: string;
  operationIcon?: string;
  operationColor?: string;
  sequence: number;
  workCenterName?: string;
  bufferBeforeName?: string;
  bufferAfterName?: string;
  routingOperationId?: number;
  zlpIds: number[];
  zlpNumbers: string[];
  workOrderIds: number[];
  itemCount: number;
  completedCount: number;
  damagedCount: number;
  componentFamilies: Set<string>;
  statuses: Set<string>;
  actualStartTime?: string;
  actualEndTime?: string;
  operatorId?: number | null;
  operatorName?: string;
  assignedOperators: AssignedOperator[];
}

function determineComponentFamily(componentName: string | undefined): ComponentFamily {
  if (!componentName) return 'mixed';
  const upper = componentName.toUpperCase();
  
  if (upper.startsWith('HDF-')) return 'formatki_hdf';
  if (upper.includes('BIAŁY') || upper.includes('BIALY') || upper.includes('WHITE')) return 'formatki_biale';
  if (upper.startsWith('SIEDZISKO-')) return 'siedziska';
  if (upper.startsWith('TAPIC-') || upper.includes('TAPICEROWANE')) return 'tapicerowane';
  if (upper.startsWith('WD-') || upper.startsWith('WG-') || upper.startsWith('BOK-') || 
      upper.startsWith('DRZWI-') || upper.startsWith('PÓŁKA-') || upper.startsWith('POLKA-') ||
      upper.startsWith('TYŁ-') || upper.startsWith('TYL-') || upper.startsWith('FRONT-') ||
      upper.startsWith('BLAT-') || upper.startsWith('DNA-') || upper.startsWith('DNO-') ||
      upper.startsWith('DENKO-') || upper.startsWith('PANEL-') || upper.startsWith('LISTWA-') ||
      upper.startsWith('BOCZKI-')) {
    return 'formatki_kolor';
  }
  return 'akcesoria';
}

function aggregateNodeStatus(statuses: Set<string>): FlowNodeStatus {
  if (statuses.has('in_progress')) return 'in_progress';
  if (statuses.has('done') && statuses.size === 1) return 'completed';
  if (statuses.has('ready')) return 'ready';
  if (statuses.has('blocked')) return 'blocked';
  return 'pending';
}

export async function getPlanFlowGraph(pool: Pool, planId: number): Promise<ProductionPlanFlowGraph | null> {
  const plan = await getPlanById(pool, planId);
  if (!plan) return null;
  
  // Get all ZLPs and their work orders for this plan
  const zlpsResult = await pool.query(`
    SELECT DISTINCT ON (po.id)
      po.id,
      po.order_number,
      po.status,
      po.color_code,
      po.quantity_planned
    FROM production.production_orders po
    WHERE po.source_order_number = $2
       OR po.id IN (
         SELECT ppl.production_order_id 
         FROM production.production_plan_lines ppl 
         WHERE ppl.plan_id = $1 AND ppl.deleted_at IS NULL AND ppl.production_order_id IS NOT NULL
       )
    ORDER BY po.id
  `, [planId, plan.planNumber]);
  
  if (zlpsResult.rows.length === 0) {
    return {
      planId,
      planName: plan.name,
      nodes: [],
      edges: [],
      metadata: {
        totalZlps: 0,
        totalItems: 0,
        completedItems: 0,
        damagedItems: 0,
        componentFamilies: [],
        generatedAt: new Date().toISOString(),
      },
    };
  }
  
  const zlpIds = zlpsResult.rows.map(r => r.id);
  const zlpMap = new Map(zlpsResult.rows.map(r => [r.id, r]));
  
  // Get pallets for all ZLPs with carrier names
  const palletsResult = await pool.query(`
    SELECT 
      pop.id,
      pop.production_order_id,
      pop.pallet_label,
      pop.sequence,
      pop.flow_code,
      pop.routing_id,
      pop.routing_variant_code,
      pop.carrier_id,
      pop.status,
      pop.split_after_operation,
      pop.split_operation_id,
      pop.component_count,
      po.order_number as zlp_number,
      po.color_code,
      pc.code as carrier_code,
      pc.name as carrier_name,
      pc.barcode as carrier_barcode
    FROM production.production_order_pallets pop
    JOIN production.production_orders po ON po.id = pop.production_order_id
    LEFT JOIN production.production_carriers pc ON pc.id = pop.carrier_id
    WHERE pop.production_order_id = ANY($1)
    ORDER BY pop.production_order_id, pop.sequence
  `, [zlpIds]);

  // Map: zlpId -> pallets[]
  interface PalletInfo {
    id: number;
    label: string;
    flowCode: string;
    componentCount: number;
    status: string;
    splitAfter: string | null;
    zlpNumber: string;
    colorCode: string;
    carrierId: number | null;
    carrierCode: string | null;
    carrierName: string | null;
    carrierBarcode: string | null;
    routingId: number | null;
  }
  const palletsByZlp = new Map<number, PalletInfo[]>();
  const allPalletsByFlow = new Map<string, PalletInfo[]>();
  
  for (const pallet of palletsResult.rows) {
    const palletInfo: PalletInfo = {
      id: pallet.id,
      label: pallet.pallet_label,
      flowCode: pallet.flow_code,
      componentCount: pallet.component_count || 0,
      status: pallet.status,
      splitAfter: pallet.split_after_operation,
      zlpNumber: pallet.zlp_number,
      colorCode: pallet.color_code,
      carrierId: pallet.carrier_id,
      carrierCode: pallet.carrier_code,
      carrierName: pallet.carrier_name,
      carrierBarcode: pallet.carrier_barcode,
      routingId: pallet.routing_id,
    };
    
    if (!palletsByZlp.has(pallet.production_order_id)) {
      palletsByZlp.set(pallet.production_order_id, []);
    }
    palletsByZlp.get(pallet.production_order_id)!.push(palletInfo);
    
    if (!allPalletsByFlow.has(pallet.flow_code)) {
      allPalletsByFlow.set(pallet.flow_code, []);
    }
    allPalletsByFlow.get(pallet.flow_code)!.push(palletInfo);
  }

  // Collect all unique flow codes
  const allFlowCodes = Array.from(allPalletsByFlow.keys());
  const hasPallets = palletsResult.rows.length > 0;
  
  // Build a map of flowCode -> Set of routing_ids from pallets
  // This is used to filter post-split operations to only show operations that exist in each flow's routing
  // We compare work order's routing_operation.routing_id to pallet's routing_id
  const flowCodeRoutingIds = new Map<string, Set<number>>();
  
  // Map flowCode -> set of routing_ids from pallets with that flowCode
  for (const [flowCode, pallets] of Array.from(allPalletsByFlow.entries())) {
    const routingIdsSet = new Set<number>();
    for (const pallet of pallets) {
      if (pallet.routingId) {
        routingIdsSet.add(pallet.routingId);
      }
    }
    if (routingIdsSet.size > 0) {
      flowCodeRoutingIds.set(flowCode, routingIdsSet);
    }
    // If no routing info found for this flowCode, don't add to map - it will use fallback (inclusive) behavior
  }
  
  // Get work orders with operation details
  const workOrdersResult = await pool.query(`
    SELECT 
      wo.id,
      wo.production_order_id,
      wo.work_order_number,
      wo.sequence,
      wo.status,
      wo.quantity_planned,
      wo.quantity_produced,
      wo.quantity_scrap,
      wo.actual_start_time,
      wo.actual_end_time,
      wo.operator_user_id,
      wo.operator_id,
      COALESCE(wo.operator_name, op.full_name, u.first_name || ' ' || u.last_name, u.username) as operator_name,
      ro.id as routing_operation_id,
      ro.routing_id as operation_routing_id,
      ro.name as operation_name,
      ro.code as operation_code,
      COALESCE(ro.icon, dict.icon) as operation_icon,
      COALESCE(ro.color, dict.color) as operation_color,
      wc.id as work_center_id,
      wc.name as work_center_name,
      wc.code as work_center_code,
      buf_before.id as buffer_before_id,
      buf_before.name as buffer_before_name,
      buf_before.code as buffer_before_code,
      buf_after.id as buffer_after_id,
      buf_after.name as buffer_after_name,
      buf_after.code as buffer_after_code,
      loc.id as location_id,
      loc.name as location_name,
      loc.code as location_code
    FROM production.production_work_orders wo
    LEFT JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
    LEFT JOIN production.production_work_centers wc ON wc.id = wo.work_center_id
    LEFT JOIN production.production_locations buf_before ON buf_before.id = wo.buffer_before_id
    LEFT JOIN production.production_locations buf_after ON buf_after.id = wo.buffer_after_id
    LEFT JOIN production.production_locations loc ON loc.id = wc.location_id
    LEFT JOIN product_creator.dictionaries dict ON dict.dictionary_type = 'production_operation' AND LOWER(TRIM(dict.name)) = LOWER(TRIM(ro.name))
    LEFT JOIN production.production_operators op ON op.id = wo.operator_id
    LEFT JOIN users u ON u.id = wo.operator_user_id
    WHERE wo.production_order_id = ANY($1)
    ORDER BY wo.sequence, wo.production_order_id
  `, [zlpIds]);
  
  // Get all work order IDs for operator lookup
  const allWorkOrderIds = workOrdersResult.rows.map(r => r.id);
  
  // Get operators from junction table for all work orders
  const operatorsResult = await pool.query(`
    SELECT 
      woo.work_order_id,
      woo.operator_id,
      woo.is_primary,
      op.full_name as operator_name,
      op.short_code as operator_code
    FROM production.work_order_operators woo
    JOIN production.production_operators op ON op.id = woo.operator_id
    WHERE woo.work_order_id = ANY($1)
    ORDER BY woo.work_order_id, woo.is_primary DESC, op.full_name
  `, [allWorkOrderIds]);
  
  // Build map of work order -> operators
  const workOrderOperatorsMap = new Map<number, Array<{ operatorId: number; operatorName: string; isPrimary: boolean }>>();
  for (const row of operatorsResult.rows) {
    if (!workOrderOperatorsMap.has(row.work_order_id)) {
      workOrderOperatorsMap.set(row.work_order_id, []);
    }
    workOrderOperatorsMap.get(row.work_order_id)!.push({
      operatorId: row.operator_id,
      operatorName: row.operator_name,
      isPrimary: row.is_primary,
    });
  }
  
  // Get BOM items for material color analysis
  const bomItemsResult = await pool.query(`
    SELECT 
      bob.production_order_id,
      bi.component_name,
      bi.color_code,
      bi.is_damaged,
      bi.quantity
    FROM production.production_order_bom_items bi
    JOIN production.production_order_boms bob ON bob.id = bi.production_order_bom_id
    WHERE bob.production_order_id = ANY($1)
  `, [zlpIds]);
  
  // Build material colors per ZLP (use actual color_code instead of generic families)
  const zlpColors = new Map<number, Set<string>>();
  const zlpDamagedCount = new Map<number, number>();
  const zlpItemCount = new Map<number, number>();
  
  for (const item of bomItemsResult.rows) {
    const zlpId = item.production_order_id;
    if (!zlpColors.has(zlpId)) {
      zlpColors.set(zlpId, new Set());
      zlpDamagedCount.set(zlpId, 0);
      zlpItemCount.set(zlpId, 0);
    }
    // Use actual color_code from BOM, fallback to component family if not set
    const colorCode = item.color_code || determineComponentFamily(item.component_name);
    zlpColors.get(zlpId)!.add(colorCode);
    zlpItemCount.set(zlpId, (zlpItemCount.get(zlpId) || 0) + parseInt(item.quantity || 1));
    if (item.is_damaged) {
      zlpDamagedCount.set(zlpId, (zlpDamagedCount.get(zlpId) || 0) + 1);
    }
  }
  
  // Determine primary color for each ZLP (first non-HDF color, or first color)
  const zlpPrimaryColor = new Map<number, string>();
  for (const [zlpId, colors] of Array.from(zlpColors.entries())) {
    const colorArray = Array.from(colors);
    // Prefer non-HDF colors first, then HDF
    const primaryColor = colorArray.find(c => !c.startsWith('HDF')) || colorArray[0] || 'mixed';
    zlpPrimaryColor.set(zlpId, primaryColor);
  }
  
  // FIRST: Determine split point sequence (where pallet flows diverge - e.g., after edging)
  // This is needed BEFORE aggregation to properly split operations by flowCode
  let preScanSplitSequence: number | null = null;
  for (const wo of workOrdersResult.rows) {
    const isSplitOp = wo.operation_code?.toLowerCase().includes('oklej') || 
                      wo.operation_code?.toLowerCase().includes('edging') ||
                      wo.operation_name?.toLowerCase().includes('oklejanie');
    if (isSplitOp) {
      preScanSplitSequence = wo.sequence;
      break;
    }
  }
  
  // Aggregate operations by code+sequence+color for separate lanes
  // For post-split operations, also include flowCode in the key
  const operationAggregations = new Map<string, OperationAggregation & { materialColor: string; flowCode?: string }>();
  
  for (const wo of workOrdersResult.rows) {
    const zlpId = wo.production_order_id;
    const zlp = zlpMap.get(zlpId);
    const materialColor = zlpPrimaryColor.get(zlpId) || 'mixed';
    
    // For post-split operations, get flowCodes from pallets and create separate aggregations
    const isPostSplitOp = preScanSplitSequence !== null && wo.sequence > preScanSplitSequence;
    const zlpPallets = palletsByZlp.get(zlpId) || [];
    
    // Determine which flowCodes to aggregate by
    let flowCodesToAggregate: (string | undefined)[] = [undefined];
    if (isPostSplitOp && zlpPallets.length > 0) {
      // Get unique flowCodes from pallets for this ZLP
      const uniqueFlowCodes = Array.from(new Set(zlpPallets.map(p => p.flowCode)));
      if (uniqueFlowCodes.length > 0) {
        // Use operation_routing_id (the routing this operation belongs to) for reliable matching
        const operationRoutingId = wo.operation_routing_id;
        
        // If work order lacks operation_routing_id, keep on shared branch (pre-split behavior)
        // This prevents operations with missing routing metadata from spreading to all flows
        if (!operationRoutingId) {
          console.log(`⚠️ [FlowTree] WO#${wo.id} (${wo.operation_code}) brak operation_routing_id - pozostaje na wspólnej gałęzi`);
          // Keep flowCodesToAggregate as [undefined] - stays on shared branch
        } else {
          // Count how many flowCodes have routing data at all
          const flowCodesWithRoutingData = uniqueFlowCodes.filter(flowCode => {
            const palletRoutingIds = flowCodeRoutingIds.get(flowCode);
            return palletRoutingIds && palletRoutingIds.size > 0;
          });
          
          // Only filter by routing if at least one flowCode has routing data
          // Otherwise, we can't make routing-based decisions
          if (flowCodesWithRoutingData.length === 0) {
            // No pallet routing data available - include all flowCodes (data quality limitation)
            console.log(`⚠️ [FlowTree] WO#${wo.id} (${wo.operation_code}) brak danych routing w paletach - włączam wszystkie flowCodes`);
            flowCodesToAggregate = uniqueFlowCodes;
          } else {
            // We have routing data - filter strictly by routing match
            const filteredFlowCodes = uniqueFlowCodes.filter(flowCode => {
              const palletRoutingIds = flowCodeRoutingIds.get(flowCode);
              // If this flowCode has no routing data, EXCLUDE it (since other flows have data)
              if (!palletRoutingIds || palletRoutingIds.size === 0) {
                console.log(`⚠️ [FlowTree] WO#${wo.id} flowCode=${flowCode} bez routing - wykluczam (inne flow mają dane)`);
                return false;
              }
              // Check if the work order's routing matches any of this flowCode's pallet routings
              return palletRoutingIds.has(operationRoutingId);
            });
            
            // If routing matched exactly one or more flows, use them
            if (filteredFlowCodes.length > 0) {
              flowCodesToAggregate = filteredFlowCodes;
            } else {
              // No flows matched - routing mismatch (legacy/archived routing or override)
              // Log for diagnostics and keep on shared branch to avoid spreading to all
              console.log(`⚠️ [FlowTree] WO#${wo.id} (${wo.operation_code}) routing_id=${operationRoutingId} nie pasuje do żadnego flowCode - pozostaje na wspólnej gałęzi`);
              // Keep flowCodesToAggregate as [undefined] - stays on shared branch
            }
          }
        }
      }
    }
    
    // Create aggregation for each flowCode (or one without flowCode for pre-split ops)
    for (const flowCode of flowCodesToAggregate) {
      const key = flowCode 
        ? `${wo.operation_code || 'op'}-${wo.sequence}-${materialColor}-${flowCode}`
        : `${wo.operation_code || 'op'}-${wo.sequence}-${materialColor}`;
      
      if (!operationAggregations.has(key)) {
        // Get operators from junction table for this work order
        const woOperators = workOrderOperatorsMap.get(wo.id) || [];
        const primaryOp = woOperators.find(op => op.isPrimary) || woOperators[0];
        
        // For post-split ops with flowCode, calculate itemCount from matching pallets only
        let initialItemCount = 0;
        if (flowCode && isPostSplitOp) {
          const matchingPallets = zlpPallets.filter(p => p.flowCode === flowCode);
          initialItemCount = matchingPallets.reduce((sum, p) => sum + (p.componentCount || 0), 0);
        } else {
          initialItemCount = zlpItemCount.get(zlpId) || 0;
        }
        
        operationAggregations.set(key, {
          operationCode: wo.operation_code || 'unknown',
          operationName: wo.operation_name || 'Operacja',
          operationIcon: wo.operation_icon,
          operationColor: wo.operation_color,
          sequence: wo.sequence,
          workCenterName: wo.work_center_name,
          bufferBeforeName: wo.buffer_before_name,
          bufferAfterName: wo.buffer_after_name,
          routingOperationId: wo.routing_operation_id,
          zlpIds: [],
          zlpNumbers: [],
          workOrderIds: [],
          itemCount: 0,
          completedCount: 0,
          damagedCount: 0,
          componentFamilies: new Set([materialColor]),
          statuses: new Set(),
          materialColor,
          flowCode,
          actualStartTime: wo.actual_start_time ? new Date(wo.actual_start_time).toISOString() : undefined,
          actualEndTime: wo.actual_end_time ? new Date(wo.actual_end_time).toISOString() : undefined,
          operatorId: primaryOp?.operatorId || wo.operator_id || null,
          operatorName: primaryOp?.operatorName || wo.operator_name || undefined,
          assignedOperators: woOperators,
        });
      }
      
      const agg = operationAggregations.get(key)!;
      
      // Only add this ZLP if not already added to this aggregation
      if (!agg.zlpIds.includes(zlpId)) {
        agg.zlpIds.push(zlpId);
        agg.zlpNumbers.push(zlp?.order_number || '');
      }
      if (!agg.workOrderIds.includes(wo.id)) {
        agg.workOrderIds.push(wo.id);
      }
      agg.statuses.add(wo.status);
      
      // For post-split ops, use pallet-based item count; for pre-split, use ZLP item count
      if (flowCode && isPostSplitOp) {
        const matchingPallets = zlpPallets.filter(p => p.flowCode === flowCode);
        const palletItemCount = matchingPallets.reduce((sum, p) => sum + (p.componentCount || 0), 0);
        // Add only if we haven't already counted this ZLP's contribution
        if (!agg.zlpIds.slice(0, -1).includes(zlpId)) {
          agg.itemCount += palletItemCount;
        }
      } else if (!agg.zlpIds.slice(0, -1).includes(zlpId)) {
        agg.itemCount += zlpItemCount.get(zlpId) || 0;
      }
      
      if (!agg.zlpIds.slice(0, -1).includes(zlpId)) {
        agg.damagedCount += zlpDamagedCount.get(zlpId) || 0;
      }
      
      if (wo.status === 'done') {
        agg.completedCount += parseInt(wo.quantity_produced || 0);
      }
      
      // Merge operators from this work order into aggregation
      const woOperators = workOrderOperatorsMap.get(wo.id) || [];
      for (const op of woOperators) {
        // Avoid duplicates - check if operator already exists
        if (!agg.assignedOperators.find(existing => existing.operatorId === op.operatorId)) {
          agg.assignedOperators.push(op);
        }
      }
      
      // Update times and operator if this is an in_progress work order (most recent data)
      if (wo.status === 'in_progress' && wo.actual_start_time) {
        agg.actualStartTime = new Date(wo.actual_start_time).toISOString();
        // Update primary operator from junction table
        const primaryOp = woOperators.find(op => op.isPrimary) || woOperators[0];
        if (primaryOp) {
          agg.operatorId = primaryOp.operatorId;
          agg.operatorName = primaryOp.operatorName;
        } else if (wo.operator_id) {
          agg.operatorId = wo.operator_id;
        }
        if (!primaryOp && wo.operator_name) {
          agg.operatorName = wo.operator_name;
        }
      }
      if (wo.actual_end_time) {
        agg.actualEndTime = new Date(wo.actual_end_time).toISOString();
      }
    }
  }
  
  // Group operations by sequence to find merge/split points
  const opsBySequence = new Map<number, Array<{ key: string; agg: OperationAggregation & { materialColor: string } }>>();
  for (const [key, agg] of Array.from(operationAggregations.entries())) {
    if (!opsBySequence.has(agg.sequence)) {
      opsBySequence.set(agg.sequence, []);
    }
    opsBySequence.get(agg.sequence)!.push({ key, agg });
  }
  
  const sortedSequences = Array.from(opsBySequence.keys()).sort((a, b) => a - b);
  
  // Build nodes and edges
  const nodes: FlowNode[] = [];
  const edges: FlowEdge[] = [];
  const addedNodeIds = new Set<string>();
  
  // Start node
  nodes.push({
    id: 'start',
    type: 'start',
    label: 'Start',
    status: 'completed',
    data: {
      zlpIds: zlpIds,
      zlpNumbers: zlpsResult.rows.map(r => r.order_number),
      itemCount: Array.from(zlpItemCount.values()).reduce((a, b) => a + b, 0),
    },
  });
  addedNodeIds.add('start');
  
  // Track previous nodes per color lane for edge connections
  // For post-split operations, we also track by flowCode (color-flowCode key)
  const prevNodeByLane = new Map<string, string>();
  const allColorsInPlan = new Set<string>();
  for (const [, agg] of Array.from(operationAggregations.entries())) {
    allColorsInPlan.add(agg.materialColor);
  }
  
  // Connect start to first operation of each color
  for (const color of Array.from(allColorsInPlan)) {
    prevNodeByLane.set(color, 'start');
  }
  
  // First pass: find the split point sequence
  let splitPointSequence: number | null = null;
  for (const seq of sortedSequences) {
    const opsAtSeq = opsBySequence.get(seq)!;
    for (const { agg } of opsAtSeq) {
      const isSplitOperation = agg.operationCode?.toLowerCase().includes('oklej') || 
                               agg.operationCode?.toLowerCase().includes('edging') ||
                               agg.operationName?.toLowerCase().includes('oklejanie');
      if (isSplitOperation) {
        splitPointSequence = seq;
        break;
      }
    }
    if (splitPointSequence !== null) break;
  }
  
  // Process operations by sequence
  for (const seq of sortedSequences) {
    const opsAtSeq = opsBySequence.get(seq)!;
    const isSplitPoint = opsAtSeq.length > 1;
    
    // Check if this is a merge point (multiple families converging to same buffer after)
    const bufferAfterSet = new Set(opsAtSeq.map(o => o.agg.bufferAfterName).filter(Boolean));
    const isMergeToBuffer = bufferAfterSet.size === 1 && opsAtSeq.length > 1;
    
    for (const { key, agg } of opsAtSeq) {
      const materialColor = agg.materialColor;
      const nodeId = `op-${key}`;
      
      if (!addedNodeIds.has(nodeId)) {
        // Collect pallets for all ZLPs in this operation
        // For post-split operations with flowCode, filter pallets to matching flowCode only
        const palletsForOp: PalletInfo[] = [];
        const flowCodesForOp = new Set<string>();
        const aggFlowCode = (agg as any).flowCode as string | undefined;
        const isPostSplit = splitPointSequence !== null && seq > splitPointSequence;
        
        for (const zlpId of agg.zlpIds) {
          const zlpPallets = palletsByZlp.get(zlpId) || [];
          for (const p of zlpPallets) {
            // For post-split operations with a specific flowCode, only include matching pallets
            if (isPostSplit && aggFlowCode) {
              if (p.flowCode === aggFlowCode) {
                palletsForOp.push(p);
                flowCodesForOp.add(p.flowCode);
              }
            } else {
              // For pre-split operations, include all pallets
              palletsForOp.push(p);
              flowCodesForOp.add(p.flowCode);
            }
          }
        }
        
        // Check if this is a split point (after edging/oklejanie)
        const isSplitOperation = agg.operationCode?.toLowerCase().includes('oklej') || 
                                  agg.operationCode?.toLowerCase().includes('edging') ||
                                  agg.operationName?.toLowerCase().includes('oklejanie');
        
        // Use itemCount from aggregation (already calculated correctly based on flowCode)
        const effectiveItemCount = agg.itemCount;
        
        // Add operation node
        nodes.push({
          id: nodeId,
          type: 'operation',
          label: agg.operationName,
          status: aggregateNodeStatus(agg.statuses),
          data: {
            operationCode: agg.operationCode,
            operationName: agg.operationName,
            workCenterName: agg.workCenterName,
            routingOperationId: agg.routingOperationId,
            zlpIds: agg.zlpIds,
            zlpNumbers: agg.zlpNumbers,
            workOrderIds: agg.workOrderIds,
            workOrderId: agg.workOrderIds.length === 1 ? agg.workOrderIds[0] : undefined,
            itemCount: effectiveItemCount,
            completedCount: agg.completedCount,
            damagedCount: agg.damagedCount,
            materialColor,
            actualStartTime: agg.actualStartTime,
            actualEndTime: agg.actualEndTime,
            operatorId: agg.operatorId,
            operatorName: agg.operatorName,
            assignedOperators: agg.assignedOperators,
            // Pallet information
            pallets: palletsForOp,
            flowCodes: Array.from(flowCodesForOp),
            isSplitPoint: isSplitOperation && flowCodesForOp.size > 1,
            palletCount: palletsForOp.length,
          },
        });
        addedNodeIds.add(nodeId);
      }
      
      // Connect from previous node of this color lane
      // For post-split ops with flowCode, use combined key; for pre-split, use color only
      const aggFlowCodeForEdge = (agg as any).flowCode as string | undefined;
      const isPostSplitForEdge = splitPointSequence !== null && seq > splitPointSequence;
      
      // Determine the lane key for edge tracking
      const laneKey = (isPostSplitForEdge && aggFlowCodeForEdge) 
        ? `${materialColor}-${aggFlowCodeForEdge}` 
        : materialColor;
      
      // For post-split nodes, check if we have a previous node for this specific lane
      // If not, fall back to the color-only key (from pre-split operations)
      let prevNodeId = prevNodeByLane.get(laneKey);
      if (!prevNodeId && isPostSplitForEdge && aggFlowCodeForEdge) {
        // Fall back to color-only key for first post-split operation in this flow
        prevNodeId = prevNodeByLane.get(materialColor);
      }
      if (!prevNodeId) {
        prevNodeId = 'start';
      }
      
      const edgeId = `${prevNodeId}-${nodeId}`;
      if (!edges.find(e => e.id === edgeId)) {
        edges.push({
          id: edgeId,
          source: prevNodeId,
          target: nodeId,
          animated: agg.statuses.has('in_progress'),
          data: {
            itemCount: agg.itemCount,
          },
        });
      }
      
      // Update the lane tracking for subsequent operations
      prevNodeByLane.set(laneKey, nodeId);
      
      // Check if this is a split operation - if so, seed lane tracking for all flowCodes
      const isSplitOpForEdge = agg.operationCode?.toLowerCase().includes('oklej') || 
                               agg.operationCode?.toLowerCase().includes('edging') ||
                               agg.operationName?.toLowerCase().includes('oklejanie');
      
      if (isSplitOpForEdge) {
        // Seed lane tracking for all flowCodes in pallets at this split operation
        // so post-split operations can connect correctly
        for (const zlpId of agg.zlpIds) {
          const zlpPallets = palletsByZlp.get(zlpId) || [];
          for (const p of zlpPallets) {
            const combinedKey = `${materialColor}-${p.flowCode}`;
            // Set this split operation as the previous node for all flowCode lanes
            prevNodeByLane.set(combinedKey, nodeId);
          }
        }
      }
      
      // Also update the color-only key so pre-split operations link correctly
      if (!isPostSplitForEdge || !aggFlowCodeForEdge) {
        prevNodeByLane.set(materialColor, nodeId);
      }
    }
    
    // If operations converge to same buffer, add merge buffer node
    if (isMergeToBuffer && opsAtSeq.length > 1) {
      const bufferName = Array.from(bufferAfterSet)[0]!;
      const bufferNodeId = `buffer-merge-${bufferName.replace(/\s+/g, '-').toLowerCase()}`;
      
      if (!addedNodeIds.has(bufferNodeId)) {
        const allColorsAtBuffer = opsAtSeq.map(o => o.agg.materialColor);
        // Calculate active colors - colors that are still in the buffer (not yet moved to next operation)
        // A color is active if its previous operation is completed but the buffer's next operations are not started
        const activeColors = allColorsAtBuffer.filter(color => {
          const colorAgg = opsAtSeq.find(o => o.agg.materialColor === color);
          if (!colorAgg) return true;
          // Color is active (still in buffer) if previous operation is done/in_progress but next ops are pending/ready
          const prevStatus = colorAgg.agg.statuses;
          // If previous operation has any in_progress or done work, the color might be in buffer
          // If it's all done and next seq has in_progress or done, it has left the buffer
          const hasLeftBuffer = prevStatus.has('done') && !prevStatus.has('in_progress') && !prevStatus.has('pending');
          // Check if there's a next sequence with this color that's already started
          const nextSeq = seq + 1;
          const nextOps = opsBySequence.get(nextSeq);
          if (nextOps) {
            const nextColorOp = nextOps.find(o => o.agg.materialColor === color);
            if (nextColorOp && (nextColorOp.agg.statuses.has('in_progress') || nextColorOp.agg.statuses.has('done'))) {
              return false; // Color has left this buffer
            }
          }
          return true; // Color is still in buffer
        });
        
        // Determine buffer status based on active operations
        const bufferHasActive = opsAtSeq.some(o => o.agg.statuses.has('in_progress'));
        const bufferStatus = bufferHasActive ? 'in_progress' : 'ready';
        
        nodes.push({
          id: bufferNodeId,
          type: 'buffer',
          label: bufferName,
          status: bufferStatus,
          data: {
            locationName: bufferName,
            materialColors: allColorsAtBuffer,
            activeColors: activeColors,
            isMergePoint: true,
          },
        });
        addedNodeIds.add(bufferNodeId);
      }
      
      // Connect all color lanes to the merge buffer
      for (const { agg } of opsAtSeq) {
        const prevOpId = prevNodeByLane.get(agg.materialColor);
        if (prevOpId) {
          const edgeId = `${prevOpId}-${bufferNodeId}`;
          if (!edges.find(e => e.id === edgeId)) {
            edges.push({
              id: edgeId,
              source: prevOpId,
              target: bufferNodeId,
              data: {},
            });
          }
        }
        // After merge, all colors continue from the buffer
        prevNodeByLane.set(agg.materialColor, bufferNodeId);
      }
    }
  }
  
  // ============================================================================
  // SYNTHETIC NODES: Warehouse Packed Products (WZ-SPAK) & Buffer Formatki
  // ============================================================================
  
  // Fetch WZ-SPAK documents for this plan (packed products from warehouse)
  const wzSpakResult = await pool.query(`
    SELECT 
      wd.id,
      wd.doc_number,
      wd.status as doc_status,
      wd.confirmed_at,
      wdl.id as line_id,
      wdl.source_type,
      wdl.source_id,
      wdl.catalog_product_id,
      wdl.sku as product_sku,
      wdl.product_name,
      wdl.quantity_requested as quantity,
      wdl.is_picked,
      wdl.picked_at,
      cp.title as catalog_product_name
    FROM warehouse.documents wd
    JOIN warehouse.document_lines wdl ON wdl.document_id = wd.id
    LEFT JOIN catalog.products cp ON cp.id = wdl.catalog_product_id
    WHERE wd.plan_id = $1 
      AND wd.doc_type = 'WZ-SPAK'
    ORDER BY wd.id, wdl.id
  `, [planId]);
  
  // Group packed products by document
  const wzSpakDocs = new Map<number, {
    id: number;
    docNumber: string;
    status: string;
    confirmedAt: string | null;
    items: Array<{
      productName: string;
      productSku: string;
      quantity: number;
      isPicked: boolean;
    }>;
  }>();
  
  for (const row of wzSpakResult.rows) {
    if (!wzSpakDocs.has(row.id)) {
      wzSpakDocs.set(row.id, {
        id: row.id,
        docNumber: row.doc_number,
        status: row.doc_status,
        confirmedAt: row.confirmed_at,
        items: [],
      });
    }
    wzSpakDocs.get(row.id)!.items.push({
      productName: row.catalog_product_name || row.product_name || row.product_sku,
      productSku: row.product_sku || '',
      quantity: parseInt(row.quantity || '1'),
      isPicked: row.is_picked || false,
    });
  }
  
  // Fetch warehouse routing rules for this plan (global + plan-specific)
  const routingRulesResult = await pool.query(`
    SELECT 
      wrr.*,
      pro.code as dest_operation_code,
      pro.name as dest_operation_name,
      pl.code as dest_location_code,
      pl.name as dest_location_name
    FROM production.warehouse_routing_rules wrr
    LEFT JOIN production.production_routing_operations pro ON pro.id = wrr.destination_operation_id
    LEFT JOIN production.production_locations pl ON pl.id = wrr.destination_location_id
    WHERE (wrr.plan_id = $1 OR wrr.plan_id IS NULL)
      AND wrr.is_active = true
    ORDER BY wrr.priority DESC, wrr.plan_id DESC NULLS LAST
  `, [planId]);
  
  const routingRules = routingRulesResult.rows;
  
  // Helper function to find destination node based on routing rules
  const findRoutingDestination = (productType: string, completionStatus: 'incomplete' | 'complete', productSku?: string) => {
    // Find matching rule (by priority)
    const matchingRule = routingRules.find(rule => {
      // Check completion status match
      if (rule.completion_status !== 'any' && rule.completion_status !== completionStatus) {
        return false;
      }
      // Check product type match
      if (rule.product_type && rule.product_type !== productType) {
        return false;
      }
      // Check SKU match
      if (rule.product_sku && productSku && rule.product_sku !== productSku) {
        return false;
      }
      return true;
    });
    
    return matchingRule || null;
  };
  
  // Create warehouse_packed node if there are WZ-SPAK documents
  if (wzSpakDocs.size > 0) {
    const allDocs = Array.from(wzSpakDocs.values());
    const totalItems = allDocs.reduce((sum, doc) => sum + doc.items.reduce((s, i) => s + i.quantity, 0), 0);
    const allPicked = allDocs.every(doc => doc.items.every(i => i.isPicked));
    const anyPicked = allDocs.some(doc => doc.items.some(i => i.isPicked));
    
    const packedStatus: FlowNodeStatus = allPicked ? 'completed' : (anyPicked ? 'in_progress' : 'ready');
    
    const warehousePackedNode: FlowNode = {
      id: 'warehouse-packed',
      type: 'warehouse_packed',
      label: 'Magazyn produktów spakowanych',
      status: packedStatus,
      data: {
        warehouseDocumentId: allDocs[0]?.id,
        warehouseDocumentNumber: allDocs.map(d => d.docNumber).join(', '),
        packedProductsCount: totalItems,
        packedProductsItems: allDocs.flatMap(d => d.items),
      },
    };
    nodes.push(warehousePackedNode);
    addedNodeIds.add('warehouse-packed');
    
    // Add incoming edge from start to warehouse-packed
    edges.push({
      id: 'start-to-warehouse-packed',
      source: 'start',
      target: 'warehouse-packed',
      label: 'Pobierz z magazynu',
      style: {
        stroke: '#10b981', // Green for packed products
        strokeWidth: 2,
      },
    });
    
    // Check for routing rules for packed products from warehouse
    // First try 'complete' status (fully packed products ready for shipping)
    // Then try 'incomplete' status (products requiring additional packaging/assembly)
    // Products from warehouse are typically complete - they go to packing only if they need additional work
    let routingRule = findRoutingDestination('packed_product', 'complete');
    if (!routingRule) {
      routingRule = findRoutingDestination('packed_product', 'incomplete');
    }
    
    let destinationNodeId: string | null = null;
    let destinationLabel = `${totalItems} szt spakowanych`;
    
    if (routingRule) {
      // Use routing rule destination
      if (routingRule.destination === 'packing') {
        // Find packing buffer or operation
        const packingBuffer = nodes.find(n =>
          n.type === 'buffer' &&
          (n.label?.toLowerCase().includes('pakow') || n.data.locationName?.toLowerCase().includes('pakow'))
        );
        const packingNode = nodes.find(n => 
          n.type === 'operation' && 
          (n.data.operationCode?.toLowerCase().includes('pakow') || 
           n.label?.toLowerCase().includes('pakow'))
        );
        destinationNodeId = packingBuffer?.id || packingNode?.id || null;
        destinationLabel = `${totalItems} szt → pakowanie`;
      } else if (routingRule.destination === 'shipping_buffer') {
        // Go directly to end (shipping)
        destinationNodeId = 'end';
        destinationLabel = `${totalItems} szt → wysyłka`;
      } else if (routingRule.destination === 'assembly') {
        // Find assembly operation
        const assemblyNode = nodes.find(n => 
          n.type === 'operation' && 
          (n.data.operationCode?.toLowerCase().includes('mont') || 
           n.label?.toLowerCase().includes('mont') ||
           n.label?.toLowerCase().includes('komplet'))
        );
        destinationNodeId = assemblyNode?.id || null;
        destinationLabel = `${totalItems} szt → montaż`;
      } else if (routingRule.destination.startsWith('operation:')) {
        const opId = parseInt(routingRule.destination.replace('operation:', ''));
        const opNode = nodes.find(n => 
          n.type === 'operation' && n.data.routingOperationId === opId
        );
        destinationNodeId = opNode?.id || null;
        destinationLabel = `${totalItems} szt → ${routingRule.dest_operation_name || 'operacja'}`;
      } else if (routingRule.destination.startsWith('location:')) {
        const locId = parseInt(routingRule.destination.replace('location:', ''));
        const bufferNode = nodes.find(n => 
          n.type === 'buffer' && n.data.locationId === locId
        );
        destinationNodeId = bufferNode?.id || null;
        destinationLabel = `${totalItems} szt → ${routingRule.dest_location_name || 'bufor'}`;
      }
    }
    
    // If no routing rule matched or destination not found, use default logic
    if (!destinationNodeId) {
      // Find packing operation or buffer before packing to connect to
      const packingNode = nodes.find(n => 
        n.type === 'operation' && 
        (n.data.operationCode?.toLowerCase().includes('pakow') || 
         n.label?.toLowerCase().includes('pakow'))
      );
      
      // Find buffer before packing (bufor pakowania)
      const packingBuffer = nodes.find(n =>
        n.type === 'buffer' &&
        (n.label?.toLowerCase().includes('pakow') || n.data.locationName?.toLowerCase().includes('pakow'))
      );
      
      destinationNodeId = packingBuffer?.id || packingNode?.id || 'end';
    }
    
    // Add edge to destination
    edges.push({
      id: `warehouse-packed-to-${destinationNodeId}`,
      source: 'warehouse-packed',
      target: destinationNodeId,
      label: destinationLabel,
      style: {
        stroke: '#10b981',
        strokeWidth: 2,
      },
      data: {
        itemCount: totalItems,
      },
    });
  }
  
  // Fetch buffer formatki reservations for this plan
  const bufferFormatkiResult = await pool.query(`
    SELECT 
      pbr.id,
      pbr.product_sku,
      pbr.quantity_reserved,
      pbr.quantity_consumed,
      pbr.status,
      pbr.zlp_item_id,
      pbs.quantity_available,
      wm.name as material_name,
      wm.internal_code
    FROM production.production_buffer_reservations pbr
    LEFT JOIN production.production_buffer_stock pbs ON pbs.product_sku = pbr.product_sku
    LEFT JOIN warehouse.materials wm ON wm.internal_code = pbr.product_sku
    WHERE pbr.zlp_item_id IN (
      SELECT id FROM production.production_plan_lines WHERE plan_id = $1 AND deleted_at IS NULL
    )
    ORDER BY pbr.product_sku
  `, [planId]);
  
  // Fetch WZ-FORM documents for this plan (formatki from buffer)
  const wzFormResult = await pool.query(`
    SELECT 
      wd.id,
      wd.doc_number,
      wd.status as doc_status,
      wd.confirmed_at,
      wd.total_lines,
      wd.total_quantity
    FROM warehouse.documents wd
    WHERE wd.plan_id = $1 
      AND wd.doc_type = 'WZ-FORM'
    ORDER BY wd.id
  `, [planId]);
  
  // Create buffer_formatki node if there are reservations OR WZ-FORM documents
  if (bufferFormatkiResult.rows.length > 0 || wzFormResult.rows.length > 0) {
    const reservations = bufferFormatkiResult.rows;
    const totalReserved = reservations.reduce((sum, r) => sum + parseFloat(r.quantity_reserved || '0'), 0);
    const totalConsumed = reservations.reduce((sum, r) => sum + parseFloat(r.quantity_consumed || '0'), 0);
    const allConsumed = reservations.every(r => r.status === 'CONSUMED');
    const anyConsumed = reservations.some(r => r.status === 'CONSUMED' || parseFloat(r.quantity_consumed || '0') > 0);
    
    // Check WZ-FORM document status
    const wzFormDocs = wzFormResult.rows;
    const hasWzForm = wzFormDocs.length > 0;
    const wzFormConfirmed = wzFormDocs.some(d => d.doc_status === 'confirmed' || d.doc_status === 'completed');
    
    const formatkiStatus: FlowNodeStatus = allConsumed ? 'completed' : (anyConsumed || wzFormConfirmed ? 'in_progress' : 'ready');
    
    const bufferFormatkiNode: FlowNode = {
      id: 'buffer-formatki',
      type: 'buffer_formatki',
      label: 'Bufor produkcji (formatki)',
      status: formatkiStatus,
      data: {
        reservationsCount: reservations.length,
        reservedQuantity: totalReserved,
        consumedQuantity: totalConsumed,
        reservationItems: reservations.map(r => ({
          productSku: r.product_sku,
          productName: r.material_name || r.internal_code || r.product_sku,
          quantityReserved: parseFloat(r.quantity_reserved || '0'),
          quantityConsumed: parseFloat(r.quantity_consumed || '0'),
          status: r.status,
        })),
        warehouseDocumentId: hasWzForm ? wzFormDocs[0].id : undefined,
        warehouseDocumentNumber: hasWzForm ? wzFormDocs.map(d => d.doc_number).join(', ') : undefined,
        warehouseDocumentStatus: hasWzForm ? wzFormDocs[0].doc_status : undefined,
      },
    };
    nodes.push(bufferFormatkiNode);
    addedNodeIds.add('buffer-formatki');
    
    // Always add incoming edge from start to buffer-formatki
    edges.push({
      id: 'start-to-buffer-formatki',
      source: 'start',
      target: 'buffer-formatki',
      label: 'Pobierz z bufora',
      style: {
        stroke: '#f59e0b', // Amber for formatki
        strokeWidth: 2,
      },
    });
    
    // Find first assembly/drilling operation or cutting operation
    const assemblyNode = nodes.find(n => 
      n.type === 'operation' && 
      (n.data.operationCode?.toLowerCase().includes('mont') || 
       n.data.operationCode?.toLowerCase().includes('wiert') ||
       n.data.operationCode?.toLowerCase().includes('komplet') ||
       n.label?.toLowerCase().includes('mont') ||
       n.label?.toLowerCase().includes('komplet'))
    );
    
    // Find buffer before montaż or kompletowanie
    const assemblyBuffer = nodes.find(n =>
      n.type === 'buffer' &&
      (n.label?.toLowerCase().includes('mont') || 
       n.label?.toLowerCase().includes('komplet') ||
       n.data.locationName?.toLowerCase().includes('mont') ||
       n.data.locationName?.toLowerCase().includes('komplet'))
    );
    
    // Connect to assembly buffer, assembly operation, or end
    if (assemblyBuffer) {
      edges.push({
        id: 'buffer-formatki-to-assembly-buffer',
        source: 'buffer-formatki',
        target: assemblyBuffer.id,
        label: `${Math.round(totalReserved)} szt formatek`,
        style: {
          stroke: '#f59e0b',
          strokeWidth: 2,
        },
        data: {
          itemCount: Math.round(totalReserved),
        },
      });
    } else if (assemblyNode) {
      edges.push({
        id: 'buffer-formatki-to-assembly',
        source: 'buffer-formatki',
        target: assemblyNode.id,
        label: `${Math.round(totalReserved)} szt formatek`,
        style: {
          stroke: '#f59e0b',
          strokeWidth: 2,
        },
        data: {
          itemCount: Math.round(totalReserved),
        },
      });
    } else {
      // Connect directly to end if no assembly operation found
      edges.push({
        id: 'buffer-formatki-to-end',
        source: 'buffer-formatki',
        target: 'end',
        label: `${Math.round(totalReserved)} szt formatek`,
        style: {
          stroke: '#f59e0b',
          strokeWidth: 2,
        },
        data: {
          itemCount: Math.round(totalReserved),
        },
      });
    }
  }
  
  // ============================================================================
  // SHIPPING BUFFER: Add shipping buffer node before end if terminal operations have buffer_after
  // ============================================================================
  
  // Collect terminal nodes (last node of each lane) from prevNodeByLane
  const terminalNodeIds = new Set<string>();
  for (const [, nodeId] of Array.from(prevNodeByLane.entries())) {
    if (nodeId !== 'start') {
      terminalNodeIds.add(nodeId);
    }
  }
  
  // Find buffer_after names from all terminal operations (not just last sequence)
  // This ensures we catch shipping buffer from any lane's final operation
  const shippingBufferNames = new Set<string>();
  const terminalOperationStatuses: Array<{ nodeId: string; statuses: Set<string>; color: string }> = [];
  
  for (const seq of sortedSequences) {
    const opsAtSeq = opsBySequence.get(seq)!;
    for (const { key, agg } of opsAtSeq) {
      const nodeId = `op-${key}`;
      // Check if this operation is a terminal node (last in its lane)
      if (terminalNodeIds.has(nodeId) && agg.bufferAfterName) {
        shippingBufferNames.add(agg.bufferAfterName);
        terminalOperationStatuses.push({
          nodeId,
          statuses: agg.statuses,
          color: agg.materialColor,
        });
      }
    }
  }
  
  // If there's a shipping buffer (like "Bufor wysyłkowy"), create a node for it
  // But first check if a merge buffer with the same name already exists
  let shippingBufferNodeId: string | null = null;
  if (shippingBufferNames.size > 0) {
    // Use the first one (typically "Bufor wysyłkowy")
    const shippingBufferName = Array.from(shippingBufferNames)[0]!;
    const mergeBufferNodeId = `buffer-merge-${shippingBufferName.replace(/\s+/g, '-').toLowerCase()}`;
    
    // Check if merge buffer with same name already exists - if so, reuse it as shipping buffer
    if (addedNodeIds.has(mergeBufferNodeId)) {
      shippingBufferNodeId = mergeBufferNodeId;
      // Update the existing merge buffer node to also be a shipping buffer
      const existingNode = nodes.find(n => n.id === mergeBufferNodeId);
      if (existingNode) {
        existingNode.data.isShippingBuffer = true;
      }
    } else {
      // Create new shipping buffer node
      shippingBufferNodeId = `buffer-shipping-${shippingBufferName.replace(/\s+/g, '-').toLowerCase()}`;
      
      if (!addedNodeIds.has(shippingBufferNodeId)) {
        // Calculate colors from all terminal operations that have this buffer
        const colorsInShippingBuffer = terminalOperationStatuses.map(t => t.color);
        
        // Determine buffer status based on all terminal operations across all lanes
        const allTerminalsDone = terminalOperationStatuses.every(t => 
          t.statuses.has('done') && !t.statuses.has('in_progress') && !t.statuses.has('pending')
        );
        const anyTerminalInProgress = terminalOperationStatuses.some(t => t.statuses.has('in_progress'));
        
        const shippingBufferStatus = allTerminalsDone ? 'completed' : (anyTerminalInProgress ? 'in_progress' : 'ready');
        
        nodes.push({
          id: shippingBufferNodeId,
          type: 'buffer',
          label: shippingBufferName,
          status: shippingBufferStatus,
          data: {
            locationName: shippingBufferName,
            materialColors: colorsInShippingBuffer,
            activeColors: colorsInShippingBuffer,
            isShippingBuffer: true,
          },
        });
        addedNodeIds.add(shippingBufferNodeId);
      }
    }
  }
  
  // End node
  nodes.push({
    id: 'end',
    type: 'end',
    label: 'Zakończenie',
    status: zlpsResult.rows.every(r => r.status === 'done') ? 'completed' : 'pending',
    data: {},
  });
  addedNodeIds.add('end');
  
  // Connect last nodes of each color lane to shipping buffer or directly to end
  const lastNodesConnectedToEnd = new Set<string>();
  for (const [, lastNodeId] of Array.from(prevNodeByLane.entries())) {
    if (!lastNodesConnectedToEnd.has(lastNodeId) && lastNodeId !== 'start') {
      // If shipping buffer exists, connect to it; otherwise connect directly to end
      const targetNodeId = shippingBufferNodeId || 'end';
      const edgeId = `${lastNodeId}-${targetNodeId}`;
      if (!edges.find(e => e.id === edgeId)) {
        edges.push({
          id: edgeId,
          source: lastNodeId,
          target: targetNodeId,
        });
      }
      lastNodesConnectedToEnd.add(lastNodeId);
    }
  }
  
  // Connect shipping buffer to end
  if (shippingBufferNodeId) {
    const bufferToEndEdgeId = `${shippingBufferNodeId}-end`;
    if (!edges.find(e => e.id === bufferToEndEdgeId)) {
      edges.push({
        id: bufferToEndEdgeId,
        source: shippingBufferNodeId,
        target: 'end',
      });
    }
  }
  
  // Safety net: Ensure ALL operation nodes have outgoing edges
  // This catches any operation nodes that weren't tracked in prevNodeByLane
  const nodesWithOutgoingEdges = new Set(edges.map(e => e.source));
  for (const node of nodes) {
    if (node.type === 'operation' && !nodesWithOutgoingEdges.has(node.id)) {
      // Connect to shipping buffer if it exists, otherwise to end
      // Always ensure we connect to 'end' if shipping buffer is null
      const targetNodeId = shippingBufferNodeId ? shippingBufferNodeId : 'end';
      edges.push({
        id: `${node.id}-${targetNodeId}`,
        source: node.id,
        target: targetNodeId,
      });
    }
  }
  
  // Fetch merge points from database with routing operation info
  const mergePointsResult = await pool.query(`
    SELECT 
      mp.id,
      mp.merge_stage,
      mp.required_families,
      mp.completion_policy,
      mp.manual_release,
      mp.arrived_families,
      mp.family_progress,
      mp.status,
      mp.routing_operation_id,
      ro.code as routing_operation_code,
      ro.name as routing_operation_name
    FROM production.production_plan_merge_points mp
    LEFT JOIN production.production_routing_operations ro ON ro.id = mp.routing_operation_id
    WHERE mp.plan_id = $1
    ORDER BY mp.merge_stage
  `, [planId]);
  
  // Process merge points and add to appropriate nodes
  const mergePoints: MergePointProgress[] = mergePointsResult.rows.map(mp => ({
    mergePointId: mp.id,
    mergeStage: mp.merge_stage,
    requiredFamilies: mp.required_families || [],
    arrivedFamilies: mp.arrived_families || [],
    familyProgress: mp.family_progress || {},
    status: mp.status || 'waiting',
    completionPolicy: mp.completion_policy,
    manualRelease: mp.manual_release || false,
    routingOperationId: mp.routing_operation_id,
    routingOperationCode: mp.routing_operation_code,
  }));
  
  // Enhance nodes with merge point data - prioritize routing_operation_id match
  for (const node of nodes) {
    if (node.data.isMergePoint || node.type === 'buffer') {
      // Find matching merge point: priority to routing_operation_id, then fallback to label
      const matchingMergePoint = mergePoints.find(mp => {
        // Primary: If merge point has routing_operation_id configured and node has one too
        if (mp.routingOperationId != null && node.data.routingOperationId != null) {
          return mp.routingOperationId === node.data.routingOperationId;
        }
        // Skip if merge point requires specific operation but node doesn't have one
        if (mp.routingOperationId != null && node.data.routingOperationId == null) {
          return false;
        }
        // Fallback: label-based matching when merge point has no specific operation
        const nodeName = (node.label || '').toLowerCase();
        const mergeStage = (mp.mergeStage || '').toLowerCase();
        const stageMatch = nodeName.includes(mergeStage) ||
                          (mergeStage.includes('kompletowanie') && nodeName.includes('komplet')) ||
                          (mergeStage.includes('pakowanie') && nodeName.includes('pakow'));
        return stageMatch;
      });
      
      if (matchingMergePoint) {
        node.type = 'merge';
        node.data.isMergePoint = true;
        node.data.mergePointData = matchingMergePoint;
      }
    }
  }
  
  // Calculate totals - collect all unique material colors
  const allColors = new Set<string>();
  Array.from(operationAggregations.values()).forEach(agg => {
    allColors.add(agg.materialColor);
  });
  
  const totalItems = Array.from(zlpItemCount.values()).reduce((a, b) => a + b, 0);
  const damagedItems = Array.from(zlpDamagedCount.values()).reduce((a, b) => a + b, 0);
  const completedItems = Array.from(operationAggregations.values())
    .reduce((sum, agg) => sum + agg.completedCount, 0);
  
  return {
    planId,
    planName: plan.name,
    nodes,
    edges,
    mergePoints,
    metadata: {
      totalZlps: zlpsResult.rows.length,
      totalItems,
      completedItems,
      damagedItems,
      materialColors: Array.from(allColors),
      generatedAt: new Date().toISOString(),
    },
  };
}

/**
 * Get BOM items (formatki/components) for a specific operation in the flow tree
 */
export async function getOperationItems(
  pool: Pool,
  planId: number,
  zlpIds: number[],
  materialColor?: string,
  operationCode?: string
): Promise<any[]> {
  if (zlpIds.length === 0) {
    return [];
  }

  // Fetch BOM items for the given ZLPs
  const result = await pool.query(`
    SELECT 
      bi.id,
      bi.component_name,
      bi.component_type,
      bi.quantity,
      bi.unit_of_measure,
      bi.color_code,
      bi.length,
      bi.width,
      bi.thickness,
      bi.is_damaged,
      bi.damage_type,
      bi.damage_notes,
      bi.item_status,
      bi.quantity_ordered,
      bi.quantity_produced,
      bi.quantity_damaged,
      bi.quantity_scrapped,
      bi.notes,
      bi.source_furniture_reference,
      bob.production_order_id as zlp_id,
      po.order_number as zlp_number
    FROM production.production_order_bom_items bi
    JOIN production.production_order_boms bob ON bob.id = bi.production_order_bom_id
    JOIN production.production_orders po ON po.id = bob.production_order_id
    WHERE bob.production_order_id = ANY($1)
    ${materialColor ? `AND (bi.color_code = $2 OR bi.color_code ILIKE $2 || '%')` : ''}
    ORDER BY bi.color_code, bi.component_name, bi.length DESC, bi.width DESC
  `, materialColor ? [zlpIds, materialColor] : [zlpIds]);

  return result.rows.map((row: any) => ({
    id: row.id,
    componentName: row.component_name,
    componentType: row.component_type,
    quantity: parseFloat(row.quantity) || 1,
    unitOfMeasure: row.unit_of_measure,
    colorCode: row.color_code,
    length: row.length ? parseFloat(row.length) : null,
    width: row.width ? parseFloat(row.width) : null,
    thickness: row.thickness ? parseFloat(row.thickness) : null,
    isDamaged: row.is_damaged,
    damageType: row.damage_type,
    damageNotes: row.damage_notes,
    itemStatus: row.item_status,
    quantityOrdered: row.quantity_ordered ? parseFloat(row.quantity_ordered) : null,
    quantityProduced: row.quantity_produced ? parseFloat(row.quantity_produced) : 0,
    quantityDamaged: row.quantity_damaged ? parseFloat(row.quantity_damaged) : 0,
    quantityScrapped: row.quantity_scrapped ? parseFloat(row.quantity_scrapped) : 0,
    notes: row.notes,
    sourceFurnitureReference: row.source_furniture_reference,
    zlpId: row.zlp_id,
    zlpNumber: row.zlp_number,
  }));
}
