import type { Pool } from "pg";
import type { ProductionLocation, InsertProductionLocation } from "@shared/schema";

export async function getLocations(pool: Pool, parentId?: number | null): Promise<ProductionLocation[]> {
  const query = parentId !== undefined
    ? `SELECT * FROM production.production_locations WHERE parent_id ${parentId === null ? 'IS NULL' : '= $1'} ORDER BY code ASC`
    : `SELECT * FROM production.production_locations ORDER BY code ASC`;
  
  const result = parentId === null || parentId === undefined
    ? await pool.query(query)
    : await pool.query(query, [parentId]);
  
  return result.rows.map(row => ({
    id: row.id,
    locationGroupId: row.location_group_id,
    parentId: row.parent_id,
    code: row.code,
    name: row.name,
    path: row.path,
    level: row.level,
    barcode: row.barcode,
    capacity: row.capacity,
    capacityUnit: row.capacity_unit,
    currentLoad: row.current_load,
    dimensions: row.dimensions,
    status: row.status,
    allowsStorage: row.allows_storage,
    isActive: row.is_active,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getLocationById(pool: Pool, id: number): Promise<ProductionLocation | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_locations WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    locationGroupId: row.location_group_id,
    parentId: row.parent_id,
    code: row.code,
    name: row.name,
    path: row.path,
    level: row.level,
    barcode: row.barcode,
    capacity: row.capacity,
    capacityUnit: row.capacity_unit,
    currentLoad: row.current_load,
    dimensions: row.dimensions,
    status: row.status,
    allowsStorage: row.allows_storage,
    isActive: row.is_active,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createLocation(pool: Pool, data: InsertProductionLocation): Promise<ProductionLocation> {
  // Normalize parentId: 0 should be treated as null (root level)
  if (data.parentId !== undefined && data.parentId === 0) {
    data.parentId = null;
  }

  // Calculate path and level based on parent
  let path = `/${data.code}`;
  let level = 0;
  
  if (data.parentId) {
    const parent = await getLocationById(pool, data.parentId);
    if (parent) {
      path = `${parent.path}/${data.code}`;
      level = (parent.level || 0) + 1;
    }
  }

  const result = await pool.query(`
    INSERT INTO production.production_locations 
    (location_group_id, parent_id, code, name, path, level, barcode, capacity, capacity_unit, 
     current_load, dimensions, status, allows_storage, is_active, notes)
    VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15)
    RETURNING *
  `, [
    data.locationGroupId,
    data.parentId || null,
    data.code,
    data.name,
    path,
    level,
    data.barcode || null,
    data.capacity || null,
    data.capacityUnit || null,
    data.currentLoad || 0,
    data.dimensions ? JSON.stringify(data.dimensions) : null,
    data.status || 'active',
    data.allowsStorage ?? true,
    data.isActive ?? true,
    data.notes || null,
  ]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    locationGroupId: row.location_group_id,
    parentId: row.parent_id,
    code: row.code,
    name: row.name,
    path: row.path,
    level: row.level,
    barcode: row.barcode,
    capacity: row.capacity,
    capacityUnit: row.capacity_unit,
    currentLoad: row.current_load,
    dimensions: row.dimensions,
    status: row.status,
    allowsStorage: row.allows_storage,
    isActive: row.is_active,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

// Helper function to update descendant paths when a node's hierarchy changes
async function updateDescendantPaths(pool: Pool, parentId: number): Promise<void> {
  // Get the parent location to use its path
  const parent = await getLocationById(pool, parentId);
  if (!parent) return;

  // Get all direct children
  const children = await pool.query(
    `SELECT id, code FROM production.production_locations WHERE parent_id = $1`,
    [parentId]
  );

  // Update each child's path and level, then recursively update their children
  for (const child of children.rows) {
    const newPath = `${parent.path}/${child.code}`;
    const newLevel = (parent.level || 0) + 1;

    await pool.query(
      `UPDATE production.production_locations SET path = $1, level = $2, updated_at = NOW() WHERE id = $3`,
      [newPath, newLevel, child.id]
    );

    // Recursively update this child's descendants
    await updateDescendantPaths(pool, child.id);
  }
}

export async function updateLocation(pool: Pool, id: number, data: Partial<InsertProductionLocation>): Promise<ProductionLocation | null> {
  // Get current location first to calculate hierarchy
  const currentLocation = await getLocationById(pool, id);
  if (!currentLocation) return null;

  // Normalize parentId: 0 should be treated as null (root level)
  if (data.parentId !== undefined && data.parentId === 0) {
    data.parentId = null;
  }

  // Validate parentId to prevent cycles
  if (data.parentId !== undefined && data.parentId !== null) {
    // Cannot be its own parent
    if (data.parentId === id) {
      throw new Error("Location cannot be its own parent");
    }

    // Cannot move to one of its own descendants (would create cycle)
    const descendants = await getLocationChildren(pool, id);
    const descendantIds = descendants.map(d => d.id);
    if (descendantIds.includes(data.parentId)) {
      throw new Error("Location cannot be moved to one of its descendants");
    }
  }

  const updates: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  // Determine if we need to recalculate path/level
  const parentChanged = data.parentId !== undefined;
  const codeChanged = data.code !== undefined;
  const needsHierarchyUpdate = parentChanged || codeChanged;

  if (data.locationGroupId !== undefined) {
    updates.push(`location_group_id = $${paramIndex++}`);
    values.push(data.locationGroupId);
  }
  if (data.parentId !== undefined) {
    updates.push(`parent_id = $${paramIndex++}`);
    values.push(data.parentId);
  }
  if (data.code !== undefined) {
    updates.push(`code = $${paramIndex++}`);
    values.push(data.code);
  }

  // Recalculate path and level if parent or code changed
  if (needsHierarchyUpdate) {
    const newParentId = data.parentId !== undefined ? data.parentId : currentLocation.parentId;
    const newCode = data.code !== undefined ? data.code : currentLocation.code;
    
    let newPath: string;
    let newLevel: number;
    
    if (newParentId === null || newParentId === undefined) {
      // Moving to root or already at root
      newPath = `/${newCode}`;
      newLevel = 0;
    } else {
      // Has a parent - fetch parent to build path
      const parent = await getLocationById(pool, newParentId);
      if (parent) {
        newPath = `${parent.path}/${newCode}`;
        newLevel = (parent.level || 0) + 1;
      } else {
        // Parent not found, treat as root
        newPath = `/${newCode}`;
        newLevel = 0;
      }
    }
    
    updates.push(`path = $${paramIndex++}`);
    values.push(newPath);
    updates.push(`level = $${paramIndex++}`);
    values.push(newLevel);
  }
  if (data.name !== undefined) {
    updates.push(`name = $${paramIndex++}`);
    values.push(data.name);
  }
  if (data.barcode !== undefined) {
    updates.push(`barcode = $${paramIndex++}`);
    values.push(data.barcode);
  }
  if (data.capacity !== undefined) {
    updates.push(`capacity = $${paramIndex++}`);
    values.push(data.capacity);
  }
  if (data.capacityUnit !== undefined) {
    updates.push(`capacity_unit = $${paramIndex++}`);
    values.push(data.capacityUnit);
  }
  if (data.currentLoad !== undefined) {
    updates.push(`current_load = $${paramIndex++}`);
    values.push(data.currentLoad);
  }
  if (data.dimensions !== undefined) {
    updates.push(`dimensions = $${paramIndex++}`);
    values.push(data.dimensions ? JSON.stringify(data.dimensions) : null);
  }
  if (data.status !== undefined) {
    updates.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }
  if (data.allowsStorage !== undefined) {
    updates.push(`allows_storage = $${paramIndex++}`);
    values.push(data.allowsStorage);
  }
  if (data.isActive !== undefined) {
    updates.push(`is_active = $${paramIndex++}`);
    values.push(data.isActive);
  }
  if (data.notes !== undefined) {
    updates.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (updates.length === 0) return getLocationById(pool, id);

  updates.push(`updated_at = NOW()`);
  values.push(id);

  const result = await pool.query(`
    UPDATE production.production_locations
    SET ${updates.join(', ')}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);

  if (result.rows.length === 0) return null;

  // If hierarchy changed, cascade update to all descendants
  if (needsHierarchyUpdate) {
    await updateDescendantPaths(pool, id);
  }

  const row = result.rows[0];
  return {
    id: row.id,
    locationGroupId: row.location_group_id,
    parentId: row.parent_id,
    code: row.code,
    name: row.name,
    path: row.path,
    level: row.level,
    barcode: row.barcode,
    capacity: row.capacity,
    capacityUnit: row.capacity_unit,
    currentLoad: row.current_load,
    dimensions: row.dimensions,
    status: row.status,
    allowsStorage: row.allows_storage,
    isActive: row.is_active,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function deleteLocation(pool: Pool, id: number): Promise<boolean> {
  const result = await pool.query(`
    DELETE FROM production.production_locations WHERE id = $1
  `, [id]);
  
  return result.rowCount !== null && result.rowCount > 0;
}

// Helper: Get location hierarchy (breadcrumb path)
export async function getLocationPath(pool: Pool, id: number): Promise<ProductionLocation[]> {
  const result = await pool.query(`
    WITH RECURSIVE location_path AS (
      SELECT * FROM production.production_locations WHERE id = $1
      UNION ALL
      SELECT l.* FROM production.production_locations l
      INNER JOIN location_path lp ON l.id = lp.parent_id
    )
    SELECT * FROM location_path ORDER BY level ASC
  `, [id]);
  
  return result.rows.map(row => ({
    id: row.id,
    locationGroupId: row.location_group_id,
    parentId: row.parent_id,
    code: row.code,
    name: row.name,
    path: row.path,
    level: row.level,
    barcode: row.barcode,
    capacity: row.capacity,
    capacityUnit: row.capacity_unit,
    currentLoad: row.current_load,
    dimensions: row.dimensions,
    status: row.status,
    allowsStorage: row.allows_storage,
    isActive: row.is_active,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

// Helper: Get all child locations (tree)
export async function getLocationChildren(pool: Pool, id: number): Promise<ProductionLocation[]> {
  const result = await pool.query(`
    WITH RECURSIVE location_tree AS (
      SELECT * FROM production.production_locations WHERE id = $1
      UNION ALL
      SELECT l.* FROM production.production_locations l
      INNER JOIN location_tree lt ON l.parent_id = lt.id
    )
    SELECT * FROM location_tree WHERE id != $1 ORDER BY path ASC
  `, [id]);
  
  return result.rows.map(row => ({
    id: row.id,
    locationGroupId: row.location_group_id,
    parentId: row.parent_id,
    code: row.code,
    name: row.name,
    path: row.path,
    level: row.level,
    barcode: row.barcode,
    capacity: row.capacity,
    capacityUnit: row.capacity_unit,
    currentLoad: row.current_load,
    dimensions: row.dimensions,
    status: row.status,
    allowsStorage: row.allows_storage,
    isActive: row.is_active,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}
