import { Pool, PoolClient } from "pg";
import type { ProductionBufferStock, InsertProductionBufferStock } from "@shared/schema";

type PoolOrClient = Pool | PoolClient;

export async function getBufferStock(pool: Pool): Promise<ProductionBufferStock[]> {
  const result = await pool.query(`
    SELECT 
      bs.*,
      l.name as location_name,
      l.code as location_code
    FROM production.production_buffer_stock bs
    LEFT JOIN production.production_locations l ON bs.location_id = l.id
    ORDER BY bs.product_sku ASC
  `);
  
  return result.rows.map(row => ({
    id: row.id,
    productSku: row.product_sku,
    productName: row.product_name,
    locationId: row.location_id,
    quantityAvailable: row.quantity_available,
    quantityReserved: row.quantity_reserved,
    quantityTotal: row.quantity_total,
    unitOfMeasure: row.unit_of_measure,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getBufferStockBySku(pool: Pool, productSku: string): Promise<ProductionBufferStock[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_buffer_stock 
    WHERE product_sku = $1
    ORDER BY location_id ASC
  `, [productSku]);
  
  return result.rows.map(row => ({
    id: row.id,
    productSku: row.product_sku,
    productName: row.product_name,
    locationId: row.location_id,
    quantityAvailable: row.quantity_available,
    quantityReserved: row.quantity_reserved,
    quantityTotal: row.quantity_total,
    unitOfMeasure: row.unit_of_measure,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getBufferStockByLocation(pool: Pool, locationId: number): Promise<ProductionBufferStock[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_buffer_stock 
    WHERE location_id = $1
    ORDER BY product_sku ASC
  `, [locationId]);
  
  return result.rows.map(row => ({
    id: row.id,
    productSku: row.product_sku,
    productName: row.product_name,
    locationId: row.location_id,
    quantityAvailable: row.quantity_available,
    quantityReserved: row.quantity_reserved,
    quantityTotal: row.quantity_total,
    unitOfMeasure: row.unit_of_measure,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getOrCreateBufferStock(
  pool: PoolOrClient, 
  productSku: string, 
  locationId: number | null,
  productName?: string
): Promise<ProductionBufferStock> {
  const existing = await pool.query(`
    SELECT * FROM production.production_buffer_stock 
    WHERE product_sku = $1 AND location_id IS NOT DISTINCT FROM $2
  `, [productSku, locationId]);
  
  if (existing.rows.length > 0) {
    const row = existing.rows[0];
    return {
      id: row.id,
      productSku: row.product_sku,
      productName: row.product_name,
      locationId: row.location_id,
      quantityAvailable: row.quantity_available,
      quantityReserved: row.quantity_reserved,
      quantityTotal: row.quantity_total,
      unitOfMeasure: row.unit_of_measure,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }
  
  const result = await pool.query(`
    INSERT INTO production.production_buffer_stock 
    (product_sku, product_name, location_id, quantity_available, quantity_reserved, quantity_total)
    VALUES ($1, $2, $3, 0, 0, 0)
    RETURNING *
  `, [productSku, productName || null, locationId]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    productSku: row.product_sku,
    productName: row.product_name,
    locationId: row.location_id,
    quantityAvailable: row.quantity_available,
    quantityReserved: row.quantity_reserved,
    quantityTotal: row.quantity_total,
    unitOfMeasure: row.unit_of_measure,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function updateBufferQuantities(
  pool: PoolOrClient,
  productSku: string,
  locationId: number | null,
  availableDelta: number,
  reservedDelta: number
): Promise<ProductionBufferStock> {
  const result = await pool.query(`
    UPDATE production.production_buffer_stock 
    SET 
      quantity_available = quantity_available + $3,
      quantity_reserved = quantity_reserved + $4,
      quantity_total = quantity_total + $3 + $4,
      updated_at = NOW()
    WHERE product_sku = $1 AND location_id IS NOT DISTINCT FROM $2
    RETURNING *
  `, [productSku, locationId, availableDelta, reservedDelta]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    productSku: row.product_sku,
    productName: row.product_name,
    locationId: row.location_id,
    quantityAvailable: row.quantity_available,
    quantityReserved: row.quantity_reserved,
    quantityTotal: row.quantity_total,
    unitOfMeasure: row.unit_of_measure,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}
