import { Router } from "express";
import { pool } from "../../postgres";
import { 
  getAllTemplates, 
  getTemplateById, 
  createTemplate, 
  updateTemplate, 
  deleteTemplate,
  generatePlanName 
} from "../../services/production/plan-name-templates";
import { insertProductionPlanNameTemplateSchema } from "@shared/schema";
import { z } from "zod";

const router = Router();

router.get("/", async (req, res) => {
  try {
    const templates = await getAllTemplates(pool);
    res.json(templates);
  } catch (error: any) {
    console.error("Failed to fetch plan name templates:", error);
    res.status(500).json({ message: "Failed to fetch plan name templates" });
  }
});

router.get("/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    if (isNaN(id)) {
      return res.status(400).json({ message: "Invalid template ID" });
    }

    const template = await getTemplateById(pool, id);
    if (!template) {
      return res.status(404).json({ message: "Template not found" });
    }

    res.json(template);
  } catch (error: any) {
    console.error("Failed to fetch template:", error);
    res.status(500).json({ message: "Failed to fetch template" });
  }
});

router.post("/", async (req, res) => {
  try {
    const validatedData = insertProductionPlanNameTemplateSchema.parse(req.body);
    const template = await createTemplate(pool, validatedData);
    res.status(201).json(template);
  } catch (error: any) {
    if (error instanceof z.ZodError) {
      return res.status(400).json({ message: "Validation error", errors: error.errors });
    }
    
    console.error("Failed to create template:", error);
    res.status(500).json({ message: error.message || "Failed to create template" });
  }
});

router.patch("/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    if (isNaN(id)) {
      return res.status(400).json({ message: "Invalid template ID" });
    }

    const partialSchema = insertProductionPlanNameTemplateSchema.partial();
    const validatedData = partialSchema.parse(req.body);

    const template = await updateTemplate(pool, id, validatedData);
    res.json(template);
  } catch (error: any) {
    if (error instanceof z.ZodError) {
      return res.status(400).json({ message: "Validation error", errors: error.errors });
    }
    
    console.error("Failed to update template:", error);
    res.status(500).json({ message: error.message || "Failed to update template" });
  }
});

router.delete("/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    if (isNaN(id)) {
      return res.status(400).json({ message: "Invalid template ID" });
    }

    await deleteTemplate(pool, id);
    res.status(204).send();
  } catch (error: any) {
    console.error("Failed to delete template:", error);
    res.status(500).json({ message: "Failed to delete template" });
  }
});

const generateRequestSchema = z.object({
  templateId: z.number().int().positive(),
  dateOverride: z.string().datetime().optional(),
  preview: z.boolean().optional().default(true),
});

router.post("/generate", async (req, res) => {
  const client = await pool.connect();
  
  try {
    const { templateId, dateOverride, preview } = generateRequestSchema.parse(req.body);
    
    await client.query("BEGIN");
    
    const result = await generatePlanName(
      {
        templateId,
        dateOverride: dateOverride ? new Date(dateOverride) : undefined,
        preview,
      },
      client
    );
    
    await client.query("COMMIT");
    
    res.json(result);
  } catch (error: any) {
    await client.query("ROLLBACK");
    
    if (error instanceof z.ZodError) {
      return res.status(400).json({ message: "Validation error", errors: error.errors });
    }
    
    console.error("Failed to generate plan name:", error);
    res.status(500).json({ message: error.message || "Failed to generate plan name" });
  } finally {
    client.release();
  }
});

export default router;
