import type { Express } from "express";
import { z } from "zod";
import { pool } from "../../postgres";
import { requirePermission } from "../../auth";

const createPalletSchema = z.object({
  productionOrderId: z.number(),
  palletLabel: z.string().optional(),
  colorCode: z.string().optional(),
  flowCode: z.string(),
  routingId: z.number().optional().nullable(),
  routingVariantCode: z.string().optional(),
  carrierId: z.number().optional().nullable(),
  carrierCode: z.string().optional(),
  maxCapacity: z.number().optional().default(100),
  notes: z.string().optional(),
});

const updatePalletSchema = z.object({
  status: z.enum(['open', 'filled', 'in_buffer', 'in_operation', 'in_warehouse', 'completed']).optional(),
  isFilled: z.boolean().optional(),
  carrierId: z.number().optional().nullable(),
  carrierCode: z.string().optional(),
  currentOperationId: z.number().optional().nullable(),
  currentOperationCode: z.string().optional(),
  currentLocationId: z.number().optional().nullable(),
  currentLoad: z.number().optional(),
  maxCapacity: z.number().optional(),
  notes: z.string().optional(),
});

const markFilledSchema = z.object({
  isFilled: z.boolean(),
  operatorId: z.number().optional(),
});

const changeCarrierSchema = z.object({
  toCarrierId: z.number(),
  toCarrierCode: z.string(),
  operationId: z.number().optional(),
  operationCode: z.string().optional(),
  locationId: z.number().optional(),
  operatorId: z.number().optional(),
  reason: z.string().optional(),
});

const startWarehouseStaySchema = z.object({
  warehouseId: z.number(),
  warehouseName: z.string().optional(),
  reason: z.string().optional(),
  previousOperationId: z.number().optional(),
  previousOperationCode: z.string().optional(),
  operatorId: z.number().optional(),
  notes: z.string().optional(),
});

const endWarehouseStaySchema = z.object({
  nextOperationId: z.number().optional(),
  nextOperationCode: z.string().optional(),
  operatorId: z.number().optional(),
  notes: z.string().optional(),
});

const addPalletEventSchema = z.object({
  eventType: z.string(),
  eventData: z.record(z.any()).optional(),
  operatorId: z.number().optional(),
  operatorName: z.string().optional(),
  locationId: z.number().optional(),
  operationId: z.number().optional(),
  notes: z.string().optional(),
});

export function registerPalletRoutes(app: Express) {
  
  // GET /api/production/orders/:orderId/pallets - Pobierz palety dla ZLP
  app.get("/api/production/orders/:orderId/pallets", requirePermission('view_production'), async (req, res) => {
    try {
      const orderId = parseInt(req.params.orderId);
      if (isNaN(orderId)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const result = await pool.query(`
        SELECT 
          p.*,
          c.code as carrier_code_resolved,
          c.name as carrier_name,
          cg.name as carrier_group_name,
          loc.name as location_name,
          wh.name as warehouse_name_resolved
        FROM production.production_order_pallets p
        LEFT JOIN production.production_carriers c ON p.carrier_id = c.id
        LEFT JOIN production.production_carrier_groups cg ON c.carrier_group_id = cg.id
        LEFT JOIN production.production_locations loc ON p.current_location_id = loc.id
        LEFT JOIN production.production_locations wh ON p.warehouse_id = wh.id
        WHERE p.production_order_id = $1
        ORDER BY p.sequence, p.created_at
      `, [orderId]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching pallets:", error);
      res.status(500).json({ message: "Failed to fetch pallets" });
    }
  });

  // GET /api/production/pallets/:id - Pobierz szczegóły palety
  app.get("/api/production/pallets/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid pallet ID" });
      }

      const result = await pool.query(`
        SELECT 
          p.*,
          c.code as carrier_code_resolved,
          c.name as carrier_name,
          cg.name as carrier_group_name,
          loc.name as location_name,
          wh.name as warehouse_name_resolved
        FROM production.production_order_pallets p
        LEFT JOIN production.production_carriers c ON p.carrier_id = c.id
        LEFT JOIN production.production_carrier_groups cg ON c.carrier_group_id = cg.id
        LEFT JOIN production.production_locations loc ON p.current_location_id = loc.id
        LEFT JOIN production.production_locations wh ON p.warehouse_id = wh.id
        WHERE p.id = $1
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Pallet not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error fetching pallet:", error);
      res.status(500).json({ message: "Failed to fetch pallet" });
    }
  });

  // POST /api/production/orders/:orderId/pallets - Utwórz nową paletę
  app.post("/api/production/orders/:orderId/pallets", requirePermission('manage_production'), async (req, res) => {
    try {
      const orderId = parseInt(req.params.orderId);
      if (isNaN(orderId)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const data = createPalletSchema.parse({ ...req.body, productionOrderId: orderId });

      const seqResult = await pool.query(`
        SELECT COALESCE(MAX(sequence), 0) + 1 as next_seq
        FROM production.production_order_pallets
        WHERE production_order_id = $1
      `, [orderId]);
      const nextSeq = seqResult.rows[0].next_seq;

      const palletLabel = data.palletLabel || `${data.colorCode || 'PAL'}-${data.flowCode}-${nextSeq}`;

      const result = await pool.query(`
        INSERT INTO production.production_order_pallets (
          production_order_id, pallet_label, sequence, color_code, flow_code,
          routing_id, routing_variant_code, carrier_id, carrier_code,
          status, is_filled, max_capacity, current_load, fill_percentage, notes
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, 'open', false, $10, 0, 0, $11)
        RETURNING *
      `, [
        orderId, palletLabel, nextSeq, data.colorCode, data.flowCode,
        data.routingId, data.routingVariantCode, data.carrierId, data.carrierCode,
        data.maxCapacity, data.notes
      ]);

      await pool.query(`
        INSERT INTO production.production_pallet_events (pallet_id, event_type, event_data, notes)
        VALUES ($1, 'created', $2, 'Paleta utworzona')
      `, [result.rows[0].id, JSON.stringify({ flowCode: data.flowCode, colorCode: data.colorCode })]);

      res.status(201).json(result.rows[0]);
    } catch (error) {
      console.error("Error creating pallet:", error);
      res.status(500).json({ message: "Failed to create pallet" });
    }
  });

  // PATCH /api/production/pallets/:id - Aktualizuj paletę
  app.patch("/api/production/pallets/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid pallet ID" });
      }

      const data = updatePalletSchema.parse(req.body);

      const updates: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      if (data.status !== undefined) {
        updates.push(`status = $${paramIndex++}`);
        values.push(data.status);
      }
      if (data.isFilled !== undefined) {
        updates.push(`is_filled = $${paramIndex++}`);
        values.push(data.isFilled);
      }
      if (data.carrierId !== undefined) {
        updates.push(`carrier_id = $${paramIndex++}`);
        values.push(data.carrierId);
      }
      if (data.carrierCode !== undefined) {
        updates.push(`carrier_code = $${paramIndex++}`);
        values.push(data.carrierCode);
      }
      if (data.currentOperationId !== undefined) {
        updates.push(`current_operation_id = $${paramIndex++}`);
        values.push(data.currentOperationId);
      }
      if (data.currentOperationCode !== undefined) {
        updates.push(`current_operation_code = $${paramIndex++}`);
        values.push(data.currentOperationCode);
      }
      if (data.currentLocationId !== undefined) {
        updates.push(`current_location_id = $${paramIndex++}`);
        values.push(data.currentLocationId);
      }
      if (data.currentLoad !== undefined) {
        updates.push(`current_load = $${paramIndex++}`);
        values.push(data.currentLoad);
        if (data.maxCapacity) {
          updates.push(`fill_percentage = $${paramIndex++}`);
          values.push((data.currentLoad / data.maxCapacity) * 100);
        }
      }
      if (data.maxCapacity !== undefined) {
        updates.push(`max_capacity = $${paramIndex++}`);
        values.push(data.maxCapacity);
      }
      if (data.notes !== undefined) {
        updates.push(`notes = $${paramIndex++}`);
        values.push(data.notes);
      }

      updates.push(`updated_at = NOW()`);

      if (updates.length === 1) {
        return res.status(400).json({ message: "No fields to update" });
      }

      values.push(id);
      const result = await pool.query(`
        UPDATE production.production_order_pallets
        SET ${updates.join(', ')}
        WHERE id = $${paramIndex}
        RETURNING *
      `, values);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Pallet not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error updating pallet:", error);
      res.status(500).json({ message: "Failed to update pallet" });
    }
  });

  // POST /api/production/pallets/:id/mark-filled - Oznacz paletę jako zapełnioną
  app.post("/api/production/pallets/:id/mark-filled", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid pallet ID" });
      }

      const data = markFilledSchema.parse(req.body);

      const result = await pool.query(`
        UPDATE production.production_order_pallets
        SET is_filled = $1, status = CASE WHEN $1 THEN 'filled' ELSE status END, updated_at = NOW()
        WHERE id = $2
        RETURNING *
      `, [data.isFilled, id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Pallet not found" });
      }

      await pool.query(`
        INSERT INTO production.production_pallet_events (pallet_id, event_type, event_data, operator_id)
        VALUES ($1, 'marked_filled', $2, $3)
      `, [id, JSON.stringify({ isFilled: data.isFilled }), data.operatorId]);

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error marking pallet as filled:", error);
      res.status(500).json({ message: "Failed to mark pallet as filled" });
    }
  });

  // POST /api/production/pallets/:id/change-carrier - Zmień nośnik palety
  app.post("/api/production/pallets/:id/change-carrier", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid pallet ID" });
      }

      const data = changeCarrierSchema.parse(req.body);

      const currentResult = await pool.query(`
        SELECT carrier_id, carrier_code FROM production.production_order_pallets WHERE id = $1
      `, [id]);

      if (currentResult.rows.length === 0) {
        return res.status(404).json({ message: "Pallet not found" });
      }

      const current = currentResult.rows[0];

      const result = await pool.query(`
        UPDATE production.production_order_pallets
        SET carrier_id = $1, carrier_code = $2, updated_at = NOW()
        WHERE id = $3
        RETURNING *
      `, [data.toCarrierId, data.toCarrierCode, id]);

      await pool.query(`
        INSERT INTO production.production_carrier_changes (
          pallet_id, from_carrier_id, from_carrier_code, to_carrier_id, to_carrier_code,
          operation_id, operation_code, location_id, operator_id, reason
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
      `, [
        id, current.carrier_id, current.carrier_code,
        data.toCarrierId, data.toCarrierCode,
        data.operationId, data.operationCode, data.locationId, data.operatorId, data.reason
      ]);

      await pool.query(`
        INSERT INTO production.production_pallet_events (pallet_id, event_type, event_data, operator_id, operation_id, location_id)
        VALUES ($1, 'carrier_changed', $2, $3, $4, $5)
      `, [
        id,
        JSON.stringify({
          fromCarrierId: current.carrier_id,
          fromCarrierCode: current.carrier_code,
          toCarrierId: data.toCarrierId,
          toCarrierCode: data.toCarrierCode
        }),
        data.operatorId, data.operationId, data.locationId
      ]);

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error changing pallet carrier:", error);
      res.status(500).json({ message: "Failed to change carrier" });
    }
  });

  // POST /api/production/pallets/:id/warehouse-stay/start - Rozpocznij pobyt na magazynie
  app.post("/api/production/pallets/:id/warehouse-stay/start", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid pallet ID" });
      }

      const data = startWarehouseStaySchema.parse(req.body);

      const palletResult = await pool.query(`
        SELECT production_order_id FROM production.production_order_pallets WHERE id = $1
      `, [id]);

      if (palletResult.rows.length === 0) {
        return res.status(404).json({ message: "Pallet not found" });
      }

      const productionOrderId = palletResult.rows[0].production_order_id;

      const stayResult = await pool.query(`
        INSERT INTO production.production_warehouse_stays (
          pallet_id, production_order_id, warehouse_id, warehouse_name,
          start_time, reason, previous_operation_id, previous_operation_code,
          status, started_by_operator_id, notes
        ) VALUES ($1, $2, $3, $4, NOW(), $5, $6, $7, 'active', $8, $9)
        RETURNING *
      `, [
        id, productionOrderId, data.warehouseId, data.warehouseName,
        data.reason, data.previousOperationId, data.previousOperationCode,
        data.operatorId, data.notes
      ]);

      await pool.query(`
        UPDATE production.production_order_pallets
        SET status = 'in_warehouse', warehouse_id = $1, warehouse_entry_time = NOW(), updated_at = NOW()
        WHERE id = $2
      `, [data.warehouseId, id]);

      await pool.query(`
        INSERT INTO production.production_pallet_events (pallet_id, event_type, event_data, operator_id, location_id)
        VALUES ($1, 'warehouse_in', $2, $3, $4)
      `, [
        id,
        JSON.stringify({ warehouseId: data.warehouseId, warehouseName: data.warehouseName, reason: data.reason }),
        data.operatorId, data.warehouseId
      ]);

      res.json(stayResult.rows[0]);
    } catch (error) {
      console.error("Error starting warehouse stay:", error);
      res.status(500).json({ message: "Failed to start warehouse stay" });
    }
  });

  // POST /api/production/pallets/:id/warehouse-stay/end - Zakończ pobyt na magazynie
  app.post("/api/production/pallets/:id/warehouse-stay/end", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid pallet ID" });
      }

      const data = endWarehouseStaySchema.parse(req.body);

      const stayResult = await pool.query(`
        UPDATE production.production_warehouse_stays
        SET 
          end_time = NOW(),
          duration_seconds = EXTRACT(EPOCH FROM (NOW() - start_time))::integer,
          next_operation_id = $1,
          next_operation_code = $2,
          ended_by_operator_id = $3,
          status = 'completed',
          notes = COALESCE(notes || E'\\n' || $4, $4),
          updated_at = NOW()
        WHERE pallet_id = $5 AND status = 'active'
        RETURNING *
      `, [data.nextOperationId, data.nextOperationCode, data.operatorId, data.notes, id]);

      if (stayResult.rows.length === 0) {
        return res.status(404).json({ message: "Active warehouse stay not found" });
      }

      await pool.query(`
        UPDATE production.production_order_pallets
        SET status = 'open', warehouse_exit_time = NOW(), updated_at = NOW()
        WHERE id = $1
      `, [id]);

      await pool.query(`
        INSERT INTO production.production_pallet_events (pallet_id, event_type, event_data, operator_id)
        VALUES ($1, 'warehouse_out', $2, $3)
      `, [
        id,
        JSON.stringify({
          durationSeconds: stayResult.rows[0].duration_seconds,
          nextOperationCode: data.nextOperationCode
        }),
        data.operatorId
      ]);

      res.json(stayResult.rows[0]);
    } catch (error) {
      console.error("Error ending warehouse stay:", error);
      res.status(500).json({ message: "Failed to end warehouse stay" });
    }
  });

  // GET /api/production/pallets/:id/warehouse-stays - Pobierz historię pobytów na magazynie
  app.get("/api/production/pallets/:id/warehouse-stays", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid pallet ID" });
      }

      const result = await pool.query(`
        SELECT 
          ws.*,
          wh.name as warehouse_name_resolved,
          so.name as started_by_name,
          eo.name as ended_by_name
        FROM production.production_warehouse_stays ws
        LEFT JOIN production.production_locations wh ON ws.warehouse_id = wh.id
        LEFT JOIN production.production_operators so ON ws.started_by_operator_id = so.id
        LEFT JOIN production.production_operators eo ON ws.ended_by_operator_id = eo.id
        WHERE ws.pallet_id = $1
        ORDER BY ws.start_time DESC
      `, [id]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching warehouse stays:", error);
      res.status(500).json({ message: "Failed to fetch warehouse stays" });
    }
  });

  // GET /api/production/pallets/:id/events - Pobierz historię zdarzeń palety
  app.get("/api/production/pallets/:id/events", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid pallet ID" });
      }

      const result = await pool.query(`
        SELECT 
          pe.*,
          op.name as operator_name_resolved,
          loc.name as location_name_resolved,
          ro.name as operation_name_resolved
        FROM production.production_pallet_events pe
        LEFT JOIN production.production_operators op ON pe.operator_id = op.id
        LEFT JOIN production.production_locations loc ON pe.location_id = loc.id
        LEFT JOIN production.production_routing_operations ro ON pe.operation_id = ro.id
        WHERE pe.pallet_id = $1
        ORDER BY pe.created_at DESC
      `, [id]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching pallet events:", error);
      res.status(500).json({ message: "Failed to fetch pallet events" });
    }
  });

  // POST /api/production/pallets/:id/events - Dodaj zdarzenie do palety
  app.post("/api/production/pallets/:id/events", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid pallet ID" });
      }

      const data = addPalletEventSchema.parse(req.body);

      const result = await pool.query(`
        INSERT INTO production.production_pallet_events (
          pallet_id, event_type, event_data, operator_id, operator_name, location_id, operation_id, notes
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        RETURNING *
      `, [
        id, data.eventType, data.eventData ? JSON.stringify(data.eventData) : null,
        data.operatorId, data.operatorName, data.locationId, data.operationId, data.notes
      ]);

      res.status(201).json(result.rows[0]);
    } catch (error) {
      console.error("Error adding pallet event:", error);
      res.status(500).json({ message: "Failed to add pallet event" });
    }
  });

  // GET /api/production/pallets/:id/carrier-changes - Pobierz historię zmian nośników
  app.get("/api/production/pallets/:id/carrier-changes", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid pallet ID" });
      }

      const result = await pool.query(`
        SELECT 
          cc.*,
          fc.name as from_carrier_name,
          tc.name as to_carrier_name,
          op.name as operator_name,
          ro.name as operation_name
        FROM production.production_carrier_changes cc
        LEFT JOIN production.production_carriers fc ON cc.from_carrier_id = fc.id
        LEFT JOIN production.production_carriers tc ON cc.to_carrier_id = tc.id
        LEFT JOIN production.production_operators op ON cc.operator_id = op.id
        LEFT JOIN production.production_routing_operations ro ON cc.operation_id = ro.id
        WHERE cc.pallet_id = $1
        ORDER BY cc.created_at DESC
      `, [id]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching carrier changes:", error);
      res.status(500).json({ message: "Failed to fetch carrier changes" });
    }
  });

  // GET /api/production/orders/:orderId/warehouse-stays/active - Aktywne pobyty magazynowe dla ZLP
  app.get("/api/production/orders/:orderId/warehouse-stays/active", requirePermission('view_production'), async (req, res) => {
    try {
      const orderId = parseInt(req.params.orderId);
      if (isNaN(orderId)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const result = await pool.query(`
        SELECT 
          ws.*,
          p.pallet_label,
          p.color_code,
          p.flow_code,
          wh.name as warehouse_name_resolved,
          EXTRACT(EPOCH FROM (NOW() - ws.start_time))::integer as current_duration_seconds
        FROM production.production_warehouse_stays ws
        JOIN production.production_order_pallets p ON ws.pallet_id = p.id
        LEFT JOIN production.production_locations wh ON ws.warehouse_id = wh.id
        WHERE ws.production_order_id = $1 AND ws.status = 'active'
        ORDER BY ws.start_time
      `, [orderId]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching active warehouse stays:", error);
      res.status(500).json({ message: "Failed to fetch active warehouse stays" });
    }
  });
}
