import type { Express } from "express";
import { z } from "zod";
import { pool } from "../../postgres";
import * as ordersService from "../../services/production/orders";
import { generateProductionCardPDF, generateSawCSV } from "../../services/production/order-exports";
import { requirePermission } from "../../auth";

const createOrderSchema = z.object({
  orderNumber: z.string().optional(),
  productId: z.number(),
  bomId: z.number().optional().nullable(),
  routingId: z.number().optional().nullable(),
  status: z.enum(['draft', 'confirmed', 'planned', 'in_progress', 'paused', 'done', 'cancelled']).optional(),
  priority: z.enum(['low', 'normal', 'high', 'urgent']).optional(),
  quantityPlanned: z.number().positive(),
  unitOfMeasure: z.string().optional(),
  plannedStartDate: z.string().optional().nullable(),
  plannedEndDate: z.string().optional().nullable(),
  responsibleUserId: z.number().optional().nullable(),
  sourceOrderNumber: z.string().optional().nullable(),
  locationId: z.number().optional().nullable(),
  notes: z.string().optional().nullable(),
});

const filtersSchema = z.object({
  status: z.string().optional(),
  workflowStage: z.string().optional(),
  priority: z.string().optional(),
  productId: z.coerce.number().optional(),
  responsibleUserId: z.coerce.number().optional(),
  startDate: z.string().optional(),
  endDate: z.string().optional(),
  search: z.string().optional(),
  limit: z.coerce.number().optional(),
  offset: z.coerce.number().optional(),
});

const updateWorkflowStageSchema = z.object({
  workflowStage: z.string(),
});

const updateDamageSchema = z.object({
  isDamaged: z.boolean(),
  damageType: z.string().optional().nullable(),
  damageNotes: z.string().optional().nullable(),
}).refine(
  (data) => !data.isDamaged || (data.isDamaged && data.damageType),
  {
    message: "damageType is required when isDamaged is true",
    path: ["damageType"],
  }
);

export function registerOrderRoutes(app: Express) {
  // GET /api/production/orders - Get all orders with filtering
  app.get("/api/production/orders", requirePermission('view_production'), async (req, res) => {
    try {
      const filters = filtersSchema.parse(req.query);
      
      // Convert string dates to Date objects
      const processedFilters = {
        ...filters,
        startDate: filters.startDate ? new Date(filters.startDate) : undefined,
        endDate: filters.endDate ? new Date(filters.endDate) : undefined,
      };

      const orders = await ordersService.getOrders(pool, processedFilters);
      res.json(orders);
    } catch (error) {
      console.error("Error fetching production orders:", error);
      res.status(500).json({ message: "Failed to fetch production orders" });
    }
  });

  // GET /api/production/orders/:id - Get order by ID
  app.get("/api/production/orders/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.getOrderById(pool, id);
      if (!order) {
        return res.status(404).json({ message: "Production order not found" });
      }

      console.log(`📦 [ORDER DETAIL] Order ${id} - sourcePlanId: ${order.sourcePlanId}`);
      res.json(order);
    } catch (error) {
      console.error("Error fetching production order:", error);
      res.status(500).json({ message: "Failed to fetch production order" });
    }
  });

  // GET /api/production/orders/:id/items - Get BOM items (formatki/komponenty) for a ZLP
  app.get("/api/production/orders/:id/items", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const items = await ordersService.getOrderBomItems(pool, id);
      res.json(items);
    } catch (error) {
      console.error("Error fetching order BOM items:", error);
      res.status(500).json({ message: "Failed to fetch order items" });
    }
  });

  // POST /api/production/orders - Create new order
  app.post("/api/production/orders", requirePermission('manage_production'), async (req, res) => {
    try {
      const data = createOrderSchema.parse(req.body);
      
      // Auto-generate order number if not provided
      const orderNumber = data.orderNumber || await ordersService.generateOrderNumber(pool);
      
      const order = await ordersService.createOrder(pool, {
        ...data,
        orderNumber,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : null,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : null,
      });
      
      res.status(201).json(order);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Order with this number already exists" });
      }
      console.error("Error creating production order:", error);
      res.status(500).json({ message: "Failed to create production order" });
    }
  });

  // PATCH /api/production/orders/:id - Update order
  app.patch("/api/production/orders/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const data = createOrderSchema.partial().parse(req.body);
      const order = await ordersService.updateOrder(pool, id, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : undefined,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : undefined,
      });
      
      if (!order) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.json(order);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Order with this number already exists" });
      }
      console.error("Error updating production order:", error);
      res.status(500).json({ message: "Failed to update production order" });
    }
  });

  // DELETE /api/production/orders/:id - Delete order
  app.delete("/api/production/orders/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const success = await ordersService.deleteOrder(pool, id);
      if (!success) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting production order:", error);
      res.status(500).json({ message: "Failed to delete production order" });
    }
  });

  // POST /api/production/orders/:id/start - Start order
  app.post("/api/production/orders/:id/start", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.startOrder(pool, id);
      if (!order) {
        return res.status(400).json({ message: "Cannot start order - invalid status or order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error starting production order:", error);
      res.status(500).json({ message: "Failed to start production order" });
    }
  });

  // POST /api/production/orders/:id/complete - Complete order
  app.post("/api/production/orders/:id/complete", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.completeOrder(pool, id);
      if (!order) {
        return res.status(400).json({ message: "Cannot complete order - must be in_progress or order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error completing production order:", error);
      res.status(500).json({ message: "Failed to complete production order" });
    }
  });

  // POST /api/production/orders/:id/cancel - Cancel order
  app.post("/api/production/orders/:id/cancel", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.cancelOrder(pool, id);
      if (!order) {
        return res.status(400).json({ message: "Cannot cancel order - already done/cancelled or order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error cancelling production order:", error);
      res.status(500).json({ message: "Failed to cancel production order" });
    }
  });

  // PATCH /api/production/orders/:id/workflow-stage - Update workflow stage
  app.patch("/api/production/orders/:id/workflow-stage", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const { workflowStage } = updateWorkflowStageSchema.parse(req.body);
      
      // Get user ID from session (if authenticated)
      const userId = (req as any).user?.id;

      const order = await ordersService.updateWorkflowStage(pool, id, workflowStage, userId);
      if (!order) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error updating workflow stage:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      res.status(500).json({ message: "Failed to update workflow stage" });
    }
  });

  // PATCH /api/production/order-bom-items/:id/damage - Mark/unmark component as damaged
  app.patch("/api/production/order-bom-items/:id/damage", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid BOM item ID" });
      }

      const data = updateDamageSchema.parse(req.body);
      
      // Get user ID from session (if authenticated)
      const userId = (req as any).user?.id;

      const item = await ordersService.updateBomItemDamage(pool, id, {
        isDamaged: data.isDamaged,
        damageType: data.damageType,
        damageNotes: data.damageNotes,
        damagedBy: userId,
      });
      
      if (!item) {
        return res.status(404).json({ message: "BOM item not found" });
      }

      res.json(item);
    } catch (error) {
      console.error("Error updating BOM item damage status:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      res.status(500).json({ message: "Failed to update damage status" });
    }
  });

  // PATCH /api/production/order-bom-items/:id/drilling - Mark/unmark component as drilled
  app.patch("/api/production/order-bom-items/:id/drilling", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid BOM item ID" });
      }

      const { isDrilled } = req.body;
      const userId = (req as any).user?.id ? parseInt((req as any).user.id) : null;

      const result = await pool.query(`
        UPDATE production.production_order_bom_items
        SET 
          is_drilled = $2,
          drilled_at = CASE WHEN $2 = true THEN NOW() ELSE NULL END,
          drilled_by_user_id = CASE WHEN $2 = true THEN $3::integer ELSE NULL END,
          updated_at = NOW()
        WHERE id = $1
        RETURNING id, component_name, is_drilled, drilled_at
      `, [id, isDrilled === true, userId]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ message: "BOM item not found" });
      }

      res.json({
        id: result.rows[0].id,
        componentName: result.rows[0].component_name,
        isDrilled: result.rows[0].is_drilled,
        drilledAt: result.rows[0].drilled_at,
      });
    } catch (error) {
      console.error("Error updating BOM item drilling status:", error);
      res.status(500).json({ message: "Failed to update drilling status" });
    }
  });

  // ====== WORK ORDER ROUTES ======

  // POST /api/production/work-orders/:id/start - Start work order
  app.post("/api/production/work-orders/:id/start", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const result = await ordersService.startWorkOrder(pool, id);
      
      if (result.error) {
        const statusMap: Record<string, number> = {
          'not_found': 404,
          'invalid_status': 400,
          'invalid_order_status': 400,
          'sequence_error': 400,
          'concurrent_update': 409,
        };
        return res.status(statusMap[result.error] || 400).json({ message: result.message });
      }
      
      res.json(result);
    } catch (error: any) {
      console.error("Error starting work order:", error);
      res.status(500).json({ message: "Internal server error while starting work order" });
    }
  });

  // POST /api/production/work-orders/:id/complete - Complete work order
  app.post("/api/production/work-orders/:id/complete", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const options: ordersService.CompleteWorkOrderOptions = {};
      
      if (req.body.quantityProduced !== undefined) {
        options.quantityProduced = parseFloat(req.body.quantityProduced);
      }
      if (req.body.quantityScrap !== undefined) {
        options.quantityScrap = parseFloat(req.body.quantityScrap);
      }
      if (req.body.qualityCheckPassed !== undefined) {
        options.qualityCheckPassed = req.body.qualityCheckPassed;
      }
      if (req.body.qualityCheckNotes !== undefined) {
        options.qualityCheckNotes = req.body.qualityCheckNotes;
      }
      if (req.body.notes !== undefined) {
        options.notes = req.body.notes;
      }

      const result = await ordersService.completeWorkOrder(pool, id, options);
      
      if (result.error) {
        const statusMap: Record<string, number> = {
          'not_found': 404,
          'invalid_status': 400,
          'concurrent_update': 409,
        };
        return res.status(statusMap[result.error] || 400).json({ message: result.message });
      }
      
      res.json(result);
    } catch (error: any) {
      console.error("Error completing work order:", error);
      res.status(500).json({ message: "Internal server error while completing work order" });
    }
  });

  // ========== QR Code Scanning for Shared Operations ==========
  
  const scanOperationSchema = z.object({
    productionOrderId: z.number(),
    operationCode: z.string(),
  });
  
  // Helper to parse QR code format: "ZLP:993:magazynowanie" or JSON
  const parseQRCode = (input: any): { productionOrderId: number; operationCode: string } | null => {
    // Handle JSON body directly
    if (typeof input === 'object' && input.productionOrderId && input.operationCode) {
      return {
        productionOrderId: Number(input.productionOrderId),
        operationCode: String(input.operationCode),
      };
    }
    
    // Handle text format from scanner: "ZLP:993:magazynowanie"
    if (typeof input === 'object' && input.qrCode) {
      const text = String(input.qrCode).trim();
      const match = text.match(/^ZLP:(\d+):(\w+)$/i);
      if (match) {
        return {
          productionOrderId: parseInt(match[1], 10),
          operationCode: match[2],
        };
      }
    }
    
    return null;
  };

  // POST /api/production/scan-operation - Toggle operation status via QR scan
  // First scan: Start all pending work orders for this operation
  // Second scan: Complete all in_progress work orders for this operation
  // Accepts: { productionOrderId, operationCode } or { qrCode: "ZLP:993:magazynowanie" }
  app.post("/api/production/scan-operation", requirePermission('manage_production'), async (req, res) => {
    try {
      const data = parseQRCode(req.body);
      
      if (!data) {
        return res.status(400).json({ 
          message: "Nieprawidłowy format kodu QR. Oczekiwany format: ZLP:ID:OPERACJA" 
        });
      }
      
      // Find all work orders for this production order and operation code
      const workOrdersResult = await pool.query(`
        SELECT wo.id, wo.status, wo.work_order_number, ro.code as operation_code
        FROM production.production_work_orders wo
        JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
        WHERE wo.production_order_id = $1 
          AND LOWER(ro.code) = LOWER($2)
        ORDER BY wo.sequence
      `, [data.productionOrderId, data.operationCode]);
      
      if (workOrdersResult.rows.length === 0) {
        return res.status(404).json({ 
          message: `Nie znaleziono operacji ${data.operationCode} dla zlecenia ${data.productionOrderId}` 
        });
      }
      
      const workOrders = workOrdersResult.rows;
      const pendingWos = workOrders.filter(wo => wo.status === 'pending' || wo.status === 'ready');
      const inProgressWos = workOrders.filter(wo => wo.status === 'in_progress');
      const doneWos = workOrders.filter(wo => wo.status === 'done');
      
      // If all are done, nothing to do
      if (doneWos.length === workOrders.length) {
        return res.json({
          action: 'none',
          message: 'Wszystkie operacje już zakończone',
          operationCode: data.operationCode,
          totalWorkOrders: workOrders.length,
          completed: doneWos.length,
        });
      }
      
      // If there are in_progress work orders, complete them
      if (inProgressWos.length > 0) {
        const results = [];
        for (const wo of inProgressWos) {
          const result = await ordersService.completeWorkOrder(pool, wo.id, {});
          results.push(result);
        }
        
        return res.json({
          action: 'complete',
          message: `Zakończono ${inProgressWos.length} operacji ${data.operationCode}`,
          operationCode: data.operationCode,
          totalWorkOrders: workOrders.length,
          completed: inProgressWos.length,
          workOrderIds: inProgressWos.map(wo => wo.id),
        });
      }
      
      // If there are pending work orders, start them
      if (pendingWos.length > 0) {
        const results: any[] = [];
        const errors: { id: number; error: string }[] = [];
        
        for (const wo of pendingWos) {
          const result = await ordersService.startWorkOrder(pool, wo.id);
          if (result.error) {
            errors.push({ id: wo.id, error: result.message });
          } else {
            results.push(result);
          }
        }
        
        if (errors.length > 0 && results.length === 0) {
          return res.status(400).json({
            action: 'start',
            message: `Nie można rozpocząć operacji: ${errors[0].error}`,
            operationCode: data.operationCode,
            errors,
          });
        }
        
        return res.json({
          action: 'start',
          message: `Rozpoczęto ${results.length} operacji ${data.operationCode}`,
          operationCode: data.operationCode,
          totalWorkOrders: workOrders.length,
          started: results.length,
          workOrderIds: pendingWos.filter((_, i) => !errors.some(e => e.id === pendingWos[i].id)).map(wo => wo.id),
          errors: errors.length > 0 ? errors : undefined,
        });
      }
      
      return res.json({
        action: 'none',
        message: 'Brak operacji do wykonania',
        operationCode: data.operationCode,
      });
      
    } catch (error: any) {
      console.error("Error scanning operation:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Nieprawidłowe dane QR", errors: error.errors });
      }
      res.status(500).json({ message: "Błąd podczas skanowania operacji" });
    }
  });

  // ========== Multi-Source Material APIs ==========

  const materialSourceSchema = z.object({
    workOrderId: z.number(),
    sourceType: z.enum(['hdf', 'formatka', 'okucia', 'opakowania', 'tapicernia', 'surowiec', 'polprodukt']),
    sourceLocationId: z.number().optional().nullable(),
    sourceLocationName: z.string().optional().nullable(),
    quantityRequired: z.number().positive(),
    notes: z.string().optional().nullable(),
  });

  const updateMaterialSourceSchema = z.object({
    quantityProvided: z.number().min(0).optional(),
    documentId: z.number().optional().nullable(),
    documentNumber: z.string().optional().nullable(),
    notes: z.string().optional().nullable(),
  });

  // GET /api/production/work-orders/:id/material-sources - Get material sources for work order
  app.get("/api/production/work-orders/:id/material-sources", requirePermission('view_production'), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      if (isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const result = await pool.query(`
        SELECT 
          oms.id,
          oms.work_order_id as "workOrderId",
          oms.source_type as "sourceType",
          oms.source_location_id as "sourceLocationId",
          oms.source_location_name as "sourceLocationName",
          oms.quantity_required as "quantityRequired",
          oms.quantity_provided as "quantityProvided",
          oms.document_id as "documentId",
          oms.document_number as "documentNumber",
          oms.status,
          oms.notes,
          oms.created_at as "createdAt",
          oms.updated_at as "updatedAt",
          pl.name as "locationName",
          pl.code as "locationCode"
        FROM production.operation_material_sources oms
        LEFT JOIN production.production_locations pl ON oms.source_location_id = pl.id
        WHERE oms.work_order_id = $1
        ORDER BY oms.source_type, oms.created_at
      `, [workOrderId]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching material sources:", error);
      res.status(500).json({ message: "Failed to fetch material sources" });
    }
  });

  // POST /api/production/work-orders/material-sources - Add material source requirement
  app.post("/api/production/work-orders/material-sources", requirePermission('manage_production'), async (req, res) => {
    try {
      const parsed = materialSourceSchema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: "Invalid request body", errors: parsed.error.errors });
      }

      const { workOrderId, sourceType, sourceLocationId, sourceLocationName, quantityRequired, notes } = parsed.data;

      const result = await pool.query(`
        INSERT INTO production.operation_material_sources (
          work_order_id, source_type, source_location_id, source_location_name,
          quantity_required, notes
        ) VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING 
          id,
          work_order_id as "workOrderId",
          source_type as "sourceType",
          source_location_id as "sourceLocationId",
          source_location_name as "sourceLocationName",
          quantity_required as "quantityRequired",
          quantity_provided as "quantityProvided",
          status,
          notes,
          created_at as "createdAt"
      `, [workOrderId, sourceType, sourceLocationId || null, sourceLocationName || null, quantityRequired, notes || null]);

      res.status(201).json(result.rows[0]);
    } catch (error) {
      console.error("Error adding material source:", error);
      res.status(500).json({ message: "Failed to add material source" });
    }
  });

  // PATCH /api/production/work-orders/material-sources/:id - Update material source (quantity provided, document link)
  app.patch("/api/production/work-orders/material-sources/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid material source ID" });
      }

      const parsed = updateMaterialSourceSchema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: "Invalid request body", errors: parsed.error.errors });
      }

      const updates: string[] = ['updated_at = NOW()'];
      const values: any[] = [];
      let paramIndex = 1;

      const { quantityProvided, documentId, documentNumber, notes } = parsed.data;

      if (quantityProvided !== undefined) {
        updates.push(`quantity_provided = LEAST(quantity_required, $${paramIndex++})`);
        values.push(quantityProvided);
        updates.push(`status = CASE 
          WHEN LEAST(quantity_required, $${paramIndex - 1}) >= quantity_required THEN 'complete'
          WHEN LEAST(quantity_required, $${paramIndex - 1}) > 0 THEN 'partial'
          ELSE 'pending'
        END`);
      }
      if (documentId !== undefined) {
        updates.push(`document_id = $${paramIndex++}`);
        values.push(documentId);
      }
      if (documentNumber !== undefined) {
        updates.push(`document_number = $${paramIndex++}`);
        values.push(documentNumber);
      }
      if (notes !== undefined) {
        updates.push(`notes = $${paramIndex++}`);
        values.push(notes);
      }

      values.push(id);

      const result = await pool.query(`
        UPDATE production.operation_material_sources
        SET ${updates.join(', ')}
        WHERE id = $${paramIndex}
        RETURNING 
          id,
          work_order_id as "workOrderId",
          source_type as "sourceType",
          quantity_required as "quantityRequired",
          quantity_provided as "quantityProvided",
          document_id as "documentId",
          document_number as "documentNumber",
          status,
          notes,
          updated_at as "updatedAt"
      `, values);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Material source not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error updating material source:", error);
      res.status(500).json({ message: "Failed to update material source" });
    }
  });

  // DELETE /api/production/work-orders/material-sources/:id - Remove material source requirement
  app.delete("/api/production/work-orders/material-sources/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid material source ID" });
      }

      const result = await pool.query(`
        DELETE FROM production.operation_material_sources
        WHERE id = $1
        RETURNING id
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Material source not found" });
      }

      res.json({ message: "Material source removed", id });
    } catch (error) {
      console.error("Error removing material source:", error);
      res.status(500).json({ message: "Failed to remove material source" });
    }
  });

  // GET /api/production/work-orders/:id/material-readiness - Check if all material sources are ready
  app.get("/api/production/work-orders/:id/material-readiness", requirePermission('view_production'), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      if (isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const result = await pool.query(`
        SELECT 
          COUNT(*) as "totalSources",
          COUNT(CASE WHEN status = 'complete' THEN 1 END) as "completeSources",
          COUNT(CASE WHEN status = 'partial' THEN 1 END) as "partialSources",
          COUNT(CASE WHEN status = 'pending' THEN 1 END) as "pendingSources",
          SUM(quantity_required) as "totalRequired",
          SUM(quantity_provided) as "totalProvided",
          CASE 
            WHEN COUNT(*) = 0 THEN true
            WHEN COUNT(*) = COUNT(CASE WHEN status = 'complete' THEN 1 END) THEN true
            ELSE false
          END as "isReady"
        FROM production.operation_material_sources
        WHERE work_order_id = $1
      `, [workOrderId]);

      const readiness = result.rows[0];
      res.json({
        workOrderId,
        totalSources: parseInt(readiness.totalSources) || 0,
        completeSources: parseInt(readiness.completeSources) || 0,
        partialSources: parseInt(readiness.partialSources) || 0,
        pendingSources: parseInt(readiness.pendingSources) || 0,
        totalRequired: parseFloat(readiness.totalRequired) || 0,
        totalProvided: parseFloat(readiness.totalProvided) || 0,
        isReady: readiness.isReady,
      });
    } catch (error) {
      console.error("Error checking material readiness:", error);
      res.status(500).json({ message: "Failed to check material readiness" });
    }
  });

  // PATCH /api/production/work-orders/:id/operator - Update operator assignment for a work order (legacy - single operator)
  app.patch("/api/production/work-orders/:id/operator", requirePermission('manage_production'), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      if (isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const schema = z.object({
        operatorId: z.number().nullable(),
        operatorName: z.string().nullable().optional(),
      });

      const parsed = schema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: "Invalid request body", errors: parsed.error.errors });
      }

      const { operatorId } = parsed.data;

      // Walidacja: sprawdź czy operator istnieje (jeśli podano ID)
      let finalOperatorId: number | null = null;
      let finalOperatorName: string | null = null;
      
      if (operatorId !== null) {
        const opResult = await pool.query(
          `SELECT id, full_name FROM production.production_operators WHERE id = $1`,
          [operatorId]
        );
        if (opResult.rows.length === 0) {
          return res.status(400).json({ message: "Operator not found", operatorId });
        }
        finalOperatorId = opResult.rows[0].id;
        finalOperatorName = opResult.rows[0].full_name;
      }

      const result = await pool.query(`
        UPDATE production.production_work_orders
        SET operator_id = $1, operator_name = $2, updated_at = NOW()
        WHERE id = $3
        RETURNING 
          id,
          operator_id as "operatorId",
          operator_name as "operatorName"
      `, [finalOperatorId, finalOperatorName, workOrderId]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Work order not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error updating work order operator:", error);
      res.status(500).json({ message: "Failed to update work order operator" });
    }
  });

  // GET /api/production/work-orders/:id/operators - Get all operators assigned to a work order
  app.get("/api/production/work-orders/:id/operators", requirePermission('view_production'), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      if (isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const result = await pool.query(`
        SELECT 
          woo.id,
          woo.work_order_id as "workOrderId",
          woo.operator_id as "operatorId",
          woo.is_primary as "isPrimary",
          woo.assigned_at as "assignedAt",
          woo.notes,
          po.full_name as "operatorName",
          po.short_code as "operatorCode"
        FROM production.work_order_operators woo
        JOIN production.production_operators po ON woo.operator_id = po.id
        WHERE woo.work_order_id = $1
        ORDER BY woo.is_primary DESC, po.full_name
      `, [workOrderId]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching work order operators:", error);
      res.status(500).json({ message: "Failed to fetch work order operators" });
    }
  });

  // PUT /api/production/work-orders/:id/operators - Update all operators for a work order
  app.put("/api/production/work-orders/:id/operators", requirePermission('manage_production'), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      if (isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const schema = z.object({
        operatorIds: z.array(z.number()),
        primaryOperatorId: z.number().nullable().optional(),
      });

      const parsed = schema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: "Invalid request body", errors: parsed.error.errors });
      }

      const { operatorIds, primaryOperatorId } = parsed.data;

      // Sprawdź czy work order istnieje
      const woCheck = await pool.query(
        `SELECT id FROM production.production_work_orders WHERE id = $1`,
        [workOrderId]
      );
      if (woCheck.rows.length === 0) {
        return res.status(404).json({ message: "Work order not found" });
      }

      // Walidacja: sprawdź czy wszyscy operatorzy istnieją
      if (operatorIds.length > 0) {
        const opResult = await pool.query(
          `SELECT id, full_name FROM production.production_operators WHERE id = ANY($1)`,
          [operatorIds]
        );
        if (opResult.rows.length !== operatorIds.length) {
          const foundIds = opResult.rows.map(r => r.id);
          const missingIds = operatorIds.filter(id => !foundIds.includes(id));
          return res.status(400).json({ message: "Some operators not found", missingIds });
        }
      }

      const client = await pool.connect();
      try {
        await client.query('BEGIN');

        // Usuń wszystkie obecne przypisania
        await client.query(
          `DELETE FROM production.work_order_operators WHERE work_order_id = $1`,
          [workOrderId]
        );

        // Dodaj nowe przypisania
        for (const opId of operatorIds) {
          const isPrimary = opId === primaryOperatorId;
          await client.query(`
            INSERT INTO production.work_order_operators (work_order_id, operator_id, is_primary)
            VALUES ($1, $2, $3)
          `, [workOrderId, opId, isPrimary]);
        }

        // Zaktualizuj legacy pola w work_orders dla kompatybilności
        if (primaryOperatorId) {
          const primaryOp = await client.query(
            `SELECT full_name FROM production.production_operators WHERE id = $1`,
            [primaryOperatorId]
          );
          await client.query(`
            UPDATE production.production_work_orders
            SET operator_id = $1, operator_name = $2, updated_at = NOW()
            WHERE id = $3
          `, [primaryOperatorId, primaryOp.rows[0]?.full_name || null, workOrderId]);
        } else if (operatorIds.length > 0) {
          // Pierwszy operator jako główny jeśli nie określono
          const firstOp = await client.query(
            `SELECT full_name FROM production.production_operators WHERE id = $1`,
            [operatorIds[0]]
          );
          await client.query(`
            UPDATE production.production_work_orders
            SET operator_id = $1, operator_name = $2, updated_at = NOW()
            WHERE id = $3
          `, [operatorIds[0], firstOp.rows[0]?.full_name || null, workOrderId]);
        } else {
          // Brak operatorów - wyczyść legacy pola
          await client.query(`
            UPDATE production.production_work_orders
            SET operator_id = NULL, operator_name = NULL, updated_at = NOW()
            WHERE id = $1
          `, [workOrderId]);
        }

        await client.query('COMMIT');

        // Pobierz zaktualizowaną listę operatorów
        const result = await client.query(`
          SELECT 
            woo.id,
            woo.work_order_id as "workOrderId",
            woo.operator_id as "operatorId",
            woo.is_primary as "isPrimary",
            woo.assigned_at as "assignedAt",
            woo.notes,
            po.full_name as "operatorName",
            po.short_code as "operatorCode"
          FROM production.work_order_operators woo
          JOIN production.production_operators po ON woo.operator_id = po.id
          WHERE woo.work_order_id = $1
          ORDER BY woo.is_primary DESC, po.full_name
        `, [workOrderId]);

        res.json(result.rows);
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      console.error("Error updating work order operators:", error);
      res.status(500).json({ message: "Failed to update work order operators" });
    }
  });

  // POST /api/production/work-orders/bulk-assign - Bulk assign operators to work orders by operation type
  app.post("/api/production/work-orders/bulk-assign", requirePermission('manage_production'), async (req, res) => {
    try {
      const schema = z.object({
        planId: z.number().int().positive(),
        operationCode: z.string().optional().nullable(),
        workCenterId: z.number().int().positive().optional().nullable(),
        colorCode: z.string().optional().nullable(),
        operatorIds: z.array(z.number().int().positive()).min(1, "Musisz wybrać co najmniej jednego operatora"),
        primaryOperatorId: z.number().int().positive().optional().nullable(),
        mode: z.enum(['overwrite', 'fill_empty']).default('overwrite'),
      }).refine(
        (data) => !data.primaryOperatorId || data.operatorIds.includes(data.primaryOperatorId),
        { message: "Główny operator musi być na liście wybranych operatorów" }
      );

      const parsed = schema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: parsed.error.errors[0]?.message || "Invalid request body", errors: parsed.error.errors });
      }

      const { planId, operationCode, workCenterId, colorCode, operatorIds, primaryOperatorId, mode } = parsed.data;

      // Build query to find matching work orders
      let query = `
        SELECT pwo.id
        FROM production.production_work_orders pwo
        JOIN production.production_orders po ON pwo.production_order_id = po.id
        WHERE po.plan_id = $1
      `;
      const params: any[] = [planId];
      let paramIndex = 2;

      if (operationCode) {
        query += ` AND pwo.operation_code = $${paramIndex++}`;
        params.push(operationCode);
      }
      if (workCenterId) {
        query += ` AND pwo.work_center_id = $${paramIndex++}`;
        params.push(workCenterId);
      }
      if (colorCode) {
        query += ` AND po.color_code = $${paramIndex++}`;
        params.push(colorCode);
      }

      // If mode is 'fill_empty', only get work orders without any operators
      if (mode === 'fill_empty') {
        query += ` AND NOT EXISTS (
          SELECT 1 FROM production.work_order_operators woo WHERE woo.work_order_id = pwo.id
        )`;
      }

      const workOrdersResult = await pool.query(query, params);
      const workOrderIds = workOrdersResult.rows.map(r => r.id);

      if (workOrderIds.length === 0) {
        return res.json({ 
          success: true, 
          message: "Nie znaleziono zleceń do przypisania", 
          assignedCount: 0 
        });
      }

      // Validate operators exist
      if (operatorIds.length > 0) {
        const opResult = await pool.query(
          `SELECT id FROM production.production_operators WHERE id = ANY($1)`,
          [operatorIds]
        );
        if (opResult.rows.length !== operatorIds.length) {
          return res.status(400).json({ message: "Niektórzy operatorzy nie zostali znalezieni" });
        }
      }

      const client = await pool.connect();
      try {
        await client.query('BEGIN');

        for (const workOrderId of workOrderIds) {
          // Delete existing assignments
          await client.query(
            `DELETE FROM production.work_order_operators WHERE work_order_id = $1`,
            [workOrderId]
          );

          // Add new assignments
          for (const opId of operatorIds) {
            const isPrimary = opId === primaryOperatorId || (operatorIds.length === 1 && !primaryOperatorId);
            await client.query(`
              INSERT INTO production.work_order_operators (work_order_id, operator_id, is_primary)
              VALUES ($1, $2, $3)
            `, [workOrderId, opId, isPrimary]);
          }

          // Update legacy fields
          const primaryId = primaryOperatorId || (operatorIds.length > 0 ? operatorIds[0] : null);
          if (primaryId) {
            const primaryOp = await client.query(
              `SELECT full_name FROM production.production_operators WHERE id = $1`,
              [primaryId]
            );
            await client.query(`
              UPDATE production.production_work_orders
              SET operator_id = $1, operator_name = $2, updated_at = NOW()
              WHERE id = $3
            `, [primaryId, primaryOp.rows[0]?.full_name || null, workOrderId]);
          } else {
            await client.query(`
              UPDATE production.production_work_orders
              SET operator_id = NULL, operator_name = NULL, updated_at = NOW()
              WHERE id = $1
            `, [workOrderId]);
          }
        }

        await client.query('COMMIT');

        res.json({ 
          success: true, 
          message: `Przypisano operatorów do ${workOrderIds.length} zleceń`,
          assignedCount: workOrderIds.length 
        });
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      console.error("Error bulk assigning operators:", error);
      res.status(500).json({ message: "Failed to bulk assign operators" });
    }
  });

  // GET /api/production/orders/:id/pdf - Generate PDF production card
  app.get("/api/production/orders/:id/pdf", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const pdfBuffer = await generateProductionCardPDF(pool, id);
      if (!pdfBuffer) {
        return res.status(404).json({ message: "Production order not found" });
      }

      // Get order number for filename
      const order = await ordersService.getOrderById(pool, id);
      const filename = order?.orderNumber 
        ? `${order.orderNumber.replace(/[^a-zA-Z0-9-]/g, '_')}_karta.pdf`
        : `ZLP-${id}_karta.pdf`;

      res.setHeader('Content-Type', 'application/pdf');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      res.setHeader('Content-Length', pdfBuffer.length);
      res.send(pdfBuffer);
    } catch (error) {
      console.error("Error generating production card PDF:", error);
      res.status(500).json({ message: "Failed to generate PDF" });
    }
  });

  // GET /api/production/orders/:id/csv - Generate CSV for saw machine
  app.get("/api/production/orders/:id/csv", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const csvContent = await generateSawCSV(pool, id);
      if (csvContent === null) {
        return res.status(404).json({ message: "Production order not found" });
      }

      // Get order details for filename
      const order = await ordersService.getOrderById(pool, id);
      const colorCode = order?.colorCode ? `_${order.colorCode.replace(/\s+/g, '_').toUpperCase()}` : '';
      const filename = order?.orderNumber 
        ? `${order.orderNumber.replace(/[^a-zA-Z0-9-]/g, '_')}${colorCode}.csv`
        : `ZLP-${id}.csv`;

      res.setHeader('Content-Type', 'text/csv; charset=utf-8');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      res.send(csvContent);
    } catch (error) {
      console.error("Error generating saw CSV:", error);
      res.status(500).json({ message: "Failed to generate CSV" });
    }
  });

  // GET /api/production/orders/:id/pallet-flow - Get pallet flow data (buffers, pallets, formatki counts)
  app.get("/api/production/orders/:id/pallet-flow", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const palletFlow = await ordersService.getOrderPalletFlow(pool, id);
      res.json(palletFlow);
    } catch (error) {
      console.error("Error fetching pallet flow:", error);
      res.status(500).json({ message: "Failed to fetch pallet flow data" });
    }
  });

  // GET /api/production/orders/:id/path-formatki/:pathCode - Get formatki for a specific routing path
  app.get("/api/production/orders/:id/path-formatki/:pathCode", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const pathCode = req.params.pathCode;
      
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const formatki = await ordersService.getOrderPathFormatki(pool, id, pathCode);
      res.json(formatki);
    } catch (error) {
      console.error("Error fetching path formatki:", error);
      res.status(500).json({ message: "Failed to fetch formatki for path" });
    }
  });

  // POST /api/production/orders/:id/pallets - Create a new pallet for the order
  app.post("/api/production/orders/:id/pallets", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const createPalletSchema = z.object({
        flowCode: z.string(),
        colorCode: z.string().optional(),
        operationCode: z.string().optional(),
        operationId: z.number().optional(),
        routingId: z.number().optional(),
        carrierId: z.number().optional(),
        notes: z.string().optional(),
      });

      const data = createPalletSchema.parse(req.body);
      const pallet = await ordersService.createOrderPallet(pool, id, data);
      res.status(201).json(pallet);
    } catch (error) {
      console.error("Error creating pallet:", error);
      res.status(500).json({ message: "Failed to create pallet" });
    }
  });

  // POST /api/production/orders/:id/insert-warehouse-operation - Wstaw operację magazynowania do ZLP
  app.post("/api/production/orders/:id/insert-warehouse-operation", requirePermission('view_production'), async (req, res) => {
    const client = await pool.connect();
    try {
      const orderId = parseInt(req.params.id);
      if (isNaN(orderId)) {
        client.release();
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const insertWarehouseSchema = z.object({
        afterWorkOrderId: z.number(), // Po którym work order wstawić magazynowanie
        warehouseId: z.number(), // Który magazyn
        warehouseName: z.string().optional(),
        estimatedDuration: z.number().optional(), // Szacowany czas w minutach
        notes: z.string().optional(),
      });

      const data = insertWarehouseSchema.parse(req.body);
      
      console.log('[INSERT WAREHOUSE] Request data:', JSON.stringify(data));

      // Rozpocznij transakcję
      await client.query('BEGIN');

      // Znajdź work order po którym wstawiamy
      const afterWoResult = await client.query(`
        SELECT wo.*, po.routing_id 
        FROM production.production_work_orders wo
        JOIN production.production_orders po ON wo.production_order_id = po.id
        WHERE wo.id = $1 AND wo.production_order_id = $2
        FOR UPDATE
      `, [data.afterWorkOrderId, orderId]);

      if (afterWoResult.rows.length === 0) {
        await client.query('ROLLBACK');
        client.release();
        return res.status(404).json({ message: "Work order not found" });
      }

      const afterWo = afterWoResult.rows[0];
      const newSequence = afterWo.sequence + 1;

      // Sprawdź czy następna operacja (ta która byłaby po wstawionej) jest już zakończona
      // Nie pozwalamy wstawiać magazynowania między zakończonymi operacjami
      const nextWoResult = await client.query(`
        SELECT id, status, sequence FROM production.production_work_orders
        WHERE production_order_id = $1 AND sequence = $2
      `, [orderId, newSequence]);
      
      if (nextWoResult.rows.length > 0 && nextWoResult.rows[0].status === 'done') {
        await client.query('ROLLBACK');
        client.release();
        return res.status(400).json({ 
          message: "Nie można wstawić magazynowania przed zakończoną operacją. Wybierz operację która jeszcze nie została wykonana." 
        });
      }

      // Zablokuj wszystkie rekordy, które będziemy modyfikować
      const workOrdersToShift = await client.query(`
        SELECT id, sequence FROM production.production_work_orders
        WHERE production_order_id = $1 AND sequence >= $2
        ORDER BY sequence DESC
        FOR UPDATE
      `, [orderId, newSequence]);

      // Przesuń wszystkie kolejne work orders o 1 w sekwencji
      // Musimy aktualizować od największej sekwencji do najmniejszej, żeby uniknąć konfliktów z unikalnym kluczem
      for (const wo of workOrdersToShift.rows) {
        await client.query(`
          UPDATE production.production_work_orders
          SET sequence = $1, updated_at = NOW()
          WHERE id = $2
        `, [wo.sequence + 1, wo.id]);
      }

      // Generuj numer work order
      const nextNumResult = await client.query(`
        SELECT COALESCE(MAX(CAST(SUBSTRING(work_order_number FROM 'WO-([0-9]+)') AS INTEGER)), 0) + 1 as next_num
        FROM production.production_work_orders
      `);
      const nextNum = nextNumResult.rows[0].next_num;
      const workOrderNumber = `WO-${String(nextNum).padStart(6, '0')}`;

      // Pobierz buffer_after_id z operacji poprzedzającej - będzie przeniesiony na nową operację magazynowania
      const originalBufferAfterId = afterWo.buffer_after_id;
      
      // Wstaw nowy work order dla magazynowania (z buffer_after_id z poprzedniej operacji)
      const insertResult = await client.query(`
        INSERT INTO production.production_work_orders (
          work_order_number, production_order_id, routing_operation_id,
          sequence, status, quantity_planned, estimated_duration,
          location_id, notes, buffer_after_id
        ) VALUES ($1, $2, NULL, $3, 'pending', $4, $5, $6, $7, $8)
        RETURNING *
      `, [
        workOrderNumber,
        orderId,
        newSequence,
        afterWo.quantity_planned,
        data.estimatedDuration || 60,
        data.warehouseId,
        `Magazynowanie w ${data.warehouseName || 'magazynie'}. ${data.notes || ''}`,
        originalBufferAfterId
      ]);
      
      // Wyczyść buffer_after_id z oryginalnej operacji (teraz bufor jest po magazynowaniu)
      if (originalBufferAfterId) {
        await client.query(`
          UPDATE production.production_work_orders
          SET buffer_after_id = NULL, updated_at = NOW()
          WHERE id = $1
        `, [data.afterWorkOrderId]);
      }

      // Dodaj log
      await client.query(`
        INSERT INTO production.production_order_logs (
          production_order_id, work_order_id, event_type, event_data, message
        ) VALUES ($1, $2, 'warehouse_operation_inserted', $3, $4)
      `, [
        orderId,
        insertResult.rows[0].id,
        JSON.stringify({ 
          afterWorkOrderId: data.afterWorkOrderId, 
          warehouseId: data.warehouseId,
          warehouseName: data.warehouseName 
        }),
        `Wstawiono operację magazynowania po WO #${afterWo.sequence}`
      ]);

      await client.query('COMMIT');
      client.release();

      res.json({
        success: true,
        workOrder: insertResult.rows[0],
        message: `Operacja magazynowania wstawiona po operacji #${afterWo.sequence}`
      });
    } catch (error) {
      await client.query('ROLLBACK').catch(() => {});
      client.release();
      console.error("Error inserting warehouse operation:", error);
      res.status(500).json({ message: "Failed to insert warehouse operation" });
    }
  });

  // DELETE /api/production/orders/:id/warehouse-operation/:workOrderId - Usuń zaplanowaną operację magazynowania
  app.delete("/api/production/orders/:id/warehouse-operation/:workOrderId", requirePermission('view_production'), async (req, res) => {
    try {
      const orderId = parseInt(req.params.id);
      const workOrderId = parseInt(req.params.workOrderId);
      
      if (isNaN(orderId) || isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid ID" });
      }

      // Sprawdź czy to jest warehouse operation i czy jest w statusie pending
      const woResult = await pool.query(`
        SELECT wo.*, loc.name as location_name
        FROM production.production_work_orders wo
        LEFT JOIN production.production_locations loc ON loc.id = wo.location_id
        WHERE wo.id = $1 AND wo.production_order_id = $2
      `, [workOrderId, orderId]);

      if (woResult.rows.length === 0) {
        return res.status(404).json({ message: "Work order not found" });
      }

      const wo = woResult.rows[0];

      // Sprawdź czy to warehouse operation (brak routing_operation_id, ale ma location_id)
      if (wo.routing_operation_id !== null) {
        return res.status(400).json({ message: "Można usunąć tylko operacje magazynowania" });
      }

      // Sprawdź status - tylko pending można usunąć
      if (wo.status !== 'pending') {
        return res.status(400).json({ message: "Można usunąć tylko zaplanowane operacje (status: pending)" });
      }

      const deletedSequence = wo.sequence;

      // Użyj transakcji aby bezpiecznie usunąć i przeliczyć sekwencje
      const client = await pool.connect();
      try {
        await client.query('BEGIN');
        
        // Usuń work order
        await client.query(`
          DELETE FROM production.production_work_orders
          WHERE id = $1
        `, [workOrderId]);

        // Przesuń sekwencje - aktualizuj pojedynczo od najniższej do najwyższej
        // żeby uniknąć naruszenia unikalności
        const toUpdate = await client.query(`
          SELECT id, sequence FROM production.production_work_orders
          WHERE production_order_id = $1 AND sequence > $2
          ORDER BY sequence ASC
        `, [orderId, deletedSequence]);
        
        for (const row of toUpdate.rows) {
          await client.query(`
            UPDATE production.production_work_orders
            SET sequence = $1, updated_at = NOW()
            WHERE id = $2
          `, [row.sequence - 1, row.id]);
        }
        
        await client.query('COMMIT');
      } catch (txError) {
        await client.query('ROLLBACK');
        throw txError;
      } finally {
        client.release();
      }

      // Dodaj log
      await pool.query(`
        INSERT INTO production.production_order_logs (
          production_order_id, event_type, event_data, message
        ) VALUES ($1, 'warehouse_operation_deleted', $2, $3)
      `, [
        orderId,
        JSON.stringify({ 
          workOrderId,
          workOrderNumber: wo.work_order_number,
          locationName: wo.location_name
        }),
        `Usunięto operację magazynowania: ${wo.location_name || 'magazyn'}`
      ]);

      res.json({
        success: true,
        message: `Operacja magazynowania usunięta`
      });
    } catch (error) {
      console.error("Error deleting warehouse operation:", error);
      res.status(500).json({ message: "Failed to delete warehouse operation" });
    }
  });

  // GET /api/production/orders/:id/work-orders - Get all work orders for an order
  app.get("/api/production/orders/:id/work-orders", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const result = await pool.query(`
        SELECT 
          wo.*,
          ro.name as operation_name,
          ro.operation_code,
          wc.name as work_center_name,
          loc.name as location_name
        FROM production.production_work_orders wo
        LEFT JOIN production.production_routing_operations ro ON wo.routing_operation_id = ro.id
        LEFT JOIN production.production_work_centers wc ON wo.work_center_id = wc.id
        LEFT JOIN production.production_locations loc ON wo.location_id = loc.id
        WHERE wo.production_order_id = $1
        ORDER BY wo.sequence
      `, [id]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching work orders:", error);
      res.status(500).json({ message: "Failed to fetch work orders" });
    }
  });

  // POST /api/production/orders/:orderId/formatki/warehouse-intake
  // Bulk dodawanie formatek na magazyn z tworzeniem dokumentu PZ-PROD
  const warehouseIntakeSchema = z.object({
    bomItemIds: z.array(z.number()).min(1, "Wymagana co najmniej jedna formatka"),
    workOrderId: z.number().optional(),
    targetLocationId: z.number().optional(),
    remarks: z.string().optional(),
  });

  app.post("/api/production/orders/:orderId/formatki/warehouse-intake", requirePermission('manage_production'), async (req, res) => {
    const client = await pool.connect();
    try {
      const orderId = parseInt(req.params.orderId);
      if (isNaN(orderId)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const data = warehouseIntakeSchema.parse(req.body);
      const { bomItemIds, workOrderId, targetLocationId, remarks } = data;
      const userId = (req as any).user?.id;
      const userName = (req as any).user?.username || 'System';

      await client.query('BEGIN');

      // Pobierz dane zlecenia produkcyjnego
      const orderResult = await client.query(`
        SELECT po.id, po.order_number, pob.color_code
        FROM production.production_orders po
        LEFT JOIN production.production_order_boms pob ON pob.production_order_id = po.id
        WHERE po.id = $1
      `, [orderId]);

      if (orderResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ message: "Zlecenie produkcyjne nie znalezione" });
      }

      const order = orderResult.rows[0];

      // Pobierz formatki do dodania na magazyn - z walidacją przynależności do production order
      const itemsResult = await client.query(`
        SELECT 
          bi.id,
          bi.component_name,
          bi.cz1,
          bi.length,
          bi.width,
          bi.thickness,
          bi.color_code,
          bi.quantity,
          bi.is_cut,
          bi.is_edged,
          bi.is_drilled,
          bi.last_warehouse_doc_id
        FROM production.production_order_bom_items bi
        INNER JOIN production.production_order_boms bom ON bi.production_order_bom_id = bom.id
        WHERE bi.id = ANY($1::int[])
          AND bom.production_order_id = $2
      `, [bomItemIds, orderId]);

      // Walidacja: sprawdź czy wszystkie żądane formatki zostały znalezione (należą do tego zlecenia)
      if (itemsResult.rows.length !== bomItemIds.length) {
        await client.query('ROLLBACK');
        const foundIds = new Set(itemsResult.rows.map(r => r.id));
        const missingIds = bomItemIds.filter(id => !foundIds.has(id));
        return res.status(400).json({ 
          message: "Niektóre formatki nie należą do tego zlecenia produkcyjnego",
          invalidIds: missingIds
        });
      }

      if (itemsResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(400).json({ message: "Nie znaleziono formatek o podanych ID" });
      }

      // Sprawdź czy któraś formatka już była dodana na magazyn
      const alreadyAdded = itemsResult.rows.filter(item => item.last_warehouse_doc_id !== null);
      if (alreadyAdded.length > 0) {
        await client.query('ROLLBACK');
        return res.status(400).json({ 
          message: `Niektóre formatki zostały już dodane na magazyn`,
          alreadyAddedIds: alreadyAdded.map(i => i.id)
        });
      }

      // Wygeneruj numer dokumentu PZ-PROD
      const year = new Date().getFullYear();
      const docNumResult = await client.query(`
        SELECT COUNT(*) + 1 as next_num 
        FROM warehouse.documents 
        WHERE doc_type = 'PZ-PROD' 
          AND EXTRACT(YEAR FROM created_at) = $1
      `, [year]);
      const docNumber = `PZ-PROD/${year}/${String(docNumResult.rows[0].next_num).padStart(4, '0')}`;

      // Utwórz nagłówek dokumentu PZ-PROD
      // work_order_id odnosi się do konkretnego work order (operacji), nie do production order
      const docResult = await client.query(`
        INSERT INTO warehouse.documents (
          doc_number, doc_type, status,
          work_order_id, target_location_id,
          issued_at, issued_by, issued_by_name,
          remarks, total_lines, total_quantity
        ) VALUES ($1, 'PZ-PROD', 'confirmed', $2, $3, NOW(), $4, $5, $6, $7, $8)
        RETURNING id
      `, [
        docNumber,
        workOrderId || null, // Prawidłowe work_order_id (z operacji), nie production_order_id
        targetLocationId || null,
        userId,
        userName,
        remarks || `Przyjęcie formatek z ZLP ${order.order_number}`,
        itemsResult.rows.length,
        itemsResult.rows.reduce((sum, item) => sum + parseFloat(item.quantity || 1), 0)
      ]);

      const documentId = docResult.rows[0].id;
      let lineNumber = 0;
      
      // Mapa do śledzenia bieżącej ilości dla każdego stock_panel_id
      // Gdy wiele formatek aktualizuje ten sam panel, musimy śledzić running quantity
      const runningQuantities: Map<number, number> = new Map();

      // Przetwórz każdą formatkę
      for (const item of itemsResult.rows) {
        lineNumber++;

        // Generowanie nazwy formatki w magazynie
        // Po cięciu/oklejaniu: "długość x szerokość - kolor"
        // Po wierceniu: pełna nazwa z cz1 i cz2 np. "BOK-L-NADST-1000x180-ARTISAN"
        let generatedName: string;
        const lengthVal = parseFloat(item.length || 0);
        const widthVal = parseFloat(item.width || 0);
        const colorCode = item.color_code || order.color_code || '';

        // Wyodrębnij CZ2 z component_name (np. "BOK-L-NADST-1183x280-BIALY" -> "NADST")
        let cz2 = '';
        if (item.component_name && item.cz1) {
          // Usuń prefix cz1 z component_name
          const withoutCz1 = item.component_name.substring(item.cz1.length + 1); // +1 dla myślnika
          // Wyodrębnij część przed wymiarami (szukaj wzorca liczba x liczba)
          const match = withoutCz1.match(/^([A-Z0-9-]+?)-\d+x\d+/i);
          if (match && match[1]) {
            cz2 = match[1];
          }
        }

        if (item.is_drilled && item.cz1) {
          // Po wierceniu - pełna nazwa z cz1 i cz2
          if (cz2) {
            generatedName = `${item.cz1}-${cz2}-${lengthVal}x${widthVal}-${colorCode}`;
          } else {
            generatedName = `${item.cz1}-${lengthVal}x${widthVal}-${colorCode}`;
          }
        } else {
          // Po cięciu lub oklejaniu - wymiary i kolor
          generatedName = `${lengthVal}x${widthVal}-${colorCode}`;
        }

        // Sprawdź czy taka formatka już istnieje w stock_panels
        // Sprawdzamy tylko po generated_name, która zawiera wymiary i kolor
        const existingPanel = await client.query(`
          SELECT id, quantity FROM warehouse.stock_panels
          WHERE generated_name = $1
            AND is_active = true
            AND is_archived = false
          LIMIT 1
        `, [generatedName]);

        let stockPanelId: number;
        const qty = parseInt(item.quantity || 1);

        let quantityBefore = 0;
        
        if (existingPanel.rows.length > 0) {
          // Aktualizuj ilość istniejącej formatki
          stockPanelId = existingPanel.rows[0].id;
          
          // Użyj running quantity jeśli już przetwarzaliśmy ten panel w tej transakcji
          if (runningQuantities.has(stockPanelId)) {
            quantityBefore = runningQuantities.get(stockPanelId)!;
          } else {
            quantityBefore = parseFloat(existingPanel.rows[0].quantity || 0);
          }
          
          await client.query(`
            UPDATE warehouse.stock_panels
            SET quantity = quantity + $1,
                updated_at = NOW(),
                updated_by = $2,
                location_id = COALESCE($3, location_id)
            WHERE id = $4
          `, [qty, userName, targetLocationId || null, stockPanelId]);
          
          // Zaktualizuj running quantity dla tego panelu
          runningQuantities.set(stockPanelId, quantityBefore + qty);
        } else {
          // Utwórz nową formatkę w magazynie
          const insertResult = await client.query(`
            INSERT INTO warehouse.stock_panels (
              generated_name, cz1, length, width, thickness,
              color_code, is_drilled, is_edged,
              source, source_reference, quantity,
              location_id, is_active, updated_by
            ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, 'production', $9, $10, $11, true, $12)
            RETURNING id
          `, [
            generatedName,
            item.cz1,
            lengthVal,
            widthVal,
            parseFloat(item.thickness || 18),
            colorCode,
            item.is_drilled || false,
            item.is_edged || false,
            `ZLP-${order.order_number}`,
            qty,
            targetLocationId || null,
            userName
          ]);
          stockPanelId = insertResult.rows[0].id;
          
          // Zapisz running quantity dla nowego panelu (gdyby kolejna formatka miała tę samą nazwę)
          runningQuantities.set(stockPanelId, qty);
        }
        
        // Zapisz do historii zmian magazynowych
        await client.query(`
          INSERT INTO warehouse.inventory_history (
            stock_panel_id, operation_type, quantity_change,
            quantity_before, quantity_after,
            production_order_id, document_number,
            notes, performed_by
          ) VALUES ($1, 'PZ-PROD', $2, $3, $4, $5, $6, $7, $8)
        `, [
          stockPanelId,
          qty,
          quantityBefore,
          quantityBefore + qty,
          orderId,
          docNumber,
          `Przyjęcie z produkcji: ${generatedName}`,
          userName
        ]);

        // Dodaj linię dokumentu
        await client.query(`
          INSERT INTO warehouse.document_lines (
            document_id, line_number, source_type, source_id,
            product_name, quantity_requested, quantity_picked,
            unit, is_picked, picked_at
          ) VALUES ($1, $2, 'formatka', $3, $4, $5, $6, 'szt', true, NOW())
        `, [
          documentId,
          lineNumber,
          stockPanelId,
          generatedName,
          qty,
          qty
        ]);

        // Aktualizuj BOM item z ID dokumentu
        await client.query(`
          UPDATE production.production_order_bom_items
          SET last_warehouse_doc_id = $1, updated_at = NOW()
          WHERE id = $2
        `, [documentId, item.id]);
      }

      // Dodaj log
      await client.query(`
        INSERT INTO production.production_order_logs (
          production_order_id, event_type, event_data, message, user_id
        ) VALUES ($1, 'formatki_warehouse_intake', $2, $3, $4)
      `, [
        orderId,
        JSON.stringify({ 
          documentId,
          docNumber,
          itemCount: itemsResult.rows.length,
          bomItemIds
        }),
        `Dodano ${itemsResult.rows.length} formatek na magazyn (${docNumber})`,
        userId
      ]);

      // Check if all formatki have been added to warehouse - auto-complete ZLP if so
      // Count total formatki (non-damaged) and those already added to warehouse
      // Use COALESCE to treat NULL is_damaged as false (not damaged)
      const formatkiStatusResult = await client.query(`
        SELECT 
          COUNT(*) FILTER (WHERE COALESCE(bi.is_damaged, false) = false) as total_formatki,
          COUNT(*) FILTER (WHERE COALESCE(bi.is_damaged, false) = false AND bi.last_warehouse_doc_id IS NOT NULL) as added_to_warehouse
        FROM production.production_order_bom_items bi
        INNER JOIN production.production_order_boms bom ON bi.production_order_bom_id = bom.id
        WHERE bom.production_order_id = $1
      `, [orderId]);

      const totalFormatki = parseInt(formatkiStatusResult.rows[0].total_formatki || 0);
      const addedToWarehouse = parseInt(formatkiStatusResult.rows[0].added_to_warehouse || 0);
      let orderAutoCompleted = false;

      // If all non-damaged formatki have been added to warehouse, auto-complete the production order
      if (totalFormatki > 0 && addedToWarehouse >= totalFormatki) {
        // Check current order status - only complete if not already done/cancelled
        const orderStatusResult = await client.query(`
          SELECT status FROM production.production_orders WHERE id = $1
        `, [orderId]);

        const currentStatus = orderStatusResult.rows[0]?.status;
        if (currentStatus && currentStatus !== 'done' && currentStatus !== 'cancelled') {
          await client.query(`
            UPDATE production.production_orders
            SET 
              status = 'done',
              actual_end_date = NOW(),
              quantity_produced = quantity_planned,
              updated_at = NOW()
            WHERE id = $1
          `, [orderId]);

          await client.query(`
            INSERT INTO production.production_order_logs 
            (production_order_id, event_type, event_data, message, user_id, created_at)
            VALUES ($1, 'order_auto_completed', $2, $3, $4, NOW())
          `, [
            orderId,
            JSON.stringify({ 
              reason: 'all_formatki_added_to_warehouse',
              totalFormatki,
              addedToWarehouse
            }),
            `ZLP automatycznie zakończone - wszystkie formatki (${totalFormatki}) dodane na magazyn`,
            userId
          ]);

          orderAutoCompleted = true;
          console.log(`✅ Auto-completed production order ${orderId} - all ${totalFormatki} formatki added to warehouse`);
        }
      }

      await client.query('COMMIT');

      res.json({
        success: true,
        message: orderAutoCompleted 
          ? `Dodano ${itemsResult.rows.length} formatek na magazyn. ZLP automatycznie zakończone!`
          : `Dodano ${itemsResult.rows.length} formatek na magazyn`,
        documentId,
        docNumber,
        itemsProcessed: itemsResult.rows.length,
        orderAutoCompleted,
        formatkiStatus: {
          total: totalFormatki,
          addedToWarehouse
        }
      });

    } catch (error) {
      await client.query('ROLLBACK');
      console.error("Error adding formatki to warehouse:", error);
      
      if (error instanceof z.ZodError) {
        return res.status(400).json({ 
          message: "Nieprawidłowe dane wejściowe", 
          errors: error.errors 
        });
      }
      
      res.status(500).json({ message: "Błąd podczas dodawania formatek na magazyn" });
    } finally {
      client.release();
    }
  });

  // GET /api/production/orders/:id/summaries - Get damage and warehouse intake summaries
  app.get("/api/production/orders/:id/summaries", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      // Query formatki statistics
      const formatkiStatsResult = await pool.query(`
        SELECT 
          COUNT(*) as total,
          COUNT(*) FILTER (WHERE COALESCE(bi.is_damaged, false) = false) as non_damaged,
          COUNT(*) FILTER (WHERE COALESCE(bi.is_damaged, false) = false AND bi.last_warehouse_doc_id IS NOT NULL) as in_warehouse,
          COUNT(*) FILTER (WHERE bi.is_damaged = true) as damaged,
          COUNT(*) FILTER (WHERE COALESCE(bi.is_damaged, false) = false AND bi.last_warehouse_doc_id IS NULL) as pending
        FROM production.production_order_bom_items bi
        INNER JOIN production.production_order_boms bom ON bi.production_order_bom_id = bom.id
        WHERE bom.production_order_id = $1
      `, [id]);

      const formatkiStats = {
        total: parseInt(formatkiStatsResult.rows[0]?.total || 0),
        nonDamaged: parseInt(formatkiStatsResult.rows[0]?.non_damaged || 0),
        inWarehouse: parseInt(formatkiStatsResult.rows[0]?.in_warehouse || 0),
        damaged: parseInt(formatkiStatsResult.rows[0]?.damaged || 0),
        pending: parseInt(formatkiStatsResult.rows[0]?.pending || 0),
      };

      // Query order completion info
      const orderResult = await pool.query(`
        SELECT 
          status,
          actual_end_date,
          quantity_produced,
          quantity_planned
        FROM production.production_orders
        WHERE id = $1
      `, [id]);

      const orderInfo = orderResult.rows[0] ? {
        status: orderResult.rows[0].status,
        actualEndDate: orderResult.rows[0].actual_end_date,
        quantityProduced: parseFloat(orderResult.rows[0].quantity_produced || 0),
        quantityPlanned: parseFloat(orderResult.rows[0].quantity_planned || 0),
        isCompleted: orderResult.rows[0].status === 'done',
        isAutoCompleted: formatkiStats.nonDamaged > 0 && formatkiStats.inWarehouse >= formatkiStats.nonDamaged && formatkiStats.pending === 0 && orderResult.rows[0].status === 'done',
      } : null;

      // Query damaged items with details (no work order join to avoid duplicates)
      const damagedResult = await pool.query(`
        SELECT 
          bi.id,
          bi.component_name,
          bi.damage_type,
          bi.damage_notes,
          bi.damaged_at,
          u.username as damaged_by_username,
          u.full_name as damaged_by_name
        FROM production.production_order_bom_items bi
        INNER JOIN production.production_order_boms bom ON bi.production_order_bom_id = bom.id
        LEFT JOIN users u ON bi.damaged_by_user_id = u.id
        WHERE bom.production_order_id = $1
          AND bi.is_damaged = true
        ORDER BY bi.damaged_at DESC
      `, [id]);

      // Query warehouse intake documents for this order
      const warehouseResult = await pool.query(`
        SELECT DISTINCT
          d.id as document_id,
          d.doc_number,
          d.issued_at,
          d.issued_by_name,
          d.total_lines,
          d.total_quantity,
          d.remarks,
          wo.work_order_number,
          ro.name as operation_name,
          ro.code as operation_code
        FROM warehouse.documents d
        LEFT JOIN production.production_work_orders wo ON d.work_order_id = wo.id
        LEFT JOIN production.production_routing_operations ro ON wo.routing_operation_id = ro.id
        WHERE d.doc_type = 'PZ-PROD'
          AND (
            wo.production_order_id = $1
            OR d.id IN (
              SELECT DISTINCT bi.last_warehouse_doc_id
              FROM production.production_order_bom_items bi
              INNER JOIN production.production_order_boms bom ON bi.production_order_bom_id = bom.id
              WHERE bom.production_order_id = $1
                AND bi.last_warehouse_doc_id IS NOT NULL
            )
          )
        ORDER BY d.issued_at DESC
      `, [id]);

      res.json({
        formatkiStats,
        orderInfo,
        damageSummary: damagedResult.rows.map(row => ({
          id: row.id,
          componentName: row.component_name,
          damageType: row.damage_type,
          damageNotes: row.damage_notes,
          damagedAt: row.damaged_at,
          damagedByUsername: row.damaged_by_username,
          damagedByName: row.damaged_by_name,
        })),
        warehouseIntakeSummary: warehouseResult.rows.map(row => ({
          documentId: row.document_id,
          docNumber: row.doc_number,
          issuedAt: row.issued_at,
          issuedByName: row.issued_by_name,
          totalLines: row.total_lines || 0,
          totalQuantity: row.total_quantity ? parseFloat(row.total_quantity) : 0,
          remarks: row.remarks,
          workOrderNumber: row.work_order_number,
          operationName: row.operation_name,
          operationCode: row.operation_code,
        })),
      });
    } catch (error) {
      console.error("Error fetching order summaries:", error);
      res.status(500).json({ message: "Failed to fetch order summaries" });
    }
  });

  // GET /api/production/plans/:planId/assembly-operations - Get plan-level assembly operations
  app.get("/api/production/plans/:planId/assembly-operations", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const result = await ordersService.getPlanAssemblyOperations(pool, planId);
      if (!result) {
        return res.status(404).json({ message: "Plan not found or has no ZLPs" });
      }

      res.json(result);
    } catch (error) {
      console.error("Error fetching plan assembly operations:", error);
      res.status(500).json({ message: "Failed to fetch plan assembly operations" });
    }
  });

  // POST /api/production/plans/:planId/assembly-operations/:operationCode/start - Batch start plan assembly operation
  app.post("/api/production/plans/:planId/assembly-operations/:operationCode/start", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      const operationCode = req.params.operationCode;
      
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const result = await ordersService.batchStartPlanAssemblyOperation(pool, planId, operationCode);
      
      if (result.startedCount === 0) {
        return res.status(400).json({ message: "No pending work orders to start" });
      }

      res.json({
        message: `Started ${result.startedCount} work orders for ${operationCode}`,
        ...result
      });
    } catch (error) {
      console.error("Error starting plan assembly operation:", error);
      res.status(500).json({ message: "Failed to start plan assembly operation" });
    }
  });

  // POST /api/production/plans/:planId/assembly-operations/:operationCode/complete - Batch complete plan assembly operation
  app.post("/api/production/plans/:planId/assembly-operations/:operationCode/complete", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      const operationCode = req.params.operationCode;
      
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const result = await ordersService.batchCompletePlanAssemblyOperation(pool, planId, operationCode);
      
      if (result.completedCount === 0) {
        return res.status(400).json({ message: "No in-progress work orders to complete" });
      }

      res.json({
        message: `Completed ${result.completedCount} work orders for ${operationCode}`,
        ...result
      });
    } catch (error) {
      console.error("Error completing plan assembly operation:", error);
      res.status(500).json({ message: "Failed to complete plan assembly operation" });
    }
  });

}
