import type { Express } from "express";
import { requirePermission } from "../../auth";
import { z } from "zod";
import { pool } from "../../postgres";
import { 
  getCarrierGroups, 
  getCarrierGroupById, 
  createCarrierGroup, 
  updateCarrierGroup, 
  deleteCarrierGroup,
  getCarriers,
  getCarrierById,
  createCarrier,
  updateCarrier,
  deleteCarrier
} from "../../services/production/carriers";
import { insertProductionCarrierGroupSchema, insertProductionCarrierSchema } from "@shared/schema";

const partialCarrierGroupSchema = insertProductionCarrierGroupSchema.partial();
const partialCarrierSchema = insertProductionCarrierSchema.partial();

export function setupCarrierRoutes(app: Express) {
  // Carrier Groups
  app.get("/api/production/carriers/groups", requirePermission("view_production"), async (req, res) => {
    try {
      const groups = await getCarrierGroups(pool);
      res.json(groups);
    } catch (error: any) {
      console.error("Error fetching carrier groups:", error);
      res.status(500).json({ error: "Failed to fetch carrier groups" });
    }
  });

  app.get("/api/production/carriers/groups/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const group = await getCarrierGroupById(pool, id);
      if (!group) {
        return res.status(404).json({ error: "Carrier group not found" });
      }
      res.json(group);
    } catch (error: any) {
      console.error("Error fetching carrier group:", error);
      res.status(500).json({ error: "Failed to fetch carrier group" });
    }
  });

  app.post("/api/production/carriers/groups", requirePermission("manage_production"), async (req, res) => {
    try {
      const validated = insertProductionCarrierGroupSchema.parse(req.body);
      const group = await createCarrierGroup(pool, validated);
      res.status(201).json(group);
    } catch (error: any) {
      console.error("Error creating carrier group:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to create carrier group" });
    }
  });

  app.patch("/api/production/carriers/groups/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const validated = partialCarrierGroupSchema.parse(req.body);
      const group = await updateCarrierGroup(pool, id, validated);
      if (!group) {
        return res.status(404).json({ error: "Carrier group not found" });
      }
      res.json(group);
    } catch (error: any) {
      console.error("Error updating carrier group:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to update carrier group" });
    }
  });

  app.delete("/api/production/carriers/groups/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      await deleteCarrierGroup(pool, id);
      res.status(204).send();
    } catch (error: any) {
      console.error("Error deleting carrier group:", error);
      res.status(500).json({ error: "Failed to delete carrier group" });
    }
  });

  // Carriers
  app.get("/api/production/carriers", requirePermission("view_production"), async (req, res) => {
    try {
      const groupId = req.query.groupId ? parseInt(req.query.groupId as string) : undefined;
      const carriers = await getCarriers(pool, groupId);
      res.json(carriers);
    } catch (error: any) {
      console.error("Error fetching carriers:", error);
      res.status(500).json({ error: "Failed to fetch carriers" });
    }
  });

  app.get("/api/production/carriers/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const carrier = await getCarrierById(pool, id);
      if (!carrier) {
        return res.status(404).json({ error: "Carrier not found" });
      }
      res.json(carrier);
    } catch (error: any) {
      console.error("Error fetching carrier:", error);
      res.status(500).json({ error: "Failed to fetch carrier" });
    }
  });

  app.post("/api/production/carriers", requirePermission("manage_production"), async (req, res) => {
    try {
      const validated = insertProductionCarrierSchema.parse(req.body);
      const carrier = await createCarrier(pool, validated);
      res.status(201).json(carrier);
    } catch (error: any) {
      console.error("Error creating carrier:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to create carrier" });
    }
  });

  app.patch("/api/production/carriers/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const validated = partialCarrierSchema.parse(req.body);
      const carrier = await updateCarrier(pool, id, validated);
      if (!carrier) {
        return res.status(404).json({ error: "Carrier not found" });
      }
      res.json(carrier);
    } catch (error: any) {
      console.error("Error updating carrier:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to update carrier" });
    }
  });

  app.delete("/api/production/carriers/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      await deleteCarrier(pool, id);
      res.status(204).send();
    } catch (error: any) {
      console.error("Error deleting carrier:", error);
      res.status(500).json({ error: "Failed to delete carrier" });
    }
  });

  // Assign carrier to work order
  app.patch("/api/production/work-orders/:id/carrier", requirePermission("manage_production"), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      const { carrierId } = req.body;
      
      if (isNaN(workOrderId)) {
        return res.status(400).json({ error: "Invalid work order ID" });
      }
      
      const result = await pool.query(`
        UPDATE production.production_work_orders
        SET carrier_id = $2, updated_at = NOW()
        WHERE id = $1
        RETURNING id, carrier_id
      `, [workOrderId, carrierId || null]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Work order not found" });
      }
      
      // If carrierId is set, get carrier details
      let carrier = null;
      if (carrierId) {
        const carrierResult = await pool.query(`
          SELECT c.id, c.code, c.name, cg.name as group_name
          FROM production.production_carriers c
          LEFT JOIN production.production_carrier_groups cg ON c.carrier_group_id = cg.id
          WHERE c.id = $1
        `, [carrierId]);
        carrier = carrierResult.rows[0] || null;
      }
      
      res.json({
        workOrderId: result.rows[0].id,
        carrierId: result.rows[0].carrier_id,
        carrier,
      });
    } catch (error: any) {
      console.error("Error assigning carrier to work order:", error);
      res.status(500).json({ error: "Failed to assign carrier" });
    }
  });

  // Get available carriers (not assigned to any in-progress work order)
  app.get("/api/production/carriers/available", requirePermission("view_production"), async (req, res) => {
    try {
      // Get carriers that are either not assigned to any work order,
      // or only assigned to completed/cancelled work orders
      const result = await pool.query(`
        SELECT 
          c.id, c.code, c.name, c.barcode, c.status, c.capacity, c.capacity_unit,
          cg.name as group_name, cg.code as group_code
        FROM production.production_carriers c
        LEFT JOIN production.production_carrier_groups cg ON c.carrier_group_id = cg.id
        WHERE c.is_active = true
          AND c.status IN ('available', 'in_use')
          AND c.id NOT IN (
            SELECT DISTINCT wo.carrier_id 
            FROM production.production_work_orders wo
            WHERE wo.carrier_id IS NOT NULL 
              AND wo.status IN ('pending', 'ready', 'in_progress', 'paused')
          )
        ORDER BY cg.name, c.code
      `);
      
      res.json(result.rows.map(r => ({
        id: r.id,
        code: r.code,
        name: r.name,
        barcode: r.barcode,
        status: r.status,
        capacity: r.capacity,
        capacityUnit: r.capacity_unit,
        groupName: r.group_name,
        groupCode: r.group_code,
      })));
    } catch (error: any) {
      console.error("Error fetching available carriers:", error);
      res.status(500).json({ error: "Failed to fetch available carriers" });
    }
  });
}
