import { pool } from "./postgres";

async function migrateColors() {
  console.log('🔄 Rozpoczynam migrację kolorów...');
  
  const client = await pool.connect();
  
  try {
    // Pobierz wszystkie matryce
    const result = await client.query(`
      SELECT id, colors, colors_v2, color_images, color_options, selected_colors
      FROM product_creator.product_matrices
    `);
    
    for (const row of result.rows) {
      const { id, colors, colors_v2, color_images, color_options, selected_colors } = row;
      
      if (!colors_v2 || colors_v2.length === 0) {
        console.log(`⏩ Matrix ID ${id}: brak kolorów, pomijam`);
        continue;
      }
      
      // Stwórz mapę: nazwa koloru -> ID
      const colorNameToId: Record<string, string> = {};
      for (const colorObj of colors_v2) {
        colorNameToId[colorObj.name] = colorObj.id;
      }
      
      console.log(`📝 Matrix ID ${id}: mapowanie ${Object.keys(colorNameToId).length} kolorów`);
      
      // Przekonwertuj color_images
      const newColorImages: Record<string, any> = {};
      if (color_images && typeof color_images === 'object') {
        for (const [colorName, images] of Object.entries(color_images)) {
          const colorId = colorNameToId[colorName];
          if (colorId) {
            newColorImages[colorId] = images;
          }
        }
      }
      
      // Przekonwertuj color_options
      const newColorOptions: Record<string, any> = {};
      if (color_options && typeof color_options === 'object') {
        for (const [colorName, options] of Object.entries(color_options)) {
          const colorId = colorNameToId[colorName];
          if (colorId) {
            newColorOptions[colorId] = options;
          }
        }
      }
      
      // Przekonwertuj selected_colors
      const newSelectedColors: string[] = [];
      if (selected_colors && Array.isArray(selected_colors)) {
        for (const colorName of selected_colors) {
          const colorId = colorNameToId[colorName];
          if (colorId) {
            newSelectedColors.push(colorId);
          }
        }
      }
      
      // Zapisz przekonwertowane dane
      await client.query(`
        UPDATE product_creator.product_matrices
        SET 
          color_images_v2 = $1,
          color_options_v2 = $2,
          selected_colors_v2 = $3
        WHERE id = $4
      `, [
        JSON.stringify(newColorImages),
        JSON.stringify(newColorOptions),
        JSON.stringify(newSelectedColors),
        id
      ]);
      
      console.log(`✅ Matrix ID ${id}: migracja zakończona`);
    }
    
    console.log('\n🎯 Migracja zakończona pomyślnie!');
    console.log('Następny krok: Usuń stare kolumny i zmień nazwy nowych kolumn.');
    
  } catch (error) {
    console.error('❌ Błąd podczas migracji:', error);
    throw error;
  } finally {
    client.release();
  }
}

// Uruchom migrację
migrateColors()
  .then(() => {
    console.log('\n✨ Gotowe!');
    process.exit(0);
  })
  .catch((error) => {
    console.error('\n💥 Błąd:', error);
    process.exit(1);
  });
