import sharp from 'sharp';

/**
 * Image processing utility for generating optimized thumbnails
 * Generates multiple versions: thumbnail (80x80), medium (400x400), and preserves original
 */

export interface ImageVersions {
  original: Buffer;
  thumbnail: Buffer;
  medium: Buffer;
}

export interface ProcessedImage {
  buffers: ImageVersions;
  format: 'webp' | 'jpeg' | 'png';
}

/**
 * Process an image buffer to generate multiple optimized versions
 * @param buffer - Original image buffer
 * @param options - Processing options
 * @returns Processed image with multiple versions
 */
export async function processImage(
  buffer: Buffer,
  options: {
    thumbnailSize?: number;
    mediumSize?: number;
    quality?: number;
    format?: 'webp' | 'jpeg' | 'png';
  } = {}
): Promise<ProcessedImage> {
  const {
    thumbnailSize = 80,
    mediumSize = 400,
    quality = 85,
    format = 'webp'
  } = options;

  try {
    // Create sharp instance from buffer
    const image = sharp(buffer);
    const metadata = await image.metadata();

    // Determine output format (prefer WebP for better compression)
    const outputFormat = format;

    // Generate thumbnail (80x80 by default) - cover mode to fill the square
    const thumbnailBuffer = await sharp(buffer)
      .resize(thumbnailSize, thumbnailSize, {
        fit: 'cover',
        position: 'center'
      })
      .toFormat(outputFormat, { quality })
      .toBuffer();

    // Generate medium size (400x400 by default) - inside mode to preserve aspect ratio
    const mediumBuffer = await sharp(buffer)
      .resize(mediumSize, mediumSize, {
        fit: 'inside',
        withoutEnlargement: true
      })
      .toFormat(outputFormat, { quality })
      .toBuffer();

    // Convert original to WebP for storage optimization
    const originalBuffer = await sharp(buffer)
      .toFormat(outputFormat, { quality: 90 })
      .toBuffer();

    return {
      buffers: {
        original: originalBuffer,
        thumbnail: thumbnailBuffer,
        medium: mediumBuffer
      },
      format: outputFormat
    };
  } catch (error) {
    console.error('❌ Error processing image:', error);
    throw new Error('Failed to process image');
  }
}

/**
 * Get file extension for image format
 */
export function getExtensionForFormat(format: 'webp' | 'jpeg' | 'png'): string {
  switch (format) {
    case 'webp':
      return '.webp';
    case 'jpeg':
      return '.jpg';
    case 'png':
      return '.png';
    default:
      return '.webp';
  }
}

/**
 * Generate filenames for different image versions
 */
export function generateImageVersionFilenames(baseFilename: string, format: 'webp' | 'jpeg' | 'png') {
  // Remove existing extension
  const nameWithoutExt = baseFilename.replace(/\.[^/.]+$/, '');
  const ext = getExtensionForFormat(format);

  return {
    original: `${nameWithoutExt}${ext}`,
    thumbnail: `${nameWithoutExt}_thumb${ext}`,
    medium: `${nameWithoutExt}_medium${ext}`
  };
}
