import axios from 'axios';
import * as fs from 'fs/promises';
import * as path from 'path';
import { pool } from './postgres.js';

async function downloadAndSaveImage(imageUrl: string, externalId: string): Promise<string | null> {
  try {
    console.log(`📥 Pobieranie: ${imageUrl}`);
    
    const response = await axios.get(imageUrl, {
      responseType: 'arraybuffer',
      timeout: 10000,
    });

    const productsDir = path.join(process.cwd(), 'attached_assets', 'products');
    await fs.mkdir(productsDir, { recursive: true });

    const extension = imageUrl.endsWith('.png') ? 'png' : 'jpg';
    const filename = `${externalId}_1.${extension}`;
    const filepath = path.join(productsDir, filename);

    await fs.writeFile(filepath, response.data);
    console.log(`✅ Zapisano: ${filename}`);

    return `/api/products/images/${filename}`;
  } catch (error: any) {
    console.error(`❌ Błąd dla ${externalId}:`, error.message);
    return null;
  }
}

async function fixExternalImages() {
  try {
    console.log('🔄 Naprawa zewnętrznych URL-i obrazków...\n');

    // Pobierz produkty z zewnętrznymi URL-ami
    const result = await pool.query(`
      SELECT offer_external_id, image_url
      FROM commerce.marketplace_products
      WHERE image_url LIKE 'http%'
    `);

    console.log(`📊 Znaleziono ${result.rows.length} produktów z zewnętrznymi URL-ami\n`);

    let success = 0;
    let failed = 0;

    for (const product of result.rows) {
      const localPath = await downloadAndSaveImage(product.image_url, product.offer_external_id);
      
      if (localPath) {
        // Zaktualizuj marketplace_products
        await pool.query(`
          UPDATE commerce.marketplace_products
          SET image_url = $1, updated_at = NOW()
          WHERE offer_external_id = $2
        `, [localPath, product.offer_external_id]);

        // Zaktualizuj order_items
        await pool.query(`
          UPDATE commerce.order_items
          SET image_url = $1
          WHERE offer_external_id = $2
        `, [localPath, product.offer_external_id]);

        success++;
      } else {
        failed++;
      }

      // Throttle
      await new Promise(resolve => setTimeout(resolve, 200));
    }

    console.log(`\n✅ Zakończono: ${success} sukces, ${failed} niepowodzeń`);
    
  } catch (error) {
    console.error('❌ Błąd:', error);
  } finally {
    await pool.end();
  }
}

fixExternalImages();
