import SftpClient from 'ssh2-sftp-client';
import { pool } from "./postgres";
import fs from 'fs';
import path from 'path';

export interface FileUploadOptions {
  filename: string;
  buffer: Buffer;
  mimetype: string;
  subfolder: string; // e.g., 'products/images', 'accessories/images', 'exports/pdf'
  metadata?: {
    productId?: string;
    timestamp?: string;
    [key: string]: any;
  };
}

export interface FileStorageAdapter {
  upload(options: FileUploadOptions): Promise<string>; // Returns URL
  delete(url: string): Promise<void>;
  exists(url: string): Promise<boolean>;
  getUrl(path: string): string;
  getBuffer(url: string): Promise<Buffer>; // Download file as buffer
}

// ===== LOCAL STORAGE ADAPTER =====
export class LocalStorageAdapter implements FileStorageAdapter {
  private uploadsDir = path.join(process.cwd(), 'uploads');
  private baseUrl = 'https://files.alpsys.pl';
  private pathPrefix = 'OMS';

  constructor() {
    // Ensure uploads directory exists
    if (!fs.existsSync(this.uploadsDir)) {
      fs.mkdirSync(this.uploadsDir, { recursive: true });
    }
  }

  async upload(options: FileUploadOptions): Promise<string> {
    const { filename, buffer, subfolder } = options;

    const fullPath = path.join(this.uploadsDir, subfolder);
    
    // Ensure directory exists
    if (!fs.existsSync(fullPath)) {
      fs.mkdirSync(fullPath, { recursive: true });
    }

    // Write file
    const filePath = path.join(fullPath, filename);
    fs.writeFileSync(filePath, buffer);

    // Return full URL (consistent with SFTP adapter)
    const relativePath = `${subfolder}/${filename}`.replace(/\\/g, '/');
    return `${this.baseUrl}/${this.pathPrefix}/${relativePath}`.replace(/([^:]\/)\/+/g, '$1');
  }

  async delete(url: string): Promise<void> {
    // Extract path from URL (remove /uploads/ prefix)
    const relativePath = url.replace(/^\/uploads\//, '');
    const fullPath = path.join(this.uploadsDir, relativePath);

    if (fs.existsSync(fullPath)) {
      fs.unlinkSync(fullPath);
    }
  }

  async exists(url: string): Promise<boolean> {
    // Handle both relative and absolute URLs
    let relativePath: string;
    if (url.startsWith('http://') || url.startsWith('https://')) {
      // Extract path from full URL
      relativePath = url
        .replace(this.baseUrl, '')
        .replace(`/${this.pathPrefix}/`, '')
        .replace(/^\/uploads\//, '');
    } else {
      relativePath = url.replace(/^\/uploads\//, '');
    }
    
    const fullPath = path.join(this.uploadsDir, relativePath);
    return fs.existsSync(fullPath);
  }

  getUrl(filePath: string): string {
    // Return full URL (consistent with SFTP adapter)
    const normalizedPath = filePath.replace(/\\/g, '/').replace(/^\/+/, '');
    return `${this.baseUrl}/${this.pathPrefix}/${normalizedPath}`.replace(/([^:]\/)\/+/g, '$1');
  }

  async getBuffer(url: string): Promise<Buffer> {
    // Handle both relative and absolute URLs
    let relativePath: string;
    if (url.startsWith('http://') || url.startsWith('https://')) {
      // Extract path from full URL
      relativePath = url
        .replace(this.baseUrl, '')
        .replace(`/${this.pathPrefix}/`, '')
        .replace(/^\/uploads\//, '');
    } else {
      relativePath = url.replace(/^\/uploads\//, '');
    }
    
    const fullPath = path.join(this.uploadsDir, relativePath);
    
    if (!fs.existsSync(fullPath)) {
      throw new Error(`File not found: ${fullPath}`);
    }
    
    return fs.readFileSync(fullPath);
  }
}

// ===== SFTP STORAGE ADAPTER =====
export class SFTPStorageAdapter implements FileStorageAdapter {
  private config: {
    host: string;
    port: number;
    username: string;
    password: string;
    baseUrl: string;
    pathPrefix: string;
  };

  constructor(config: {
    host: string;
    port: number;
    username: string;
    password: string;
    baseUrl: string;
    pathPrefix: string;
  }) {
    this.config = config;
  }

  private async getClient(): Promise<SftpClient> {
    const sftp = new SftpClient();
    await sftp.connect({
      host: this.config.host,
      port: this.config.port,
      username: this.config.username,
      password: this.config.password,
    });
    return sftp;
  }

  async upload(options: FileUploadOptions): Promise<string> {
    const { filename, buffer, subfolder } = options;

    // CRITICAL FIX: SFTP user's home directory IS already the pathPrefix folder
    // So we should NOT add pathPrefix to SFTP paths - only use it for public URLs
    // If SFTP user home = /var/www/files/OMS, then pathPrefix should only be used in getUrl()
    const remotePath = `${subfolder}/${filename}`.replace(/\/+/g, '/');
    const remoteDir = path.dirname(remotePath);

    console.log('🔍 [SFTP Upload Debug]:');
    console.log('  - pathPrefix (for URL only):', this.config.pathPrefix);
    console.log('  - subfolder:', subfolder);
    console.log('  - filename:', filename);
    console.log('  - remotePath (SFTP):', remotePath);
    console.log('  - remoteDir (SFTP):', remoteDir);

    const sftp = await this.getClient();

    try {
      // Ensure remote directory exists
      await sftp.mkdir(remoteDir, true);

      // Upload file from buffer
      await sftp.put(buffer, remotePath);

      // Return public URL with pathPrefix
      const normalizedPrefix = this.config.pathPrefix.replace(/^\/+/, '');
      const publicUrl = `${this.config.baseUrl}/${normalizedPrefix}/${remotePath}`.replace(/([^:]\/)\/+/g, '$1');
      console.log('  - publicUrl:', publicUrl);
      return publicUrl;
    } finally {
      await sftp.end();
    }
  }

  async delete(url: string): Promise<void> {
    // Extract remote path from URL - remove baseUrl and pathPrefix
    const normalizedPrefix = this.config.pathPrefix.replace(/^\/+/, '');
    const remotePath = url
      .replace(this.config.baseUrl, '')
      .replace(new RegExp(`^/${normalizedPrefix}/`), '')
      .replace(/^\/+/, '');

    const sftp = await this.getClient();

    try {
      await sftp.delete(remotePath);
    } catch (error: any) {
      // Ignore if file doesn't exist
      if (error.code !== 'ENOENT') {
        throw error;
      }
    } finally {
      await sftp.end();
    }
  }

  async exists(url: string): Promise<boolean> {
    // Extract remote path from URL - remove baseUrl and pathPrefix
    const normalizedPrefix = this.config.pathPrefix.replace(/^\/+/, '');
    const remotePath = url
      .replace(this.config.baseUrl, '')
      .replace(new RegExp(`^/${normalizedPrefix}/`), '')
      .replace(/^\/+/, '');

    const sftp = await this.getClient();

    try {
      await sftp.stat(remotePath);
      return true;
    } catch (error: any) {
      if (error.code === 'ENOENT') {
        return false;
      }
      throw error;
    } finally {
      await sftp.end();
    }
  }

  getUrl(filePath: string): string {
    // Remove leading slash from pathPrefix to make it relative
    const normalizedPrefix = this.config.pathPrefix.replace(/^\/+/, '');
    const remotePath = `${normalizedPrefix}/${filePath}`.replace(/\/+/g, '/');
    return `${this.config.baseUrl}/${remotePath}`.replace(/([^:]\/)\/+/g, '$1');
  }

  async getBuffer(url: string): Promise<Buffer> {
    // Extract remote path from URL - remove baseUrl and pathPrefix
    const normalizedPrefix = this.config.pathPrefix.replace(/^\/+/, '');
    let remotePath = url
      .replace(this.config.baseUrl, '')
      .replace(new RegExp(`^/${normalizedPrefix}/`), '')
      .replace(/^\/+/, '');

    // Handle legacy URLs that start with 'uploads/'
    // Old format: /uploads/product-matrix-colors/file.jpg
    // SFTP location: product-matrix-colors/file.jpg (no 'uploads' folder on SFTP)
    if (remotePath.startsWith('uploads/')) {
      remotePath = remotePath.substring('uploads/'.length);
    }

    const sftp = await this.getClient();

    try {
      const buffer = await sftp.get(remotePath);
      return buffer as Buffer;
    } finally {
      await sftp.end();
    }
  }
}

// ===== FACTORY =====
let cachedAdapter: FileStorageAdapter | null = null;

export async function getFileStorageAdapter(): Promise<FileStorageAdapter> {
  // Return cached adapter if available
  if (cachedAdapter) {
    return cachedAdapter;
  }

  // Fetch active configuration from database
  const result = await pool.query(
    'SELECT * FROM file_storage_settings WHERE is_active = true LIMIT 1'
  );

  const activeConfig = result.rows[0];

  // If no active config or provider is 'local', use LocalStorageAdapter
  if (!activeConfig || activeConfig.provider === 'local') {
    cachedAdapter = new LocalStorageAdapter();
    return cachedAdapter;
  }

  // If provider is 'sftp', use SFTPStorageAdapter
  if (activeConfig.provider === 'sftp') {
    // Get password from environment variable
    const password = process.env.FILE_STORAGE_PASSWORD || '';

    if (!activeConfig.host || !activeConfig.username || !password) {
      console.error('SFTP configuration incomplete, falling back to local storage');
      cachedAdapter = new LocalStorageAdapter();
      return cachedAdapter;
    }

    // Normalize pathPrefix to remove leading slashes
    const normalizedPathPrefix = (activeConfig.path_prefix || 'OMS').replace(/^\/+/, '');
    
    cachedAdapter = new SFTPStorageAdapter({
      host: activeConfig.host,
      port: activeConfig.port || 22,
      username: activeConfig.username,
      password: password,
      baseUrl: activeConfig.base_url || '',
      pathPrefix: normalizedPathPrefix,
    });
    return cachedAdapter;
  }

  // Default to local storage
  cachedAdapter = new LocalStorageAdapter();
  return cachedAdapter;
}

// Reset cache (useful for testing or when configuration changes)
export function resetFileStorageAdapterCache() {
  cachedAdapter = null;
}
