import { Pool } from "pg";
import dotenv from "dotenv";

// Load environment variables
dotenv.config();

// Database connection
const pool = new Pool({
  connectionString: process.env.DATABASE_URL,
});

// Helper function to build stable matrix variant ID
// Format: "matrixId-length-width-height-color" or "matrixId-length-width-height-color-option1-option2-..."
function buildMatrixVariantId(
  matrixId: number,
  length: number | null,
  width: number | null,
  height: number | null,
  color: string | null,
  colorOptions?: string[]
): string {
  const dimParts = [
    length !== null ? length.toString() : 'null',
    width !== null ? width.toString() : 'null',
    height !== null ? height.toString() : 'null',
  ];
  
  let id = `${matrixId}-${dimParts.join('-')}-${color || 'no-color'}`;
  
  // Add color options if they exist (sorted for deterministic ID)
  if (colorOptions && colorOptions.length > 0) {
    const sortedOptions = colorOptions.slice().sort().join('-');
    id += `-${sortedOptions}`;
  }
  
  return id;
}

async function backfillMatrixVariantIds() {
  console.log('🚀 === BACKFILL MATRIX_VARIANT_ID (with color_options) ===\n');
  
  try {
    // Get ALL products generated from matrices (re-generate IDs for all)
    const result = await pool.query(`
      SELECT id, matrix_id, length, width, height, color, color_options, sku
      FROM catalog.products
      WHERE generated_from_matrix = true
        AND matrix_id IS NOT NULL
      ORDER BY id ASC
    `);
    
    const products = result.rows;
    console.log(`📦 Znaleziono ${products.length} produktów do backfillu (re-generating all IDs)\n`);
    
    if (products.length === 0) {
      console.log('✅ Brak produktów z matryc!');
      return;
    }
    
    let successCount = 0;
    let errorCount = 0;
    const errors: any[] = [];
    
    // Process each product
    for (const product of products) {
      try {
        const { id, matrix_id, length, width, height, color, color_options, sku } = product;
        
        // Parse dimensions (they're stored as decimal strings)
        const lengthNum = length ? parseFloat(length) : null;
        const widthNum = width ? parseFloat(width) : null;
        const heightNum = height ? parseFloat(height) : null;
        
        // Parse color_options (stored as JSONB array)
        const colorOptions = color_options || [];
        
        // Build matrix variant ID (including color options)
        const matrixVariantId = buildMatrixVariantId(
          matrix_id,
          lengthNum,
          widthNum,
          heightNum,
          color,
          colorOptions
        );
        
        // Update product
        await pool.query(
          `UPDATE catalog.products 
           SET matrix_variant_id = $1, updated_at = CURRENT_TIMESTAMP
           WHERE id = $2`,
          [matrixVariantId, id]
        );
        
        successCount++;
        
        if (successCount % 50 === 0) {
          console.log(`   ✅ Przetworzono ${successCount}/${products.length} produktów...`);
        }
      } catch (error) {
        errorCount++;
        errors.push({
          productId: product.id,
          sku: product.sku,
          error: error instanceof Error ? error.message : 'Unknown error'
        });
        console.error(`   ❌ Błąd dla produktu ID ${product.id} (${product.sku}):`, error);
      }
    }
    
    console.log(`\n✅ === BACKFILL ZAKOŃCZONY ===`);
    console.log(`   Sukces: ${successCount}/${products.length}`);
    console.log(`   Błędy: ${errorCount}/${products.length}`);
    
    if (errors.length > 0) {
      console.log(`\n❌ Błędy przy przetwarzaniu ${errors.length} produktów:`);
      errors.forEach(err => {
        console.log(`   - Produkt ID ${err.productId} (${err.sku}): ${err.error}`);
      });
    }
    
    // Verify results
    const verifyResult = await pool.query(`
      SELECT COUNT(*) as count
      FROM catalog.products
      WHERE generated_from_matrix = true
        AND matrix_id IS NOT NULL
        AND matrix_variant_id IS NULL
    `);
    
    const remainingCount = parseInt(verifyResult.rows[0].count);
    console.log(`\n📊 Weryfikacja:`);
    console.log(`   Produkty bez matrix_variant_id: ${remainingCount}`);
    
    if (remainingCount === 0) {
      console.log(`   ✅ Wszystkie produkty zostały zaktualizowane pomyślnie!`);
    }
    
  } catch (error) {
    console.error('❌ Błąd podczas backfillu:', error);
    throw error;
  } finally {
    await pool.end();
  }
}

// Run the backfill
backfillMatrixVariantIds()
  .then(() => {
    console.log('\n🎉 Backfill zakończony pomyślnie!');
    process.exit(0);
  })
  .catch((error) => {
    console.error('\n💥 Backfill zakończony z błędem:', error);
    process.exit(1);
  });
