#!/bin/bash
# Skrypt weryfikacyjny dla konfiguracji SFTP file storage
# Uruchom na serwerze Hetzner jako root lub oms-sftp

echo "🔍 Sprawdzam konfigurację SFTP file storage..."
echo ""

# Kolory
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

ERRORS=0
WARNINGS=0

# Funkcje pomocnicze
check_pass() {
    echo -e "${GREEN}✅ $1${NC}"
}

check_fail() {
    echo -e "${RED}❌ $1${NC}"
    ((ERRORS++))
}

check_warn() {
    echo -e "${YELLOW}⚠️  $1${NC}"
    ((WARNINGS++))
}

# 1. Sprawdź czy folder główny istnieje
echo "📁 Sprawdzam główny folder /var/www/files/OMS..."
if [ -d "/var/www/files/OMS" ]; then
    check_pass "Folder /var/www/files/OMS istnieje"
    
    # Sprawdź uprawnienia
    OWNER=$(stat -c '%U' /var/www/files/OMS)
    PERMS=$(stat -c '%a' /var/www/files/OMS)
    
    if [ "$OWNER" = "oms-sftp" ]; then
        check_pass "Właściciel folderu: $OWNER (OK)"
    else
        check_fail "Właściciel folderu: $OWNER (powinno być: oms-sftp)"
    fi
    
    if [ "$PERMS" -ge "755" ]; then
        check_pass "Uprawnienia folderu: $PERMS (OK)"
    else
        check_warn "Uprawnienia folderu: $PERMS (zalecane: 755)"
    fi
else
    check_fail "Folder /var/www/files/OMS NIE ISTNIEJE!"
    echo "   Utwórz go komendą: sudo mkdir -p /var/www/files/OMS"
fi

echo ""

# 2. Sprawdź podfoldery
echo "📂 Sprawdzam podfoldery..."
REQUIRED_FOLDERS=(
    "/var/www/files/OMS/products/images"
    "/var/www/files/OMS/accessories/images"
    "/var/www/files/OMS/exports/pdf"
    "/var/www/files/OMS/exports/csv"
    "/var/www/files/OMS/ai-cache"
)

for FOLDER in "${REQUIRED_FOLDERS[@]}"; do
    if [ -d "$FOLDER" ]; then
        OWNER=$(stat -c '%U' "$FOLDER")
        if [ "$OWNER" = "oms-sftp" ]; then
            check_pass "$(basename $(dirname $FOLDER))/$(basename $FOLDER) - istnieje, właściciel: $OWNER"
        else
            check_warn "$(basename $(dirname $FOLDER))/$(basename $FOLDER) - istnieje, ale właściciel: $OWNER (powinno być: oms-sftp)"
        fi
    else
        check_warn "$(basename $(dirname $FOLDER))/$(basename $FOLDER) - NIE ISTNIEJE (zostanie utworzony automatycznie)"
    fi
done

echo ""

# 3. Sprawdź użytkownika oms-sftp
echo "👤 Sprawdzam użytkownika oms-sftp..."
if id "oms-sftp" &>/dev/null; then
    check_pass "Użytkownik oms-sftp istnieje"
    
    # Sprawdź grupy
    GROUPS=$(groups oms-sftp)
    echo "   Grupy: $GROUPS"
    
    # Sprawdź home directory
    HOME_DIR=$(eval echo ~oms-sftp)
    if [ -d "$HOME_DIR" ]; then
        check_pass "Home directory: $HOME_DIR"
    else
        check_warn "Home directory nie istnieje: $HOME_DIR"
    fi
else
    check_fail "Użytkownik oms-sftp NIE ISTNIEJE!"
fi

echo ""

# 4. Test zapisu
echo "✍️  Test zapisu pliku..."
TEST_FILE="/var/www/files/OMS/test-$(date +%s).txt"

if [ -d "/var/www/files/OMS" ]; then
    # Spróbuj jako obecny użytkownik
    if touch "$TEST_FILE" 2>/dev/null; then
        check_pass "Zapis pliku testowego: OK"
        
        # Sprawdź czy można usunąć
        if rm "$TEST_FILE" 2>/dev/null; then
            check_pass "Usuwanie pliku testowego: OK"
        else
            check_warn "Nie można usunąć pliku testowego"
        fi
    else
        check_fail "Nie można utworzyć pliku testowego w /var/www/files/OMS"
        echo "   Możliwe przyczyny:"
        echo "   - Brak uprawnień zapisu"
        echo "   - Folder należy do innego użytkownika"
    fi
fi

echo ""

# 5. Sprawdź nginx (jeśli jest)
echo "🌐 Sprawdzam konfigurację nginx..."
if command -v nginx &> /dev/null; then
    check_pass "Nginx zainstalowany"
    
    # Sprawdź czy działa
    if systemctl is-active --quiet nginx; then
        check_pass "Nginx działa"
    else
        check_warn "Nginx nie działa"
    fi
    
    # Sprawdź konfigurację dla files.alpsys.pl
    if [ -f "/etc/nginx/sites-enabled/files.alpsys.pl" ] || [ -f "/etc/nginx/sites-available/files.alpsys.pl" ]; then
        check_pass "Konfiguracja nginx dla files.alpsys.pl istnieje"
    else
        check_warn "Nie znaleziono konfiguracji nginx dla files.alpsys.pl"
    fi
else
    check_warn "Nginx nie zainstalowany (opcjonalny)"
fi

echo ""

# 6. Sprawdź SSH/SFTP
echo "🔐 Sprawdzam konfigurację SSH..."
if [ -f "/etc/ssh/sshd_config" ]; then
    check_pass "Plik konfiguracyjny SSH istnieje"
    
    # Sprawdź czy SFTP subsystem jest włączony
    if grep -q "^Subsystem.*sftp" /etc/ssh/sshd_config; then
        check_pass "SFTP subsystem włączony"
    else
        check_fail "SFTP subsystem NIE włączony w /etc/ssh/sshd_config"
    fi
else
    check_warn "Nie znaleziono /etc/ssh/sshd_config"
fi

echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "📊 PODSUMOWANIE:"
echo ""

if [ $ERRORS -eq 0 ] && [ $WARNINGS -eq 0 ]; then
    echo -e "${GREEN}✅ Wszystko OK! Konfiguracja jest prawidłowa.${NC}"
elif [ $ERRORS -eq 0 ]; then
    echo -e "${YELLOW}⚠️  Ostrzeżenia: $WARNINGS${NC}"
    echo "   Konfiguracja powinna działać, ale można poprawić niektóre rzeczy."
else
    echo -e "${RED}❌ Błędy: $ERRORS, Ostrzeżenia: $WARNINGS${NC}"
    echo "   Napraw błędy przed uruchomieniem migracji!"
fi

echo ""
echo "💡 Jeśli są błędy, uruchom komendy naprawcze:"
echo ""
echo "   sudo mkdir -p /var/www/files/OMS/{products/images,accessories/images,exports/{pdf,csv},ai-cache}"
echo "   sudo chown -R oms-sftp:oms-sftp /var/www/files/OMS"
echo "   sudo chmod -R 755 /var/www/files/OMS"
echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

exit $ERRORS
