import { Pool } from 'pg';

const pool = new Pool({
  connectionString: process.env.DATABASE_URL,
});

async function setupProductionDatabase() {
  const client = await pool.connect();
  
  try {
    console.log('🚀 Setting up production database...\n');

    // Krok 1: Tworzenie schematów
    console.log('📁 Creating schemas...');
    await client.query('CREATE SCHEMA IF NOT EXISTS allegro');
    await client.query('CREATE SCHEMA IF NOT EXISTS shoper');
    await client.query('CREATE SCHEMA IF NOT EXISTS commerce');
    console.log('✅ Schemas created: allegro, shoper, commerce\n');

    // Krok 2: Tabela allegro.orders
    console.log('📊 Creating allegro.orders table...');
    await client.query(`
      CREATE TABLE IF NOT EXISTS allegro.orders (
        id SERIAL PRIMARY KEY,
        order_id VARCHAR(255) UNIQUE NOT NULL,
        seller_id VARCHAR(255),
        seller_login VARCHAR(255),
        order_date TIMESTAMP,
        seller_status VARCHAR(50),
        marketplace VARCHAR(50),
        buyer_id VARCHAR(255),
        buyer_login VARCHAR(255),
        buyer_email VARCHAR(255),
        buyer_company VARCHAR(255),
        buyer_name VARCHAR(255),
        buyer_phone VARCHAR(50),
        buyer_address TEXT,
        buyer_zip VARCHAR(20),
        buyer_city VARCHAR(100),
        buyer_country_code VARCHAR(10),
        payment_id VARCHAR(255),
        payment_status VARCHAR(50),
        payment_provider VARCHAR(100),
        allegro_pay BOOLEAN DEFAULT false,
        payment_amount DECIMAL(10,2),
        payment_currency VARCHAR(10),
        payment_type VARCHAR(50),
        invoice_required BOOLEAN DEFAULT false,
        delivery_method VARCHAR(255),
        delivery_amount DECIMAL(10,2),
        delivery_currency VARCHAR(10),
        total_to_pay_amount DECIMAL(10,2),
        total_to_pay_currency VARCHAR(10),
        buyer_notes TEXT,
        smart BOOLEAN DEFAULT false,
        tracking_numbers TEXT,
        payment_last_date TIMESTAMP,
        imported_at TIMESTAMP DEFAULT NOW()
      )
    `);
    console.log('✅ allegro.orders created\n');

    // Krok 3: Tabela allegro.products
    console.log('📦 Creating allegro.products table...');
    await client.query(`
      CREATE TABLE IF NOT EXISTS allegro.products (
        id SERIAL PRIMARY KEY,
        product_id VARCHAR(255),
        external_id VARCHAR(255) UNIQUE NOT NULL,
        name TEXT,
        description TEXT,
        image_url TEXT,
        updated_at TIMESTAMP DEFAULT NOW()
      )
    `);
    console.log('✅ allegro.products created\n');

    // Krok 4: Tabela allegro.order_items
    console.log('🛒 Creating allegro.order_items table...');
    await client.query(`
      CREATE TABLE IF NOT EXISTS allegro.order_items (
        id SERIAL PRIMARY KEY,
        order_id VARCHAR(255) REFERENCES allegro.orders(order_id) ON DELETE CASCADE,
        product_id INTEGER REFERENCES allegro.products(id),
        offer_id VARCHAR(255),
        offer_name TEXT,
        quantity INTEGER,
        unit_price DECIMAL(10,2),
        total_price DECIMAL(10,2),
        currency VARCHAR(10),
        created_at TIMESTAMP DEFAULT NOW()
      )
    `);
    console.log('✅ allegro.order_items created\n');

    // Krok 5: Tabela shoper.orders
    console.log('📊 Creating shoper.orders table...');
    await client.query(`
      CREATE TABLE IF NOT EXISTS shoper.orders (
        id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
        shoper_order_id VARCHAR(255) UNIQUE NOT NULL,
        buyer_login VARCHAR(255),
        buyer_email VARCHAR(255),
        total_amount DECIMAL(10,2),
        currency VARCHAR(10),
        payment_status VARCHAR(50),
        fulfillment_status VARCHAR(50),
        items_count VARCHAR(50),
        order_date TIMESTAMP,
        payment_date TIMESTAMP,
        raw_data JSONB,
        created_at TIMESTAMP DEFAULT NOW(),
        updated_at TIMESTAMP DEFAULT NOW()
      )
    `);
    console.log('✅ shoper.orders created\n');

    // Krok 6: Tabela shoper.order_items
    console.log('🛒 Creating shoper.order_items table...');
    await client.query(`
      CREATE TABLE IF NOT EXISTS shoper.order_items (
        id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
        shoper_order_id VARCHAR(255) REFERENCES shoper.orders(shoper_order_id) ON DELETE CASCADE,
        product_id VARCHAR(255),
        product_name TEXT,
        product_code VARCHAR(255),
        quantity INTEGER,
        unit_price DECIMAL(10,2),
        total_price DECIMAL(10,2),
        currency VARCHAR(10),
        created_at TIMESTAMP DEFAULT NOW()
      )
    `);
    console.log('✅ shoper.order_items created\n');

    // Krok 7: Tabela commerce.orders (DOKŁADNA kopia z development!)
    console.log('🔄 Creating commerce.orders table...');
    await client.query(`
      CREATE TABLE IF NOT EXISTS commerce.orders (
        id VARCHAR(255) PRIMARY KEY DEFAULT gen_random_uuid()::text,
        order_number BIGINT,
        order_code VARCHAR(50) GENERATED ALWAYS AS (
          CASE 
            WHEN source = 'ALLEGRO' THEN 'AL-' || order_number
            WHEN source = 'SHOPER' THEN 'SH-' || order_number
          END
        ) STORED,
        source VARCHAR(50) NOT NULL,
        source_order_id VARCHAR(255) NOT NULL,
        order_date TIMESTAMP,
        status VARCHAR(50),
        buyer_login VARCHAR(255),
        buyer_email VARCHAR(255),
        buyer_company VARCHAR(255),
        buyer_first_name VARCHAR(255),
        buyer_last_name VARCHAR(255),
        buyer_phone VARCHAR(50),
        buyer_address TEXT,
        buyer_zip VARCHAR(20),
        buyer_city VARCHAR(100),
        buyer_country_code VARCHAR(10),
        payment_id VARCHAR(255),
        payment_status VARCHAR(50),
        payment_type VARCHAR(50),
        payment_provider VARCHAR(100),
        payment_amount DECIMAL(10,2),
        payment_currency VARCHAR(10),
        delivery_method VARCHAR(255),
        delivery_amount DECIMAL(10,2),
        delivery_currency VARCHAR(10),
        total_to_pay_amount DECIMAL(10,2),
        total_to_pay_currency VARCHAR(10),
        buyer_notes TEXT,
        invoice_required BOOLEAN DEFAULT false,
        tracking_numbers TEXT,
        raw_payload JSONB,
        created_at TIMESTAMP DEFAULT NOW(),
        updated_at TIMESTAMP DEFAULT NOW(),
        buyer_company_name TEXT,
        delivery_address JSONB,
        UNIQUE(source, source_order_id)
      )
    `);
    console.log('✅ commerce.orders created\n');

    // Krok 8: Sekwencja dla order_number
    console.log('🔢 Creating order_number sequence...');
    await client.query(`
      CREATE SEQUENCE IF NOT EXISTS commerce.order_number_seq START 1000
    `);
    console.log('✅ Sequence created\n');

    // Krok 9: Triggery dla Allegro (DOKŁADNA kopia z development!)
    console.log('⚡ Creating Allegro triggers...');
    await client.query(`
      CREATE OR REPLACE FUNCTION public.sync_allegro_to_commerce()
      RETURNS TRIGGER AS $$
      DECLARE
          v_first_name VARCHAR;
          v_last_name VARCHAR;
          name_parts TEXT[];
      BEGIN
          -- Rozdziel imię i nazwisko (jeśli istnieje)
          IF NEW.buyer_name IS NOT NULL THEN
              name_parts := string_to_array(TRIM(NEW.buyer_name), ' ');
              v_first_name := name_parts[1];
              IF array_length(name_parts, 1) > 1 THEN
                  v_last_name := array_to_string(name_parts[2:array_length(name_parts, 1)], ' ');
              END IF;
          END IF;
          
          -- Wstaw lub zaktualizuj w commerce.orders
          INSERT INTO commerce.orders (
              id,
              source,
              source_order_id,
              order_date,
              status,
              buyer_login,
              buyer_email,
              buyer_company,
              buyer_first_name,
              buyer_last_name,
              buyer_phone,
              buyer_address,
              buyer_zip,
              buyer_city,
              buyer_country_code,
              payment_id,
              payment_status,
              payment_type,
              payment_provider,
              payment_amount,
              payment_currency,
              delivery_method,
              delivery_amount,
              delivery_currency,
              total_to_pay_amount,
              total_to_pay_currency,
              buyer_notes,
              invoice_required,
              tracking_numbers,
              raw_payload,
              created_at,
              updated_at
          ) VALUES (
              gen_random_uuid(),
              'ALLEGRO',
              NEW.order_id,
              NEW.order_date,
              NEW.seller_status,
              NEW.buyer_login,
              NEW.buyer_email,
              NEW.buyer_company,
              v_first_name,
              v_last_name,
              NEW.buyer_phone,
              NEW.buyer_address,
              NEW.buyer_zip,
              NEW.buyer_city,
              NEW.buyer_country_code,
              NEW.payment_id,
              NEW.payment_status,
              NEW.payment_type,
              NEW.payment_provider,
              NEW.payment_amount,
              NEW.payment_currency,
              NEW.delivery_method,
              NEW.delivery_amount,
              NEW.delivery_currency,
              NEW.total_to_pay_amount,
              NEW.total_to_pay_currency,
              NEW.buyer_notes,
              NEW.invoice_required,
              NEW.tracking_numbers,
              jsonb_build_object(
                  'order_id', NEW.order_id,
                  'marketplace', NEW.marketplace,
                  'smart', NEW.smart,
                  'allegro_pay', NEW.allegro_pay
              ),
              NOW(),
              NOW()
          )
          ON CONFLICT (source_order_id, source) DO UPDATE SET
              order_date = EXCLUDED.order_date,
              status = EXCLUDED.status,
              buyer_login = EXCLUDED.buyer_login,
              buyer_email = EXCLUDED.buyer_email,
              buyer_company = EXCLUDED.buyer_company,
              buyer_first_name = EXCLUDED.buyer_first_name,
              buyer_last_name = EXCLUDED.buyer_last_name,
              buyer_phone = EXCLUDED.buyer_phone,
              buyer_address = EXCLUDED.buyer_address,
              buyer_zip = EXCLUDED.buyer_zip,
              buyer_city = EXCLUDED.buyer_city,
              buyer_country_code = EXCLUDED.buyer_country_code,
              payment_id = EXCLUDED.payment_id,
              payment_status = EXCLUDED.payment_status,
              payment_type = EXCLUDED.payment_type,
              payment_provider = EXCLUDED.payment_provider,
              payment_amount = EXCLUDED.payment_amount,
              payment_currency = EXCLUDED.payment_currency,
              delivery_method = EXCLUDED.delivery_method,
              delivery_amount = EXCLUDED.delivery_amount,
              delivery_currency = EXCLUDED.delivery_currency,
              total_to_pay_amount = EXCLUDED.total_to_pay_amount,
              total_to_pay_currency = EXCLUDED.total_to_pay_currency,
              buyer_notes = EXCLUDED.buyer_notes,
              invoice_required = EXCLUDED.invoice_required,
              tracking_numbers = EXCLUDED.tracking_numbers,
              updated_at = NOW();
          
          RETURN NEW;
      END;
      $$ LANGUAGE plpgsql;
    `);

    await client.query(`
      DROP TRIGGER IF EXISTS allegro_to_commerce_trigger ON allegro.orders;
      CREATE TRIGGER allegro_to_commerce_trigger
      AFTER INSERT OR UPDATE ON allegro.orders
      FOR EACH ROW
      EXECUTE FUNCTION public.sync_allegro_to_commerce();
    `);
    console.log('✅ Allegro triggers created\n');

    // Krok 10: Triggery dla Shoper (DOKŁADNA kopia z development!)
    console.log('⚡ Creating Shoper triggers...');
    await client.query(`
      CREATE OR REPLACE FUNCTION public.sync_shoper_to_commerce()
      RETURNS TRIGGER AS $$
      DECLARE
          v_raw jsonb;
          v_billing_address jsonb;
          v_delivery_address jsonb;
          v_first_name VARCHAR;
          v_last_name VARCHAR;
          v_phone VARCHAR;
          v_company VARCHAR;
          v_street VARCHAR;
          v_zip VARCHAR;
          v_city VARCHAR;
          v_country_code VARCHAR;
          v_delivery_cost NUMERIC;
      BEGIN
          v_raw := NEW.raw_data;
          
          -- Wyciągnij dane z billing_address
          IF v_raw->>'billing_address' IS NOT NULL THEN
              v_billing_address := (v_raw->>'billing_address')::jsonb;
              v_first_name := v_billing_address->>'firstname';
              v_last_name := v_billing_address->>'lastname';
              v_phone := v_billing_address->>'phone';
              v_company := v_billing_address->>'company';
          END IF;
          
          -- Wyciągnij dane z delivery_address
          IF v_raw->>'delivery_address' IS NOT NULL THEN
              v_delivery_address := (v_raw->>'delivery_address')::jsonb;
              v_street := v_delivery_address->>'street1';
              v_zip := v_delivery_address->>'postcode';
              v_city := v_delivery_address->>'city';
              v_country_code := v_delivery_address->>'country_code';
          END IF;
          
          v_delivery_cost := (v_raw->>'shipping_cost')::numeric;
          
          -- Wstaw lub zaktualizuj w commerce.orders
          INSERT INTO commerce.orders (
              id,
              source,
              source_order_id,
              order_date,
              status,
              buyer_login,
              buyer_email,
              buyer_company,
              buyer_first_name,
              buyer_last_name,
              buyer_phone,
              buyer_address,
              buyer_zip,
              buyer_city,
              buyer_country_code,
              payment_status,
              payment_type,
              payment_amount,
              payment_currency,
              delivery_amount,
              delivery_currency,
              total_to_pay_amount,
              total_to_pay_currency,
              buyer_notes,
              invoice_required,
              raw_payload,
              created_at,
              updated_at
          ) VALUES (
              gen_random_uuid(),
              'SHOPER',
              NEW.shoper_order_id,
              NEW.order_date,
              NEW.fulfillment_status,
              NEW.buyer_login,
              NEW.buyer_email,
              v_company,
              v_first_name,
              v_last_name,
              v_phone,
              v_street,
              v_zip,
              v_city,
              v_country_code,
              NEW.payment_status,
              CASE 
                  WHEN (v_raw->>'payment_id')::int = 2 THEN 'ONLINE'
                  ELSE 'CASH_ON_DELIVERY'
              END,
              NEW.total_amount,
              NEW.currency,
              v_delivery_cost,
              NEW.currency,
              NEW.total_amount,
              NEW.currency,
              v_raw->>'notes',
              false,
              v_raw,
              NOW(),
              NOW()
          )
          ON CONFLICT (source_order_id, source) DO UPDATE SET
              order_date = EXCLUDED.order_date,
              status = EXCLUDED.status,
              buyer_login = EXCLUDED.buyer_login,
              buyer_email = EXCLUDED.buyer_email,
              buyer_company = EXCLUDED.buyer_company,
              buyer_first_name = EXCLUDED.buyer_first_name,
              buyer_last_name = EXCLUDED.buyer_last_name,
              buyer_phone = EXCLUDED.buyer_phone,
              buyer_address = EXCLUDED.buyer_address,
              buyer_zip = EXCLUDED.buyer_zip,
              buyer_city = EXCLUDED.buyer_city,
              buyer_country_code = EXCLUDED.buyer_country_code,
              payment_status = EXCLUDED.payment_status,
              payment_type = EXCLUDED.payment_type,
              payment_amount = EXCLUDED.payment_amount,
              delivery_amount = EXCLUDED.delivery_amount,
              total_to_pay_amount = EXCLUDED.total_to_pay_amount,
              buyer_notes = EXCLUDED.buyer_notes,
              raw_payload = EXCLUDED.raw_payload,
              updated_at = NOW();
          
          RETURN NEW;
      END;
      $$ LANGUAGE plpgsql;
    `);

    await client.query(`
      DROP TRIGGER IF EXISTS shoper_to_commerce_trigger ON shoper.orders;
      CREATE TRIGGER shoper_to_commerce_trigger
      AFTER INSERT OR UPDATE ON shoper.orders
      FOR EACH ROW
      EXECUTE FUNCTION public.sync_shoper_to_commerce();
    `);
    console.log('✅ Shoper triggers created\n');

    console.log('🎉 Production database setup completed successfully!\n');
    console.log('Next steps:');
    console.log('1. Go to https://alp-oms.replit.app/settings');
    console.log('2. Connect to Allegro');
    console.log('3. Connect to Shoper');
    console.log('4. All orders will be automatically synced!\n');

  } catch (error) {
    console.error('❌ Error setting up database:', error);
    throw error;
  } finally {
    client.release();
    await pool.end();
  }
}

setupProductionDatabase().catch(console.error);
