import { pool } from '../server/postgres.js';
import { promises as fs } from 'fs';
import path from 'path';

async function restoreDatabase(backupFileName: string) {
  const backupDir = path.join(process.cwd(), 'backups');
  const backupFile = path.join(backupDir, backupFileName);
  
  try {
    // Sprawdź czy plik istnieje
    await fs.access(backupFile);
    
    console.log('🔄 Rozpoczynam restore bazy danych...');
    console.log(`📁 Plik: ${backupFile}`);
    
    // Wczytaj plik SQL
    const sqlContent = await fs.readFile(backupFile, 'utf-8');
    
    // Podziel na instrukcje SQL
    const statements = sqlContent
      .split(';')
      .map(s => s.trim())
      .filter(s => s.length > 0 && !s.startsWith('--'));
    
    console.log(`📊 Liczba instrukcji SQL: ${statements.length}`);
    
    // Wykonaj każdą instrukcję
    const client = await pool.connect();
    
    try {
      await client.query('BEGIN');
      
      for (let i = 0; i < statements.length; i++) {
        const stmt = statements[i];
        
        if (i % 100 === 0) {
          console.log(`⏳ Progress: ${i}/${statements.length}`);
        }
        
        try {
          await client.query(stmt);
        } catch (error: any) {
          console.warn(`⚠️  Błąd przy instrukcji ${i}: ${error.message}`);
          // Kontynuuj mimo błędów (może tabela już istnieje itp.)
        }
      }
      
      await client.query('COMMIT');
      console.log('✅ Restore zakończony pomyślnie!');
      
    } catch (error) {
      await client.query('ROLLBACK');
      throw error;
    } finally {
      client.release();
    }
    
  } catch (error) {
    console.error('❌ Błąd podczas restore:', error);
    throw error;
  } finally {
    await pool.end();
  }
}

// Pobierz nazwę pliku z argumentów
const backupFileName = process.argv[2];

if (!backupFileName) {
  console.error('❌ Podaj nazwę pliku backupu!');
  console.error('Użycie: tsx scripts/restore-database.ts <nazwa-pliku.sql>');
  process.exit(1);
}

restoreDatabase(backupFileName).catch(console.error);
