#!/bin/bash
set -e

# Alpma OMS Backup Script
# Creates database dump and code archive (excluding images)

TIMESTAMP=$(date +%Y%m%d_%H%M%S)
BACKUP_DIR="backups"
DB_BACKUP="${BACKUP_DIR}/db-${TIMESTAMP}.dump"
CODE_BACKUP="${BACKUP_DIR}/code-${TIMESTAMP}.tar.gz"

echo "=== Alpma OMS Backup Started: ${TIMESTAMP} ==="

# Create backup directory if it doesn't exist
mkdir -p "${BACKUP_DIR}"

# 1. Database Backup
echo "📦 Creating database backup..."
if [ -z "${DATABASE_URL}" ]; then
  echo "❌ ERROR: DATABASE_URL not set"
  exit 1
fi

pg_dump "${DATABASE_URL}" --format=custom --file="${DB_BACKUP}"
DB_SIZE=$(du -h "${DB_BACKUP}" | cut -f1)
echo "✓ Database backup created: ${DB_BACKUP} (${DB_SIZE})"

# Calculate checksum
DB_CHECKSUM=$(sha256sum "${DB_BACKUP}" | cut -d' ' -f1)
echo "  Checksum: ${DB_CHECKSUM}"

# 2. Code Archive (excluding images and node_modules)
echo "📦 Creating code archive..."
tar --exclude='uploads' \
    --exclude='attached_assets' \
    --exclude='node_modules' \
    --exclude='.git' \
    --exclude='backups' \
    --exclude='dist' \
    --exclude='.npm' \
    --exclude='.cache' \
    --exclude='.upm' \
    --exclude='*.log' \
    -czf "${CODE_BACKUP}" \
    client/ server/ shared/ scripts/ \
    package.json package-lock.json tsconfig.json \
    vite.config.ts drizzle.config.ts tailwind.config.ts \
    *.md *.json 2>/dev/null || true

CODE_SIZE=$(du -h "${CODE_BACKUP}" | cut -f1)
echo "✓ Code archive created: ${CODE_BACKUP} (${CODE_SIZE})"

# Calculate checksum
CODE_CHECKSUM=$(sha256sum "${CODE_BACKUP}" | cut -d' ' -f1)
echo "  Checksum: ${CODE_CHECKSUM}"

# 3. Create backup manifest
MANIFEST="${BACKUP_DIR}/backup-${TIMESTAMP}.manifest"
cat > "${MANIFEST}" << EOF
Alpma OMS Backup Manifest
========================
Timestamp: ${TIMESTAMP}
Date: $(date)

Database Backup:
  File: ${DB_BACKUP}
  Size: ${DB_SIZE}
  SHA256: ${DB_CHECKSUM}

Code Archive:
  File: ${CODE_BACKUP}
  Size: ${CODE_SIZE}
  SHA256: ${CODE_CHECKSUM}

Restore Instructions:
=====================
1. Database: pg_restore -d \$DATABASE_URL ${DB_BACKUP}
2. Code: tar -xzf ${CODE_BACKUP}

Notes:
- Images (uploads/, attached_assets/) are excluded from code archive
- Node modules and build artifacts are excluded
EOF

echo "✓ Manifest created: ${MANIFEST}"

# 4. Cleanup old backups (keep last 10)
echo "🧹 Cleaning old backups (keeping last 10)..."
ls -t "${BACKUP_DIR}"/db-*.dump 2>/dev/null | tail -n +11 | xargs -r rm
ls -t "${BACKUP_DIR}"/code-*.tar.gz 2>/dev/null | tail -n +11 | xargs -r rm
ls -t "${BACKUP_DIR}"/backup-*.manifest 2>/dev/null | tail -n +11 | xargs -r rm

echo ""
echo "=== Backup Completed Successfully ==="
echo "Database: ${DB_BACKUP}"
echo "Code: ${CODE_BACKUP}"
echo "Manifest: ${MANIFEST}"
echo ""
echo "To restore database: pg_restore -d \$DATABASE_URL ${DB_BACKUP}"
echo "To restore code: tar -xzf ${CODE_BACKUP}"
