import { pool } from './server/postgres';
import { generateProductionOrdersFromPlan } from './server/services/production/production-order-generator';

async function freshSimulation() {
  console.log('\n🏭 ═══════════════════════════════════════════════════════════');
  console.log('   ŚWIEŻA SYMULACJA PROCESU PRODUKCYJNEGO');
  console.log('═══════════════════════════════════════════════════════════\n');

  // Krok 1: Nowy plan
  console.log('┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 1: UTWORZENIE NOWEGO PLANU PRODUKCYJNEGO          │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  const planResult = await pool.query(`
    INSERT INTO production.production_plans (
      plan_number,
      name,
      status,
      planned_start_date
    ) VALUES (
      'PLAN-SIM-001',
      'Symulacja - Kompletny proces produkcyjny',
      'draft',
      CURRENT_TIMESTAMP
    )
    RETURNING id, plan_number, name
  `);
  
  const planId = planResult.rows[0].id;
  console.log(`✅ Utworzono plan: ${planResult.rows[0].plan_number}`);
  console.log(`   Nazwa: ${planResult.rows[0].name}`);
  console.log(`   ID: ${planId}\n`);
  
  // Krok 2: Dodaj produkty
  console.log('┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 2: DODANIE PRODUKTÓW DO PLANU                     │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  const products = [
    { id: 277, qty: 2, color: 'BIALY', ref: '01994', name: 'VB-50 BIALY' },
    { id: 305, qty: 1, color: 'WOTAN', ref: '01967', name: 'VB-50 WOTAN' },
  ];
  
  for (const prod of products) {
    await pool.query(`
      INSERT INTO production.production_plan_lines (
        plan_id,
        product_id,
        quantity,
        color_code,
        source_type,
        source_reference,
        status
      ) VALUES ($1, $2, $3, $4, 'sales_order', $5, 'pending')
    `, [planId, prod.id, prod.qty, prod.color, prod.ref]);
    
    console.log(`   ✓ Dodano: ${prod.qty}× ${prod.name} (zamówienie ${prod.ref})`);
  }
  
  console.log('');
  
  // Krok 3: Generuj ZLP
  console.log('┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 3: GENEROWANIE ZLP (ZLECENIA PRODUKCYJNE)         │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  const genResult = await generateProductionOrdersFromPlan(planId);
  
  if (!genResult.success) {
    console.error('❌ Błąd:', genResult.errors);
    await pool.end();
    process.exit(1);
  }
  
  console.log(`✅ Wygenerowano ${genResult.generatedOrders.length} ZLP:\n`);
  genResult.generatedOrders.forEach(order => {
    console.log(`   📦 ${order.orderNumber} (${order.colorCode}):`);
    console.log(`      - Formatek: ${order.componentCount} (quantity=1 każda)`);
    console.log(`      - Work Orders: ${order.workOrderIds.length}`);
  });
  
  console.log(`\n📊 Breakdown po kolorach:`);
  Object.entries(genResult.summary.colorBreakdown).forEach(([color, count]) => {
    console.log(`   - ${color}: ${count} formatek`);
  });
  
  // Krok 4: Routing resolution
  console.log('\n\n┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 4: WERYFIKACJA ROUTING RESOLUTION                 │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  const routingData = await pool.query(`
    SELECT 
      po.order_number,
      rv.variant_code,
      pobi.required_operations,
      COUNT(*) as ile
    FROM production.production_order_bom_items pobi
    JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
    JOIN production.production_orders po ON po.id = pob.production_order_id
    JOIN production.production_plan_lines ppl ON ppl.production_order_id = po.id
    LEFT JOIN production.production_routing_variants rv ON rv.id = pobi.routing_variant_id
    WHERE ppl.plan_id = $1
    GROUP BY po.order_number, rv.variant_code, pobi.required_operations
    ORDER BY po.order_number, rv.variant_code
  `, [planId]);
  
  let currentZLP = '';
  routingData.rows.forEach(row => {
    if (row.order_number !== currentZLP) {
      console.log(`\n   ${row.order_number}:`);
      currentZLP = row.order_number;
    }
    const ops = JSON.parse(row.required_operations).join(' → ');
    console.log(`      ${row.variant_code}: ${row.ile} formatek → ${ops}`);
  });
  
  // Krok 5: Work Orders
  console.log('\n\n┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 5: WORK ORDERS (OPERACJE DO WYKONANIA)            │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  const workOrders = await pool.query(`
    SELECT 
      pwo.id,
      pwo.work_order_number,
      po.order_number,
      pwo.sequence,
      pwo.quantity_planned,
      pwo.status,
      COALESCE(pro.creates_buffer, false) as creates_buffer,
      pl.name as buffer_location
    FROM production.production_work_orders pwo
    JOIN production.production_orders po ON po.id = pwo.production_order_id
    JOIN production.production_plan_lines ppl ON ppl.production_order_id = po.id
    LEFT JOIN production.production_routing_operations pro ON pro.id = pwo.routing_operation_id
    LEFT JOIN production.production_locations pl ON pl.id = pro.buffer_location_id
    WHERE ppl.plan_id = $1
    ORDER BY po.order_number, pwo.sequence
  `, [planId]);
  
  console.log(`📋 Utworzono ${workOrders.rows.length} Work Orders:\n`);
  
  currentZLP = '';
  workOrders.rows.forEach(wo => {
    if (wo.order_number !== currentZLP) {
      console.log(`\n   ${wo.order_number}:`);
      currentZLP = wo.order_number;
    }
    const bufferFlag = wo.creates_buffer ? ' 📦 → BUFOR' : '';
    console.log(`      ${wo.sequence}. ${wo.work_order_number}: ${wo.quantity_planned} szt${bufferFlag}`);
    if (wo.buffer_location) {
      console.log(`         └─ Składowanie → ${wo.buffer_location}`);
    }
  });
  
  // Krok 6: Symulacja wykonania WO z buforem
  console.log('\n\n┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 6: WYKONANIE WORK ORDER (symulacja)               │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  const woWithBuffer = workOrders.rows.find(w => w.creates_buffer);
  
  if (woWithBuffer) {
    console.log(`🎯 Wykonuję: ${woWithBuffer.work_order_number}\n`);
    console.log(`   1️⃣  START operacji...`);
    
    await pool.query(`UPDATE production.production_work_orders SET status = 'in_progress', actual_start_time = NOW() WHERE id = $1`, [woWithBuffer.id]);
    
    console.log(`   2️⃣  Produkcja: ${woWithBuffer.quantity_planned} szt...`);
    
    await pool.query(`UPDATE production.production_work_orders SET quantity_produced = $1 WHERE id = $2`, [woWithBuffer.quantity_planned, woWithBuffer.id]);
    
    console.log(`   3️⃣  Zakończono operację ✓`);
    console.log(`\n   📦 BUFFER SYSTEM - Auto-składowanie:\n`);
    
    // SKU półproduktu
    const sku = `SEMI-${woWithBuffer.order_number}-SEQ${woWithBuffer.sequence}`;
    
    // Ruch IN
    await pool.query(`
      INSERT INTO production.production_buffer_movements (
        movement_type, product_sku, quantity, unit_of_measure, 
        source_type, source_id, notes
      ) VALUES ('IN', $1, $2, 'szt', 'WORK_ORDER', $3, 'Auto-składowanie po zakończeniu operacji')
    `, [sku, woWithBuffer.quantity_planned, woWithBuffer.id]);
    
    console.log(`      ✅ Utworzono ruch IN: +${woWithBuffer.quantity_planned} szt ${sku}`);
    
    // Stan magazynowy
    const bufferLocationId = await pool.query(`
      SELECT pro.buffer_location_id 
      FROM production.production_routing_operations pro
      JOIN production.production_work_orders pwo ON pwo.routing_operation_id = pro.id
      WHERE pwo.id = $1
    `, [woWithBuffer.id]);
    
    if (bufferLocationId.rows[0]?.buffer_location_id) {
      await pool.query(`
        INSERT INTO production.production_buffer_stock (
          product_sku, product_name, location_id, quantity_available, quantity_total
        ) VALUES ($1, $1, $2, $3, $3)
        ON CONFLICT (product_sku, location_id) 
        DO UPDATE SET
          quantity_available = production_buffer_stock.quantity_available + $3,
          quantity_total = production_buffer_stock.quantity_total + $3
      `, [sku, bufferLocationId.rows[0].buffer_location_id, woWithBuffer.quantity_planned]);
      
      const stock = await pool.query(`
        SELECT quantity_available, quantity_reserved, quantity_total
        FROM production.production_buffer_stock
        WHERE product_sku = $1
      `, [sku]);
      
      console.log(`\n      📊 STAN MAGAZYNU:`);
      console.log(`         Dostępne: ${stock.rows[0].quantity_available} szt`);
      console.log(`         Zarezerwowane: ${stock.rows[0].quantity_reserved} szt`);
      console.log(`         RAZEM: ${stock.rows[0].quantity_total} szt`);
    }
    
    await pool.query(`UPDATE production.production_work_orders SET status = 'completed', actual_end_time = NOW() WHERE id = $1`, [woWithBuffer.id]);
    console.log(`\n   ✅ Work Order zakończony!`);
  } else {
    console.log('⚠️  Brak WO z flagą creates_buffer');
  }
  
  // Podsumowanie
  console.log('\n\n┌═════════════════════════════════════════════════════════┐');
  console.log('│ 🎯 DIAGRAM PROCESU - CO SIĘ WYDARZYŁO                 │');
  console.log('└═════════════════════════════════════════════════════════┘\n');
  
  console.log(`
  📋 PLAN PRODUKCYJNY (PLAN-SIM-001)
      ├─ 2× VB-50 BIALY (zamówienie 01994)
      └─ 1× VB-50 WOTAN (zamówienie 01967)
             ↓
  ⚙️  GENERATOR (production-order-generator.ts)
      ├─ Analizuje BOM każdego produktu
      ├─ Tworzy INDYWIDUALNE rekordy (quantity=1)
      ├─ Resolve routing variant dla każdej formatki
      └─ Agreguje po kolorze → ZLP
             ↓
  📦 PRODUCTION ORDERS (ZLP)
      ├─ ZLP-XXX-BIALY: ${genResult.summary.colorBreakdown['BIALY'] || 0} formatek
      ├─ ZLP-XXX-WOTAN: ${genResult.summary.colorBreakdown['WOTAN'] || 0} formatek
      └─ Każda formatka: routing_variant_id + required_operations
             ↓
  ⚙️  WORK ORDERS (operacje)
      ├─ WO-1: Cięcie (${workOrders.rows.filter(w => w.sequence === 1).length || 0} ZLP)
      ├─ WO-2: Oklejanie (${workOrders.rows.filter(w => w.sequence === 2).length || 0} ZLP)
      └─ WO-3: Wiercenie (${workOrders.rows.filter(w => w.sequence === 3).length || 0} ZLP)
             ↓
  🏭 WYKONANIE (jeśli creates_buffer=true)
      ├─ Work Order → status: in_progress
      ├─ Produkcja: quantity_produced
      ├─ Status: completed
      └─ 📦 BUFFER SYSTEM aktywuje się automatycznie:
             ↓
  📦 BUFFER MOVEMENTS (production_buffer_movements)
      └─ Ruch IN: +N szt → lokalizacja magazynowa
             ↓
  📊 BUFFER STOCK (production_buffer_stock)
      └─ quantity_available += N
      └─ quantity_total += N
             ↓
  🔄 NASTĘPNY ZLP może zarezerwować ten półprodukt
      ├─ Ruch RESERVE: quantity_available → quantity_reserved
      └─ Ruch OUT: pobieranie do kompletacji
  `);
  
  console.log('\n═══════════════════════════════════════════════════════════');
  console.log('✨ SYMULACJA ZAKOŃCZONA!');
  console.log('═══════════════════════════════════════════════════════════\n');
  
  await pool.end();
}

freshSimulation().catch(err => {
  console.error('❌ Błąd:', err);
  process.exit(1);
});
