import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ArrowLeft, Package, FileText, History, ExternalLink, Trash2, Plus, MapPin, Building2, Truck, FileCheck, ChevronDown, Pencil, List, Archive, ArchiveRestore, Lock, RotateCcw, AlertTriangle, User, ClipboardList, Clock, Loader2 } from "lucide-react";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { useState, useEffect } from "react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Link } from "wouter";

interface Material {
  id: number;
  name: string;
  internalCode: string;
  supplierCode?: string;
  description?: string;
  specifications?: any;
  unitOfMeasure?: string;
  price?: number;
  quantity?: string;
  reservedQuantity?: number;
  availableQuantity?: number;
  gallery?: string[];
  primaryImageUrl?: string;
  isActive: boolean;
  isArchived?: boolean;
  groupName?: string;
  categoryName?: string;
  locationId?: number;
  carrierId?: number;
  locationName?: string;
  carrierName?: string;
  
  // Pola z produktu katalogowego (dla produkty-spakowane)
  catalogProductId?: number | null;
  catalogProductSku?: string | null;
  
  // Pola specyficzne dla formatek
  parentId?: number | null;
  length?: number | null;
  width?: number | null;
  materialThickness?: number | null;
  cz1?: string | null;
  cz2?: string | null;
  furnitureType?: string | null;
  color?: string | null;
  plateType?: string | null;
  edgingMaterial?: string | null;
  edge1?: boolean;
  edge2?: boolean;
  edge3?: boolean;
  edge4?: boolean;
  edge5?: boolean;
  status?: string | null;
  source?: string | null;
  referenceCode?: string | null;
}

interface Location {
  id: number;
  name: string;
  level: number;
  parentId: number | null;
}

interface Carrier {
  id: number;
  name: string;
  type: string;
}

interface MaterialUsage {
  id: number;
  type?: string;
  productionOrderId?: number;
  productionPlanId?: number;
  quantityUsed: number;
  usedAt: string;
  notes?: string;
  // Pola dla rezerwacji planów (type = 'plan_reservation')
  planId?: number;
  planNumber?: string;
  planName?: string;
  planStatus?: string;
  planPriority?: string;
  lineStatus?: string;
  updatedAt?: string;
}

interface InventoryHistory {
  id: number;
  materialId: number;
  operationType: string;
  quantityChange: string;
  quantityBefore: string;
  quantityAfter: string;
  productionOrderId?: number;
  productionPlanId?: number;
  notes?: string;
  documentNumber?: string;
  performedBy?: string;
  createdAt: string;
}

interface FormatkaReservationHistoryItem {
  id: number;
  materialId: number;
  operationType: 'reservation' | 'consumption' | 'cancellation';
  quantityChange: string;
  quantityBefore?: number | null;
  quantityAfter?: number | null;
  productionPlanId?: number | null;
  notes?: string | null;
  documentNumber?: string | null;
  performedBy?: string | null;
  createdAt?: string | null;
  originalStatus: 'ACTIVE' | 'CONSUMED' | 'CANCELLED';
  quantityReserved: number;
  quantityConsumed: number | null;
}

interface InventoryCount {
  countId: number;
  countName: string;
  countStatus: string;
  countCreatedAt: string;
  countFinalizedAt?: string | null;
  expectedQuantity?: string | null;
  countedQuantity?: string | null;
  difference?: number | null;
}

interface ProductHistoryEvent {
  id: number;
  serialNumber: string;
  action: string;
  timestamp: string;
  performedBy?: string;
  details?: string;
  status: string;
  planId?: number;
  planLineId?: number;
  orderId?: string;
  sourceType?: string;
  sourceId?: number;
}

interface ProductHistoryResponse {
  events: ProductHistoryEvent[];
  stats: {
    available: number;
    reserved: number;
    shipped: number;
    returned: number;
    defective: number;
    scrapped: number;
    total: number;
  };
  totalItems: number;
}

const actionConfig: Record<string, { label: string; icon: any; color: string }> = {
  created: { label: 'Przyjęcie', icon: Plus, color: 'text-green-500' },
  reserved: { label: 'Rezerwacja', icon: Lock, color: 'text-blue-500' },
  unreserved: { label: 'Anulowanie rezerwacji', icon: RotateCcw, color: 'text-orange-500' },
  shipped: { label: 'Wysyłka', icon: Truck, color: 'text-purple-500' },
  returned: { label: 'Zwrot', icon: RotateCcw, color: 'text-yellow-500' },
  moved: { label: 'Przeniesienie', icon: MapPin, color: 'text-cyan-500' },
  scrapped: { label: 'Złomowanie', icon: Trash2, color: 'text-red-500' },
  defective: { label: 'Oznaczenie wady', icon: AlertTriangle, color: 'text-red-400' },
  consumed: { label: 'Wydanie', icon: Package, color: 'text-purple-500' },
};

export default function WarehouseMaterialDetail() {
  const params = useParams<{ id: string; category: string }>();
  const [location, setLocation] = useLocation();
  const { toast } = useToast();
  const materialId = parseInt(params.id || "0");
  const category = params.category;
  
  // Inventory adjustment dialog state
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [operationType, setOperationType] = useState<string>("inventory_count");
  const [quantityChange, setQuantityChange] = useState<string>("");
  const [documentNumber, setDocumentNumber] = useState<string>("");
  const [notes, setNotes] = useState<string>("");
  
  // Location editor state
  const [isLocationDialogOpen, setIsLocationDialogOpen] = useState(false);
  const [selectedLocationId, setSelectedLocationId] = useState<string>("");
  const [selectedCarrierId, setSelectedCarrierId] = useState<string>("");

  // Right sidebar collapsed state
  const [isInventoryCountsSectionCollapsed, setIsInventoryCountsSectionCollapsed] = useState(false);

  // Load sidebar collapse state from localStorage on mount (SSR-safe)
  useEffect(() => {
    if (typeof window !== 'undefined') {
      const saved = localStorage.getItem('inventoryCountsSectionCollapsed');
      if (saved === 'true') {
        setIsInventoryCountsSectionCollapsed(true);
      }
    }
  }, []);

  // Save sidebar collapse state to localStorage
  useEffect(() => {
    if (typeof window !== 'undefined') {
      localStorage.setItem('inventoryCountsSectionCollapsed', String(isInventoryCountsSectionCollapsed));
    }
  }, [isInventoryCountsSectionCollapsed]);

  // Determine back path based on category param
  const getBackPath = () => {
    return `/warehouse/${category}`;
  };

  const { data: material, isLoading } = useQuery<Material>({
    queryKey: [`/api/warehouse/materials/${materialId}`, { category }],
    enabled: !!materialId,
  });

  const { data: usageHistory } = useQuery<MaterialUsage[]>({
    queryKey: [`/api/warehouse/materials/${materialId}/usage`, { category }],
    enabled: !!materialId,
  });

  const { data: inventoryHistory } = useQuery<InventoryHistory[]>({
    queryKey: [`/api/warehouse/materials/${materialId}/inventory-history`, { category }],
    enabled: !!materialId,
  });

  // Fetch inventory counts containing this material
  const { data: inventoryCounts, isLoading: isCountsLoading } = useQuery<InventoryCount[]>({
    queryKey: [`/api/warehouse/materials/${materialId}/inventory-counts`, { category }],
    enabled: !!materialId,
  });

  // Fetch locations and carriers for selection
  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ["/api/production/locations"],
  });

  const { data: carriers = [] } = useQuery<Carrier[]>({
    queryKey: ["/api/production/carriers"],
  });

  // Fetch product history for packed products
  const { data: productHistory, isLoading: isHistoryLoading } = useQuery<ProductHistoryResponse>({
    queryKey: [`/api/warehouse/packed-products/${materialId}/history`],
    enabled: !!materialId && category === 'produkty-spakowane',
  });

  // Fetch reservation history for formatki
  const { data: formatkaReservationHistory, isLoading: isReservationHistoryLoading } = useQuery<{
    items: FormatkaReservationHistoryItem[];
    total: number;
    limit: number;
    offset: number;
  }>({
    queryKey: [`/api/warehouse/formatki/${materialId}/reservation-history`],
    enabled: !!materialId && category === 'formatki',
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/warehouse/materials/${id}?category=${category}`);
    },
    onSuccess: () => {
      // Invalidate materials cache so list refreshes
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/materials"] });
      toast({
        title: "Sukces",
        description: "Materiał został usunięty",
      });
      setLocation(getBackPath());
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się usunąć materiału",
        variant: "destructive",
      });
    },
  });

  const inventoryAdjustmentMutation = useMutation({
    mutationFn: async (data: {
      operationType: string;
      quantityChange: number;
      documentNumber?: string;
      notes?: string;
    }) => {
      return await apiRequest("POST", `/api/warehouse/materials/${materialId}/inventory-adjustment?category=${category}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}`] });
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}/inventory-history`] });
      toast({
        title: "Sukces",
        description: "Wpis inwentaryzacyjny został dodany",
      });
      setIsDialogOpen(false);
      resetDialogForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się dodać wpisu inwentaryzacyjnego",
        variant: "destructive",
      });
    },
  });

  const updateLocationMutation = useMutation({
    mutationFn: async (data: {
      locationId: number | null;
      carrierId: number | null;
    }) => {
      return await apiRequest("PATCH", `/api/warehouse/materials/${materialId}?category=${category}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}`] });
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/materials"] });
      toast({
        title: "Sukces",
        description: "Lokalizacja magazynowa została zaktualizowana",
      });
      setIsLocationDialogOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować lokalizacji",
        variant: "destructive",
      });
    },
  });

  const handleDelete = () => {
    if (confirm("Czy na pewno chcesz usunąć ten materiał?")) {
      deleteMutation.mutate(materialId);
    }
  };

  const resetDialogForm = () => {
    setOperationType("inventory_count");
    setQuantityChange("");
    setDocumentNumber("");
    setNotes("");
  };

  const handleSubmitInventoryAdjustment = () => {
    const change = parseFloat(quantityChange);
    if (isNaN(change) || change === 0) {
      toast({
        title: "Błąd",
        description: "Zmiana ilości musi być liczbą różną od zera",
        variant: "destructive",
      });
      return;
    }

    inventoryAdjustmentMutation.mutate({
      operationType,
      quantityChange: change,
      documentNumber: documentNumber || undefined,
      notes: notes || undefined,
    });
  };

  const calculateNewQuantity = () => {
    if (!material) return null;
    const change = parseFloat(quantityChange);
    if (isNaN(change)) return null;
    const currentQty = parseFloat(material.quantity || "0");
    return currentQty + change;
  };

  const handleOpenLocationEditor = () => {
    setSelectedLocationId(material?.locationId?.toString() || "");
    setSelectedCarrierId(material?.carrierId?.toString() || "");
    setIsLocationDialogOpen(true);
  };

  const handleSubmitLocation = () => {
    updateLocationMutation.mutate({
      locationId: selectedLocationId ? parseInt(selectedLocationId) : null,
      carrierId: selectedCarrierId ? parseInt(selectedCarrierId) : null,
    });
  };

  // Archive mutation for packed products
  const archiveMutation = useMutation({
    mutationFn: async (data: { archive: boolean }) => {
      return await apiRequest("PATCH", `/api/warehouse/packed-products/${materialId}/archive`, data);
    },
    onSuccess: (_, variables) => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}`] });
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/materials"] });
      toast({
        title: "Sukces",
        description: variables.archive 
          ? "Produkt został zarchiwizowany" 
          : "Produkt został przywrócony z archiwum",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zmienić statusu archiwizacji",
        variant: "destructive",
      });
    },
  });

  const handleArchiveToggle = () => {
    const isCurrentlyArchived = material?.isArchived;
    const confirmMessage = isCurrentlyArchived
      ? "Czy na pewno chcesz przywrócić ten produkt z archiwum?"
      : "Czy na pewno chcesz zarchiwizować ten produkt?";
    
    if (confirm(confirmMessage)) {
      archiveMutation.mutate({ archive: !isCurrentlyArchived });
    }
  };

  if (isLoading) {
    return (
      <WarehouseLayout category={category}>
        <div className="container mx-auto p-6 space-y-6">
          <Skeleton className="h-10 w-64" />
          <Skeleton className="h-96 w-full" />
        </div>
      </WarehouseLayout>
    );
  }

  if (!material) {
    return (
      <WarehouseLayout category={category}>
        <div className="container mx-auto p-6">
          <Card>
            <CardContent className="pt-6">
              <p className="text-center text-muted-foreground">Materiał nie znaleziony</p>
            </CardContent>
          </Card>
        </div>
      </WarehouseLayout>
    );
  }

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'draft':
        return <Badge variant="secondary">Robocze</Badge>;
      case 'finalized':
        return <Badge variant="default" className="bg-green-600">Zakończone</Badge>;
      default:
        return <Badge variant="outline">{status}</Badge>;
    }
  };

  return (
    <WarehouseLayout category={category}>
      <div className="flex h-screen">
        {/* Main content */}
        <div className="flex-1 overflow-y-auto">
          <div className="p-4 md:p-6 space-y-6">
        {/* Header */}
        <div className="flex items-center gap-4">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => setLocation(getBackPath())}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
          </Button>
          {/* Miniaturka zdjęcia */}
          {(material.primaryImageUrl || (material.gallery && material.gallery.length > 0)) && (
            <div className="h-16 w-16 rounded-md overflow-hidden border bg-muted flex-shrink-0">
              <img
                src={material.primaryImageUrl || material.gallery![0]}
                alt={material.name}
                className="h-full w-full object-cover"
                data-testid="img-material-thumbnail"
              />
            </div>
          )}
          <div className="flex-1">
            <h1 className="text-2xl font-bold">{material.name}</h1>
            <p className="text-sm text-muted-foreground">
              Kod: {material.internalCode}
              {material.supplierCode && ` | Kod dostawcy: ${material.supplierCode}`}
              {material.catalogProductSku && material.catalogProductId && (
                <>
                  {' | SKU: '}
                  <Link 
                    href={`/catalog-products/${material.catalogProductId}`}
                    className="text-primary hover:underline"
                  >
                    {material.catalogProductSku}
                  </Link>
                </>
              )}
            </p>
          </div>
          <div className="flex items-center gap-2">
            <Badge variant={material.isActive ? "default" : "secondary"}>
              {material.isActive ? "Aktywny" : "Nieaktywny"}
            </Badge>
            {category === 'produkty-spakowane' && (
              <Badge variant={material.isArchived ? "secondary" : "outline"}>
                {material.isArchived ? "Zarchiwizowany" : "Aktualny"}
              </Badge>
            )}
            <Link href={`${location}/edit`}>
              <Button
                variant="outline"
                size="sm"
                data-testid="button-edit-material"
              >
                <Pencil className="h-4 w-4 mr-2" />
                Edytuj
              </Button>
            </Link>
            {category === 'produkty-spakowane' && (
              <Button
                variant="outline"
                size="sm"
                onClick={handleArchiveToggle}
                disabled={archiveMutation.isPending}
                data-testid="button-archive-material"
              >
                {material.isArchived ? (
                  <>
                    <ArchiveRestore className="h-4 w-4 mr-2" />
                    Przywróć
                  </>
                ) : (
                  <>
                    <Archive className="h-4 w-4 mr-2" />
                    Archiwizuj
                  </>
                )}
              </Button>
            )}
            <Button
              variant="outline"
              size="sm"
              onClick={handleDelete}
              data-testid="button-delete-material"
            >
              <Trash2 className="h-4 w-4 mr-2" />
              Usuń
            </Button>
          </div>
        </div>

        {/* Location Card */}
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle className="flex items-center gap-2">
                  <MapPin className="h-5 w-5" />
                  Lokalizacja magazynowa
                </CardTitle>
                <CardDescription className="mt-1">
                  Miejsce składowania materiału w magazynie
                </CardDescription>
              </div>
              <Button
                size="sm"
                variant="outline"
                onClick={handleOpenLocationEditor}
                data-testid="button-edit-location"
              >
                <MapPin className="h-4 w-4 mr-1" />
                Edytuj
              </Button>
            </div>
          </CardHeader>
          <CardContent>
            <div className="flex items-center gap-4">
              {material.locationName ? (
                <>
                  <div className="flex items-center gap-2">
                    <Building2 className="h-5 w-5 text-muted-foreground" />
                    <div>
                      <p className="text-sm text-muted-foreground">Hala</p>
                      <p className="font-medium">{material.locationName}</p>
                    </div>
                  </div>
                  {material.carrierName && (
                    <>
                      <div className="h-8 w-px bg-border" />
                      <div className="flex items-center gap-2">
                        <Truck className="h-5 w-5 text-muted-foreground" />
                        <div>
                          <p className="text-sm text-muted-foreground">Nośnik</p>
                          <p className="font-medium">{material.carrierName}</p>
                        </div>
                      </div>
                    </>
                  )}
                </>
              ) : (
                <p className="text-sm text-muted-foreground">
                  Lokalizacja nie została przypisana
                </p>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Inventory Stock Card */}
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle className="flex items-center gap-2">
                  <Package className="h-5 w-5" />
                  Stan magazynowy
                </CardTitle>
                <CardDescription className="mt-1">
                  Bieżący stan zapasów dla tego materiału
                </CardDescription>
              </div>
              <div className="flex items-center gap-3">
                {category === 'produkty-spakowane' ? (
                <>
                  <Link href={`/warehouse/produkty-spakowane/${materialId}/items`}>
                    <Button
                      variant="outline"
                      size="sm"
                      data-testid="button-view-items"
                    >
                      <List className="h-4 w-4 mr-2" />
                      Pojedyncze sztuki
                    </Button>
                  </Link>
                  <Badge variant="outline" className="text-lg font-bold px-4 py-2">
                    <Package className="h-4 w-4 mr-2" />
                    Stan mag.: {material.quantity || "0"} {material.unitOfMeasure}
                  </Badge>
                  <Badge variant="outline" className="text-lg font-bold px-4 py-2 bg-yellow-100 dark:bg-yellow-900/30 border-yellow-500">
                    Zarezerwowane: {material.reservedQuantity || "0"} {material.unitOfMeasure}
                  </Badge>
                  <Badge variant="outline" className="text-lg font-bold px-4 py-2 bg-green-100 dark:bg-green-900/30 border-green-500">
                    Dostępne: {material.availableQuantity || "0"} {material.unitOfMeasure}
                  </Badge>
                </>
              ) : (
                <Badge variant="outline" className="text-lg font-bold px-4 py-2">
                  {material.quantity || "0"} {material.unitOfMeasure}
                </Badge>
              )}
                <Button
                  size="sm"
                  onClick={() => setIsDialogOpen(true)}
                  data-testid="button-add-inventory-entry"
                >
                  <Plus className="h-4 w-4 mr-1" />
                  Dodaj wpis
                </Button>
              </div>
            </div>
          </CardHeader>
        </Card>

        {/* Product History Card for produkty-spakowane */}
        {category === 'produkty-spakowane' && (
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle className="flex items-center gap-2">
                    <History className="h-5 w-5" />
                    Historia ruchu produktu
                  </CardTitle>
                  <CardDescription className="mt-1">
                    Zbiorcza historia wszystkich sztuk - od przyjęcia do magazynu po wysyłkę
                  </CardDescription>
                </div>
                <Link href={`/warehouse/produkty-spakowane/${materialId}/items`}>
                  <Button variant="outline" size="sm" data-testid="button-view-all-items">
                    <List className="h-4 w-4 mr-2" />
                    Wszystkie sztuki
                  </Button>
                </Link>
              </div>
            </CardHeader>
            <CardContent>
              {isHistoryLoading ? (
                <div className="space-y-3">
                  {[...Array(5)].map((_, i) => (
                    <Skeleton key={i} className="h-16 w-full" />
                  ))}
                </div>
              ) : productHistory?.events && productHistory.events.length > 0 ? (
                <Tabs defaultValue="all" className="w-full">
                  <TabsList className="mb-4">
                    <TabsTrigger value="all">
                      Wszystkie ({productHistory.events.length})
                    </TabsTrigger>
                    <TabsTrigger value="created">
                      Przyjęcia ({productHistory.events.filter(e => e.action === 'created').length})
                    </TabsTrigger>
                    <TabsTrigger value="reserved">
                      Rezerwacje ({productHistory.events.filter(e => e.action === 'reserved' || e.action === 'unreserved').length})
                    </TabsTrigger>
                    <TabsTrigger value="scrapped">
                      Złomowania ({productHistory.events.filter(e => e.action === 'scrapped').length})
                    </TabsTrigger>
                  </TabsList>
                  
                  <TabsContent value="all">
                    <ScrollArea className="h-[400px] pr-4">
                      <div className="relative">
                        <div className="absolute left-[19px] top-0 bottom-0 w-0.5 bg-border" />
                        <div className="space-y-4">
                          {productHistory.events.slice(0, 50).map((event, index) => {
                            const config = actionConfig[event.action] || { label: event.action, icon: History, color: 'text-muted-foreground' };
                            const EventIcon = config.icon;
                            return (
                              <div key={`${event.id}-${index}`} className="relative flex gap-4 pl-0">
                                <div className={`relative z-10 flex h-10 w-10 shrink-0 items-center justify-center rounded-full border-2 bg-background ${config.color.replace('text-', 'border-')}`}>
                                  <EventIcon className={`w-4 h-4 ${config.color}`} />
                                </div>
                                <div className="flex-1 pt-1">
                                  <div className="flex items-center gap-2 flex-wrap">
                                    <Badge variant="outline" className="font-mono text-xs">
                                      {event.serialNumber}
                                    </Badge>
                                    <span className="font-medium text-sm">{config.label}</span>
                                    <span className="text-xs text-muted-foreground">
                                      {format(new Date(event.timestamp), "dd.MM.yyyy HH:mm", { locale: pl })}
                                    </span>
                                  </div>
                                  <p className="text-sm text-muted-foreground mt-0.5">
                                    {event.details}
                                  </p>
                                  <div className="flex items-center gap-2 mt-1 flex-wrap">
                                    {event.performedBy && (
                                      <Badge variant="outline" className="text-xs gap-1">
                                        <User className="w-3 h-3" />
                                        {event.performedBy}
                                      </Badge>
                                    )}
                                    {event.planId && (
                                      <Link href={`/production/plans/${event.planId}`}>
                                        <Badge variant="outline" className="text-xs gap-1 cursor-pointer hover:bg-accent">
                                          <ClipboardList className="w-3 h-3" />
                                          Plan #{event.planId}
                                        </Badge>
                                      </Link>
                                    )}
                                    {event.sourceType === 'inventory_count' && event.sourceId && (
                                      <Link href={`/warehouse/inventory-counts/${event.sourceId}`}>
                                        <Badge variant="outline" className="text-xs gap-1 cursor-pointer hover:bg-accent bg-yellow-500/10 text-yellow-700 border-yellow-300">
                                          <FileCheck className="w-3 h-3" />
                                          Spis #{event.sourceId}
                                        </Badge>
                                      </Link>
                                    )}
                                  </div>
                                </div>
                              </div>
                            );
                          })}
                          {productHistory.events.length > 50 && (
                            <div className="text-center py-4 text-muted-foreground text-sm">
                              Pokazano 50 z {productHistory.events.length} wydarzeń. 
                              <Link href={`/warehouse/produkty-spakowane/${materialId}/items`} className="text-primary ml-1 hover:underline">
                                Zobacz wszystkie sztuki
                              </Link>
                            </div>
                          )}
                        </div>
                      </div>
                    </ScrollArea>
                  </TabsContent>
                  
                  <TabsContent value="created">
                    <ScrollArea className="h-[400px] pr-4">
                      <div className="relative">
                        <div className="absolute left-[19px] top-0 bottom-0 w-0.5 bg-border" />
                        <div className="space-y-4">
                          {productHistory.events.filter(e => e.action === 'created').slice(0, 50).map((event, index) => {
                            const EventIcon = Plus;
                            return (
                              <div key={`${event.id}-${index}`} className="relative flex gap-4 pl-0">
                                <div className="relative z-10 flex h-10 w-10 shrink-0 items-center justify-center rounded-full border-2 bg-background border-green-500">
                                  <EventIcon className="w-4 h-4 text-green-500" />
                                </div>
                                <div className="flex-1 pt-1">
                                  <div className="flex items-center gap-2 flex-wrap">
                                    <Badge variant="outline" className="font-mono text-xs">
                                      {event.serialNumber}
                                    </Badge>
                                    <span className="text-xs text-muted-foreground">
                                      {format(new Date(event.timestamp), "dd.MM.yyyy HH:mm", { locale: pl })}
                                    </span>
                                  </div>
                                  <p className="text-sm text-muted-foreground mt-0.5">
                                    {event.details}
                                  </p>
                                  {event.sourceType === 'inventory_count' && event.sourceId && (
                                    <Link href={`/warehouse/inventory-counts/${event.sourceId}`}>
                                      <Badge variant="outline" className="text-xs gap-1 cursor-pointer hover:bg-accent bg-yellow-500/10 text-yellow-700 border-yellow-300 mt-1">
                                        <FileCheck className="w-3 h-3" />
                                        Spis #{event.sourceId}
                                      </Badge>
                                    </Link>
                                  )}
                                </div>
                              </div>
                            );
                          })}
                        </div>
                      </div>
                    </ScrollArea>
                  </TabsContent>
                  
                  <TabsContent value="reserved">
                    <ScrollArea className="h-[400px] pr-4">
                      <div className="relative">
                        <div className="absolute left-[19px] top-0 bottom-0 w-0.5 bg-border" />
                        <div className="space-y-4">
                          {productHistory.events.filter(e => e.action === 'reserved' || e.action === 'unreserved').slice(0, 50).map((event, index) => {
                            const isReserved = event.action === 'reserved';
                            const EventIcon = isReserved ? Lock : RotateCcw;
                            const color = isReserved ? 'text-blue-500 border-blue-500' : 'text-orange-500 border-orange-500';
                            return (
                              <div key={`${event.id}-${index}`} className="relative flex gap-4 pl-0">
                                <div className={`relative z-10 flex h-10 w-10 shrink-0 items-center justify-center rounded-full border-2 bg-background ${color.split(' ')[1]}`}>
                                  <EventIcon className={`w-4 h-4 ${color.split(' ')[0]}`} />
                                </div>
                                <div className="flex-1 pt-1">
                                  <div className="flex items-center gap-2 flex-wrap">
                                    <Badge variant="outline" className="font-mono text-xs">
                                      {event.serialNumber}
                                    </Badge>
                                    <span className="font-medium text-sm">
                                      {isReserved ? 'Rezerwacja' : 'Anulowanie rezerwacji'}
                                    </span>
                                    <span className="text-xs text-muted-foreground">
                                      {format(new Date(event.timestamp), "dd.MM.yyyy HH:mm", { locale: pl })}
                                    </span>
                                  </div>
                                  <p className="text-sm text-muted-foreground mt-0.5">
                                    {event.details}
                                  </p>
                                  <div className="flex items-center gap-2 mt-1 flex-wrap">
                                    {event.performedBy && (
                                      <Badge variant="outline" className="text-xs gap-1">
                                        <User className="w-3 h-3" />
                                        {event.performedBy}
                                      </Badge>
                                    )}
                                    {event.planId && (
                                      <Link href={`/production/plans/${event.planId}`}>
                                        <Badge variant="outline" className="text-xs gap-1 cursor-pointer hover:bg-accent">
                                          <ClipboardList className="w-3 h-3" />
                                          Plan #{event.planId}
                                        </Badge>
                                      </Link>
                                    )}
                                  </div>
                                </div>
                              </div>
                            );
                          })}
                        </div>
                      </div>
                    </ScrollArea>
                  </TabsContent>
                  
                  <TabsContent value="scrapped">
                    <ScrollArea className="h-[400px] pr-4">
                      <div className="relative">
                        <div className="absolute left-[19px] top-0 bottom-0 w-0.5 bg-border" />
                        <div className="space-y-4">
                          {productHistory.events.filter(e => e.action === 'scrapped').slice(0, 50).map((event, index) => {
                            return (
                              <div key={`${event.id}-${index}`} className="relative flex gap-4 pl-0">
                                <div className="relative z-10 flex h-10 w-10 shrink-0 items-center justify-center rounded-full border-2 bg-background border-red-500">
                                  <Trash2 className="w-4 h-4 text-red-500" />
                                </div>
                                <div className="flex-1 pt-1">
                                  <div className="flex items-center gap-2 flex-wrap">
                                    <Badge variant="outline" className="font-mono text-xs">
                                      {event.serialNumber}
                                    </Badge>
                                    <span className="font-medium text-sm text-red-500">Złomowanie</span>
                                    <span className="text-xs text-muted-foreground">
                                      {format(new Date(event.timestamp), "dd.MM.yyyy HH:mm", { locale: pl })}
                                    </span>
                                  </div>
                                  <p className="text-sm text-muted-foreground mt-0.5">
                                    {event.details}
                                  </p>
                                  {event.performedBy && (
                                    <Badge variant="outline" className="text-xs gap-1 mt-1">
                                      <User className="w-3 h-3" />
                                      {event.performedBy}
                                    </Badge>
                                  )}
                                </div>
                              </div>
                            );
                          })}
                        </div>
                      </div>
                    </ScrollArea>
                  </TabsContent>
                </Tabs>
              ) : (
                <div className="text-center py-8 text-muted-foreground">
                  <Clock className="h-12 w-12 mx-auto mb-3 opacity-50" />
                  <p>Brak historii ruchu dla tego produktu</p>
                  <p className="text-sm mt-1">Historia pojawi się po dodaniu sztuk do magazynu</p>
                </div>
              )}
            </CardContent>
          </Card>
        )}

        {/* Historia rezerwacji formatek */}
        {category === 'formatki' && (
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle className="flex items-center gap-2">
                    <History className="h-5 w-5" />
                    Historia rezerwacji
                  </CardTitle>
                  <CardDescription className="mt-1">
                    Zbiorcza historia rezerwacji formatki - od przyjęcia do magazynu po wydanie
                  </CardDescription>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              {isReservationHistoryLoading ? (
                <div className="flex items-center justify-center py-8">
                  <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
                </div>
              ) : formatkaReservationHistory?.items && formatkaReservationHistory.items.length > 0 ? (
              <ScrollArea className="h-[300px] pr-4">
                <div className="relative">
                  <div className="absolute left-[19px] top-0 bottom-0 w-0.5 bg-border" />
                  <div className="space-y-4">
                    {formatkaReservationHistory.items.map((entry, index) => {
                      const isReservation = entry.operationType === 'reservation';
                      const isConsumption = entry.operationType === 'consumption';
                      const isCancellation = entry.operationType === 'cancellation';
                      
                      const iconClass = isReservation ? 'text-blue-500 border-blue-500' : 
                                       isConsumption ? 'text-purple-500 border-purple-500' : 
                                       isCancellation ? 'text-orange-500 border-orange-500' : 
                                       'text-muted-foreground border-muted-foreground';
                      const IconComponent = isReservation ? Lock : 
                                           isConsumption ? Package : 
                                           isCancellation ? RotateCcw : History;
                      const label = isReservation ? 'Rezerwacja' : 
                                   isConsumption ? 'Wydanie' : 
                                   isCancellation ? 'Anulowanie rezerwacji' : 'Operacja';
                      
                      return (
                        <div key={entry.id || index} className="relative flex gap-4 pl-0">
                          <div className={`relative z-10 flex h-10 w-10 shrink-0 items-center justify-center rounded-full border-2 bg-background ${iconClass}`}>
                            <IconComponent className={`w-4 h-4 ${iconClass.split(' ')[0]}`} />
                          </div>
                          <div className="flex-1 pt-1">
                            <div className="flex items-center gap-2 flex-wrap">
                              <span className="font-medium text-sm">{label}</span>
                              <Badge 
                                variant="outline" 
                                className={`text-xs ${
                                  entry.quantityChange?.startsWith('-') 
                                    ? 'bg-red-100 dark:bg-red-900/30 text-red-700 dark:text-red-300 border-red-300' 
                                    : 'bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300 border-green-300'
                                }`}
                              >
                                {entry.quantityChange} szt.
                              </Badge>
                              <span className="text-xs text-muted-foreground">
                                {entry.createdAt ? format(new Date(entry.createdAt), "dd.MM.yyyy HH:mm", { locale: pl }) : '-'}
                              </span>
                            </div>
                            <p className="text-sm text-muted-foreground mt-0.5">
                              {entry.notes}
                            </p>
                            <div className="flex items-center gap-2 mt-1 flex-wrap">
                              {entry.performedBy && (
                                <Badge variant="outline" className="text-xs gap-1">
                                  <User className="w-3 h-3" />
                                  {entry.performedBy}
                                </Badge>
                              )}
                              {entry.documentNumber && (
                                <Link href={`/production/plans/${entry.productionPlanId}`}>
                                  <Badge variant="outline" className="text-xs gap-1 cursor-pointer hover:bg-accent">
                                    <ClipboardList className="w-3 h-3" />
                                    {entry.documentNumber}
                                  </Badge>
                                </Link>
                              )}
                            </div>
                          </div>
                        </div>
                      );
                    })}
                  </div>
                </div>
              </ScrollArea>
              ) : (
                <div className="text-center py-8 text-muted-foreground">
                  <Clock className="h-12 w-12 mx-auto mb-3 opacity-50" />
                  <p>Brak historii rezerwacji dla tej formatki</p>
                  <p className="text-sm mt-1">Historia pojawi się gdy formatka zostanie zarezerwowana do planu produkcyjnego</p>
                </div>
              )}
            </CardContent>
          </Card>
        )}

        {/* Formatki-specific Info Card */}
        {category === 'formatki' && (
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileText className="h-5 w-5" />
                Parametry formatki
              </CardTitle>
              <CardDescription className="mt-1">
                Szczegółowe informacje o formatce
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                {/* Wymiary */}
                {material.specifications?.length && (
                  <div>
                    <p className="text-muted-foreground mb-1">Długość</p>
                    <p className="font-medium">{material.specifications.length} cm</p>
                  </div>
                )}
                {material.specifications?.width && (
                  <div>
                    <p className="text-muted-foreground mb-1">Szerokość</p>
                    <p className="font-medium">{material.specifications.width} cm</p>
                  </div>
                )}
                {material.specifications?.thickness && (
                  <div>
                    <p className="text-muted-foreground mb-1">Grubość</p>
                    <p className="font-medium">{material.specifications.thickness} mm</p>
                  </div>
                )}
                {material.specifications?.parentId && (
                  <div>
                    <p className="text-muted-foreground mb-1">Rodzic (ID)</p>
                    <p className="font-medium">{material.specifications.parentId}</p>
                  </div>
                )}
                
                {/* Części nazwy i typ */}
                {material.specifications?.cz1 && (
                  <div>
                    <p className="text-muted-foreground mb-1">CZ1</p>
                    <p className="font-medium">{material.specifications.cz1}</p>
                  </div>
                )}
                {material.specifications?.cz2 && (
                  <div>
                    <p className="text-muted-foreground mb-1">CZ2</p>
                    <p className="font-medium">{material.specifications.cz2}</p>
                  </div>
                )}
                {material.specifications?.furnitureType && (
                  <div>
                    <p className="text-muted-foreground mb-1">Typ mebla</p>
                    <p className="font-medium">{material.specifications.furnitureType}</p>
                  </div>
                )}
                {material.specifications?.colorCode && (
                  <div>
                    <p className="text-muted-foreground mb-1">Kolor</p>
                    <p className="font-medium">{material.specifications.colorCode}</p>
                  </div>
                )}
                
                {/* Materiały */}
                {material.specifications?.boardCode && (
                  <div>
                    <p className="text-muted-foreground mb-1">Płyta</p>
                    <p className="font-medium">{material.specifications.boardCode}</p>
                  </div>
                )}
                {material.specifications?.edgingCode && (
                  <div>
                    <p className="text-muted-foreground mb-1">Obrzeże</p>
                    <p className="font-medium">{material.specifications.edgingCode}</p>
                  </div>
                )}
                
                {/* Status i źródło */}
                {(material.specifications?.isEdged || material.specifications?.isDrilled) && (
                  <div>
                    <p className="text-muted-foreground mb-1">Status</p>
                    <p className="font-medium">
                      {material.specifications?.isEdged ? 'Oklejone' : material.specifications?.isDrilled ? 'Wiercone' : '-'}
                    </p>
                  </div>
                )}
                {material.specifications?.source && (
                  <div>
                    <p className="text-muted-foreground mb-1">Źródło</p>
                    <p className="font-medium">{material.specifications.source}</p>
                  </div>
                )}
                {material.specifications?.sourceReference && (
                  <div>
                    <p className="text-muted-foreground mb-1">Referencja</p>
                    <p className="font-medium">{material.specifications.sourceReference}</p>
                  </div>
                )}
              </div>
              
              {/* Kompletne (Obrzeża) */}
              {(material.specifications?.edge1 || material.specifications?.edge2 || material.specifications?.edge3 || material.specifications?.edge4) && (
                <div className="mt-4 pt-4 border-t">
                  <p className="text-muted-foreground mb-2">Kompletne (obrzeża)</p>
                  <div className="flex items-center gap-2 flex-wrap">
                    {material.specifications?.edge1 && <Badge variant="outline">E1 ({material.specifications.edge1})</Badge>}
                    {material.specifications?.edge2 && <Badge variant="outline">E2 ({material.specifications.edge2})</Badge>}
                    {material.specifications?.edge3 && <Badge variant="outline">E3 ({material.specifications.edge3})</Badge>}
                    {material.specifications?.edge4 && <Badge variant="outline">E4 ({material.specifications.edge4})</Badge>}
                  </div>
                </div>
              )}
              
              {/* Koszt materiału */}
              {(material.specifications?.boardPrice || material.specifications?.calculatedBoardCost || material.specifications?.calculatedEdgingCost || material.specifications?.calculatedMaterialCost) && (
                <div className="mt-4 pt-4 border-t">
                  <p className="text-muted-foreground mb-2">Koszt materiału</p>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                    {/* Płyta z linkiem */}
                    <div>
                      <p className="text-muted-foreground mb-1">Płyta</p>
                      {material.specifications?.boardId ? (
                        <Link href={`/warehouse/plyty/${material.specifications.boardId}`}>
                          <span className="font-medium text-primary hover:underline cursor-pointer">
                            {material.specifications?.boardName || material.specifications?.boardCode || '-'}
                          </span>
                        </Link>
                      ) : (
                        <p className="font-medium">{material.specifications?.boardCode || '-'}</p>
                      )}
                      {material.specifications?.boardPrice && (
                        <p className="text-xs text-muted-foreground">
                          {Number(material.specifications.boardPrice).toFixed(2)} zł/m²
                        </p>
                      )}
                    </div>
                    
                    {/* Obrzeże z linkiem */}
                    <div>
                      <p className="text-muted-foreground mb-1">Obrzeże</p>
                      {material.specifications?.edgingId ? (
                        <Link href={`/warehouse/obrzeza/${material.specifications.edgingId}`}>
                          <span className="font-medium text-primary hover:underline cursor-pointer">
                            {material.specifications?.edgingName || material.specifications?.edgingCode || '-'}
                          </span>
                        </Link>
                      ) : (
                        <p className="font-medium">{material.specifications?.edgingCode || '-'}</p>
                      )}
                      {material.specifications?.edgingPrice && (
                        <p className="text-xs text-muted-foreground">
                          {Number(material.specifications.edgingPrice).toFixed(2)} zł/mb
                        </p>
                      )}
                    </div>
                    
                    {/* Koszt płyty */}
                    <div>
                      <p className="text-muted-foreground mb-1">Koszt płyty</p>
                      <p className="font-medium">
                        {material.specifications?.calculatedBoardCost 
                          ? `${Number(material.specifications.calculatedBoardCost).toFixed(4)} zł`
                          : '-'
                        }
                      </p>
                      {material.specifications?.length && material.specifications?.width && (
                        <p className="text-xs text-muted-foreground">
                          {((Number(material.specifications.length) * Number(material.specifications.width)) / 1000000).toFixed(4)} m²
                        </p>
                      )}
                    </div>
                    
                    {/* Koszt obrzeża */}
                    <div>
                      <p className="text-muted-foreground mb-1">Koszt obrzeża</p>
                      <p className="font-medium">
                        {material.specifications?.calculatedEdgingCost 
                          ? `${Number(material.specifications.calculatedEdgingCost).toFixed(4)} zł`
                          : (material.specifications?.edge1 === 'T' && material.specifications?.edge2 === 'T' && material.specifications?.edge3 === 'T' && material.specifications?.edge4 === 'T') 
                            ? 'brak ceny obrzeża'
                            : 'nie wszystkie krawędzie oklejone'
                        }
                      </p>
                      {material.specifications?.length && material.specifications?.width && material.specifications?.edge1 === 'T' && material.specifications?.edge2 === 'T' && material.specifications?.edge3 === 'T' && material.specifications?.edge4 === 'T' && (
                        <p className="text-xs text-muted-foreground">
                          {(2 * (Number(material.specifications.length) + Number(material.specifications.width)) / 1000).toFixed(3)} mb
                        </p>
                      )}
                    </div>
                    
                    {/* Łączny koszt */}
                    <div className="col-span-2 md:col-span-4 pt-2 border-t">
                      <p className="text-muted-foreground mb-1">Łączny koszt materiału</p>
                      <p className="font-medium text-lg text-primary">
                        {material.specifications?.calculatedMaterialCost 
                          ? `${Number(material.specifications.calculatedMaterialCost).toFixed(4)} zł`
                          : '-'
                        }
                      </p>
                    </div>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>
        )}

        <div className="grid grid-cols-1 gap-6">
          {/* Main Info Card */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Package className="h-5 w-5" />
                Szczegóły materiału
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              {material.description && (
                <div>
                  <h3 className="font-medium mb-2">Opis</h3>
                  <p className="text-sm text-muted-foreground">{material.description}</p>
                </div>
              )}

              {material.specifications && Object.keys(material.specifications).length > 0 && (
                <div>
                  <h3 className="font-medium mb-2">Specyfikacja</h3>
                  
                  {/* Specjalna sekcja dla formatek */}
                  {material.specifications.category === 'formatki' ? (
                    <div className="space-y-4">
                      {/* Wymiary */}
                      {(material.specifications.length || material.specifications.width || material.specifications.thickness) && (
                        <div>
                          <p className="text-sm text-muted-foreground mb-1">Wymiary</p>
                          <p className="font-medium">
                            {material.specifications.length} × {material.specifications.width} × {material.specifications.thickness} mm
                          </p>
                        </div>
                      )}
                      
                      {/* Kod płyty i obrzeża */}
                      <div className="grid grid-cols-2 gap-4">
                        {material.specifications.boardCode && (
                          <div>
                            <p className="text-sm text-muted-foreground mb-1">Kod płyty</p>
                            <p className="font-medium">{material.specifications.boardCode}</p>
                          </div>
                        )}
                        {material.specifications.edgingCode && (
                          <div>
                            <p className="text-sm text-muted-foreground mb-1">Kod obrzeża</p>
                            <p className="font-medium">{material.specifications.edgingCode}</p>
                          </div>
                        )}
                      </div>
                      
                      {/* Kolor */}
                      {(material.specifications.colorCode || material.specifications.colorName) && (
                        <div>
                          <p className="text-sm text-muted-foreground mb-1">Kolor</p>
                          <div className="flex items-center gap-2">
                            {material.specifications.colorHex && (
                              <div 
                                className="w-6 h-6 rounded border border-border" 
                                style={{ backgroundColor: material.specifications.colorHex }}
                              />
                            )}
                            <p className="font-medium">
                              {material.specifications.colorName || material.specifications.colorCode}
                            </p>
                          </div>
                        </div>
                      )}
                      
                      {/* Obrzeża krawędzi */}
                      {(material.specifications.edge1 || material.specifications.edge2 || material.specifications.edge3 || material.specifications.edge4) && (
                        <div>
                          <p className="text-sm text-muted-foreground mb-1">Obrzeża krawędzi</p>
                          <div className="grid grid-cols-2 gap-2 text-sm">
                            {material.specifications.edge1 && (
                              <div>
                                <span className="text-muted-foreground">Krawędź 1:</span>
                                <span className="font-medium ml-2">{material.specifications.edge1}</span>
                              </div>
                            )}
                            {material.specifications.edge2 && (
                              <div>
                                <span className="text-muted-foreground">Krawędź 2:</span>
                                <span className="font-medium ml-2">{material.specifications.edge2}</span>
                              </div>
                            )}
                            {material.specifications.edge3 && (
                              <div>
                                <span className="text-muted-foreground">Krawędź 3:</span>
                                <span className="font-medium ml-2">{material.specifications.edge3}</span>
                              </div>
                            )}
                            {material.specifications.edge4 && (
                              <div>
                                <span className="text-muted-foreground">Krawędź 4:</span>
                                <span className="font-medium ml-2">{material.specifications.edge4}</span>
                              </div>
                            )}
                          </div>
                        </div>
                      )}
                      
                      {/* Status obróbki */}
                      <div className="grid grid-cols-2 gap-4">
                        {material.specifications.isDrilled !== undefined && (
                          <div>
                            <p className="text-sm text-muted-foreground mb-1">Wiercone</p>
                            <Badge variant={material.specifications.isDrilled ? "default" : "secondary"}>
                              {material.specifications.isDrilled ? "Tak" : "Nie"}
                            </Badge>
                          </div>
                        )}
                        {material.specifications.isEdged !== undefined && (
                          <div>
                            <p className="text-sm text-muted-foreground mb-1">Obrzeżowane</p>
                            <Badge variant={material.specifications.isEdged ? "default" : "secondary"}>
                              {material.specifications.isEdged ? "Tak" : "Nie"}
                            </Badge>
                          </div>
                        )}
                      </div>
                      
                      {/* Źródło */}
                      {material.specifications.source && (
                        <div>
                          <p className="text-sm text-muted-foreground mb-1">Źródło</p>
                          <div className="space-y-1">
                            <p className="font-medium">{material.specifications.source}</p>
                            {material.specifications.sourceReference && (
                              <p className="text-sm text-muted-foreground">
                                Referencja: {material.specifications.sourceReference}
                              </p>
                            )}
                          </div>
                        </div>
                      )}
                      
                      {/* Dodatkowe informacje */}
                      {(material.specifications.cz1 || material.specifications.cz2 || material.specifications.furnitureType) && (
                        <div className="grid grid-cols-2 gap-4 pt-4 border-t">
                          {material.specifications.cz1 && (
                            <div>
                              <p className="text-sm text-muted-foreground">CZ1</p>
                              <p className="font-medium">{material.specifications.cz1}</p>
                            </div>
                          )}
                          {material.specifications.cz2 && (
                            <div>
                              <p className="text-sm text-muted-foreground">CZ2</p>
                              <p className="font-medium">{material.specifications.cz2}</p>
                            </div>
                          )}
                          {material.specifications.furnitureType && (
                            <div>
                              <p className="text-sm text-muted-foreground">Typ mebla</p>
                              <p className="font-medium">{material.specifications.furnitureType}</p>
                            </div>
                          )}
                        </div>
                      )}
                    </div>
                  ) : (
                    /* Standardowa sekcja dla innych materiałów */
                    <dl className="grid grid-cols-2 gap-2 text-sm">
                      {Object.entries(material.specifications)
                        .filter(([key]) => key !== 'category')
                        .map(([key, value]) => (
                          <div key={key}>
                            <dt className="text-muted-foreground">{key}:</dt>
                            <dd className="font-medium">{String(value)}</dd>
                          </div>
                        ))}
                    </dl>
                  )}
                </div>
              )}

              <div className="grid grid-cols-2 gap-4 pt-4 border-t">
                {material.groupName && (
                  <div>
                    <p className="text-sm text-muted-foreground">Grupa</p>
                    <p className="font-medium">{material.groupName}</p>
                  </div>
                )}
                {material.unitOfMeasure && (
                  <div>
                    <p className="text-sm text-muted-foreground">Jednostka miary</p>
                    <p className="font-medium">{material.unitOfMeasure}</p>
                  </div>
                )}
                {material.price && (
                  <div>
                    <p className="text-sm text-muted-foreground">Cena</p>
                    <p className="font-medium">{material.price} PLN</p>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Image Card */}
          {material.primaryImageUrl && (
            <Card>
              <CardHeader>
                <CardTitle>Zdjęcie</CardTitle>
              </CardHeader>
              <CardContent>
                <img
                  src={material.primaryImageUrl}
                  alt={material.name}
                  className="w-full h-auto rounded-md"
                />
              </CardContent>
            </Card>
          )}
        </div>

        {/* Inventory History */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <History className="h-5 w-5" />
              Historia inwentaryzacji
            </CardTitle>
            <CardDescription>
              Wszystkie zmiany stanu magazynowego z powodami (inwentaryzacja, zużycie materiału, reklamacja)
            </CardDescription>
          </CardHeader>
          <CardContent>
            {!inventoryHistory || inventoryHistory.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-8">
                Brak historii zmian
              </p>
            ) : (
              <div className="space-y-2">
                {inventoryHistory.map((entry) => (
                  <div
                    key={entry.id}
                    className="flex items-start justify-between p-3 border rounded-md"
                  >
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-1">
                        <Badge variant={
                          entry.operationType === 'inventory_count' ? 'default' :
                          entry.operationType === 'material_usage' ? 'secondary' :
                          entry.operationType === 'complaint' ? 'destructive' :
                          entry.operationType === 'purchase' ? 'default' :
                          entry.operationType === 'adjustment' ? 'outline' :
                          'outline'
                        }>
                          {entry.operationType === 'inventory_count' ? 'Inwentaryzacja' :
                           entry.operationType === 'material_usage' ? 'Zużycie materiału' :
                           entry.operationType === 'complaint' ? 'Reklamacja' :
                           entry.operationType === 'purchase' ? 'Zakup' :
                           entry.operationType === 'adjustment' ? 'Korekta' :
                           entry.operationType}
                        </Badge>
                        <span className="font-medium">
                          {parseFloat(entry.quantityChange) > 0 ? '+' : ''}
                          {entry.quantityChange} {material.unitOfMeasure}
                        </span>
                      </div>
                      <p className="text-sm text-muted-foreground">
                        Stan: {entry.quantityBefore} → {entry.quantityAfter} {material.unitOfMeasure}
                      </p>
                      <p className="text-sm text-muted-foreground">
                        {new Date(entry.createdAt).toLocaleString("pl-PL")}
                        {entry.performedBy && ` • ${entry.performedBy}`}
                      </p>
                      {entry.notes && (
                        <p className="text-sm text-muted-foreground mt-1">{entry.notes}</p>
                      )}
                      {entry.documentNumber && (
                        <p className="text-sm text-muted-foreground">Dokument: {entry.documentNumber}</p>
                      )}
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Usage History */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <History className="h-5 w-5" />
              Historia zużycia w produkcji
            </CardTitle>
            <CardDescription>
              Szczegółowa historia użycia tego materiału w zleceniach produkcyjnych
            </CardDescription>
          </CardHeader>
          <CardContent>
            {!usageHistory || usageHistory.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-8">
                Brak historii zużycia
              </p>
            ) : (
              <div className="space-y-2">
                {usageHistory.map((usage) => (
                  <div
                    key={usage.id}
                    className="flex items-center justify-between p-3 border rounded-md"
                  >
                    <div className="flex-1">
                      {usage.type === 'plan_reservation' ? (
                        <>
                          <div className="flex items-center gap-2 mb-1">
                            <p className="font-medium">
                              Rezerwacja: {usage.quantityUsed} {material.unitOfMeasure}
                            </p>
                            {usage.lineStatus === 'pending' && (
                              <Badge variant="outline" className="bg-yellow-100 dark:bg-yellow-900/30 text-xs">
                                Oczekuje
                              </Badge>
                            )}
                            {usage.lineStatus === 'in_progress' && (
                              <Badge variant="outline" className="bg-blue-100 dark:bg-blue-900/30 text-xs">
                                W produkcji
                              </Badge>
                            )}
                            {usage.lineStatus === 'completed' && (
                              <Badge variant="outline" className="bg-green-100 dark:bg-green-900/30 text-xs">
                                Zakończone
                              </Badge>
                            )}
                            {usage.lineStatus === 'cancelled' && (
                              <Badge variant="outline" className="bg-gray-100 dark:bg-gray-900/30 text-xs">
                                Anulowane
                              </Badge>
                            )}
                          </div>
                          <p className="text-sm text-muted-foreground">
                            Plan: <span className="font-medium">{usage.planNumber}</span> {usage.planName && `- ${usage.planName}`}
                          </p>
                          <p className="text-sm text-muted-foreground">
                            Zarezerwowano: {new Date(usage.usedAt).toLocaleString("pl-PL")}
                          </p>
                          {usage.notes && (
                            <p className="text-sm text-muted-foreground mt-1">{usage.notes}</p>
                          )}
                        </>
                      ) : (
                        <>
                          <p className="font-medium">
                            Zużyto: {usage.quantityUsed} {material.unitOfMeasure}
                          </p>
                          <p className="text-sm text-muted-foreground">
                            {new Date(usage.usedAt).toLocaleString("pl-PL")}
                          </p>
                          {usage.notes && (
                            <p className="text-sm text-muted-foreground mt-1">{usage.notes}</p>
                          )}
                        </>
                      )}
                    </div>
                    {usage.planId && (
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => setLocation(`/production/plans/${usage.planId}`)}
                        data-testid={`button-view-plan-${usage.planId}`}
                      >
                        <ExternalLink className="h-4 w-4 mr-1" />
                        Plan
                      </Button>
                    )}
                    {usage.productionOrderId && (
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => setLocation(`/production/orders/${usage.productionOrderId}`)}
                        data-testid={`button-view-order-${usage.productionOrderId}`}
                      >
                        <ExternalLink className="h-4 w-4 mr-1" />
                        Zlecenie
                      </Button>
                    )}
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Link to material */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="h-5 w-5" />
              Link do materiału
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center gap-2">
              <code className="flex-1 p-2 bg-muted rounded text-sm">
                {window.location.origin}{location}
              </code>
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  navigator.clipboard.writeText(`${window.location.origin}${location}`);
                }}
                data-testid="button-copy-link"
              >
                Kopiuj
              </Button>
            </div>
          </CardContent>
        </Card>

        {/* Inventory Adjustment Dialog */}
        <Dialog open={isDialogOpen} onOpenChange={(open) => {
          setIsDialogOpen(open);
          if (!open) {
            resetDialogForm();
          }
        }}>
          <DialogContent className="sm:max-w-[500px]">
            <DialogHeader>
              <DialogTitle>Dodaj wpis inwentaryzacyjny</DialogTitle>
              <DialogDescription>
                Zarejestruj zmianę stanu magazynowego dla: {material.name}
              </DialogDescription>
            </DialogHeader>
            
            <div className="space-y-4 py-4">
              <div className="space-y-2">
                <Label htmlFor="operation-type">Typ operacji</Label>
                <Select value={operationType} onValueChange={setOperationType}>
                  <SelectTrigger id="operation-type" data-testid="select-operation-type">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="inventory_count">Inwentaryzacja</SelectItem>
                    <SelectItem value="material_usage">Zużycie materiału</SelectItem>
                    <SelectItem value="complaint">Reklamacja</SelectItem>
                    <SelectItem value="adjustment">Korekta</SelectItem>
                    <SelectItem value="purchase">Zakup</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="quantity-change">
                  Zmiana ilości ({material.unitOfMeasure})
                </Label>
                <Input
                  id="quantity-change"
                  type="number"
                  step="0.001"
                  placeholder="np. +10 lub -5"
                  value={quantityChange}
                  onChange={(e) => setQuantityChange(e.target.value)}
                  data-testid="input-quantity-change"
                />
                {quantityChange && !isNaN(parseFloat(quantityChange)) && (
                  <p className="text-sm text-muted-foreground">
                    Stan: {material.quantity || "0"} → {calculateNewQuantity()} {material.unitOfMeasure}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="document-number">Numer dokumentu (opcjonalnie)</Label>
                <Input
                  id="document-number"
                  placeholder="np. WZ/2025/001"
                  value={documentNumber}
                  onChange={(e) => setDocumentNumber(e.target.value)}
                  data-testid="input-document-number"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="notes">Notatki (opcjonalnie)</Label>
                <Textarea
                  id="notes"
                  placeholder="Dodatkowe informacje..."
                  value={notes}
                  onChange={(e) => setNotes(e.target.value)}
                  data-testid="textarea-notes"
                  rows={3}
                />
              </div>
            </div>

            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => {
                  setIsDialogOpen(false);
                  resetDialogForm();
                }}
                data-testid="button-cancel-inventory"
              >
                Anuluj
              </Button>
              <Button
                onClick={handleSubmitInventoryAdjustment}
                disabled={inventoryAdjustmentMutation.isPending}
                data-testid="button-submit-inventory"
              >
                {inventoryAdjustmentMutation.isPending ? "Zapisywanie..." : "Zapisz"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        {/* Location Editor Dialog */}
        <Dialog open={isLocationDialogOpen} onOpenChange={setIsLocationDialogOpen}>
          <DialogContent className="sm:max-w-[500px]">
            <DialogHeader>
              <DialogTitle>Lokalizacja magazynowa</DialogTitle>
              <DialogDescription>
                Ustaw lokalizację i nośnik dla: {material.name}
              </DialogDescription>
            </DialogHeader>

            <div className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="location-select" className="flex items-center gap-2">
                  <Building2 className="h-4 w-4" />
                  Hala
                </Label>
                <Select value={selectedLocationId || "none"} onValueChange={(val) => setSelectedLocationId(val === "none" ? "" : val)}>
                  <SelectTrigger id="location-select" data-testid="select-location">
                    <SelectValue placeholder="Wybierz halę" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Brak lokalizacji</SelectItem>
                    {locations
                      .filter(loc => loc.level === 1)
                      .map((loc) => (
                        <SelectItem key={loc.id} value={loc.id.toString()}>
                          {loc.name}
                        </SelectItem>
                      ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="carrier-select" className="flex items-center gap-2">
                  <Truck className="h-4 w-4" />
                  Nośnik
                </Label>
                <Select value={selectedCarrierId || "none"} onValueChange={(val) => setSelectedCarrierId(val === "none" ? "" : val)}>
                  <SelectTrigger id="carrier-select" data-testid="select-carrier">
                    <SelectValue placeholder="Wybierz nośnik" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Brak nośnika</SelectItem>
                    {carriers.map((carrier) => (
                      <SelectItem key={carrier.id} value={carrier.id.toString()}>
                        {carrier.name} ({carrier.type})
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>

            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => setIsLocationDialogOpen(false)}
                data-testid="button-cancel-location"
              >
                Anuluj
              </Button>
              <Button
                onClick={handleSubmitLocation}
                disabled={updateLocationMutation.isPending}
                data-testid="button-submit-location"
              >
                {updateLocationMutation.isPending ? "Zapisywanie..." : "Zapisz"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
          </div>
        </div>

        {/* Right Sidebar */}
        <div className="hidden md:flex w-80 border-l bg-card flex-shrink-0 overflow-y-auto flex-col">
          <div className="border-b px-4 py-4 flex items-center justify-between gap-2 flex-shrink-0">
            <h2 className="text-lg font-semibold flex-1">Powiązania</h2>
          </div>

          <div className="flex-1 overflow-y-auto">
            {/* Inventory Counts Section */}
            {isCountsLoading ? (
              <div className="p-4 space-y-3 border-b">
                <Skeleton className="h-5 w-1/2" />
                {[...Array(3)].map((_, i) => (
                  <Skeleton key={i} className="h-10 w-full" />
                ))}
              </div>
            ) : inventoryCounts && inventoryCounts.length > 0 ? (
              <div className="border-b">
                <button
                  onClick={() => setIsInventoryCountsSectionCollapsed(!isInventoryCountsSectionCollapsed)}
                  className="w-full px-4 py-3 flex items-center gap-2 bg-primary/5 hover-elevate active-elevate-2 transition-colors"
                  data-testid="button-toggle-inventory-counts-section"
                >
                  <FileCheck className="h-5 w-5 text-primary" />
                  <h3 className="font-semibold text-base flex-1 text-left text-primary">Spisy inwentaryzacyjne</h3>
                  <Badge variant="secondary" className="text-xs font-bold">
                    {inventoryCounts.length}
                  </Badge>
                  <ChevronDown 
                    className={`h-4 w-4 transition-transform ${isInventoryCountsSectionCollapsed ? '-rotate-90' : ''}`} 
                  />
                </button>
                {!isInventoryCountsSectionCollapsed && (
                  <div className="p-2 space-y-2">
                    {inventoryCounts.map((count) => (
                      <Link key={count.countId} href={`/warehouse/inventory-counts/${count.countId}`}>
                        <div
                          className="flex flex-col gap-2 px-3 py-3 rounded-md text-sm transition-colors hover-elevate active-elevate-2 border border-border/50"
                          data-testid={`inventory-count-item-${count.countId}`}
                        >
                          <div className="flex items-start justify-between gap-2">
                            <div className="flex-1 min-w-0">
                              <div className="font-medium truncate">{count.countName}</div>
                              <div className="text-xs text-muted-foreground truncate mt-0.5">
                                {new Date(count.countCreatedAt).toLocaleDateString('pl-PL')}
                              </div>
                            </div>
                            {getStatusBadge(count.countStatus)}
                          </div>
                          <div className="flex flex-col gap-1 text-xs">
                            {count.expectedQuantity !== null && count.expectedQuantity !== undefined && (
                              <div>
                                <span className="text-muted-foreground">Oczekiwano: </span>
                                <span className="font-medium">{count.expectedQuantity}</span>
                              </div>
                            )}
                            {count.countedQuantity !== null && count.countedQuantity !== undefined && (
                              <div>
                                <span className="text-muted-foreground">Policzono: </span>
                                <span className="font-medium">{count.countedQuantity}</span>
                              </div>
                            )}
                            {count.difference !== null && count.difference !== undefined && count.difference !== 0 && (
                              <div>
                                <span className="text-muted-foreground">Różnica: </span>
                                <span className={`font-medium ${(count.difference ?? 0) > 0 ? 'text-green-600' : 'text-red-600'}`}>
                                  {(count.difference ?? 0) > 0 ? '+' : ''}{count.difference}
                                </span>
                              </div>
                            )}
                          </div>
                        </div>
                      </Link>
                    ))}
                  </div>
                )}
              </div>
            ) : null}
          </div>
        </div>
      </div>
    </WarehouseLayout>
  );
}
