import { useQuery, useMutation } from "@tanstack/react-query";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { 
  FileText, ArrowLeft, Trash2, CheckCircle, Ban, Plus, 
  Package, Hash, MapPin, Calendar, User, Edit2, Search, Loader2, Printer 
} from "lucide-react";
import { Link, useParams, useLocation } from "wouter";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Separator } from "@/components/ui/separator";
import { useState, useMemo } from "react";
import { safeGoBack } from "@/lib/navigation";

interface DocumentLine {
  id: number;
  documentId: number;
  lineNumber: number;
  sourceType: string;
  sourceId: number | null;
  catalogProductId: number | null;
  sku: string | null;
  productName: string;
  quantityRequested: number;
  quantityPicked: number;
  unit: string;
  warehouseLocationId: number | null;
  warehouseLocationName: string | null;
  serialNumbers: string[];
  planLineId: number | null;
  isPicked: boolean;
  pickedAt: string | null;
  pickedBy: number | null;
  notes: string | null;
  createdAt: string;
  updatedAt: string;
}

interface WarehouseDocumentWithLines {
  id: number;
  docNumber: string;
  docType: string;
  status: string;
  planId: number | null;
  planName: string | null;
  targetDepartment: string | null;
  targetLocationId: number | null;
  targetLocationName: string | null;
  issuedAt: string | null;
  issuedBy: number | null;
  issuedByName: string | null;
  confirmedAt: string | null;
  confirmedByName: string | null;
  completedAt: string | null;
  completedByName: string | null;
  remarks: string | null;
  totalLines: number;
  totalQuantity: number;
  createdAt: string;
  updatedAt: string;
  lines: DocumentLine[];
}

const DOCUMENT_TYPE_LABELS: Record<string, string> = {
  'WZ-SPAK': 'Produkty spakowane',
  'WZ-TAP': 'Tapicernia',
  'WZ-FORM': 'Formatki',
  'WZ-OKUC': 'Okucia',
  'WZ-OPAK': 'Opakowania',
};

const STATUS_LABELS: Record<string, string> = {
  'draft': 'Szkic',
  'confirmed': 'Potwierdzony',
  'printed': 'Wydrukowany',
  'completed': 'Zrealizowany',
  'cancelled': 'Anulowany',
};

const STATUS_VARIANTS: Record<string, "secondary" | "default" | "outline" | "destructive"> = {
  'draft': 'secondary',
  'confirmed': 'default',
  'printed': 'default',
  'completed': 'outline',
  'cancelled': 'destructive',
};

const SOURCE_TYPE_LABELS: Record<string, string> = {
  'packed_product': 'Produkty spakowane',
  'formatka': 'Formatki',
  'okucia': 'Okucia',
  'opakowania': 'Opakowania',
  'tapicernia': 'Tapicernia',
  'custom': 'Inne',
};

const DOC_TYPE_TO_SOURCE: Record<string, string> = {
  'WZ-SPAK': 'packed_product',
  'WZ-TAP': 'tapicernia',
  'WZ-FORM': 'formatka',
  'WZ-OKUC': 'okucia',
  'WZ-OPAK': 'opakowania',
};

interface WarehouseItem {
  id: number;
  name: string;
  sku?: string;
  quantity: number;
  reservedQuantity: number;
  unit: string;
  catalogProductId?: number;
}

export default function WarehouseDocumentDetailPage() {
  const { id } = useParams<{ id: string }>();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [cancelDialogOpen, setCancelDialogOpen] = useState(false);
  const [confirmDialogOpen, setConfirmDialogOpen] = useState(false);
  const [completeDialogOpen, setCompleteDialogOpen] = useState(false);
  const [deleteLineDialogOpen, setDeleteLineDialogOpen] = useState(false);
  const [selectedLineId, setSelectedLineId] = useState<number | null>(null);
  
  const [addLineDialogOpen, setAddLineDialogOpen] = useState(false);
  const [productSearchOpen, setProductSearchOpen] = useState(false);
  const [productSearch, setProductSearch] = useState("");
  const [selectedProduct, setSelectedProduct] = useState<WarehouseItem | null>(null);
  const [lineQuantity, setLineQuantity] = useState<number>(1);

  const { data: document, isLoading, error } = useQuery<WarehouseDocumentWithLines>({
    queryKey: ['/api/warehouse/documents', id],
    queryFn: async () => {
      const response = await fetch(`/api/warehouse/documents/${id}`, { credentials: 'include' });
      if (!response.ok) throw new Error('Failed to fetch document');
      return response.json();
    },
    enabled: !!id,
  });

  const deleteMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest('DELETE', `/api/warehouse/documents/${id}`);
    },
    onSuccess: () => {
      toast({ title: "Sukces", description: "Dokument został usunięty" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents'] });
      setLocation('/warehouse/documents');
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się usunąć dokumentu", variant: "destructive" });
    },
  });

  const cancelMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest('POST', `/api/warehouse/documents/${id}/cancel`);
    },
    onSuccess: () => {
      toast({ title: "Sukces", description: "Dokument został anulowany" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents', id] });
      setCancelDialogOpen(false);
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się anulować dokumentu", variant: "destructive" });
    },
  });

  const confirmMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest('POST', `/api/warehouse/documents/${id}/confirm`);
    },
    onSuccess: () => {
      toast({ title: "Sukces", description: "Dokument został potwierdzony" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents', id] });
      setConfirmDialogOpen(false);
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się potwierdzić dokumentu", variant: "destructive" });
    },
  });

  const completeMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest('POST', `/api/warehouse/documents/${id}/complete`);
    },
    onSuccess: () => {
      toast({ title: "Sukces", description: "Dokument został zrealizowany" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents', id] });
      setCompleteDialogOpen(false);
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się zrealizować dokumentu", variant: "destructive" });
    },
  });

  const deleteLineMutation = useMutation({
    mutationFn: async (lineId: number) => {
      return await apiRequest('DELETE', `/api/warehouse/documents/${id}/lines/${lineId}`);
    },
    onSuccess: () => {
      toast({ title: "Sukces", description: "Pozycja została usunięta" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents', id] });
      setDeleteLineDialogOpen(false);
      setSelectedLineId(null);
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się usunąć pozycji", variant: "destructive" });
    },
  });

  const sourceType = document ? DOC_TYPE_TO_SOURCE[document.docType] : null;
  
  const { data: warehouseItems = [], isLoading: isLoadingItems } = useQuery<WarehouseItem[]>({
    queryKey: ['/api/warehouse/items-for-document', sourceType, productSearch],
    queryFn: async () => {
      if (!sourceType) return [];
      
      let endpoint = '';
      let categoryParam = '';
      
      switch (sourceType) {
        case 'packed_product':
          categoryParam = 'produkty-spakowane';
          break;
        case 'formatka':
          categoryParam = 'formatki';
          break;
        case 'opakowania':
          categoryParam = 'opakowania';
          break;
        case 'okucia':
          categoryParam = 'okucia';
          break;
        case 'tapicernia':
          categoryParam = 'tkaniny';
          break;
      }
      
      endpoint = `/api/warehouse/materials/search?category=${categoryParam}&search=${encodeURIComponent(productSearch)}&limit=500&stockFilter=inStock`;
      
      const response = await fetch(endpoint, { credentials: 'include' });
      if (!response.ok) throw new Error('Failed to fetch items');
      const data = await response.json();
      
      const items = Array.isArray(data) ? data : (data.materials || data.items || []);
      
      return items.map((item: any) => ({
        id: item.id,
        name: item.name || item.productName,
        sku: item.sku || item.catalogProductSku || item.code,
        quantity: Number(item.quantity) || 0,
        reservedQuantity: Number(item.quantityReserved || item.reserved_quantity || item.reservedQuantity) || 0,
        unit: item.unit || 'szt',
        catalogProductId: item.catalogProductId || item.catalog_product_id,
      }));
    },
    enabled: !!sourceType && addLineDialogOpen,
  });

  const addLineMutation = useMutation({
    mutationFn: async (data: { 
      sourceType: string;
      sourceId: number;
      productName: string;
      sku?: string;
      quantityRequested: number;
      unit: string;
      catalogProductId?: number;
    }) => {
      return await apiRequest('POST', `/api/warehouse/documents/${id}/lines`, data);
    },
    onSuccess: () => {
      toast({ title: "Sukces", description: "Pozycja została dodana" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents', id] });
      setAddLineDialogOpen(false);
      setSelectedProduct(null);
      setLineQuantity(1);
      setProductSearch("");
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się dodać pozycji", variant: "destructive" });
    },
  });

  const handleAddLine = () => {
    if (!selectedProduct || !sourceType) return;
    
    addLineMutation.mutate({
      sourceType,
      sourceId: selectedProduct.id,
      productName: selectedProduct.name,
      sku: selectedProduct.sku,
      quantityRequested: lineQuantity,
      unit: selectedProduct.unit,
      catalogProductId: selectedProduct.catalogProductId,
    });
  };

  const filteredItems = useMemo(() => {
    return warehouseItems.filter(item => {
      const available = item.quantity - item.reservedQuantity;
      return available > 0;
    });
  }, [warehouseItems]);

  const formatDate = (dateStr: string | null) => {
    if (!dateStr) return '-';
    return new Date(dateStr).toLocaleDateString('pl-PL', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const getStatusBadge = (status: string) => {
    return (
      <Badge variant={STATUS_VARIANTS[status] || 'secondary'} data-testid={`badge-status-${status}`}>
        {STATUS_LABELS[status] || status}
      </Badge>
    );
  };

  if (isLoading) {
    return (
      <WarehouseLayout category="all" hideSubmenu={true}>
        <div className="p-6">Ładowanie...</div>
      </WarehouseLayout>
    );
  }

  if (error || !document) {
    return (
      <WarehouseLayout category="all" hideSubmenu={true}>
        <div className="p-6">
          <div className="text-center py-12">
            <FileText className="w-12 h-12 mx-auto mb-4 text-muted-foreground" />
            <h2 className="text-lg font-semibold mb-2">Nie znaleziono dokumentu</h2>
            <Link href="/warehouse/documents">
              <Button variant="outline">
                <ArrowLeft className="w-4 h-4 mr-2" />
                Wróć do listy
              </Button>
            </Link>
          </div>
        </div>
      </WarehouseLayout>
    );
  }

  const isDraft = document.status === 'draft';
  const isActive = ['confirmed', 'printed'].includes(document.status);

  return (
    <WarehouseLayout category="all" hideSubmenu={true}>
      <div className="p-6 space-y-6">
        <div className="flex items-center justify-between gap-4">
          <div className="flex items-center gap-3">
            <Button variant="ghost" size="icon" onClick={() => safeGoBack('/warehouse/documents', setLocation)} data-testid="button-back">
              <ArrowLeft className="w-5 h-5" />
            </Button>
            <div>
              <div className="flex items-center gap-3">
                <h1 className="text-2xl font-bold font-mono" data-testid="text-doc-number">{document.docNumber}</h1>
                {getStatusBadge(document.status)}
              </div>
              <p className="text-sm text-muted-foreground">
                {DOCUMENT_TYPE_LABELS[document.docType] || document.docType}
              </p>
            </div>
          </div>
          <div className="flex items-center gap-2">
            <Button
              variant="outline"
              onClick={() => window.open(`/api/warehouse/documents/${id}/pdf`, '_blank')}
              data-testid="button-pdf"
            >
              <Printer className="w-4 h-4 mr-2" />
              Drukuj PDF
            </Button>
            {isDraft && (
              <>
                <Button onClick={() => setConfirmDialogOpen(true)} data-testid="button-confirm">
                  <CheckCircle className="w-4 h-4 mr-2" />
                  Potwierdź
                </Button>
                <Button variant="destructive" onClick={() => setDeleteDialogOpen(true)} data-testid="button-delete">
                  <Trash2 className="w-4 h-4 mr-2" />
                  Usuń
                </Button>
              </>
            )}
            {isActive && (
              <>
                <Button onClick={() => setCompleteDialogOpen(true)} data-testid="button-complete">
                  <CheckCircle className="w-4 h-4 mr-2" />
                  Zrealizuj
                </Button>
                <Button variant="destructive" onClick={() => setCancelDialogOpen(true)} data-testid="button-cancel">
                  <Ban className="w-4 h-4 mr-2" />
                  Anuluj
                </Button>
              </>
            )}
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Informacje podstawowe</CardDescription>
            </CardHeader>
            <CardContent className="space-y-2">
              <div className="flex items-center gap-2 text-sm">
                <Hash className="w-4 h-4 text-muted-foreground" />
                <span className="text-muted-foreground">Typ:</span>
                <span className="font-medium">{document.docType}</span>
              </div>
              {document.planName && (
                <div className="flex items-center gap-2 text-sm">
                  <Package className="w-4 h-4 text-muted-foreground" />
                  <span className="text-muted-foreground">Plan:</span>
                  <Link href={`/production/plans/${document.planId}`}>
                    <span className="font-medium hover:underline cursor-pointer">{document.planName}</span>
                  </Link>
                </div>
              )}
              {(document.targetDepartment || document.targetLocationName) && (
                <div className="flex items-center gap-2 text-sm">
                  <MapPin className="w-4 h-4 text-muted-foreground" />
                  <span className="text-muted-foreground">Cel:</span>
                  <span className="font-medium">{document.targetDepartment || document.targetLocationName}</span>
                </div>
              )}
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Statystyki</CardDescription>
            </CardHeader>
            <CardContent className="space-y-2">
              <div className="flex items-center justify-between">
                <span className="text-sm text-muted-foreground">Liczba pozycji:</span>
                <span className="text-lg font-bold" data-testid="text-total-lines">{document.totalLines}</span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm text-muted-foreground">Suma ilości:</span>
                <span className="text-lg font-bold" data-testid="text-total-quantity">{document.totalQuantity}</span>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Historia</CardDescription>
            </CardHeader>
            <CardContent className="space-y-2">
              <div className="flex items-center gap-2 text-sm">
                <Calendar className="w-4 h-4 text-muted-foreground" />
                <span className="text-muted-foreground">Utworzono:</span>
                <span>{formatDate(document.createdAt)}</span>
              </div>
              {document.issuedByName && (
                <div className="flex items-center gap-2 text-sm">
                  <User className="w-4 h-4 text-muted-foreground" />
                  <span className="text-muted-foreground">Wystawił:</span>
                  <span>{document.issuedByName}</span>
                </div>
              )}
              {document.confirmedAt && (
                <div className="flex items-center gap-2 text-sm">
                  <CheckCircle className="w-4 h-4 text-green-600" />
                  <span className="text-muted-foreground">Potwierdzone:</span>
                  <span>{formatDate(document.confirmedAt)}</span>
                </div>
              )}
              {document.completedAt && (
                <div className="flex items-center gap-2 text-sm">
                  <CheckCircle className="w-4 h-4 text-blue-600" />
                  <span className="text-muted-foreground">Zrealizowane:</span>
                  <span>{formatDate(document.completedAt)}</span>
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        {document.remarks && (
          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Uwagi</CardDescription>
            </CardHeader>
            <CardContent>
              <p className="text-sm whitespace-pre-wrap">{document.remarks}</p>
            </CardContent>
          </Card>
        )}

        <Separator />

        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <h2 className="text-lg font-semibold">Pozycje dokumentu ({document.lines.length})</h2>
            {isDraft && (
              <Button 
                variant="outline" 
                size="sm" 
                onClick={() => setAddLineDialogOpen(true)}
                data-testid="button-add-line"
              >
                <Plus className="w-4 h-4 mr-2" />
                Dodaj pozycję
              </Button>
            )}
          </div>

          {document.lines.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground border rounded-md">
              <Package className="w-8 h-8 mx-auto mb-2 opacity-50" />
              <p>Brak pozycji w dokumencie</p>
            </div>
          ) : (
            <div className="border rounded-md">
              <Table>
                <TableHeader>
                  <TableRow className="bg-muted/50">
                    <TableHead className="w-[5%] text-center">#</TableHead>
                    <TableHead className="w-[30%]">Produkt</TableHead>
                    <TableHead className="w-[12%]">SKU</TableHead>
                    <TableHead className="w-[10%]">Źródło</TableHead>
                    <TableHead className="w-[8%] text-center">Ilość</TableHead>
                    <TableHead className="w-[8%] text-center">Pobrano</TableHead>
                    <TableHead className="w-[6%]">J.m.</TableHead>
                    <TableHead className="w-[12%]">Lokalizacja</TableHead>
                    {isDraft && <TableHead className="w-[9%] text-right">Akcje</TableHead>}
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {document.lines.map((line) => (
                    <TableRow key={line.id} data-testid={`row-line-${line.id}`}>
                      <TableCell className="text-center font-mono text-muted-foreground">
                        {line.lineNumber}
                      </TableCell>
                      <TableCell>
                        <span className="font-medium">{line.productName}</span>
                      </TableCell>
                      <TableCell>
                        <span className="font-mono text-sm">{line.sku || '-'}</span>
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline" className="text-xs">
                          {SOURCE_TYPE_LABELS[line.sourceType] || line.sourceType}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-center font-mono font-bold">
                        {line.quantityRequested}
                      </TableCell>
                      <TableCell className="text-center font-mono">
                        <span className={line.quantityPicked >= line.quantityRequested ? 'text-green-600' : 'text-muted-foreground'}>
                          {line.quantityPicked}
                        </span>
                      </TableCell>
                      <TableCell>{line.unit}</TableCell>
                      <TableCell>
                        <span className="text-sm">{line.warehouseLocationName || '-'}</span>
                      </TableCell>
                      {isDraft && (
                        <TableCell className="text-right">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => {
                              setSelectedLineId(line.id);
                              setDeleteLineDialogOpen(true);
                            }}
                            data-testid={`button-delete-line-${line.id}`}
                          >
                            <Trash2 className="w-4 h-4 text-destructive" />
                          </Button>
                        </TableCell>
                      )}
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </div>

        <AlertDialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Usuń dokument</AlertDialogTitle>
              <AlertDialogDescription>
                Czy na pewno chcesz usunąć dokument {document.docNumber}? Ta operacja jest nieodwracalna.
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Anuluj</AlertDialogCancel>
              <AlertDialogAction
                onClick={() => deleteMutation.mutate()}
                className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              >
                Usuń
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        <AlertDialog open={cancelDialogOpen} onOpenChange={setCancelDialogOpen}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Anuluj dokument</AlertDialogTitle>
              <AlertDialogDescription>
                Czy na pewno chcesz anulować dokument {document.docNumber}?
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Nie</AlertDialogCancel>
              <AlertDialogAction
                onClick={() => cancelMutation.mutate()}
                className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              >
                Anuluj dokument
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        <AlertDialog open={confirmDialogOpen} onOpenChange={setConfirmDialogOpen}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Potwierdź dokument</AlertDialogTitle>
              <AlertDialogDescription>
                Czy na pewno chcesz potwierdzić dokument {document.docNumber}? Po potwierdzeniu dokument nie będzie mógł być edytowany.
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Anuluj</AlertDialogCancel>
              <AlertDialogAction onClick={() => confirmMutation.mutate()}>
                Potwierdź
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        <AlertDialog open={completeDialogOpen} onOpenChange={setCompleteDialogOpen}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Zrealizuj dokument</AlertDialogTitle>
              <AlertDialogDescription>
                Czy na pewno chcesz oznaczyć dokument {document.docNumber} jako zrealizowany? Ta operacja potwierdzi wydanie materiałów z magazynu.
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Anuluj</AlertDialogCancel>
              <AlertDialogAction onClick={() => completeMutation.mutate()}>
                Zrealizuj
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        <AlertDialog open={deleteLineDialogOpen} onOpenChange={setDeleteLineDialogOpen}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Usuń pozycję</AlertDialogTitle>
              <AlertDialogDescription>
                Czy na pewno chcesz usunąć tę pozycję z dokumentu?
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Anuluj</AlertDialogCancel>
              <AlertDialogAction
                onClick={() => selectedLineId && deleteLineMutation.mutate(selectedLineId)}
                className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              >
                Usuń
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        <Dialog open={addLineDialogOpen} onOpenChange={(open) => {
          setAddLineDialogOpen(open);
          if (!open) {
            setSelectedProduct(null);
            setLineQuantity(1);
            setProductSearch("");
          }
        }}>
          <DialogContent className="sm:max-w-[500px]">
            <DialogHeader>
              <DialogTitle>Dodaj pozycję</DialogTitle>
              <DialogDescription>
                Wybierz produkt z magazynu i podaj ilość do wydania
              </DialogDescription>
            </DialogHeader>
            
            <div className="space-y-4 py-4">
              <div className="space-y-2">
                <Label>Produkt</Label>
                <Popover open={productSearchOpen} onOpenChange={setProductSearchOpen}>
                  <PopoverTrigger asChild>
                    <Button
                      variant="outline"
                      role="combobox"
                      aria-expanded={productSearchOpen}
                      className="w-full justify-between"
                      data-testid="combobox-product"
                    >
                      {selectedProduct ? (
                        <span className="truncate">{selectedProduct.name}</span>
                      ) : (
                        <span className="text-muted-foreground">Wyszukaj produkt...</span>
                      )}
                      <Search className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-[460px] p-0" align="start">
                    <Command shouldFilter={false}>
                      <CommandInput 
                        placeholder="Szukaj produktu..." 
                        value={productSearch}
                        onValueChange={setProductSearch}
                        data-testid="input-product-search"
                      />
                      <CommandList>
                        {isLoadingItems ? (
                          <div className="flex items-center justify-center py-6">
                            <Loader2 className="h-4 w-4 animate-spin" />
                          </div>
                        ) : filteredItems.length === 0 ? (
                          <CommandEmpty>
                            Nie znaleziono produktów z dostępnym stanem
                          </CommandEmpty>
                        ) : (
                          <CommandGroup>
                            {filteredItems.map((item) => {
                              const available = item.quantity - item.reservedQuantity;
                              return (
                                <CommandItem
                                  key={item.id}
                                  value={item.id.toString()}
                                  onSelect={() => {
                                    setSelectedProduct(item);
                                    setProductSearchOpen(false);
                                    setLineQuantity(Math.min(1, available));
                                  }}
                                  className="flex items-center justify-between"
                                  data-testid={`item-product-${item.id}`}
                                >
                                  <div className="flex flex-col flex-1 min-w-0">
                                    <span className="font-medium truncate">{item.name}</span>
                                    {item.sku && (
                                      <span className="text-xs text-muted-foreground font-mono">
                                        {item.sku}
                                      </span>
                                    )}
                                  </div>
                                  <Badge variant="outline" className="ml-2 shrink-0">
                                    {available} {item.unit}
                                  </Badge>
                                </CommandItem>
                              );
                            })}
                          </CommandGroup>
                        )}
                      </CommandList>
                    </Command>
                  </PopoverContent>
                </Popover>
              </div>

              {selectedProduct && (
                <div className="space-y-2">
                  <Label htmlFor="quantity">Ilość do wydania</Label>
                  <div className="flex items-center gap-2">
                    <Input
                      id="quantity"
                      type="number"
                      min={1}
                      max={selectedProduct.quantity - selectedProduct.reservedQuantity}
                      value={lineQuantity}
                      onChange={(e) => setLineQuantity(Math.max(1, parseInt(e.target.value) || 1))}
                      className="w-32"
                      data-testid="input-quantity"
                    />
                    <span className="text-muted-foreground">{selectedProduct.unit}</span>
                    <span className="text-xs text-muted-foreground ml-auto">
                      Dostępne: {selectedProduct.quantity - selectedProduct.reservedQuantity} {selectedProduct.unit}
                    </span>
                  </div>
                </div>
              )}
            </div>

            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => setAddLineDialogOpen(false)}
                data-testid="button-cancel-add-line"
              >
                Anuluj
              </Button>
              <Button
                onClick={handleAddLine}
                disabled={!selectedProduct || lineQuantity < 1 || addLineMutation.isPending}
                data-testid="button-confirm-add-line"
              >
                {addLineMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
                Dodaj
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>
    </WarehouseLayout>
  );
}
