import { useState, useEffect } from "react";
import { useParams, useLocation } from "wouter";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { ArrowLeft, Play, CheckCircle2, XCircle, AlertCircle, Package, RefreshCw, Target } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { safeGoBack } from "@/lib/navigation";

interface ProductCreatorDictionary {
  id: number;
  name: string;
  code: string;
  color?: string;
  readable_name?: string;
  dictionary_type: string;
  is_active: boolean;
}

interface ValidationResult {
  variants: Array<{
    matrixId: number;
    matrixName: string;
    variantName: string;
    color: string | null;
    colorOptions?: string[];
    canGenerate: boolean;
    components: Array<{
      name: string;
      componentType: string;
      length?: number;
      width?: number;
      color?: string | null;
      colorOptions?: string[];
      quantity: number;
      available: boolean;
      availableCount: number;
      productId?: number | null;
      productSource?: 'override' | 'auto';
    }>;
  }>;
  summary: {
    totalVariants: number;
    readyCount: number;
    blockedCount: number;
  };
}

// Helper function to get contrasting text color (black or white) based on background
function getContrastColor(hexColor: string): string {
  // Remove # if present
  const hex = hexColor.replace('#', '');
  
  // Convert to RGB
  const r = parseInt(hex.substr(0, 2), 16);
  const g = parseInt(hex.substr(2, 2), 16);
  const b = parseInt(hex.substr(4, 2), 16);
  
  // Calculate relative luminance
  const luminance = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
  
  // Return black for light backgrounds, white for dark backgrounds
  return luminance > 0.5 ? '#000000' : '#ffffff';
}

export default function SetMatrixGeneratePage() {
  const params = useParams();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const matrixId = params.id ? parseInt(params.id) : null;
  
  const [step, setStep] = useState<1 | 2>(1);
  const [isGenerating, setIsGenerating] = useState(false);
  const [progress, setProgress] = useState(0);
  const [logs, setLogs] = useState<Array<{ message: string; type: 'info' | 'success' | 'error' | 'warning' }>>([]);
  const [generationComplete, setGenerationComplete] = useState(false);

  // Fetch validation data
  const { data: validationData, isLoading } = useQuery<ValidationResult>({
    queryKey: [`/api/set-matrices/validate-generation`, matrixId],
    queryFn: async () => {
      const res = await apiRequest("POST", "/api/set-matrices/validate-generation", {
        matrixIds: matrixId ? [matrixId] : undefined
      });
      return res.json() as Promise<ValidationResult>;
    },
    enabled: !!matrixId && step === 1,
  });

  // Fetch color dictionary for variant badges
  const { data: colors } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });

  const handleRefreshValidation = () => {
    if (!matrixId) return;
    queryClient.invalidateQueries({
      queryKey: [`/api/set-matrices/validate-generation`, matrixId]
    });
    toast({
      title: "Odświeżanie walidacji",
      description: "Ponownie sprawdzam dostępność komponentów...",
    });
  };

  const handleStartGeneration = async () => {
    if (!matrixId) return;
    
    setStep(2);
    setIsGenerating(true);
    setProgress(25);
    setLogs([{ message: "Rozpoczynanie generowania zestawów...", type: 'info' }]);
    setGenerationComplete(false);

    try {
      setProgress(50);
      setLogs(prev => [...prev, { message: `Generowanie zestawów z matrycy ID: ${matrixId}`, type: 'info' }]);
      
      // Start generation
      const res = await apiRequest("POST", `/api/set-matrices/${matrixId}/generate`, {});
      const result = await res.json() as { success: boolean; setsGenerated: number };
      
      setProgress(100);
      setLogs(prev => [
        ...prev,
        { message: `Pomyślnie wygenerowano ${result.setsGenerated} zestawów`, type: 'success' }
      ]);
      setIsGenerating(false);
      setGenerationComplete(true);
      
      toast({
        title: "Generowanie zakończone",
        description: `Wygenerowano ${result.setsGenerated} zestawów`,
      });
      
    } catch (error: any) {
      setIsGenerating(false);
      setLogs(prev => [
        ...prev,
        { message: `Błąd: ${error.message || 'Nie udało się wygenerować zestawów'}`, type: 'error' }
      ]);
      toast({
        title: "Błąd generowania",
        description: error.message || "Nie udało się wygenerować zestawów",
        variant: "destructive",
      });
    }
  };

  const canGenerate = validationData?.summary?.readyCount && validationData.summary.readyCount > 0;

  return (
    <div className="p-6 space-y-6 max-w-6xl mx-auto">
      {/* Header */}
      <div className="flex items-center gap-4">
        <Button
          variant="ghost"
          size="icon"
          onClick={() => safeGoBack(`/set-matrices/settings/${matrixId}`, setLocation)}
          data-testid="button-back"
        >
          <ArrowLeft className="h-5 w-5" />
        </Button>
        <div>
          <h1 className="text-2xl font-bold">Generowanie zestawów produktów</h1>
          <p className="text-sm text-muted-foreground">
            {step === 1 ? "Krok 1: Walidacja komponentów" : "Krok 2: Generowanie zestawów"}
          </p>
        </div>
      </div>

      {/* Step 1: Validation */}
      {step === 1 && (
        <div className="space-y-6">
          {isLoading ? (
            <Card>
              <CardContent className="p-12 text-center">
                <p className="text-muted-foreground">Sprawdzanie dostępności komponentów...</p>
              </CardContent>
            </Card>
          ) : (
            <>
              {/* Summary */}
              <Card>
                <CardHeader className="flex flex-row items-center justify-between gap-4 space-y-0 pb-2">
                  <CardTitle className="flex items-center gap-2">
                    <Package className="h-5 w-5" />
                    Podsumowanie walidacji
                  </CardTitle>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={handleRefreshValidation}
                    disabled={isLoading}
                    data-testid="button-refresh-validation"
                  >
                    <RefreshCw className={`h-4 w-4 mr-2 ${isLoading ? 'animate-spin' : ''}`} />
                    Odśwież
                  </Button>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-3 gap-4">
                    <div className="text-center">
                      <div className="text-3xl font-bold">{validationData?.summary?.totalVariants || 0}</div>
                      <div className="text-sm text-muted-foreground">Wariantów</div>
                    </div>
                    <div className="text-center">
                      <div className="text-3xl font-bold text-green-600">{validationData?.summary?.readyCount || 0}</div>
                      <div className="text-sm text-muted-foreground">Gotowych</div>
                    </div>
                    <div className="text-center">
                      <div className="text-3xl font-bold text-red-600">{validationData?.summary?.blockedCount || 0}</div>
                      <div className="text-sm text-muted-foreground">Zablokowanych</div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Variants Table */}
              {validationData?.variants && validationData.variants.length > 0 && (
                <Card>
                  <CardHeader>
                    <CardTitle>Warianty do wygenerowania</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-3">
                      {validationData.variants.map((variant, idx) => {
                        // Find color from dictionary for variant badge
                        const colorDict = colors?.find(c => c.name === variant.color || c.code === variant.color);
                        const colorHex = colorDict?.color;
                        
                        return (
                          <div
                            key={idx}
                            className={`p-4 rounded-none border ${
                              variant.canGenerate
                                ? 'bg-green-100 dark:bg-green-950/40 border-green-300 dark:border-green-800'
                                : 'bg-muted/50 border-muted-foreground/20'
                            }`}
                            data-testid={`variant-row-${idx}`}
                          >
                            <div className="flex items-start gap-4">
                              {/* Variant name with color badge */}
                              <div className="min-w-[200px]">
                                <div 
                                  className="inline-block px-3 py-1 text-sm font-semibold rounded-none"
                                  style={colorHex ? {
                                    backgroundColor: colorHex,
                                    color: getContrastColor(colorHex)
                                  } : undefined}
                                  data-testid={`variant-badge-${idx}`}
                                >
                                  {variant.variantName}
                                </div>
                                {!variant.canGenerate && (
                                  <Badge variant="destructive" className="text-xs mt-2 rounded-none">
                                    Brakuje komponentów
                                  </Badge>
                                )}
                              </div>
                              
                              {/* Component rectangles */}
                              <div className="flex-1 flex flex-wrap gap-2">
                                {variant.components.map((comp, compIdx) => (
                                  <div
                                    key={compIdx}
                                    className={`px-3 py-2 rounded-none border text-sm ${
                                      comp.available
                                        ? 'bg-emerald-200 dark:bg-emerald-900/50 border-emerald-300 dark:border-emerald-800 text-foreground'
                                        : 'bg-muted/30 border-muted-foreground/30 text-muted-foreground'
                                    }`}
                                    data-testid={`component-${idx}-${compIdx}`}
                                    title={comp.available ? `Dostępne: ${comp.availableCount}` : 'Brak w systemie'}
                                  >
                                    <div className="font-medium">{comp.name}</div>
                                    <div className="text-xs opacity-70">
                                      {comp.quantity > 1 && `${comp.quantity}x`}
                                      {comp.colorOptions && comp.colorOptions.length > 0 && (
                                        <div className="text-xs mt-0.5 opacity-60">
                                          {comp.colorOptions.join(', ')}
                                        </div>
                                      )}
                                      {comp.productId && (
                                        <div className="text-xs mt-1 font-mono flex items-center gap-1">
                                          <a 
                                            href={`/catalog-products/${comp.productId}`} 
                                            className="text-primary hover:underline"
                                            onClick={(e) => {
                                              e.preventDefault();
                                              setLocation(`/catalog-products/${comp.productId}`);
                                            }}
                                            data-testid={`link-product-${comp.productId}`}
                                          >
                                            #{comp.productId}
                                          </a>
                                          {comp.productSource === 'override' && (
                                            <Badge variant="secondary" className="text-xs px-1 py-0 h-4">
                                              <Target className="h-3 w-3 mr-0.5" />
                                              Nadpisanie
                                            </Badge>
                                          )}
                                        </div>
                                      )}
                                    </div>
                                  </div>
                                ))}
                              </div>
                            </div>
                          </div>
                        );
                      })}
                    </div>
                  </CardContent>
                </Card>
              )}

              {/* Action buttons */}
              <div className="flex justify-end gap-3">
                <Button
                  variant="outline"
                  onClick={() => safeGoBack(`/set-matrices/settings/${matrixId}`, setLocation)}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button
                  onClick={handleStartGeneration}
                  disabled={!canGenerate}
                  data-testid="button-start-generation"
                >
                  <Play className="h-4 w-4 mr-2" />
                  Generuj zestawy ({validationData?.summary?.readyCount || 0})
                </Button>
              </div>
            </>
          )}
        </div>
      )}

      {/* Step 2: Generation Progress */}
      {step === 2 && (
        <div className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>
                {generationComplete ? "Generowanie zakończone" : "Generowanie w toku..."}
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <div className="flex justify-between text-sm">
                  <span>Postęp:</span>
                  <span>{Math.round(progress)}%</span>
                </div>
                <Progress value={progress} className="h-2" />
              </div>

              {/* Logs */}
              <div className="border rounded-lg p-4 bg-muted/30 max-h-96 overflow-y-auto space-y-1">
                {logs.length === 0 ? (
                  <p className="text-sm text-muted-foreground">Oczekiwanie na logi...</p>
                ) : (
                  logs.map((log, idx) => (
                    <div
                      key={idx}
                      className={`text-sm font-mono ${
                        log.type === 'error' ? 'text-red-600' :
                        log.type === 'success' ? 'text-green-600' :
                        log.type === 'warning' ? 'text-yellow-600' :
                        'text-foreground'
                      }`}
                    >
                      {log.message}
                    </div>
                  ))
                )}
              </div>

              {generationComplete && (
                <div className="flex justify-end gap-3 pt-4 border-t">
                  <Button
                    variant="outline"
                    onClick={() => safeGoBack("/set-matrices", setLocation)}
                    data-testid="button-back-to-list"
                  >
                    Powrót do listy
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => safeGoBack(`/set-matrices/settings/${matrixId}`, setLocation)}
                    data-testid="button-back-to-matrix"
                  >
                    Powrót do matrycy
                  </Button>
                  <Button
                    onClick={() => setLocation("/catalog-sets")}
                    data-testid="button-go-to-sets"
                  >
                    <Package className="h-4 w-4 mr-2" />
                    Przejdź do zestawów
                  </Button>
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      )}
    </div>
  );
}
