import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Package, Search, X, Store, Download, Link2, CheckSquare, Square } from "lucide-react";
import { useState, useMemo, useEffect, useCallback, useRef } from "react";
import { Skeleton } from "@/components/ui/skeleton";
import { useLocation } from "wouter";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { ConnectionStatusIcon } from "@/components/connection-status-icon";
import { MarketplaceLinkDialog, parseMarketplaceTitle, DictionaryItem } from "@/components/marketplace-catalog-link-dialog";
import { SavedFiltersManager } from "@/components/saved-filters-manager";

// Catalog suggestion from batch-suggest
interface CatalogSuggestion {
  id: number;
  sku: string;
  title: string;
  color?: string | null;
  length?: number | null;
  width?: number | null;
  doors?: string | null;
  legs?: string | null;
  primary_images?: Array<{ url: string; imageType: string }> | null;
}

// Parameter tile colors
const PARAM_TILE_COLORS: Record<string, string> = {
  dimension: '#6B7280',
  color: '#9CA3AF',
  door: '#8688e4',
  leg: '#94A3B8',
};

// Get contrasting text color
function getContrastColor(hexColor: string): string {
  const hex = hexColor.replace('#', '');
  const r = parseInt(hex.substr(0, 2), 16);
  const g = parseInt(hex.substr(2, 2), 16);
  const b = parseInt(hex.substr(4, 2), 16);
  const luminance = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
  return luminance > 0.5 ? '#1f2937' : '#ffffff';
}

interface MarketplaceProduct {
  id: number;
  offer_external_id: string;
  source: 'allegro' | 'shoper';
  name: string;
  image_url: string | null;
  last_sold_at: string | null;
  times_sold: number;
  total_quantity_sold: number;
  avg_unit_price: string | null;
  created_at: string;
  updated_at: string;
  linkType: 'product' | 'set' | null;
  catalogProductId: number | null;
  catalogProductSku: string | null;
  catalogSetId: number | null;
  catalogSetSku: string | null;
  isLinked: boolean;
  catalogImageUrl: string | null;
}

const PAGE_SLUG = 'marketplace-products';

interface StoredFilters {
  searchQuery: string;
  sourceFilter: "all" | "allegro" | "shoper";
  linkedFilter: "all" | "linked" | "unlinked";
}

export default function ProductsPage() {
  const [searchQuery, setSearchQuery] = useState("");
  const [sourceFilter, setSourceFilter] = useState<"all" | "allegro" | "shoper">("all");
  const [linkedFilter, setLinkedFilter] = useState<"all" | "linked" | "unlinked">("all");
  const [filtersInitialized, setFiltersInitialized] = useState(false);
  const [linkDialogOpen, setLinkDialogOpen] = useState(false);
  const [selectedProductForLinking, setSelectedProductForLinking] = useState<MarketplaceProduct | null>(null);
  const [selectedProducts, setSelectedProducts] = useState<Set<string>>(new Set());
  const [suggestions, setSuggestions] = useState<Record<string, CatalogSuggestion | null>>({});
  const [, navigate] = useLocation();
  const { toast } = useToast();

  // Load filters from database
  const { data: savedSettings } = useQuery<{ filters: StoredFilters | null }>({
    queryKey: [`/api/user-settings/${PAGE_SLUG}`],
  });

  // Initialize filters from database once loaded
  useEffect(() => {
    if (savedSettings && !filtersInitialized) {
      const f = savedSettings.filters;
      if (f) {
        if (f.searchQuery) setSearchQuery(f.searchQuery);
        if (f.sourceFilter) setSourceFilter(f.sourceFilter);
        if (f.linkedFilter) setLinkedFilter(f.linkedFilter);
      }
      setFiltersInitialized(true);
    }
  }, [savedSettings, filtersInitialized]);

  // Save filters to database mutation
  const saveFiltersMutation = useMutation({
    mutationFn: async (filters: StoredFilters) => {
      return apiRequest('PUT', `/api/user-settings/${PAGE_SLUG}`, { filters });
    },
  });

  // Auto-save filters to database when they change (debounced)
  const saveTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  useEffect(() => {
    if (!filtersInitialized) return;
    
    if (saveTimeoutRef.current) {
      clearTimeout(saveTimeoutRef.current);
    }
    
    saveTimeoutRef.current = setTimeout(() => {
      saveFiltersMutation.mutate({ searchQuery, sourceFilter, linkedFilter });
    }, 500);
    
    return () => {
      if (saveTimeoutRef.current) {
        clearTimeout(saveTimeoutRef.current);
      }
    };
  }, [searchQuery, sourceFilter, linkedFilter, filtersInitialized]);

  // Fetch dictionaries for smart matching - use isSuccess instead of isFetched
  // to ensure data is actually available, not just cache miss resolved
  const { data: productTypesDict = [], isSuccess: productTypesSuccess } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries", { type: "product_type" }],
  });
  const { data: colorsDict = [], isSuccess: colorsSuccess } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries", { type: "color" }],
  });
  const { data: doorsDict = [], isSuccess: doorsSuccess } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries", { type: "door" }],
  });
  const { data: legsDict = [], isSuccess: legsSuccess } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries", { type: "leg" }],
  });
  
  // Consider dictionaries ready when all queries succeeded AND have actual data
  // This prevents parsing with empty arrays from stale cache
  const dictionariesReady = 
    productTypesSuccess && colorsSuccess && doorsSuccess && legsSuccess &&
    productTypesDict.length > 0 && colorsDict.length > 0 && 
    doorsDict.length > 0 && legsDict.length > 0;

  // Pre-compute dictionary lookup sets once (memoized)
  const dictionaryLookups = useMemo(() => {
    if (!dictionariesReady) return null;
    
    const colorCodes = new Set<string>();
    const typeCodes = new Set<string>();
    const doorCodes = new Set<string>();
    const legCodes = new Set<string>();
    
    colorsDict.filter(c => c.isActive).forEach(c => {
      if (c.code) colorCodes.add(c.code.toUpperCase());
      if (c.name) colorCodes.add(c.name.toUpperCase());
      if (c.readableName) colorCodes.add(c.readableName.toUpperCase());
    });
    
    productTypesDict.filter(t => t.isActive).forEach(t => {
      if (t.code) typeCodes.add(t.code.toUpperCase());
    });
    
    doorsDict.filter(d => d.isActive).forEach(d => {
      if (d.code) doorCodes.add(d.code.toUpperCase());
      if (d.name) doorCodes.add(d.name.toUpperCase());
    });
    
    legsDict.filter(l => l.isActive).forEach(l => {
      if (l.code) legCodes.add(l.code.toUpperCase());
      if (l.name) legCodes.add(l.name.toUpperCase());
    });
    
    return { colorCodes, typeCodes, doorCodes, legCodes };
  }, [dictionariesReady, colorsDict, productTypesDict, doorsDict, legsDict]);

  // Helper to get token category (memoized function)
  const getTokenCategory = useCallback((token: string): 'color' | 'type' | 'door' | 'leg' | 'dimension' | 'other' => {
    if (!dictionaryLookups) return 'other';
    const upper = token.toUpperCase();
    if (dictionaryLookups.colorCodes.has(upper)) return 'color';
    if (dictionaryLookups.typeCodes.has(upper)) return 'type';
    if (dictionaryLookups.doorCodes.has(upper)) return 'door';
    if (dictionaryLookups.legCodes.has(upper)) return 'leg';
    if (/^\d+$/.test(token)) return 'dimension';
    return 'other';
  }, [dictionaryLookups]);

  // Build smart tokens for each unlinked product
  const getSmartTokens = useCallback((name: string) => {
    const tokens = parseMarketplaceTitle(name, {
      productTypes: productTypesDict,
      colors: colorsDict,
      doors: doorsDict,
      legs: legsDict,
    });
    return tokens;
  }, [productTypesDict, colorsDict, doorsDict, legsDict]);

  const queryParams = useMemo(() => {
    const params: Record<string, string> = {};
    if (searchQuery) params.search = searchQuery;
    if (sourceFilter !== "all") params.source = sourceFilter;
    if (linkedFilter !== "all") params.linked = linkedFilter;
    return params;
  }, [searchQuery, sourceFilter, linkedFilter]);

  const queryKey = useMemo(() => ['/api/products', queryParams], [queryParams]);

  const { data: products = [], isLoading } = useQuery<MarketplaceProduct[]>({
    queryKey,
  });

  // Pre-compute tokens for all products immediately when dictionaries are ready
  const productTokens = useMemo(() => {
    if (!dictionariesReady || products.length === 0) return {};
    
    const tokens: Record<string, string[]> = {};
    const dictionaries = {
      productTypes: productTypesDict,
      colors: colorsDict,
      doors: doorsDict,
      legs: legsDict,
    };
    
    products.forEach(p => {
      if (!p.isLinked) {
        tokens[p.offer_external_id] = parseMarketplaceTitle(p.name, dictionaries);
      }
    });
    
    return tokens;
  }, [dictionariesReady, products, productTypesDict, colorsDict, doorsDict, legsDict]);

  const syncImageMutation = useMutation({
    mutationFn: async (externalId: string) => {
      const response = await fetch(`/api/marketplace-products/${externalId}/sync-image`, {
        method: 'POST',
        credentials: 'include',
      });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to sync image');
      }
      return response.json();
    },
    onSuccess: (data, externalId) => {
      toast({
        title: "✅ Zdjęcie zsynchronizowane",
        description: "Zdjęcie produktu zostało pobrane z Allegro",
      });
      queryClient.invalidateQueries({ queryKey: ['/api/products'] });
    },
    onError: (error: any, externalId) => {
      toast({
        title: "❌ Błąd synchronizacji",
        description: error.message || "Nie udało się pobrać zdjęcia",
        variant: "destructive",
      });
    },
  });

  // Log dictionary loading status separately from effect
  useEffect(() => {
    console.log('[DictStatus] dictionaries updated:', {
      colors: colorsDict.length,
      types: productTypesDict.length, 
      doors: doorsDict.length,
      legs: legsDict.length,
      colorsSuccess,
      productTypesSuccess,
      dictionariesReady,
    });
  }, [colorsDict.length, productTypesDict.length, doorsDict.length, legsDict.length, colorsSuccess, productTypesSuccess, dictionariesReady]);

  // Fetch best match suggestions for unlinked products (chunked in batches of 50)
  // This runs in background - tokens are already displayed from productTokens useMemo
  useEffect(() => {
    // Wait for tokens to be computed
    if (Object.keys(productTokens).length === 0 || products.length === 0) return;
    
    const unlinkedProducts = products.filter(p => !p.isLinked);
    if (unlinkedProducts.length === 0) return;
    
    // Use pre-computed tokens from productTokens
    const batchData = unlinkedProducts
      .filter(p => productTokens[p.offer_external_id]?.length > 0)
      .map(p => ({
        externalId: p.offer_external_id,
        searchTokens: productTokens[p.offer_external_id],
      }));
    
    if (batchData.length === 0) return;
    
    // Split into chunks of 50 for API limit
    const CHUNK_SIZE = 50;
    const chunks: typeof batchData[] = [];
    for (let i = 0; i < batchData.length; i += CHUNK_SIZE) {
      chunks.push(batchData.slice(i, i + CHUNK_SIZE));
    }
    
    // Fetch all chunks and merge results
    const fetchAllChunks = async () => {
      const allSuggestions: Record<string, CatalogSuggestion | null> = {};
      
      for (const chunk of chunks) {
        try {
          const res = await fetch('/api/catalog/products/batch-suggest', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            credentials: 'include',
            body: JSON.stringify({ products: chunk }),
          });
          const data = await res.json();
          Object.assign(allSuggestions, data);
        } catch (err) {
          console.error('Failed to fetch suggestions chunk:', err);
        }
      }
      
      setSuggestions(allSuggestions);
    };
    
    fetchAllChunks();
  }, [products, productTokens]);

  // Bulk link mutation
  const bulkLinkMutation = useMutation({
    mutationFn: async (links: Array<{ catalogProductId: number; externalId: string; platform: string }>) => {
      const response = await apiRequest('POST', '/api/catalog/products/bulk-link', { links });
      return response.json();
    },
    onSuccess: (data: any) => {
      toast({
        title: "Produkty połączone",
        description: `Połączono ${data.linked} produktów${data.skipped > 0 ? `, pominięto ${data.skipped}` : ''}`,
      });
      setSelectedProducts(new Set());
      queryClient.invalidateQueries({ queryKey: ['/api/products'] });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd łączenia",
        description: error.message || "Nie udało się połączyć produktów",
        variant: "destructive",
      });
    },
  });

  // Single quick link mutation
  const quickLinkMutation = useMutation({
    mutationFn: async ({ catalogProductId, externalId, platform }: { catalogProductId: number; externalId: string; platform: string }) => {
      const response = await apiRequest('POST', '/api/catalog/products/link-marketplace', { catalogProductId, externalId, platform, linkType: 'product' });
      return response.json();
    },
    onSuccess: () => {
      toast({ title: "Produkt połączony" });
      queryClient.invalidateQueries({ queryKey: ['/api/products'] });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd łączenia",
        description: error.message || "Nie udało się połączyć produktu",
        variant: "destructive",
      });
    },
  });

  // Toggle product selection
  const toggleProductSelection = (externalId: string) => {
    setSelectedProducts(prev => {
      const next = new Set(prev);
      if (next.has(externalId)) {
        next.delete(externalId);
      } else {
        next.add(externalId);
      }
      return next;
    });
  };

  // Select all unlinked products with suggestions
  const selectAllWithSuggestions = () => {
    const unlinkedWithSuggestions = products
      .filter(p => !p.isLinked && suggestions[p.offer_external_id])
      .map(p => p.offer_external_id);
    setSelectedProducts(new Set(unlinkedWithSuggestions));
  };

  // Clear selection
  const clearSelection = () => setSelectedProducts(new Set());

  // Bulk link selected products
  const handleBulkLink = () => {
    const links = Array.from(selectedProducts)
      .map(externalId => {
        const product = products.find(p => p.offer_external_id === externalId);
        const suggestion = suggestions[externalId];
        if (!product || !suggestion) return null;
        return {
          catalogProductId: suggestion.id,
          externalId,
          platform: product.source,
        };
      })
      .filter((l): l is NonNullable<typeof l> => l !== null);
    
    if (links.length > 0) {
      bulkLinkMutation.mutate(links);
    }
  };

  const clearAllFilters = () => {
    setSearchQuery("");
    setSourceFilter("all");
    setLinkedFilter("all");
  };

  const hasActiveFilters = searchQuery || sourceFilter !== "all" || linkedFilter !== "all";

  const currentFilters = useMemo(() => ({
    searchQuery,
    sourceFilter,
    linkedFilter,
  }), [searchQuery, sourceFilter, linkedFilter]);

  const handleLoadFilters = (filters: Record<string, any>) => {
    if (!filters) return;
    if (filters.searchQuery !== undefined) setSearchQuery(filters.searchQuery);
    if (filters.sourceFilter !== undefined) setSourceFilter(filters.sourceFilter);
    if (filters.linkedFilter !== undefined) setLinkedFilter(filters.linkedFilter);
  };

  // Compute linking statistics
  const linkingStats = useMemo(() => {
    const total = products.length;
    const linked = products.filter(p => p.isLinked).length;
    const unlinked = total - linked;
    const linkedPercent = total > 0 ? Math.round((linked / total) * 100) : 0;
    return { total, linked, unlinked, linkedPercent };
  }, [products]);

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
        <div>
          <h1 className="text-3xl font-bold mb-2">Produkty Marketplace</h1>
          <p className="text-muted-foreground">
            Produkty sprzedane na Allegro i Shoper (agregowane z zamówień)
          </p>
        </div>
        
        {/* Linking statistics */}
        {!isLoading && products.length > 0 && (
          <div className="flex items-center gap-3 bg-muted/50 rounded-lg px-4 py-2 border">
            <div className="text-center">
              <div className="text-2xl font-bold">{linkingStats.total}</div>
              <div className="text-xs text-muted-foreground">Wszystkich</div>
            </div>
            <div className="h-8 w-px bg-border" />
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600">{linkingStats.linked}</div>
              <div className="text-xs text-muted-foreground">Połączonych</div>
            </div>
            <div className="h-8 w-px bg-border" />
            <div className="text-center">
              <div className="text-2xl font-bold text-orange-500">{linkingStats.unlinked}</div>
              <div className="text-xs text-muted-foreground">Niepołączonych</div>
            </div>
            <div className="h-8 w-px bg-border" />
            <div className="text-center">
              <div className="text-2xl font-bold text-primary">{linkingStats.linkedPercent}%</div>
              <div className="text-xs text-muted-foreground">Pokrycie</div>
            </div>
          </div>
        )}
      </div>

      {/* Wyszukiwarka */}
      <Card className="border-2 border-primary/20 bg-primary/5">
        <CardContent className="pt-6">
          <div className="space-y-2">
            <label className="text-sm font-medium text-primary flex items-center gap-2">
              <Search className="h-4 w-4" />
              Wyszukaj produkt
            </label>
            <div className="relative">
              <Input
                placeholder="Wpisz nazwę produktu lub SKU..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="text-lg h-12 bg-background"
                data-testid="input-search-main"
              />
              {searchQuery && (
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() => setSearchQuery("")}
                  className="absolute right-2 top-1/2 -translate-y-1/2"
                  data-testid="button-clear-search"
                >
                  <X className="h-4 w-4" />
                </Button>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Filtry */}
      <Card>
        <CardHeader className="flex flex-row items-center justify-between gap-4">
          <CardTitle className="flex items-center gap-2">
            <Store className="h-5 w-5" />
            Filtry
          </CardTitle>
          <SavedFiltersManager
            context="marketplace_products"
            currentFilters={currentFilters}
            onFiltersLoad={handleLoadFilters}
          />
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="space-y-2">
              <label className="text-sm font-medium">Źródło</label>
              <Select value={sourceFilter} onValueChange={(val: "all" | "allegro" | "shoper") => setSourceFilter(val)}>
                <SelectTrigger data-testid="select-source">
                  <SelectValue placeholder="Wszystkie" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie platformy</SelectItem>
                  <SelectItem value="allegro">Allegro</SelectItem>
                  <SelectItem value="shoper">Shoper</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <label className="text-sm font-medium">Status połączenia</label>
              <Select value={linkedFilter} onValueChange={(val: "all" | "linked" | "unlinked") => setLinkedFilter(val)}>
                <SelectTrigger data-testid="select-linked">
                  <SelectValue placeholder="Wszystkie" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  <SelectItem value="linked">Połączone</SelectItem>
                  <SelectItem value="unlinked">Niepołączone</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Aktywne filtry */}
      {hasActiveFilters && (
        <div className="flex flex-wrap items-center gap-2 p-4 bg-muted/50 rounded-lg border">
          <span className="text-sm font-medium text-muted-foreground">Aktywne filtry:</span>
          
          {searchQuery && (
            <Badge
              variant="default"
              className="gap-1 bg-blue-600 hover:bg-blue-700 cursor-pointer"
              onClick={() => setSearchQuery("")}
              data-testid="badge-search"
            >
              {searchQuery}
              <X className="h-3 w-3" />
            </Badge>
          )}

          {sourceFilter !== "all" && (
            <Badge
              variant="default"
              className="gap-1 bg-blue-600 hover:bg-blue-700 cursor-pointer"
              onClick={() => setSourceFilter("all")}
              data-testid="badge-source"
            >
              {sourceFilter === "allegro" ? "Allegro" : "Shoper"}
              <X className="h-3 w-3" />
            </Badge>
          )}

          {linkedFilter !== "all" && (
            <Badge
              variant="default"
              className="gap-1 bg-emerald-600 hover:bg-emerald-700 cursor-pointer"
              onClick={() => setLinkedFilter("all")}
              data-testid="badge-linked"
            >
              {linkedFilter === "linked" ? "Połączone" : "Niepołączone"}
              <X className="h-3 w-3" />
            </Badge>
          )}

          <Button
            variant="ghost"
            size="sm"
            onClick={clearAllFilters}
            className="ml-auto"
            data-testid="button-clear-all-filters"
          >
            Wyczyść wszystko
          </Button>
        </div>
      )}

      {/* Tabela produktów */}
      <Card>
        <CardHeader>
          <CardTitle>Produkty ({products.length})</CardTitle>
          <CardDescription>
            Unikalne produkty sprzedane na marketplace (Allegro + Shoper)
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[...Array(5)].map((_, i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          ) : products.length === 0 ? (
            <div className="text-center py-12">
              <Package className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <p className="text-muted-foreground">Nie znaleziono produktów</p>
            </div>
          ) : (
            <>
              {/* Bulk action bar */}
              {selectedProducts.size > 0 && (
                <div className="flex items-center gap-3 p-3 mb-3 bg-primary/10 border border-primary/20 rounded-lg">
                  <CheckSquare className="h-5 w-5 text-primary" />
                  <span className="font-medium">Zaznaczono: {selectedProducts.size}</span>
                  <Button
                    size="sm"
                    onClick={handleBulkLink}
                    disabled={bulkLinkMutation.isPending}
                    data-testid="button-bulk-link"
                  >
                    <Link2 className="h-4 w-4 mr-2" />
                    {bulkLinkMutation.isPending ? 'Łączenie...' : 'Połącz zaznaczone'}
                  </Button>
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={clearSelection}
                    data-testid="button-clear-selection"
                  >
                    Odznacz
                  </Button>
                  <Button
                    size="sm"
                    variant="ghost"
                    onClick={selectAllWithSuggestions}
                    data-testid="button-select-all"
                  >
                    Zaznacz wszystkie z dopasowaniem
                  </Button>
                </div>
              )}
              <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="w-10"></TableHead>
                  <TableHead className="w-20">Zdjęcie</TableHead>
                  <TableHead>SKU</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Dopasowanie</TableHead>
                  <TableHead>Źródło</TableHead>
                  <TableHead>Sprzedane</TableHead>
                  <TableHead className="text-center">Połączenie</TableHead>
                  <TableHead className="w-24">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {products.map((product) => (
                  <TableRow 
                    key={product.id} 
                    data-testid={`row-product-${product.offer_external_id}`}
                  >
                    {/* Checkbox for unlinked products */}
                    <TableCell onClick={(e) => e.stopPropagation()}>
                      {!product.isLinked && suggestions[product.offer_external_id] && (
                        <Checkbox
                          checked={selectedProducts.has(product.offer_external_id)}
                          onCheckedChange={() => toggleProductSelection(product.offer_external_id)}
                          data-testid={`checkbox-${product.offer_external_id}`}
                        />
                      )}
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}?source=${product.source}`)}
                    >
                      {(product.catalogImageUrl || product.image_url) ? (
                        <div className="relative">
                          <img
                            src={product.catalogImageUrl || product.image_url || ''}
                            alt={product.name}
                            className="h-20 w-20 object-cover rounded border"
                            onError={(e) => {
                              (e.target as HTMLImageElement).src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iODAiIGhlaWdodD0iODAiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PHJlY3Qgd2lkdGg9IjgwIiBoZWlnaHQ9IjgwIiBmaWxsPSIjZjNmNGY2Ii8+PC9zdmc+';
                            }}
                          />
                          {product.catalogImageUrl && (
                            <div className="absolute -top-1 -right-1 w-4 h-4 bg-green-500 rounded-full flex items-center justify-center" title="Zdjęcie z katalogu">
                              <span className="text-white text-[8px] font-bold">K</span>
                            </div>
                          )}
                        </div>
                      ) : (
                        <div className="h-20 w-20 bg-muted rounded border flex items-center justify-center">
                          <Package className="h-10 w-10 text-muted-foreground" />
                        </div>
                      )}
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}?source=${product.source}`)}
                    >
                      <span className="font-mono text-sm" data-testid={`text-sku-${product.offer_external_id}`}>
                        {product.offer_external_id}
                      </span>
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}?source=${product.source}`)}
                    >
                      <div className="max-w-md">
                        <div className="font-medium line-clamp-2">{product.name}</div>
                      </div>
                    </TableCell>
                    {/* Suggestion/Matching cell */}
                    <TableCell onClick={(e) => e.stopPropagation()}>
                      {!product.isLinked && (() => {
                        const suggestion = suggestions[product.offer_external_id];
                        // Use pre-computed tokens (instant, no waiting for API)
                        const marketplaceTokens = productTokens[product.offer_external_id] || [];
                        
                        // Show marketplace tokens even without suggestion
                        if (!suggestion && marketplaceTokens.length === 0) {
                          return <span className="text-xs text-muted-foreground">—</span>;
                        }
                        
                        // Build parameter tiles for catalog suggestion
                        const colorItem = suggestion ? colorsDict.find(c => c.code === suggestion.color) : null;
                        const doorItem = suggestion ? doorsDict.find(d => d.code === suggestion.doors) : null;
                        const legItem = suggestion ? legsDict.find(l => l.code === suggestion.legs) : null;
                        
                        const paramTiles: Array<{ value: string; bgColor: string; matched: boolean }> = [];
                        
                        // Helper: check if dimension token matches catalog dimension (handles cm→mm conversion)
                        // Token "60" from title matches catalog dimension 600 (60cm = 600mm)
                        const matchDimension = (dimMmRaw: number | string) => {
                          const dimMm = typeof dimMmRaw === 'string' ? parseInt(dimMmRaw) : dimMmRaw;
                          if (isNaN(dimMm)) return false;
                          const result = marketplaceTokens.some(t => {
                            const tokenNum = parseInt(t);
                            if (isNaN(tokenNum)) return false;
                            // Direct match (both in same unit)
                            if (tokenNum === dimMm) return true;
                            // Token in cm, dimension in mm (60 → 600)
                            if (tokenNum * 10 === dimMm) return true;
                            // Token in mm, dimension in cm (600 → 60) - unlikely but safe
                            if (tokenNum === dimMm * 10) return true;
                            return false;
                          });
                          return result;
                        };
                        
                        if (suggestion?.color) {
                          const matched = marketplaceTokens.some(t => t.toLowerCase() === suggestion.color?.toLowerCase());
                          paramTiles.push({ value: suggestion.color, bgColor: colorItem?.color || PARAM_TILE_COLORS.color, matched });
                        }
                        if (suggestion?.width) {
                          const matched = matchDimension(suggestion.width);
                          paramTiles.push({ value: String(suggestion.width), bgColor: PARAM_TILE_COLORS.dimension, matched });
                        }
                        if (suggestion?.length) {
                          const matched = matchDimension(suggestion.length);
                          paramTiles.push({ value: String(suggestion.length), bgColor: PARAM_TILE_COLORS.dimension, matched });
                        }
                        if (suggestion?.doors) {
                          const matched = marketplaceTokens.some(t => t.toLowerCase() === suggestion.doors?.toLowerCase());
                          paramTiles.push({ value: suggestion.doors, bgColor: doorItem?.color || PARAM_TILE_COLORS.door, matched });
                        }
                        if (suggestion?.legs) {
                          const matched = marketplaceTokens.some(t => t.toLowerCase() === suggestion.legs?.toLowerCase());
                          paramTiles.push({ value: suggestion.legs, bgColor: legItem?.color || PARAM_TILE_COLORS.leg, matched });
                        }
                        
                        // Count matched params
                        const matchedCount = paramTiles.filter(t => t.matched).length;
                        const totalParams = paramTiles.length;
                        
                        return (
                          <div className="space-y-2">
                            {/* Marketplace tokens - green for dictionary matches, blue for dimensions */}
                            {marketplaceTokens.length > 0 && (
                              <div className="flex flex-wrap gap-1">
                                <span className="text-[10px] text-muted-foreground mr-1">Szukane:</span>
                                {marketplaceTokens.map((token, idx) => {
                                  const category = getTokenCategory(token);
                                  const isDictMatch = category === 'color' || category === 'type' || category === 'door' || category === 'leg';
                                  const isDimension = category === 'dimension';
                                  return (
                                    <Badge 
                                      key={idx} 
                                      variant={isDictMatch ? "default" : "outline"}
                                      className={`text-[10px] px-1 py-0 h-4 ${isDictMatch ? 'bg-green-600 hover:bg-green-700 text-white' : isDimension ? 'bg-blue-600/20 text-blue-400 border-blue-500/50' : ''}`}
                                      data-testid={`token-${product.offer_external_id}-${idx}`}
                                    >
                                      {token}
                                    </Badge>
                                  );
                                })}
                              </div>
                            )}
                            
                            {/* Catalog suggestion with matching */}
                            {suggestion && (
                              <div 
                                className="cursor-pointer p-2 rounded border hover:border-primary hover:bg-primary/5 transition-colors"
                                onClick={() => {
                                  quickLinkMutation.mutate({
                                    catalogProductId: suggestion.id,
                                    externalId: product.offer_external_id,
                                    platform: product.source,
                                  });
                                }}
                                title={`Kliknij aby połączyć z: ${suggestion.sku}`}
                                data-testid={`suggestion-${product.offer_external_id}`}
                              >
                                <div className="flex items-center gap-2 mb-1">
                                  <span className="text-xs font-mono text-muted-foreground">
                                    {suggestion.sku}
                                  </span>
                                  {totalParams > 0 && (
                                    <Badge 
                                      variant={matchedCount === totalParams ? "default" : "secondary"}
                                      className={`text-[10px] px-1 py-0 h-4 ${matchedCount === totalParams ? 'bg-green-600' : ''}`}
                                    >
                                      {matchedCount}/{totalParams}
                                    </Badge>
                                  )}
                                </div>
                                <div className="flex flex-wrap gap-1">
                                  {paramTiles.map((tile, idx) => (
                                    <span
                                      key={idx}
                                      className={`px-1.5 py-0.5 rounded text-xs font-medium ${tile.matched ? 'ring-2 ring-green-500 ring-offset-1' : 'opacity-60'}`}
                                      style={{ 
                                        backgroundColor: tile.bgColor, 
                                        color: getContrastColor(tile.bgColor) 
                                      }}
                                    >
                                      {tile.value}
                                    </span>
                                  ))}
                                </div>
                                <div className="text-[10px] text-primary mt-1 flex items-center gap-1">
                                  <Link2 className="h-3 w-3" />
                                  Połącz
                                </div>
                              </div>
                            )}
                            
                            {/* No catalog match found */}
                            {!suggestion && marketplaceTokens.length > 0 && (
                              <div className="text-[10px] text-muted-foreground italic">
                                Brak dopasowania w katalogu
                              </div>
                            )}
                          </div>
                        );
                      })()}
                      {product.isLinked && (
                        <span className="text-xs text-green-600">Połączony</span>
                      )}
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}?source=${product.source}`)}
                    >
                      <Badge variant={product.source === 'allegro' ? 'default' : 'secondary'}>
                        {product.source === 'allegro' ? 'Allegro' : 'Shoper'}
                      </Badge>
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}?source=${product.source}`)}
                    >
                      <div className="text-sm">
                        <div className="font-medium">{product.total_quantity_sold} szt.</div>
                        <div className="text-xs text-muted-foreground">{product.times_sold} zam.</div>
                      </div>
                    </TableCell>
                    <TableCell className="text-center" onClick={(e) => e.stopPropagation()}>
                      <ConnectionStatusIcon
                        isLinked={product.isLinked}
                        targetPath={
                          product.linkType === 'product' && product.catalogProductId
                            ? `/catalog-products/${product.catalogProductId}`
                            : product.linkType === 'set' && product.catalogSetId
                            ? `/catalog-sets/${product.catalogSetId}`
                            : null
                        }
                        targetSku={product.catalogProductSku || product.catalogSetSku}
                        productId={product.id}
                        onManageLinks={() => {
                          setSelectedProductForLinking(product);
                          setLinkDialogOpen(true);
                        }}
                      />
                    </TableCell>
                    <TableCell>
                      {(!product.image_url || product.image_url === '') && product.source.toUpperCase() === 'ALLEGRO' && (
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={(e) => {
                            e.stopPropagation();
                            syncImageMutation.mutate(product.offer_external_id);
                          }}
                          disabled={syncImageMutation.isPending}
                          data-testid={`button-sync-image-${product.offer_external_id}`}
                        >
                          <Download className="h-4 w-4 mr-2" />
                          {syncImageMutation.isPending ? 'Pobieranie...' : 'Pobierz zdjęcie'}
                        </Button>
                      )}
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
            </>
          )}
        </CardContent>
      </Card>

      {/* Marketplace Link Management Dialog */}
      {selectedProductForLinking && (
        <MarketplaceLinkDialog
          open={linkDialogOpen}
          onOpenChange={setLinkDialogOpen}
          mode="marketplace-to-catalog"
          marketplaceProduct={{
            id: selectedProductForLinking.id,
            externalId: selectedProductForLinking.offer_external_id,
            platform: selectedProductForLinking.source,
            sku: selectedProductForLinking.catalogProductSku || selectedProductForLinking.offer_external_id,
            name: selectedProductForLinking.name,
          }}
          onSuccess={() => {
            queryClient.invalidateQueries({ queryKey: ['/api/products'] });
          }}
        />
      )}
    </div>
  );
}
