import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Link, useLocation } from "wouter";
import { Plus, Edit, Trash2, Loader2, Copy } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { insertProductionWorkCenterSchema } from "@shared/schema";
import type { ProductionWorkCenter, ProductionLocation } from "@shared/schema";
import { z } from "zod";

// Extend shared schema with frontend-specific validation
const workCenterFormSchema = insertProductionWorkCenterSchema.extend({
  code: z.string().min(1, "Kod jest wymagany"),
  name: z.string().min(1, "Nazwa jest wymagana"),
  locationId: z.coerce.number().optional().nullable(),
  throughputPerHour: z.coerce.number().optional().nullable(),
  operatingCostPerHour: z.coerce.number().optional().nullable(),
  oeeTarget: z.coerce.number().min(0).max(100).optional().nullable(),
  currentOee: z.coerce.number().min(0).max(100).optional().nullable(),
});

type WorkCenterFormData = z.infer<typeof workCenterFormSchema>;

export default function ProductionWorkCenters() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [, setLocation] = useLocation();
  const { toast } = useToast();

  const form = useForm<WorkCenterFormData>({
    resolver: zodResolver(workCenterFormSchema),
    defaultValues: { 
      code: "", 
      name: "", 
      status: "available", 
      isActive: true,
      capabilities: [],
    },
  });

  const { data: workCenters = [], isLoading } = useQuery<ProductionWorkCenter[]>({
    queryKey: ["/api/production/work-centers"],
  });

  const { data: locations = [] } = useQuery<ProductionLocation[]>({
    queryKey: ["/api/production/locations"],
  });

  const createMutation = useMutation({
    mutationFn: (data: WorkCenterFormData) => {
      const payload = {
        ...data,
        throughputPerHour: data.throughputPerHour != null ? String(data.throughputPerHour) : null,
        operatingCostPerHour: data.operatingCostPerHour != null ? String(data.operatingCostPerHour) : null,
        oeeTarget: data.oeeTarget != null ? String(data.oeeTarget) : null,
        currentOee: data.currentOee != null ? String(data.currentOee) : null,
      };
      return apiRequest("POST", "/api/production/work-centers", payload);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/work-centers"] });
      setIsDialogOpen(false);
      form.reset();
      toast({ title: "Gniazdo robocze zostało utworzone" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się utworzyć gniazda roboczego",
        variant: "destructive" 
      });
    },
  });


  const deleteMutation = useMutation({
    mutationFn: (id: number) => apiRequest("DELETE", `/api/production/work-centers/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/work-centers"] });
      toast({ title: "Gniazdo robocze zostało usunięte" });
    },
  });

  const handleDuplicate = (workCenter: ProductionWorkCenter) => {
    const timestamp = new Date().getTime().toString().slice(-6);
    const duplicatedData: WorkCenterFormData = {
      code: `${workCenter.code}-${timestamp}`,
      name: `${workCenter.name} (kopia)`,
      description: workCenter.description ?? "",
      locationId: workCenter.locationId ?? null,
      capabilities: workCenter.capabilities ?? [],
      status: workCenter.status,
      throughputPerHour: workCenter.throughputPerHour ? Number(workCenter.throughputPerHour) : null,
      throughputUnit: workCenter.throughputUnit ?? "",
      operatingCostPerHour: workCenter.operatingCostPerHour ? Number(workCenter.operatingCostPerHour) : null,
      oeeTarget: workCenter.oeeTarget ? Number(workCenter.oeeTarget) : null,
      currentOee: workCenter.currentOee ? Number(workCenter.currentOee) : null,
      isActive: workCenter.isActive ?? true,
      notes: workCenter.notes ?? "",
    };
    
    createMutation.mutate(duplicatedData);
  };

  useEffect(() => {
    if (!isDialogOpen) {
      form.reset({ 
        code: "", 
        name: "", 
        status: "available", 
        isActive: true,
        capabilities: [],
      });
    }
  }, [isDialogOpen, form]);

  const onSubmit = (data: WorkCenterFormData) => {
    // Normalize locationId: 0 or undefined should be sent as null
    const normalizedData = {
      ...data,
      locationId: data.locationId && data.locationId !== 0 ? data.locationId : null,
    };
    createMutation.mutate(normalizedData);
  };

  const getLocationName = (locationId: number | null) => {
    if (!locationId) return "-";
    const location = locations.find(l => l.id === locationId);
    return location ? location.name : "-";
  };

  const getStatusBadge = (status: string) => {
    const variants: Record<string, "default" | "secondary" | "destructive" | "outline"> = {
      available: "default",
      busy: "secondary",
      maintenance: "outline",
      offline: "destructive",
    };
    const labels: Record<string, string> = {
      available: "Dostępne",
      busy: "Zajęte",
      maintenance: "Konserwacja",
      offline: "Wyłączone",
    };
    return <Badge variant={variants[status] || "default"}>{labels[status] || status}</Badge>;
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <h1 className="text-3xl font-bold" data-testid="text-page-title">Gniazda Robocze</h1>
      
      <Card>
        <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-4">
          <CardTitle>Stanowiska Produkcyjne</CardTitle>
          <Button
            onClick={() => setIsDialogOpen(true)}
            data-testid="button-add-workcenter"
          >
            <Plus className="mr-2 h-4 w-4" />
            Dodaj Gniazdo
          </Button>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex justify-center p-8">
              <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-workcenters" />
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Kod</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Lokalizacja</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Przepustowość</TableHead>
                  <TableHead>OEE Target</TableHead>
                  <TableHead>Aktywne</TableHead>
                  <TableHead>Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {workCenters.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={8} className="text-center text-muted-foreground">
                      Brak gniazd roboczych
                    </TableCell>
                  </TableRow>
                ) : (
                  workCenters.map((wc) => (
                    <TableRow 
                      key={wc.id} 
                      data-testid={`row-workcenter-${wc.id}`}
                      className="cursor-pointer hover-elevate"
                      onClick={() => setLocation(`/production/work-centers/${wc.id}`)}
                    >
                      <TableCell>
                        <span className="font-mono">
                          {wc.code}
                        </span>
                      </TableCell>
                      <TableCell>
                        <span>
                          {wc.name}
                        </span>
                      </TableCell>
                      <TableCell>{getLocationName(wc.locationId)}</TableCell>
                      <TableCell>{getStatusBadge(wc.status)}</TableCell>
                      <TableCell>
                        {wc.throughputPerHour ? `${wc.throughputPerHour} ${wc.throughputUnit || ''}/h` : '-'}
                      </TableCell>
                      <TableCell>
                        {wc.oeeTarget ? `${wc.oeeTarget}%` : '-'}
                      </TableCell>
                      <TableCell>
                        <Badge variant={wc.isActive ? "default" : "secondary"}>
                          {wc.isActive ? "Tak" : "Nie"}
                        </Badge>
                      </TableCell>
                      <TableCell onClick={(e) => e.stopPropagation()}>
                        <div className="flex gap-1">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => setLocation(`/production/work-centers/${wc.id}/edit`)}
                            data-testid={`button-edit-workcenter-${wc.id}`}
                          >
                            <Edit className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleDuplicate(wc)}
                            disabled={createMutation.isPending}
                            data-testid={`button-duplicate-workcenter-${wc.id}`}
                          >
                            <Copy className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => {
                              if (confirm("Czy na pewno chcesz usunąć to gniazdo robocze?")) {
                                deleteMutation.mutate(wc.id);
                              }
                            }}
                            data-testid={`button-delete-workcenter-${wc.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Dodaj Gniazdo Robocze</DialogTitle>
            <DialogDescription>
              Wypełnij formularz aby utworzyć nowe gniazdo robocze
            </DialogDescription>
          </DialogHeader>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="code"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kod *</FormLabel>
                      <FormControl>
                        <Input {...field} data-testid="input-code" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Nazwa *</FormLabel>
                      <FormControl>
                        <Input {...field} data-testid="input-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} value={field.value ?? ""} rows={2} data-testid="input-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="locationId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Lokalizacja</FormLabel>
                      <Select 
                        onValueChange={(value) => field.onChange(value === "0" ? null : parseInt(value))}
                        value={field.value?.toString() || "0"}
                      >
                        <FormControl>
                          <SelectTrigger data-testid="select-location">
                            <SelectValue />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="0">Brak</SelectItem>
                          {locations.map((loc) => (
                            <SelectItem key={loc.id} value={loc.id.toString()}>
                              {loc.path}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="status"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Status</FormLabel>
                      <Select onValueChange={field.onChange} value={field.value}>
                        <FormControl>
                          <SelectTrigger data-testid="select-status">
                            <SelectValue />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="available">Dostępne</SelectItem>
                          <SelectItem value="busy">Zajęte</SelectItem>
                          <SelectItem value="maintenance">Konserwacja</SelectItem>
                          <SelectItem value="offline">Wyłączone</SelectItem>
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="throughputPerHour"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Przepustowość / godz.</FormLabel>
                      <FormControl>
                        <Input 
                          type="number" 
                          step="0.01"
                          {...field} 
                          value={field.value ?? ""}
                          onChange={(e) => field.onChange(e.target.value === "" ? null : parseFloat(e.target.value))}
                          data-testid="input-throughput" 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="throughputUnit"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Jednostka</FormLabel>
                      <FormControl>
                        <Input {...field} value={field.value ?? ""} placeholder="np. szt, m2, kg" data-testid="input-unit" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="operatingCostPerHour"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Koszt operacyjny / godz. (PLN)</FormLabel>
                      <FormControl>
                        <Input 
                          type="number" 
                          step="0.01"
                          {...field} 
                          value={field.value ?? ""}
                          onChange={(e) => field.onChange(e.target.value === "" ? null : parseFloat(e.target.value))}
                          data-testid="input-cost" 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="oeeTarget"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Docelowe OEE (%)</FormLabel>
                      <FormControl>
                        <Input 
                          type="number" 
                          step="0.01"
                          min="0"
                          max="100"
                          {...field} 
                          value={field.value ?? ""}
                          onChange={(e) => field.onChange(e.target.value === "" ? null : parseFloat(e.target.value))}
                          data-testid="input-oee-target" 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Notatki</FormLabel>
                    <FormControl>
                      <Textarea {...field} value={field.value ?? ""} rows={2} data-testid="input-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="isActive"
                render={({ field }) => (
                  <FormItem className="flex flex-row items-start space-x-3 space-y-0 rounded-md border p-4">
                    <FormControl>
                      <Checkbox
                        checked={field.value ?? true}
                        onCheckedChange={field.onChange}
                        data-testid="checkbox-active"
                      />
                    </FormControl>
                    <div className="space-y-1 leading-none">
                      <FormLabel>Aktywne</FormLabel>
                    </div>
                  </FormItem>
                )}
              />

              <DialogFooter>
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => setIsDialogOpen(false)}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button
                  type="submit"
                  disabled={createMutation.isPending}
                  data-testid="button-submit"
                >
                  {createMutation.isPending && (
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  )}
                  Utwórz
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
