import { useState, useMemo } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Plus, Edit, Trash2, Loader2, ListOrdered, Search, Link2, CheckCircle2, XCircle, ChevronDown, ChevronRight, X } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage, FormDescription } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import { Separator } from "@/components/ui/separator";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from "@/components/ui/command";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { ScrollArea } from "@/components/ui/scroll-area";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { z } from "zod";

interface LinkedRouting {
  id: number;
  code: string;
  name: string;
}

interface RoutingVariant {
  id: number;
  variantCode: string;
  variantName: string;
  description: string | null;
  defaultOperations: string[];
  routingIds: number[];
  linkedRoutings: LinkedRouting[];
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
  rulesCount: number;
}

interface RoutingVariantRule {
  id: number;
  routingVariantId: number;
  variantCode?: string;
  variantName?: string;
  namePattern: string | null;
  formatkaCodes: string[];
  colorPattern: string | null;
  priority: number;
  isActive: boolean;
  createdAt: string;
}

interface ProductionRouting {
  id: number;
  code: string;
  name: string;
}

interface DictionaryEntry {
  id: number;
  code: string;
  name: string;
  readableName?: string;
}

const variantFormSchema = z.object({
  variantCode: z.string().min(1, "Kod jest wymagany"),
  variantName: z.string().min(1, "Nazwa jest wymagana"),
  description: z.string().optional().nullable(),
  defaultOperations: z.string().min(1, "Operacje są wymagane"),
  routingIds: z.array(z.number()).default([]),
  isActive: z.boolean().default(true),
});

const ruleFormSchema = z.object({
  routingVariantId: z.number(),
  namePattern: z.string().optional().nullable(),
  formatkaCodes: z.array(z.string()).default([]),
  colorPattern: z.string().optional().nullable(),
  priority: z.coerce.number().min(0).default(0),
  isActive: z.boolean().default(true),
});

type VariantFormData = z.infer<typeof variantFormSchema>;
type RuleFormData = z.infer<typeof ruleFormSchema>;

export default function ProductionRoutingVariants() {
  const [isVariantDialogOpen, setIsVariantDialogOpen] = useState(false);
  const [editingVariant, setEditingVariant] = useState<RoutingVariant | null>(null);
  const [isRuleDialogOpen, setIsRuleDialogOpen] = useState(false);
  const [editingRule, setEditingRule] = useState<RoutingVariantRule | null>(null);
  const [selectedVariantId, setSelectedVariantId] = useState<number | null>(null);
  const [expandedVariants, setExpandedVariants] = useState<Set<number>>(new Set());
  const [selectedRoutingIds, setSelectedRoutingIds] = useState<number[]>([]);
  const [routingSearchOpen, setRoutingSearchOpen] = useState(false);
  const [routingSearchValue, setRoutingSearchValue] = useState("");
  
  // Rule dialog state for formatka multi-select
  const [selectedFormatkaCodes, setSelectedFormatkaCodes] = useState<string[]>([]);
  const [formatkaSearchOpen, setFormatkaSearchOpen] = useState(false);
  const [formatkaSearchValue, setFormatkaSearchValue] = useState("");
  
  // Test section state
  const [testComponentOpen, setTestComponentOpen] = useState(false);
  const [testComponentSearch, setTestComponentSearch] = useState("");
  const [testComponentName, setTestComponentName] = useState("");
  const [testColorOpen, setTestColorOpen] = useState(false);
  const [testColorSearch, setTestColorSearch] = useState("");
  const [testColor, setTestColor] = useState("");
  const [testResult, setTestResult] = useState<any>(null);
  const [isTesting, setIsTesting] = useState(false);
  
  const { toast } = useToast();

  const variantForm = useForm<VariantFormData>({
    resolver: zodResolver(variantFormSchema),
    defaultValues: {
      variantCode: "",
      variantName: "",
      description: "",
      defaultOperations: "",
      routingIds: [],
      isActive: true,
    },
  });

  const ruleForm = useForm<RuleFormData>({
    resolver: zodResolver(ruleFormSchema),
    defaultValues: {
      routingVariantId: 0,
      namePattern: "",
      colorPattern: "",
      priority: 0,
      isActive: true,
    },
  });

  const { data: variants = [], isLoading: isLoadingVariants } = useQuery<RoutingVariant[]>({
    queryKey: ["/api/production/routing-variants"],
  });

  const { data: allRules = [], isLoading: isLoadingRules } = useQuery<RoutingVariantRule[]>({
    queryKey: ["/api/production/routing-variant-rules"],
  });

  const { data: routings = [] } = useQuery<ProductionRouting[]>({
    queryKey: ["/api/production/routings"],
  });

  const { data: componentDictionaries = [] } = useQuery<DictionaryEntry[]>({
    queryKey: ["/api/dictionaries", "component_cz1"],
    queryFn: async () => {
      const res = await fetch("/api/dictionaries?type=component_cz1");
      if (!res.ok) throw new Error("Failed to fetch dictionaries");
      return res.json();
    },
  });

  const { data: colorDictionaries = [] } = useQuery<DictionaryEntry[]>({
    queryKey: ["/api/dictionaries", "color"],
    queryFn: async () => {
      const res = await fetch("/api/dictionaries?type=color");
      if (!res.ok) throw new Error("Failed to fetch dictionaries");
      return res.json();
    },
  });

  // Generate variant name from selected routings
  const generatedVariantName = useMemo(() => {
    if (selectedRoutingIds.length === 0) return "";
    const selectedRoutings = routings.filter(r => selectedRoutingIds.includes(r.id));
    return selectedRoutings.map(r => r.name).join(" / ");
  }, [selectedRoutingIds, routings]);

  // Filter routings by search
  const filteredRoutings = useMemo(() => {
    if (!routingSearchValue) return routings;
    const search = routingSearchValue.toLowerCase();
    return routings.filter(r => 
      r.code.toLowerCase().includes(search) || 
      r.name.toLowerCase().includes(search)
    );
  }, [routings, routingSearchValue]);

  // Filter components by search
  const filteredComponents = useMemo(() => {
    if (!testComponentSearch) return componentDictionaries;
    const search = testComponentSearch.toLowerCase();
    return componentDictionaries.filter(c => 
      c.code.toLowerCase().includes(search) || 
      c.name.toLowerCase().includes(search) ||
      (c.readableName && c.readableName.toLowerCase().includes(search))
    );
  }, [componentDictionaries, testComponentSearch]);

  // Filter colors by search
  const filteredColors = useMemo(() => {
    if (!testColorSearch) return colorDictionaries;
    const search = testColorSearch.toLowerCase();
    return colorDictionaries.filter(c => 
      c.code.toLowerCase().includes(search) || 
      c.name.toLowerCase().includes(search) ||
      (c.readableName && c.readableName.toLowerCase().includes(search))
    );
  }, [colorDictionaries, testColorSearch]);

  // Filter formatkas for rule dialog
  const filteredFormatkas = useMemo(() => {
    if (!formatkaSearchValue) return componentDictionaries;
    const search = formatkaSearchValue.toLowerCase();
    return componentDictionaries.filter(c => 
      c.code.toLowerCase().includes(search) || 
      c.name.toLowerCase().includes(search) ||
      (c.readableName && c.readableName.toLowerCase().includes(search))
    );
  }, [componentDictionaries, formatkaSearchValue]);

  const createVariantMutation = useMutation({
    mutationFn: (data: any) => apiRequest("POST", "/api/production/routing-variants", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routing-variants"] });
      setIsVariantDialogOpen(false);
      variantForm.reset();
      setSelectedRoutingIds([]);
      toast({ title: "Wariant marszruty został utworzony" });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się utworzyć wariantu", variant: "destructive" });
    },
  });

  const updateVariantMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: any }) => apiRequest("PATCH", `/api/production/routing-variants/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routing-variants"] });
      setIsVariantDialogOpen(false);
      setEditingVariant(null);
      variantForm.reset();
      setSelectedRoutingIds([]);
      toast({ title: "Wariant marszruty został zaktualizowany" });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się zaktualizować wariantu", variant: "destructive" });
    },
  });

  const deleteVariantMutation = useMutation({
    mutationFn: (id: number) => apiRequest("DELETE", `/api/production/routing-variants/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routing-variants"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/routing-variant-rules"] });
      toast({ title: "Wariant marszruty został usunięty" });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się usunąć wariantu", variant: "destructive" });
    },
  });

  const createRuleMutation = useMutation({
    mutationFn: (data: any) => apiRequest("POST", "/api/production/routing-variant-rules", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routing-variant-rules"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/routing-variants"] });
      setIsRuleDialogOpen(false);
      ruleForm.reset();
      setSelectedFormatkaCodes([]);
      toast({ title: "Reguła została utworzona" });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się utworzyć reguły", variant: "destructive" });
    },
  });

  const updateRuleMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: any }) => apiRequest("PATCH", `/api/production/routing-variant-rules/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routing-variant-rules"] });
      setIsRuleDialogOpen(false);
      setEditingRule(null);
      ruleForm.reset();
      setSelectedFormatkaCodes([]);
      toast({ title: "Reguła została zaktualizowana" });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się zaktualizować reguły", variant: "destructive" });
    },
  });

  const deleteRuleMutation = useMutation({
    mutationFn: (id: number) => apiRequest("DELETE", `/api/production/routing-variant-rules/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routing-variant-rules"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/routing-variants"] });
      toast({ title: "Reguła została usunięta" });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się usunąć reguły", variant: "destructive" });
    },
  });

  const handleVariantSubmit = (data: VariantFormData) => {
    const payload = {
      variantCode: data.variantCode,
      variantName: data.variantName || generatedVariantName,
      description: data.description || null,
      defaultOperations: data.defaultOperations.split(",").map(op => op.trim()).filter(Boolean),
      routingIds: selectedRoutingIds,
      isActive: data.isActive,
    };

    if (editingVariant) {
      updateVariantMutation.mutate({ id: editingVariant.id, data: payload });
    } else {
      createVariantMutation.mutate(payload);
    }
  };

  const handleRuleSubmit = (data: RuleFormData) => {
    const payload = {
      routingVariantId: data.routingVariantId,
      namePattern: data.namePattern || null,
      formatkaCodes: selectedFormatkaCodes,
      colorPattern: data.colorPattern || null,
      priority: data.priority,
      isActive: data.isActive,
    };

    if (editingRule) {
      updateRuleMutation.mutate({ id: editingRule.id, data: payload });
    } else {
      createRuleMutation.mutate(payload);
    }
  };

  const toggleFormatkaSelection = (code: string) => {
    setSelectedFormatkaCodes(prev => 
      prev.includes(code) 
        ? prev.filter(c => c !== code)
        : [...prev, code]
    );
  };

  const openVariantDialog = (variant?: RoutingVariant) => {
    if (variant) {
      setEditingVariant(variant);
      setSelectedRoutingIds(variant.routingIds || []);
      variantForm.reset({
        variantCode: variant.variantCode,
        variantName: variant.variantName,
        description: variant.description || "",
        defaultOperations: variant.defaultOperations.join(", "),
        routingIds: variant.routingIds || [],
        isActive: variant.isActive,
      });
    } else {
      setEditingVariant(null);
      setSelectedRoutingIds([]);
      variantForm.reset({
        variantCode: "",
        variantName: "",
        description: "",
        defaultOperations: "",
        routingIds: [],
        isActive: true,
      });
    }
    setIsVariantDialogOpen(true);
  };

  const openRuleDialog = (variantId: number, rule?: RoutingVariantRule) => {
    setSelectedVariantId(variantId);
    if (rule) {
      setEditingRule(rule);
      setSelectedFormatkaCodes(rule.formatkaCodes || []);
      ruleForm.reset({
        routingVariantId: variantId,
        namePattern: rule.namePattern || "",
        formatkaCodes: rule.formatkaCodes || [],
        colorPattern: rule.colorPattern || "",
        priority: rule.priority,
        isActive: rule.isActive,
      });
    } else {
      setEditingRule(null);
      setSelectedFormatkaCodes([]);
      ruleForm.reset({
        routingVariantId: variantId,
        namePattern: "",
        formatkaCodes: [],
        colorPattern: "",
        priority: 0,
        isActive: true,
      });
    }
    setIsRuleDialogOpen(true);
  };

  const toggleVariantExpand = (variantId: number) => {
    const newExpanded = new Set(expandedVariants);
    if (newExpanded.has(variantId)) {
      newExpanded.delete(variantId);
    } else {
      newExpanded.add(variantId);
    }
    setExpandedVariants(newExpanded);
  };

  const getRulesForVariant = (variantId: number) => {
    return allRules.filter(rule => rule.routingVariantId === variantId);
  };

  const toggleRoutingSelection = (routingId: number) => {
    setSelectedRoutingIds(prev => 
      prev.includes(routingId) 
        ? prev.filter(id => id !== routingId)
        : [...prev, routingId]
    );
  };

  const handleTestPattern = async () => {
    if (!testComponentName) {
      toast({ title: "Wprowadź nazwę komponentu", variant: "destructive" });
      return;
    }

    setIsTesting(true);
    try {
      const response = await apiRequest("POST", "/api/production/routing-variant-rules/test", {
        componentName: testComponentName,
        color: testColor || null,
      });
      setTestResult(response);
    } catch (error: any) {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    } finally {
      setIsTesting(false);
    }
  };

  if (isLoadingVariants) {
    return (
      <div className="flex items-center justify-center h-64">
        <Loader2 className="w-8 h-8 animate-spin" />
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">Warianty Marszrut</h1>
          <p className="text-muted-foreground">Zarządzaj wariantami marszrut i regułami dopasowania dla formatek</p>
        </div>
        <Button onClick={() => openVariantDialog()} data-testid="button-add-variant">
          <Plus className="w-4 h-4 mr-2" />
          Dodaj wariant
        </Button>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <div className="lg:col-span-2 space-y-4">
          {variants.map((variant) => {
            const rules = getRulesForVariant(variant.id);
            const isExpanded = expandedVariants.has(variant.id);

            return (
              <Card key={variant.id} data-testid={`card-variant-${variant.id}`}>
                <Collapsible open={isExpanded} onOpenChange={() => toggleVariantExpand(variant.id)}>
                  <CardHeader className="pb-2">
                    <div className="flex items-center justify-between gap-2">
                      <CollapsibleTrigger className="flex items-center gap-2 hover-elevate rounded p-1 -m-1">
                        {isExpanded ? <ChevronDown className="w-4 h-4" /> : <ChevronRight className="w-4 h-4" />}
                        <div className="flex items-center gap-3">
                          <CardTitle className="text-lg">{variant.variantCode}</CardTitle>
                          <Badge variant={variant.isActive ? "default" : "secondary"}>
                            {variant.isActive ? "Aktywny" : "Nieaktywny"}
                          </Badge>
                        </div>
                      </CollapsibleTrigger>
                      <div className="flex items-center gap-2">
                        <Badge variant="outline">{rules.length} reguł</Badge>
                        <Button variant="ghost" size="icon" onClick={() => openVariantDialog(variant)} data-testid={`button-edit-variant-${variant.id}`}>
                          <Edit className="w-4 h-4" />
                        </Button>
                        <Button variant="ghost" size="icon" onClick={() => deleteVariantMutation.mutate(variant.id)} data-testid={`button-delete-variant-${variant.id}`}>
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                    <CardDescription>{variant.variantName}</CardDescription>
                  </CardHeader>

                  <CardContent className="pt-0">
                    <div className="flex flex-wrap gap-2 mb-2">
                      <span className="text-sm text-muted-foreground">Operacje:</span>
                      {variant.defaultOperations.map((op, i) => (
                        <Badge key={i} variant="secondary" className="text-xs">{op}</Badge>
                      ))}
                    </div>
                    
                    {variant.linkedRoutings && variant.linkedRoutings.length > 0 && (
                      <div className="flex items-start gap-2 text-sm text-muted-foreground mt-2">
                        <Link2 className="w-4 h-4 mt-0.5 flex-shrink-0" />
                        <div className="flex flex-wrap gap-1">
                          <span>Marszruty:</span>
                          {variant.linkedRoutings.map((r, i) => (
                            <Badge key={r.id} variant="outline" className="text-xs">
                              {r.code} - {r.name}
                            </Badge>
                          ))}
                        </div>
                      </div>
                    )}

                    <CollapsibleContent>
                      <Separator className="my-4" />
                      
                      <div className="flex items-center justify-between mb-3">
                        <h4 className="font-medium">Reguły dopasowania</h4>
                        <Button size="sm" variant="outline" onClick={() => openRuleDialog(variant.id)} data-testid={`button-add-rule-${variant.id}`}>
                          <Plus className="w-3 h-3 mr-1" />
                          Dodaj regułę
                        </Button>
                      </div>

                      {rules.length === 0 ? (
                        <p className="text-sm text-muted-foreground italic">Brak reguł - ten wariant nie będzie dopasowywany automatycznie</p>
                      ) : (
                        <Table>
                          <TableHeader>
                            <TableRow>
                              <TableHead>Formatki / Wzorzec</TableHead>
                              <TableHead>Kolor</TableHead>
                              <TableHead className="text-center">Priorytet</TableHead>
                              <TableHead className="text-center">Status</TableHead>
                              <TableHead className="w-[100px]">Akcje</TableHead>
                            </TableRow>
                          </TableHeader>
                          <TableBody>
                            {rules.map((rule) => (
                              <TableRow key={rule.id} data-testid={`row-rule-${rule.id}`}>
                                <TableCell>
                                  {rule.formatkaCodes && rule.formatkaCodes.length > 0 ? (
                                    <div className="flex flex-wrap gap-1">
                                      {rule.formatkaCodes.map(code => (
                                        <Badge key={code} variant="default" className="text-xs">
                                          {componentDictionaries.find(d => d.code === code)?.name || code}
                                        </Badge>
                                      ))}
                                    </div>
                                  ) : rule.namePattern ? (
                                    <span className="font-mono text-sm">{rule.namePattern}</span>
                                  ) : (
                                    <span className="text-muted-foreground">-</span>
                                  )}
                                </TableCell>
                                <TableCell>{rule.colorPattern || "-"}</TableCell>
                                <TableCell className="text-center">{rule.priority}</TableCell>
                                <TableCell className="text-center">
                                  {rule.isActive ? (
                                    <CheckCircle2 className="w-4 h-4 text-green-500 mx-auto" />
                                  ) : (
                                    <XCircle className="w-4 h-4 text-red-500 mx-auto" />
                                  )}
                                </TableCell>
                                <TableCell>
                                  <div className="flex gap-1">
                                    <Button variant="ghost" size="icon" onClick={() => openRuleDialog(variant.id, rule)} data-testid={`button-edit-rule-${rule.id}`}>
                                      <Edit className="w-4 h-4" />
                                    </Button>
                                    <Button variant="ghost" size="icon" onClick={() => deleteRuleMutation.mutate(rule.id)} data-testid={`button-delete-rule-${rule.id}`}>
                                      <Trash2 className="w-4 h-4" />
                                    </Button>
                                  </div>
                                </TableCell>
                              </TableRow>
                            ))}
                          </TableBody>
                        </Table>
                      )}
                    </CollapsibleContent>
                  </CardContent>
                </Collapsible>
              </Card>
            );
          })}

          {variants.length === 0 && (
            <Card>
              <CardContent className="py-8 text-center text-muted-foreground">
                <ListOrdered className="w-12 h-12 mx-auto mb-4 opacity-50" />
                <p>Brak wariantów marszrut</p>
                <p className="text-sm">Dodaj pierwszy wariant, aby rozpocząć</p>
              </CardContent>
            </Card>
          )}
        </div>

        <div>
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Search className="w-5 h-5" />
                Testuj dopasowanie
              </CardTitle>
              <CardDescription>
                Sprawdź, który wariant zostanie dopasowany dla danej formatki
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <label className="text-sm font-medium mb-2 block">Typ komponentu</label>
                <Popover open={testComponentOpen} onOpenChange={setTestComponentOpen}>
                  <PopoverTrigger asChild>
                    <Button 
                      variant="outline" 
                      role="combobox" 
                      className="w-full justify-between"
                      data-testid="select-test-component"
                    >
                      {testComponentName || "Wybierz komponent..."}
                      <ChevronDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-full p-0" align="start">
                    <Command>
                      <CommandInput 
                        placeholder="Szukaj komponentu..." 
                        value={testComponentSearch}
                        onValueChange={setTestComponentSearch}
                      />
                      <CommandList>
                        <CommandEmpty>Nie znaleziono komponentów</CommandEmpty>
                        <CommandGroup>
                          <ScrollArea className="h-[200px]">
                            {filteredComponents.map((component) => (
                              <CommandItem
                                key={component.id}
                                value={component.code}
                                onSelect={() => {
                                  setTestComponentName(component.code);
                                  setTestComponentOpen(false);
                                  setTestComponentSearch("");
                                }}
                              >
                                <span className="font-mono">{component.code}</span>
                                <span className="ml-2 text-muted-foreground text-sm">
                                  {component.readableName || component.name}
                                </span>
                              </CommandItem>
                            ))}
                          </ScrollArea>
                        </CommandGroup>
                      </CommandList>
                    </Command>
                  </PopoverContent>
                </Popover>
                <Input
                  className="mt-2"
                  placeholder="lub wpisz własny wzorzec np. BOCZKI-SZUFLADA-100x260"
                  value={testComponentName}
                  onChange={(e) => setTestComponentName(e.target.value)}
                  data-testid="input-test-component-name"
                />
              </div>
              
              <div>
                <label className="text-sm font-medium mb-2 block">Kolor</label>
                <Popover open={testColorOpen} onOpenChange={setTestColorOpen}>
                  <PopoverTrigger asChild>
                    <Button 
                      variant="outline" 
                      role="combobox" 
                      className="w-full justify-between"
                      data-testid="select-test-color"
                    >
                      {testColor || "Wybierz kolor..."}
                      <ChevronDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-full p-0" align="start">
                    <Command>
                      <CommandInput 
                        placeholder="Szukaj koloru..." 
                        value={testColorSearch}
                        onValueChange={setTestColorSearch}
                      />
                      <CommandList>
                        <CommandEmpty>Nie znaleziono kolorów</CommandEmpty>
                        <CommandGroup>
                          <ScrollArea className="h-[200px]">
                            {filteredColors.map((color) => (
                              <CommandItem
                                key={color.id}
                                value={color.code}
                                onSelect={() => {
                                  setTestColor(color.code);
                                  setTestColorOpen(false);
                                  setTestColorSearch("");
                                }}
                              >
                                <span className="font-mono">{color.code}</span>
                                <span className="ml-2 text-muted-foreground text-sm">
                                  {color.readableName || color.name}
                                </span>
                              </CommandItem>
                            ))}
                          </ScrollArea>
                        </CommandGroup>
                      </CommandList>
                    </Command>
                  </PopoverContent>
                </Popover>
              </div>

              <Button onClick={handleTestPattern} disabled={isTesting} className="w-full" data-testid="button-test-pattern">
                {isTesting ? <Loader2 className="w-4 h-4 animate-spin mr-2" /> : <Search className="w-4 h-4 mr-2" />}
                Testuj
              </Button>

              {testResult && (
                <div className="mt-4 p-4 rounded-lg bg-muted">
                  {testResult.matched ? (
                    <div className="space-y-2">
                      <div className="flex items-center gap-2 text-green-600">
                        <CheckCircle2 className="w-5 h-5" />
                        <span className="font-medium">Znaleziono dopasowanie!</span>
                      </div>
                      <div className="text-sm space-y-1">
                        <p><strong>Wariant:</strong> {testResult.matchedRule.variantCode}</p>
                        <p><strong>Nazwa:</strong> {testResult.matchedRule.variantName}</p>
                        <p><strong>Wzorzec:</strong> {testResult.matchedRule.namePattern || "dowolny"}</p>
                        {testResult.matchedRule.colorPattern && (
                          <p><strong>Wzorzec koloru:</strong> {testResult.matchedRule.colorPattern}</p>
                        )}
                        <p><strong>Priorytet:</strong> {testResult.matchedRule.priority}</p>
                        <div className="flex flex-wrap gap-1 mt-2">
                          {testResult.matchedRule.defaultOperations?.map((op: string, i: number) => (
                            <Badge key={i} variant="secondary" className="text-xs">{op}</Badge>
                          ))}
                        </div>
                        {testResult.matchedRule.linkedRoutings && testResult.matchedRule.linkedRoutings.length > 0 && (
                          <div className="mt-2">
                            <strong>Marszruty:</strong>
                            <div className="flex flex-wrap gap-1 mt-1">
                              {testResult.matchedRule.linkedRoutings.map((r: any) => (
                                <Badge key={r.id} variant="outline" className="text-xs">
                                  {r.code} - {r.name}
                                </Badge>
                              ))}
                            </div>
                          </div>
                        )}
                      </div>
                    </div>
                  ) : (
                    <div className="flex items-center gap-2 text-red-600">
                      <XCircle className="w-5 h-5" />
                      <span className="font-medium">Brak dopasowania</span>
                    </div>
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>

      {/* Variant Dialog */}
      <Dialog open={isVariantDialogOpen} onOpenChange={setIsVariantDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>{editingVariant ? "Edytuj wariant" : "Nowy wariant marszruty"}</DialogTitle>
            <DialogDescription>
              Wariant definiuje zestaw operacji produkcyjnych dla typu formatki
            </DialogDescription>
          </DialogHeader>
          <Form {...variantForm}>
            <form onSubmit={variantForm.handleSubmit(handleVariantSubmit)} className="space-y-4">
              <FormField
                control={variantForm.control}
                name="variantCode"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kod wariantu</FormLabel>
                    <FormControl>
                      <Input placeholder="np. CUT_EDGE_DRILL" {...field} data-testid="input-variant-code" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              {/* Multi-select for routings */}
              <div className="space-y-2">
                <FormLabel>Powiązane marszruty</FormLabel>
                <Popover open={routingSearchOpen} onOpenChange={setRoutingSearchOpen} modal={true}>
                  <PopoverTrigger asChild>
                    <Button 
                      variant="outline" 
                      role="combobox" 
                      type="button"
                      className="w-full justify-between"
                      data-testid="select-variant-routings"
                    >
                      {selectedRoutingIds.length > 0 
                        ? `Wybrano ${selectedRoutingIds.length} marszrut`
                        : "Wybierz marszruty..."}
                      <ChevronDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-[400px] p-0 z-[10010]" align="start" side="bottom" sideOffset={4}>
                    <Command shouldFilter={false}>
                      <CommandInput 
                        placeholder="Szukaj marszruty..." 
                        value={routingSearchValue}
                        onValueChange={setRoutingSearchValue}
                      />
                      <CommandList className="max-h-[250px] overflow-y-auto">
                        <CommandEmpty>Nie znaleziono marszrut ({routings.length} dostępnych)</CommandEmpty>
                        <CommandGroup>
                          {filteredRoutings.map((routing) => (
                            <CommandItem
                              key={routing.id}
                              value={routing.code}
                              onSelect={() => toggleRoutingSelection(routing.id)}
                              className="cursor-pointer"
                            >
                              <Checkbox 
                                checked={selectedRoutingIds.includes(routing.id)}
                                className="mr-2 pointer-events-none"
                              />
                              <span className="font-mono text-sm">{routing.code}</span>
                              <span className="ml-2 text-muted-foreground text-sm truncate">
                                {routing.name}
                              </span>
                            </CommandItem>
                          ))}
                        </CommandGroup>
                      </CommandList>
                    </Command>
                  </PopoverContent>
                </Popover>
                
                {/* Selected routings chips */}
                {selectedRoutingIds.length > 0 && (
                  <div className="flex flex-wrap gap-1 mt-2">
                    {selectedRoutingIds.map(id => {
                      const routing = routings.find(r => r.id === id);
                      return routing ? (
                        <Badge key={id} variant="secondary" className="flex items-center gap-1">
                          {routing.code}
                          <X 
                            className="w-3 h-3 cursor-pointer hover:text-destructive" 
                            onClick={() => toggleRoutingSelection(id)}
                          />
                        </Badge>
                      ) : null;
                    })}
                  </div>
                )}
                <FormDescription>
                  Ten sam typ formatki może być produkowany różnymi maszynami (innymi marszrutami)
                </FormDescription>
              </div>

              <FormField
                control={variantForm.control}
                name="variantName"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa wariantu</FormLabel>
                    <FormControl>
                      <Input 
                        placeholder={generatedVariantName || "np. Cięcie + Oklejanie + Wiercenie"} 
                        {...field}
                        value={field.value || generatedVariantName}
                        data-testid="input-variant-name" 
                      />
                    </FormControl>
                    <FormDescription>
                      {generatedVariantName && `Sugerowana nazwa: ${generatedVariantName}`}
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              
              <FormField
                control={variantForm.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis (opcjonalnie)</FormLabel>
                    <FormControl>
                      <Textarea placeholder="Opis wariantu..." {...field} value={field.value || ""} data-testid="input-variant-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={variantForm.control}
                name="defaultOperations"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Operacje (oddzielone przecinkiem)</FormLabel>
                    <FormControl>
                      <Input placeholder="cutting, edging, drilling" {...field} data-testid="input-variant-operations" />
                    </FormControl>
                    <FormDescription>
                      Np: cutting, edging, drilling, upholstering
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={variantForm.control}
                name="isActive"
                render={({ field }) => (
                  <FormItem className="flex items-center gap-2">
                    <FormControl>
                      <Checkbox 
                        checked={field.value} 
                        onCheckedChange={field.onChange}
                        data-testid="checkbox-variant-active"
                      />
                    </FormControl>
                    <FormLabel className="!mt-0">Aktywny</FormLabel>
                  </FormItem>
                )}
              />
              <DialogFooter>
                <Button type="button" variant="outline" onClick={() => setIsVariantDialogOpen(false)}>
                  Anuluj
                </Button>
                <Button type="submit" disabled={createVariantMutation.isPending || updateVariantMutation.isPending} data-testid="button-save-variant">
                  {(createVariantMutation.isPending || updateVariantMutation.isPending) && (
                    <Loader2 className="w-4 h-4 animate-spin mr-2" />
                  )}
                  {editingVariant ? "Zapisz" : "Utwórz"}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      {/* Rule Dialog */}
      <Dialog open={isRuleDialogOpen} onOpenChange={setIsRuleDialogOpen}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>{editingRule ? "Edytuj regułę" : "Nowa reguła dopasowania"}</DialogTitle>
            <DialogDescription>
              Reguła określa, które formatki będą używać tego wariantu marszruty
            </DialogDescription>
          </DialogHeader>
          <Form {...ruleForm}>
            <form onSubmit={ruleForm.handleSubmit(handleRuleSubmit)} className="space-y-4">
              {/* Formatki multi-select */}
              <div className="space-y-2">
                <FormLabel>Formatki ze słownika</FormLabel>
                <Popover open={formatkaSearchOpen} onOpenChange={setFormatkaSearchOpen} modal={true}>
                  <PopoverTrigger asChild>
                    <Button 
                      variant="outline" 
                      role="combobox" 
                      type="button"
                      className="w-full justify-between"
                      data-testid="select-rule-formatkas"
                    >
                      {selectedFormatkaCodes.length > 0 
                        ? `Wybrano ${selectedFormatkaCodes.length} formatek`
                        : "Wybierz formatki ze słownika..."}
                      <ChevronDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-[400px] p-0 z-[10010]" align="start" side="bottom" sideOffset={4}>
                    <Command shouldFilter={false}>
                      <CommandInput 
                        placeholder="Szukaj formatki..." 
                        value={formatkaSearchValue}
                        onValueChange={setFormatkaSearchValue}
                      />
                      <CommandList className="max-h-[250px] overflow-y-auto">
                        <CommandEmpty>Nie znaleziono formatek ({componentDictionaries.length} dostępnych)</CommandEmpty>
                        <CommandGroup>
                          {filteredFormatkas.map((formatka) => (
                            <CommandItem
                              key={formatka.code}
                              value={formatka.code}
                              onSelect={() => toggleFormatkaSelection(formatka.code)}
                              className="cursor-pointer"
                            >
                              <Checkbox 
                                checked={selectedFormatkaCodes.includes(formatka.code)}
                                className="mr-2 pointer-events-none"
                              />
                              <span className="font-mono text-sm">{formatka.code}</span>
                              <span className="ml-2 text-muted-foreground text-sm truncate">
                                {formatka.name !== formatka.code ? formatka.name : ""}
                              </span>
                            </CommandItem>
                          ))}
                        </CommandGroup>
                      </CommandList>
                    </Command>
                  </PopoverContent>
                </Popover>
                
                {/* Selected formatkas chips */}
                {selectedFormatkaCodes.length > 0 && (
                  <div className="flex flex-wrap gap-1 mt-2">
                    {selectedFormatkaCodes.map(code => {
                      const formatka = componentDictionaries.find(f => f.code === code);
                      return (
                        <Badge key={code} variant="secondary" className="flex items-center gap-1">
                          {formatka?.name || code}
                          <X 
                            className="w-3 h-3 cursor-pointer hover:text-destructive" 
                            onClick={() => toggleFormatkaSelection(code)}
                          />
                        </Badge>
                      );
                    })}
                  </div>
                )}
                <p className="text-sm text-muted-foreground">
                  Wybrane formatki będą automatycznie dopasowane do tej marszruty. Ma priorytet nad wzorcem SQL LIKE.
                </p>
              </div>

              <Separator />

              <FormField
                control={ruleForm.control}
                name="namePattern"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Wzorzec nazwy - alternatywny (SQL LIKE)</FormLabel>
                    <FormControl>
                      <Input placeholder="np. BOCZKI-S%, WD-%, SIEDZISKO%" {...field} value={field.value || ""} data-testid="input-rule-name-pattern" />
                    </FormControl>
                    <FormDescription>
                      Używany gdy formatka nie pasuje do wybranych kodów. Użyj % jako wildcard.
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={ruleForm.control}
                name="colorPattern"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Wzorzec koloru (opcjonalnie)</FormLabel>
                    <FormControl>
                      <Input placeholder="np. SUROWA, CZARNY" {...field} value={field.value || ""} data-testid="input-rule-color-pattern" />
                    </FormControl>
                    <FormDescription>
                      Dokładne dopasowanie koloru. Puste = wszystkie kolory.
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={ruleForm.control}
                name="priority"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Priorytet</FormLabel>
                    <FormControl>
                      <Input type="number" {...field} data-testid="input-rule-priority" />
                    </FormControl>
                    <FormDescription>
                      Wyższy priorytet = ważniejsza reguła (przy konfliktach)
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={ruleForm.control}
                name="isActive"
                render={({ field }) => (
                  <FormItem className="flex items-center gap-2">
                    <FormControl>
                      <Checkbox 
                        checked={field.value} 
                        onCheckedChange={field.onChange}
                        data-testid="checkbox-rule-active"
                      />
                    </FormControl>
                    <FormLabel className="!mt-0">Aktywna</FormLabel>
                  </FormItem>
                )}
              />
              <DialogFooter>
                <Button type="button" variant="outline" onClick={() => setIsRuleDialogOpen(false)}>
                  Anuluj
                </Button>
                <Button type="submit" disabled={createRuleMutation.isPending || updateRuleMutation.isPending} data-testid="button-save-rule">
                  {(createRuleMutation.isPending || updateRuleMutation.isPending) && (
                    <Loader2 className="w-4 h-4 animate-spin mr-2" />
                  )}
                  {editingRule ? "Zapisz" : "Utwórz"}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
