import { useQuery, useMutation } from "@tanstack/react-query";
import { useParams, useLocation } from "wouter";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { ArrowLeft, Loader2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { ProductionRoutingForm, routingFormSchema, type RoutingFormData } from "@/features/production-routings/production-routing-form";
import type { ProductionRouting, ProductionWorkCenter } from "@shared/schema";
import { useEffect } from "react";

export default function ProductionRoutingEdit() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();

  const { data: routing, isLoading } = useQuery<ProductionRouting>({
    queryKey: ["/api/production/routings", id],
    queryFn: async () => {
      const res = await fetch(`/api/production/routings/${id}`);
      if (!res.ok) throw new Error("Failed to fetch routing");
      return res.json();
    },
  });

  const { data: workCenters = [] } = useQuery<ProductionWorkCenter[]>({
    queryKey: ["/api/production/work-centers"],
  });

  const form = useForm<RoutingFormData>({
    resolver: zodResolver(routingFormSchema),
    defaultValues: {
      code: "",
      name: "",
      isActive: true,
    },
  });

  useEffect(() => {
    if (routing) {
      form.reset({
        code: routing.code,
        name: routing.name,
        description: routing.description ?? "",
        productId: routing.productId ?? null,
        productType: routing.productType ?? "",
        defaultWorkCenterId: routing.defaultWorkCenterId ?? null,
        isActive: routing.isActive ?? true,
        version: routing.version ?? 1,
        notes: routing.notes ?? "",
      });
    }
  }, [routing, form]);

  const updateMutation = useMutation({
    mutationFn: (data: RoutingFormData) => {
      const normalizedData = {
        ...data,
        defaultWorkCenterId: data.defaultWorkCenterId && data.defaultWorkCenterId !== 0 ? data.defaultWorkCenterId : null,
      };
      return apiRequest("PATCH", `/api/production/routings/${id}`, normalizedData);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings", id] });
      toast({ title: "Marszruta została zaktualizowana" });
      navigate(`/production/routings/${id}`);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować marszruty",
        variant: "destructive",
      });
    },
  });

  const handleCancel = () => {
    navigate(`/production/routings/${id}`);
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6">
        <div className="flex justify-center p-8">
          <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-routing-edit" />
        </div>
      </div>
    );
  }

  if (!routing) {
    return (
      <div className="container mx-auto p-6">
        <Card>
          <CardContent className="p-8 text-center text-muted-foreground">
            Nie znaleziono marszruty
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center gap-4">
        <Button
          variant="ghost"
          size="icon"
          onClick={handleCancel}
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4" />
        </Button>
        <div>
          <h1 className="text-3xl font-bold" data-testid="text-page-title">
            Edycja marszruty
          </h1>
          <p className="text-muted-foreground font-mono">
            {routing.code}
          </p>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Dane marszruty</CardTitle>
        </CardHeader>
        <CardContent>
          <ProductionRoutingForm
            form={form}
            onSubmit={(data) => updateMutation.mutate(data)}
            mutation={updateMutation}
            workCenters={workCenters}
            mode="edit"
            onCancel={handleCancel}
          />
        </CardContent>
      </Card>
    </div>
  );
}
