import { useMutation, useQuery } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useLocation } from "wouter";
import { ArrowLeft, CalendarDays, Clock } from "lucide-react";
import { useEffect, useMemo } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import { z } from "zod";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Badge } from "@/components/ui/badge";

const planFormSchema = z.object({
  name: z.string().min(1, "Nazwa jest wymagana"),
  description: z.string().optional(),
  plannedStartDate: z.string().optional(),
  plannedEndDate: z.string().optional(),
  status: z.enum(['draft', 'approved', 'in_progress', 'completed', 'cancelled']).optional(),
  priority: z.enum(['low', 'normal', 'high', 'urgent']).optional(),
  notes: z.string().optional(),
  autoAssignRoutings: z.boolean().optional().default(false),
});

type PlanFormValues = z.infer<typeof planFormSchema>;

function formatDateTimeLocal(date: Date): string {
  const pad = (n: number) => n.toString().padStart(2, '0');
  return `${date.getFullYear()}-${pad(date.getMonth() + 1)}-${pad(date.getDate())}T${pad(date.getHours())}:${pad(date.getMinutes())}`;
}

function addDays(date: Date, days: number): Date {
  const result = new Date(date);
  result.setDate(result.getDate() + days);
  return result;
}

function setTime(date: Date, hours: number, minutes: number = 0): Date {
  const result = new Date(date);
  result.setHours(hours, minutes, 0, 0);
  return result;
}

export default function ProductionPlanNew() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();

  const now = useMemo(() => new Date(), []);
  const defaultStartDate = formatDateTimeLocal(now);
  const defaultEndDate = formatDateTimeLocal(setTime(now, 14, 0));

  const form = useForm<PlanFormValues>({
    resolver: zodResolver(planFormSchema),
    defaultValues: {
      name: "",
      description: "",
      plannedStartDate: defaultStartDate,
      plannedEndDate: defaultEndDate,
      status: "draft",
      priority: "normal",
      notes: "",
      autoAssignRoutings: false,
    },
  });

  const { data: templates, isLoading: templatesLoading } = useQuery<any[]>({
    queryKey: ["/api/production/plan-name-templates"],
  });

  const generateNameMutation = useMutation({
    mutationFn: async (templateId: number) => {
      const response = await fetch("/api/production/plan-name-templates/generate", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({ templateId }),
      });
      if (!response.ok) {
        const error = await response.json().catch(() => ({ message: "Failed to generate name" }));
        throw new Error(error.message);
      }
      return await response.json();
    },
    onSuccess: (data) => {
      form.setValue("name", data.generatedName);
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się wygenerować nazwy",
      });
    },
  });

  useEffect(() => {
    if (templates && templates.length > 0 && !form.getValues("name")) {
      const defaultTemplate = templates.find(t => t.isDefault) || templates[0];
      if (defaultTemplate) {
        generateNameMutation.mutate(defaultTemplate.id);
      }
    }
  }, [templates]);

  const handleSelectTemplate = (templateId: string) => {
    generateNameMutation.mutate(parseInt(templateId, 10));
  };

  const createMutation = useMutation({
    mutationFn: async (data: PlanFormValues) => {
      const response = await fetch("/api/production/planning/plans", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify(data),
      });
      if (!response.ok) {
        const error = await response.json().catch(() => ({ message: "Failed to create plan" }));
        throw new Error(error.message);
      }
      return await response.json();
    },
    onSuccess: (createdPlan) => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/planning/plans"] });
      toast({
        title: "Sukces",
        description: "Plan produkcji został utworzony",
      });
      setLocation(`/production/plans/${createdPlan.id}`);
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć planu",
      });
    },
  });

  const handleSubmit = (data: PlanFormValues) => {
    createMutation.mutate(data);
  };

  const handleCancel = () => {
    setLocation("/production/plans");
  };

  const setStartDatePreset = (daysFromNow: number) => {
    const date = addDays(now, daysFromNow);
    const startDateTime = setTime(date, now.getHours(), now.getMinutes());
    form.setValue("plannedStartDate", formatDateTimeLocal(startDateTime));
  };

  const setDurationDays = (days: number) => {
    const startDateStr = form.getValues("plannedStartDate");
    if (startDateStr) {
      const startDate = new Date(startDateStr);
      const endDate = addDays(startDate, days - 1);
      const endDateTime = setTime(endDate, 14, 0);
      form.setValue("plannedEndDate", formatDateTimeLocal(endDateTime));
    }
  };

  return (
    <div className="container mx-auto p-6 max-w-4xl">
      <div className="mb-6">
        <Button
          variant="ghost"
          onClick={handleCancel}
          className="mb-4"
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Powrót do listy planów
        </Button>
        <h1 className="text-3xl font-bold" data-testid="text-page-title">Nowy Plan Produkcji</h1>
        <p className="text-muted-foreground mt-2">Utwórz nowy plan produkcji</p>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Dane Planu</CardTitle>
          <CardDescription>Wypełnij formularz aby utworzyć nowy plan produkcji</CardDescription>
        </CardHeader>
        <CardContent>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(handleSubmit)} className="space-y-4">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <div className="flex items-center justify-between">
                      <FormLabel>Nazwa planu *</FormLabel>
                      <div className="flex gap-1 flex-wrap">
                        {!templatesLoading && templates?.map((template) => (
                          <Badge
                            key={template.id}
                            variant="outline"
                            className="cursor-pointer hover-elevate text-xs"
                            onClick={() => handleSelectTemplate(String(template.id))}
                            data-testid={`badge-template-${template.id}`}
                          >
                            {template.templateName}
                          </Badge>
                        ))}
                      </div>
                    </div>
                    <FormControl>
                      <Input
                        {...field}
                        placeholder={generateNameMutation.isPending ? "Generuję..." : "Plan produkcji styczeń 2025"}
                        disabled={generateNameMutation.isPending}
                        data-testid="input-name"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} placeholder="Opcjonalny opis planu..." data-testid="input-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="space-y-3">
                <div className="flex items-center gap-2 text-sm text-muted-foreground">
                  <CalendarDays className="h-4 w-4" />
                  <span>Predefiniowane daty rozpoczęcia:</span>
                </div>
                <div className="flex gap-2 flex-wrap">
                  <Badge
                    variant="outline"
                    className="cursor-pointer hover-elevate"
                    onClick={() => setStartDatePreset(0)}
                    data-testid="badge-today"
                  >
                    Dzisiaj
                  </Badge>
                  <Badge
                    variant="outline"
                    className="cursor-pointer hover-elevate"
                    onClick={() => setStartDatePreset(1)}
                    data-testid="badge-tomorrow"
                  >
                    Jutro
                  </Badge>
                  <Badge
                    variant="outline"
                    className="cursor-pointer hover-elevate"
                    onClick={() => setStartDatePreset(2)}
                    data-testid="badge-day-after"
                  >
                    Pojutrze
                  </Badge>
                  <Badge
                    variant="outline"
                    className="cursor-pointer hover-elevate"
                    onClick={() => setStartDatePreset(3)}
                    data-testid="badge-plus-3"
                  >
                    +3 dni
                  </Badge>
                  <Badge
                    variant="outline"
                    className="cursor-pointer hover-elevate"
                    onClick={() => setStartDatePreset(7)}
                    data-testid="badge-plus-7"
                  >
                    +7 dni
                  </Badge>
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="plannedStartDate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Data i godzina rozpoczęcia</FormLabel>
                      <FormControl>
                        <Input type="datetime-local" {...field} data-testid="input-start-date" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="plannedEndDate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Data i godzina zakończenia</FormLabel>
                      <FormControl>
                        <Input type="datetime-local" {...field} data-testid="input-end-date" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="space-y-3">
                <div className="flex items-center gap-2 text-sm text-muted-foreground">
                  <Clock className="h-4 w-4" />
                  <span>Czas trwania (od daty rozpoczęcia):</span>
                </div>
                <div className="flex gap-2 flex-wrap">
                  <Badge
                    variant="outline"
                    className="cursor-pointer hover-elevate"
                    onClick={() => setDurationDays(1)}
                    data-testid="badge-duration-1"
                  >
                    1 dzień
                  </Badge>
                  <Badge
                    variant="outline"
                    className="cursor-pointer hover-elevate"
                    onClick={() => setDurationDays(2)}
                    data-testid="badge-duration-2"
                  >
                    2 dni
                  </Badge>
                  <Badge
                    variant="outline"
                    className="cursor-pointer hover-elevate"
                    onClick={() => setDurationDays(3)}
                    data-testid="badge-duration-3"
                  >
                    3 dni
                  </Badge>
                  <Badge
                    variant="outline"
                    className="cursor-pointer hover-elevate"
                    onClick={() => setDurationDays(5)}
                    data-testid="badge-duration-5"
                  >
                    5 dni
                  </Badge>
                  <Badge
                    variant="outline"
                    className="cursor-pointer hover-elevate"
                    onClick={() => setDurationDays(7)}
                    data-testid="badge-duration-7"
                  >
                    7 dni
                  </Badge>
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="status"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Status</FormLabel>
                      <Select value={field.value} onValueChange={field.onChange}>
                        <FormControl>
                          <SelectTrigger data-testid="select-status">
                            <SelectValue />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="draft">Projekt</SelectItem>
                          <SelectItem value="approved">Zatwierdzony</SelectItem>
                          <SelectItem value="in_progress">W realizacji</SelectItem>
                          <SelectItem value="completed">Zakończony</SelectItem>
                          <SelectItem value="cancelled">Anulowany</SelectItem>
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="priority"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Priorytet</FormLabel>
                      <Select value={field.value} onValueChange={field.onChange}>
                        <FormControl>
                          <SelectTrigger data-testid="select-priority">
                            <SelectValue />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="low">Niski</SelectItem>
                          <SelectItem value="normal">Normalny</SelectItem>
                          <SelectItem value="high">Wysoki</SelectItem>
                          <SelectItem value="urgent">Pilny</SelectItem>
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Uwagi</FormLabel>
                    <FormControl>
                      <Textarea {...field} placeholder="Dodatkowe uwagi..." data-testid="input-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2 pt-4">
                <Button
                  type="button"
                  variant="outline"
                  onClick={handleCancel}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button
                  type="submit"
                  disabled={createMutation.isPending}
                  data-testid="button-submit"
                >
                  {createMutation.isPending ? "Tworzenie..." : "Utwórz Plan"}
                </Button>
              </div>
            </form>
          </Form>
        </CardContent>
      </Card>
    </div>
  );
}
