import { useQuery, useMutation } from "@tanstack/react-query";
import { useRoute, useLocation } from "wouter";
import { ArrowLeft, Edit, Play, CheckCircle, XCircle, Loader2, Calendar, Package, User, MapPin, Image, Filter, Grid3x3, List, AlertTriangle, Download, FileSpreadsheet, Warehouse, Clock } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import { apiRequest } from "@/lib/queryClient";
import type { ProductionOrder } from "@shared/schema";
import { isFormatka, FORMATKI_PREFIXES } from "@shared/schema";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { useState, useMemo } from "react";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { WorkOrderFlow, WorkOrderTable } from "@/components/work-order-flow";
import { PalletManager } from "@/components/pallet-manager";
import { InsertWarehouseOperation } from "@/components/insert-warehouse-operation";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Combobox } from "@/components/ui/combobox";

const statusColors: Record<string, string> = {
  draft: "bg-gray-500",
  confirmed: "bg-blue-500",
  planned: "bg-indigo-500",
  in_progress: "bg-yellow-500",
  paused: "bg-orange-500",
  done: "bg-green-500",
  cancelled: "bg-red-500",
};

const statusLabels: Record<string, string> = {
  draft: "Projekt",
  confirmed: "Potwierdzone",
  planned: "Zaplanowane",
  in_progress: "W realizacji",
  paused: "Wstrzymane",
  done: "Zakończone",
  cancelled: "Anulowane",
};

const priorityColors: Record<string, string> = {
  low: "bg-gray-400",
  normal: "bg-blue-400",
  high: "bg-orange-400",
  urgent: "bg-red-500",
};

const priorityLabels: Record<string, string> = {
  low: "Niski",
  normal: "Normalny",
  high: "Wysoki",
  urgent: "Pilny",
};

const workflowStageColors: Record<string, string> = {
  magazine: "bg-gray-500",
  cutting: "bg-blue-500",
  edging: "bg-indigo-500",
  drilling: "bg-purple-500",
  upholstering: "bg-pink-500",
  picking: "bg-yellow-500",
  packing: "bg-orange-500",
  strapping: "bg-amber-500",
  ready: "bg-green-500",
  shipped: "bg-emerald-600",
};

const workflowStageLabels: Record<string, string> = {
  magazine: "Magazyn",
  cutting: "Cięcie",
  edging: "Oklejanie",
  drilling: "Wiercenie",
  upholstering: "Tapicerowanie",
  picking: "Kompletowanie",
  packing: "Pakowanie",
  strapping: "Bandowanie",
  ready: "Gotowe",
  shipped: "Wysłane",
};

export default function ProductionOrderDetailPage() {
  const [, params] = useRoute("/production/orders/:id");
  const [, setLocation] = useLocation();
  const { toast } = useToast();

  const orderId = params?.id ? parseInt(params.id) : null;
  
  // Formatki filters state
  const [viewMode, setViewMode] = useState<'list' | 'gallery'>('list');
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedColor, setSelectedColor] = useState<string>('all');
  const [selectedEdging, setSelectedEdging] = useState<string>('all');
  const [selectedDrilling, setSelectedDrilling] = useState<string>('all');
  const [selectedBoardType, setSelectedBoardType] = useState<string>('all');
  const [showDamagedOnly, setShowDamagedOnly] = useState(false);
  const [selectedFormatka, setSelectedFormatka] = useState<any | null>(null);
  const [damageDialogOpen, setDamageDialogOpen] = useState(false);
  const [selectedDamageItem, setSelectedDamageItem] = useState<any | null>(null);
  const [damageType, setDamageType] = useState<string>('');
  const [damageNotes, setDamageNotes] = useState<string>('');

  // Fetch order
  const { data: order, isLoading } = useQuery<ProductionOrder>({
    queryKey: ["/api/production/orders", orderId],
    queryFn: async () => {
      const response = await fetch(`/api/production/orders/${orderId}`, {
        credentials: "include",
      });
      if (!response.ok) {
        throw new Error("Failed to fetch order");
      }
      return await response.json();
    },
    enabled: !!orderId,
  });

  // Fetch dynamic formatki prefixes from dictionary
  const { data: dynamicPrefixesData } = useQuery<{ prefixes: string[] }>({
    queryKey: ["/api/dictionary-utils/formatki-prefixes"],
    staleTime: 5 * 60 * 1000, // Cache for 5 minutes
  });

  // Fetch scrap reason dictionary for damage types
  interface DictionaryItem {
    id: number;
    code: string;
    name: string | null;
    readableName: string | null;
    dictionaryType: string;
    isActive: boolean;
  }
  const { data: scrapReasons } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries", { type: "scrap_reason_cutting" }],
    queryFn: async () => {
      const response = await fetch("/api/dictionaries?type=scrap_reason_cutting", {
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to fetch scrap reasons");
      return response.json();
    },
    staleTime: 5 * 60 * 1000, // Cache for 5 minutes
  });

  // Fetch damage and warehouse summaries
  interface DamageSummaryItem {
    id: number;
    componentName: string;
    damageType: string | null;
    damageNotes: string | null;
    damagedAt: string | null;
    damagedByUsername: string | null;
    damagedByName: string | null;
  }
  interface WarehouseIntakeSummaryItem {
    documentId: number;
    docNumber: string;
    issuedAt: string;
    issuedByName: string | null;
    totalLines: number;
    totalQuantity: number;
    remarks: string | null;
    workOrderNumber: string | null;
    operationName: string | null;
    operationCode: string | null;
  }
  interface FormatkiStats {
    total: number;
    nonDamaged: number;
    inWarehouse: number;
    damaged: number;
    pending: number;
  }
  interface OrderInfo {
    status: string;
    actualEndDate: string | null;
    quantityProduced: number;
    quantityPlanned: number;
    isCompleted: boolean;
    isAutoCompleted: boolean;
  }
  interface OrderSummaries {
    formatkiStats?: FormatkiStats;
    orderInfo?: OrderInfo;
    damageSummary: DamageSummaryItem[];
    warehouseIntakeSummary: WarehouseIntakeSummaryItem[];
  }
  const { data: orderSummaries } = useQuery<OrderSummaries>({
    queryKey: ["/api/production/orders", orderId, "summaries"],
    queryFn: async () => {
      const response = await fetch(`/api/production/orders/${orderId}/summaries`, {
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to fetch summaries");
      return response.json();
    },
    enabled: !!orderId,
  });

  // Status transition mutations
  const startMutation = useMutation({
    mutationFn: async () => {
      const res = await apiRequest("POST", `/api/production/orders/${orderId}/start`);
      return await res.json();
    },
    onSuccess: () => {
      // Invalidate all queries related to orders (including filtered views)
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Zlecenie rozpoczęte",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się rozpocząć zlecenia",
      });
    },
  });

  const completeMutation = useMutation({
    mutationFn: async () => {
      const res = await apiRequest("POST", `/api/production/orders/${orderId}/complete`);
      return await res.json();
    },
    onSuccess: () => {
      // Invalidate all queries related to orders (including filtered views)
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Zlecenie zakończone",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zakończyć zlecenia",
      });
    },
  });

  const cancelMutation = useMutation({
    mutationFn: async () => {
      const res = await apiRequest("POST", `/api/production/orders/${orderId}/cancel`);
      return await res.json();
    },
    onSuccess: () => {
      // Invalidate all queries related to orders (including filtered views)
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Zlecenie anulowane",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się anulować zlecenia",
      });
    },
  });

  // Update workflow stage mutation
  const updateWorkflowStageMutation = useMutation({
    mutationFn: async (workflowStage: string) => {
      const res = await apiRequest("PATCH", `/api/production/orders/${orderId}/workflow-stage`, { workflowStage });
      return await res.json();
    },
    onSuccess: () => {
      // Invalidate all queries related to orders (including filtered views)
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Etap workflow zaktualizowany",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować etapu workflow",
      });
    },
  });

  // Update damage status mutation
  const damageMutation = useMutation({
    mutationFn: async ({ itemId, isDamaged, damageType, damageNotes }: { itemId: number; isDamaged: boolean; damageType?: string; damageNotes?: string }) => {
      const res = await apiRequest("PATCH", `/api/production/order-bom-items/${itemId}/damage`, { 
        isDamaged, 
        damageType: isDamaged ? (damageType || null) : null, 
        damageNotes: isDamaged ? (damageNotes || null) : null 
      });
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      setDamageDialogOpen(false);
      setSelectedDamageItem(null);
      setDamageType('');
      setDamageNotes('');
      toast({
        title: "Sukces",
        description: "Status uszkodzenia zaktualizowany",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować statusu",
      });
    },
  });

  const handleOpenDamageDialog = (item: any) => {
    setSelectedDamageItem(item);
    setDamageType(item.damageType || '');
    setDamageNotes(item.damageNotes || '');
    setDamageDialogOpen(true);
  };

  const handleMarkDamaged = () => {
    if (!selectedDamageItem) return;
    if (!damageType) {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: "Wybierz typ uszkodzenia",
      });
      return;
    }
    damageMutation.mutate({
      itemId: selectedDamageItem.id,
      isDamaged: true,
      damageType,
      damageNotes,
    });
  };

  const handleUnmarkDamaged = () => {
    if (!selectedDamageItem) return;
    damageMutation.mutate({
      itemId: selectedDamageItem.id,
      isDamaged: false,
    });
  };

  const formatDate = (date: Date | null | undefined) => {
    if (!date) return "-";
    return format(new Date(date), "dd.MM.yyyy HH:mm", { locale: pl });
  };

  const formatDateShort = (date: Date | null | undefined) => {
    if (!date) return "-";
    return format(new Date(date), "dd.MM.yyyy", { locale: pl });
  };

  // Separate components and formatki based on generated name prefix
  const { components: standardComponents, formatki } = useMemo(() => {
    if (!order || !(order as any).components) return { components: [], formatki: [] };
    
    const allComponents = (order as any).components;
    
    // Use dynamic prefixes from API (with fallback to static FORMATKI_PREFIXES)
    const prefixesToUse = dynamicPrefixesData?.prefixes || FORMATKI_PREFIXES;
    
    // Helper to check if component is formatka using current prefixes
    const isFormatkaDynamic = (generatedName: string | null | undefined): boolean => {
      if (!generatedName) return false;
      const upperName = generatedName.toUpperCase();
      return prefixesToUse.some(prefix => upperName.startsWith(prefix.toUpperCase()));
    };
    
    // Classify components
    const formatki = allComponents.filter((comp: any) => isFormatkaDynamic(comp.generatedName));
    const components = allComponents.filter((comp: any) => !isFormatkaDynamic(comp.generatedName));
    
    return { components, formatki };
  }, [order, dynamicPrefixesData]);
  
  // Get damaged formatki for workflow display (with full details)
  const damagedFormatki = useMemo(() => {
    return formatki
      .filter((comp: any) => comp.isDamaged)
      .map((comp: any) => ({
        id: comp.id,
        generatedName: comp.generatedName,
        damageType: comp.damageType,
        damageNotes: comp.damageNotes,
      }));
  }, [formatki]);

  // Helper to get damage type label from dictionary
  const getDamageTypeLabel = (damageTypeCode: string | null | undefined): string => {
    if (!damageTypeCode) return "Uszkodzony";
    if (scrapReasons && scrapReasons.length > 0) {
      const reason = scrapReasons.find(r => r.code === damageTypeCode);
      if (reason) return reason.readableName || reason.name || reason.code;
    }
    return damageTypeCode;
  };
  
  // Filter components
  const filteredComponents = useMemo(() => {
    return standardComponents.filter((comp: any) => {
      // Search filter
      if (searchQuery && !comp.generatedName?.toLowerCase().includes(searchQuery.toLowerCase())) {
        return false;
      }
      
      // Color filter
      if (selectedColor !== 'all' && comp.color !== selectedColor) {
        return false;
      }
      
      // Damaged only filter
      if (showDamagedOnly && !comp.isDamaged) {
        return false;
      }
      
      return true;
    });
  }, [standardComponents, searchQuery, selectedColor, showDamagedOnly]);
  
  // Filter formatki
  const filteredFormatki = useMemo(() => {
    return formatki.filter((comp: any) => {
      // Search filter
      if (searchQuery && !comp.generatedName?.toLowerCase().includes(searchQuery.toLowerCase())) {
        return false;
      }
      
      // Color filter
      if (selectedColor !== 'all' && comp.color !== selectedColor) {
        return false;
      }
      
      // Edging filter
      if (selectedEdging !== 'all' && comp.edgingPattern !== selectedEdging) {
        return false;
      }
      
      // Drilling filter
      if (selectedDrilling !== 'all' && comp.drillingPattern !== selectedDrilling) {
        return false;
      }
      
      // Board type filter
      if (selectedBoardType !== 'all' && comp.boardType !== selectedBoardType) {
        return false;
      }
      
      // Damaged only filter
      if (showDamagedOnly && !comp.isDamaged) {
        return false;
      }
      
      return true;
    });
  }, [formatki, searchQuery, selectedColor, selectedEdging, selectedDrilling, selectedBoardType, showDamagedOnly]);

  // Extract unique values for filters
  const filterOptions = useMemo(() => {
    if (!order || !(order as any).components) return { colors: [], edgingPatterns: [], drillingPatterns: [], boardTypes: [] };
    
    const components = (order as any).components;
    
    return {
      colors: Array.from(new Set(components.map((c: any) => c.color).filter(Boolean))),
      edgingPatterns: Array.from(new Set(components.map((c: any) => c.edgingPattern).filter(Boolean))),
      drillingPatterns: Array.from(new Set(components.map((c: any) => c.drillingPattern).filter(Boolean))),
      boardTypes: Array.from(new Set(components.map((c: any) => c.boardType).filter(Boolean))),
    };
  }, [order]);

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
      </div>
    );
  }

  if (!order) {
    return (
      <div className="flex flex-col items-center justify-center h-full">
        <Package className="h-16 w-16 text-muted-foreground mb-4" />
        <p className="text-muted-foreground">Nie znaleziono zlecenia produkcyjnego</p>
        <Button
          variant="outline"
          className="mt-4"
          onClick={() => setLocation("/production/orders")}
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Powrót do listy
        </Button>
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full">
      {/* Header */}
      <div className="border-b p-3 md:p-4">
        <div className="flex flex-col md:flex-row md:items-center gap-3 md:gap-4 mb-3 md:mb-4">
          <div className="flex items-center gap-3">
            <Button
              variant="ghost"
              size="icon"
              onClick={() => setLocation("/production/orders")}
              data-testid="button-back"
            >
              <ArrowLeft className="h-4 w-4" />
            </Button>
            <div className="flex-1">
              <h1 className="text-lg md:text-2xl font-bold font-mono" data-testid="text-title">
                {order.orderNumber}
              </h1>
              <p className="text-xs md:text-sm text-muted-foreground">Zlecenie Produkcyjne</p>
            </div>
          </div>
          <div className="flex flex-wrap items-center gap-2 ml-0 md:ml-auto">
            <Button
              size="sm"
              variant="outline"
              onClick={() => window.open(`/api/production/orders/${orderId}/pdf`, '_blank')}
              data-testid="button-download-pdf"
              className="text-xs md:text-sm"
            >
              <Download className="h-4 w-4 md:mr-2" />
              <span className="hidden md:inline">Karta PDF</span>
            </Button>
            <Button
              size="sm"
              variant="outline"
              onClick={() => window.open(`/api/production/orders/${orderId}/csv`, '_blank')}
              data-testid="button-download-csv"
              className="text-xs md:text-sm"
            >
              <FileSpreadsheet className="h-4 w-4 md:mr-2" />
              <span className="hidden md:inline">CSV Piła</span>
            </Button>
            <Button
              size="sm"
              variant="outline"
              onClick={() => {/* TODO: Open edit dialog */}}
              data-testid="button-edit"
              className="text-xs md:text-sm"
            >
              <Edit className="h-4 w-4 md:mr-2" />
              <span className="hidden md:inline">Edytuj</span>
            </Button>
            {(order as any).workOrders && (order as any).workOrders.length > 0 && (
              <InsertWarehouseOperation
                productionOrderId={order.id}
                workOrders={(order as any).workOrders.map((wo: any) => ({
                  id: wo.id,
                  work_order_number: wo.workOrderNumber,
                  sequence: wo.sequence,
                  status: wo.status,
                  operation_name: wo.operationName,
                  operation_code: wo.operationCode,
                  location_name: wo.locationName,
                }))}
              />
            )}
            {order.status === 'draft' || order.status === 'confirmed' || order.status === 'planned' ? (
              <Button
                size="sm"
                onClick={() => startMutation.mutate()}
                disabled={startMutation.isPending}
                data-testid="button-start"
                className="text-xs md:text-sm"
              >
                <Play className="h-4 w-4 md:mr-2" />
                <span className="hidden md:inline">Rozpocznij</span>
              </Button>
            ) : null}
            {order.status === 'in_progress' ? (
              <Button
                size="sm"
                onClick={() => completeMutation.mutate()}
                disabled={completeMutation.isPending}
                data-testid="button-complete"
                className="text-xs md:text-sm"
              >
                <CheckCircle className="h-4 w-4 md:mr-2" />
                <span className="hidden md:inline">Zakończ</span>
              </Button>
            ) : null}
            {order.status !== 'done' && order.status !== 'cancelled' ? (
              <Button
                size="sm"
                variant="destructive"
                onClick={() => cancelMutation.mutate()}
                disabled={cancelMutation.isPending}
                data-testid="button-cancel"
                className="text-xs md:text-sm"
              >
                <XCircle className="h-4 w-4 md:mr-2" />
                <span className="hidden md:inline">Anuluj</span>
              </Button>
            ) : null}
          </div>
        </div>

        {/* Status and Workflow badges */}
        <div className="flex flex-wrap items-center gap-2 md:gap-4">
          <div className="flex items-center gap-1.5 md:gap-2">
            <span className="text-xs md:text-sm text-muted-foreground">Status:</span>
            <Badge className={`text-xs ${statusColors[order.status]}`}>
              {statusLabels[order.status]}
            </Badge>
          </div>
          
          <div className="flex items-center gap-1.5 md:gap-2">
            <span className="text-xs md:text-sm text-muted-foreground">Etap:</span>
            {order.workflowStage ? (
              <select
                value={order.workflowStage}
                onChange={(e) => updateWorkflowStageMutation.mutate(e.target.value)}
                disabled={updateWorkflowStageMutation.isPending || order.status === 'done' || order.status === 'cancelled'}
                className="h-7 md:h-8 px-2 md:px-3 rounded-md border bg-background text-xs md:text-sm font-medium"
                data-testid="select-workflow-stage"
              >
                {Object.entries(workflowStageLabels).map(([value, label]) => (
                  <option key={value} value={value}>{label}</option>
                ))}
              </select>
            ) : (
              <select
                onChange={(e) => updateWorkflowStageMutation.mutate(e.target.value)}
                disabled={updateWorkflowStageMutation.isPending || order.status === 'done' || order.status === 'cancelled'}
                className="h-7 md:h-8 px-2 md:px-3 rounded-md border bg-background text-xs md:text-sm text-muted-foreground"
                data-testid="select-workflow-stage"
                value=""
              >
                <option value="">Ustaw etap...</option>
                {Object.entries(workflowStageLabels).map(([value, label]) => (
                  <option key={value} value={value}>{label}</option>
                ))}
              </select>
            )}
          </div>

          <div className="flex items-center gap-1.5 md:gap-2">
            <span className="text-xs md:text-sm text-muted-foreground">Priorytet:</span>
            <Badge variant="outline" className={`text-xs ${priorityColors[order.priority]}`}>
              {priorityLabels[order.priority]}
            </Badge>
          </div>
        </div>

      </div>

      {/* Content */}
      <div className="flex-1 overflow-auto p-4">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
          {/* === SEKCJA 1: INFORMACJE PODSTAWOWE === */}
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-2">
                <Package className="h-4 w-4" />
                Informacje Podstawowe
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="grid grid-cols-2 gap-3">
                <div>
                  <p className="text-xs text-muted-foreground">Numer ZLP</p>
                  <p className="font-mono font-bold text-lg">{order.orderNumber}</p>
                </div>
                <div>
                  <p className="text-xs text-muted-foreground">Produkt</p>
                  <p className="font-semibold">{(order as any).productName || `ID: ${order.productId}`}</p>
                </div>
              </div>
              <div className="grid grid-cols-3 gap-3">
                <div>
                  <p className="text-xs text-muted-foreground">Ilość</p>
                  <p className="font-semibold">{order.quantityPlanned} {order.unitOfMeasure}</p>
                </div>
                {order.sourceOrderNumber && (
                  <div>
                    <p className="text-xs text-muted-foreground">Zamówienie źródłowe</p>
                    <p className="font-mono text-sm">{order.sourceOrderNumber}</p>
                  </div>
                )}
                {(order.locationId || (order as any).locationName) && (
                  <div>
                    <p className="text-xs text-muted-foreground">Lokalizacja</p>
                    <p className="text-sm flex items-center gap-1">
                      <MapPin className="h-3 w-3" />
                      {(order as any).locationName || `ID: ${order.locationId}`}
                    </p>
                  </div>
                )}
              </div>
              {((order.routingId || (order as any).routingName) || order.bomId) && (
                <div className="grid grid-cols-2 gap-3 pt-2 border-t">
                  {(order.routingId || (order as any).routingName) && (
                    <div>
                      <p className="text-xs text-muted-foreground">Marszruta</p>
                      <p className="text-sm">{(order as any).routingName || `ID: ${order.routingId}`}</p>
                    </div>
                  )}
                  {order.bomId && (
                    <div>
                      <p className="text-xs text-muted-foreground">BOM</p>
                      <p className="text-sm">ID: {order.bomId}</p>
                    </div>
                  )}
                </div>
              )}
            </CardContent>
          </Card>

          {/* === SEKCJA 2: HARMONOGRAM === */}
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-2">
                <Calendar className="h-4 w-4" />
                Harmonogram
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="grid grid-cols-2 gap-3">
                <div>
                  <p className="text-xs text-muted-foreground">Plan: Start</p>
                  <p className="font-semibold">{formatDateShort(order.plannedStartDate)}</p>
                </div>
                <div>
                  <p className="text-xs text-muted-foreground">Plan: Koniec</p>
                  <p className="font-semibold">{formatDateShort(order.plannedEndDate)}</p>
                </div>
              </div>
              {(order.actualStartDate || order.actualEndDate) && (
                <div className="grid grid-cols-2 gap-3 pt-2 border-t">
                  <div>
                    <p className="text-xs text-muted-foreground">Rzeczywisty Start</p>
                    <p className="font-semibold text-green-600 dark:text-green-400">{formatDate(order.actualStartDate)}</p>
                  </div>
                  <div>
                    <p className="text-xs text-muted-foreground">Rzeczywisty Koniec</p>
                    <p className="font-semibold text-green-600 dark:text-green-400">{formatDate(order.actualEndDate)}</p>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>

          {/* === SEKCJA 3: PROCES PRODUKCJI === */}
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-2">
                <Play className="h-4 w-4" />
                Proces Produkcji
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="grid grid-cols-3 gap-3">
                <div>
                  <p className="text-xs text-muted-foreground">Status</p>
                  <Badge className={`text-xs mt-1 ${statusColors[order.status]}`}>
                    {statusLabels[order.status]}
                  </Badge>
                </div>
                <div>
                  <p className="text-xs text-muted-foreground">Etap Workflow</p>
                  {order.workflowStage ? (
                    <Badge variant="outline" className={`text-xs mt-1 ${workflowStageColors[order.workflowStage]}`}>
                      {workflowStageLabels[order.workflowStage]}
                    </Badge>
                  ) : (
                    <span className="text-xs text-muted-foreground">-</span>
                  )}
                </div>
                <div>
                  <p className="text-xs text-muted-foreground">Priorytet</p>
                  <Badge variant="outline" className={`text-xs mt-1 ${priorityColors[order.priority]}`}>
                    {priorityLabels[order.priority]}
                  </Badge>
                </div>
              </div>
              {(order as any).workOrders && (order as any).workOrders.length > 0 && (
                <div className="pt-2 border-t">
                  <p className="text-xs text-muted-foreground mb-2">Operacje ({(order as any).workOrders.length})</p>
                  <div className="flex gap-1 flex-wrap">
                    {(order as any).workOrders.slice(0, 8).map((wo: any) => (
                      <Badge 
                        key={wo.id} 
                        variant="outline" 
                        className={`text-xs ${
                          wo.status === 'done' ? 'bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300' :
                          wo.status === 'in_progress' ? 'bg-yellow-100 dark:bg-yellow-900/30 text-yellow-700 dark:text-yellow-300' :
                          'bg-gray-100 dark:bg-gray-800 text-gray-600 dark:text-gray-400'
                        }`}
                      >
                        {wo.operationName || wo.operationCode || `#${wo.sequence}`}
                      </Badge>
                    ))}
                    {(order as any).workOrders.length > 8 && (
                      <Badge variant="outline" className="text-xs">+{(order as any).workOrders.length - 8}</Badge>
                    )}
                  </div>
                </div>
              )}
            </CardContent>
          </Card>

          {/* === SEKCJA 4: ZAKOŃCZENIE I JAKOŚĆ === */}
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-2">
                <CheckCircle className="h-4 w-4" />
                Zakończenie i Jakość
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              {/* Status zakończenia ZLP */}
              {orderSummaries?.orderInfo && (
                <div className={`p-3 rounded-md ${orderSummaries.orderInfo.isCompleted ? 'bg-green-50 dark:bg-green-900/20' : 'bg-blue-50 dark:bg-blue-900/20'}`}>
                  <div className="flex items-center gap-2 mb-2">
                    {orderSummaries.orderInfo.isCompleted ? (
                      <CheckCircle className="h-5 w-5 text-green-600" />
                    ) : (
                      <Clock className="h-5 w-5 text-blue-600" />
                    )}
                    <span className={`text-sm font-semibold ${orderSummaries.orderInfo.isCompleted ? 'text-green-700 dark:text-green-300' : 'text-blue-700 dark:text-blue-300'}`}>
                      {orderSummaries.orderInfo.isCompleted 
                        ? (orderSummaries.orderInfo.isAutoCompleted ? 'ZLP Zakończone (auto)' : 'ZLP Zakończone')
                        : 'ZLP W trakcie realizacji'}
                    </span>
                  </div>
                  {orderSummaries.orderInfo.isCompleted && orderSummaries.orderInfo.actualEndDate && (
                    <p className="text-xs text-muted-foreground">
                      Zakończono: {format(new Date(orderSummaries.orderInfo.actualEndDate), "dd.MM.yyyy HH:mm", { locale: pl })}
                    </p>
                  )}
                </div>
              )}

              {/* Statystyki formatek */}
              {orderSummaries?.formatkiStats && (
                <div className="pt-2 border-t">
                  <p className="text-xs font-semibold text-muted-foreground mb-2">Status formatek</p>
                  <div className="grid grid-cols-4 gap-2 text-center">
                    <div className="bg-gray-100 dark:bg-gray-800 rounded p-2">
                      <p className="text-lg font-bold">{orderSummaries.formatkiStats.total}</p>
                      <p className="text-xs text-muted-foreground">Razem</p>
                    </div>
                    <div className="bg-green-100 dark:bg-green-900/30 rounded p-2">
                      <p className="text-lg font-bold text-green-700 dark:text-green-300">{orderSummaries.formatkiStats.inWarehouse}</p>
                      <p className="text-xs text-green-600 dark:text-green-400">Magazyn</p>
                    </div>
                    <div className="bg-yellow-100 dark:bg-yellow-900/30 rounded p-2">
                      <p className="text-lg font-bold text-yellow-700 dark:text-yellow-300">{orderSummaries.formatkiStats.pending}</p>
                      <p className="text-xs text-yellow-600 dark:text-yellow-400">Oczekuje</p>
                    </div>
                    <div className="bg-red-100 dark:bg-red-900/30 rounded p-2">
                      <p className="text-lg font-bold text-red-700 dark:text-red-300">{orderSummaries.formatkiStats.damaged}</p>
                      <p className="text-xs text-red-600 dark:text-red-400">Uszkodzone</p>
                    </div>
                  </div>
                  {/* Progress bar */}
                  {orderSummaries.formatkiStats.total > 0 && (
                    <div className="mt-2">
                      <div className="h-2 bg-gray-200 dark:bg-gray-700 rounded-full overflow-hidden">
                        <div 
                          className="h-full bg-green-500 transition-all duration-300"
                          style={{ width: `${orderSummaries.formatkiStats.nonDamaged > 0 ? Math.round((orderSummaries.formatkiStats.inWarehouse / orderSummaries.formatkiStats.nonDamaged) * 100) : 0}%` }}
                        />
                      </div>
                      <p className="text-xs text-muted-foreground mt-1 text-center">
                        {orderSummaries.formatkiStats.nonDamaged > 0 ? Math.round((orderSummaries.formatkiStats.inWarehouse / orderSummaries.formatkiStats.nonDamaged) * 100) : 0}% formatek na magazynie
                      </p>
                    </div>
                  )}
                </div>
              )}

              {/* Damage Summary - Detailed */}
              {(orderSummaries?.damageSummary?.length ?? 0) > 0 && orderSummaries?.damageSummary && (
                <div className="pt-2 border-t">
                  <div className="flex items-center gap-2 mb-2">
                    <AlertTriangle className="h-4 w-4 text-red-500" />
                    <span className="text-xs font-semibold text-red-700 dark:text-red-400">
                      Uszkodzenia ({orderSummaries.damageSummary.length})
                    </span>
                  </div>
                  <div className="space-y-1.5 max-h-32 overflow-y-auto bg-red-50 dark:bg-red-900/20 rounded-md p-2">
                    {orderSummaries.damageSummary.map((item) => (
                      <div key={item.id} className="text-xs bg-white dark:bg-red-900/30 rounded px-2 py-1">
                        <div className="font-medium text-red-700 dark:text-red-300">{item.componentName}</div>
                        <div className="flex flex-wrap gap-x-3 text-muted-foreground">
                          {item.damageType && (
                            <span>Typ: {scrapReasons?.find(r => r.code === item.damageType)?.readableName || item.damageType}</span>
                          )}
                          {item.damagedAt && (
                            <span className="flex items-center gap-0.5">
                              <Clock className="h-3 w-3" />
                              {format(new Date(item.damagedAt), "dd.MM HH:mm", { locale: pl })}
                            </span>
                          )}
                          {(item.damagedByName || item.damagedByUsername) && (
                            <span className="flex items-center gap-0.5">
                              <User className="h-3 w-3" />
                              {item.damagedByName || item.damagedByUsername}
                            </span>
                          )}
                        </div>
                        {item.damageNotes && (
                          <div className="text-muted-foreground italic mt-0.5 truncate" title={item.damageNotes}>
                            {item.damageNotes}
                          </div>
                        )}
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {/* Warehouse Intake Summary - Detailed with links */}
              {(orderSummaries?.warehouseIntakeSummary?.length ?? 0) > 0 && orderSummaries?.warehouseIntakeSummary && (
                <div className="pt-2 border-t">
                  <div className="flex items-center gap-2 mb-2">
                    <Warehouse className="h-4 w-4 text-green-600" />
                    <span className="text-xs font-semibold text-green-700 dark:text-green-400">
                      Dokumenty magazynowe ({orderSummaries.warehouseIntakeSummary.length} dok. / {orderSummaries.warehouseIntakeSummary.reduce((sum, d) => sum + d.totalLines, 0)} szt.)
                    </span>
                  </div>
                  <div className="space-y-1.5 max-h-40 overflow-y-auto bg-green-50 dark:bg-green-900/20 rounded-md p-2">
                    {orderSummaries.warehouseIntakeSummary.map((doc) => (
                      <div key={doc.documentId} className="text-xs bg-white dark:bg-green-900/30 rounded px-2 py-1.5">
                        <div className="flex items-center justify-between">
                          <a 
                            href={`/warehouse/documents/${doc.documentId}`}
                            className="font-medium text-green-700 dark:text-green-300 font-mono hover:underline cursor-pointer"
                            onClick={(e) => {
                              e.preventDefault();
                              setLocation(`/warehouse/documents/${doc.documentId}`);
                            }}
                            data-testid={`link-warehouse-doc-${doc.documentId}`}
                          >
                            {doc.docNumber}
                          </a>
                          <Badge variant="outline" className="text-xs bg-green-100 dark:bg-green-800/50">
                            {doc.totalLines} szt.
                          </Badge>
                        </div>
                        <div className="flex flex-wrap gap-x-3 text-muted-foreground mt-1">
                          {doc.operationName && (
                            <span className="flex items-center gap-0.5">
                              Op: {doc.operationName}
                            </span>
                          )}
                          <span className="flex items-center gap-0.5">
                            <Clock className="h-3 w-3" />
                            {format(new Date(doc.issuedAt), "dd.MM HH:mm", { locale: pl })}
                          </span>
                          {doc.issuedByName && (
                            <span className="flex items-center gap-0.5">
                              <User className="h-3 w-3" />
                              {doc.issuedByName}
                            </span>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {/* Brak danych info */}
              {(!orderSummaries?.formatkiStats || orderSummaries.formatkiStats.total === 0) && 
               (!orderSummaries?.damageSummary || orderSummaries.damageSummary.length === 0) &&
               (!orderSummaries?.warehouseIntakeSummary || orderSummaries.warehouseIntakeSummary.length === 0) && (
                <div className="text-center text-muted-foreground text-sm py-4">
                  Brak danych do wyświetlenia
                </div>
              )}
            </CardContent>
          </Card>

          {/* === SEKCJA 5: CZASY SYSTEMOWE === */}
          <Card className="lg:col-span-2">
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-2">
                <Clock className="h-4 w-4" />
                Czasy Systemowe
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <div>
                  <p className="text-xs text-muted-foreground">Utworzone</p>
                  <p className="text-sm">{formatDate(order.createdAt)}</p>
                </div>
                <div>
                  <p className="text-xs text-muted-foreground">Ostatnia aktualizacja</p>
                  <p className="text-sm">{formatDate(order.updatedAt)}</p>
                </div>
                {order.responsibleUserId && (
                  <div>
                    <p className="text-xs text-muted-foreground">Odpowiedzialny</p>
                    <p className="text-sm flex items-center gap-1">
                      <User className="h-3 w-3" />
                      ID: {order.responsibleUserId}
                    </p>
                  </div>
                )}
                {order.notes && (
                  <div className="md:col-span-1">
                    <p className="text-xs text-muted-foreground">Uwagi</p>
                    <p className="text-sm truncate" title={order.notes}>{order.notes}</p>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Work Orders / Operations - Visual Flow */}
          {(order as any).workOrders && (order as any).workOrders.length > 0 && (
            <WorkOrderFlow 
              workOrders={(order as any).workOrders}
              productionOrderId={order.id}
              orderStatus={order.status}
              damagedFormatki={damagedFormatki}
              sourcePlanId={(order as any).sourcePlanId}
            />
          )}

          {/* Work Orders - Detailed Table */}
          {(order as any).workOrders && (order as any).workOrders.length > 0 && (
            <WorkOrderTable 
              workOrders={(order as any).workOrders}
              productionOrderId={order.id}
              orderStatus={order.status}
              damagedFormatki={damagedFormatki}
            />
          )}

          {/* Pallet Manager - Zarządzanie paletami */}
          <PalletManager 
            productionOrderId={order.id}
            colorCode={order.colorCode}
          />

          {/* Komponenty i Formatki */}
          {(order as any).components && (order as any).components.length > 0 && (
            <Card className="lg:col-span-2">
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle className="flex items-center gap-2">
                    <Package className="h-5 w-5" />
                    Komponenty i Formatki ({(order as any).components.length})
                  </CardTitle>
                  <div className="flex items-center gap-2">
                    <Button
                      size="sm"
                      variant={viewMode === 'list' ? 'default' : 'outline'}
                      onClick={() => setViewMode('list')}
                      data-testid="button-view-list"
                    >
                      <List className="h-4 w-4" />
                    </Button>
                    <Button
                      size="sm"
                      variant={viewMode === 'gallery' ? 'default' : 'outline'}
                      onClick={() => setViewMode('gallery')}
                      data-testid="button-view-gallery"
                    >
                      <Grid3x3 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>

                {/* Filters */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-3 mt-4">
                  <Input
                    placeholder="Szukaj komponentu..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    data-testid="input-search-component"
                  />
                  
                  <Select value={selectedColor} onValueChange={setSelectedColor}>
                    <SelectTrigger data-testid="select-color-filter">
                      <SelectValue placeholder="Wszystkie kolory" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Wszystkie kolory</SelectItem>
                      {(filterOptions.colors as string[]).map((color: string) => (
                        <SelectItem key={color} value={color}>{color}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  
                  <Select value={selectedEdging} onValueChange={setSelectedEdging}>
                    <SelectTrigger data-testid="select-edging-filter">
                      <SelectValue placeholder="Wszystkie oklejania" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Wszystkie oklejania</SelectItem>
                      {(filterOptions.edgingPatterns as string[]).map((pattern: string) => (
                        <SelectItem key={pattern} value={pattern}>{pattern}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  
                  <Select value={selectedDrilling} onValueChange={setSelectedDrilling}>
                    <SelectTrigger data-testid="select-drilling-filter">
                      <SelectValue placeholder="Wszystkie wiercenia" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Wszystkie wiercenia</SelectItem>
                      {(filterOptions.drillingPatterns as string[]).map((pattern: string) => (
                        <SelectItem key={pattern} value={pattern}>{pattern}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </CardHeader>

              <CardContent>
                <Tabs defaultValue="formatki" className="w-full">
                  <TabsList className="grid w-full grid-cols-2 bg-purple-600/20">
                    <TabsTrigger value="komponenty" data-testid="tab-komponenty" className="data-[state=active]:bg-purple-600 data-[state=active]:text-white">
                      Komponenty ({filteredComponents.length})
                    </TabsTrigger>
                    <TabsTrigger value="formatki" data-testid="tab-formatki" className="data-[state=active]:bg-purple-600 data-[state=active]:text-white">
                      Formatki ({filteredFormatki.length})
                    </TabsTrigger>
                  </TabsList>
                  
                  <TabsContent value="komponenty" className="mt-4">
                    {viewMode === 'list' ? (
                      <div>
                        {filteredComponents.map((comp: any) => (
                          <div
                            key={comp.id}
                            className="flex items-center gap-2 px-2 border-b hover-elevate cursor-pointer h-[30px]"
                            onClick={() => setSelectedFormatka(comp)}
                            data-testid={`component-item-${comp.id}`}
                          >
                            {comp.visualizationUrl ? (
                              <img
                                src={comp.visualizationUrl}
                                alt={comp.generatedName}
                                className="w-5 h-5 object-contain shrink-0"
                              />
                            ) : (
                              <div className="w-5 h-5 bg-muted flex items-center justify-center shrink-0">
                                <Image className="h-3 w-3 text-muted-foreground" />
                              </div>
                            )}
                            <div className="flex-1 min-w-0 flex items-center gap-3">
                              <p className="font-medium truncate text-xs leading-none">{comp.generatedName}</p>
                              {comp.isDamaged && (
                                <Badge variant="destructive" className="shrink-0 text-[10px] h-4 px-1 leading-none">
                                  <AlertTriangle className="h-2 w-2 mr-0.5" />
                                  {getDamageTypeLabel(comp.damageType)}
                                </Badge>
                              )}
                              {comp.color && <span className="text-xs text-muted-foreground shrink-0 leading-none">{comp.color}</span>}
                              {comp.calculatedLength && comp.calculatedWidth && (
                                <span className="text-xs text-muted-foreground shrink-0 leading-none">{comp.calculatedLength}×{comp.calculatedWidth}</span>
                              )}
                            </div>
                            <Button
                              size="icon"
                              variant="ghost"
                              className="h-6 w-6 shrink-0"
                              onClick={(e) => {
                                e.stopPropagation();
                                handleOpenDamageDialog(comp);
                              }}
                              data-testid={`button-damage-${comp.id}`}
                            >
                              {damageMutation.isPending && selectedDamageItem?.id === comp.id ? (
                                <Loader2 className="h-3 w-3 animate-spin" />
                              ) : (
                                <AlertTriangle className={`h-3 w-3 ${comp.isDamaged ? 'text-destructive' : 'text-muted-foreground'}`} />
                              )}
                            </Button>
                          </div>
                        ))}
                      </div>
                    ) : (
                      <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                        {filteredComponents.map((comp: any) => (
                          <div
                            key={comp.id}
                            className="border rounded-md p-3 hover-elevate cursor-pointer"
                            onClick={() => setSelectedFormatka(comp)}
                            data-testid={`component-card-${comp.id}`}
                          >
                            {comp.visualizationUrl ? (
                              <img
                                src={comp.visualizationUrl}
                                alt={comp.generatedName}
                                className="w-full aspect-square object-contain border rounded mb-2"
                              />
                            ) : (
                              <div className="w-full aspect-square bg-muted flex items-center justify-center border rounded mb-2">
                                <Image className="h-12 w-12 text-muted-foreground" />
                              </div>
                            )}
                            <p className="text-sm font-medium truncate">{comp.generatedName}</p>
                            {comp.isDamaged && (
                              <Badge variant="destructive" className="mt-2">{getDamageTypeLabel(comp.damageType)}</Badge>
                            )}
                          </div>
                        ))}
                      </div>
                    )}
                    {filteredComponents.length === 0 && (
                      <div className="text-center py-8 text-muted-foreground">
                        Brak komponentów spełniających kryteria filtrowania
                      </div>
                    )}
                  </TabsContent>
                  
                  <TabsContent value="formatki" className="mt-4">
                    {viewMode === 'list' ? (
                      <div>
                        {filteredFormatki.map((comp: any) => (
                          <div
                            key={comp.id}
                            className="flex items-center gap-2 px-2 border-b hover-elevate cursor-pointer h-[30px]"
                            onClick={() => setSelectedFormatka(comp)}
                            data-testid={`formatka-item-${comp.id}`}
                          >
                            {comp.visualizationUrl ? (
                              <img
                                src={comp.visualizationUrl}
                                alt={comp.generatedName}
                                className="w-5 h-5 object-contain shrink-0"
                              />
                            ) : (
                              <div className="w-5 h-5 bg-muted flex items-center justify-center shrink-0">
                                <Image className="h-3 w-3 text-muted-foreground" />
                              </div>
                            )}
                            <div className="flex-1 min-w-0 flex items-center gap-3">
                              <p className="font-medium truncate text-xs leading-none">{comp.generatedName}</p>
                              {comp.isDamaged && (
                                <Badge variant="destructive" className="shrink-0 text-[10px] h-4 px-1 leading-none">
                                  <AlertTriangle className="h-2 w-2 mr-0.5" />
                                  {getDamageTypeLabel(comp.damageType)}
                                </Badge>
                              )}
                              {comp.color && <span className="text-xs text-muted-foreground shrink-0 leading-none">{comp.color}</span>}
                              {comp.edgingPattern && <span className="text-xs text-muted-foreground shrink-0 leading-none">Okl: {comp.edgingPattern}</span>}
                              {comp.drillingPattern && <span className="text-xs text-muted-foreground shrink-0 leading-none">Wierc: {comp.drillingPattern}</span>}
                              {comp.calculatedLength && comp.calculatedWidth && (
                                <span className="text-xs text-muted-foreground shrink-0 leading-none">{comp.calculatedLength}×{comp.calculatedWidth}</span>
                              )}
                            </div>
                            <Button
                              size="icon"
                              variant="ghost"
                              className="h-6 w-6 shrink-0"
                              onClick={(e) => {
                                e.stopPropagation();
                                handleOpenDamageDialog(comp);
                              }}
                              data-testid={`button-damage-${comp.id}`}
                            >
                              {damageMutation.isPending && selectedDamageItem?.id === comp.id ? (
                                <Loader2 className="h-3 w-3 animate-spin" />
                              ) : (
                                <AlertTriangle className={`h-3 w-3 ${comp.isDamaged ? 'text-destructive' : 'text-muted-foreground'}`} />
                              )}
                            </Button>
                          </div>
                        ))}
                      </div>
                    ) : (
                      <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                        {filteredFormatki.map((comp: any) => (
                          <div
                            key={comp.id}
                            className="border rounded-md p-3 hover-elevate cursor-pointer"
                            onClick={() => setSelectedFormatka(comp)}
                            data-testid={`formatka-card-${comp.id}`}
                          >
                            {comp.visualizationUrl ? (
                              <img
                                src={comp.visualizationUrl}
                                alt={comp.generatedName}
                                className="w-full aspect-square object-contain border rounded mb-2"
                              />
                            ) : (
                              <div className="w-full aspect-square bg-muted flex items-center justify-center border rounded mb-2">
                                <Image className="h-12 w-12 text-muted-foreground" />
                              </div>
                            )}
                            <p className="text-sm font-medium truncate">{comp.generatedName}</p>
                            {comp.isDamaged && (
                              <Badge variant="destructive" className="mt-2">{getDamageTypeLabel(comp.damageType)}</Badge>
                            )}
                          </div>
                        ))}
                      </div>
                    )}
                    {filteredFormatki.length === 0 && (
                      <div className="text-center py-8 text-muted-foreground">
                        Brak formatek spełniających kryteria filtrowania
                      </div>
                    )}
                  </TabsContent>
                </Tabs>
              </CardContent>
            </Card>
          )}
        </div>
      </div>

      {/* Damage Dialog */}
      <Dialog 
        open={damageDialogOpen} 
        onOpenChange={(open) => {
          if (!open && damageMutation.isPending) return;
          setDamageDialogOpen(open);
        }}
      >
        <DialogContent 
          className={`max-w-md ${damageMutation.isPending ? '[&>button]:pointer-events-none [&>button]:opacity-50' : ''}`}
          onPointerDownOutside={(e) => {
            if (damageMutation.isPending) e.preventDefault();
          }}
          onEscapeKeyDown={(e) => {
            if (damageMutation.isPending) e.preventDefault();
          }}
          onInteractOutside={(e) => {
            if (damageMutation.isPending) e.preventDefault();
          }}
        >
          <DialogHeader>
            <DialogTitle>Zarządzaj uszkodzeniem</DialogTitle>
            <DialogDescription>
              {selectedDamageItem?.generatedName || 'Komponent'}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="damage-type">Typ uszkodzenia</Label>
              <Combobox
                value={damageType}
                onChange={setDamageType}
                placeholder="Wybierz typ uszkodzenia"
                searchPlaceholder="Szukaj typu uszkodzenia..."
                emptyText="Nie znaleziono"
                className="w-full"
                options={
                  scrapReasons && scrapReasons.length > 0
                    ? scrapReasons
                        .filter(reason => reason.isActive)
                        .map(reason => ({
                          value: reason.code,
                          label: reason.readableName || reason.name || reason.code
                        }))
                    : [
                        { value: "SIZE_ERROR", label: "Błąd wymiarów" },
                        { value: "COLOR_ERROR", label: "Błąd koloru" },
                        { value: "DRILLING_ERROR", label: "Błąd wiercenia" },
                        { value: "EDGING_ERROR", label: "Błąd oklejania" },
                        { value: "UPHOLSTERY_ERROR", label: "Błąd tapicerki" },
                        { value: "OTHER", label: "Inne" }
                      ]
                }
              />
            </div>
            <div>
              <Label htmlFor="damage-notes">Notatki</Label>
              <Textarea
                id="damage-notes"
                data-testid="input-damage-notes"
                placeholder="Opisz uszkodzenie..."
                value={damageNotes}
                onChange={(e) => setDamageNotes(e.target.value)}
                rows={3}
              />
            </div>
          </div>
          <DialogFooter className="gap-2">
            {selectedDamageItem?.isDamaged && (
              <Button
                data-testid="button-unmark-damaged"
                variant="outline"
                onClick={handleUnmarkDamaged}
                disabled={damageMutation.isPending}
              >
                {damageMutation.isPending ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : null}
                Usuń oznaczenie
              </Button>
            )}
            <Button
              data-testid="button-mark-damaged"
              onClick={handleMarkDamaged}
              disabled={damageMutation.isPending || !damageType}
            >
              {damageMutation.isPending ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : null}
              Oznacz jako uszkodzony
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Formatka Detail Modal */}
      <Dialog open={!!selectedFormatka} onOpenChange={(open) => !open && setSelectedFormatka(null)}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle>{selectedFormatka?.generatedName}</DialogTitle>
            <DialogDescription>
              Szczegóły komponentu
            </DialogDescription>
          </DialogHeader>
          {selectedFormatka && (
            <div className="space-y-4">
              {selectedFormatka.visualizationUrl && (
                <img
                  src={selectedFormatka.visualizationUrl}
                  alt={selectedFormatka.generatedName}
                  className="w-full max-h-96 object-contain border rounded"
                />
              )}
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <p className="text-muted-foreground">Typ</p>
                  <p className="font-medium">{selectedFormatka.componentType}</p>
                </div>
                {selectedFormatka.color && (
                  <div>
                    <p className="text-muted-foreground">Kolor</p>
                    <p className="font-medium">{selectedFormatka.color}</p>
                  </div>
                )}
                {selectedFormatka.boardType && (
                  <div>
                    <p className="text-muted-foreground">Typ płyty</p>
                    <p className="font-medium">{selectedFormatka.boardType}</p>
                  </div>
                )}
                {selectedFormatka.edgingPattern && (
                  <div>
                    <p className="text-muted-foreground">Oklejanie</p>
                    <p className="font-medium">{selectedFormatka.edgingPattern}</p>
                  </div>
                )}
                {selectedFormatka.drillingPattern && (
                  <div>
                    <p className="text-muted-foreground">Wiercenie</p>
                    <p className="font-medium">{selectedFormatka.drillingPattern}</p>
                  </div>
                )}
                {selectedFormatka.calculatedLength && selectedFormatka.calculatedWidth && (
                  <div>
                    <p className="text-muted-foreground">Wymiary</p>
                    <p className="font-medium">{selectedFormatka.calculatedLength} × {selectedFormatka.calculatedWidth} mm</p>
                  </div>
                )}
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
