import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Plus, Edit, Trash2, Loader2, User, Building2, BadgeCheck, Star } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Switch } from "@/components/ui/switch";
import { Checkbox } from "@/components/ui/checkbox";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { z } from "zod";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";

interface OperatorWorkCenter {
  id: number;
  code: string;
  name: string;
  isPrimary: boolean;
  proficiencyLevel: string;
}

interface ProductionOperator {
  id: number;
  full_name: string;
  short_code: string | null;
  badge_id: string | null;
  user_id: number | null;
  work_center_id: number | null;
  is_active: boolean;
  notes: string | null;
  created_at: string;
  updated_at: string;
  work_centers: OperatorWorkCenter[];
}

interface ProductionWorkCenter {
  id: number;
  code: string;
  name: string;
}

const operatorFormSchema = z.object({
  fullName: z.string().min(2, "Imię i nazwisko musi mieć minimum 2 znaki"),
  shortCode: z.string().max(20).optional().nullable(),
  badgeId: z.string().max(100).optional().nullable(),
  isActive: z.boolean().default(true),
  notes: z.string().optional().nullable(),
});

type OperatorFormData = z.infer<typeof operatorFormSchema>;

export default function ProductionOperators() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingOperator, setEditingOperator] = useState<ProductionOperator | null>(null);
  const [deleteOperatorId, setDeleteOperatorId] = useState<number | null>(null);
  const [search, setSearch] = useState("");
  const [filterActive, setFilterActive] = useState<string>("all");
  const [selectedWorkCenterIds, setSelectedWorkCenterIds] = useState<number[]>([]);
  const [primaryWorkCenterId, setPrimaryWorkCenterId] = useState<number | null>(null);
  const { toast } = useToast();

  const form = useForm<OperatorFormData>({
    resolver: zodResolver(operatorFormSchema),
    defaultValues: {
      fullName: "",
      shortCode: "",
      badgeId: "",
      isActive: true,
      notes: "",
    },
  });

  const { data: operatorsData, isLoading } = useQuery<{
    operators: ProductionOperator[];
    total: number;
  }>({
    queryKey: ["/api/production/operators", { search, is_active: filterActive !== "all" ? filterActive : undefined }],
  });

  const { data: workCenters = [] } = useQuery<ProductionWorkCenter[]>({
    queryKey: ["/api/production/work-centers"],
  });

  const operators = operatorsData?.operators || [];

  const createMutation = useMutation({
    mutationFn: async (data: OperatorFormData) => {
      const response = await apiRequest("POST", "/api/production/operators", data);
      const newOperator = await response.json();
      
      // Zapisz gniazda robocze jeśli są wybrane
      if (selectedWorkCenterIds.length > 0) {
        await apiRequest("PUT", `/api/production/operators/${newOperator.id}/work-centers`, {
          workCenterIds: selectedWorkCenterIds,
          primaryWorkCenterId: primaryWorkCenterId,
        });
      }
      
      return newOperator;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/operators"] });
      setIsDialogOpen(false);
      form.reset();
      setSelectedWorkCenterIds([]);
      setPrimaryWorkCenterId(null);
      toast({ title: "Operator został utworzony" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć operatora",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async (data: OperatorFormData & { id: number }) => {
      await apiRequest("PATCH", `/api/production/operators/${data.id}`, data);
      
      // Zapisz gniazda robocze
      await apiRequest("PUT", `/api/production/operators/${data.id}/work-centers`, {
        workCenterIds: selectedWorkCenterIds,
        primaryWorkCenterId: primaryWorkCenterId,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/operators"] });
      setIsDialogOpen(false);
      setEditingOperator(null);
      form.reset();
      setSelectedWorkCenterIds([]);
      setPrimaryWorkCenterId(null);
      toast({ title: "Operator został zaktualizowany" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować operatora",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) => apiRequest("DELETE", `/api/production/operators/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/operators"] });
      setDeleteOperatorId(null);
      toast({ title: "Operator został usunięty" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć operatora",
        variant: "destructive",
      });
      setDeleteOperatorId(null);
    },
  });

  const openCreateDialog = () => {
    setEditingOperator(null);
    form.reset({
      fullName: "",
      shortCode: "",
      badgeId: "",
      isActive: true,
      notes: "",
    });
    setSelectedWorkCenterIds([]);
    setPrimaryWorkCenterId(null);
    setIsDialogOpen(true);
  };

  const openEditDialog = (operator: ProductionOperator) => {
    setEditingOperator(operator);
    form.reset({
      fullName: operator.full_name,
      shortCode: operator.short_code || "",
      badgeId: operator.badge_id || "",
      isActive: operator.is_active,
      notes: operator.notes || "",
    });
    
    // Ustaw gniazda robocze z danych operatora
    const wcIds = operator.work_centers?.map(wc => wc.id) || [];
    setSelectedWorkCenterIds(wcIds);
    
    const primary = operator.work_centers?.find(wc => wc.isPrimary);
    setPrimaryWorkCenterId(primary?.id || (wcIds.length > 0 ? wcIds[0] : null));
    
    setIsDialogOpen(true);
  };

  const onSubmit = (data: OperatorFormData) => {
    if (editingOperator) {
      updateMutation.mutate({ ...data, id: editingOperator.id });
    } else {
      createMutation.mutate(data);
    }
  };

  const toggleWorkCenter = (wcId: number) => {
    setSelectedWorkCenterIds(prev => {
      if (prev.includes(wcId)) {
        const newIds = prev.filter(id => id !== wcId);
        // Jeśli usuwamy primary, ustaw nowy primary
        if (primaryWorkCenterId === wcId) {
          setPrimaryWorkCenterId(newIds.length > 0 ? newIds[0] : null);
        }
        return newIds;
      } else {
        // Jeśli to pierwszy wybór, ustaw jako primary
        if (prev.length === 0) {
          setPrimaryWorkCenterId(wcId);
        }
        return [...prev, wcId];
      }
    });
  };

  const setPrimary = (wcId: number) => {
    if (selectedWorkCenterIds.includes(wcId)) {
      setPrimaryWorkCenterId(wcId);
    }
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between flex-wrap gap-4">
        <div>
          <h1 className="text-3xl font-bold">Operatorzy produkcji</h1>
          <p className="text-muted-foreground">
            Zarządzaj operatorami przypisanymi do operacji produkcyjnych
          </p>
        </div>
        <Button onClick={openCreateDialog} data-testid="button-add-operator">
          <Plus className="w-4 h-4 mr-2" />
          Dodaj operatora
        </Button>
      </div>

      <Card>
        <CardHeader>
          <div className="flex items-center justify-between flex-wrap gap-4">
            <div>
              <CardTitle>Lista operatorów</CardTitle>
              <CardDescription>
                {operatorsData?.total || 0} operatorów w systemie
              </CardDescription>
            </div>
            <div className="flex items-center gap-4">
              <Input
                placeholder="Szukaj operatora..."
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                className="w-64"
                data-testid="input-search-operators"
              />
              <Select value={filterActive} onValueChange={setFilterActive}>
                <SelectTrigger className="w-40" data-testid="select-filter-active">
                  <SelectValue placeholder="Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszyscy</SelectItem>
                  <SelectItem value="true">Aktywni</SelectItem>
                  <SelectItem value="false">Nieaktywni</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex items-center justify-center py-8">
              <Loader2 className="w-6 h-6 animate-spin" />
            </div>
          ) : operators.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground">
              Brak operatorów do wyświetlenia
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Imię i nazwisko</TableHead>
                  <TableHead>Kod</TableHead>
                  <TableHead>ID karty</TableHead>
                  <TableHead>Gniazda robocze</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead className="text-right">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {operators.map((operator) => (
                  <TableRow key={operator.id} data-testid={`row-operator-${operator.id}`}>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <div className="w-8 h-8 rounded-full bg-primary/10 flex items-center justify-center">
                          <User className="w-4 h-4" />
                        </div>
                        <span className="font-medium" data-testid={`text-name-${operator.id}`}>
                          {operator.full_name}
                        </span>
                      </div>
                    </TableCell>
                    <TableCell data-testid={`text-code-${operator.id}`}>
                      {operator.short_code ? (
                        <Badge variant="outline">{operator.short_code}</Badge>
                      ) : (
                        "-"
                      )}
                    </TableCell>
                    <TableCell data-testid={`text-badge-${operator.id}`}>
                      {operator.badge_id ? (
                        <div className="flex items-center gap-1">
                          <BadgeCheck className="w-4 h-4 text-muted-foreground" />
                          {operator.badge_id}
                        </div>
                      ) : (
                        "-"
                      )}
                    </TableCell>
                    <TableCell data-testid={`text-workcenters-${operator.id}`}>
                      {operator.work_centers && operator.work_centers.length > 0 ? (
                        <div className="flex flex-wrap gap-1">
                          {operator.work_centers.map((wc) => (
                            <Badge 
                              key={wc.id} 
                              variant={wc.isPrimary ? "default" : "secondary"}
                              className="text-xs"
                            >
                              {wc.isPrimary && <Star className="w-3 h-3 mr-1" />}
                              {wc.code}
                            </Badge>
                          ))}
                        </div>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </TableCell>
                    <TableCell>
                      {operator.is_active ? (
                        <Badge data-testid={`badge-status-${operator.id}`}>Aktywny</Badge>
                      ) : (
                        <Badge variant="secondary" data-testid={`badge-status-${operator.id}`}>
                          Nieaktywny
                        </Badge>
                      )}
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex items-center justify-end gap-1">
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => openEditDialog(operator)}
                          data-testid={`button-edit-${operator.id}`}
                        >
                          <Edit className="w-4 h-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => setDeleteOperatorId(operator.id)}
                          data-testid={`button-delete-${operator.id}`}
                        >
                          <Trash2 className="w-4 h-4 text-destructive" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog
        open={isDialogOpen}
        onOpenChange={(open) => {
          setIsDialogOpen(open);
          if (!open) {
            setEditingOperator(null);
            form.reset();
            setSelectedWorkCenterIds([]);
            setPrimaryWorkCenterId(null);
          }
        }}
      >
        <DialogContent className="max-w-lg" data-testid="dialog-operator">
          <DialogHeader>
            <DialogTitle>
              {editingOperator ? "Edytuj operatora" : "Dodaj nowego operatora"}
            </DialogTitle>
            <DialogDescription>
              {editingOperator
                ? "Zaktualizuj dane operatora produkcji"
                : "Wprowadź dane nowego operatora produkcji"}
            </DialogDescription>
          </DialogHeader>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
              <FormField
                control={form.control}
                name="fullName"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Imię i nazwisko</FormLabel>
                    <FormControl>
                      <Input
                        placeholder="np. Jan Kowalski"
                        {...field}
                        data-testid="input-fullname"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="shortCode"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kod (inicjały)</FormLabel>
                      <FormControl>
                        <Input
                          placeholder="np. JK"
                          {...field}
                          value={field.value || ""}
                          data-testid="input-shortcode"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="badgeId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>ID karty</FormLabel>
                      <FormControl>
                        <Input
                          placeholder="np. 12345"
                          {...field}
                          value={field.value || ""}
                          data-testid="input-badgeid"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="space-y-3">
                <FormLabel>Gniazda robocze</FormLabel>
                <p className="text-sm text-muted-foreground">
                  Wybierz maszyny, które może obsługiwać operator. Kliknij gwiazdkę aby ustawić główne gniazdo.
                </p>
                <div className="border rounded-md p-3 max-h-48 overflow-y-auto space-y-2">
                  {workCenters.length === 0 ? (
                    <p className="text-sm text-muted-foreground">Brak gniazd roboczych</p>
                  ) : (
                    workCenters.map((wc) => {
                      const isSelected = selectedWorkCenterIds.includes(wc.id);
                      const isPrimary = primaryWorkCenterId === wc.id;
                      
                      return (
                        <div 
                          key={wc.id} 
                          className="flex items-center justify-between py-1"
                        >
                          <div className="flex items-center gap-2">
                            <Checkbox
                              id={`wc-${wc.id}`}
                              checked={isSelected}
                              onCheckedChange={() => toggleWorkCenter(wc.id)}
                              data-testid={`checkbox-wc-${wc.id}`}
                            />
                            <label 
                              htmlFor={`wc-${wc.id}`}
                              className="text-sm cursor-pointer"
                            >
                              <span className="font-medium">{wc.code}</span>
                              <span className="text-muted-foreground ml-2">{wc.name}</span>
                            </label>
                          </div>
                          {isSelected && (
                            <Button
                              type="button"
                              variant="ghost"
                              size="icon"
                              className="h-6 w-6"
                              onClick={() => setPrimary(wc.id)}
                              data-testid={`button-primary-${wc.id}`}
                            >
                              <Star 
                                className={`w-4 h-4 ${isPrimary ? 'fill-yellow-400 text-yellow-400' : 'text-muted-foreground'}`} 
                              />
                            </Button>
                          )}
                        </div>
                      );
                    })
                  )}
                </div>
                {selectedWorkCenterIds.length > 0 && (
                  <p className="text-xs text-muted-foreground">
                    Wybrano {selectedWorkCenterIds.length} gniazd
                    {primaryWorkCenterId && (
                      <span>
                        , główne: <span className="font-medium">{workCenters.find(wc => wc.id === primaryWorkCenterId)?.code}</span>
                      </span>
                    )}
                  </p>
                )}
              </div>

              <FormField
                control={form.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Notatki</FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="Dodatkowe informacje..."
                        {...field}
                        value={field.value || ""}
                        data-testid="input-notes"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="isActive"
                render={({ field }) => (
                  <FormItem className="flex items-center justify-between rounded-lg border p-3">
                    <div className="space-y-0.5">
                      <FormLabel>Aktywny</FormLabel>
                      <p className="text-sm text-muted-foreground">
                        Nieaktywni operatorzy nie są wyświetlani w listach wyboru
                      </p>
                    </div>
                    <FormControl>
                      <Switch
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        data-testid="switch-active"
                      />
                    </FormControl>
                  </FormItem>
                )}
              />

              <DialogFooter>
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => {
                    setIsDialogOpen(false);
                    setEditingOperator(null);
                    form.reset();
                    setSelectedWorkCenterIds([]);
                    setPrimaryWorkCenterId(null);
                  }}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button
                  type="submit"
                  disabled={createMutation.isPending || updateMutation.isPending}
                  data-testid="button-submit"
                >
                  {createMutation.isPending || updateMutation.isPending ? (
                    <>
                      <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                      Zapisywanie...
                    </>
                  ) : editingOperator ? (
                    "Zapisz zmiany"
                  ) : (
                    "Utwórz operatora"
                  )}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      <AlertDialog
        open={deleteOperatorId !== null}
        onOpenChange={() => setDeleteOperatorId(null)}
      >
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno chcesz usunąć operatora?</AlertDialogTitle>
            <AlertDialogDescription>
              Ta akcja jest nieodwracalna. Operator zostanie trwale usunięty z systemu.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => deleteOperatorId && deleteMutation.mutate(deleteOperatorId)}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              data-testid="button-confirm-delete"
            >
              Usuń
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
