import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Plus, Edit, Trash2, Loader2, ChevronRight, MapPin } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { insertProductionLocationSchema, insertProductionLocationGroupSchema } from "@shared/schema";
import type { ProductionLocation, ProductionLocationGroup, InsertProductionLocation } from "@shared/schema";
import { z } from "zod";

// Extend the shared schema with frontend-specific validation
const locationFormSchema = insertProductionLocationSchema.extend({
  locationGroupId: z.coerce.number().min(1, "Grupa jest wymagana"),
  parentId: z.coerce.number().optional().nullable(),
  capacity: z.string().optional(),
});

type LocationFormData = z.infer<typeof locationFormSchema>;

export default function ProductionLocations() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingLocation, setEditingLocation] = useState<ProductionLocation | null>(null);
  const { toast } = useToast();

  const form = useForm<LocationFormData>({
    resolver: zodResolver(locationFormSchema),
    defaultValues: { 
      locationGroupId: 0, 
      code: "", 
      name: "", 
      status: "active", 
      isActive: true,
      allowsStorage: true,
    },
  });

  const { data: locationGroups = [] } = useQuery<ProductionLocationGroup[]>({
    queryKey: ["/api/production/location-groups"],
  });

  const { data: locations = [], isLoading } = useQuery<ProductionLocation[]>({
    queryKey: ["/api/production/locations"],
  });

  const createMutation = useMutation({
    mutationFn: (data: any) => apiRequest("POST", "/api/production/locations", data),
    onSuccess: () => {
      console.log("✅ Mutation success!");
      queryClient.invalidateQueries({ queryKey: ["/api/production/locations"] });
      setIsDialogOpen(false);
      form.reset();
      toast({ title: "Lokalizacja została utworzona" });
    },
    onError: (error: any) => {
      console.log("❌ Mutation error:", error);
      toast({ title: "Błąd tworzenia lokalizacji", description: error.message || "Nieznany błąd", variant: "destructive" });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: any }) =>
      apiRequest("PATCH", `/api/production/locations/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/locations"] });
      setIsDialogOpen(false);
      setEditingLocation(null);
      form.reset();
      toast({ title: "Lokalizacja została zaktualizowana" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) => apiRequest("DELETE", `/api/production/locations/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/locations"] });
      toast({ title: "Lokalizacja została usunięta" });
    },
  });

  useEffect(() => {
    if (editingLocation) {
      form.reset({
        locationGroupId: editingLocation.locationGroupId,
        parentId: editingLocation.parentId ?? undefined,
        code: editingLocation.code,
        name: editingLocation.name,
        barcode: editingLocation.barcode ?? "",
        capacity: editingLocation.capacity?.toString() ?? "",
        capacityUnit: editingLocation.capacityUnit ?? "",
        status: editingLocation.status || "active",
        notes: editingLocation.notes ?? "",
        isActive: editingLocation.isActive ?? true,
        allowsStorage: editingLocation.allowsStorage ?? true,
      });
    } else {
      form.reset({ 
        locationGroupId: 0, 
        code: "", 
        name: "", 
        barcode: "",
        capacity: "",
        capacityUnit: "",
        status: "active", 
        notes: "",
        isActive: true,
        allowsStorage: true,
      });
    }
  }, [editingLocation, form]);

  const onSubmit = (data: LocationFormData) => {
    console.log("📝 Form submit - data:", data);
    // Normalize parentId: 0 or undefined should be sent as null for root level
    const normalizedData = {
      ...data,
      parentId: data.parentId && data.parentId !== 0 ? data.parentId : null,
      capacity: data.capacity === "" || !data.capacity ? undefined : Number(data.capacity),
      barcode: data.barcode || undefined,
      capacityUnit: data.capacityUnit || undefined,
      notes: data.notes || undefined,
    };
    
    console.log("📤 Sending to API:", normalizedData);
    console.log("📊 editingLocation:", editingLocation);
    console.log("🔍 Will call:", editingLocation ? "updateMutation" : "createMutation");
    
    if (editingLocation) {
      console.log("🔄 Calling updateMutation...");
      updateMutation.mutate({ id: editingLocation.id, data: normalizedData });
    } else {
      console.log("➕ Calling createMutation...");
      createMutation.mutate(normalizedData);
    }
  };
  
  const onError = (errors: any) => {
    console.log("❌ Form validation errors:", errors);
    toast({ title: "Błąd walidacji formularza", description: "Sprawdź wymagane pola", variant: "destructive" });
  };

  const getParentName = (parentId: number | null) => {
    if (!parentId) return "-";
    const parent = locations.find(l => l.id === parentId);
    return parent?.name || "Nieznana";
  };

  const getLocationGroupName = (groupId: number) => {
    const group = locationGroups.find(g => g.id === groupId);
    return group?.name || "Nieznana grupa";
  };

  const renderLocationPath = (location: ProductionLocation) => {
    if (!location.path) return location.name;
    const parts = location.path.split('/').filter(Boolean);
    return (
      <div className="flex items-center gap-1">
        {parts.map((part, idx) => (
          <div key={idx} className="flex items-center gap-1">
            {idx > 0 && <ChevronRight className="w-3 h-3 text-muted-foreground" />}
            <span className={idx === parts.length - 1 ? "" : "text-muted-foreground text-sm"}>{part}</span>
          </div>
        ))}
      </div>
    );
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex justify-between items-center">
        <h1 className="text-3xl font-bold" data-testid="text-page-title">Lokalizacje Produkcyjne</h1>
        <Button onClick={() => { setEditingLocation(null); setIsDialogOpen(true); }} data-testid="button-add-location">
          <Plus className="w-4 h-4 mr-2" />Dodaj Lokalizację
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Hierarchia Lokalizacji</CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex items-center justify-center p-8" data-testid="loading-locations">
              <Loader2 className="w-8 h-8 animate-spin text-muted-foreground" />
            </div>
          ) : locations.length === 0 ? (
            <p className="text-muted-foreground text-center p-8" data-testid="empty-locations">Brak lokalizacji</p>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Ścieżka</TableHead>
                  <TableHead>Kod</TableHead>
                  <TableHead>Grupa</TableHead>
                  <TableHead>Pojemność</TableHead>
                  <TableHead>Obciążenie</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead className="text-right">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {locations
                  .sort((a, b) => (a.path || "").localeCompare(b.path || ""))
                  .map((location) => (
                    <TableRow key={location.id} data-testid={`row-location-${location.id}`}>
                      <TableCell data-testid={`text-location-path-${location.id}`}>
                        {renderLocationPath(location)}
                      </TableCell>
                      <TableCell className="font-mono" data-testid={`text-location-code-${location.id}`}>
                        {location.code}
                      </TableCell>
                      <TableCell data-testid={`text-location-group-${location.id}`}>
                        <Badge variant="outline">{getLocationGroupName(location.locationGroupId)}</Badge>
                      </TableCell>
                      <TableCell data-testid={`text-location-capacity-${location.id}`}>
                        {location.capacity ? `${location.capacity} ${location.capacityUnit || ""}` : "-"}
                      </TableCell>
                      <TableCell data-testid={`text-location-load-${location.id}`}>
                        {location.currentLoad || 0} / {location.capacity || "-"}
                      </TableCell>
                      <TableCell data-testid={`text-location-status-${location.id}`}>
                        <Badge variant={
                          location.status === "active" ? "default" :
                          location.status === "full" ? "destructive" :
                          location.status === "maintenance" ? "secondary" : "outline"
                        }>
                          {location.status === "active" ? "Aktywna" :
                           location.status === "inactive" ? "Nieaktywna" :
                           location.status === "full" ? "Pełna" : "Konserwacja"}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-right space-x-2">
                        <Button size="icon" variant="ghost" onClick={() => { setEditingLocation(location); setIsDialogOpen(true); }} data-testid={`button-edit-location-${location.id}`}>
                          <Edit className="w-4 h-4" />
                        </Button>
                        <Button size="icon" variant="ghost" onClick={() => deleteMutation.mutate(location.id)} data-testid={`button-delete-location-${location.id}`}>
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={(open) => { setIsDialogOpen(open); if (!open) { setEditingLocation(null); form.reset(); } }}>
        <DialogContent className="max-w-2xl" data-testid="dialog-location">
          <DialogHeader>
            <DialogTitle>{editingLocation ? "Edytuj Lokalizację" : "Dodaj Lokalizację"}</DialogTitle>
            <DialogDescription>
              {editingLocation ? "Zaktualizuj informacje o lokalizacji" : "Utwórz nową lokalizację w hierarchii produkcji"}
            </DialogDescription>
          </DialogHeader>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit, onError)} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <FormField control={form.control} name="locationGroupId" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Grupa*</FormLabel>
                    <Select onValueChange={(value) => field.onChange(parseInt(value))} value={field.value?.toString() || ""}>
                      <FormControl>
                        <SelectTrigger data-testid="select-location-group">
                          <SelectValue placeholder="Wybierz grupę" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        {locationGroups.map(group => (
                          <SelectItem key={group.id} value={group.id.toString()}>{group.name}</SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="parentId" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Lokalizacja nadrzędna</FormLabel>
                    <Select onValueChange={(value) => field.onChange(value ? parseInt(value) : undefined)} value={field.value?.toString() || ""}>
                      <FormControl>
                        <SelectTrigger data-testid="select-parent-location">
                          <SelectValue placeholder="Brak (główna)" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="0">Brak (główna)</SelectItem>
                        {locations.map(loc => (
                          <SelectItem key={loc.id} value={loc.id.toString()}>{loc.path || loc.name}</SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="code" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kod*</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. HALA-A" data-testid="input-location-code" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="name" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa*</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. Hala produkcyjna A" data-testid="input-location-name" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="barcode" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kod kreskowy</FormLabel>
                    <FormControl>
                      <Input {...field} value={field.value || ""} placeholder="Opcjonalny" data-testid="input-location-barcode" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="status" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Status*</FormLabel>
                    <Select onValueChange={field.onChange} value={field.value || "active"}>
                      <FormControl>
                        <SelectTrigger data-testid="select-location-status">
                          <SelectValue />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="active">Aktywna</SelectItem>
                        <SelectItem value="inactive">Nieaktywna</SelectItem>
                        <SelectItem value="full">Pełna</SelectItem>
                        <SelectItem value="maintenance">Konserwacja</SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="capacity" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Pojemność</FormLabel>
                    <FormControl>
                      <Input {...field} value={field.value ?? ""} type="number" step="0.01" placeholder="np. 1000" onChange={field.onChange} data-testid="input-location-capacity" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="capacityUnit" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Jednostka pojemności</FormLabel>
                    <FormControl>
                      <Input {...field} value={field.value || ""} placeholder="np. szt, m³" data-testid="input-location-unit" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="notes" render={({ field }) => (
                  <FormItem className="col-span-2">
                    <FormLabel>Uwagi</FormLabel>
                    <FormControl>
                      <Textarea {...field} value={field.value || ""} placeholder="Dodatkowe informacje" data-testid="input-location-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
              </div>
              <DialogFooter>
                <Button type="button" variant="outline" onClick={() => { setIsDialogOpen(false); setEditingLocation(null); form.reset(); }} data-testid="button-cancel-location">
                  Anuluj
                </Button>
                <Button type="submit" disabled={createMutation.isPending || updateMutation.isPending} data-testid="button-submit-location">
                  {(createMutation.isPending || updateMutation.isPending) && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
                  {editingLocation ? "Zapisz" : "Utwórz"}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
