import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { ArrowLeft, Package, Store, ShoppingCart, Calendar, Download, ExternalLink, Link as LinkIcon, Loader2, Search, X } from "lucide-react";
import { Link } from "wouter";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog";
import { Input } from "@/components/ui/input";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useNavigation } from "@/lib/navigation";
import { useState, useEffect, useMemo } from "react";
import { MarketplaceLinkDialog } from "@/components/marketplace-catalog-link-dialog";

interface MarketplaceProduct {
  id: number;
  offer_external_id: string;
  source: 'allegro' | 'shoper';
  name: string;
  image_url: string | null;
  last_sold_at: string | null;
  times_sold: number;
  total_quantity_sold: number;
  avg_unit_price: string | null;
  created_at: string;
  updated_at: string;
}

interface ProductOrder {
  id: number;
  order_number: string;
  source: string;
  source_order_id: string;
  order_date: string;
  payment_status: string;
  payment_amount: string;
  buyer_first_name: string;
  buyer_last_name: string;
  buyer_login: string;
  quantity: number;
  unit_price: string;
  total_price: string;
}

interface CatalogProduct {
  id: number;
  sku: string;
  title: string;
  linkType?: 'product' | 'set';
  linkId?: number;
}

interface OmsStats {
  totalQuantity: number;
  orderCount: number;
}

interface ProductDetailsResponse {
  product: MarketplaceProduct;
  orders: ProductOrder[];
  omsStats: OmsStats;
  catalogProduct: CatalogProduct | null;
  catalogImageUrl: string | null;
}

export default function ProductDetails() {
  const { sku } = useParams<{ sku: string }>();
  const [location, navigate] = useLocation();
  const { toast } = useToast();
  const [linkDialogOpen, setLinkDialogOpen] = useState(false);
  const [confirmUnlink, setConfirmUnlink] = useState<number | null>(null);
  const navigation = useNavigation();

  // Pobierz parametr source z URL (np. /product/123?source=allegro)
  // Używamy useMemo aby reagować na zmiany w location
  const source = useMemo(() => {
    const urlParams = new URLSearchParams(window.location.search);
    return urlParams.get('source');
  }, [location]);

  const { data, isLoading, error } = useQuery<ProductDetailsResponse>({
    queryKey: [`/api/marketplace-products/${sku}`, { source }],
    queryFn: async () => {
      const url = source 
        ? `/api/marketplace-products/${sku}?source=${encodeURIComponent(source)}`
        : `/api/marketplace-products/${sku}`;
      const res = await fetch(url, { credentials: 'include' });
      if (!res.ok) throw new Error('Failed to fetch product');
      return res.json();
    },
    enabled: !!sku,
  });

  const syncImageMutation = useMutation({
    mutationFn: async (externalId: string) => {
      const response = await fetch(`/api/marketplace-products/${externalId}/sync-image`, {
        method: 'POST',
        credentials: 'include',
      });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to sync image');
      }
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "✅ Zdjęcie zsynchronizowane",
        description: "Zdjęcie produktu zostało pobrane z Allegro",
      });
      queryClient.invalidateQueries({ queryKey: [`/api/marketplace-products/${sku}`] });
    },
    onError: (error: any) => {
      toast({
        title: "❌ Błąd synchronizacji",
        description: error.message || "Nie udało się pobrać zdjęcia",
        variant: "destructive",
      });
    },
  });

  const syncCatalogImageMutation = useMutation({
    mutationFn: async (externalId: string) => {
      const response = await fetch(`/api/marketplace-products/${externalId}/sync-catalog-image`, {
        method: 'POST',
        credentials: 'include',
      });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to sync catalog image');
      }
      return response.json();
    },
    onSuccess: (data) => {
      toast({
        title: "✅ Zdjęcie zsynchronizowane",
        description: data.message || "Zdjęcie produktu zostało pobrane z katalogu",
      });
      queryClient.invalidateQueries({ queryKey: [`/api/marketplace-products/${sku}`] });
    },
    onError: (error: any) => {
      toast({
        title: "❌ Błąd synchronizacji",
        description: error.message || "Nie udało się pobrać zdjęcia z katalogu",
        variant: "destructive",
      });
    },
  });

  const unlinkMutation = useMutation({
    mutationFn: async (linkId: number) => {
      const response = await fetch(`/api/catalog/products/platform-links/${linkId}`, {
        method: 'DELETE',
        credentials: 'include',
      });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to unlink product');
      }
      return response.json();
    },
    onSuccess: async () => {
      toast({
        title: "✅ Połączenie usunięte",
        description: "Produkt marketplace został odłączony od katalogu",
      });
      // Force refetch to ensure UI updates
      await queryClient.refetchQueries({ queryKey: [`/api/marketplace-products/${sku}`] });
      setConfirmUnlink(null);
    },
    onError: (error: any) => {
      toast({
        title: "❌ Błąd rozłączania",
        description: error.message || "Nie udało się rozłączyć produktu",
        variant: "destructive",
      });
    },
  });

  if (isLoading) {
    return (
      <div className="container mx-auto p-6 space-y-6">
        <Skeleton className="h-10 w-64" />
        <div className="grid gap-6 md:grid-cols-3">
          <Card className="md:col-span-2">
            <CardHeader>
              <Skeleton className="h-8 w-3/4" />
            </CardHeader>
            <CardContent className="space-y-4">
              <Skeleton className="h-64 w-full" />
              <Skeleton className="h-20 w-full" />
            </CardContent>
          </Card>
          <div className="space-y-4">
            <Skeleton className="h-48 w-full" />
          </div>
        </div>
      </div>
    );
  }

  if (error || !data) {
    return (
      <div className="container mx-auto p-6 space-y-6">
        <Button
          variant="ghost"
          onClick={() => navigation.goBack('/catalog-products', navigate)}
          className="gap-2"
          data-testid="button-back-products"
        >
          <ArrowLeft className="h-4 w-4" />
          Powrót do produktów
        </Button>
        <Card>
          <CardContent className="pt-6">
            <div className="text-center py-12">
              <Package className="h-16 w-16 mx-auto text-muted-foreground mb-4" />
              <h2 className="text-xl font-semibold mb-2">Nie znaleziono produktu</h2>
              <p className="text-muted-foreground">
                Produkt o SKU <code className="px-2 py-1 bg-muted rounded">{sku}</code> nie istnieje.
              </p>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  const { product, orders, omsStats, catalogProduct, catalogImageUrl } = data;

  // Determine catalog link href based on linkType
  const catalogHref = catalogProduct?.linkType === 'set'
    ? `/catalog-sets/${catalogProduct.id}`
    : catalogProduct?.linkType === 'product'
    ? `/catalog-products/${catalogProduct.id}`
    : null;

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <Button
          variant="ghost"
          onClick={() => navigation.goBack('/catalog-products', navigate)}
          className="gap-2"
          data-testid="button-back-products"
        >
          <ArrowLeft className="h-4 w-4" />
          Powrót do produktów
        </Button>
        
        <div className="flex items-center gap-2">
          {catalogProduct ? (
            <Card className="px-4 py-2.5 bg-green-50 dark:bg-green-950/20 border-green-200 dark:border-green-800">
              <div className="flex items-center gap-3">
                <div className="flex items-center gap-2">
                  <div className="relative flex h-2 w-2">
                    <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-green-400 opacity-75"></span>
                    <span className="relative inline-flex rounded-full h-2 w-2 bg-green-500"></span>
                  </div>
                  <span className="text-sm font-medium text-green-700 dark:text-green-300">
                    Połączono
                  </span>
                </div>
                <div className="h-4 w-px bg-green-200 dark:bg-green-800" />
                {catalogHref ? (
                  <Link href={catalogHref} asChild>
                    <button
                      className="inline-flex items-center gap-1.5 px-2.5 py-1 rounded-md text-xs font-medium bg-white dark:bg-gray-900 border border-green-200 dark:border-green-800 text-green-700 dark:text-green-300 hover:bg-green-50 dark:hover:bg-green-900/30 transition-colors cursor-pointer"
                      title={`Przejdź do ${catalogProduct.linkType === 'set' ? 'zestawu' : 'produktu'} katalogowego: ${catalogProduct.title}`}
                      data-testid="button-catalog-sku"
                    >
                      SKU: {catalogProduct.sku}
                    </button>
                  </Link>
                ) : (
                  <span className="inline-flex items-center gap-1.5 px-2.5 py-1 rounded-md text-xs font-medium bg-white dark:bg-gray-900 border border-green-200 dark:border-green-800 text-green-700 dark:text-green-300">
                    SKU: {catalogProduct.sku}
                  </span>
                )}
                {catalogHref && (
                  <Link href={catalogHref} asChild>
                    <Button
                      size="sm"
                      variant="ghost"
                      data-testid="button-view-catalog"
                      title={`Otwórz ${catalogProduct.linkType === 'set' ? 'zestaw' : 'produkt'} katalogowy: ${catalogProduct.title}`}
                      className="h-auto px-2 py-1 text-green-700 dark:text-green-300 hover:text-green-800 dark:hover:text-green-200 hover:bg-green-100 dark:hover:bg-green-900/20 gap-1"
                    >
                      <ExternalLink className="h-3.5 w-3.5" />
                      Zobacz {catalogProduct.linkType === 'set' ? 'zestaw' : 'produkt'}
                    </Button>
                  </Link>
                )}
                <div className="h-4 w-px bg-green-200 dark:bg-green-800" />
                <Button
                  size="sm"
                  variant="ghost"
                  onClick={() => catalogProduct.linkId && setConfirmUnlink(catalogProduct.linkId)}
                  disabled={!catalogProduct.linkId || unlinkMutation.isPending}
                  data-testid="button-unlink-catalog"
                  title="Rozłącz połączenie"
                  className="h-auto px-2 py-1 text-red-600 dark:text-red-400 hover:text-red-700 dark:hover:text-red-300 hover:bg-red-50 dark:hover:bg-red-900/20"
                >
                  {unlinkMutation.isPending ? <Loader2 className="h-3.5 w-3.5 animate-spin" /> : <X className="h-3.5 w-3.5" />}
                </Button>
              </div>
            </Card>
          ) : (
            <Button
              size="default"
              variant="outline"
              onClick={() => setLinkDialogOpen(true)}
              data-testid="button-link-catalog"
              title="Połącz z produktem katalogowym"
              className="bg-blue-600 hover:bg-blue-700 border-blue-600 text-white gap-2"
            >
              <Package className="h-4 w-4" />
              Połącz z katalogiem
            </Button>
          )}
        </div>
      </div>

      <div className="grid gap-6 lg:grid-cols-3">
        {/* Główna karta produktu */}
        <Card className="lg:col-span-2">
          <CardHeader>
            <div className="flex items-start justify-between gap-4">
              <div className="flex-1">
                <CardTitle className="text-2xl mb-2" data-testid="text-product-name">
                  {product.name}
                </CardTitle>
                <div className="flex flex-wrap gap-2">
                  <Badge variant="outline" data-testid="badge-sku">
                    SKU: {product.offer_external_id}
                  </Badge>
                  <Badge variant={product.source === 'allegro' ? 'default' : 'secondary'} data-testid="badge-source">
                    {product.source === 'allegro' ? 'Allegro' : 'Shoper'}
                  </Badge>
                  {catalogProduct && (
                    <Badge variant="secondary" className="bg-blue-100 dark:bg-blue-950 text-blue-700 dark:text-blue-300" data-testid="badge-catalog-linked">
                      Połączony z katalogiem
                    </Badge>
                  )}
                </div>
              </div>
            </div>
          </CardHeader>
          <CardContent className="space-y-6">
            <div className="flex flex-col md:flex-row gap-6">
              <div className="flex-shrink-0 space-y-4 w-full md:w-auto flex flex-col items-center md:items-start">
                {(catalogImageUrl || product.image_url) ? (
                  <img
                    src={catalogImageUrl || product.image_url || undefined}
                    alt={product.name}
                    className="w-full max-w-[280px] md:w-64 h-auto aspect-square object-cover rounded-lg border"
                    data-testid="img-product"
                    onError={(e) => {
                      (e.target as HTMLImageElement).src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjU2IiBoZWlnaHQ9IjI1NiIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMjU2IiBoZWlnaHQ9IjI1NiIgZmlsbD0iI2YzZjRmNiIvPjwvc3ZnPg==';
                    }}
                  />
                ) : (
                  <>
                    <div className="w-full max-w-[280px] md:w-64 aspect-square bg-muted rounded-lg border flex items-center justify-center">
                      <Package className="h-24 w-24 md:h-32 md:w-32 text-muted-foreground" />
                    </div>
                    {catalogProduct ? (
                      <Button
                        onClick={() => syncCatalogImageMutation.mutate(product.offer_external_id)}
                        disabled={syncCatalogImageMutation.isPending}
                        className="w-full max-w-[280px] md:w-64"
                        data-testid="button-sync-catalog-image"
                      >
                        <Download className="h-4 w-4 mr-2" />
                        {syncCatalogImageMutation.isPending ? 'Pobieranie...' : 'Pobierz zdjęcie z katalogu'}
                      </Button>
                    ) : product.source.toUpperCase() === 'ALLEGRO' && (
                      <Button
                        onClick={() => syncImageMutation.mutate(product.offer_external_id)}
                        disabled={syncImageMutation.isPending}
                        className="w-full max-w-[280px] md:w-64"
                        data-testid="button-sync-image"
                      >
                        <Download className="h-4 w-4 mr-2" />
                        {syncImageMutation.isPending ? 'Pobieranie...' : 'Pobierz zdjęcie z Allegro'}
                      </Button>
                    )}
                  </>
                )}
              </div>

              <div className="flex-1 space-y-4">
                <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                  <div>
                    <p className="text-sm text-muted-foreground">Średnia cena</p>
                    <p className="text-2xl font-bold" data-testid="text-avg-price">
                      {product.avg_unit_price 
                        ? `${parseFloat(product.avg_unit_price).toFixed(2)} PLN`
                        : '—'}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground">Sprzedaż marketplace (lifetime)</p>
                    <p className="text-2xl font-bold" data-testid="text-quantity-sold">
                      {product.total_quantity_sold} szt.
                    </p>
                    <p className="text-xs text-muted-foreground">Historyczne dane z platformy</p>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground">Zamówienia w OMS</p>
                    <p className="text-2xl font-bold" data-testid="text-oms-orders-count">
                      {omsStats?.orderCount ?? 0} zam. ({omsStats?.totalQuantity ?? 0} szt.)
                    </p>
                    <p className="text-xs text-muted-foreground">Zaimportowane do systemu</p>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground">Ostatnia sprzedaż</p>
                    <p className="text-base font-medium" data-testid="text-last-sold">
                      {product.last_sold_at
                        ? new Date(product.last_sold_at).toLocaleDateString('pl-PL')
                        : '—'}
                    </p>
                  </div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Boczna karta z dodatkowymi info */}
        <div className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle className="text-lg flex items-center gap-2">
                <Store className="h-5 w-5" />
                Informacje
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div>
                <p className="text-sm text-muted-foreground">ID produktu</p>
                <p className="text-sm font-medium" data-testid="text-product-id">
                  #{product.id}
                </p>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Pierwsze wystąpienie</p>
                <p className="text-sm font-medium" data-testid="text-created-at">
                  {new Date(product.created_at).toLocaleDateString("pl-PL", {
                    year: 'numeric',
                    month: 'long',
                    day: 'numeric'
                  })}
                </p>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Ostatnia aktualizacja</p>
                <p className="text-sm font-medium" data-testid="text-updated-at">
                  {new Date(product.updated_at).toLocaleDateString("pl-PL", {
                    year: 'numeric',
                    month: 'long',
                    day: 'numeric'
                  })}
                </p>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>

      {/* Lista zamówień */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <ShoppingCart className="h-5 w-5" />
            Zamówienia z tym produktem ({orders.length})
          </CardTitle>
        </CardHeader>
        <CardContent>
          {orders.length === 0 ? (
            <div className="text-center py-12">
              <ShoppingCart className="h-16 w-16 mx-auto text-muted-foreground mb-4" />
              <p className="text-muted-foreground">Brak zamówień z tym produktem</p>
            </div>
          ) : (
            <div className="overflow-x-auto -mx-6 px-6 md:mx-0 md:px-0">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Nr zamówienia</TableHead>
                    <TableHead>Data</TableHead>
                    <TableHead>Klient</TableHead>
                    <TableHead>Ilość</TableHead>
                    <TableHead>Cena jedn.</TableHead>
                    <TableHead>Suma</TableHead>
                    <TableHead>Status</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {orders.map((order) => (
                    <TableRow 
                      key={order.id}
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/order/${order.order_number}`)}
                      data-testid={`row-order-${order.order_number}`}
                    >
                      <TableCell>
                        <div>
                          <div className="font-mono text-sm font-medium">
                            {order.order_number}
                          </div>
                          <Badge variant="outline" className="text-xs mt-1">
                            {order.source === 'ALLEGRO' ? 'Allegro' : 'Shoper'}
                          </Badge>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="text-sm">
                          {new Date(order.order_date).toLocaleDateString('pl-PL')}
                        </div>
                        <div className="text-xs text-muted-foreground">
                          {new Date(order.order_date).toLocaleTimeString('pl-PL', {
                            hour: '2-digit',
                            minute: '2-digit'
                          })}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="text-sm">
                          {order.buyer_first_name && order.buyer_last_name
                            ? `${order.buyer_first_name} ${order.buyer_last_name}`
                            : order.buyer_login}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="font-medium">{order.quantity} szt.</div>
                      </TableCell>
                      <TableCell>
                        <div className="text-sm">
                          {parseFloat(order.unit_price).toFixed(2)} PLN
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="font-medium">
                          {parseFloat(order.total_price).toFixed(2)} PLN
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge 
                          variant={order.payment_status === 'PAID' ? 'default' : 'secondary'}
                          data-testid={`badge-status-${order.source_order_id}`}
                        >
                          {order.payment_status === 'PAID' ? 'Opłacone' : order.payment_status}
                        </Badge>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Marketplace-Catalog Link Dialog */}
      <MarketplaceLinkDialog
        open={linkDialogOpen}
        onOpenChange={setLinkDialogOpen}
        mode="marketplace-to-catalog"
        marketplaceProduct={{
          id: product.id,
          externalId: product.offer_external_id,
          platform: product.source.toLowerCase() as 'allegro' | 'shoper',
          sku: product.offer_external_id,
          name: product.name,
        }}
        onSuccess={() => {
          queryClient.invalidateQueries({ queryKey: [`/api/marketplace-products/${sku}`] });
        }}
      />

      {/* Unlink Confirmation Dialog */}
      <AlertDialog open={!!confirmUnlink} onOpenChange={(open) => !open && setConfirmUnlink(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Rozłączyć połączenie?</AlertDialogTitle>
            <AlertDialogDescription>
              Produkt marketplace zostanie odłączony od {catalogProduct?.linkType === 'set' ? 'zestawu' : 'produktu'} katalogowego.
              Ta operacja usunie połączenie, ale nie skasuje ani produktu marketplace ani katalogu.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-unlink">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => confirmUnlink && unlinkMutation.mutate(confirmUnlink)}
              disabled={unlinkMutation.isPending}
              data-testid="button-confirm-unlink"
              className="bg-red-600 hover:bg-red-700"
            >
              {unlinkMutation.isPending ? <Loader2 className="h-4 w-4 mr-2 animate-spin" /> : null}
              Rozłącz
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
