import { useState, useMemo, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useRoute, Link, useLocation } from "wouter";
import { 
  ArrowLeft,
  Package,
  Loader2,
  CheckCircle2,
  Clock,
  AlertTriangle,
  Play,
  Square,
  Box,
  Palette,
  Settings2,
  ClipboardList,
  Layers,
  PackageOpen,
  Truck,
  AlertCircle,
  Filter,
  Check,
  X,
  ArrowUpDown,
  ChevronDown,
  ChevronRight,
  Image as ImageIcon,
  ShoppingCart
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Progress } from "@/components/ui/progress";
import { Separator } from "@/components/ui/separator";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Input } from "@/components/ui/input";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";

interface ZlpColor {
  orderId: number;
  orderNumber: string;
  colorCode: string;
  colorName?: string;
  status: string;
  assemblyReady: boolean;
  packingReady: boolean;
  componentsTotal: number;
  componentsDone: number;
}

interface PlanAssemblyData {
  planId: number;
  planNumber: string;
  planName: string;
  zlpColors: ZlpColor[];
  assemblyOperations: {
    operationCode: string;
    operationName: string;
    status: string;
    totalItems: number;
    completedItems: number;
    canStart: boolean;
  }[];
  packingOperations: {
    operationCode: string;
    operationName: string;
    status: string;
    totalItems: number;
    completedItems: number;
    canStart: boolean;
  }[];
  allFormatkiCount: number;
  allFormatkiDone: number;
  allFormatkiDamaged: number;
}

interface FormatkaItem {
  id: number;
  componentName: string;
  componentType: string;
  colorCode: string;
  length: number;
  width: number;
  thickness: number;
  quantity: number;
  isDamaged: boolean;
  damageType?: string;
  damageNotes?: string;
  isAssembled: boolean;
  assembledAt?: string;
  isCut: boolean;
  isEdged: boolean;
  isDrilled: boolean;
  isFromWarehouse?: boolean;
  zlpOrderNumber: string;
}

interface PackingProduct {
  lineId: number;
  productId: number;
  productSku: string;
  productTitle: string;
  productImage?: string;
  quantity: number;
  reservedQuantity: number;
  sourceReference?: string;
  formatki: (FormatkaItem & { zlpOrderNumber?: string; isFromWarehouse?: boolean })[];
  formatkiTotal: number;
  formatkiAssembled: number;
  formatkiDamaged: number;
}

const statusColors: Record<string, string> = {
  pending: "bg-gray-400",
  in_progress: "bg-blue-500",
  done: "bg-green-500",
  completed: "bg-green-500",
};

const statusLabels: Record<string, string> = {
  pending: "Oczekuje",
  in_progress: "W trakcie",
  done: "Ukończone",
  completed: "Ukończone",
};

export default function PlanAssemblyStation() {
  const [, params] = useRoute("/production/plans/:id/assembly/:section?");
  const planId = params?.id ? parseInt(params.id) : null;
  const section = params?.section || "overview";
  const { toast } = useToast();
  const [, navigate] = useLocation();
  const [selectedFormatki, setSelectedFormatki] = useState<Set<number>>(new Set());
  const [filterColor, setFilterColor] = useState<string>("all");
  const [filterZlp, setFilterZlp] = useState<string>("all");
  const [filterAssembled, setFilterAssembled] = useState<string>("all");
  const [searchText, setSearchText] = useState<string>("");
  const [filterLength, setFilterLength] = useState<string>("");
  const [filterWidth, setFilterWidth] = useState<string>("");
  const [sortField, setSortField] = useState<string>("zlpOrderNumber");
  const [sortDir, setSortDir] = useState<"asc" | "desc">("asc");
  
  // Debounced search text for server-side filtering
  const [debouncedSearch, setDebouncedSearch] = useState<string>("");
  useEffect(() => {
    const timer = setTimeout(() => setDebouncedSearch(searchText), 300);
    return () => clearTimeout(timer);
  }, [searchText]);

  const { data: assemblyData, isLoading } = useQuery<PlanAssemblyData>({
    queryKey: ['/api/production/plans', planId, 'assembly-station'],
    enabled: !!planId,
  });

  // Build query string for server-side filtering
  const formatkiQueryString = useMemo(() => {
    const params = new URLSearchParams();
    if (debouncedSearch) params.set('searchText', debouncedSearch);
    if (filterColor !== 'all') params.set('colorCode', filterColor);
    if (filterZlp !== 'all') params.set('zlpOrderNumber', filterZlp);
    if (filterAssembled !== 'all') params.set('isAssembled', filterAssembled);
    if (filterLength) params.set('length', filterLength);
    if (filterWidth) params.set('width', filterWidth);
    return params.toString();
  }, [debouncedSearch, filterColor, filterZlp, filterAssembled, filterLength, filterWidth]);

  const { data: formatki = [], isLoading: formatkiLoading } = useQuery<FormatkaItem[]>({
    queryKey: ['/api/production/plans', planId, 'all-formatki', formatkiQueryString],
    queryFn: async () => {
      const url = `/api/production/plans/${planId}/all-formatki${formatkiQueryString ? `?${formatkiQueryString}` : ''}`;
      const res = await fetch(url, { credentials: 'include' });
      if (!res.ok) throw new Error('Failed to fetch formatki');
      return res.json();
    },
    enabled: !!planId && section === 'formatki',
  });

  // Fetch packing products with their formatki
  const { data: packingProducts = [], isLoading: packingProductsLoading } = useQuery<PackingProduct[]>({
    queryKey: ['/api/production/plans', planId, 'packing-products'],
    enabled: !!planId && section === 'packing',
  });

  const startAssemblyMutation = useMutation({
    mutationFn: async () => {
      const res = await apiRequest('POST', `/api/production/plans/${planId}/assembly-operations/start`);
      return res.json();
    },
    onSuccess: (data: { message: string; updatedCount: number }) => {
      toast({ 
        title: data.updatedCount > 0 ? "Rozpoczęto kompletowanie" : "Informacja", 
        description: data.message 
      });
      queryClient.invalidateQueries({ queryKey: ['/api/production/plans', planId, 'assembly-station'] });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się rozpocząć kompletowania", variant: "destructive" });
    },
  });

  const completeAssemblyMutation = useMutation({
    mutationFn: async () => {
      return apiRequest('POST', `/api/production/plans/${planId}/assembly-operations/complete`);
    },
    onSuccess: () => {
      toast({ title: "Zakończono kompletowanie", description: "Operacja kompletowania została zakończona" });
      queryClient.invalidateQueries({ queryKey: ['/api/production/plans', planId, 'assembly-station'] });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się zakończyć kompletowania", variant: "destructive" });
    },
  });

  const startPackingMutation = useMutation({
    mutationFn: async () => {
      const res = await apiRequest('POST', `/api/production/plans/${planId}/packing-operations/start`);
      return res.json();
    },
    onSuccess: (data: { message: string; updatedCount: number }) => {
      toast({ 
        title: data.updatedCount > 0 ? "Rozpoczęto pakowanie" : "Informacja", 
        description: data.message 
      });
      queryClient.invalidateQueries({ queryKey: ['/api/production/plans', planId, 'assembly-station'] });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się rozpocząć pakowania", variant: "destructive" });
    },
  });

  const completePackingMutation = useMutation({
    mutationFn: async () => {
      return apiRequest('POST', `/api/production/plans/${planId}/packing-operations/complete`);
    },
    onSuccess: () => {
      toast({ title: "Zakończono pakowanie", description: "Operacja pakowania została zakończona" });
      queryClient.invalidateQueries({ queryKey: ['/api/production/plans', planId, 'assembly-station'] });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się zakończyć pakowania", variant: "destructive" });
    },
  });

  const bulkAssembleMutation = useMutation({
    mutationFn: async ({ ids, isAssembled }: { ids: number[]; isAssembled: boolean }) => {
      const res = await apiRequest('PATCH', `/api/production/plans/${planId}/formatki/bulk-assemble`, { ids, isAssembled });
      return res.json();
    },
    onSuccess: (data: { message: string; updatedCount: number }) => {
      toast({ title: "Sukces", description: data.message });
      setSelectedFormatki(new Set());
      queryClient.invalidateQueries({ queryKey: ['/api/production/plans', planId, 'all-formatki'] });
      queryClient.invalidateQueries({ queryKey: ['/api/production/plans', planId, 'packing-products'] });
      queryClient.invalidateQueries({ queryKey: ['/api/production/plans', planId, 'assembly-station'] });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się zaktualizować formatek", variant: "destructive" });
    },
  });

  // Computed values for filters
  const uniqueColors = useMemo(() => {
    if (!formatki) return [];
    const colors = Array.from(new Set(formatki.map(f => f.colorCode).filter(Boolean)));
    return colors.sort();
  }, [formatki]);

  const uniqueZlps = useMemo(() => {
    if (!formatki) return [];
    const zlps = Array.from(new Set(formatki.map(f => f.zlpOrderNumber).filter(Boolean)));
    return zlps.sort();
  }, [formatki]);

  // Sorted formatki (filtering is done server-side)
  const filteredFormatki = useMemo(() => {
    if (!formatki) return [];
    let result = [...formatki];

    // Sort (client-side)
    result.sort((a, b) => {
      let valA: any = a[sortField as keyof FormatkaItem];
      let valB: any = b[sortField as keyof FormatkaItem];
      if (typeof valA === 'string') valA = valA.toLowerCase();
      if (typeof valB === 'string') valB = valB.toLowerCase();
      if (valA < valB) return sortDir === 'asc' ? -1 : 1;
      if (valA > valB) return sortDir === 'asc' ? 1 : -1;
      return 0;
    });

    return result;
  }, [formatki, sortField, sortDir]);

  // Reset filters
  const resetFilters = () => {
    setSearchText("");
    setFilterColor("all");
    setFilterZlp("all");
    setFilterAssembled("all");
    setFilterLength("");
    setFilterWidth("");
  };

  const toggleSort = (field: string) => {
    if (sortField === field) {
      setSortDir(sortDir === 'asc' ? 'desc' : 'asc');
    } else {
      setSortField(field);
      setSortDir('asc');
    }
  };

  const toggleFormatkaSelection = (id: number) => {
    setSelectedFormatki(prev => {
      const newSet = new Set(prev);
      if (newSet.has(id)) {
        newSet.delete(id);
      } else {
        newSet.add(id);
      }
      return newSet;
    });
  };

  const toggleAllVisible = () => {
    if (selectedFormatki.size === filteredFormatki.length) {
      setSelectedFormatki(new Set());
    } else {
      setSelectedFormatki(new Set(filteredFormatki.map(f => f.id)));
    }
  };

  const overallProgress = useMemo(() => {
    if (!assemblyData) return 0;
    const total = assemblyData.allFormatkiCount;
    if (total === 0) return 0;
    return Math.round((assemblyData.allFormatkiDone / total) * 100);
  }, [assemblyData]);

  // Można zawsze rozpocząć kompletowanie - formatki można odkładać na wózek sukcesywnie
  const canStartAssembly = useMemo(() => {
    if (!assemblyData) return false;
    return assemblyData.zlpColors.length > 0;
  }, [assemblyData]);

  // Można rozpocząć pakowanie gdy jest przynajmniej jeden ZLP z formatkami
  const canStartPacking = useMemo(() => {
    if (!assemblyData) return false;
    return assemblyData.zlpColors.length > 0;
  }, [assemblyData]);

  // Informacja o gotowości - ile ZLP jest w pełni gotowych
  const readyForAssemblyCount = useMemo(() => {
    if (!assemblyData) return 0;
    return assemblyData.zlpColors.filter(zlp => zlp.assemblyReady).length;
  }, [assemblyData]);

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
      </div>
    );
  }

  if (!assemblyData) {
    return (
      <div className="flex flex-col items-center justify-center h-full gap-4">
        <AlertTriangle className="h-12 w-12 text-muted-foreground" />
        <p className="text-muted-foreground">Nie znaleziono danych planu</p>
        <Link href="/production/plans">
          <Button variant="outline">
            <ArrowLeft className="h-4 w-4 mr-2" />
            Powrót do planów
          </Button>
        </Link>
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full">
      <div className="border-b bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60 px-6 py-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Link href={`/production/plans/${planId}`}>
              <Button variant="ghost" size="icon" data-testid="button-back-to-plan">
                <ArrowLeft className="h-4 w-4" />
              </Button>
            </Link>
            <div>
              <h1 className="text-xl font-semibold flex items-center gap-2">
                <Package className="h-5 w-5" />
                Stanowisko Kompletowania
              </h1>
              <p className="text-sm text-muted-foreground">
                {assemblyData.planNumber} - {assemblyData.planName}
              </p>
            </div>
          </div>
          
          <div className="flex items-center gap-4">
            <div className="text-right">
              <p className="text-sm text-muted-foreground">Postęp ogólny</p>
              <p className="text-lg font-semibold">{overallProgress}%</p>
            </div>
            <Progress value={overallProgress} className="w-32 h-2" />
          </div>
        </div>
      </div>

      <div className="flex-1 overflow-auto p-6">
        <Tabs value={section} onValueChange={(val) => navigate(`/production/plans/${planId}/assembly/${val}`)} className="h-full">
          <TabsList className="mb-4">
            <TabsTrigger value="overview" className="gap-2" data-testid="tab-overview">
              <ClipboardList className="h-4 w-4" />
              Przegląd
            </TabsTrigger>
            <TabsTrigger value="formatki" className="gap-2" data-testid="tab-formatki">
              <Layers className="h-4 w-4" />
              Formatki
            </TabsTrigger>
            <TabsTrigger value="assembly" className="gap-2" data-testid="tab-assembly">
              <Settings2 className="h-4 w-4" />
              Kompletowanie
            </TabsTrigger>
            <TabsTrigger value="packing" className="gap-2" data-testid="tab-packing">
              <PackageOpen className="h-4 w-4" />
              Pakowanie
            </TabsTrigger>
            <TabsTrigger value="carriers" className="gap-2" data-testid="tab-carriers">
              <Truck className="h-4 w-4" />
              Nośniki
            </TabsTrigger>
          </TabsList>

          <TabsContent value="overview" className="mt-0">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Palette className="h-5 w-5" />
                    Kolory ZLP w planie
                  </CardTitle>
                  <CardDescription>
                    Lista wszystkich zleceń produkcyjnych pogrupowanych po kolorze
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {assemblyData.zlpColors.map((zlp) => (
                      <div 
                        key={zlp.orderId}
                        className="flex items-center justify-between p-3 rounded-lg border bg-card"
                        data-testid={`zlp-color-${zlp.orderId}`}
                      >
                        <div className="flex items-center gap-3">
                          <div 
                            className="w-4 h-4 rounded-full border"
                            style={{ backgroundColor: getColorHex(zlp.colorCode) }}
                          />
                          <div>
                            <p className="font-medium">{zlp.orderNumber}</p>
                            <p className="text-sm text-muted-foreground">{zlp.colorName || zlp.colorCode}</p>
                          </div>
                        </div>
                        <div className="flex items-center gap-2">
                          <Badge 
                            variant={zlp.status === 'done' ? 'default' : 'secondary'}
                            className={statusColors[zlp.status]}
                          >
                            {statusLabels[zlp.status] || zlp.status}
                          </Badge>
                          {zlp.assemblyReady ? (
                            <CheckCircle2 className="h-4 w-4 text-green-500" />
                          ) : (
                            <Clock className="h-4 w-4 text-yellow-500" />
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Box className="h-5 w-5" />
                    Podsumowanie formatek
                  </CardTitle>
                  <CardDescription>
                    Status wszystkich formatek ze wszystkich kolorów
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="grid grid-cols-3 gap-4">
                      <div className="text-center p-4 rounded-lg bg-muted/50">
                        <p className="text-2xl font-bold">{assemblyData.allFormatkiCount}</p>
                        <p className="text-sm text-muted-foreground">Wszystkie</p>
                      </div>
                      <div className="text-center p-4 rounded-lg bg-green-500/10">
                        <p className="text-2xl font-bold text-green-600">{assemblyData.allFormatkiDone}</p>
                        <p className="text-sm text-muted-foreground">Ukończone</p>
                      </div>
                      <div className="text-center p-4 rounded-lg bg-red-500/10">
                        <p className="text-2xl font-bold text-red-600">{assemblyData.allFormatkiDamaged}</p>
                        <p className="text-sm text-muted-foreground">Uszkodzone</p>
                      </div>
                    </div>
                    
                    <Separator />
                    
                    <div className="space-y-2">
                      <div className="flex justify-between text-sm">
                        <span>Postęp</span>
                        <span>{assemblyData.allFormatkiDone} / {assemblyData.allFormatkiCount}</span>
                      </div>
                      <Progress value={overallProgress} className="h-3" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="lg:col-span-2">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Settings2 className="h-5 w-5" />
                    Sterowanie operacjami
                  </CardTitle>
                  <CardDescription>
                    Zarządzaj kompletowaniem i pakowaniem na poziomie całego planu
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div className="space-y-4">
                      <h4 className="font-medium flex items-center gap-2">
                        <Settings2 className="h-4 w-4" />
                        Kompletowanie
                      </h4>
                      <div className="flex items-start gap-2 p-3 rounded-lg bg-blue-500/10 text-blue-700 dark:text-blue-400">
                        <Clock className="h-4 w-4 mt-0.5 shrink-0" />
                        <p className="text-sm">
                          Gotowe do kompletowania: {readyForAssemblyCount} / {assemblyData.zlpColors.length} kolorów.
                          Możesz rozpocząć kompletowanie i odkładać formatki na wózek sukcesywnie.
                        </p>
                      </div>
                      <div className="flex gap-2">
                        <Button 
                          onClick={() => startAssemblyMutation.mutate()}
                          disabled={!canStartAssembly || startAssemblyMutation.isPending}
                          className="flex-1"
                          data-testid="button-start-assembly"
                        >
                          {startAssemblyMutation.isPending ? (
                            <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                          ) : (
                            <Play className="h-4 w-4 mr-2" />
                          )}
                          Rozpocznij kompletowanie
                        </Button>
                        <Button 
                          variant="outline"
                          onClick={() => completeAssemblyMutation.mutate()}
                          disabled={completeAssemblyMutation.isPending}
                          data-testid="button-complete-assembly"
                        >
                          {completeAssemblyMutation.isPending ? (
                            <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                          ) : (
                            <Square className="h-4 w-4 mr-2" />
                          )}
                          Zakończ
                        </Button>
                      </div>
                    </div>

                    <div className="space-y-4">
                      <h4 className="font-medium flex items-center gap-2">
                        <PackageOpen className="h-4 w-4" />
                        Pakowanie
                      </h4>
                      <div className="flex items-start gap-2 p-3 rounded-lg bg-blue-500/10 text-blue-700 dark:text-blue-400">
                        <Clock className="h-4 w-4 mt-0.5 shrink-0" />
                        <p className="text-sm">
                          Pakowanie można rozpocząć równolegle z kompletowaniem - pakuj skompletowane zestawy.
                        </p>
                      </div>
                      <div className="flex gap-2">
                        <Button 
                          onClick={() => startPackingMutation.mutate()}
                          disabled={!canStartPacking || startPackingMutation.isPending}
                          className="flex-1"
                          data-testid="button-start-packing"
                        >
                          {startPackingMutation.isPending ? (
                            <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                          ) : (
                            <Play className="h-4 w-4 mr-2" />
                          )}
                          Rozpocznij pakowanie
                        </Button>
                        <Button 
                          variant="outline"
                          onClick={() => completePackingMutation.mutate()}
                          disabled={completePackingMutation.isPending}
                          data-testid="button-complete-packing"
                        >
                          {completePackingMutation.isPending ? (
                            <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                          ) : (
                            <Square className="h-4 w-4 mr-2" />
                          )}
                          Zakończ
                        </Button>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="formatki" className="mt-0">
            <Card>
              <CardHeader>
                <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
                  <div>
                    <CardTitle className="flex items-center gap-2">
                      <Layers className="h-5 w-5" />
                      Wszystkie formatki planu
                    </CardTitle>
                    <CardDescription>
                      Zaznacz formatki i oznacz jako skompletowane
                    </CardDescription>
                  </div>
                  
                  {/* Bulk actions */}
                  {selectedFormatki.size > 0 && (
                    <div className="flex items-center gap-2 p-2 rounded-lg bg-primary/10">
                      <span className="text-sm font-medium">
                        Zaznaczono: {selectedFormatki.size}
                      </span>
                      <Button
                        size="sm"
                        onClick={() => bulkAssembleMutation.mutate({ ids: Array.from(selectedFormatki), isAssembled: true })}
                        disabled={bulkAssembleMutation.isPending}
                        data-testid="button-bulk-assemble"
                      >
                        {bulkAssembleMutation.isPending ? (
                          <Loader2 className="h-4 w-4 mr-1 animate-spin" />
                        ) : (
                          <Check className="h-4 w-4 mr-1" />
                        )}
                        Oznacz jako skompletowane
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => bulkAssembleMutation.mutate({ ids: Array.from(selectedFormatki), isAssembled: false })}
                        disabled={bulkAssembleMutation.isPending}
                        data-testid="button-bulk-unassemble"
                      >
                        <X className="h-4 w-4 mr-1" />
                        Cofnij
                      </Button>
                    </div>
                  )}
                </div>
              </CardHeader>
              <CardContent>
                {/* Filters */}
                <div className="space-y-3 mb-4 p-3 rounded-lg bg-muted/50">
                  {/* Search row */}
                  <div className="flex flex-wrap gap-3 items-center">
                    <div className="flex items-center gap-2">
                      <Filter className="h-4 w-4 text-muted-foreground" />
                      <span className="text-sm text-muted-foreground">Szukaj:</span>
                    </div>
                    <Input
                      placeholder="Szukaj (użyj ; dla wielu np. bo;cza)"
                      value={searchText}
                      onChange={(e) => setSearchText(e.target.value)}
                      className="w-[300px]"
                      data-testid="input-search-text"
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={resetFilters}
                      data-testid="button-reset-filters"
                    >
                      <X className="h-4 w-4 mr-1" />
                      Wyczyść filtry
                    </Button>
                    <div className="flex-1" />
                    <div className="text-sm text-muted-foreground">
                      Wyświetlono: {filteredFormatki.length}
                    </div>
                  </div>
                  
                  {/* Filters row */}
                  <div className="flex flex-wrap gap-3 items-center">
                    <span className="text-sm text-muted-foreground">Filtry:</span>
                    <Select value={filterColor} onValueChange={setFilterColor}>
                      <SelectTrigger className="w-[150px]" data-testid="select-filter-color">
                        <SelectValue placeholder="Kolor" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">Wszystkie kolory</SelectItem>
                        {uniqueColors.map(color => (
                          <SelectItem key={color} value={color}>{color}</SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <Select value={filterZlp} onValueChange={setFilterZlp}>
                      <SelectTrigger className="w-[180px]" data-testid="select-filter-zlp">
                        <SelectValue placeholder="ZLP" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">Wszystkie ZLP</SelectItem>
                        {uniqueZlps.map(zlp => (
                          <SelectItem key={zlp} value={zlp}>{zlp}</SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <Select value={filterAssembled} onValueChange={setFilterAssembled}>
                      <SelectTrigger className="w-[180px]" data-testid="select-filter-assembled">
                        <SelectValue placeholder="Status" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">Wszystkie</SelectItem>
                        <SelectItem value="false">Do skompletowania</SelectItem>
                        <SelectItem value="true">Skompletowane</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  
                  {/* Dimension filters row */}
                  <div className="flex flex-wrap gap-3 items-center">
                    <span className="text-sm text-muted-foreground">Wymiary:</span>
                    <div className="flex items-center gap-1">
                      <span className="text-xs text-muted-foreground">Długość:</span>
                      <Input
                        type="number"
                        placeholder="np. 500"
                        value={filterLength}
                        onChange={(e) => setFilterLength(e.target.value)}
                        className="w-[100px]"
                        data-testid="input-filter-length"
                      />
                    </div>
                    <div className="flex items-center gap-1">
                      <span className="text-xs text-muted-foreground">Szerokość:</span>
                      <Input
                        type="number"
                        placeholder="np. 300"
                        value={filterWidth}
                        onChange={(e) => setFilterWidth(e.target.value)}
                        className="w-[100px]"
                        data-testid="input-filter-width"
                      />
                    </div>
                  </div>
                </div>

                {formatkiLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
                  </div>
                ) : filteredFormatki.length === 0 ? (
                  <p className="text-center text-muted-foreground py-8">
                    {formatki?.length === 0 ? 'Brak formatek w tym planie' : 'Brak formatek pasujących do filtrów'}
                  </p>
                ) : (
                  <div className="border rounded-lg overflow-hidden">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead className="w-[50px]">
                            <Checkbox 
                              checked={selectedFormatki.size === filteredFormatki.length && filteredFormatki.length > 0}
                              onCheckedChange={toggleAllVisible}
                              data-testid="checkbox-select-all"
                            />
                          </TableHead>
                          <TableHead 
                            className="cursor-pointer hover-elevate"
                            onClick={() => toggleSort('componentName')}
                          >
                            <div className="flex items-center gap-1">
                              Nazwa
                              <ArrowUpDown className="h-3 w-3" />
                            </div>
                          </TableHead>
                          <TableHead 
                            className="cursor-pointer hover-elevate"
                            onClick={() => toggleSort('colorCode')}
                          >
                            <div className="flex items-center gap-1">
                              Kolor
                              <ArrowUpDown className="h-3 w-3" />
                            </div>
                          </TableHead>
                          <TableHead>Wymiary</TableHead>
                          <TableHead 
                            className="cursor-pointer hover-elevate"
                            onClick={() => toggleSort('zlpOrderNumber')}
                          >
                            <div className="flex items-center gap-1">
                              ZLP
                              <ArrowUpDown className="h-3 w-3" />
                            </div>
                          </TableHead>
                          <TableHead className="text-center">Status</TableHead>
                          <TableHead className="w-[100px] text-center">Akcje</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {filteredFormatki.map((formatka) => (
                          <TableRow 
                            key={formatka.id}
                            className={`${formatka.isDamaged ? 'bg-red-500/10' : ''} ${formatka.isAssembled ? 'bg-green-500/5' : ''}`}
                            data-testid={`formatka-row-${formatka.id}`}
                          >
                            <TableCell>
                              <Checkbox 
                                checked={selectedFormatki.has(formatka.id)}
                                onCheckedChange={() => toggleFormatkaSelection(formatka.id)}
                                data-testid={`checkbox-formatka-${formatka.id}`}
                              />
                            </TableCell>
                            <TableCell>
                              <div className="font-medium">{formatka.componentName}</div>
                              <div className="text-xs text-muted-foreground">{formatka.componentType}</div>
                            </TableCell>
                            <TableCell>
                              <div className="flex items-center gap-2">
                                <div 
                                  className="w-3 h-3 rounded-full border"
                                  style={{ backgroundColor: getColorHex(formatka.colorCode) }}
                                />
                                <span className="text-sm">{formatka.colorCode}</span>
                              </div>
                            </TableCell>
                            <TableCell>
                              <span className="text-sm">
                                {formatka.length} x {formatka.width} x {formatka.thickness}
                              </span>
                            </TableCell>
                            <TableCell>
                              <span className="text-sm font-mono">{formatka.zlpOrderNumber}</span>
                            </TableCell>
                            <TableCell>
                              <div className="flex items-center justify-center gap-1">
                                {formatka.isDamaged && (
                                  <Badge variant="destructive" className="text-xs">
                                    <AlertTriangle className="h-3 w-3 mr-1" />
                                    Uszkodzona
                                  </Badge>
                                )}
                                {formatka.isAssembled ? (
                                  <Badge className="bg-green-500 text-xs">
                                    <CheckCircle2 className="h-3 w-3 mr-1" />
                                    Skompletowana
                                  </Badge>
                                ) : (
                                  <Badge variant="secondary" className="text-xs">
                                    <Clock className="h-3 w-3 mr-1" />
                                    Oczekuje
                                  </Badge>
                                )}
                              </div>
                            </TableCell>
                            <TableCell>
                              <div className="flex items-center justify-center">
                                <Button
                                  size="sm"
                                  variant={formatka.isAssembled ? "outline" : "default"}
                                  onClick={() => bulkAssembleMutation.mutate({ ids: [formatka.id], isAssembled: !formatka.isAssembled })}
                                  disabled={bulkAssembleMutation.isPending}
                                  data-testid={`button-toggle-formatka-${formatka.id}`}
                                >
                                  {formatka.isAssembled ? (
                                    <>
                                      <X className="h-3 w-3 mr-1" />
                                      Cofnij
                                    </>
                                  ) : (
                                    <>
                                      <Check className="h-3 w-3 mr-1" />
                                      Gotowe
                                    </>
                                  )}
                                </Button>
                              </div>
                            </TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="assembly" className="mt-0">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Settings2 className="h-5 w-5" />
                  Panel kompletowania
                </CardTitle>
                <CardDescription>
                  Status kompletowania dla każdego koloru ZLP
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-6">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {assemblyData.zlpColors.map((zlp) => (
                      <div 
                        key={zlp.orderId}
                        className={`p-4 rounded-lg border ${
                          zlp.assemblyReady ? 'border-green-500/30 bg-green-500/5' : 'border-yellow-500/30 bg-yellow-500/5'
                        }`}
                        data-testid={`assembly-zlp-${zlp.orderId}`}
                      >
                        <div className="flex items-center justify-between mb-3">
                          <div className="flex items-center gap-2">
                            <div 
                              className="w-4 h-4 rounded-full border"
                              style={{ backgroundColor: getColorHex(zlp.colorCode) }}
                            />
                            <span className="font-semibold">{zlp.orderNumber}</span>
                            <span className="text-sm text-muted-foreground">{zlp.colorName || zlp.colorCode}</span>
                          </div>
                          {zlp.assemblyReady ? (
                            <Badge className="bg-green-500">
                              <CheckCircle2 className="h-3 w-3 mr-1" />
                              Gotowe
                            </Badge>
                          ) : (
                            <Badge variant="secondary">
                              <Clock className="h-3 w-3 mr-1" />
                              Oczekuje
                            </Badge>
                          )}
                        </div>
                        <div className="space-y-2">
                          <div className="flex justify-between text-sm">
                            <span className="text-muted-foreground">Formatki:</span>
                            <span>{zlp.componentsDone} / {zlp.componentsTotal}</span>
                          </div>
                          <Progress 
                            value={zlp.componentsTotal > 0 ? (zlp.componentsDone / zlp.componentsTotal) * 100 : 0} 
                            className="h-2" 
                          />
                        </div>
                      </div>
                    ))}
                  </div>
                  
                  <Separator />
                  
                  <div className="flex items-center justify-between p-4 rounded-lg bg-muted/50">
                    <div>
                      <p className="font-medium">Operacje kompletowania dla całego planu</p>
                      <p className="text-sm text-muted-foreground">
                        Rozpocznij lub zakończ kompletowanie dla wszystkich kolorów naraz
                      </p>
                    </div>
                    <div className="flex gap-2">
                      <Button 
                        onClick={() => startAssemblyMutation.mutate()}
                        disabled={!canStartAssembly || startAssemblyMutation.isPending}
                        data-testid="button-start-assembly-panel"
                      >
                        {startAssemblyMutation.isPending ? (
                          <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                        ) : (
                          <Play className="h-4 w-4 mr-2" />
                        )}
                        Rozpocznij
                      </Button>
                      <Button 
                        variant="outline"
                        onClick={() => completeAssemblyMutation.mutate()}
                        disabled={completeAssemblyMutation.isPending}
                        data-testid="button-complete-assembly-panel"
                      >
                        {completeAssemblyMutation.isPending ? (
                          <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                        ) : (
                          <CheckCircle2 className="h-4 w-4 mr-2" />
                        )}
                        Zakończ
                      </Button>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="packing" className="mt-0">
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div>
                    <CardTitle className="flex items-center gap-2">
                      <PackageOpen className="h-5 w-5" />
                      Produkty do pakowania
                    </CardTitle>
                    <CardDescription>
                      Lista produktów z ich formatkami do skompletowania
                    </CardDescription>
                  </div>
                  <div className="flex gap-2">
                    <Button 
                      onClick={() => startPackingMutation.mutate()}
                      disabled={!canStartPacking || startPackingMutation.isPending}
                      data-testid="button-start-packing-panel"
                    >
                      {startPackingMutation.isPending ? (
                        <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                      ) : (
                        <Play className="h-4 w-4 mr-2" />
                      )}
                      Rozpocznij pakowanie
                    </Button>
                    <Button 
                      variant="outline"
                      onClick={() => completePackingMutation.mutate()}
                      disabled={completePackingMutation.isPending}
                      data-testid="button-complete-packing-panel"
                    >
                      {completePackingMutation.isPending ? (
                        <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                      ) : (
                        <CheckCircle2 className="h-4 w-4 mr-2" />
                      )}
                      Zakończ
                    </Button>
                  </div>
                </div>
              </CardHeader>
              <CardContent>
                {packingProductsLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
                  </div>
                ) : packingProducts.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    <Package className="h-12 w-12 mx-auto mb-4 opacity-50" />
                    <p>Brak produktów w tym planie</p>
                  </div>
                ) : (
                  <div className="space-y-4">
                    {(() => {
                      const groupedByOrder = packingProducts.reduce((acc: Record<string, typeof packingProducts>, product) => {
                        const baseOrderNumber = product.sourceReference ? product.sourceReference.split(' #')[0] : 'Z_KATALOGU';
                        if (!acc[baseOrderNumber]) acc[baseOrderNumber] = [];
                        acc[baseOrderNumber].push(product);
                        return acc;
                      }, {});
                      
                      return Object.entries(groupedByOrder).map(([orderNumber, orderProducts]) => (
                        <Collapsible key={orderNumber} defaultOpen={true} data-testid={`order-group-${orderNumber}`}>
                          <CollapsibleTrigger asChild>
                            <div className="w-full bg-muted/50 border rounded-lg p-3 hover-elevate cursor-pointer">
                              <div className="flex items-center justify-between">
                                <div className="flex items-center gap-2">
                                  <ChevronRight className="h-4 w-4 text-muted-foreground" />
                                  <ShoppingCart className="h-4 w-4 text-primary" />
                                  {orderNumber === 'Z_KATALOGU' ? (
                                    <span className="font-medium text-orange-500">Z katalogu</span>
                                  ) : (
                                    <Link href={`/order/${orderNumber}`} className="font-medium text-blue-500 hover:underline" onClick={(e) => e.stopPropagation()}>
                                      Zamówienie: {orderNumber}
                                    </Link>
                                  )}
                                  <Badge variant="secondary">{orderProducts.length} {orderProducts.length === 1 ? 'produkt' : 'produktów'}</Badge>
                                </div>
                                <div className="flex items-center gap-2">
                                  {(() => {
                                    const totalFormatki = orderProducts.reduce((sum, p) => sum + p.formatkiTotal, 0);
                                    const assembledFormatki = orderProducts.reduce((sum, p) => sum + p.formatkiAssembled, 0);
                                    return totalFormatki > 0 ? (
                                      <Badge variant={assembledFormatki === totalFormatki ? "default" : "secondary"} className={assembledFormatki === totalFormatki ? "bg-green-500" : ""}>
                                        {assembledFormatki}/{totalFormatki}
                                      </Badge>
                                    ) : null;
                                  })()}
                                </div>
                              </div>
                            </div>
                          </CollapsibleTrigger>
                          <CollapsibleContent>
                            <div className="ml-4 mt-2 space-y-2 border-l-2 border-muted pl-4">
                              {orderProducts.map((product) => (
                                <Collapsible key={product.lineId} defaultOpen={false} data-testid={`packing-product-${product.lineId}`}>
                                  <CollapsibleTrigger asChild>
                                    <div className="w-full border rounded-lg p-3 hover-elevate cursor-pointer">
                                      <div className="flex items-center gap-3">
                                        <ChevronRight className="h-4 w-4 text-muted-foreground" />
                                        {product.productImage ? (
                                          <img src={product.productImage} alt={product.productTitle} className="w-12 h-12 object-contain rounded border bg-white" />
                                        ) : (
                                          <div className="w-12 h-12 rounded border bg-muted flex items-center justify-center">
                                            <ImageIcon className="h-5 w-5 text-muted-foreground" />
                                          </div>
                                        )}
                                        <div className="flex-1 min-w-0">
                                          <p className="font-medium truncate">{product.productTitle}</p>
                                          <p className="text-sm text-muted-foreground">
                                            <span className="font-mono text-xs bg-muted px-1 rounded">#{product.lineId}</span>
                                            {' '}SKU: {product.productSku} | Ilość: {product.quantity} szt.
                                          </p>
                                          {product.sourceReference?.includes('#') && (
                                            <p className="text-xs text-muted-foreground">Sztuka: {product.sourceReference.split(' ')[1]}</p>
                                          )}
                                        </div>
                                        <div className="flex items-center gap-2 flex-shrink-0">
                                          {product.reservedQuantity > 0 && (
                                            <Badge variant="outline" className="bg-blue-500/10 border-blue-500 text-blue-500">
                                              <Box className="h-3 w-3 mr-1" />Z magazynu
                                            </Badge>
                                          )}
                                          {product.formatkiTotal > 0 && (
                                            <Badge variant={product.formatkiAssembled === product.formatkiTotal ? "default" : "secondary"} className={product.formatkiAssembled === product.formatkiTotal ? "bg-green-500" : ""}>
                                              {product.formatkiAssembled}/{product.formatkiTotal}
                                            </Badge>
                                          )}
                                          {product.formatkiDamaged > 0 && <Badge variant="destructive"><AlertTriangle className="h-3 w-3" /></Badge>}
                                          {(() => {
                                            const allFromWarehouse = product.formatki.length > 0 && product.formatki.every((f: any) => f.isFromWarehouse);
                                            const editableFormatki = product.formatki.filter((f: any) => !f.isFromWarehouse);
                                            return allFromWarehouse ? (
                                              <Badge className="bg-green-500">
                                                <CheckCircle2 className="h-3 w-3 mr-1" />Spakowano
                                              </Badge>
                                            ) : (
                                              <Button
                                                size="sm"
                                                variant={product.formatkiAssembled === product.formatkiTotal ? "outline" : "default"}
                                                className={product.formatkiAssembled === product.formatkiTotal ? "bg-green-500/20 border-green-500" : ""}
                                                onClick={(e) => { e.stopPropagation(); bulkAssembleMutation.mutate({ ids: editableFormatki.map((f: any) => f.id), isAssembled: product.formatkiAssembled !== product.formatkiTotal }); }}
                                                disabled={bulkAssembleMutation.isPending || editableFormatki.length === 0}
                                                data-testid={`button-pack-product-${product.lineId}`}
                                              >
                                                {product.formatkiAssembled === product.formatkiTotal ? <><CheckCircle2 className="h-3 w-3 mr-1" />Spakowano</> : <><Package className="h-3 w-3 mr-1" />Spakuj</>}
                                              </Button>
                                            );
                                          })()}
                                        </div>
                                      </div>
                                    </div>
                                  </CollapsibleTrigger>
                                  <CollapsibleContent>
                                    {product.formatki.length > 0 ? (
                                      <div className="ml-6 mt-1 border-l-2 border-muted pl-4 space-y-1">
                                        {product.reservedQuantity > 0 && (
                                          <div className="p-2 bg-blue-500/10 rounded text-xs text-blue-500 flex items-center gap-2">
                                            <Box className="h-3 w-3" />
                                            <span>Produkt z magazynu - formatki zostały już wcześniej skompletowane i spakowane</span>
                                          </div>
                                        )}
                                        {product.formatki.map((formatka) => (
                                          <div key={formatka.id} className={`p-2 rounded flex items-center justify-between ${formatka.isDamaged ? 'bg-red-500/10' : formatka.isAssembled ? 'bg-green-500/10' : 'bg-muted/30'}`} data-testid={`packing-formatka-${formatka.id}`}>
                                            <div className="flex items-center gap-2">
                                              <div className="w-3 h-3 rounded-full border" style={{ backgroundColor: getColorHex(formatka.colorCode) }} />
                                              <span className="font-medium text-sm">{formatka.componentName}</span>
                                              <span className="text-xs text-muted-foreground">{formatka.length}×{formatka.width}×{formatka.thickness}</span>
                                              {formatka.zlpOrderNumber && <Badge variant="outline" className="text-xs font-mono">{formatka.zlpOrderNumber}</Badge>}
                                              {formatka.isFromWarehouse && <Badge variant="outline" className="text-xs bg-blue-500/10 border-blue-500 text-blue-500">z magazynu</Badge>}
                                            </div>
                                            <div className="flex items-center gap-2">
                                              {formatka.isDamaged && <Badge variant="destructive" className="text-xs">Uszkodzona</Badge>}
                                              {formatka.isFromWarehouse ? (
                                                <Badge className="bg-green-500 text-xs">
                                                  <CheckCircle2 className="h-3 w-3 mr-1" />OK
                                                </Badge>
                                              ) : (
                                                <Button size="sm" variant={formatka.isAssembled ? "outline" : "default"} onClick={(e) => { e.stopPropagation(); bulkAssembleMutation.mutate({ ids: [formatka.id], isAssembled: !formatka.isAssembled }); }} disabled={bulkAssembleMutation.isPending} data-testid={`button-toggle-packing-formatka-${formatka.id}`}>
                                                  {formatka.isAssembled ? <><X className="h-3 w-3 mr-1" />Cofnij</> : <><Check className="h-3 w-3 mr-1" />OK</>}
                                                </Button>
                                              )}
                                            </div>
                                          </div>
                                        ))}
                                      </div>
                                    ) : (
                                      <div className="ml-6 mt-1 p-2 text-sm text-muted-foreground">Brak formatek dla tego produktu</div>
                                    )}
                                  </CollapsibleContent>
                                </Collapsible>
                              ))}
                            </div>
                          </CollapsibleContent>
                        </Collapsible>
                      ));
                    })()}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="carriers" className="mt-0">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Truck className="h-5 w-5" />
                  Nośniki i kolejność pakowania
                </CardTitle>
                <CardDescription>
                  Przypisz nośniki transportowe do produktów planu
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div className="p-4 rounded-lg border bg-muted/30">
                    <div className="flex items-center gap-2 text-muted-foreground">
                      <Truck className="h-5 w-5" />
                      <p>Nośniki zostaną przypisane automatycznie podczas generowania ZLP lub można je przypisać ręcznie z poziomu szczegółów ZLP.</p>
                    </div>
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    {assemblyData.zlpColors.map((zlp) => (
                      <Link key={zlp.orderId} href={`/production/orders/${zlp.orderId}`}>
                        <div className="p-4 rounded-lg border hover-elevate cursor-pointer" data-testid={`carrier-zlp-${zlp.orderId}`}>
                          <div className="flex items-center gap-2 mb-2">
                            <div 
                              className="w-3 h-3 rounded-full border"
                              style={{ backgroundColor: getColorHex(zlp.colorCode) }}
                            />
                            <span className="font-medium">{zlp.orderNumber}</span>
                          </div>
                          <p className="text-sm text-muted-foreground">{zlp.colorName || zlp.colorCode}</p>
                          <div className="mt-2 text-xs text-muted-foreground">
                            Kliknij aby zarządzać nośnikami
                          </div>
                        </div>
                      </Link>
                    ))}
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}

function getColorHex(colorCode: string): string {
  const colorMap: Record<string, string> = {
    'CZARNY': '#1a1a1a',
    'BIALY': '#ffffff',
    'SZARY': '#808080',
    'DĄB': '#8b6914',
    'ANTRACYT': '#383838',
    'ORZECH': '#5d4037',
    'SONOMA': '#c9a66b',
    'HDF-BIALY': '#f5f5f5',
    'HDF': '#d4c4a8',
  };
  return colorMap[colorCode?.toUpperCase()] || '#6b7280';
}
