import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ArrowLeft, Package, Hash, Calendar, MapPin, AlertTriangle, CheckCircle, Clock, Lock, Plus, Trash2, FileCheck, History, ArrowRight, User, ClipboardList, Truck, RotateCcw, Ban } from "lucide-react";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { useState } from "react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Link } from "wouter";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { DictionaryCombobox } from "@/components/dictionary-combobox";
import type { ProductCreatorDictionary } from "@shared/schema";

interface PackedProductItem {
  id: number;
  serialNumber: string;
  productSku: string;
  status: 'available' | 'reserved' | 'shipped' | 'returned' | 'defective' | 'scrapped';
  packedAt: string;
  producedAt?: string;
  consumedAt?: string;
  reservedAt?: string;
  reservedForPlanLineId?: number;
  reservedForPlanId?: number;
  locationId?: number;
  carrierId?: number;
  locationName?: string;
  carrierName?: string;
  defectDescription?: string;
  notes?: string;
  bomId?: number;
  productionPlanLineId?: number;
  metadata?: {
    auditHistory?: Array<{
      action: string;
      reason?: string;
      timestamp: string;
      performedBy?: string;
      previousStatus?: string;
      planId?: number;
      planLineId?: number;
      orderId?: string;
      locationFrom?: string;
      locationTo?: string;
    }>;
  };
  sourceType?: 'inventory_count' | 'production' | 'manual';
  sourceId?: number;
  updatedAt?: string;
}

interface PackedProduct {
  id: number;
  productSku: string;
  productName: string;
  quantity: number;
  reservedQuantity: number;
  locationName?: string;
  carrierName?: string;
  imageUrl?: string;
}

interface ItemsResponse {
  items: PackedProductItem[];
  stats: {
    available: number;
    reserved: number;
    shipped: number;
    returned: number;
    defective: number;
    scrapped: number;
    total: number;
  };
  pagination: {
    limit: number;
    offset: number;
    total: number;
  };
}

interface Location {
  id: number;
  name: string;
}

interface Carrier {
  id: number;
  name: string;
}

const statusConfig: Record<string, { label: string; variant: "default" | "secondary" | "destructive" | "outline"; icon: any }> = {
  available: { label: "Dostępny", variant: "default", icon: CheckCircle },
  reserved: { label: "Zarezerwowany", variant: "secondary", icon: Lock },
  shipped: { label: "Wysłany", variant: "outline", icon: Package },
  returned: { label: "Zwrócony", variant: "secondary", icon: Clock },
  defective: { label: "Wadliwy", variant: "destructive", icon: AlertTriangle },
  scrapped: { label: "Zezłomowany", variant: "destructive", icon: Trash2 },
};

export default function PackedProductItemsPage() {
  const params = useParams<{ id: string }>();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const productId = parseInt(params.id || "0");
  
  const [activeTab, setActiveTab] = useState<string>("all");
  const [isScrapDialogOpen, setIsScrapDialogOpen] = useState(false);
  const [selectedItemId, setSelectedItemId] = useState<number | null>(null);
  const [scrapReasonCode, setScrapReasonCode] = useState("");
  const [scrapReasonCustom, setScrapReasonCustom] = useState("");
  const [isHistoryDialogOpen, setIsHistoryDialogOpen] = useState(false);
  const [historyItem, setHistoryItem] = useState<PackedProductItem | null>(null);
  
  const { data: product, isLoading: isProductLoading } = useQuery<PackedProduct>({
    queryKey: [`/api/warehouse/packed-products/${productId}`],
    enabled: !!productId,
  });
  
  const { data: itemsData, isLoading: isItemsLoading } = useQuery<ItemsResponse>({
    queryKey: [`/api/warehouse/packed-products/${productId}/items`, activeTab],
    queryFn: async () => {
      const statusParam = activeTab !== 'all' ? `?status=${activeTab}` : '';
      const response = await fetch(`/api/warehouse/packed-products/${productId}/items${statusParam}`, {
        credentials: 'include',
      });
      if (!response.ok) throw new Error('Failed to fetch items');
      return response.json();
    },
    enabled: !!productId,
  });
  
  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ["/api/production/locations"],
  });
  
  const { data: carriers = [] } = useQuery<Carrier[]>({
    queryKey: ["/api/production/carriers"],
  });
  
  const { data: scrapReasons = [] } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries", { type: "scrap_reason_product" }],
  });
  
  const createInventoryCountMutation = useMutation({
    mutationFn: async () => {
      const today = new Date();
      const day = String(today.getDate()).padStart(2, '0');
      const month = String(today.getMonth() + 1).padStart(2, '0');
      const year = today.getFullYear();
      const countName = `Spis ${product?.productName || 'produktu'} ${day}-${month}-${year}`;
      
      return await apiRequest("POST", `/api/warehouse/packed-products/${productId}/create-inventory-count`, {
        name: countName,
      });
    },
    onSuccess: (data: any) => {
      toast({
        title: "Sukces",
        description: "Utworzono spis inwentaryzacyjny",
      });
      if (data.inventoryCountId) {
        setLocation(`/warehouse/inventory-counts/${data.inventoryCountId}`);
      }
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się utworzyć spisu inwentaryzacyjnego",
        variant: "destructive",
      });
    },
  });
  
  const scrapItemMutation = useMutation({
    mutationFn: async ({ itemId, reason }: { itemId: number; reason: string }) => {
      return await apiRequest("POST", `/api/warehouse/packed-products/items/${itemId}/scrap`, { reason });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/packed-products/${productId}/items`] });
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/packed-products/${productId}`] });
      toast({
        title: "Sukces",
        description: "Sztuka została zezłomowana",
      });
      setIsScrapDialogOpen(false);
      setSelectedItemId(null);
      setScrapReasonCode("");
      setScrapReasonCustom("");
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się zezłomować sztuki",
        variant: "destructive",
      });
    },
  });

  const formatDate = (dateString?: string) => {
    if (!dateString) return "-";
    return format(new Date(dateString), "dd.MM.yyyy HH:mm", { locale: pl });
  };

  const getStatusBadge = (status: string) => {
    const config = statusConfig[status] || statusConfig.available;
    const Icon = config.icon;
    return (
      <Badge variant={config.variant} className="gap-1">
        <Icon className="w-3 h-3" />
        {config.label}
      </Badge>
    );
  };

  const buildHistoryTimeline = (item: PackedProductItem) => {
    const events: Array<{
      action: string;
      timestamp: string;
      performedBy?: string;
      details?: string;
      icon: any;
      color: string;
      planId?: number;
      orderId?: string;
    }> = [];

    if (item.packedAt) {
      const sourceLabel = item.sourceType === 'inventory_count' 
        ? `ze spisu #${item.sourceId}` 
        : item.sourceType === 'production' 
          ? `z produkcji #${item.sourceId}` 
          : 'ręcznie';
      events.push({
        action: 'created',
        timestamp: item.packedAt,
        details: `Dodano do magazynu ${sourceLabel}`,
        icon: Plus,
        color: 'text-green-500',
      });
    }

    if (item.metadata?.auditHistory) {
      item.metadata.auditHistory.forEach(entry => {
        let icon = History;
        let color = 'text-muted-foreground';
        let details = '';

        switch (entry.action) {
          case 'reserved':
            icon = Lock;
            color = 'text-blue-500';
            details = entry.planId 
              ? `Zarezerwowano do planu #${entry.planId}` 
              : 'Zarezerwowano';
            if (entry.orderId) {
              details += ` (zamówienie: ${entry.orderId})`;
            }
            break;
          case 'unreserved':
            icon = RotateCcw;
            color = 'text-orange-500';
            details = 'Anulowano rezerwację';
            break;
          case 'shipped':
            icon = Truck;
            color = 'text-purple-500';
            details = entry.orderId 
              ? `Wysłano z zamówieniem ${entry.orderId}` 
              : 'Wysłano';
            break;
          case 'returned':
            icon = RotateCcw;
            color = 'text-yellow-500';
            details = entry.reason || 'Zwrócono';
            break;
          case 'moved':
            icon = ArrowRight;
            color = 'text-cyan-500';
            details = entry.locationFrom && entry.locationTo
              ? `Przeniesiono z ${entry.locationFrom} do ${entry.locationTo}`
              : 'Przeniesiono';
            break;
          case 'scrapped':
            icon = Trash2;
            color = 'text-red-500';
            details = entry.reason?.replace(/^\[.*?\]\s*/, '') || 'Zezłomowano';
            break;
          case 'defective':
            icon = AlertTriangle;
            color = 'text-red-400';
            details = entry.reason || 'Oznaczono jako wadliwy';
            break;
          default:
            details = entry.action;
        }

        events.push({
          action: entry.action,
          timestamp: entry.timestamp,
          performedBy: entry.performedBy,
          details,
          icon,
          color,
          planId: entry.planId,
          orderId: entry.orderId,
        });
      });
    }

    if (item.reservedAt && !events.some(e => e.action === 'reserved')) {
      events.push({
        action: 'reserved',
        timestamp: item.reservedAt,
        details: item.reservedForPlanId 
          ? `Zarezerwowano do planu #${item.reservedForPlanId}` 
          : 'Zarezerwowano',
        icon: Lock,
        color: 'text-blue-500',
        planId: item.reservedForPlanId,
      });
    }

    if (item.consumedAt && !events.some(e => e.action === 'shipped')) {
      events.push({
        action: 'consumed',
        timestamp: item.consumedAt,
        details: 'Zużyto / wydano',
        icon: Package,
        color: 'text-purple-500',
      });
    }

    events.sort((a, b) => new Date(a.timestamp).getTime() - new Date(b.timestamp).getTime());

    return events;
  };

  const getActionLabel = (action: string) => {
    const labels: Record<string, string> = {
      created: 'Przyjęcie',
      reserved: 'Rezerwacja',
      unreserved: 'Anulowanie rezerwacji',
      shipped: 'Wysyłka',
      returned: 'Zwrot',
      moved: 'Przeniesienie',
      scrapped: 'Złomowanie',
      defective: 'Oznaczenie wady',
      consumed: 'Wydanie',
    };
    return labels[action] || action;
  };

  if (isProductLoading) {
    return (
      <WarehouseLayout category="produkty-spakowane">
        <div className="p-6 space-y-6">
          <Skeleton className="h-8 w-48" />
          <Skeleton className="h-32 w-full" />
        </div>
      </WarehouseLayout>
    );
  }

  if (!product) {
    return (
      <WarehouseLayout category="produkty-spakowane">
        <div className="p-6">
          <Card>
            <CardContent className="py-12 text-center">
              <Package className="w-12 h-12 mx-auto text-muted-foreground mb-4" />
              <p className="text-muted-foreground">Nie znaleziono produktu</p>
              <Button variant="outline" className="mt-4" onClick={() => setLocation("/warehouse/produkty-spakowane")}>
                <ArrowLeft className="w-4 h-4 mr-2" />
                Powrót do listy
              </Button>
            </CardContent>
          </Card>
        </div>
      </WarehouseLayout>
    );
  }

  return (
    <WarehouseLayout category="produkty-spakowane">
      <div className="p-6 space-y-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button variant="ghost" size="icon" onClick={() => setLocation("/warehouse/produkty-spakowane")} data-testid="button-back">
              <ArrowLeft className="w-4 h-4" />
            </Button>
            <div>
              <h1 className="text-2xl font-bold">{product.productName}</h1>
              <p className="text-muted-foreground">{product.productSku}</p>
            </div>
          </div>
          <Button 
            onClick={() => createInventoryCountMutation.mutate()} 
            disabled={createInventoryCountMutation.isPending}
            data-testid="button-add-to-inventory"
          >
            <FileCheck className="w-4 h-4 mr-2" />
            {createInventoryCountMutation.isPending ? "Tworzenie spisu..." : "Dodaj do spisu"}
          </Button>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground">Stan magazynowy</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{itemsData?.stats?.total || 0}</div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-1">
                <CheckCircle className="w-4 h-4 text-green-500" />
                Dostępne
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600">{itemsData?.stats?.available || 0}</div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-1">
                <Lock className="w-4 h-4 text-yellow-500" />
                Zarezerwowane
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-yellow-600">{itemsData?.stats?.reserved || 0}</div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-1">
                <Package className="w-4 h-4 text-blue-500" />
                Wysłane
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-blue-600">{itemsData?.stats?.shipped || 0}</div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-1">
                <Clock className="w-4 h-4 text-orange-500" />
                Zwrócone
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-orange-600">{itemsData?.stats?.returned || 0}</div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-1">
                <AlertTriangle className="w-4 h-4 text-red-500" />
                Wadliwe
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-red-600">{itemsData?.stats?.defective || 0}</div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-1">
                <Trash2 className="w-4 h-4 text-gray-500" />
                Zezłomowane
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-gray-600">{itemsData?.stats?.scrapped || 0}</div>
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Hash className="w-5 h-5" />
              Pojedyncze sztuki
            </CardTitle>
            <CardDescription>
              Lista wszystkich fizycznych egzemplarzy produktu z numerami seryjnymi
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Tabs value={activeTab} onValueChange={setActiveTab}>
              <TabsList className="mb-4">
                <TabsTrigger value="all" data-testid="tab-all">Wszystkie ({itemsData?.stats?.total || 0})</TabsTrigger>
                <TabsTrigger value="available" data-testid="tab-available">Dostępne ({itemsData?.stats?.available || 0})</TabsTrigger>
                <TabsTrigger value="reserved" data-testid="tab-reserved">Zarezerwowane ({itemsData?.stats?.reserved || 0})</TabsTrigger>
                <TabsTrigger value="shipped" data-testid="tab-shipped">Wysłane ({itemsData?.stats?.shipped || 0})</TabsTrigger>
                <TabsTrigger value="returned" data-testid="tab-returned">Zwrócone ({itemsData?.stats?.returned || 0})</TabsTrigger>
                <TabsTrigger value="defective" data-testid="tab-defective">Wadliwe ({itemsData?.stats?.defective || 0})</TabsTrigger>
                <TabsTrigger value="scrapped" data-testid="tab-scrapped">Zezłomowane ({itemsData?.stats?.scrapped || 0})</TabsTrigger>
              </TabsList>
              
              <TabsContent value={activeTab}>
                {isItemsLoading ? (
                  <div className="space-y-2">
                    {[...Array(5)].map((_, i) => (
                      <Skeleton key={i} className="h-12 w-full" />
                    ))}
                  </div>
                ) : (
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="w-[60px]">Lp.</TableHead>
                        <TableHead>Numer seryjny</TableHead>
                        <TableHead>Status</TableHead>
                        <TableHead>Data zapakowania</TableHead>
                        <TableHead>Lokalizacja</TableHead>
                        <TableHead>Źródło</TableHead>
                        <TableHead>Rezerwacja</TableHead>
                        {activeTab === 'scrapped' && <TableHead>Powód / Data złomowania</TableHead>}
                        <TableHead className="w-[100px]">Akcje</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {itemsData?.items?.length === 0 ? (
                        <TableRow>
                          <TableCell colSpan={activeTab === 'scrapped' ? 9 : 8} className="text-center py-8 text-muted-foreground">
                            Brak sztuk w tej kategorii
                          </TableCell>
                        </TableRow>
                      ) : (
                        itemsData?.items?.map((item, index) => (
                          <TableRow key={item.id} data-testid={`row-item-${item.id}`}>
                            <TableCell className="text-muted-foreground">{index + 1}</TableCell>
                            <TableCell className="font-mono text-sm">{item.serialNumber}</TableCell>
                            <TableCell>{getStatusBadge(item.status)}</TableCell>
                            <TableCell>{formatDate(item.packedAt)}</TableCell>
                            <TableCell>
                              {item.locationName ? (
                                <span className="flex items-center gap-1">
                                  <MapPin className="w-3 h-3 text-muted-foreground" />
                                  {item.locationName}
                                </span>
                              ) : (
                                <span className="text-muted-foreground">-</span>
                              )}
                            </TableCell>
                            <TableCell>
                              {item.sourceType === 'inventory_count' ? (
                                <Link href={`/warehouse/inventory-counts/${item.sourceId}`}>
                                  <Badge variant="outline" className="cursor-pointer hover:bg-accent bg-yellow-500/10 text-yellow-700 border-yellow-300">
                                    <FileCheck className="w-3 h-3 mr-1" />
                                    Spis #{item.sourceId}
                                  </Badge>
                                </Link>
                              ) : item.sourceType === 'production' ? (
                                <Link href={`/production/plans/${item.sourceId}`}>
                                  <Badge variant="outline" className="cursor-pointer hover:bg-accent bg-blue-500/10 text-blue-700 border-blue-300">
                                    <Package className="w-3 h-3 mr-1" />
                                    Produkcja #{item.sourceId}
                                  </Badge>
                                </Link>
                              ) : (
                                <span className="text-muted-foreground">Ręczne</span>
                              )}
                            </TableCell>
                            <TableCell>
                              {item.reservedForPlanId ? (
                                <Link href={`/production/plans/${item.reservedForPlanId}`}>
                                  <Badge variant="outline" className="cursor-pointer hover:bg-accent">
                                    Plan #{item.reservedForPlanId}
                                  </Badge>
                                </Link>
                              ) : (
                                <span className="text-muted-foreground">-</span>
                              )}
                            </TableCell>
                            {activeTab === 'scrapped' && (
                              <TableCell>
                                {item.metadata?.auditHistory?.find(h => h.action === 'scrapped') ? (
                                  <div className="text-sm space-y-1">
                                    <div className="text-muted-foreground">
                                      {item.metadata.auditHistory.find(h => h.action === 'scrapped')?.reason?.replace(/^\[.*?\]\s*/, '')}
                                    </div>
                                    <div className="text-xs text-muted-foreground">
                                      {formatDate(item.metadata.auditHistory.find(h => h.action === 'scrapped')?.timestamp || '')}
                                      {item.metadata.auditHistory.find(h => h.action === 'scrapped')?.performedBy && (
                                        <span> przez {item.metadata.auditHistory.find(h => h.action === 'scrapped')?.performedBy}</span>
                                      )}
                                    </div>
                                  </div>
                                ) : item.notes ? (
                                  <div className="text-sm text-muted-foreground max-w-xs truncate" title={item.notes}>
                                    {item.notes}
                                  </div>
                                ) : (
                                  <span className="text-muted-foreground">-</span>
                                )}
                              </TableCell>
                            )}
                            <TableCell>
                              <div className="flex items-center gap-1">
                                <Button
                                  variant="ghost"
                                  size="icon"
                                  onClick={() => {
                                    setHistoryItem(item);
                                    setIsHistoryDialogOpen(true);
                                  }}
                                  title="Historia ruchu"
                                  data-testid={`button-history-${item.id}`}
                                >
                                  <History className="w-4 h-4 text-muted-foreground" />
                                </Button>
                                {item.status !== 'reserved' && item.status !== 'scrapped' && (
                                  <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={() => {
                                      setSelectedItemId(item.id);
                                      setIsScrapDialogOpen(true);
                                    }}
                                    title="Zezłomuj"
                                    data-testid={`button-scrap-${item.id}`}
                                  >
                                    <Trash2 className="w-4 h-4 text-red-500" />
                                  </Button>
                                )}
                              </div>
                            </TableCell>
                          </TableRow>
                        ))
                      )}
                    </TableBody>
                  </Table>
                )}
              </TabsContent>
            </Tabs>
          </CardContent>
        </Card>

        <Dialog open={isScrapDialogOpen} onOpenChange={setIsScrapDialogOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Zezłomuj sztukę</DialogTitle>
              <DialogDescription>
                Podaj powód złomowania. Sztuka zostanie trwale oznaczona jako zezłomowana i nie będzie dostępna do użycia.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <div className="space-y-2">
                <Label>Powód złomowania</Label>
                <DictionaryCombobox
                  items={scrapReasons}
                  value={scrapReasonCode}
                  onChange={(value) => {
                    setScrapReasonCode(value);
                    if (value !== "other") {
                      setScrapReasonCustom("");
                    }
                  }}
                  placeholder="Wybierz powód..."
                  searchPlaceholder="Szukaj powodu..."
                  emptyText="Nie znaleziono powodu"
                  testId="select-scrap-reason"
                  valueField="name"
                  displayField="readableName"
                />
              </div>
              {scrapReasonCode === "other" && (
                <div className="space-y-2">
                  <Label htmlFor="scrapReasonCustom">Opisz powód</Label>
                  <Textarea
                    id="scrapReasonCustom"
                    value={scrapReasonCustom}
                    onChange={(e) => setScrapReasonCustom(e.target.value)}
                    placeholder="Opisz powód złomowania..."
                    data-testid="input-scrap-reason-custom"
                  />
                </div>
              )}
            </div>
            <DialogFooter>
              <Button variant="outline" onClick={() => {
                setIsScrapDialogOpen(false);
                setScrapReasonCode("");
                setScrapReasonCustom("");
              }}>
                Anuluj
              </Button>
              <Button
                variant="destructive"
                onClick={() => {
                  let finalReason = "";
                  if (scrapReasonCode === "other") {
                    finalReason = `[other] ${scrapReasonCustom.trim()}`;
                  } else {
                    const selectedReason = scrapReasons.find(r => r.name === scrapReasonCode);
                    const displayName = selectedReason?.readableName || selectedReason?.name || scrapReasonCode;
                    finalReason = `[${scrapReasonCode}] ${displayName}`;
                  }
                  if (selectedItemId && finalReason) {
                    scrapItemMutation.mutate({ itemId: selectedItemId, reason: finalReason });
                  }
                }}
                disabled={
                  !scrapReasonCode || 
                  (scrapReasonCode === "other" && !scrapReasonCustom.trim()) ||
                  scrapItemMutation.isPending
                }
                data-testid="button-confirm-scrap"
              >
                {scrapItemMutation.isPending ? "Zapisywanie..." : "Zezłomuj"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        <Dialog open={isHistoryDialogOpen} onOpenChange={setIsHistoryDialogOpen}>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2">
                <History className="w-5 h-5" />
                Historia ruchu produktu
              </DialogTitle>
              <DialogDescription>
                {historyItem && (
                  <div className="flex flex-col gap-1 mt-2">
                    <span className="font-mono text-sm">{historyItem.serialNumber}</span>
                    <div className="flex items-center gap-2">
                      {getStatusBadge(historyItem.status)}
                      {historyItem.locationName && (
                        <span className="text-xs flex items-center gap-1">
                          <MapPin className="w-3 h-3" />
                          {historyItem.locationName}
                        </span>
                      )}
                    </div>
                  </div>
                )}
              </DialogDescription>
            </DialogHeader>
            
            {historyItem && (
              <ScrollArea className="max-h-[400px] pr-4">
                <div className="relative">
                  <div className="absolute left-[19px] top-0 bottom-0 w-0.5 bg-border" />
                  
                  <div className="space-y-4">
                    {buildHistoryTimeline(historyItem).map((event, index) => {
                      const EventIcon = event.icon;
                      return (
                        <div key={index} className="relative flex gap-4 pl-0">
                          <div className={`relative z-10 flex h-10 w-10 shrink-0 items-center justify-center rounded-full border-2 bg-background ${event.color.replace('text-', 'border-')}`}>
                            <EventIcon className={`w-4 h-4 ${event.color}`} />
                          </div>
                          
                          <div className="flex-1 pt-1.5">
                            <div className="flex items-center gap-2 flex-wrap">
                              <span className="font-medium text-sm">{getActionLabel(event.action)}</span>
                              <span className="text-xs text-muted-foreground">
                                {formatDate(event.timestamp)}
                              </span>
                            </div>
                            
                            <p className="text-sm text-muted-foreground mt-0.5">
                              {event.details}
                            </p>
                            
                            <div className="flex items-center gap-2 mt-1 flex-wrap">
                              {event.performedBy && (
                                <Badge variant="outline" className="text-xs gap-1">
                                  <User className="w-3 h-3" />
                                  {event.performedBy}
                                </Badge>
                              )}
                              {event.planId && (
                                <Link href={`/production/plans/${event.planId}`}>
                                  <Badge variant="outline" className="text-xs gap-1 cursor-pointer hover:bg-accent">
                                    <ClipboardList className="w-3 h-3" />
                                    Plan #{event.planId}
                                  </Badge>
                                </Link>
                              )}
                              {event.orderId && (
                                <Badge variant="outline" className="text-xs gap-1 bg-blue-500/10 text-blue-700 border-blue-300">
                                  <Package className="w-3 h-3" />
                                  {event.orderId}
                                </Badge>
                              )}
                            </div>
                          </div>
                        </div>
                      );
                    })}
                    
                    {buildHistoryTimeline(historyItem).length === 0 && (
                      <div className="text-center py-8 text-muted-foreground">
                        Brak zapisanej historii ruchu
                      </div>
                    )}
                  </div>
                </div>
              </ScrollArea>
            )}
            
            <DialogFooter>
              <Button variant="outline" onClick={() => setIsHistoryDialogOpen(false)}>
                Zamknij
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>
    </WarehouseLayout>
  );
}
