import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Plus, Trash2, LayoutGrid, Loader2 } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Badge } from "@/components/ui/badge";

interface MaterialGroup {
  id: number;
  name: string;
  code: string;
  category: string;
  description: string | null;
  displayOrder: number | null;
}

interface Category {
  id: number;
  name: string;
  code: string;
}

// Category display names and colors
const CATEGORY_INFO: Record<string, { name: string; color: string }> = {
  formatki: { name: "Formatki", color: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200" },
  plyty: { name: "Płyty", color: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200" },
  okucia: { name: "Okucia", color: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200" },
  obrzeza: { name: "Obrzęża", color: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200" },
  tkaniny: { name: "Tkaniny", color: "bg-pink-100 text-pink-800 dark:bg-pink-900 dark:text-pink-200" },
  sruby: { name: "Śruby", color: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200" },
  pianki: { name: "Pianki", color: "bg-indigo-100 text-indigo-800 dark:bg-indigo-900 dark:text-indigo-200" },
  opakowania: { name: "Opakowania", color: "bg-teal-100 text-teal-800 dark:bg-teal-900 dark:text-teal-200" },
  "produkty-spakowane": { name: "Produkty spakowane", color: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200" },
};

export default function Groups() {
  const { toast } = useToast();
  const [createDialogOpen, setCreateDialogOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState("");
  const [categoryFilter, setCategoryFilter] = useState<string>("all");
  const [formData, setFormData] = useState({
    name: "",
    code: "",
    category: "",
    description: "",
  });

  // Fetch all groups
  const { data: groups = [], isLoading } = useQuery<MaterialGroup[]>({
    queryKey: ["/api/warehouse/material-groups"],
  });

  // Fetch categories
  const { data: categories = [] } = useQuery<Category[]>({
    queryKey: ["/api/warehouse/categories"],
  });

  // Create group mutation
  const createGroupMutation = useMutation({
    mutationFn: async (data: { name: string; code: string; category: string; description: string }) => {
      // Find category ID
      const category = categories.find(c => c.code === data.category);
      if (!category) throw new Error("Category not found");

      return apiRequest("POST", "/api/warehouse/groups", {
        categoryId: category.id,
        name: data.name,
        code: data.code,
        description: data.description || null,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      toast({ title: "Grupa utworzona pomyślnie" });
      setCreateDialogOpen(false);
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć grupy",
        variant: "destructive",
      });
    },
  });

  // Delete group mutation
  const deleteGroupMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/warehouse/groups/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      toast({ title: "Grupa usunięta pomyślnie" });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się usunąć grupy",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({ name: "", code: "", category: "", description: "" });
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!formData.name || !formData.code || !formData.category) {
      toast({
        title: "Błąd walidacji",
        description: "Wypełnij wszystkie wymagane pola",
        variant: "destructive",
      });
      return;
    }
    createGroupMutation.mutate(formData);
  };

  const handleDelete = (id: number, name: string) => {
    if (confirm(`Czy na pewno chcesz usunąć grupę "${name}"?`)) {
      deleteGroupMutation.mutate(id);
    }
  };

  // Filter groups
  const filteredGroups = groups.filter((group) => {
    const matchesSearch = searchQuery === "" || 
      group.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
      group.code.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesCategory = categoryFilter === "all" || group.category === categoryFilter;
    return matchesSearch && matchesCategory;
  });

  // Group by category
  const groupedByCategory = filteredGroups.reduce((acc, group) => {
    if (!acc[group.category]) {
      acc[group.category] = [];
    }
    acc[group.category].push(group);
    return acc;
  }, {} as Record<string, MaterialGroup[]>);

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <LayoutGrid className="w-8 h-8 text-primary" />
          <div>
            <h1 className="text-3xl font-bold" data-testid="text-page-title">
              Grupy
            </h1>
            <p className="text-muted-foreground">
              Zarządzaj grupami materiałów wszystkich kategorii
            </p>
          </div>
        </div>
        <Button onClick={() => setCreateDialogOpen(true)} data-testid="button-create-group">
          <Plus className="w-4 h-4 mr-2" />
          Dodaj grupę
        </Button>
      </div>

      {/* Filters */}
      <Card>
        <CardContent className="pt-6">
          <div className="flex gap-4">
            <div className="flex-1">
              <Input
                placeholder="Szukaj grup po nazwie lub kodzie..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                data-testid="input-search"
              />
            </div>
            <Select value={categoryFilter} onValueChange={setCategoryFilter}>
              <SelectTrigger className="w-[250px]" data-testid="select-category-filter">
                <SelectValue placeholder="Wszystkie kategorie" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Wszystkie kategorie</SelectItem>
                {categories.map((cat) => (
                  <SelectItem key={cat.code} value={cat.code}>
                    {CATEGORY_INFO[cat.code]?.name || cat.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      {/* Groups List */}
      {isLoading ? (
        <Card>
          <CardContent className="py-12">
            <div className="flex items-center justify-center gap-2 text-muted-foreground">
              <Loader2 className="w-5 h-5 animate-spin" />
              Ładowanie grup...
            </div>
          </CardContent>
        </Card>
      ) : filteredGroups.length === 0 ? (
        <Card>
          <CardContent className="py-12">
            <div className="text-center text-muted-foreground">
              {searchQuery || categoryFilter !== "all" 
                ? "Nie znaleziono grup spełniających kryteria" 
                : "Brak grup. Dodaj pierwszą grupę używając przycisku powyżej."}
            </div>
          </CardContent>
        </Card>
      ) : (
        <div className="space-y-6">
          {Object.entries(groupedByCategory)
            .sort(([a], [b]) => a.localeCompare(b))
            .map(([category, categoryGroups]) => (
              <Card key={category}>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Badge className={CATEGORY_INFO[category]?.color || "bg-gray-100 text-gray-800"}>
                      {CATEGORY_INFO[category]?.name || category}
                    </Badge>
                    <span className="text-sm font-normal text-muted-foreground">
                      ({categoryGroups.length} {categoryGroups.length === 1 ? "grupa" : "grup"})
                    </span>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="w-[80px]">Kod</TableHead>
                        <TableHead>Nazwa</TableHead>
                        <TableHead>Opis</TableHead>
                        <TableHead className="w-[80px]">Akcje</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {categoryGroups
                        .sort((a, b) => (a.displayOrder || 999) - (b.displayOrder || 999))
                        .map((group) => (
                          <TableRow key={group.id} data-testid={`row-group-${group.id}`}>
                            <TableCell className="font-mono text-sm" data-testid={`text-group-code-${group.id}`}>
                              {group.code}
                            </TableCell>
                            <TableCell className="font-medium" data-testid={`text-group-name-${group.id}`}>
                              {group.name}
                            </TableCell>
                            <TableCell className="text-muted-foreground" data-testid={`text-group-description-${group.id}`}>
                              {group.description || "-"}
                            </TableCell>
                            <TableCell>
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => handleDelete(group.id, group.name)}
                                data-testid={`button-delete-group-${group.id}`}
                              >
                                <Trash2 className="w-4 h-4" />
                              </Button>
                            </TableCell>
                          </TableRow>
                        ))}
                    </TableBody>
                  </Table>
                </CardContent>
              </Card>
            ))}
        </div>
      )}

      {/* Create Group Dialog */}
      <Dialog open={createDialogOpen} onOpenChange={setCreateDialogOpen}>
        <DialogContent data-testid="dialog-create-group">
          <DialogHeader>
            <DialogTitle>Dodaj nową grupę</DialogTitle>
            <DialogDescription>
              Utwórz nową grupę materiałów dla wybranej kategorii
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleSubmit}>
            <div className="space-y-4 py-4">
              <div>
                <Label htmlFor="category">Kategoria *</Label>
                <Select 
                  value={formData.category} 
                  onValueChange={(value) => setFormData({ ...formData, category: value })}
                >
                  <SelectTrigger id="category" data-testid="select-category">
                    <SelectValue placeholder="Wybierz kategorię" />
                  </SelectTrigger>
                  <SelectContent>
                    {categories.map((cat) => (
                      <SelectItem key={cat.code} value={cat.code}>
                        {CATEGORY_INFO[cat.code]?.name || cat.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="code">Kod grupy *</Label>
                <Input
                  id="code"
                  value={formData.code}
                  onChange={(e) => setFormData({ ...formData, code: e.target.value })}
                  placeholder="np. 05, 10, A1"
                  required
                  data-testid="input-group-code"
                />
              </div>
              <div>
                <Label htmlFor="name">Nazwa grupy *</Label>
                <Input
                  id="name"
                  value={formData.name}
                  onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                  placeholder="np. Płyta biała, Zawiasy"
                  required
                  data-testid="input-group-name"
                />
              </div>
              <div>
                <Label htmlFor="description">Opis</Label>
                <Input
                  id="description"
                  value={formData.description}
                  onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                  placeholder="Opcjonalny opis grupy"
                  data-testid="input-group-description"
                />
              </div>
            </div>
            <DialogFooter>
              <Button
                type="button"
                variant="outline"
                onClick={() => {
                  setCreateDialogOpen(false);
                  resetForm();
                }}
                data-testid="button-cancel"
              >
                Anuluj
              </Button>
              <Button
                type="submit"
                disabled={createGroupMutation.isPending}
                data-testid="button-submit"
              >
                {createGroupMutation.isPending && (
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                )}
                Dodaj grupę
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
