import { useEffect } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation, useParams } from "wouter";
import { ArrowLeft, Save, Loader2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { cutPatternTemplateFormSchema, type CutPatternTemplateFormData, type CutPatternTemplate } from "@shared/schema";

export default function CuttingPatternsEditPage() {
  const { toast } = useToast();
  const [, navigate] = useLocation();
  const params = useParams<{ id: string }>();
  const templateId = parseInt(params.id || "0");

  const { data: template, isLoading, error } = useQuery<CutPatternTemplate>({
    queryKey: ["/api/production/cut-pattern-templates", templateId],
    queryFn: async () => {
      const response = await fetch(`/api/production/cut-pattern-templates/${templateId}`);
      if (!response.ok) {
        throw new Error("Nie znaleziono szablonu");
      }
      return response.json();
    },
    enabled: templateId > 0,
  });

  const form = useForm<CutPatternTemplateFormData>({
    resolver: zodResolver(cutPatternTemplateFormSchema),
    defaultValues: {
      name: "",
      description: "",
      boardTypeCode: "",
      boardLength: 2800,
      boardWidth: 2070,
      boardThickness: 18,
      kerf: 4,
      imageUrl: "",
      opis: "",
      notes: "",
    },
  });

  // Wypełnij formularz danymi z API
  useEffect(() => {
    if (template) {
      form.reset({
        name: template.name || "",
        description: template.description || "",
        boardTypeCode: template.boardTypeCode || "",
        boardLength: parseFloat(template.boardLength as any) || 2800,
        boardWidth: parseFloat(template.boardWidth as any) || 2070,
        boardThickness: parseFloat(template.boardThickness as any) || 18,
        kerf: parseFloat(template.kerf as any) || 4,
        imageUrl: template.imageUrl || "",
        opis: template.opis || "",
        notes: template.notes || "",
      });
    }
  }, [template, form]);

  const updateMutation = useMutation({
    mutationFn: async (data: CutPatternTemplateFormData) => {
      const res = await apiRequest("PATCH", `/api/production/cut-pattern-templates/${templateId}`, data);
      return await res.json();
    },
    onSuccess: (response: { success: boolean; message: string }) => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/cut-pattern-templates"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/cut-pattern-templates", templateId] });
      toast({
        title: "Szablon zaktualizowany",
        description: response.message,
      });
      navigate("/cutting-patterns");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji szablonu",
        description: error.message || "Nie udało się zaktualizować szablonu rozkroju",
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: CutPatternTemplateFormData) => {
    updateMutation.mutate(data);
  };

  // Obsługa błędów i przekierowanie
  useEffect(() => {
    if (error) {
      toast({
        title: "Szablon nie znaleziony",
        description: "Przekierowanie do listy szablonów",
        variant: "destructive",
      });
      navigate("/cutting-patterns");
    }
  }, [error, navigate, toast]);

  if (isLoading) {
    return (
      <div className="container mx-auto py-6 max-w-4xl">
        <div className="flex items-center gap-4 mb-6">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => navigate("/cutting-patterns")}
            data-testid="button-back"
          >
            <ArrowLeft className="h-5 w-5" />
          </Button>
          <div>
            <Skeleton className="h-9 w-64 mb-2" />
            <Skeleton className="h-5 w-48" />
          </div>
        </div>
        <Card>
          <CardHeader>
            <Skeleton className="h-7 w-48 mb-2" />
            <Skeleton className="h-5 w-96" />
          </CardHeader>
          <CardContent className="space-y-6">
            {[1, 2, 3, 4].map((i) => (
              <div key={i} className="space-y-2">
                <Skeleton className="h-5 w-32" />
                <Skeleton className="h-10 w-full" />
              </div>
            ))}
          </CardContent>
        </Card>
      </div>
    );
  }

  if (!template) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
      </div>
    );
  }

  return (
    <div className="container mx-auto py-6 max-w-4xl">
      <div className="flex items-center gap-4 mb-6">
        <Button
          variant="ghost"
          size="icon"
          onClick={() => navigate("/cutting-patterns")}
          data-testid="button-back"
        >
          <ArrowLeft className="h-5 w-5" />
        </Button>
        <div>
          <h1 className="text-3xl font-bold">Edycja szablonu rozkroju</h1>
          <p className="text-muted-foreground">
            Kod: <span className="font-mono font-semibold">{template.code}</span>
          </p>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Podstawowe informacje</CardTitle>
          <CardDescription>
            Edytuj dane szablonu rozkroju. Pola oznaczone * są wymagane.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa *</FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        placeholder="np. Rozkrój szafki kuchennej"
                        data-testid="input-name"
                      />
                    </FormControl>
                    <FormDescription>
                      Krótka, opisowa nazwa szablonu rozkroju
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea
                        {...field}
                        placeholder="Dodatkowy opis szablonu rozkroju"
                        rows={3}
                        data-testid="input-description"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <FormField
                  control={form.control}
                  name="boardLength"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Długość płyty (mm) *</FormLabel>
                      <FormControl>
                        <Input
                          {...field}
                          type="number"
                          step="0.01"
                          placeholder="2800"
                          data-testid="input-board-length"
                        />
                      </FormControl>
                      <FormDescription>Długość płyty bazowej</FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="boardWidth"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Szerokość płyty (mm) *</FormLabel>
                      <FormControl>
                        <Input
                          {...field}
                          type="number"
                          step="0.01"
                          placeholder="2070"
                          data-testid="input-board-width"
                        />
                      </FormControl>
                      <FormDescription>Szerokość płyty bazowej</FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="boardThickness"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Grubość płyty (mm)</FormLabel>
                      <FormControl>
                        <Input
                          {...field}
                          type="number"
                          step="0.01"
                          placeholder="18"
                          data-testid="input-board-thickness"
                        />
                      </FormControl>
                      <FormDescription>Domyślnie: 18 mm</FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="kerf"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kerf - szerokość cięcia (mm)</FormLabel>
                      <FormControl>
                        <Input
                          {...field}
                          type="number"
                          step="0.01"
                          placeholder="4"
                          data-testid="input-kerf"
                        />
                      </FormControl>
                      <FormDescription>Domyślnie: 4 mm</FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="boardTypeCode"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kod typu płyty</FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        placeholder="np. 18_WOTAN"
                        data-testid="input-board-type-code"
                      />
                    </FormControl>
                    <FormDescription>
                      Opcjonalny kod oznaczenia materiału płyty
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="opis"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis rozszerzony</FormLabel>
                    <FormControl>
                      <Textarea
                        {...field}
                        placeholder="Szczegółowy opis techniczny rozkroju"
                        rows={4}
                        data-testid="input-opis"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Uwagi</FormLabel>
                    <FormControl>
                      <Textarea
                        {...field}
                        placeholder="Dodatkowe uwagi produkcyjne"
                        rows={3}
                        data-testid="input-notes"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="imageUrl"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>URL obrazka rozkroju</FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        type="url"
                        placeholder="https://example.com/image.jpg"
                        data-testid="input-image-url"
                      />
                    </FormControl>
                    <FormDescription>
                      Link do wizualizacji rozkroju (opcjonalny)
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex gap-3 justify-end pt-4 border-t">
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => navigate("/cutting-patterns")}
                  disabled={updateMutation.isPending}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button
                  type="submit"
                  disabled={updateMutation.isPending}
                  data-testid="button-submit"
                >
                  {updateMutation.isPending ? (
                    <>
                      <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                      Aktualizowanie...
                    </>
                  ) : (
                    <>
                      <Save className="h-4 w-4 mr-2" />
                      Zaktualizuj szablon
                    </>
                  )}
                </Button>
              </div>
            </form>
          </Form>
        </CardContent>
      </Card>
    </div>
  );
}
