import { useQuery } from "@tanstack/react-query";
import { useParams, useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { ArrowLeft, Package, Ruler } from "lucide-react";

interface ComponentTemplate {
  id: number;
  furniture_type: string;
  cz1: string;
  cz2: string;
  base_length: number;
  base_width: number;
  thickness: number;
  color: string;
  edge1: boolean;
  edge2: boolean;
  edge3: boolean;
  edge4: boolean;
  edging_material: string;
  plate_type: string;
  door: string | null;
  leg: string | null;
  production_category: string;
  is_active: boolean;
  created_at: string;
}

export default function ComponentTemplateDetail() {
  const params = useParams();
  const templateId = params.id;

  const { data: template, isLoading } = useQuery<ComponentTemplate>({
    queryKey: [`/api/component-templates/${templateId}`],
    enabled: !!templateId,
  });

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="text-center">
          <Package className="w-12 h-12 mx-auto mb-4 text-muted-foreground animate-pulse" />
          <p className="text-muted-foreground">Ładowanie formatki...</p>
        </div>
      </div>
    );
  }

  if (!template) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="text-center">
          <Package className="w-12 h-12 mx-auto mb-4 text-muted-foreground" />
          <p className="text-muted-foreground">Formatka nie została znaleziona</p>
          <Button 
            variant="link" 
            className="mt-4"
            onClick={() => window.history.back()}
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Wstecz
          </Button>
        </div>
      </div>
    );
  }

  // Calculate SVG dimensions (max width 400px, maintain aspect ratio)
  const maxSvgWidth = 400;
  const aspectRatio = template.base_length / template.base_width;
  const svgWidth = maxSvgWidth;
  const svgHeight = svgWidth / aspectRatio;

  // Padding for SVG
  const padding = 40;
  const rectX = padding;
  const rectY = padding;
  const rectWidth = svgWidth - padding * 2;
  const rectHeight = svgHeight - padding * 2;

  // Edge labels
  const edgeOffset = 10;
  const edges = [
    { active: template.edge1, label: "Krawędź 1 (góra)", x1: rectX, y1: rectY - edgeOffset, x2: rectX + rectWidth, y2: rectY - edgeOffset },
    { active: template.edge2, label: "Krawędź 2 (prawo)", x1: rectX + rectWidth + edgeOffset, y1: rectY, x2: rectX + rectWidth + edgeOffset, y2: rectY + rectHeight },
    { active: template.edge3, label: "Krawędź 3 (dół)", x1: rectX, y1: rectY + rectHeight + edgeOffset, x2: rectX + rectWidth, y2: rectY + rectHeight + edgeOffset },
    { active: template.edge4, label: "Krawędź 4 (lewo)", x1: rectX - edgeOffset, y1: rectY, x2: rectX - edgeOffset, y2: rectY + rectHeight },
  ];

  return (
    <div className="container mx-auto p-6 max-w-6xl">
      {/* Header */}
      <div className="mb-6">
        <Button 
          variant="ghost" 
          size="sm" 
          data-testid="button-back"
          onClick={() => window.history.back()}
        >
          <ArrowLeft className="w-4 h-4 mr-2" />
          Wstecz
        </Button>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Left Column: Visualization */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Ruler className="w-5 h-5" />
              Wizualizacja formatki
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="bg-muted/20 rounded-lg p-4 flex items-center justify-center">
              <svg
                width={svgWidth + padding * 2}
                height={svgHeight + padding * 2}
                viewBox={`0 0 ${svgWidth + padding * 2} ${svgHeight + padding * 2}`}
                className="mx-auto"
              >
                {/* Main rectangle */}
                <rect
                  x={rectX}
                  y={rectY}
                  width={rectWidth}
                  height={rectHeight}
                  fill="hsl(var(--card))"
                  stroke="hsl(var(--border))"
                  strokeWidth="2"
                  data-testid="svg-rectangle"
                />

                {/* Dimension labels */}
                <text
                  x={rectX + rectWidth / 2}
                  y={rectY - 20}
                  textAnchor="middle"
                  fontSize="14"
                  fill="hsl(var(--foreground))"
                  fontWeight="600"
                >
                  {template.base_length} mm
                </text>
                <text
                  x={rectX + rectWidth + 20}
                  y={rectY + rectHeight / 2}
                  textAnchor="start"
                  fontSize="14"
                  fill="hsl(var(--foreground))"
                  fontWeight="600"
                  transform={`rotate(90, ${rectX + rectWidth + 20}, ${rectY + rectHeight / 2})`}
                >
                  {template.base_width} mm
                </text>

                {/* Edging lines */}
                {edges.map((edge, index) => {
                  if (!edge.active) return null;
                  
                  return (
                    <g key={index}>
                      {/* Dashed line for edge */}
                      <line
                        x1={edge.x1}
                        y1={edge.y1}
                        x2={edge.x2}
                        y2={edge.y2}
                        stroke="hsl(var(--primary))"
                        strokeWidth="2"
                        strokeDasharray="5,5"
                        data-testid={`svg-edge-${index + 1}`}
                      />
                      {/* Label */}
                      <text
                        x={(edge.x1 + edge.x2) / 2}
                        y={(edge.y1 + edge.y2) / 2 + (index % 2 === 0 ? -5 : 15)}
                        textAnchor="middle"
                        fontSize="11"
                        fill="hsl(var(--primary))"
                        fontWeight="500"
                      >
                        {template.edging_material || "Obrzeże"}
                      </text>
                    </g>
                  );
                })}

                {/* Center label */}
                <text
                  x={rectX + rectWidth / 2}
                  y={rectY + rectHeight / 2}
                  textAnchor="middle"
                  fontSize="16"
                  fill="hsl(var(--muted-foreground))"
                  fontWeight="500"
                >
                  {template.cz1} {template.cz2}
                </text>
              </svg>
            </div>

            {/* Edge legend */}
            <div className="mt-4 grid grid-cols-2 gap-2">
              {edges.map((edge, index) => (
                <div key={index} className="flex items-center gap-2 text-sm">
                  <div className={`w-3 h-3 rounded-sm ${edge.active ? 'bg-primary' : 'bg-muted'}`} />
                  <span className={edge.active ? 'text-foreground' : 'text-muted-foreground'}>
                    {edge.label}
                  </span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* Right Column: Details */}
        <div className="space-y-6">
          {/* Basic Info */}
          <Card>
            <CardHeader>
              <CardTitle>Informacje podstawowe</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="grid grid-cols-2 gap-3">
                <div>
                  <p className="text-sm text-muted-foreground">ID</p>
                  <p className="font-medium" data-testid="text-template-id">#{template.id}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Status</p>
                  <Badge variant={template.is_active ? "default" : "secondary"}>
                    {template.is_active ? "Aktywna" : "Nieaktywna"}
                  </Badge>
                </div>
              </div>

              <div>
                <p className="text-sm text-muted-foreground">Rodzaj mebla</p>
                <Badge variant="outline" className="mt-1">{template.furniture_type}</Badge>
              </div>

              <div className="grid grid-cols-2 gap-3">
                <div>
                  <p className="text-sm text-muted-foreground">CZ1</p>
                  <Badge variant="secondary" className="mt-1">{template.cz1}</Badge>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">CZ2</p>
                  <Badge variant="outline" className="mt-1">{template.cz2}</Badge>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Dimensions */}
          <Card>
            <CardHeader>
              <CardTitle>Wymiary</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="grid grid-cols-3 gap-3">
                <div>
                  <p className="text-sm text-muted-foreground">Długość</p>
                  <p className="font-medium font-mono">{template.base_length} mm</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Szerokość</p>
                  <p className="font-medium font-mono">{template.base_width} mm</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Grubość</p>
                  <p className="font-medium font-mono">{template.thickness || "-"} mm</p>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Materials */}
          <Card>
            <CardHeader>
              <CardTitle>Materiały</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div>
                <p className="text-sm text-muted-foreground">Kolor</p>
                <p className="font-medium">{template.color || "-"}</p>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Typ płyty</p>
                <p className="font-medium">{template.plate_type || "-"}</p>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Materiał obrzeża</p>
                <p className="font-medium">{template.edging_material || "-"}</p>
              </div>
              {template.production_category && (
                <div>
                  <p className="text-sm text-muted-foreground">Kategoria produkcji</p>
                  <Badge variant="outline" className="mt-1">{template.production_category}</Badge>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Options */}
          {(template.door || template.leg) && (
            <Card>
              <CardHeader>
                <CardTitle>Opcje</CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                {template.door && (
                  <div>
                    <p className="text-sm text-muted-foreground">Drzwi</p>
                    <Badge variant="secondary" className="mt-1">{template.door}</Badge>
                  </div>
                )}
                {template.leg && (
                  <div>
                    <p className="text-sm text-muted-foreground">Nóżki</p>
                    <Badge variant="secondary" className="mt-1">{template.leg}</Badge>
                  </div>
                )}
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </div>
  );
}
