import { useState } from 'react';
import { useDropzone } from 'react-dropzone';
import { Upload, CheckCircle, XCircle, AlertCircle, Save } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { useToast } from '@/hooks/use-toast';
import { queryClient } from '@/lib/queryClient';
import { Badge } from '@/components/ui/badge';
import { ScrollArea } from '@/components/ui/scroll-area';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';

interface AirtableProduct {
  sku: string;
  name: string;
  length: number | null;
  width: number | null;
  color: string | null;
  doors: string | null;
  legs: string | null;
}

interface SystemProduct {
  id: number;
  sku: string;
  title: string;
  length: string | null;
  width: string | null;
  color: string | null;
  doors: string | null;
  legs: string | null;
  skuAirtable: string | null;
}

interface MatchResult {
  airtableProduct: AirtableProduct;
  systemProduct: SystemProduct | null;
  matchScore: number;
  matchDetails: string[];
}

export default function AirtableMatcherPage() {
  const [matchResults, setMatchResults] = useState<MatchResult[]>([]);
  const [isUploading, setIsUploading] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [stats, setStats] = useState<{ total: number; matched: number; unmatched: number } | null>(null);
  const { toast } = useToast();

  const onDrop = async (acceptedFiles: File[]) => {
    if (acceptedFiles.length === 0) return;

    const file = acceptedFiles[0];
    setIsUploading(true);

    try {
      const formData = new FormData();
      formData.append('file', file);

      const response = await fetch('/api/airtable/match-products', {
        method: 'POST',
        body: formData,
        credentials: 'include',
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.error || 'Błąd matchowania');
      }

      setMatchResults(data.results);
      setStats({
        total: data.total,
        matched: data.matched,
        unmatched: data.unmatched,
      });

      toast({
        title: 'Matchowanie zakończone',
        description: `Dopasowano ${data.matched} z ${data.total} produktów (${Math.round((data.matched / data.total) * 100)}%)`,
      });
    } catch (error: any) {
      toast({
        title: 'Błąd matchowania',
        description: error.message,
        variant: 'destructive',
      });
    } finally {
      setIsUploading(false);
    }
  };

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: {
      'text/csv': ['.csv'],
    },
    maxFiles: 1,
  });

  const handleApplySku = async () => {
    // Tylko dopasowane produkty
    const matchedProducts = matchResults.filter(r => r.systemProduct !== null);

    if (matchedProducts.length === 0) {
      toast({
        title: 'Brak dopasowań',
        description: 'Nie ma produktów do zapisania',
        variant: 'destructive',
      });
      return;
    }

    setIsSaving(true);

    try {
      const matches = matchedProducts.map(r => ({
        productId: r.systemProduct!.id,
        airtableSku: r.airtableProduct.sku,
      }));

      const response = await fetch('/api/airtable/apply-sku', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ matches }),
        credentials: 'include',
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.error || 'Błąd zapisu');
      }

      toast({
        title: 'Zapisano SKU',
        description: `Zaktualizowano ${data.updated} produktów`,
      });

      // Odśwież cache produktów katalogowych
      await queryClient.invalidateQueries({ queryKey: ['/api/catalog-products'] });

      // Wyczyść wyniki po zapisie
      setMatchResults([]);
      setStats(null);
    } catch (error: any) {
      toast({
        title: 'Błąd zapisu',
        description: error.message,
        variant: 'destructive',
      });
    } finally {
      setIsSaving(false);
    }
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div>
        <h1 className="text-3xl font-bold">Matchowanie produktów z Airtable</h1>
        <p className="text-muted-foreground mt-2">
          Prześlij CSV z Airtable aby dopasować SKU do produktów w systemie
        </p>
      </div>

      {/* Upload Area */}
      <Card>
        <CardHeader>
          <CardTitle>Prześlij plik CSV</CardTitle>
          <CardDescription>
            Przeciągnij plik CSV z Airtable lub kliknij aby wybrać
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div
            {...getRootProps()}
            className={`border-2 border-dashed rounded-lg p-12 text-center cursor-pointer transition-colors
              ${isDragActive ? 'border-primary bg-primary/5' : 'border-border hover:border-primary/50'}`}
          >
            <input {...getInputProps()} />
            <Upload className="w-12 h-12 mx-auto mb-4 text-muted-foreground" />
            {isUploading ? (
              <p className="text-sm text-muted-foreground">Przetwarzanie...</p>
            ) : isDragActive ? (
              <p className="text-sm text-muted-foreground">Upuść plik tutaj...</p>
            ) : (
              <div>
                <p className="text-sm font-medium">Przeciągnij plik CSV lub kliknij aby wybrać</p>
                <p className="text-xs text-muted-foreground mt-1">
                  Obsługiwane formaty: CSV
                </p>
              </div>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Stats */}
      {stats && (
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-sm font-medium">Wszystkie</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{stats.total}</div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-sm font-medium">Dopasowane</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600">{stats.matched}</div>
              <p className="text-xs text-muted-foreground mt-1">
                {Math.round((stats.matched / stats.total) * 100)}%
              </p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-sm font-medium">Niedopasowane</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-red-600">{stats.unmatched}</div>
              <p className="text-xs text-muted-foreground mt-1">
                {Math.round((stats.unmatched / stats.total) * 100)}%
              </p>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Results Table */}
      {matchResults.length > 0 && (
        <Card>
          <CardHeader className="flex flex-row items-center justify-between">
            <div>
              <CardTitle>Wyniki matchowania</CardTitle>
              <CardDescription>
                Zielone - dopasowane (≥60%), Czerwone - niedopasowane
              </CardDescription>
            </div>
            <Button
              onClick={handleApplySku}
              disabled={isSaving || !matchResults.some(r => r.systemProduct)}
              data-testid="button-apply-sku"
            >
              <Save className="w-4 h-4 mr-2" />
              {isSaving ? 'Zapisywanie...' : 'Zapisz SKU do produktów'}
            </Button>
          </CardHeader>
          <CardContent>
            <ScrollArea className="h-[600px]">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Status</TableHead>
                    <TableHead>SKU Airtable</TableHead>
                    <TableHead>Nazwa (Airtable)</TableHead>
                    <TableHead>Produkt w systemie</TableHead>
                    <TableHead>Score</TableHead>
                    <TableHead>Szczegóły</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {matchResults.map((result, idx) => (
                    <TableRow key={idx} data-testid={`row-match-${idx}`}>
                      <TableCell>
                        {result.systemProduct ? (
                          <CheckCircle className="w-5 h-5 text-green-600" />
                        ) : (
                          <XCircle className="w-5 h-5 text-red-600" />
                        )}
                      </TableCell>
                      <TableCell className="font-mono text-sm">
                        {result.airtableProduct.sku}
                      </TableCell>
                      <TableCell className="max-w-xs truncate">
                        {result.airtableProduct.name}
                      </TableCell>
                      <TableCell>
                        {result.systemProduct ? (
                          <div>
                            <div className="font-medium">{result.systemProduct.title}</div>
                            <div className="text-xs text-muted-foreground">
                              SKU: {result.systemProduct.sku}
                            </div>
                          </div>
                        ) : (
                          <span className="text-muted-foreground text-sm">Brak dopasowania</span>
                        )}
                      </TableCell>
                      <TableCell>
                        <Badge
                          variant={result.matchScore >= 60 ? 'default' : 'destructive'}
                          data-testid={`badge-score-${idx}`}
                        >
                          {result.matchScore}%
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="space-y-1">
                          {result.matchDetails.map((detail, i) => (
                            <div key={i} className="text-xs flex items-start gap-1">
                              {detail.startsWith('✓') ? (
                                <CheckCircle className="w-3 h-3 text-green-600 mt-0.5 flex-shrink-0" />
                              ) : detail.startsWith('✗') ? (
                                <AlertCircle className="w-3 h-3 text-red-600 mt-0.5 flex-shrink-0" />
                              ) : null}
                              <span>{detail.replace(/^[✓✗]\s*/, '')}</span>
                            </div>
                          ))}
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </ScrollArea>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
