import { useState, useRef } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Plus, Edit2, Trash2, Loader2, Package, Upload, X, Image as ImageIcon } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { apiRequest, queryClient } from "@/lib/queryClient";
import type { ProductAddon } from "@shared/schema";
import { CSVImportDialog } from "@/components/csv-import-dialog";

const addonFormSchema = z.object({
  addonType: z.enum(["fabric", "board", "certificate", "accessory", "component"]),
  name: z.string().min(1, "Nazwa jest wymagana"),
  code: z.string().optional(),
  description: z.string().optional(),
  htmlTemplate: z.string().optional(),
  isActive: z.boolean().default(true),
});

type AddonFormData = z.infer<typeof addonFormSchema>;

const ADDON_TYPE_LABELS: Record<string, string> = {
  fabric: "Tkanina",
  board: "Płyta",
  certificate: "Certyfikat",
  accessory: "Akcesorium",
  component: "Komponent",
};

const CSV_TEMPLATE = `addonType,name,code,description,htmlTemplate,isActive
fabric,Tkanina przykładowa,TK001,Opis tkaniny,<p>HTML template</p>,true
board,Płyta meblowa,PL001,Opis płyty,,true
certificate,Certyfikat FSC,CERT001,Certyfikat jakości,,true
accessory,Nóżki metalowe,AKC001,Nóżki do mebli,,true
component,Zawiasy,KOMP001,Zawiasy drzwiowe,,true`;

const CSV_FILENAME = "szablon_dodatki.csv";

export default function AddonsManager() {
  const { toast } = useToast();
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingAddon, setEditingAddon] = useState<ProductAddon | null>(null);
  const [filterType, setFilterType] = useState<string>("all");
  const [uploadingImages, setUploadingImages] = useState(false);
  const [isDraggingFiles, setIsDraggingFiles] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  // Fetch addons
  const { data: addons, isLoading } = useQuery<ProductAddon[]>({
    queryKey: ["/api/product-addons", { type: filterType === "all" ? undefined : filterType }],
  });

  // Create mutation
  const createMutation = useMutation({
    mutationFn: async (data: AddonFormData) => {
      return await apiRequest("POST", "/api/product-addons", {
        addonType: data.addonType,
        name: data.name,
        code: data.code || null,
        description: data.description || null,
        htmlTemplate: data.htmlTemplate || null,
        attributes: {},
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/product-addons"] });
      toast({ title: "Dodatek utworzony pomyślnie" });
      setIsDialogOpen(false);
      form.reset();
    },
    onError: () => {
      toast({ title: "Błąd tworzenia dodatku", variant: "destructive" });
    },
  });

  // Update mutation
  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: Partial<AddonFormData> }) => {
      return await apiRequest("PATCH", `/api/product-addons/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/product-addons"] });
      toast({ title: "Dodatek zaktualizowany" });
      setIsDialogOpen(false);
      setEditingAddon(null);
      form.reset();
    },
    onError: () => {
      toast({ title: "Błąd aktualizacji dodatku", variant: "destructive" });
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest("DELETE", `/api/product-addons/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/product-addons"] });
      toast({ title: "Dodatek usunięty" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania dodatku", variant: "destructive" });
    },
  });

  // Addon images query (only when editing)
  const { data: addonImages } = useQuery<any[]>({
    queryKey: editingAddon ? [`/api/product-addons/${editingAddon.id}/images`] : [],
    enabled: !!editingAddon,
  });

  // Delete image mutation
  const deleteImageMutation = useMutation({
    mutationFn: async ({ addonId, imageId }: { addonId: number; imageId: number }) => {
      return await apiRequest("DELETE", `/api/product-addons/${addonId}/images/${imageId}`);
    },
    onSuccess: () => {
      if (editingAddon) {
        queryClient.invalidateQueries({ queryKey: [`/api/product-addons/${editingAddon.id}/images`] });
      }
      toast({ title: "Zdjęcie usunięte" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania zdjęcia", variant: "destructive" });
    },
  });

  const form = useForm<AddonFormData>({
    resolver: zodResolver(addonFormSchema),
    defaultValues: {
      addonType: "fabric",
      name: "",
      code: "",
      description: "",
      htmlTemplate: "",
      isActive: true,
    },
  });

  const onSubmit = (data: AddonFormData) => {
    if (editingAddon) {
      updateMutation.mutate({ id: editingAddon.id, data });
    } else {
      createMutation.mutate(data);
    }
  };

  const handleEdit = (addon: ProductAddon) => {
    setEditingAddon(addon);
    form.reset({
      addonType: addon.addonType as any,
      name: addon.name,
      code: addon.code || "",
      description: addon.description || "",
      htmlTemplate: addon.htmlTemplate || "",
      isActive: addon.isActive ?? true,
    });
    setIsDialogOpen(true);
  };

  const handleAdd = () => {
    setEditingAddon(null);
    form.reset({
      addonType: "fabric",
      name: "",
      code: "",
      description: "",
      htmlTemplate: "",
      isActive: true,
    });
    setIsDialogOpen(true);
  };

  const handleDelete = (addon: ProductAddon) => {
    if (window.confirm(`Czy na pewno chcesz usunąć dodatek "${addon.name}"?`)) {
      deleteMutation.mutate(addon.id);
    }
  };

  // Image upload handlers
  const uploadFiles = async (files: FileList) => {
    if (!editingAddon) {
      toast({ title: "Zapisz dodatek przed dodaniem zdjęć", variant: "destructive" });
      return;
    }

    setUploadingImages(true);
    let successCount = 0;
    let failedCount = 0;

    for (const file of Array.from(files)) {
      const formData = new FormData();
      formData.append("image", file);

      try {
        const response = await fetch(`/api/product-addons/${editingAddon.id}/images`, {
          method: "POST",
          body: formData,
          credentials: "include",
        });

        if (response.ok) {
          successCount++;
        } else {
          const errorData = await response.json().catch(() => ({ error: "Unknown error" }));
          console.error("Upload failed:", errorData);
          failedCount++;
        }
      } catch (error) {
        console.error("Upload error:", error);
        failedCount++;
      }
    }

    queryClient.invalidateQueries({ queryKey: [`/api/product-addons/${editingAddon.id}/images`] });

    if (successCount > 0 && failedCount === 0) {
      toast({ title: `${successCount} zdjęć dodanych` });
    } else if (successCount > 0 && failedCount > 0) {
      toast({ title: `${successCount} zdjęć dodanych, ${failedCount} nieudanych` });
    } else {
      toast({ title: "Wszystkie uploady nieudane", variant: "destructive" });
    }

    setUploadingImages(false);
    if (fileInputRef.current) {
      fileInputRef.current.value = "";
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files) {
      uploadFiles(e.target.files);
    }
  };

  const handleFileDrop = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDraggingFiles(false);

    const files = e.dataTransfer.files;
    if (files.length > 0) {
      uploadFiles(files);
    }
  };

  const handleDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDraggingFiles(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    const rect = e.currentTarget.getBoundingClientRect();
    const x = e.clientX;
    const y = e.clientY;
    if (x <= rect.left || x >= rect.right || y <= rect.top || y >= rect.bottom) {
      setIsDraggingFiles(false);
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  const handleDeleteImage = (imageId: number) => {
    if (!editingAddon) return;
    if (window.confirm("Czy na pewno chcesz usunąć to zdjęcie?")) {
      deleteImageMutation.mutate({ addonId: editingAddon.id, imageId });
    }
  };

  const filteredAddons = addons || [];

  return (
    <div className="container mx-auto py-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Zarządzanie dodatkami</h1>
          <p className="text-muted-foreground mt-2">
            Tkaniny, płyty, certyfikaty, akcesoria i komponenty
          </p>
        </div>
        <div className="flex gap-2">
          <CSVImportDialog
            endpoint="/api/product-addons/import-csv"
            templateCsv={CSV_TEMPLATE}
            templateFilename={CSV_FILENAME}
            invalidateQueryKey={["/api/product-addons"]}
            title="Import dodatków z CSV"
            description="Importuj wiele dodatków jednocześnie z pliku CSV"
          />
          <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
            <DialogTrigger asChild>
              <Button onClick={handleAdd} data-testid="button-add-addon">
                <Plus className="mr-2 h-4 w-4" />
                Dodaj dodatek
              </Button>
            </DialogTrigger>
          <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>
                {editingAddon ? "Edytuj dodatek" : "Nowy dodatek"}
              </DialogTitle>
              <DialogDescription>
                {editingAddon ? "Zaktualizuj informacje o dodatku" : "Dodaj nowy dodatek do katalogu"}
              </DialogDescription>
            </DialogHeader>
            <Form {...form}>
              <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
                <FormField
                  control={form.control}
                  name="addonType"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Typ dodatku</FormLabel>
                      <Select onValueChange={field.onChange} value={field.value}>
                        <FormControl>
                          <SelectTrigger data-testid="select-addon-type">
                            <SelectValue placeholder="Wybierz typ" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="fabric">Tkanina</SelectItem>
                          <SelectItem value="board">Płyta</SelectItem>
                          <SelectItem value="certificate">Certyfikat</SelectItem>
                          <SelectItem value="accessory">Akcesorium</SelectItem>
                          <SelectItem value="component">Komponent</SelectItem>
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Nazwa</FormLabel>
                      <FormControl>
                        <Input placeholder="np. Tkanina velvet szary" {...field} data-testid="input-addon-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="code"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kod/SKU (opcjonalnie)</FormLabel>
                      <FormControl>
                        <Input placeholder="np. VEL-GREY-001" {...field} data-testid="input-addon-code" />
                      </FormControl>
                      <FormDescription>
                        Unikalny kod dodatku do identyfikacji
                      </FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="description"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Opis (opcjonalnie)</FormLabel>
                      <FormControl>
                        <Textarea 
                          placeholder="Opis dodatku..."
                          rows={3}
                          {...field}
                          data-testid="textarea-addon-description"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="htmlTemplate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Szablon HTML (opcjonalnie)</FormLabel>
                      <FormControl>
                        <Textarea 
                          placeholder="<div>Twój szablon HTML...</div>"
                          rows={4}
                          className="font-mono text-sm"
                          {...field}
                          data-testid="textarea-addon-html"
                        />
                      </FormControl>
                      <FormDescription>
                        Szablon HTML do wstawienia w opis produktu
                      </FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="isActive"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                      <div className="space-y-0.5">
                        <FormLabel className="text-base">Aktywny</FormLabel>
                        <FormDescription>
                          Dodatek będzie widoczny i dostępny do przypisania
                        </FormDescription>
                      </div>
                      <FormControl>
                        <Switch
                          checked={field.value}
                          onCheckedChange={field.onChange}
                          data-testid="switch-addon-active"
                        />
                      </FormControl>
                    </FormItem>
                  )}
                />

                {/* Image Upload Section - Only visible when editing */}
                {editingAddon && (
                  <div className="space-y-4 pt-4 border-t">
                    <h3 className="text-sm font-medium">Zdjęcia dodatku</h3>
                    
                    {/* Drag & Drop Zone */}
                    <div
                      onDrop={handleFileDrop}
                      onDragOver={handleDragOver}
                      onDragEnter={handleDragEnter}
                      onDragLeave={handleDragLeave}
                      className={`
                        border-2 border-dashed rounded-lg p-6 text-center transition-colors
                        ${isDraggingFiles ? 'border-primary bg-primary/5' : 'border-border hover:border-primary/50'}
                      `}
                    >
                      <input
                        ref={fileInputRef}
                        type="file"
                        accept="image/*"
                        multiple
                        onChange={handleFileChange}
                        className="hidden"
                        data-testid="input-addon-image"
                      />
                      <Upload className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
                      <p className="text-sm text-muted-foreground mb-2">
                        Przeciągnij i upuść zdjęcia tutaj lub
                      </p>
                      <Button
                        type="button"
                        variant="outline"
                        size="sm"
                        onClick={() => fileInputRef.current?.click()}
                        disabled={uploadingImages}
                        data-testid="button-upload-image"
                      >
                        {uploadingImages ? (
                          <>
                            <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                            Przesyłanie...
                          </>
                        ) : (
                          <>
                            <ImageIcon className="mr-2 h-4 w-4" />
                            Wybierz pliki
                          </>
                        )}
                      </Button>
                      <p className="text-xs text-muted-foreground mt-2">
                        JPG, PNG, GIF, WEBP (max 10MB każde)
                      </p>
                    </div>

                    {/* Image Gallery */}
                    {addonImages && addonImages.length > 0 && (
                      <div className="grid grid-cols-3 gap-4">
                        {addonImages.map((image: any) => (
                          <div key={image.id} className="relative group rounded-lg overflow-hidden border">
                            <img
                              src={image.url}
                              alt={image.altText || image.title || 'Addon image'}
                              className="w-full h-32 object-cover"
                            />
                            <Button
                              type="button"
                              variant="destructive"
                              size="icon"
                              className="absolute top-2 right-2 h-8 w-8 opacity-0 group-hover:opacity-100 transition-opacity"
                              onClick={() => handleDeleteImage(image.id)}
                              data-testid={`button-delete-image-${image.id}`}
                            >
                              <X className="h-4 w-4" />
                            </Button>
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                )}

                <div className="flex justify-end gap-2 pt-4">
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() => {
                      setIsDialogOpen(false);
                      setEditingAddon(null);
                      form.reset();
                    }}
                    data-testid="button-cancel"
                  >
                    Anuluj
                  </Button>
                  <Button 
                    type="submit"
                    disabled={createMutation.isPending || updateMutation.isPending}
                    data-testid="button-save-addon"
                  >
                    {(createMutation.isPending || updateMutation.isPending) && (
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    )}
                    {editingAddon ? "Zapisz zmiany" : "Dodaj dodatek"}
                  </Button>
                </div>
              </form>
            </Form>
          </DialogContent>
        </Dialog>
        </div>
      </div>

      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>Lista dodatków</CardTitle>
              <CardDescription>
                Wszystkie dodatki dostępne w systemie
              </CardDescription>
            </div>
            <Select value={filterType} onValueChange={setFilterType}>
              <SelectTrigger className="w-[200px]" data-testid="select-filter-type">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Wszystkie typy</SelectItem>
                <SelectItem value="fabric">Tkaniny</SelectItem>
                <SelectItem value="board">Płyty</SelectItem>
                <SelectItem value="certificate">Certyfikaty</SelectItem>
                <SelectItem value="accessory">Akcesoria</SelectItem>
                <SelectItem value="component">Komponenty</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex justify-center py-12">
              <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
            </div>
          ) : filteredAddons.length === 0 ? (
            <div className="text-center py-12">
              <Package className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
              <p className="text-muted-foreground">Brak dodatków</p>
              <p className="text-sm text-muted-foreground mt-2">
                Kliknij "Dodaj dodatek" aby utworzyć pierwszy dodatek
              </p>
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Typ</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Kod</TableHead>
                  <TableHead>Opis</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead className="text-right">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredAddons.map((addon) => (
                  <TableRow key={addon.id} data-testid={`row-addon-${addon.id}`}>
                    <TableCell>
                      <Badge variant="outline" data-testid={`badge-type-${addon.id}`}>
                        {ADDON_TYPE_LABELS[addon.addonType] || addon.addonType}
                      </Badge>
                    </TableCell>
                    <TableCell className="font-medium" data-testid={`text-name-${addon.id}`}>
                      {addon.name}
                    </TableCell>
                    <TableCell className="text-muted-foreground">
                      {addon.code || "-"}
                    </TableCell>
                    <TableCell className="max-w-xs truncate text-muted-foreground">
                      {addon.description || "-"}
                    </TableCell>
                    <TableCell>
                      {addon.isActive ? (
                        <Badge variant="default">Aktywny</Badge>
                      ) : (
                        <Badge variant="secondary">Nieaktywny</Badge>
                      )}
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex justify-end gap-2">
                        <Button
                          size="icon"
                          variant="ghost"
                          onClick={() => handleEdit(addon)}
                          data-testid={`button-edit-${addon.id}`}
                        >
                          <Edit2 className="h-4 w-4" />
                        </Button>
                        <Button
                          size="icon"
                          variant="ghost"
                          onClick={() => handleDelete(addon)}
                          disabled={deleteMutation.isPending}
                          data-testid={`button-delete-${addon.id}`}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
