import { useState, useEffect, useMemo, useCallback } from 'react';
import { useQuery, useMutation } from '@tanstack/react-query';
import { apiRequest, queryClient } from '@/lib/queryClient';

export type FilterState = 'include' | 'exclude';
export type FilterStates = Record<string, FilterState>;
export type StockFilter = 'all' | 'inStock' | 'outOfStock';

export interface ColumnDefinition {
  key: string;
  label: string;
  defaultWidth: number;
  defaultVisible?: boolean;
  sortKey?: string;
}

export interface FilterGroupDefinition {
  key: string;
  label: string;
  type: 'triState' | 'stock' | 'multiSelect';
  paramName?: string;
  excludeParamName?: string;
}

export interface WarehouseTableConfig {
  categorySlug: string;
  userSettingsKey: string;
  columns: ColumnDefinition[];
  filterGroups: FilterGroupDefinition[];
  defaultSortBy?: string;
  defaultSortOrder?: 'ASC' | 'DESC';
}

export interface WarehouseTableState {
  searchQuery: string;
  setSearchQuery: (q: string) => void;
  page: number;
  setPage: (p: number) => void;
  limit: number;
  setLimit: (l: number) => void;
  sortBy: string;
  setSortBy: (s: string) => void;
  sortOrder: 'ASC' | 'DESC';
  setSortOrder: (o: 'ASC' | 'DESC') => void;
  
  columnVisibility: Record<string, boolean>;
  setColumnVisibility: (v: Record<string, boolean>) => void;
  columnOrder: string[];
  setColumnOrder: (o: string[]) => void;
  columnWidths: Record<string, number>;
  setColumnWidths: (w: Record<string, number>) => void;
  isColumnVisible: (key: string) => boolean;
  getColumnWidth: (key: string) => number;
  getOrderedColumns: () => ColumnDefinition[];
  
  filterStates: Record<string, FilterStates>;
  setFilterState: (groupKey: string, itemKey: string, state: FilterState | undefined) => void;
  cycleFilterState: (groupKey: string, itemKey: string) => void;
  getFilterState: (groupKey: string, itemKey: string) => FilterState | undefined;
  clearFilterGroup: (groupKey: string) => void;
  
  stockFilter: StockFilter;
  setStockFilter: (f: StockFilter) => void;
  
  multiSelectFilters: Record<string, string[]>;
  setMultiSelectFilter: (groupKey: string, values: string[]) => void;
  
  filtersExpanded: boolean;
  setFiltersExpanded: (e: boolean) => void;
  filterGroupsVisible: Record<string, boolean>;
  setFilterGroupVisible: (key: string, visible: boolean) => void;
  
  selectedIds: number[];
  setSelectedIds: (ids: number[]) => void;
  toggleSelection: (id: number) => void;
  selectAll: (ids: number[]) => void;
  clearSelection: () => void;
  
  buildQueryParams: () => Record<string, string>;
  
  userSettingsLoaded: boolean;
  saveSettings: () => void;
}

export function useWarehouseTableState(config: WarehouseTableConfig): WarehouseTableState {
  const { categorySlug, userSettingsKey, columns, filterGroups, defaultSortBy = 'name', defaultSortOrder = 'ASC' } = config;
  
  const defaultColumnVisibility = useMemo(() => 
    columns.reduce((acc, col) => {
      acc[col.key] = col.defaultVisible !== false;
      return acc;
    }, {} as Record<string, boolean>),
    [columns]
  );
  
  const defaultColumnWidths = useMemo(() => 
    columns.reduce((acc, col) => {
      acc[col.key] = col.defaultWidth;
      return acc;
    }, {} as Record<string, number>),
    [columns]
  );
  
  const defaultColumnOrder = useMemo(() => columns.map(col => col.key), [columns]);
  
  const [searchQuery, setSearchQuery] = useState('');
  const [page, setPage] = useState(1);
  const [limit, setLimit] = useState(25);
  const [sortBy, setSortBy] = useState(defaultSortBy);
  const [sortOrder, setSortOrder] = useState<'ASC' | 'DESC'>(defaultSortOrder);
  
  const [columnVisibility, setColumnVisibility] = useState<Record<string, boolean>>(defaultColumnVisibility);
  const [columnOrder, setColumnOrder] = useState<string[]>(defaultColumnOrder);
  const [columnWidths, setColumnWidths] = useState<Record<string, number>>(defaultColumnWidths);
  
  const [filterStates, setFilterStates] = useState<Record<string, FilterStates>>({});
  const [stockFilter, setStockFilter] = useState<StockFilter>('all');
  const [multiSelectFilters, setMultiSelectFiltersState] = useState<Record<string, string[]>>({});
  
  const [filtersExpanded, setFiltersExpanded] = useState(true);
  const [filterGroupsVisible, setFilterGroupsVisibleState] = useState<Record<string, boolean>>(() => {
    return filterGroups.reduce((acc, fg) => {
      acc[fg.key] = true;
      return acc;
    }, {} as Record<string, boolean>);
  });
  
  const [selectedIds, setSelectedIds] = useState<number[]>([]);
  const [userSettingsLoaded, setUserSettingsLoaded] = useState(false);
  
  const { data: serverSettings, isSuccess: settingsLoaded } = useQuery<any>({
    queryKey: [`/api/user-settings/${userSettingsKey}`],
  });
  
  const saveSettingsMutation = useMutation({
    mutationFn: (settings: any) => apiRequest('PUT', `/api/user-settings/${userSettingsKey}`, settings),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/user-settings/${userSettingsKey}`] });
    }
  });
  
  useEffect(() => {
    if (settingsLoaded && serverSettings && !userSettingsLoaded) {
      const f = serverSettings.filters || {};
      
      if (f.searchQuery) setSearchQuery(f.searchQuery);
      if (f.page) setPage(f.page);
      if (f.limit) setLimit(f.limit);
      if (f.sortBy) setSortBy(f.sortBy);
      if (f.sortOrder) setSortOrder(f.sortOrder);
      if (f.stockFilter) setStockFilter(f.stockFilter);
      if (f.filtersExpanded !== undefined) setFiltersExpanded(f.filtersExpanded);
      if (f.filterGroupsVisible) setFilterGroupsVisibleState(f.filterGroupsVisible);
      
      const loadedFilterStates: Record<string, FilterStates> = {};
      filterGroups.forEach(fg => {
        const stateKey = `${fg.key}FilterStates`;
        if (f[stateKey]) {
          loadedFilterStates[fg.key] = f[stateKey];
        }
      });
      if (Object.keys(loadedFilterStates).length > 0) {
        setFilterStates(loadedFilterStates);
      }
      
      const loadedMultiSelects: Record<string, string[]> = {};
      filterGroups.filter(fg => fg.type === 'multiSelect').forEach(fg => {
        const stateKey = `selected${fg.key.charAt(0).toUpperCase() + fg.key.slice(1)}`;
        if (f[stateKey]) {
          loadedMultiSelects[fg.key] = f[stateKey];
        }
      });
      if (Object.keys(loadedMultiSelects).length > 0) {
        setMultiSelectFiltersState(loadedMultiSelects);
      }
      
      if (serverSettings.columnVisibility) {
        setColumnVisibility(serverSettings.columnVisibility);
      }
      if (serverSettings.columnWidths) {
        setColumnWidths(serverSettings.columnWidths);
      }
      if (serverSettings.columnOrder && serverSettings.columnOrder.length > 0) {
        const savedOrder = serverSettings.columnOrder;
        const allKeys = columns.map(col => col.key);
        const missingKeys = allKeys.filter(k => !savedOrder.includes(k));
        setColumnOrder([...savedOrder.filter((k: string) => allKeys.includes(k)), ...missingKeys]);
      }
      
      setUserSettingsLoaded(true);
    }
  }, [settingsLoaded, serverSettings, userSettingsLoaded, columns, filterGroups]);
  
  const saveSettings = useCallback(() => {
    if (!userSettingsLoaded) return;
    
    const filtersToSave: Record<string, any> = {
      searchQuery,
      page,
      limit,
      sortBy,
      sortOrder,
      stockFilter,
      filtersExpanded,
      filterGroupsVisible,
    };
    
    filterGroups.forEach(fg => {
      const stateKey = `${fg.key}FilterStates`;
      filtersToSave[stateKey] = filterStates[fg.key] || {};
    });
    
    filterGroups.filter(fg => fg.type === 'multiSelect').forEach(fg => {
      const stateKey = `selected${fg.key.charAt(0).toUpperCase() + fg.key.slice(1)}`;
      filtersToSave[stateKey] = multiSelectFilters[fg.key] || [];
    });
    
    saveSettingsMutation.mutate({
      filters: filtersToSave,
      columnVisibility,
      columnWidths,
      columnOrder,
    });
  }, [
    userSettingsLoaded, searchQuery, page, limit, sortBy, sortOrder, stockFilter,
    filtersExpanded, filterGroupsVisible, filterStates, multiSelectFilters,
    columnVisibility, columnWidths, columnOrder, filterGroups, saveSettingsMutation
  ]);
  
  useEffect(() => {
    if (userSettingsLoaded) {
      const timeoutId = setTimeout(saveSettings, 500);
      return () => clearTimeout(timeoutId);
    }
  }, [
    searchQuery, page, limit, sortBy, sortOrder, stockFilter,
    filtersExpanded, filterGroupsVisible, filterStates, multiSelectFilters,
    columnVisibility, columnWidths, columnOrder, userSettingsLoaded
  ]);
  
  const isColumnVisible = useCallback((key: string) => columnVisibility[key] !== false, [columnVisibility]);
  
  const getColumnWidth = useCallback((key: string) => 
    columnWidths[key] || defaultColumnWidths[key] || 100,
    [columnWidths, defaultColumnWidths]
  );
  
  const getOrderedColumns = useCallback(() => {
    const orderedCols = columnOrder
      .map(key => columns.find(col => col.key === key))
      .filter((col): col is ColumnDefinition => col !== undefined);
    const missingCols = columns.filter(col => !columnOrder.includes(col.key));
    return [...orderedCols, ...missingCols];
  }, [columnOrder, columns]);
  
  const setFilterState = useCallback((groupKey: string, itemKey: string, state: FilterState | undefined) => {
    setFilterStates(prev => {
      const groupStates = { ...(prev[groupKey] || {}) };
      if (state === undefined) {
        delete groupStates[itemKey];
      } else {
        groupStates[itemKey] = state;
      }
      return { ...prev, [groupKey]: groupStates };
    });
  }, []);
  
  const cycleFilterState = useCallback((groupKey: string, itemKey: string) => {
    setFilterStates(prev => {
      const groupStates = { ...(prev[groupKey] || {}) };
      const current = groupStates[itemKey];
      if (!current) {
        groupStates[itemKey] = 'include';
      } else if (current === 'include') {
        groupStates[itemKey] = 'exclude';
      } else {
        delete groupStates[itemKey];
      }
      return { ...prev, [groupKey]: groupStates };
    });
  }, []);
  
  const getFilterState = useCallback((groupKey: string, itemKey: string) => {
    return filterStates[groupKey]?.[itemKey];
  }, [filterStates]);
  
  const clearFilterGroup = useCallback((groupKey: string) => {
    setFilterStates(prev => ({ ...prev, [groupKey]: {} }));
  }, []);
  
  const setMultiSelectFilter = useCallback((groupKey: string, values: string[]) => {
    setMultiSelectFiltersState(prev => ({ ...prev, [groupKey]: values }));
  }, []);
  
  const setFilterGroupVisible = useCallback((key: string, visible: boolean) => {
    setFilterGroupsVisibleState(prev => ({ ...prev, [key]: visible }));
  }, []);
  
  const toggleSelection = useCallback((id: number) => {
    setSelectedIds(prev => 
      prev.includes(id) ? prev.filter(i => i !== id) : [...prev, id]
    );
  }, []);
  
  const selectAll = useCallback((ids: number[]) => {
    setSelectedIds(ids);
  }, []);
  
  const clearSelection = useCallback(() => {
    setSelectedIds([]);
  }, []);
  
  const buildQueryParams = useCallback(() => {
    const params: Record<string, string> = {
      category: categorySlug,
      page: page.toString(),
      limit: limit.toString(),
      sortBy,
      sortOrder,
    };
    
    if (searchQuery) {
      params.search = searchQuery;
    }
    
    if (stockFilter !== 'all') {
      params.stockFilter = stockFilter;
    }
    
    filterGroups.forEach(fg => {
      if (fg.type === 'triState') {
        const states = filterStates[fg.key] || {};
        const includes = Object.entries(states)
          .filter(([_, state]) => state === 'include')
          .map(([key]) => key);
        const excludes = Object.entries(states)
          .filter(([_, state]) => state === 'exclude')
          .map(([key]) => key);
        
        if (includes.length > 0 && fg.paramName) {
          params[fg.paramName] = includes.join(',');
        }
        if (excludes.length > 0 && fg.excludeParamName) {
          params[fg.excludeParamName] = excludes.join(',');
        }
      } else if (fg.type === 'multiSelect') {
        const values = multiSelectFilters[fg.key] || [];
        if (values.length > 0 && fg.paramName) {
          params[fg.paramName] = values.join(',');
        }
      }
    });
    
    return params;
  }, [categorySlug, page, limit, sortBy, sortOrder, searchQuery, stockFilter, filterGroups, filterStates, multiSelectFilters]);
  
  return {
    searchQuery,
    setSearchQuery,
    page,
    setPage,
    limit,
    setLimit,
    sortBy,
    setSortBy,
    sortOrder,
    setSortOrder,
    
    columnVisibility,
    setColumnVisibility,
    columnOrder,
    setColumnOrder,
    columnWidths,
    setColumnWidths,
    isColumnVisible,
    getColumnWidth,
    getOrderedColumns,
    
    filterStates,
    setFilterState,
    cycleFilterState,
    getFilterState,
    clearFilterGroup,
    
    stockFilter,
    setStockFilter,
    
    multiSelectFilters,
    setMultiSelectFilter,
    
    filtersExpanded,
    setFiltersExpanded,
    filterGroupsVisible,
    setFilterGroupVisible,
    
    selectedIds,
    setSelectedIds,
    toggleSelection,
    selectAll,
    clearSelection,
    
    buildQueryParams,
    
    userSettingsLoaded,
    saveSettings,
  };
}
