import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from "@/components/ui/collapsible";
import { useToast } from "@/hooks/use-toast";
import {
  Plus,
  Truck,
  Package,
  ChevronDown,
  ChevronRight,
  Trash2,
  Edit,
  Play,
  CheckCircle,
  Clock,
  Loader2,
  ArrowUpDown,
  GripVertical,
} from "lucide-react";

interface TransportBatch {
  id: number;
  plan_id: number;
  batch_number: string;
  label: string | null;
  target_stage: string | null;
  destination_location_id: number | null;
  destination_location_name: string | null;
  sort_order: number;
  planned_departure_at: string | null;
  actual_departure_at: string | null;
  status: string;
  item_count: number;
  actual_item_count: number;
  ready_item_count: number;
  families_required: string[] | null;
  families_ready: string[] | null;
}

interface TransportBatchItem {
  id: number;
  batch_id: number;
  production_order_id: number | null;
  zlp_number: string | null;
  zlp_status: string | null;
  color_code: string | null;
  cutting_sequence: number | null;
  cutting_priority: string | null;
  product_name: string | null;
  component_family: string | null;
  status: string;
}

interface TransportBatchesPanelProps {
  planId: number;
}

const statusColors: Record<string, string> = {
  pending: "bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200",
  loading: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200",
  in_transit: "bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-200",
  delivered: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200",
  cancelled: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200",
};

const statusLabels: Record<string, string> = {
  pending: "Oczekuje",
  loading: "Załadunek",
  in_transit: "W trasie",
  delivered: "Dostarczono",
  cancelled: "Anulowano",
};

const colorBadgeColors: Record<string, string> = {
  BIALY: "bg-white border border-gray-300 text-gray-800",
  SONOMA: "bg-amber-200 text-amber-900",
  WOTAN: "bg-amber-700 text-white",
  SUROWA: "bg-yellow-100 text-yellow-800",
  HDF: "bg-stone-400 text-white",
  CZARNY: "bg-black text-white",
};

export function TransportBatchesPanel({ planId }: TransportBatchesPanelProps) {
  const { toast } = useToast();
  const [createDialogOpen, setCreateDialogOpen] = useState(false);
  const [editingBatch, setEditingBatch] = useState<TransportBatch | null>(null);
  const [expandedBatches, setExpandedBatches] = useState<Set<number>>(new Set());
  const [newBatch, setNewBatch] = useState({
    label: "",
    targetStage: "",
  });

  const { data: batches = [], isLoading } = useQuery<TransportBatch[]>({
    queryKey: [`/api/production/planning/plans/${planId}/transport-batches`],
    enabled: !!planId,
  });

  const createMutation = useMutation({
    mutationFn: async (data: typeof newBatch) => {
      return apiRequest("POST", `/api/production/planning/plans/${planId}/transport-batches`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/transport-batches`] });
      setCreateDialogOpen(false);
      setNewBatch({ label: "", targetStage: "" });
      toast({ title: "Utworzono partię transportową" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się utworzyć partii",
        variant: "destructive" 
      });
    },
  });

  const updateStatusMutation = useMutation({
    mutationFn: async ({ id, status }: { id: number; status: string }) => {
      return apiRequest("PATCH", `/api/production/planning/transport-batches/${id}`, { status });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/transport-batches`] });
      toast({ title: "Zaktualizowano status partii" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/production/planning/transport-batches/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/transport-batches`] });
      toast({ title: "Usunięto partię transportową" });
    },
  });

  const toggleExpanded = (batchId: number) => {
    setExpandedBatches((prev) => {
      const next = new Set(prev);
      if (next.has(batchId)) {
        next.delete(batchId);
      } else {
        next.add(batchId);
      }
      return next;
    });
  };

  const getNextStatus = (current: string): string | null => {
    const flow: Record<string, string> = {
      pending: "loading",
      loading: "in_transit",
      in_transit: "delivered",
    };
    return flow[current] || null;
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center p-8">
        <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
      </div>
    );
  }

  return (
    <div className="space-y-4">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-2">
          <Truck className="h-5 w-5 text-muted-foreground" />
          <h3 className="font-semibold">Partie transportowe</h3>
          <Badge variant="secondary">{batches.length}</Badge>
        </div>
        <Button
          size="sm"
          onClick={() => setCreateDialogOpen(true)}
          data-testid="button-create-transport-batch"
        >
          <Plus className="h-4 w-4 mr-1" />
          Nowa partia
        </Button>
      </div>

      {batches.length === 0 ? (
        <Card>
          <CardContent className="flex flex-col items-center justify-center py-8 text-center">
            <Truck className="h-12 w-12 text-muted-foreground/50 mb-3" />
            <p className="text-muted-foreground">Brak partii transportowych</p>
            <p className="text-sm text-muted-foreground mt-1">
              Utwórz partie aby pogrupować elementy do wysyłki
            </p>
          </CardContent>
        </Card>
      ) : (
        <div className="space-y-3">
          {batches.map((batch) => (
            <TransportBatchCard
              key={batch.id}
              batch={batch}
              planId={planId}
              isExpanded={expandedBatches.has(batch.id)}
              onToggle={() => toggleExpanded(batch.id)}
              onUpdateStatus={(status) => updateStatusMutation.mutate({ id: batch.id, status })}
              onDelete={() => deleteMutation.mutate(batch.id)}
              getNextStatus={getNextStatus}
            />
          ))}
        </div>
      )}

      <Dialog open={createDialogOpen} onOpenChange={setCreateDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Nowa partia transportowa</DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Numer partii</Label>
              <div className="px-3 py-2 text-sm bg-muted rounded-md text-muted-foreground">
                Numer zostanie wygenerowany automatycznie (np. T1, T2, ...)
              </div>
            </div>
            <div className="space-y-2">
              <Label>Opis (opcjonalnie)</Label>
              <Input
                placeholder="np. BIAŁY + SONOMA do oklejania"
                value={newBatch.label}
                onChange={(e) => setNewBatch({ ...newBatch, label: e.target.value })}
                data-testid="input-batch-label"
              />
            </div>
            <div className="space-y-2">
              <Label>Docelowy etap</Label>
              <Select
                value={newBatch.targetStage}
                onValueChange={(value) => setNewBatch({ ...newBatch, targetStage: value })}
              >
                <SelectTrigger data-testid="select-target-stage">
                  <SelectValue placeholder="Wybierz etap" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="oklejanie">Oklejanie</SelectItem>
                  <SelectItem value="wiercenie">Wiercenie</SelectItem>
                  <SelectItem value="tapicerowanie">Tapicerowanie</SelectItem>
                  <SelectItem value="kompletowanie">Kompletowanie</SelectItem>
                  <SelectItem value="pakowanie">Pakowanie</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setCreateDialogOpen(false)}>
              Anuluj
            </Button>
            <Button
              onClick={() => createMutation.mutate(newBatch)}
              disabled={createMutation.isPending}
              data-testid="button-confirm-create-batch"
            >
              {createMutation.isPending && <Loader2 className="h-4 w-4 mr-2 animate-spin" />}
              Utwórz
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}

function TransportBatchCard({
  batch,
  planId,
  isExpanded,
  onToggle,
  onUpdateStatus,
  onDelete,
  getNextStatus,
}: {
  batch: TransportBatch;
  planId: number;
  isExpanded: boolean;
  onToggle: () => void;
  onUpdateStatus: (status: string) => void;
  onDelete: () => void;
  getNextStatus: (current: string) => string | null;
}) {
  const { data: items = [] } = useQuery<TransportBatchItem[]>({
    queryKey: [`/api/production/planning/transport-batches/${batch.id}/items`],
    enabled: isExpanded,
  });

  const nextStatus = getNextStatus(batch.status);
  const progress = batch.actual_item_count > 0
    ? Math.round((batch.ready_item_count / batch.actual_item_count) * 100)
    : 0;

  return (
    <Card className="overflow-hidden">
      <Collapsible open={isExpanded} onOpenChange={onToggle}>
        <CollapsibleTrigger asChild>
          <CardHeader className="cursor-pointer hover-elevate py-3">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-3">
                {isExpanded ? (
                  <ChevronDown className="h-4 w-4 text-muted-foreground" />
                ) : (
                  <ChevronRight className="h-4 w-4 text-muted-foreground" />
                )}
                <div>
                  <div className="flex items-center gap-2">
                    <span className="font-semibold">{batch.batch_number}</span>
                    <Badge className={statusColors[batch.status]}>
                      {statusLabels[batch.status]}
                    </Badge>
                  </div>
                  {batch.label && (
                    <p className="text-sm text-muted-foreground mt-0.5">{batch.label}</p>
                  )}
                </div>
              </div>
              <div className="flex items-center gap-4">
                <div className="text-right text-sm">
                  <div className="flex items-center gap-1">
                    <Package className="h-3.5 w-3.5 text-muted-foreground" />
                    <span>{batch.ready_item_count}/{batch.actual_item_count}</span>
                  </div>
                  {batch.target_stage && (
                    <div className="text-muted-foreground text-xs">→ {batch.target_stage}</div>
                  )}
                </div>
                <div 
                  className="w-16 h-2 bg-muted rounded-full overflow-hidden"
                  title={`${progress}% gotowe`}
                >
                  <div 
                    className="h-full bg-green-500 transition-all"
                    style={{ width: `${progress}%` }}
                  />
                </div>
              </div>
            </div>
          </CardHeader>
        </CollapsibleTrigger>

        <CollapsibleContent>
          <Separator />
          <CardContent className="pt-4 space-y-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                {nextStatus && batch.status !== 'delivered' && batch.status !== 'cancelled' && (
                  <Button
                    size="sm"
                    onClick={(e) => {
                      e.stopPropagation();
                      onUpdateStatus(nextStatus);
                    }}
                    data-testid={`button-advance-batch-${batch.id}`}
                  >
                    <Play className="h-3.5 w-3.5 mr-1" />
                    {nextStatus === 'loading' && 'Rozpocznij załadunek'}
                    {nextStatus === 'in_transit' && 'Wyślij'}
                    {nextStatus === 'delivered' && 'Oznacz jako dostarczone'}
                  </Button>
                )}
                {batch.status === 'delivered' && (
                  <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                    <CheckCircle className="h-3.5 w-3.5 mr-1" />
                    Dostarczono
                  </Badge>
                )}
              </div>
              <Button
                size="sm"
                variant="ghost"
                className="text-destructive hover:text-destructive"
                onClick={(e) => {
                  e.stopPropagation();
                  onDelete();
                }}
                data-testid={`button-delete-batch-${batch.id}`}
              >
                <Trash2 className="h-4 w-4" />
              </Button>
            </div>

            {items.length > 0 ? (
              <ScrollArea className="h-48">
                <div className="space-y-2">
                  {items.map((item) => (
                    <div
                      key={item.id}
                      className="flex items-center justify-between p-2 bg-muted/50 rounded-md"
                    >
                      <div className="flex items-center gap-2">
                        <GripVertical className="h-4 w-4 text-muted-foreground" />
                        <div>
                          <div className="font-medium text-sm">{item.zlp_number}</div>
                          <div className="text-xs text-muted-foreground">
                            {item.product_name}
                          </div>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        {item.color_code && (
                          <Badge
                            variant="secondary"
                            className={colorBadgeColors[item.color_code] || ""}
                          >
                            {item.color_code}
                          </Badge>
                        )}
                        {item.cutting_sequence && (
                          <Badge variant="outline" className="text-xs">
                            #{item.cutting_sequence}
                          </Badge>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              </ScrollArea>
            ) : (
              <div className="text-center py-4 text-muted-foreground text-sm">
                Brak przypisanych elementów
              </div>
            )}
          </CardContent>
        </CollapsibleContent>
      </Collapsible>
    </Card>
  );
}
