import { useState, useCallback } from "react";
import { Plus, Check, Package, ExternalLink, Layers } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
} from "@/components/ui/dialog";
import { useLocation } from "wouter";
import type { SetComponent, MarketplaceOrderItem } from "./DraggableProductItem";

interface SetComponentsModalProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  setComponents: SetComponent[];
  parentSetSku: string | null;
  parentSetTitle: string | null;
  parentSetImage?: string | null;
  orderNumber: string;
  buyerName: string;
  marketplace: string;
  currentProductId: number;
  onAddComponent: (item: MarketplaceOrderItem, orderNumber: string, buyerName: string, marketplace: string, setId?: number, componentId?: number) => void;
  isAdding?: boolean;
  catalogSetId: number | null;
  originalItem: MarketplaceOrderItem;
  onHighlightInPlan?: (productId: number, orderNumber: string) => void;
}

export function SetComponentsModal({
  open,
  onOpenChange,
  setComponents,
  parentSetSku,
  parentSetTitle,
  parentSetImage,
  orderNumber,
  buyerName,
  marketplace,
  currentProductId,
  onAddComponent,
  isAdding = false,
  catalogSetId,
  originalItem,
  onHighlightInPlan,
}: SetComponentsModalProps) {
  const [, setLocation] = useLocation();
  const [addingComponentId, setAddingComponentId] = useState<number | null>(null);
  const [locallyAddedIds, setLocallyAddedIds] = useState<Set<number>>(new Set());

  const inPlanCount = setComponents.filter(c => c.is_in_any_plan || locallyAddedIds.has(c.component_id)).length;
  const totalCount = setComponents.length;

  const handleAddComponent = useCallback((component: SetComponent) => {
    if (component.is_in_any_plan || locallyAddedIds.has(component.component_id)) return;
    
    setAddingComponentId(component.component_id);
    
    const componentItem: MarketplaceOrderItem = {
      ...originalItem,
      item_id: component.component_id,
      name: component.component_title,
      product_sku: component.component_sku,
      product_length: component.component_length,
      product_width: component.component_width,
      product_height: component.component_height,
      product_color: component.component_color,
      product_type: component.component_product_type,
      product_doors: component.component_doors,
      product_legs: component.component_legs,
      image_url: component.primary_image_url,
      catalog_product_id: component.component_id,
      catalog_set_id: catalogSetId,
      is_in_plan: component.is_in_any_plan,
      in_plan_number: component.in_plan_number,
      in_plan_id: component.in_plan_id,
      in_current_plan: component.is_in_current_plan,
      quantity: component.quantity,
      set_components: null,
    };

    onAddComponent(componentItem, orderNumber, buyerName, marketplace, catalogSetId ?? undefined, component.component_id);
    
    setLocallyAddedIds(prev => {
      const newSet = new Set(prev);
      newSet.add(component.component_id);
      return newSet;
    });
    
    setTimeout(() => setAddingComponentId(null), 500);
  }, [originalItem, catalogSetId, orderNumber, buyerName, marketplace, onAddComponent, locallyAddedIds]);

  const handleAddAllNotInPlan = () => {
    const componentsToAdd = setComponents.filter(c => !c.is_in_any_plan && !locallyAddedIds.has(c.component_id));
    componentsToAdd.forEach(component => {
      handleAddComponent(component);
    });
  };

  const navigateToPlan = (planId: number) => {
    setLocation(`/production/plans/${planId}`);
    onOpenChange(false);
  };

  const handleHighlightInPlan = (componentId: number) => {
    if (onHighlightInPlan) {
      onHighlightInPlan(componentId, orderNumber);
      onOpenChange(false);
    }
  };

  const handleOpenChange = (newOpen: boolean) => {
    if (!newOpen) {
      setLocallyAddedIds(new Set());
    }
    onOpenChange(newOpen);
  };

  return (
    <Dialog open={open} onOpenChange={handleOpenChange}>
      <DialogContent className="max-w-xl max-h-[80vh] overflow-hidden flex flex-col">
        <DialogHeader className="flex-shrink-0">
          <DialogTitle className="flex items-center gap-2">
            <Package className="h-5 w-5 text-purple-500" />
            Komponenty zestawu
          </DialogTitle>
          <DialogDescription className="sr-only">
            Lista komponentów zestawu z możliwością dodania do planu produkcyjnego
          </DialogDescription>
          <div className="flex items-center gap-2 text-sm text-muted-foreground mt-1">
            {parentSetImage ? (
              <img
                src={parentSetImage}
                alt={parentSetTitle || 'Zestaw'}
                className="w-8 h-8 object-cover rounded flex-shrink-0 border border-border"
              />
            ) : (
              <div className="w-8 h-8 bg-muted rounded flex items-center justify-center flex-shrink-0 border border-border">
                <Layers className="h-4 w-4 text-muted-foreground" />
              </div>
            )}
            <Badge variant="outline" className="text-xs">
              {parentSetSku}
            </Badge>
            <span className="truncate">{parentSetTitle}</span>
          </div>
          <div className="flex items-center gap-2 mt-2">
            <Badge 
              variant="outline" 
              className={`text-xs ${inPlanCount === totalCount ? 'bg-green-500/10 border-green-500/30 text-green-700' : 'bg-yellow-500/10 border-yellow-500/30 text-yellow-700'}`}
            >
              {inPlanCount} / {totalCount} w planach
            </Badge>
            <span className="text-xs text-muted-foreground">
              Zamówienie: {orderNumber}
            </span>
          </div>
        </DialogHeader>

        <div className="flex-1 overflow-y-auto mt-4 -mx-6 px-6">
          <div className="space-y-2">
            {setComponents.map((component) => {
              const isCurrentProduct = component.component_id === currentProductId;
              const wasAddedLocally = locallyAddedIds.has(component.component_id);
              const isInPlan = component.is_in_any_plan;
              const isAddingThis = addingComponentId === component.component_id;
              const showAsAdded = isInPlan || wasAddedLocally;

              return (
                <div
                  key={component.component_id}
                  className={`
                    flex items-center gap-3 p-3 rounded-lg border transition-all duration-300
                    ${isCurrentProduct ? 'border-purple-500 bg-purple-500/5' : 'border-border'}
                    ${showAsAdded ? 'bg-green-500/5' : ''}
                  `}
                  data-testid={`set-component-${component.component_id}`}
                >
                  {component.primary_image_url ? (
                    <img
                      src={component.primary_image_url}
                      alt={component.component_title}
                      className="w-12 h-12 object-cover rounded flex-shrink-0"
                    />
                  ) : (
                    <div className="w-12 h-12 bg-muted rounded flex items-center justify-center flex-shrink-0">
                      <Package className="h-5 w-5 text-muted-foreground" />
                    </div>
                  )}

                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-2">
                      <span className="text-sm font-medium truncate">
                        {component.component_title}
                      </span>
                      {isCurrentProduct && (
                        <Badge variant="outline" className="text-[9px] h-4 px-1 bg-purple-500/10 border-purple-500/30">
                          Aktualny
                        </Badge>
                      )}
                    </div>
                    <div className="flex items-center gap-2 mt-0.5">
                      <Badge variant="secondary" className="text-[10px] h-4 px-1">
                        {component.component_sku}
                      </Badge>
                      {component.component_color && (
                        <Badge variant="outline" className="text-[10px] h-4 px-1">
                          {component.component_color}
                        </Badge>
                      )}
                      <span className="text-[10px] text-muted-foreground flex items-center gap-0.5">
                        <Layers className="h-3 w-3" />
                        {component.quantity}x
                      </span>
                    </div>
                  </div>

                  <div className="flex items-center gap-2 flex-shrink-0">
                    {isInPlan ? (
                      component.is_in_current_plan ? (
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => handleHighlightInPlan(component.component_id)}
                          className="h-7 text-xs gap-1 bg-blue-600 hover:bg-blue-700 text-white border-blue-600"
                          data-testid={`highlight-component-${component.component_id}`}
                        >
                          <Check className="h-3 w-3" />
                          w tym planie
                        </Button>
                      ) : (
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => {
                            if (component.in_plan_id) {
                              sessionStorage.setItem('highlightProductId', String(component.component_id));
                              sessionStorage.setItem('highlightOrderNumber', orderNumber);
                              navigateToPlan(component.in_plan_id);
                            }
                          }}
                          className="h-7 text-xs gap-1 bg-blue-600 hover:bg-blue-700 text-white border-blue-600"
                          data-testid={`navigate-plan-${component.component_id}`}
                        >
                          <Check className="h-3 w-3" />
                          {component.in_plan_number}
                          <ExternalLink className="h-3 w-3" />
                        </Button>
                      )
                    ) : wasAddedLocally ? (
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => handleHighlightInPlan(component.component_id)}
                        className="h-7 text-xs gap-1 bg-blue-600 hover:bg-blue-700 text-white border-blue-600"
                        data-testid={`highlight-component-${component.component_id}`}
                      >
                        <Check className="h-3 w-3" />
                        w tym planie
                      </Button>
                    ) : (
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => handleAddComponent(component)}
                        disabled={isAdding || isAddingThis}
                        className="h-7 text-xs gap-1 bg-green-600 hover:bg-green-700 text-white border-green-600"
                        data-testid={`add-component-${component.component_id}`}
                      >
                        {isAddingThis ? (
                          <span className="animate-pulse">Dodawanie...</span>
                        ) : (
                          <>
                            <Plus className="h-3 w-3" />
                            Dodaj do planu
                          </>
                        )}
                      </Button>
                    )}
                  </div>
                </div>
              );
            })}
          </div>
        </div>

        {setComponents.some(c => !c.is_in_any_plan && !locallyAddedIds.has(c.component_id)) && (
          <div className="flex-shrink-0 pt-4 mt-4 border-t">
            <Button
              onClick={handleAddAllNotInPlan}
              disabled={isAdding}
              className="w-full"
              data-testid="add-all-components"
            >
              <Plus className="h-4 w-4 mr-2" />
              Dodaj wszystkie brakujące ({setComponents.filter(c => !c.is_in_any_plan && !locallyAddedIds.has(c.component_id)).length})
            </Button>
          </div>
        )}
      </DialogContent>
    </Dialog>
  );
}
