import { useState, useMemo, useCallback } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Route, Plus, Trash2, Factory, Palette, Package, Settings2, Wrench, Layers, ChevronDown, ChevronUp, Check, ChevronsUpDown, SkipForward, ArrowRight, Box, Lock, Unlock, Loader2, AlertTriangle, X, GripVertical, Scissors, Hammer, Paintbrush, Ruler, Drill, Cog, CircleDot, Square, Star, Zap, Timer, Settings, Truck, LucideIcon, Sparkles, FileStack } from "lucide-react";
import {
  DndContext,
  closestCenter,
  KeyboardSensor,
  PointerSensor,
  useSensor,
  useSensors,
  DragEndEvent,
} from '@dnd-kit/core';
import {
  arrayMove,
  SortableContext,
  sortableKeyboardCoordinates,
  useSortable,
  verticalListSortingStrategy,
} from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Separator } from "@/components/ui/separator";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from "@/components/ui/command";
import { Switch } from "@/components/ui/switch";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";
import { Accordion, AccordionItem, AccordionTrigger, AccordionContent } from "@/components/ui/accordion";
import { Combobox } from "@/components/ui/combobox";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { cn } from "@/lib/utils";
import { ZlpGenerationProgressDialog } from "@/components/zlp-generation-progress-dialog";

interface RoutingAssignment {
  id: number;
  planId: number;
  materialType: string | null;
  colorCode: string | null;
  componentPattern: string | null;
  routingId: number;
  routingCode?: string;
  routingName?: string;
  locationId: number | null;
  locationCode?: string;
  locationName?: string;
  overrideWorkCenterId: number | null;
  workCenterCode?: string;
  workCenterName?: string;
  templateId: number | null;
  templateName?: string;
  priority: number;
  notes: string | null;
}

interface Routing {
  id: number;
  code: string;
  name: string;
}

interface Location {
  id: number;
  code: string;
  name: string;
}

interface WorkCenter {
  id: number;
  code: string;
  name: string;
}

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
}

interface ComponentRoutingTemplate {
  id: number;
  name: string;
  materialType: string | null;
  colorCode: string | null;
  componentPattern: string | null;
  defaultRoutingId: number;
  defaultWorkCenterId: number | null;
  locationId: number | null;
  priority: number;
  routingCode?: string;
  routingName?: string;
  workCenterCode?: string;
  workCenterName?: string;
  locationCode?: string;
  locationName?: string;
}

interface RoutingOperation {
  id: number;
  sequence: number;
  code: string;
  name: string;
  description: string | null;
  defaultWorkCenterId: number | null;
  defaultWorkCenterCode: string | null;
  defaultWorkCenterName: string | null;
  defaultBufferBeforeId: number | null;
  defaultBufferBeforeName: string | null;
  defaultBufferAfterId: number | null;
  defaultBufferAfterName: string | null;
  overrideId: number | null;
  overrideWorkCenterId: number | null;
  overrideWorkCenterCode: string | null;
  overrideWorkCenterName: string | null;
  overrideBufferBeforeId: number | null;
  overrideBufferBeforeName: string | null;
  overrideBufferAfterId: number | null;
  overrideBufferAfterName: string | null;
  skipOperation: boolean;
  overrideNotes: string | null;
  effectiveWorkCenterId: number | null;
  effectiveWorkCenterName: string | null;
  hasOverride: boolean;
}

interface OperationsResponse {
  assignment: {
    id: number;
    planId: number;
    routingId: number;
    routingCode: string;
    routingName: string;
    materialType: string | null;
    colorCode: string | null;
    componentPattern: string | null;
  };
  operations: RoutingOperation[];
}

interface ZlpWorkOrder {
  id: number;
  workOrderNumber: string;
  sequence: number;
  status: string;
  operationName: string | null;
  operationCode: string | null;
  operationIcon: string | null;
  operationColor: string | null;
  workCenterName: string | null;
  bufferBeforeName: string | null;
  bufferAfterName: string | null;
  quantityPlanned: number;
  quantityProduced: number;
  quantityScrap: number;
}

interface ZlpSummary {
  id: number;
  orderNumber: string;
  status: string;
  colorCode: string | null;
  quantityPlanned: number;
  quantityProduced: number;
  quantityScrap: number;
  progress: number;
  workOrders: ZlpWorkOrder[];
  activeOperationIndex: number | null;
}

interface ZlpDashboardData {
  planId: number;
  planNumber: string;
  planName: string;
  totalZlps: number;
  completedZlps: number;
  inProgressZlps: number;
  pendingZlps: number;
  overallProgress: number;
  zlps: ZlpSummary[];
}

interface PlanRoutingAssignmentsPanelProps {
  planId: number;
  zlpLocked?: boolean;
  hasZlps?: boolean;
}

const colorHexMap: Record<string, string> = {
  'WOTAN': '#8B4513',
  'BIALY': '#FFFFFF',
  'CZARNY': '#000000',
  'SONOMA': '#DEB887',
  'LANCELOT': '#6B4423',
  'SZARY': '#808080',
  'SUROWA': '#D2B48C',
  'HDF-BIALY': '#F5F5F5',
  'BRAK-KOLORU': '#CCCCCC',
};

function getColorHex(colorCode: string | null): string {
  if (!colorCode) return '#808080';
  return colorHexMap[colorCode.toUpperCase()] || '#808080';
}

const OPERATION_ICON_MAP: Record<string, LucideIcon> = {
  Scissors, Hammer, Paintbrush, Ruler, Drill, Cog, 
  Wrench, Settings, Package, Layers, Factory, Palette, 
  Truck, Timer, Star, Zap, CircleDot, Square, Box
};

function getIconComponent(iconName: string | null | undefined): LucideIcon {
  if (!iconName) return Wrench;
  return OPERATION_ICON_MAP[iconName] || Wrench;
}

interface ComboboxOption {
  value: string;
  label: string;
}

function MiniCombobox({
  value,
  onChange,
  options,
  placeholder = "Domyślna",
  className,
  disabled = false,
}: {
  value: string;
  onChange: (value: string) => void;
  options: ComboboxOption[];
  placeholder?: string;
  className?: string;
  disabled?: boolean;
}) {
  const [open, setOpen] = useState(false);
  const selectedOption = options.find(o => o.value === value);

  return (
    <Popover open={disabled ? false : open} onOpenChange={disabled ? undefined : setOpen}>
      <PopoverTrigger asChild>
        <Button
          variant="outline"
          size="sm"
          role="combobox"
          aria-expanded={open}
          disabled={disabled}
          className={cn("justify-between h-7 text-xs", className)}
        >
          <span className="truncate max-w-[120px]">
            {selectedOption ? selectedOption.label : placeholder}
          </span>
          <ChevronsUpDown className="ml-1 h-3 w-3 shrink-0 opacity-50" />
        </Button>
      </PopoverTrigger>
      <PopoverContent className="w-[200px] p-0 z-[10001]" align="start">
        <Command
          filter={(value, search) => {
            if (value.toLowerCase().includes(search.toLowerCase())) return 1;
            return 0;
          }}
        >
          <CommandInput placeholder="Szukaj..." className="h-8 text-xs" />
          <CommandList>
            <CommandEmpty>Brak wyników</CommandEmpty>
            <CommandGroup>
              {options.map((option) => (
                <CommandItem
                  key={option.value}
                  value={option.label}
                  onSelect={() => {
                    onChange(option.value);
                    setOpen(false);
                  }}
                  className="text-xs"
                >
                  <Check
                    className={cn(
                      "mr-2 h-3 w-3",
                      value === option.value ? "opacity-100" : "opacity-0"
                    )}
                  />
                  {option.label}
                </CommandItem>
              ))}
            </CommandGroup>
          </CommandList>
        </Command>
      </PopoverContent>
    </Popover>
  );
}

interface SortableAssignmentItemProps {
  assignment: RoutingAssignment;
  isExpanded: boolean;
  onToggleExpand: () => void;
  onDelete: () => void;
  onPriorityUp: () => void;
  onPriorityDown: () => void;
  isPending: boolean;
  planId: number;
  zlpLocked?: boolean;
}

function SortableAssignmentItem({ 
  assignment, 
  isExpanded, 
  onToggleExpand, 
  onDelete,
  onPriorityUp,
  onPriorityDown,
  isPending,
  planId,
  zlpLocked = false
}: SortableAssignmentItemProps) {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ id: assignment.id });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.5 : 1,
    zIndex: isDragging ? 1000 : 1,
  };

  return (
    <div ref={setNodeRef} style={style}>
      <Collapsible
        open={isExpanded}
        onOpenChange={onToggleExpand}
      >
        <div
          className={cn(
            "rounded-md border bg-card overflow-visible",
            isExpanded && "border-primary/30",
            isDragging && "shadow-lg ring-2 ring-primary"
          )}
          data-testid={`assignment-row-${assignment.id}`}
        >
          <div className="flex items-center gap-2 p-3">
            <div
              {...(zlpLocked ? {} : attributes)}
              {...(zlpLocked ? {} : listeners)}
              className={cn(
                "text-muted-foreground",
                zlpLocked 
                  ? "cursor-not-allowed opacity-30" 
                  : "cursor-grab active:cursor-grabbing hover:text-foreground"
              )}
              data-testid={`drag-handle-${assignment.id}`}
            >
              <GripVertical className="h-4 w-4" />
            </div>
            <CollapsibleTrigger asChild>
              <Button
                size="icon"
                variant="ghost"
                className="h-6 w-6"
                data-testid={`button-expand-assignment-${assignment.id}`}
              >
                {isExpanded ? (
                  <ChevronUp className="h-4 w-4" />
                ) : (
                  <ChevronDown className="h-4 w-4" />
                )}
              </Button>
            </CollapsibleTrigger>
            <div className="flex-1 flex flex-wrap items-center gap-2">
              {assignment.colorCode && (
                <Badge variant="outline" className="gap-1">
                  <Palette className="h-3 w-3" />
                  {assignment.colorCode}
                </Badge>
              )}
              {assignment.componentPattern && (
                assignment.componentPattern.split(/[|\n]/).filter(p => p.trim()).map((pattern, idx) => (
                  <Badge key={idx} variant="secondary" className="font-mono text-xs">
                    {pattern.trim()}
                  </Badge>
                ))
              )}
              <span className="text-muted-foreground">→</span>
              <Badge variant="default" className="gap-1">
                <Route className="h-3 w-3" />
                {assignment.routingCode || assignment.routingName}
              </Badge>
              {assignment.locationName && (
                <Badge variant="outline" className="gap-1">
                  <Factory className="h-3 w-3" />
                  {assignment.locationName}
                </Badge>
              )}
              {assignment.workCenterName && (
                <Badge variant="outline" className="gap-1 bg-amber-500/10 border-amber-500/30">
                  <Wrench className="h-3 w-3" />
                  {assignment.workCenterName}
                </Badge>
              )}
              {assignment.templateName && (
                <Badge variant="secondary" className="gap-1 text-xs">
                  <Layers className="h-3 w-3" />
                  {assignment.templateName}
                </Badge>
              )}
            </div>
            <div className="flex items-center gap-1">
              <Tooltip>
                <TooltipTrigger asChild>
                  <div className="flex items-center">
                    <Button
                      size="icon"
                      variant="ghost"
                      className="h-6 w-6"
                      onClick={onPriorityDown}
                      disabled={isPending || zlpLocked}
                      data-testid={`button-priority-down-${assignment.id}`}
                    >
                      <ChevronDown className="h-3 w-3" />
                    </Button>
                    <Badge 
                      variant={assignment.priority >= 100 ? "default" : assignment.priority >= 50 ? "secondary" : "outline"}
                      className={cn(
                        "min-w-[50px] justify-center font-mono text-xs rounded-none",
                        assignment.priority >= 100 && "bg-green-600 hover:bg-green-700",
                        assignment.priority >= 50 && assignment.priority < 100 && "bg-amber-500/80 text-amber-950"
                      )}
                    >
                      P:{assignment.priority}
                    </Badge>
                    <Button
                      size="icon"
                      variant="ghost"
                      className="h-6 w-6"
                      onClick={onPriorityUp}
                      disabled={isPending || zlpLocked}
                      data-testid={`button-priority-up-${assignment.id}`}
                    >
                      <ChevronUp className="h-3 w-3" />
                    </Button>
                  </div>
                </TooltipTrigger>
                <TooltipContent>
                  <div className="text-xs">
                    <div className="font-medium">Priorytet: {assignment.priority}</div>
                    <div className="text-muted-foreground">Wyższy = sprawdzany wcześniej</div>
                    {zlpLocked ? (
                      <div className="text-amber-500">ZLP zablokowane - edycja niemożliwa</div>
                    ) : (
                      <div className="text-muted-foreground">Przeciągnij lub kliknij strzałki</div>
                    )}
                  </div>
                </TooltipContent>
              </Tooltip>
              <Button
                size="icon"
                variant="ghost"
                onClick={onDelete}
                disabled={isPending || zlpLocked}
                className="h-8 w-8 text-destructive"
                data-testid={`button-delete-assignment-${assignment.id}`}
              >
                <Trash2 className="h-4 w-4" />
              </Button>
            </div>
          </div>
          <CollapsibleContent>
            <SortableAssignmentOperations
              assignmentId={assignment.id}
              planId={planId}
              zlpLocked={zlpLocked}
            />
          </CollapsibleContent>
        </div>
      </Collapsible>
    </div>
  );
}

interface SortableAssignmentOperationsProps {
  assignmentId: number;
  planId: number;
  zlpLocked?: boolean;
}

function SortableAssignmentOperations({ assignmentId, planId, zlpLocked }: SortableAssignmentOperationsProps) {
  return <AssignmentOperationsInner assignmentId={assignmentId} planId={planId} zlpLocked={zlpLocked} />;
}

interface AssignmentOperationsProps {
  assignmentId: number;
  planId: number;
  workCenters: WorkCenter[];
  locations: Location[];
  zlpLocked?: boolean;
}

interface AssignmentOperationsInnerProps {
  assignmentId: number;
  planId: number;
  zlpLocked?: boolean;
}

function AssignmentOperationsInner({ assignmentId, planId, zlpLocked }: AssignmentOperationsInnerProps) {
  const { data: workCentersData } = useQuery<WorkCenter[]>({
    queryKey: ['/api/production/work-centers'],
  });
  const { data: locationsData } = useQuery<Location[]>({
    queryKey: ['/api/production/locations'],
  });
  
  const workCenters = workCentersData || [];
  const locations = locationsData || [];
  
  return (
    <AssignmentOperations 
      assignmentId={assignmentId} 
      planId={planId} 
      workCenters={workCenters}
      locations={locations}
      zlpLocked={zlpLocked}
    />
  );
}

function AssignmentOperations({ assignmentId, planId, workCenters, locations, zlpLocked = false }: AssignmentOperationsProps) {
  const { toast } = useToast();

  const { data: operationsData, isLoading } = useQuery<OperationsResponse>({
    queryKey: [`/api/production/plan-routing-assignments/${assignmentId}/operations`],
  });

  const updateMutation = useMutation({
    mutationFn: async ({ operationId, data }: { operationId: number; data: any }) => {
      return apiRequest("PUT", `/api/production/plan-routing-assignments/${assignmentId}/operations/${operationId}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/plan-routing-assignments/${assignmentId}/operations`] });
      queryClient.invalidateQueries({ queryKey: [`/api/production/plans/${planId}/routing-assignments`] });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const workCenterOptions: ComboboxOption[] = [
    { value: "", label: "Domyślna" },
    ...workCenters.map(wc => ({ value: String(wc.id), label: `${wc.code}` }))
  ];

  const bufferOptions: ComboboxOption[] = [
    { value: "", label: "Brak" },
    ...locations.map(loc => ({ value: String(loc.id), label: loc.name }))
  ];

  if (isLoading) {
    return (
      <div className="flex items-center justify-center gap-2 py-6 px-4 text-sm">
        <Loader2 className="h-4 w-4 animate-spin text-primary" />
        <span>Ładowanie operacji...</span>
      </div>
    );
  }

  if (!operationsData?.operations?.length) {
    return (
      <div className="py-3 px-4 text-xs text-muted-foreground">
        Brak operacji w tej marszrucie
      </div>
    );
  }

  const handleOverrideChange = (operation: RoutingOperation, field: string, value: any) => {
    const currentOverride = {
      overrideWorkCenterId: operation.overrideWorkCenterId,
      overrideBufferBeforeId: operation.overrideBufferBeforeId,
      overrideBufferAfterId: operation.overrideBufferAfterId,
      skipOperation: operation.skipOperation,
    };

    let newValue = value;
    if (field !== 'skipOperation') {
      newValue = value === "" ? null : parseInt(value);
    }

    updateMutation.mutate({
      operationId: operation.id,
      data: {
        ...currentOverride,
        [field]: newValue,
      }
    });
  };

  const overrideCount = operationsData.operations.filter(op => op.hasOverride || op.skipOperation).length;

  return (
    <div className="border-t bg-muted/20">
      <div className="px-3 py-2 text-xs font-medium text-muted-foreground flex items-center gap-2">
        <Settings2 className="h-3 w-3" />
        Nadpisania operacji ({overrideCount}/{operationsData.operations.length})
      </div>
      <div className="px-3 pb-3 space-y-1.5">
        {operationsData.operations.map((operation, idx) => (
          <div 
            key={operation.id} 
            className={cn(
              "rounded border p-2 text-xs",
              operation.skipOperation && "opacity-50 bg-muted/50",
              operation.hasOverride && !operation.skipOperation && "border-amber-500/30 bg-amber-500/5"
            )}
          >
            <div className="flex items-center gap-2 mb-2">
              <Badge variant="outline" className="text-[10px] h-5 px-1.5">
                {operation.sequence}
              </Badge>
              <span className="font-medium">{operation.code}</span>
              <span className="text-muted-foreground">{operation.name}</span>
              <div className="flex-1" />
              <div className="flex items-center gap-1">
                <SkipForward className="h-3 w-3 text-muted-foreground" />
                <Switch
                  checked={operation.skipOperation}
                  onCheckedChange={(checked) => handleOverrideChange(operation, 'skipOperation', checked)}
                  className="h-4 w-7"
                  disabled={zlpLocked}
                  data-testid={`switch-skip-op-${operation.id}`}
                />
              </div>
            </div>
            
            {!operation.skipOperation && (
              <div className="grid grid-cols-3 gap-2">
                <div>
                  <Label className="text-[10px] text-muted-foreground mb-1 flex items-center gap-1">
                    <Wrench className="h-2.5 w-2.5" />
                    Stanowisko
                  </Label>
                  <MiniCombobox
                    value={operation.overrideWorkCenterId ? String(operation.overrideWorkCenterId) : ""}
                    onChange={(v) => handleOverrideChange(operation, 'overrideWorkCenterId', v)}
                    options={workCenterOptions}
                    placeholder={operation.defaultWorkCenterCode || "Domyślna"}
                    className="w-full"
                    disabled={zlpLocked}
                  />
                </div>
                <div>
                  <Label className="text-[10px] text-muted-foreground mb-1 flex items-center gap-1">
                    <ArrowRight className="h-2.5 w-2.5" />
                    Bufor przed
                  </Label>
                  <MiniCombobox
                    value={operation.overrideBufferBeforeId ? String(operation.overrideBufferBeforeId) : ""}
                    onChange={(v) => handleOverrideChange(operation, 'overrideBufferBeforeId', v)}
                    options={bufferOptions}
                    placeholder={operation.defaultBufferBeforeName || "Brak"}
                    className="w-full"
                    disabled={zlpLocked}
                  />
                </div>
                <div>
                  <Label className="text-[10px] text-muted-foreground mb-1 flex items-center gap-1">
                    <ArrowRight className="h-2.5 w-2.5 rotate-180" />
                    Bufor po
                  </Label>
                  <MiniCombobox
                    value={operation.overrideBufferAfterId ? String(operation.overrideBufferAfterId) : ""}
                    onChange={(v) => handleOverrideChange(operation, 'overrideBufferAfterId', v)}
                    options={bufferOptions}
                    placeholder={operation.defaultBufferAfterName || "Brak"}
                    className="w-full"
                    disabled={zlpLocked}
                  />
                </div>
              </div>
            )}
          </div>
        ))}
      </div>
    </div>
  );
}

interface PlanData {
  id: number;
  autoAssignRoutings: boolean;
}

export function PlanRoutingAssignmentsPanel({ planId, zlpLocked = false, hasZlps = false }: PlanRoutingAssignmentsPanelProps) {
  const { toast } = useToast();

  const [expandedAssignments, setExpandedAssignments] = useState<Set<number>>(new Set());
  const [accordionValue, setAccordionValue] = useState<string[]>(["variants", "templates", "manual"]);
  const [newAssignmentOpen, setNewAssignmentOpen] = useState(true);
  const [newAssignment, setNewAssignment] = useState({
    materialType: "",
    colorCode: "",
    componentPattern: "",
    routingId: 0,
    locationId: 0,
    overrideWorkCenterId: 0,
    templateId: 0,
    priority: 10,
    notes: "",
  });

  const toggleAssignmentExpanded = (id: number) => {
    setExpandedAssignments(prev => {
      const next = new Set(prev);
      if (next.has(id)) {
        next.delete(id);
      } else {
        next.add(id);
      }
      return next;
    });
  };

  const { data: planData } = useQuery<PlanData>({
    queryKey: [`/api/production/planning/plans/${planId}`],
  });

  const updateAutoAssignMutation = useMutation({
    mutationFn: async (autoAssignRoutings: boolean) => {
      return apiRequest("PATCH", `/api/production/planning/plans/${planId}`, { autoAssignRoutings });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}`] });
      toast({ title: "Ustawienie zaktualizowane" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const { data: assignments = [], isLoading: showAssignmentsLoading } = useQuery<RoutingAssignment[]>({
    queryKey: [`/api/production/plans/${planId}/routing-assignments`],
  });

  const { data: routings = [] } = useQuery<Routing[]>({
    queryKey: ['/api/production/routings'],
  });

  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ['/api/production/locations'],
  });

  const { data: workCenters = [] } = useQuery<WorkCenter[]>({
    queryKey: ['/api/production/work-centers'],
  });

  const { data: colors = [] } = useQuery<DictionaryItem[]>({
    queryKey: ['/api/dictionaries?type=color'],
  });

  const { data: materialTypes = [] } = useQuery<DictionaryItem[]>({
    queryKey: ['/api/dictionaries?type=material_type'],
  });

  const { data: componentCz1 = [] } = useQuery<DictionaryItem[]>({
    queryKey: ['/api/dictionaries?type=component_cz1'],
  });

  const { data: templates = [] } = useQuery<ComponentRoutingTemplate[]>({
    queryKey: ['/api/production/component-routing-templates'],
  });

  const { data: zlpDashboard } = useQuery<ZlpDashboardData>({
    queryKey: [`/api/production/planning/plans/${planId}/zlp-dashboard`],
    enabled: hasZlps,
  });

  const getPathSignature = (workOrders: ZlpWorkOrder[]): string => {
    return workOrders.map(wo => 
      `${wo.operationCode || wo.operationName || ''}@${wo.workCenterName || 'default'}` +
      `${wo.bufferBeforeName ? `<${wo.bufferBeforeName}` : ''}` +
      `${wo.bufferAfterName ? `>${wo.bufferAfterName}` : ''}`
    ).join('|');
  };

  const zlpsByPath = useMemo(() => {
    if (!zlpDashboard?.zlps?.length) return [];
    
    const pathGroups = new Map<string, {
      signature: string;
      zlps: ZlpSummary[];
      representativeFlow: ZlpWorkOrder[];
    }>();
    
    for (const zlp of zlpDashboard.zlps) {
      if (!zlp.workOrders?.length) continue;
      
      const signature = getPathSignature(zlp.workOrders);
      
      if (!pathGroups.has(signature)) {
        pathGroups.set(signature, {
          signature,
          zlps: [],
          representativeFlow: zlp.workOrders,
        });
      }
      pathGroups.get(signature)!.zlps.push(zlp);
    }
    
    return Array.from(pathGroups.values());
  }, [zlpDashboard]);

  const createMutation = useMutation({
    mutationFn: async (data: typeof newAssignment) => {
      return apiRequest("POST", `/api/production/plans/${planId}/routing-assignments`, {
        materialType: data.materialType || null,
        colorCode: data.colorCode || null,
        componentPattern: data.componentPattern || null,
        routingId: data.routingId,
        locationId: data.locationId || null,
        overrideWorkCenterId: data.overrideWorkCenterId || null,
        templateId: data.templateId || null,
        priority: data.priority,
        notes: data.notes || null,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/plans/${planId}/routing-assignments`] });
      toast({ title: "Przypisanie dodane" });
      setNewAssignment({
        materialType: "",
        colorCode: "",
        componentPattern: "",
        routingId: 0,
        locationId: 0,
        overrideWorkCenterId: 0,
        templateId: 0,
        priority: 10,
        notes: "",
      });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/production/plan-routing-assignments/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/plans/${planId}/routing-assignments`] });
      toast({ title: "Przypisanie usunięte" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const updatePriorityMutation = useMutation({
    mutationFn: async ({ id, priority }: { id: number; priority: number }) => {
      return apiRequest("PATCH", `/api/production/plan-routing-assignments/${id}`, { priority });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/plans/${planId}/routing-assignments`] });
      toast({ title: "Priorytet zaktualizowany" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const bulkUpdatePrioritiesMutation = useMutation({
    mutationFn: async (updates: { id: number; priority: number }[]) => {
      await Promise.all(
        updates.map(({ id, priority }) => 
          apiRequest("PATCH", `/api/production/plan-routing-assignments/${id}`, { priority })
        )
      );
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/plans/${planId}/routing-assignments`] });
      toast({ title: "Kolejność zaktualizowana" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const sensors = useSensors(
    useSensor(PointerSensor, {
      activationConstraint: {
        distance: 8,
      },
    }),
    useSensor(KeyboardSensor, {
      coordinateGetter: sortableKeyboardCoordinates,
    })
  );

  const handleDragEnd = useCallback((event: DragEndEvent, group: RoutingAssignment[]) => {
    const { active, over } = event;
    
    if (!over || active.id === over.id) {
      return;
    }

    const sortedGroup = [...group].sort((a, b) => b.priority - a.priority);
    const oldIndex = sortedGroup.findIndex(item => item.id === active.id);
    const newIndex = sortedGroup.findIndex(item => item.id === over.id);

    if (oldIndex === -1 || newIndex === -1) return;

    const reordered = arrayMove(sortedGroup, oldIndex, newIndex);
    
    const basePriority = 200;
    const updates = reordered.map((item, index) => ({
      id: item.id,
      priority: basePriority - index * 10,
    }));

    bulkUpdatePrioritiesMutation.mutate(updates);
  }, [bulkUpdatePrioritiesMutation]);

  const handleApplyTemplate = (template: ComponentRoutingTemplate) => {
    setNewAssignment({
      materialType: template.materialType || "",
      colorCode: template.colorCode || "",
      componentPattern: template.componentPattern || "",
      routingId: template.defaultRoutingId,
      locationId: template.locationId || 0,
      overrideWorkCenterId: template.defaultWorkCenterId || 0,
      templateId: template.id,
      priority: template.priority,
      notes: `Z szablonu: ${template.name}`,
    });
    toast({ title: "Szablon załadowany", description: `Zastosowano szablon: ${template.name}` });
  };

  const componentPatternOptions = useMemo(() => [
    { value: "", label: "Dowolny" },
    ...componentCz1.map((c) => ({ 
      value: `${c.code}%`, 
      label: c.name === c.code || !c.name ? c.code : `${c.name} (${c.code})` 
    }))
  ], [componentCz1]);

  const canSubmit = newAssignment.routingId > 0;

  const groupedAssignments = useMemo(() => {
    const groups: Record<string, RoutingAssignment[]> = {};
    assignments.forEach(a => {
      const key = a.materialType || "Wszystkie materiały";
      if (!groups[key]) groups[key] = [];
      groups[key].push(a);
    });
    return groups;
  }, [assignments]);

  const routingOptions = useMemo(() => [
    { value: "0", label: "Wybierz marszrutę..." },
    ...routings.map(r => ({ value: String(r.id), label: `${r.code} - ${r.name}` }))
  ], [routings]);

  const locationOptions = useMemo(() => [
    { value: "0", label: "Domyślna" },
    ...locations.map(l => ({ value: String(l.id), label: l.name }))
  ], [locations]);

  const workCenterOptions = useMemo(() => [
    { value: "0", label: "Domyślna" },
    ...workCenters.map(wc => ({ value: String(wc.id), label: `${wc.code} - ${wc.name}` }))
  ], [workCenters]);

  const colorOptions = useMemo(() => [
    { value: "", label: "Wszystkie kolory" },
    ...colors.map(c => ({ value: c.code, label: c.name || c.code }))
  ], [colors]);

  const materialTypeOptions = useMemo(() => [
    { value: "", label: "Wszystkie materiały" },
    ...materialTypes.map(m => ({ value: m.code, label: m.name || m.code }))
  ], [materialTypes]);

  return (
    <div className="space-y-6">
      <div className="flex items-center gap-2 mb-4">
        <Route className="h-5 w-5 text-primary" />
        <h2 className="text-lg font-semibold">Przypisanie marszrut do planu</h2>
      </div>

      {zlpLocked && (
        <div className="flex items-center gap-2 p-3 bg-amber-500/10 border border-amber-500/30 rounded-md text-amber-700 dark:text-amber-400">
          <Lock className="h-4 w-4" />
          <span className="text-sm">Plan jest zablokowany - edycja przypisań jest niemożliwa</span>
        </div>
      )}

      <Accordion 
        type="multiple" 
        value={accordionValue} 
        onValueChange={setAccordionValue}
        className="space-y-3"
        data-testid="routing-assignment-accordion"
      >
        {/* 1. Warianty automatyczne (Auto-assign variants) */}
        <AccordionItem value="variants" className="border rounded-md bg-emerald-500/10 dark:bg-emerald-500/5">
          <AccordionTrigger className="px-4 py-3 hover:no-underline" data-testid="accordion-trigger-variants">
            <div className="flex items-center gap-2">
              <Sparkles className="h-4 w-4 text-emerald-600 dark:text-emerald-400" />
              <span className="text-sm font-medium">Warianty automatyczne</span>
            </div>
          </AccordionTrigger>
          <AccordionContent className="px-4 pb-4">
            <div className="flex items-center justify-between gap-4 p-3 bg-muted/30 rounded-md">
              <div className="flex-1">
                <div className="flex items-center gap-2">
                  <Sparkles className="h-4 w-4 text-emerald-500" />
                  <span className="text-sm font-medium">Automatyczne przypisanie marszrut</span>
                </div>
                <p className="text-xs text-muted-foreground mt-1">
                  System automatycznie przypisze marszruty na podstawie słownika formatek i wariantów
                </p>
              </div>
              <Switch
                checked={planData?.autoAssignRoutings ?? false}
                onCheckedChange={(checked) => updateAutoAssignMutation.mutate(checked)}
                disabled={zlpLocked || updateAutoAssignMutation.isPending}
                data-testid="switch-auto-assign-routings"
              />
            </div>
            <p className="text-xs text-muted-foreground mt-3 bg-muted/20 p-2 rounded">
              Gdy włączone, podczas generowania ZLP system dopasuje formatki do odpowiednich wariantów marszrut 
              na podstawie słownika formatek (np. DNO → CW, POLKA → CO).
            </p>
          </AccordionContent>
        </AccordionItem>

        {/* 2. Szablony (Templates) */}
        {templates.length > 0 && (
          <AccordionItem value="templates" className="border rounded-md bg-violet-500/10 dark:bg-violet-500/5">
            <AccordionTrigger className="px-4 py-3 hover:no-underline" data-testid="accordion-trigger-templates">
              <div className="flex items-center gap-2">
                <FileStack className="h-4 w-4 text-violet-600 dark:text-violet-400" />
                <span className="text-sm font-medium">Szablony</span>
                <Badge variant="secondary" className="text-xs ml-1">{templates.length}</Badge>
              </div>
            </AccordionTrigger>
            <AccordionContent className="px-4 pb-4">
              <p className="text-xs text-muted-foreground mb-3">
                Kliknij szablon aby załadować jego ustawienia do formularza poniżej
              </p>
              <div className="flex flex-wrap gap-2">
                {templates.map((template) => (
                  <Badge
                    key={template.id}
                    variant="outline"
                    className="cursor-pointer hover-elevate gap-1"
                    onClick={() => !zlpLocked && handleApplyTemplate(template)}
                    data-testid={`template-badge-${template.id}`}
                  >
                    <Layers className="h-3 w-3" />
                    {template.name}
                  </Badge>
                ))}
              </div>
            </AccordionContent>
          </AccordionItem>
        )}

        {/* 3. Ręczne ustawienia (Manual assignments) */}
        <AccordionItem value="manual" className="border rounded-md bg-blue-500/10 dark:bg-blue-500/5">
          <AccordionTrigger className="px-4 py-3 hover:no-underline" data-testid="accordion-trigger-manual">
            <div className="flex items-center gap-2">
              <Settings2 className="h-4 w-4 text-blue-600 dark:text-blue-400" />
              <span className="text-sm font-medium">Ręczne ustawienia</span>
              <Badge variant="secondary" className="text-xs ml-1">{assignments.length}</Badge>
            </div>
          </AccordionTrigger>
          <AccordionContent className="px-4 pb-4">
            {/* New assignment form */}
            <Collapsible open={newAssignmentOpen} onOpenChange={setNewAssignmentOpen} className="mb-4">
              <CollapsibleTrigger className="flex items-center justify-between w-full py-2 hover:bg-muted/50 transition-colors rounded-md px-2">
                <span className="text-sm font-medium flex items-center gap-2">
                  <Plus className="h-4 w-4" />
                  Nowe przypisanie
                </span>
                {newAssignmentOpen ? (
                  <ChevronUp className="h-4 w-4 text-muted-foreground" />
                ) : (
                  <ChevronDown className="h-4 w-4 text-muted-foreground" />
                )}
              </CollapsibleTrigger>
              <CollapsibleContent>
                <div className="pt-2 space-y-4">
                  <div className="grid grid-cols-2 lg:grid-cols-4 gap-4">
                    <div className="space-y-2">
                      <Label className="text-xs text-muted-foreground">Typ materiału</Label>
                      <Combobox
                        options={materialTypeOptions}
                        value={newAssignment.materialType}
                        onChange={(v) => setNewAssignment(prev => ({ ...prev, materialType: v }))}
                        placeholder="Wszystkie materiały"
                        disabled={zlpLocked}
                        data-testid="combobox-material-type"
                      />
                    </div>
                    <div className="space-y-2">
                      <Label className="text-xs text-muted-foreground">Kolor</Label>
                      <Combobox
                        options={colorOptions}
                        value={newAssignment.colorCode}
                        onChange={(v) => setNewAssignment(prev => ({ ...prev, colorCode: v }))}
                        placeholder="Wszystkie kolory"
                        disabled={zlpLocked}
                        data-testid="combobox-color"
                      />
                    </div>
                    <div className="space-y-2">
                      <Label className="text-xs text-muted-foreground">Wzorzec komponentu</Label>
                      <Combobox
                        options={componentPatternOptions}
                        value={newAssignment.componentPattern}
                        onChange={(v) => setNewAssignment(prev => ({ ...prev, componentPattern: v }))}
                        placeholder="Dowolny"
                        disabled={zlpLocked}
                        data-testid="combobox-component-pattern"
                      />
                    </div>
                    <div className="space-y-2">
                      <Label className="text-xs text-muted-foreground">Priorytet</Label>
                      <Input
                        type="number"
                        value={newAssignment.priority}
                        onChange={(e) => setNewAssignment(prev => ({ ...prev, priority: parseInt(e.target.value) || 10 }))}
                        min={1}
                        max={999}
                        disabled={zlpLocked}
                        data-testid="input-priority"
                      />
                    </div>
                  </div>

                  <Separator />

                  <div className="grid grid-cols-2 lg:grid-cols-3 gap-4">
                    <div className="space-y-2">
                      <Label className="text-xs text-muted-foreground flex items-center gap-1.5">
                        <Route className="h-3.5 w-3.5" />
                        Marszruta *
                      </Label>
                      <Combobox
                        options={routingOptions}
                        value={String(newAssignment.routingId)}
                        onChange={(v) => setNewAssignment(prev => ({ ...prev, routingId: parseInt(v) || 0 }))}
                        placeholder="Wybierz marszrutę..."
                        disabled={zlpLocked}
                        data-testid="combobox-routing"
                      />
                    </div>
                    <div className="space-y-2">
                      <Label className="text-xs text-muted-foreground flex items-center gap-1.5">
                        <Factory className="h-3.5 w-3.5" />
                        Lokalizacja
                      </Label>
                      <Combobox
                        options={locationOptions}
                        value={String(newAssignment.locationId)}
                        onChange={(v) => setNewAssignment(prev => ({ ...prev, locationId: parseInt(v) || 0 }))}
                        placeholder="Domyślna"
                        disabled={zlpLocked}
                        data-testid="combobox-location"
                      />
                    </div>
                    <div className="space-y-2">
                      <Label className="text-xs text-muted-foreground flex items-center gap-1.5">
                        <Wrench className="h-3.5 w-3.5" />
                        Override maszyny
                      </Label>
                      <Combobox
                        options={workCenterOptions}
                        value={String(newAssignment.overrideWorkCenterId)}
                        onChange={(v) => setNewAssignment(prev => ({ ...prev, overrideWorkCenterId: parseInt(v) || 0 }))}
                        placeholder="Domyślna"
                        disabled={zlpLocked}
                        data-testid="combobox-work-center"
                      />
                    </div>
                  </div>

                  <div className="space-y-2">
                    <Label className="text-xs text-muted-foreground">Notatki</Label>
                    <Textarea
                      value={newAssignment.notes}
                      onChange={(e) => setNewAssignment(prev => ({ ...prev, notes: e.target.value }))}
                      placeholder="Opcjonalne notatki..."
                      className="resize-none h-16"
                      disabled={zlpLocked}
                      data-testid="textarea-notes"
                    />
                  </div>

                  <Button
                    onClick={() => createMutation.mutate(newAssignment)}
                    disabled={!canSubmit || createMutation.isPending || zlpLocked}
                    className="w-full gap-2"
                    data-testid="button-create-assignment"
                  >
                    {createMutation.isPending ? (
                      <Loader2 className="h-4 w-4 animate-spin" />
                    ) : (
                      <Plus className="h-4 w-4" />
                    )}
                    Dodaj przypisanie
                  </Button>
                </div>
              </CollapsibleContent>
            </Collapsible>

            <Separator className="my-4" />

            {/* Current assignments list */}
            {showAssignmentsLoading ? (
              <div className="text-center py-8 text-muted-foreground">
                <Loader2 className="h-6 w-6 animate-spin mx-auto mb-2" />
                Ładowanie...
              </div>
            ) : assignments.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground border rounded-md bg-muted/20">
                <Route className="h-8 w-8 mx-auto mb-2 opacity-50" />
                <p>Brak przypisań marszrut dla tego planu.</p>
                <p className="text-xs mt-1">System użyje szablonów komponentów lub domyślnych reguł wariantów.</p>
              </div>
            ) : (
              <div className="space-y-4">
                {Object.entries(groupedAssignments).map(([materialType, group]) => {
                  const sortedGroup = [...group].sort((a, b) => b.priority - a.priority);
                  return (
                    <div key={materialType} className="space-y-2">
                      <div className="text-xs font-medium text-muted-foreground uppercase tracking-wide flex items-center gap-1.5">
                        <Package className="h-3 w-3" />
                        {materialType}
                        <span className="text-xs text-muted-foreground/70 ml-1">(przeciągnij aby zmienić kolejność)</span>
                      </div>
                      <DndContext
                        sensors={sensors}
                        collisionDetection={closestCenter}
                        onDragEnd={(event) => handleDragEnd(event, group)}
                      >
                        <SortableContext
                          items={sortedGroup.map(a => a.id)}
                          strategy={verticalListSortingStrategy}
                        >
                          <div className="space-y-2">
                            {sortedGroup.map((assignment) => (
                              <SortableAssignmentItem
                                key={assignment.id}
                                assignment={assignment}
                                isExpanded={expandedAssignments.has(assignment.id)}
                                onToggleExpand={() => toggleAssignmentExpanded(assignment.id)}
                                onDelete={() => deleteMutation.mutate(assignment.id)}
                                onPriorityUp={() => {
                                  const newPriority = Math.min(999, assignment.priority + 10);
                                  updatePriorityMutation.mutate({ id: assignment.id, priority: newPriority });
                                }}
                                onPriorityDown={() => {
                                  const newPriority = Math.max(1, assignment.priority - 10);
                                  updatePriorityMutation.mutate({ id: assignment.id, priority: newPriority });
                                }}
                                isPending={updatePriorityMutation.isPending || deleteMutation.isPending}
                                planId={planId}
                                zlpLocked={zlpLocked}
                              />
                            ))}
                          </div>
                        </SortableContext>
                      </DndContext>
                    </div>
                  );
                })}
              </div>
            )}
          </AccordionContent>
        </AccordionItem>

        {/* 4. Wygenerowane ZLP (Generated work orders) - only show if hasZlps */}
        {hasZlps && zlpsByPath.length > 0 && (
          <AccordionItem value="zlp" className="border rounded-md bg-primary/10 dark:bg-primary/5">
            <AccordionTrigger className="px-4 py-3 hover:no-underline" data-testid="accordion-trigger-zlp">
              <div className="flex items-center gap-2">
                <Factory className="h-4 w-4 text-primary" />
                <span className="text-sm font-medium">Wygenerowane zlecenia ZLP</span>
                <Badge variant="secondary" className="text-xs ml-1">{zlpDashboard?.totalZlps || 0}</Badge>
                {zlpsByPath.length > 1 && (
                  <Badge variant="outline" className="text-xs ml-1">{zlpsByPath.length} ścieżki</Badge>
                )}
              </div>
            </AccordionTrigger>
            <AccordionContent className="px-4 pb-4">
              <div className="px-4 pb-4 space-y-4">
                {zlpsByPath.map((pathGroup, pathIdx) => {
                  const pathColors = [
                    { bg: "bg-blue-500/10", border: "border-blue-500/30", accent: "text-blue-600 dark:text-blue-400" },
                    { bg: "bg-emerald-500/10", border: "border-emerald-500/30", accent: "text-emerald-600 dark:text-emerald-400" },
                    { bg: "bg-violet-500/10", border: "border-violet-500/30", accent: "text-violet-600 dark:text-violet-400" },
                    { bg: "bg-orange-500/10", border: "border-orange-500/30", accent: "text-orange-600 dark:text-orange-400" },
                    { bg: "bg-pink-500/10", border: "border-pink-500/30", accent: "text-pink-600 dark:text-pink-400" },
                    { bg: "bg-cyan-500/10", border: "border-cyan-500/30", accent: "text-cyan-600 dark:text-cyan-400" },
                  ];
                  const colorScheme = pathColors[pathIdx % pathColors.length];
                  
                  return (
                  <div key={pathIdx} className="border rounded-md space-y-3">
                    <div className="flex items-center gap-2 p-3 border-b bg-muted/30">
                      <Route className={`h-5 w-5 ${colorScheme.accent}`} />
                      <span className={`text-base font-bold ${colorScheme.accent}`}>
                        Ścieżka {pathIdx + 1}
                      </span>
                      <Badge variant="outline" className="text-[10px] rounded-none">
                        {pathGroup.zlps.length} ZLP
                      </Badge>
                    </div>
                    
                    <div className="flex flex-wrap items-center gap-2 px-3 pb-3">
                      {pathGroup.representativeFlow.map((wo, idx) => (
                        <div key={wo.id} className="flex items-center flex-shrink-0">
                          {wo.bufferBeforeName && (
                            <>
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <div className={cn(
                                    "flex flex-col items-center min-w-[70px] px-2 py-2 border rounded-sm cursor-default transition-all",
                                    wo.status === 'done' && "opacity-40 bg-muted/30 border-muted",
                                    wo.status === 'in_progress' && "bg-emerald-500/10 border-emerald-500/40",
                                    (wo.status === 'pending' || wo.status === 'draft' || !wo.status) && "opacity-70 bg-muted/20 border-border"
                                  )}>
                                    <Package className={cn(
                                      "h-3.5 w-3.5 mb-1",
                                      wo.status === 'done' && "text-muted-foreground",
                                      wo.status === 'in_progress' && "text-emerald-600",
                                      (wo.status === 'pending' || wo.status === 'draft' || !wo.status) && "text-muted-foreground"
                                    )} />
                                    <span className={cn(
                                      "text-[10px] font-medium text-center leading-tight",
                                      wo.status === 'done' && "text-muted-foreground",
                                      wo.status === 'in_progress' && "text-emerald-600 dark:text-emerald-400",
                                      (wo.status === 'pending' || wo.status === 'draft' || !wo.status) && "text-muted-foreground"
                                    )}>
                                      {wo.bufferBeforeName}
                                    </span>
                                  </div>
                                </TooltipTrigger>
                                <TooltipContent side="top" className="text-xs">
                                  <div className="font-medium">Bufor przed</div>
                                  <div className="text-muted-foreground">{wo.bufferBeforeName}</div>
                                </TooltipContent>
                              </Tooltip>
                              <ArrowRight className="h-3 w-3 text-muted-foreground mx-1 flex-shrink-0" />
                            </>
                          )}
                          <Tooltip>
                            <TooltipTrigger asChild>
                              <div 
                                className={cn(
                                  "flex flex-col items-center min-w-[100px] max-w-[140px] px-3 py-2.5 border rounded-sm cursor-default transition-all",
                                  wo.status === 'done' && "opacity-40 bg-muted/30 border-muted",
                                  wo.status === 'in_progress' && "bg-emerald-500/20 border-emerald-500 animate-pulse ring-2 ring-emerald-500/50",
                                  (wo.status === 'pending' || wo.status === 'draft' || !wo.status) && "opacity-70 bg-background border-border"
                                )}
                                style={wo.status === 'in_progress' ? {
                                  backgroundColor: '#10B98133',
                                  borderColor: '#10B981',
                                } : undefined}
                              >
                                <div className="flex items-center gap-1.5 mb-1">
                                  {(() => {
                                    const IconComp = getIconComponent(wo.operationIcon);
                                    const iconColor = wo.status === 'in_progress' ? '#10B981' : 
                                                     wo.status === 'done' ? undefined : 
                                                     wo.operationColor || undefined;
                                    return IconComp ? (
                                      <IconComp 
                                        className={cn("h-4 w-4", wo.status === 'done' && "text-muted-foreground")}
                                        style={{ color: iconColor }}
                                      />
                                    ) : (
                                      <Cog className="h-4 w-4 text-muted-foreground" />
                                    );
                                  })()}
                                  <span className="text-[10px] text-muted-foreground">
                                    {wo.sequence}.
                                  </span>
                                </div>
                                <span className={cn(
                                  "text-xs font-semibold text-center leading-tight",
                                  wo.status === 'in_progress' && "text-emerald-600 dark:text-emerald-400",
                                  wo.status === 'done' && "text-muted-foreground"
                                )}>
                                  {wo.operationName || wo.operationCode || `Op ${wo.sequence}`}
                                </span>
                                {wo.workCenterName && (
                                  <span className="text-[10px] text-muted-foreground text-center leading-tight mt-1">
                                    {wo.workCenterName}
                                  </span>
                                )}
                              </div>
                            </TooltipTrigger>
                            <TooltipContent side="top" className="text-xs">
                              <div className="font-medium">{wo.operationName || wo.operationCode}</div>
                              {wo.workCenterName && <div className="text-muted-foreground">Stanowisko: {wo.workCenterName}</div>}
                            </TooltipContent>
                          </Tooltip>
                          {wo.bufferAfterName ? (
                            <>
                              <ArrowRight className="h-3 w-3 text-muted-foreground mx-1 flex-shrink-0" />
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <div className={cn(
                                    "flex flex-col items-center min-w-[70px] px-2 py-2 border rounded-sm cursor-default transition-all",
                                    wo.status === 'done' && "opacity-40 bg-muted/30 border-muted",
                                    wo.status === 'in_progress' && "bg-emerald-500/10 border-emerald-500/40",
                                    (wo.status === 'pending' || wo.status === 'draft' || !wo.status) && "opacity-70 bg-muted/20 border-border"
                                  )}>
                                    <Package className={cn(
                                      "h-3.5 w-3.5 mb-1",
                                      wo.status === 'done' && "text-muted-foreground",
                                      wo.status === 'in_progress' && "text-emerald-600",
                                      (wo.status === 'pending' || wo.status === 'draft' || !wo.status) && "text-muted-foreground"
                                    )} />
                                    <span className={cn(
                                      "text-[10px] font-medium text-center leading-tight",
                                      wo.status === 'done' && "text-muted-foreground",
                                      wo.status === 'in_progress' && "text-emerald-600 dark:text-emerald-400",
                                      (wo.status === 'pending' || wo.status === 'draft' || !wo.status) && "text-muted-foreground"
                                    )}>
                                      {wo.bufferAfterName}
                                    </span>
                                  </div>
                                </TooltipTrigger>
                                <TooltipContent side="top" className="text-xs">
                                  <div className="font-medium">Bufor po</div>
                                  <div className="text-muted-foreground">{wo.bufferAfterName}</div>
                                </TooltipContent>
                              </Tooltip>
                              {idx < pathGroup.representativeFlow.length - 1 && (
                                <ArrowRight className="h-4 w-4 text-muted-foreground mx-1 flex-shrink-0" />
                              )}
                            </>
                          ) : (
                            idx < pathGroup.representativeFlow.length - 1 && (
                              <ArrowRight className="h-4 w-4 text-muted-foreground mx-1 flex-shrink-0" />
                            )
                          )}
                        </div>
                      ))}
                    </div>
                    
                    <div className="flex flex-wrap gap-2 px-3 pb-3">
                      {pathGroup.zlps.map((zlp) => (
                        <Tooltip key={zlp.id}>
                          <TooltipTrigger asChild>
                            <div 
                              className="flex items-center gap-2 px-2.5 py-1.5 border rounded-sm cursor-default"
                              style={{ 
                                backgroundColor: getColorHex(zlp.colorCode) + '20',
                                borderColor: getColorHex(zlp.colorCode) + '60',
                              }}
                            >
                              <span className="font-mono text-xs font-medium">
                                {zlp.orderNumber}
                              </span>
                              <span className="text-[10px] text-muted-foreground">
                                {zlp.quantityPlanned} szt.
                              </span>
                              <Badge 
                                variant="secondary" 
                                className={cn("text-[9px] h-4 px-1 rounded-none", 
                                  zlp.status === 'done' && "bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400",
                                  zlp.status === 'in_progress' && "bg-yellow-100 text-yellow-700 dark:bg-yellow-900/30 dark:text-yellow-400"
                                )}
                              >
                                {zlp.status === 'done' ? 'OK' : zlp.status === 'in_progress' ? 'W trakcie' : 'Oczekuje'}
                              </Badge>
                            </div>
                          </TooltipTrigger>
                          <TooltipContent side="top" className="text-xs">
                            <div className="font-medium">{zlp.orderNumber}</div>
                            <div className="text-muted-foreground">Ilość: {zlp.quantityPlanned} szt.</div>
                            <div className="text-muted-foreground">Kolor: {zlp.colorCode || 'Brak'}</div>
                            <div className="text-muted-foreground">Postęp: {zlp.progress}%</div>
                          </TooltipContent>
                        </Tooltip>
                      ))}
                    </div>
                  </div>
                );
                })}
              </div>
            </AccordionContent>
          </AccordionItem>
        )}
      </Accordion>

      <div className="text-xs text-muted-foreground bg-muted/30 p-4 rounded-md">
        <p className="font-medium mb-2">Hierarchia rozwiązywania marszrut:</p>
        <ol className="list-decimal list-inside space-y-1">
          <li><strong>Przypisania planu</strong> (ta lista) - najwyższy priorytet</li>
          <li><strong>Szablony komponentów</strong> - globalne domyślne marszruty</li>
          <li><strong>Reguły wariantów</strong> - fallback dla nieprzypisanych</li>
        </ol>
        <p className="mt-3 text-muted-foreground/80">
          Maszyna (override) pozwala zmienić domyślne gniazdo produkcyjne z marszruty dla tego planu.
        </p>
      </div>
    </div>
  );
}
