import { useState, useMemo } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Checkbox } from "@/components/ui/checkbox";
import { Label } from "@/components/ui/label";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from "@/components/ui/collapsible";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
} from "@/components/ui/tooltip";
import { useToast } from "@/hooks/use-toast";
import {
  Users,
  User,
  Factory,
  ChevronDown,
  ChevronRight,
  Star,
  Edit,
  Loader2,
  AlertCircle,
  Wand2,
} from "lucide-react";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

interface ProductionOperator {
  id: number;
  full_name: string;
  short_code: string | null;
  is_active: boolean;
}

interface WorkCenter {
  id: number;
  code: string;
  name: string;
}

interface WorkOrderOperatorInfo {
  workOrderId: number;
  workOrderNumber: string;
  operationName: string | null;
  operationCode: string | null;
  workCenterId: number | null;
  workCenterName: string | null;
  zlpNumber: string;
  colorCode: string | null;
  assignedOperators: Array<{
    operatorId: number;
    operatorName: string;
    isPrimary: boolean;
  }>;
}

interface OperatorWorkCenterSummary {
  workCenterId: number;
  workCenterCode: string;
  workCenterName: string;
  workOrders: WorkOrderOperatorInfo[];
  operatorSummary: Record<number, { name: string; count: number; primaryCount: number }>;
}

interface OperatorsAssignmentPanelProps {
  planId: number;
}

const colorBadgeColors: Record<string, string> = {
  BIALY: "bg-white border border-gray-300 text-gray-800",
  SONOMA: "bg-amber-200 text-amber-900",
  WOTAN: "bg-amber-700 text-white",
  SUROWA: "bg-yellow-100 text-yellow-800",
  HDF: "bg-stone-400 text-white",
  CZARNY: "bg-black text-white",
};

export function OperatorsAssignmentPanel({ planId }: OperatorsAssignmentPanelProps) {
  const { toast } = useToast();
  const [expandedWorkCenters, setExpandedWorkCenters] = useState<Set<number>>(new Set());
  const [editDialogOpen, setEditDialogOpen] = useState(false);
  const [selectedWorkOrder, setSelectedWorkOrder] = useState<WorkOrderOperatorInfo | null>(null);
  const [selectedOperatorIds, setSelectedOperatorIds] = useState<number[]>([]);
  const [primaryOperatorId, setPrimaryOperatorId] = useState<number | null>(null);
  
  // Bulk assignment state
  const [bulkDialogOpen, setBulkDialogOpen] = useState(false);
  const [bulkOperationCode, setBulkOperationCode] = useState<string>("");
  const [bulkWorkCenterId, setBulkWorkCenterId] = useState<string>("");
  const [bulkOperatorIds, setBulkOperatorIds] = useState<number[]>([]);
  const [bulkPrimaryOperatorId, setBulkPrimaryOperatorId] = useState<number | null>(null);
  const [bulkMode, setBulkMode] = useState<'overwrite' | 'fill_empty'>('overwrite');

  const { data: workOrdersData, isLoading } = useQuery<{ workOrders: WorkOrderOperatorInfo[] }>({
    queryKey: [`/api/production/planning/plans/${planId}/work-order-operators`],
    enabled: !!planId,
  });

  const { data: operatorsData } = useQuery<{ operators: ProductionOperator[] }>({
    queryKey: ["/api/production/operators", { is_active: "true" }],
  });
  const operators = operatorsData?.operators || [];

  const workCenterSummaries = useMemo(() => {
    if (!workOrdersData?.workOrders) return [];
    
    const byWorkCenter = new Map<number, OperatorWorkCenterSummary>();
    
    for (const wo of workOrdersData.workOrders) {
      const wcId = wo.workCenterId || 0;
      
      if (!byWorkCenter.has(wcId)) {
        byWorkCenter.set(wcId, {
          workCenterId: wcId,
          workCenterCode: wcId === 0 ? "BRAK" : wo.workCenterName?.split(" ")[0] || "N/A",
          workCenterName: wcId === 0 ? "Bez gniazda" : wo.workCenterName || "Nieznane gniazdo",
          workOrders: [],
          operatorSummary: {},
        });
      }
      
      const summary = byWorkCenter.get(wcId)!;
      summary.workOrders.push(wo);
      
      for (const op of wo.assignedOperators) {
        if (!summary.operatorSummary[op.operatorId]) {
          summary.operatorSummary[op.operatorId] = { name: op.operatorName, count: 0, primaryCount: 0 };
        }
        summary.operatorSummary[op.operatorId].count++;
        if (op.isPrimary) {
          summary.operatorSummary[op.operatorId].primaryCount++;
        }
      }
    }
    
    return Array.from(byWorkCenter.values()).sort((a, b) => {
      if (a.workCenterId === 0) return 1;
      if (b.workCenterId === 0) return -1;
      return a.workCenterName.localeCompare(b.workCenterName);
    });
  }, [workOrdersData]);

  // Get unique operation codes for bulk assignment
  const uniqueOperationCodes = useMemo(() => {
    if (!workOrdersData?.workOrders) return [];
    const codes = new Set<string>();
    workOrdersData.workOrders.forEach(wo => {
      if (wo.operationCode) codes.add(wo.operationCode);
    });
    return Array.from(codes).sort();
  }, [workOrdersData]);

  // Get unique work centers for bulk assignment
  const uniqueWorkCenters = useMemo(() => {
    if (!workOrdersData?.workOrders) return [];
    const centers = new Map<number, string>();
    workOrdersData.workOrders.forEach(wo => {
      if (wo.workCenterId && wo.workCenterName) {
        centers.set(wo.workCenterId, wo.workCenterName);
      }
    });
    return Array.from(centers.entries()).map(([id, name]) => ({ id, name })).sort((a, b) => a.name.localeCompare(b.name));
  }, [workOrdersData]);

  const toggleWorkCenter = (wcId: number) => {
    setExpandedWorkCenters((prev) => {
      const next = new Set(prev);
      if (next.has(wcId)) {
        next.delete(wcId);
      } else {
        next.add(wcId);
      }
      return next;
    });
  };

  const openEditDialog = (wo: WorkOrderOperatorInfo) => {
    setSelectedWorkOrder(wo);
    setSelectedOperatorIds(wo.assignedOperators.map(op => op.operatorId));
    const primary = wo.assignedOperators.find(op => op.isPrimary);
    setPrimaryOperatorId(primary?.operatorId || null);
    setEditDialogOpen(true);
  };

  const handleOperatorToggle = (operatorId: number, checked: boolean) => {
    if (checked) {
      setSelectedOperatorIds(prev => [...prev, operatorId]);
      if (selectedOperatorIds.length === 0) {
        setPrimaryOperatorId(operatorId);
      }
    } else {
      setSelectedOperatorIds(prev => prev.filter(id => id !== operatorId));
      if (primaryOperatorId === operatorId) {
        const remaining = selectedOperatorIds.filter(id => id !== operatorId);
        setPrimaryOperatorId(remaining.length > 0 ? remaining[0] : null);
      }
    }
  };

  const handlePrimaryToggle = (operatorId: number) => {
    if (selectedOperatorIds.includes(operatorId)) {
      setPrimaryOperatorId(operatorId);
    }
  };

  const updateOperatorsMutation = useMutation({
    mutationFn: ({ workOrderId, operatorIds, primaryOperatorId }: {
      workOrderId: number;
      operatorIds: number[];
      primaryOperatorId: number | null;
    }) =>
      apiRequest("PUT", `/api/production/work-orders/${workOrderId}/operators`, {
        operatorIds,
        primaryOperatorId,
      }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/work-order-operators`] });
      setEditDialogOpen(false);
      setSelectedWorkOrder(null);
      toast({ title: "Operatorzy zostali zaktualizowani" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować operatorów",
        variant: "destructive",
      });
    },
  });

  const bulkAssignMutation = useMutation({
    mutationFn: (data: {
      planId: number;
      operationCode?: string | null;
      workCenterId?: number | null;
      operatorIds: number[];
      primaryOperatorId?: number | null;
      mode: 'overwrite' | 'fill_empty';
    }) => apiRequest("POST", `/api/production/work-orders/bulk-assign`, data),
    onSuccess: (data: any) => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/work-order-operators`] });
      setBulkDialogOpen(false);
      resetBulkForm();
      toast({ 
        title: "Sukces", 
        description: data.message || `Przypisano operatorów do ${data.assignedCount} zleceń` 
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się przypisać operatorów grupowo",
        variant: "destructive",
      });
    },
  });

  const resetBulkForm = () => {
    setBulkOperationCode("");
    setBulkWorkCenterId("");
    setBulkOperatorIds([]);
    setBulkPrimaryOperatorId(null);
    setBulkMode('overwrite');
  };

  const handleBulkOperatorToggle = (operatorId: number, checked: boolean) => {
    if (checked) {
      setBulkOperatorIds(prev => [...prev, operatorId]);
      if (bulkOperatorIds.length === 0) {
        setBulkPrimaryOperatorId(operatorId);
      }
    } else {
      setBulkOperatorIds(prev => prev.filter(id => id !== operatorId));
      if (bulkPrimaryOperatorId === operatorId) {
        const remaining = bulkOperatorIds.filter(id => id !== operatorId);
        setBulkPrimaryOperatorId(remaining.length > 0 ? remaining[0] : null);
      }
    }
  };

  const handleBulkAssign = () => {
    if (bulkOperatorIds.length === 0) {
      toast({
        title: "Błąd",
        description: "Musisz wybrać co najmniej jednego operatora",
        variant: "destructive",
      });
      return;
    }
    bulkAssignMutation.mutate({
      planId,
      operationCode: bulkOperationCode && bulkOperationCode !== "all" ? bulkOperationCode : null,
      workCenterId: bulkWorkCenterId && bulkWorkCenterId !== "all" ? parseInt(bulkWorkCenterId) : null,
      operatorIds: bulkOperatorIds,
      primaryOperatorId: bulkPrimaryOperatorId,
      mode: bulkMode,
    });
  };

  const handleSaveOperators = () => {
    if (!selectedWorkOrder) return;
    updateOperatorsMutation.mutate({
      workOrderId: selectedWorkOrder.workOrderId,
      operatorIds: selectedOperatorIds,
      primaryOperatorId,
    });
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center p-8">
        <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
      </div>
    );
  }

  const totalWorkOrders = workOrdersData?.workOrders?.length || 0;
  const assignedWorkOrders = workOrdersData?.workOrders?.filter(wo => wo.assignedOperators.length > 0).length || 0;
  const unassignedWorkOrders = totalWorkOrders - assignedWorkOrders;

  return (
    <>
      <div className="space-y-4">
        <div className="flex items-center justify-between flex-wrap gap-2">
          <div className="flex items-center gap-2">
            <Users className="h-5 w-5 text-muted-foreground" />
            <h3 className="font-semibold">Przypisania operatorów</h3>
          </div>
          <div className="flex items-center gap-2">
            {totalWorkOrders > 0 && (
              <Button
                variant="outline"
                size="sm"
                onClick={() => setBulkDialogOpen(true)}
                data-testid="button-bulk-assign"
              >
                <Wand2 className="h-4 w-4 mr-1" />
                Przypisz grupowo
              </Button>
            )}
            <Badge variant="secondary" data-testid="badge-total-work-orders">
              {totalWorkOrders} zleceń
            </Badge>
            {unassignedWorkOrders > 0 && (
              <Badge variant="destructive" data-testid="badge-unassigned">
                <AlertCircle className="h-3 w-3 mr-1" />
                {unassignedWorkOrders} bez operatora
              </Badge>
            )}
          </div>
        </div>

        {workCenterSummaries.length === 0 ? (
          <Card>
            <CardContent className="flex flex-col items-center justify-center py-8 text-center">
              <Factory className="h-12 w-12 text-muted-foreground/50 mb-3" />
              <p className="text-muted-foreground">Brak zleceń roboczych w tym planie</p>
              <p className="text-sm text-muted-foreground mt-1">
                Wygeneruj zlecenia ZLP aby przypisać operatorów
              </p>
            </CardContent>
          </Card>
        ) : (
          <div className="space-y-3">
            {workCenterSummaries.map((wcSummary) => (
              <Card key={wcSummary.workCenterId} data-testid={`card-workcenter-${wcSummary.workCenterId}`}>
                <Collapsible
                  open={expandedWorkCenters.has(wcSummary.workCenterId)}
                  onOpenChange={() => toggleWorkCenter(wcSummary.workCenterId)}
                >
                  <CollapsibleTrigger asChild>
                    <CardHeader className="py-3 cursor-pointer hover-elevate">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center gap-3">
                          {expandedWorkCenters.has(wcSummary.workCenterId) ? (
                            <ChevronDown className="h-4 w-4" />
                          ) : (
                            <ChevronRight className="h-4 w-4" />
                          )}
                          <Factory className="h-4 w-4 text-muted-foreground" />
                          <div>
                            <span className="font-medium">{wcSummary.workCenterName}</span>
                            <span className="text-sm text-muted-foreground ml-2">
                              ({wcSummary.workOrders.length} zleceń)
                            </span>
                          </div>
                        </div>
                        <div className="flex items-center gap-2">
                          {Object.entries(wcSummary.operatorSummary).slice(0, 3).map(([opId, info]) => (
                            <Tooltip key={opId}>
                              <TooltipTrigger>
                                <Badge variant="outline" className="text-xs">
                                  <User className="h-3 w-3 mr-1" />
                                  {info.name.split(" ")[0]}
                                  <span className="ml-1 text-muted-foreground">×{info.count}</span>
                                </Badge>
                              </TooltipTrigger>
                              <TooltipContent>
                                {info.name}: {info.count} zleceń ({info.primaryCount} jako główny)
                              </TooltipContent>
                            </Tooltip>
                          ))}
                          {Object.keys(wcSummary.operatorSummary).length > 3 && (
                            <Badge variant="secondary" className="text-xs">
                              +{Object.keys(wcSummary.operatorSummary).length - 3}
                            </Badge>
                          )}
                        </div>
                      </div>
                    </CardHeader>
                  </CollapsibleTrigger>
                  <CollapsibleContent>
                    <CardContent className="pt-0">
                      <ScrollArea className="h-[300px]">
                        <div className="space-y-2">
                          {wcSummary.workOrders.map((wo) => (
                            <div
                              key={wo.workOrderId}
                              className="flex items-center justify-between p-2 rounded-md border bg-muted/30"
                              data-testid={`row-workorder-${wo.workOrderId}`}
                            >
                              <div className="flex items-center gap-3 flex-1 min-w-0">
                                <div className="flex flex-col min-w-0">
                                  <div className="flex items-center gap-2">
                                    <span className="font-mono text-sm">{wo.workOrderNumber}</span>
                                    {wo.colorCode && (
                                      <span
                                        className={`text-[10px] px-1.5 py-0.5 rounded ${
                                          colorBadgeColors[wo.colorCode] || "bg-gray-200 text-gray-800"
                                        }`}
                                      >
                                        {wo.colorCode}
                                      </span>
                                    )}
                                  </div>
                                  <span className="text-xs text-muted-foreground truncate">
                                    {wo.operationName || wo.operationCode || "Operacja"} • {wo.zlpNumber}
                                  </span>
                                </div>
                              </div>
                              <div className="flex items-center gap-2">
                                {wo.assignedOperators.length > 0 ? (
                                  <div className="flex items-center gap-1">
                                    {wo.assignedOperators.map((op) => (
                                      <Badge
                                        key={op.operatorId}
                                        variant={op.isPrimary ? "default" : "secondary"}
                                        className="text-xs"
                                      >
                                        {op.isPrimary && <Star className="h-3 w-3 mr-1 fill-current" />}
                                        {op.operatorName.split(" ")[0]}
                                      </Badge>
                                    ))}
                                  </div>
                                ) : (
                                  <Badge variant="outline" className="text-xs text-muted-foreground">
                                    Brak operatora
                                  </Badge>
                                )}
                                <Button
                                  variant="ghost"
                                  size="icon"
                                  className="h-7 w-7"
                                  onClick={() => openEditDialog(wo)}
                                  data-testid={`button-edit-operators-${wo.workOrderId}`}
                                >
                                  <Edit className="h-4 w-4" />
                                </Button>
                              </div>
                            </div>
                          ))}
                        </div>
                      </ScrollArea>
                    </CardContent>
                  </CollapsibleContent>
                </Collapsible>
              </Card>
            ))}
          </div>
        )}
      </div>

      <Dialog open={editDialogOpen} onOpenChange={setEditDialogOpen}>
        <DialogContent className="max-w-md" data-testid="dialog-edit-operators">
          <DialogHeader>
            <DialogTitle>Edytuj operatorów</DialogTitle>
            <DialogDescription>
              {selectedWorkOrder?.workOrderNumber} • {selectedWorkOrder?.operationName}
            </DialogDescription>
          </DialogHeader>
          <div className="py-4">
            <p className="text-sm text-muted-foreground mb-3">
              Wybierz operatorów (kliknij gwiazdkę aby oznaczyć głównego):
            </p>
            <ScrollArea className="h-[250px] border rounded-md p-2">
              <div className="space-y-2">
                {operators.map((op) => {
                  const isSelected = selectedOperatorIds.includes(op.id);
                  const isPrimary = primaryOperatorId === op.id;
                  return (
                    <div
                      key={op.id}
                      className={`flex items-center gap-3 p-2 rounded-md hover-elevate ${
                        isSelected ? "bg-accent/50" : ""
                      }`}
                      data-testid={`operator-row-${op.id}`}
                    >
                      <Checkbox
                        id={`edit-operator-${op.id}`}
                        checked={isSelected}
                        onCheckedChange={(checked) => handleOperatorToggle(op.id, !!checked)}
                        data-testid={`checkbox-operator-${op.id}`}
                      />
                      <Label
                        htmlFor={`edit-operator-${op.id}`}
                        className="flex-1 cursor-pointer text-sm"
                      >
                        {op.full_name} {op.short_code && `(${op.short_code})`}
                      </Label>
                      {isSelected && (
                        <Button
                          type="button"
                          variant="ghost"
                          size="icon"
                          className={`h-7 w-7 ${isPrimary ? "text-yellow-500" : "text-muted-foreground"}`}
                          onClick={() => handlePrimaryToggle(op.id)}
                          title={isPrimary ? "Główny operator" : "Ustaw jako głównego"}
                          data-testid={`button-primary-${op.id}`}
                        >
                          <Star className={`h-4 w-4 ${isPrimary ? "fill-current" : ""}`} />
                        </Button>
                      )}
                    </div>
                  );
                })}
              </div>
            </ScrollArea>
            {selectedOperatorIds.length > 0 && (
              <p className="text-xs text-muted-foreground mt-2">
                Wybrano: {selectedOperatorIds.length} operator{selectedOperatorIds.length === 1 ? "a" : "ów"}
                {primaryOperatorId && (
                  <>, główny: {operators.find(o => o.id === primaryOperatorId)?.full_name}</>
                )}
              </p>
            )}
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setEditDialogOpen(false)}
              data-testid="button-cancel-edit"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleSaveOperators}
              disabled={updateOperatorsMutation.isPending}
              data-testid="button-save-edit"
            >
              {updateOperatorsMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Zapisywanie...
                </>
              ) : (
                "Zapisz"
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Assignment Dialog */}
      <Dialog open={bulkDialogOpen} onOpenChange={setBulkDialogOpen}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>Grupowe przypisanie operatorów</DialogTitle>
            <DialogDescription>
              Przypisz operatorów do wielu zleceń jednocześnie według typu operacji lub gniazda.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="bulk-operation">Typ operacji</Label>
                <Select value={bulkOperationCode} onValueChange={setBulkOperationCode}>
                  <SelectTrigger id="bulk-operation" data-testid="select-bulk-operation">
                    <SelectValue placeholder="Wszystkie operacje" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie operacje</SelectItem>
                    {uniqueOperationCodes.map(code => (
                      <SelectItem key={code} value={code}>{code}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="bulk-workcenter">Gniazdo robocze</Label>
                <Select value={bulkWorkCenterId} onValueChange={setBulkWorkCenterId}>
                  <SelectTrigger id="bulk-workcenter" data-testid="select-bulk-workcenter">
                    <SelectValue placeholder="Wszystkie gniazda" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie gniazda</SelectItem>
                    {uniqueWorkCenters.map(wc => (
                      <SelectItem key={wc.id} value={wc.id.toString()}>{wc.name}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="space-y-2">
              <Label>Tryb przypisania</Label>
              <Select value={bulkMode} onValueChange={(v) => setBulkMode(v as 'overwrite' | 'fill_empty')}>
                <SelectTrigger data-testid="select-bulk-mode">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="overwrite">Nadpisz wszystkie</SelectItem>
                  <SelectItem value="fill_empty">Tylko puste zlecenia</SelectItem>
                </SelectContent>
              </Select>
              <p className="text-xs text-muted-foreground">
                {bulkMode === 'overwrite' 
                  ? "Zastąpi wszystkich operatorów w wybranych zleceniach"
                  : "Przypisze operatorów tylko do zleceń bez operatora"}
              </p>
            </div>

            <div className="space-y-2">
              <Label>Wybierz operatorów</Label>
              <ScrollArea className="h-[200px] border rounded-md p-2">
                <div className="space-y-1">
                  {operators.map((op) => {
                    const isSelected = bulkOperatorIds.includes(op.id);
                    const isPrimary = bulkPrimaryOperatorId === op.id;
                    return (
                      <div
                        key={op.id}
                        className={`flex items-center gap-3 p-2 rounded-md hover-elevate ${
                          isSelected ? "bg-accent/50" : ""
                        }`}
                        data-testid={`bulk-operator-row-${op.id}`}
                      >
                        <Checkbox
                          id={`bulk-operator-${op.id}`}
                          checked={isSelected}
                          onCheckedChange={(checked) => handleBulkOperatorToggle(op.id, !!checked)}
                          data-testid={`bulk-checkbox-operator-${op.id}`}
                        />
                        <Label
                          htmlFor={`bulk-operator-${op.id}`}
                          className="flex-1 cursor-pointer text-sm"
                        >
                          {op.full_name} {op.short_code && `(${op.short_code})`}
                        </Label>
                        {isSelected && (
                          <Button
                            type="button"
                            variant="ghost"
                            size="icon"
                            className={`h-7 w-7 ${isPrimary ? "text-yellow-500" : "text-muted-foreground"}`}
                            onClick={() => setBulkPrimaryOperatorId(op.id)}
                            title={isPrimary ? "Główny operator" : "Ustaw jako głównego"}
                            data-testid={`bulk-button-primary-${op.id}`}
                          >
                            <Star className={`h-4 w-4 ${isPrimary ? "fill-current" : ""}`} />
                          </Button>
                        )}
                      </div>
                    );
                  })}
                </div>
              </ScrollArea>
              {bulkOperatorIds.length > 0 && (
                <p className="text-xs text-muted-foreground mt-2">
                  Wybrano: {bulkOperatorIds.length} operator{bulkOperatorIds.length === 1 ? "a" : "ów"}
                  {bulkPrimaryOperatorId && (
                    <>, główny: {operators.find(o => o.id === bulkPrimaryOperatorId)?.full_name}</>
                  )}
                </p>
              )}
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setBulkDialogOpen(false);
                resetBulkForm();
              }}
              data-testid="button-cancel-bulk"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleBulkAssign}
              disabled={bulkAssignMutation.isPending || bulkOperatorIds.length === 0}
              data-testid="button-save-bulk"
            >
              {bulkAssignMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Przypisywanie...
                </>
              ) : (
                <>
                  <Wand2 className="w-4 h-4 mr-1" />
                  Przypisz
                </>
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  );
}
