import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Settings2, Wrench, Package, ArrowRight, Check, X, ChevronRight } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

interface RoutingOperation {
  id: number;
  sequence: number;
  code: string;
  name: string;
  description: string | null;
  defaultWorkCenterId: number | null;
  defaultWorkCenterCode: string | null;
  defaultWorkCenterName: string | null;
  defaultBufferBeforeId: number | null;
  defaultBufferBeforeName: string | null;
  defaultBufferAfterId: number | null;
  defaultBufferAfterName: string | null;
  createsBuffer: boolean;
  bufferLocationId: number | null;
  bufferLocationName: string | null;
  overrideId: number | null;
  overrideWorkCenterId: number | null;
  overrideWorkCenterCode: string | null;
  overrideWorkCenterName: string | null;
  overrideBufferBeforeId: number | null;
  overrideBufferBeforeName: string | null;
  overrideBufferAfterId: number | null;
  overrideBufferAfterName: string | null;
  skipOperation: boolean;
  overrideNotes: string | null;
  effectiveWorkCenterId: number | null;
  effectiveWorkCenterName: string | null;
  effectiveBufferBeforeId: number | null;
  effectiveBufferAfterId: number | null;
  hasOverride: boolean;
}

interface AssignmentInfo {
  id: number;
  planId: number;
  routingId: number;
  routingCode: string;
  routingName: string;
  materialType: string | null;
  colorCode: string | null;
  componentPattern: string | null;
}

interface OperationsResponse {
  assignment: AssignmentInfo;
  operations: RoutingOperation[];
}

interface WorkCenter {
  id: number;
  code: string;
  name: string;
}

interface Location {
  id: number;
  code: string;
  name: string;
}

interface OperationOverridesDialogProps {
  assignmentId: number;
  assignmentLabel: string;
}

interface LocalOverride {
  overrideWorkCenterId: number | null | undefined; // undefined = unchanged, null = reset to default
  overrideBufferBeforeId: number | null | undefined;
  overrideBufferAfterId: number | null | undefined;
  skipOperation: boolean | undefined;
}

export function OperationOverridesDialog({ assignmentId, assignmentLabel }: OperationOverridesDialogProps) {
  const [open, setOpen] = useState(false);
  const [localOverrides, setLocalOverrides] = useState<Record<number, LocalOverride>>({});
  const [hasChanges, setHasChanges] = useState(false);
  const { toast } = useToast();

  const { data, isLoading } = useQuery<OperationsResponse>({
    queryKey: [`/api/production/plan-routing-assignments/${assignmentId}/operations`],
    enabled: open,
  });

  const { data: workCenters = [] } = useQuery<WorkCenter[]>({
    queryKey: ["/api/production/work-centers"],
    enabled: open,
  });

  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ["/api/production/locations"],
    enabled: open,
  });

  const bufferLocations = locations.filter(loc => 
    loc.code.toLowerCase().includes('buf') || 
    loc.name.toLowerCase().includes('bufor') ||
    loc.name.toLowerCase().includes('buffer')
  );

  const saveMutation = useMutation({
    mutationFn: async () => {
      if (!data) return;
      
      // Build operations list with proper handling of changes
      const operations = data.operations.map(op => {
        const local = localOverrides[op.id];
        
        // Determine final values: local change > existing override > null (default)
        // undefined in local means "unchanged from server", null means "reset to default"
        const finalWorkCenter = local?.overrideWorkCenterId !== undefined 
          ? local.overrideWorkCenterId 
          : op.overrideWorkCenterId;
        const finalBufferBefore = local?.overrideBufferBeforeId !== undefined 
          ? local.overrideBufferBeforeId 
          : op.overrideBufferBeforeId;
        const finalBufferAfter = local?.overrideBufferAfterId !== undefined 
          ? local.overrideBufferAfterId 
          : op.overrideBufferAfterId;
        const finalSkip = local?.skipOperation !== undefined 
          ? local.skipOperation 
          : op.skipOperation;

        return {
          routingOperationId: op.id,
          overrideWorkCenterId: finalWorkCenter,
          overrideBufferBeforeId: finalBufferBefore,
          overrideBufferAfterId: finalBufferAfter,
          skipOperation: finalSkip,
        };
      });

      return apiRequest("PUT", `/api/production/plan-routing-assignments/${assignmentId}/operations/bulk`, {
        operations,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ 
        queryKey: [`/api/production/plan-routing-assignments/${assignmentId}/operations`] 
      });
      setHasChanges(false);
      setLocalOverrides({});
      toast({ title: "Zapisano nadpisania operacji" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const handleOverrideChange = (opId: number, field: keyof LocalOverride, value: any) => {
    setLocalOverrides(prev => ({
      ...prev,
      [opId]: {
        ...prev[opId],
        [field]: value,
      },
    }));
    setHasChanges(true);
  };

  const getEffectiveValue = (op: RoutingOperation, field: keyof LocalOverride) => {
    const local = localOverrides[op.id];
    if (local && local[field] !== undefined) {
      return local[field];
    }
    switch (field) {
      case 'overrideWorkCenterId':
        return op.overrideWorkCenterId;
      case 'overrideBufferBeforeId':
        return op.overrideBufferBeforeId;
      case 'overrideBufferAfterId':
        return op.overrideBufferAfterId;
      case 'skipOperation':
        return op.skipOperation;
      default:
        return null;
    }
  };

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>
        <Button 
          size="icon" 
          variant="ghost" 
          className="h-7 w-7"
          data-testid={`button-operation-overrides-${assignmentId}`}
        >
          <Settings2 className="h-4 w-4" />
        </Button>
      </DialogTrigger>
      <DialogContent className="max-w-3xl max-h-[90vh]">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Settings2 className="h-5 w-5 text-primary" />
            Konfiguracja operacji
          </DialogTitle>
          <div className="text-sm text-muted-foreground flex flex-wrap items-center gap-2">
            <Badge variant="outline">{assignmentLabel}</Badge>
            {data && (
              <>
                <ArrowRight className="h-3 w-3" />
                <Badge variant="default">{data.assignment.routingName}</Badge>
              </>
            )}
          </div>
        </DialogHeader>

        {isLoading ? (
          <div className="flex items-center justify-center py-12">
            <div className="text-muted-foreground">Ładowanie operacji...</div>
          </div>
        ) : data?.operations.length === 0 ? (
          <div className="text-center py-12 text-muted-foreground">
            <Settings2 className="h-12 w-12 mx-auto mb-4 opacity-30" />
            <p>Brak operacji w tej marszrucie.</p>
          </div>
        ) : (
          <>
            <ScrollArea className="h-[400px] pr-4">
              <div className="space-y-3">
                {data?.operations.map((op, index) => {
                  const skipOperation = getEffectiveValue(op, 'skipOperation') as boolean;
                  const overrideWorkCenterId = getEffectiveValue(op, 'overrideWorkCenterId') as number | null;
                  const overrideBufferBeforeId = getEffectiveValue(op, 'overrideBufferBeforeId') as number | null;
                  const overrideBufferAfterId = getEffectiveValue(op, 'overrideBufferAfterId') as number | null;
                  
                  return (
                    <div 
                      key={op.id} 
                      className={`rounded-lg border p-4 ${skipOperation ? 'opacity-50 bg-muted/50' : 'bg-card'}`}
                      data-testid={`operation-row-${op.id}`}
                    >
                      <div className="flex items-start justify-between gap-4 mb-3">
                        <div className="flex items-center gap-3">
                          <div className="flex items-center justify-center w-8 h-8 rounded-full bg-primary/10 text-primary font-medium text-sm">
                            {op.sequence}
                          </div>
                          <div>
                            <div className="font-medium flex items-center gap-2">
                              {op.name}
                              <Badge variant="secondary" className="text-xs font-mono">
                                {op.code}
                              </Badge>
                            </div>
                            {op.description && (
                              <div className="text-xs text-muted-foreground">{op.description}</div>
                            )}
                          </div>
                        </div>
                        
                        <div className="flex items-center gap-2">
                          <Label htmlFor={`skip-${op.id}`} className="text-xs text-muted-foreground">
                            Pomiń
                          </Label>
                          <Switch
                            id={`skip-${op.id}`}
                            checked={skipOperation}
                            onCheckedChange={(v) => handleOverrideChange(op.id, 'skipOperation', v)}
                            data-testid={`switch-skip-${op.id}`}
                          />
                        </div>
                      </div>

                      {!skipOperation && (
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
                          <div className="space-y-1.5">
                            <Label className="text-xs flex items-center gap-1">
                              <Wrench className="h-3 w-3" />
                              Stanowisko pracy
                            </Label>
                            <Select
                              value={overrideWorkCenterId ? String(overrideWorkCenterId) : "default"}
                              onValueChange={(v) => handleOverrideChange(
                                op.id, 
                                'overrideWorkCenterId', 
                                v === "default" ? null : parseInt(v)
                              )}
                            >
                              <SelectTrigger className="h-8" data-testid={`select-workcenter-${op.id}`}>
                                <SelectValue />
                              </SelectTrigger>
                              <SelectContent>
                                <SelectItem value="default">
                                  <span className="flex items-center gap-1.5">
                                    <span className="text-muted-foreground">Domyślne:</span>
                                    <span>{op.defaultWorkCenterName || 'Nie przypisane'}</span>
                                  </span>
                                </SelectItem>
                                <Separator className="my-1" />
                                {workCenters.map((wc) => (
                                  <SelectItem key={wc.id} value={String(wc.id)}>
                                    {wc.name}
                                  </SelectItem>
                                ))}
                              </SelectContent>
                            </Select>
                            {overrideWorkCenterId && (
                              <div className="flex items-center gap-1 text-xs text-amber-600">
                                <Check className="h-3 w-3" />
                                Nadpisane
                              </div>
                            )}
                          </div>

                          <div className="space-y-1.5">
                            <Label className="text-xs flex items-center gap-1">
                              <Package className="h-3 w-3" />
                              Bufor przed
                            </Label>
                            <Select
                              value={overrideBufferBeforeId ? String(overrideBufferBeforeId) : "default"}
                              onValueChange={(v) => handleOverrideChange(
                                op.id, 
                                'overrideBufferBeforeId', 
                                v === "default" ? null : parseInt(v)
                              )}
                            >
                              <SelectTrigger className="h-8" data-testid={`select-buffer-before-${op.id}`}>
                                <SelectValue />
                              </SelectTrigger>
                              <SelectContent>
                                <SelectItem value="default">
                                  <span className="text-muted-foreground">
                                    {op.defaultBufferBeforeName || 'Brak bufora'}
                                  </span>
                                </SelectItem>
                                <Separator className="my-1" />
                                {bufferLocations.map((loc) => (
                                  <SelectItem key={loc.id} value={String(loc.id)}>
                                    {loc.name}
                                  </SelectItem>
                                ))}
                              </SelectContent>
                            </Select>
                          </div>

                          <div className="space-y-1.5">
                            <Label className="text-xs flex items-center gap-1">
                              <Package className="h-3 w-3" />
                              Bufor po
                            </Label>
                            <Select
                              value={overrideBufferAfterId ? String(overrideBufferAfterId) : "default"}
                              onValueChange={(v) => handleOverrideChange(
                                op.id, 
                                'overrideBufferAfterId', 
                                v === "default" ? null : parseInt(v)
                              )}
                            >
                              <SelectTrigger className="h-8" data-testid={`select-buffer-after-${op.id}`}>
                                <SelectValue />
                              </SelectTrigger>
                              <SelectContent>
                                <SelectItem value="default">
                                  <span className="text-muted-foreground">
                                    {op.defaultBufferAfterName || 'Brak bufora'}
                                  </span>
                                </SelectItem>
                                <Separator className="my-1" />
                                {bufferLocations.map((loc) => (
                                  <SelectItem key={loc.id} value={String(loc.id)}>
                                    {loc.name}
                                  </SelectItem>
                                ))}
                              </SelectContent>
                            </Select>
                          </div>
                        </div>
                      )}

                      {index < (data?.operations.length || 0) - 1 && !skipOperation && (
                        <div className="flex justify-center mt-3">
                          <ChevronRight className="h-4 w-4 text-muted-foreground rotate-90" />
                        </div>
                      )}
                    </div>
                  );
                })}
              </div>
            </ScrollArea>

            <Separator />

            <div className="flex items-center justify-between">
              <div className="text-xs text-muted-foreground">
                {hasChanges ? (
                  <span className="text-amber-600 flex items-center gap-1">
                    <span className="w-2 h-2 rounded-full bg-amber-500" />
                    Niezapisane zmiany
                  </span>
                ) : (
                  <span className="text-green-600 flex items-center gap-1">
                    <Check className="h-3 w-3" />
                    Wszystkie zmiany zapisane
                  </span>
                )}
              </div>
              <div className="flex gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => {
                    setLocalOverrides({});
                    setHasChanges(false);
                  }}
                  disabled={!hasChanges}
                  data-testid="button-reset-overrides"
                >
                  <X className="h-4 w-4 mr-1" />
                  Resetuj
                </Button>
                <Button
                  size="sm"
                  onClick={() => saveMutation.mutate()}
                  disabled={!hasChanges || saveMutation.isPending}
                  data-testid="button-save-overrides"
                >
                  <Check className="h-4 w-4 mr-1" />
                  Zapisz zmiany
                </Button>
              </div>
            </div>
          </>
        )}
      </DialogContent>
    </Dialog>
  );
}
