import { useState, useMemo } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Checkbox } from "@/components/ui/checkbox";
import { useToast } from "@/hooks/use-toast";
import {
  Scissors,
  ArrowUpDown,
  Loader2,
  Wand2,
  GripVertical,
  Package,
  Warehouse,
  ChevronUp,
  ChevronDown,
  Check,
} from "lucide-react";
import {
  DndContext,
  closestCenter,
  KeyboardSensor,
  PointerSensor,
  useSensor,
  useSensors,
  DragEndEvent,
} from "@dnd-kit/core";
import {
  arrayMove,
  SortableContext,
  sortableKeyboardCoordinates,
  useSortable,
  verticalListSortingStrategy,
} from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";

interface ZlpCuttingInfo {
  id: number;
  order_number: string;
  color_code: string | null;
  cutting_sequence: number | null;
  cutting_priority: string | null;
  requires_cutting: boolean;
  transport_batch_id: number | null;
  transport_batch_number: string | null;
  transport_batch_label: string | null;
  status: string;
  workflow_stage: string | null;
  product_name: string | null;
  product_sku: string | null;
}

interface TransportBatch {
  id: number;
  batch_number: string;
  label: string | null;
}

interface CuttingSequencePanelProps {
  planId: number;
}

const colorBadgeColors: Record<string, string> = {
  BIALY: "bg-white border border-gray-300 text-gray-800",
  SONOMA: "bg-amber-200 text-amber-900",
  WOTAN: "bg-amber-700 text-white",
  SUROWA: "bg-yellow-100 text-yellow-800",
  HDF: "bg-stone-400 text-white",
  CZARNY: "bg-black text-white",
};

const priorityLabels: Record<string, string> = {
  low: "Niski",
  normal: "Normalny",
  high: "Wysoki",
  urgent: "Pilny",
};

const priorityColors: Record<string, string> = {
  low: "bg-gray-100 text-gray-700",
  normal: "bg-blue-100 text-blue-700",
  high: "bg-orange-100 text-orange-700",
  urgent: "bg-red-100 text-red-700",
};

export function CuttingSequencePanel({ planId }: CuttingSequencePanelProps) {
  const { toast } = useToast();
  const [autoAssignDialogOpen, setAutoAssignDialogOpen] = useState(false);
  const [selectedColors, setSelectedColors] = useState<string[]>([]);
  const [localItems, setLocalItems] = useState<ZlpCuttingInfo[]>([]);
  const [hasUnsavedChanges, setHasUnsavedChanges] = useState(false);

  const { data: zlps = [], isLoading } = useQuery<ZlpCuttingInfo[]>({
    queryKey: [`/api/production/planning/plans/${planId}/cutting-sequence`],
    enabled: !!planId,
  });

  const { data: batches = [] } = useQuery<TransportBatch[]>({
    queryKey: [`/api/production/planning/plans/${planId}/transport-batches`],
    enabled: !!planId,
  });

  const cuttingItems = useMemo(() => {
    const items = hasUnsavedChanges ? localItems : zlps;
    return items.filter((z) => z.requires_cutting);
  }, [zlps, localItems, hasUnsavedChanges]);

  const warehouseItems = useMemo(() => {
    return zlps.filter((z) => !z.requires_cutting);
  }, [zlps]);

  const uniqueColors = useMemo(() => {
    const colors = new Set<string>();
    zlps.forEach((z) => {
      if (z.color_code && z.requires_cutting) {
        colors.add(z.color_code);
      }
    });
    return Array.from(colors);
  }, [zlps]);

  const sensors = useSensors(
    useSensor(PointerSensor),
    useSensor(KeyboardSensor, {
      coordinateGetter: sortableKeyboardCoordinates,
    })
  );

  const updateSequenceMutation = useMutation({
    mutationFn: async (items: { id: number; cuttingSequence: number }[]) => {
      return apiRequest("PATCH", `/api/production/planning/plans/${planId}/cutting-sequence`, { items });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/cutting-sequence`] });
      setHasUnsavedChanges(false);
      toast({ title: "Zapisano kolejność cięcia" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zapisać kolejności",
        variant: "destructive",
      });
    },
  });

  const autoAssignMutation = useMutation({
    mutationFn: async (colorOrder: string[]) => {
      return apiRequest("POST", `/api/production/planning/plans/${planId}/auto-cutting-sequence`, { colorOrder });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/cutting-sequence`] });
      setAutoAssignDialogOpen(false);
      setSelectedColors([]);
      toast({ title: "Przypisano kolejność cięcia automatycznie" });
    },
  });

  const handleDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;

    if (over && active.id !== over.id) {
      const items = hasUnsavedChanges ? localItems : [...zlps];
      const cuttingOnly = items.filter((i) => i.requires_cutting);
      const oldIndex = cuttingOnly.findIndex((i) => i.id === active.id);
      const newIndex = cuttingOnly.findIndex((i) => i.id === over.id);

      if (oldIndex !== -1 && newIndex !== -1) {
        const reordered = arrayMove(cuttingOnly, oldIndex, newIndex);
        const withSequence = reordered.map((item, index) => ({
          ...item,
          cutting_sequence: index + 1,
        }));
        
        const combined = [...withSequence, ...items.filter((i) => !i.requires_cutting)];
        setLocalItems(combined);
        setHasUnsavedChanges(true);
      }
    }
  };

  const saveChanges = () => {
    const updates = cuttingItems.map((item, index) => ({
      id: item.id,
      cuttingSequence: index + 1,
    }));
    updateSequenceMutation.mutate(updates);
  };

  const toggleColorSelection = (color: string) => {
    setSelectedColors((prev) =>
      prev.includes(color)
        ? prev.filter((c) => c !== color)
        : [...prev, color]
    );
  };

  const moveColorUp = (color: string) => {
    setSelectedColors((prev) => {
      const index = prev.indexOf(color);
      if (index > 0) {
        const newOrder = [...prev];
        [newOrder[index - 1], newOrder[index]] = [newOrder[index], newOrder[index - 1]];
        return newOrder;
      }
      return prev;
    });
  };

  const moveColorDown = (color: string) => {
    setSelectedColors((prev) => {
      const index = prev.indexOf(color);
      if (index < prev.length - 1) {
        const newOrder = [...prev];
        [newOrder[index], newOrder[index + 1]] = [newOrder[index + 1], newOrder[index]];
        return newOrder;
      }
      return prev;
    });
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center p-8">
        <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
      </div>
    );
  }

  return (
    <div className="space-y-4">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-2">
          <Scissors className="h-5 w-5 text-muted-foreground" />
          <h3 className="font-semibold">Kolejność cięcia</h3>
          <Badge variant="secondary">{cuttingItems.length} do cięcia</Badge>
          {warehouseItems.length > 0 && (
            <Badge variant="outline">{warehouseItems.length} z magazynu</Badge>
          )}
        </div>
        <div className="flex items-center gap-2">
          {hasUnsavedChanges && (
            <Button
              size="sm"
              onClick={saveChanges}
              disabled={updateSequenceMutation.isPending}
              data-testid="button-save-cutting-sequence"
            >
              {updateSequenceMutation.isPending && <Loader2 className="h-4 w-4 mr-2 animate-spin" />}
              <Check className="h-4 w-4 mr-1" />
              Zapisz kolejność
            </Button>
          )}
          <Button
            size="sm"
            variant="outline"
            onClick={() => {
              setSelectedColors(uniqueColors);
              setAutoAssignDialogOpen(true);
            }}
            data-testid="button-auto-assign-sequence"
          >
            <Wand2 className="h-4 w-4 mr-1" />
            Auto-przypisz
          </Button>
        </div>
      </div>

      {cuttingItems.length === 0 && warehouseItems.length === 0 ? (
        <Card>
          <CardContent className="flex flex-col items-center justify-center py-8 text-center">
            <Scissors className="h-12 w-12 text-muted-foreground/50 mb-3" />
            <p className="text-muted-foreground">Brak ZLP do planowania cięcia</p>
            <p className="text-sm text-muted-foreground mt-1">
              Wygeneruj ZLP aby zaplanować kolejność cięcia
            </p>
          </CardContent>
        </Card>
      ) : (
        <div className="grid gap-4 lg:grid-cols-2">
          <Card>
            <CardHeader className="py-3">
              <CardTitle className="text-sm font-medium flex items-center gap-2">
                <Scissors className="h-4 w-4" />
                Do cięcia
                <Badge variant="secondary" className="ml-auto">{cuttingItems.length}</Badge>
              </CardTitle>
            </CardHeader>
            <Separator />
            <CardContent className="p-0">
              <ScrollArea className="h-[400px]">
                <DndContext
                  sensors={sensors}
                  collisionDetection={closestCenter}
                  onDragEnd={handleDragEnd}
                >
                  <SortableContext
                    items={cuttingItems.map((i) => i.id)}
                    strategy={verticalListSortingStrategy}
                  >
                    <div className="divide-y">
                      {cuttingItems.map((item, index) => (
                        <SortableZlpItem
                          key={item.id}
                          item={item}
                          index={index}
                          batches={batches}
                        />
                      ))}
                    </div>
                  </SortableContext>
                </DndContext>
              </ScrollArea>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="py-3">
              <CardTitle className="text-sm font-medium flex items-center gap-2">
                <Warehouse className="h-4 w-4" />
                Z magazynu (pomijają cięcie)
                <Badge variant="outline" className="ml-auto">{warehouseItems.length}</Badge>
              </CardTitle>
            </CardHeader>
            <Separator />
            <CardContent className="p-0">
              <ScrollArea className="h-[400px]">
                <div className="divide-y">
                  {warehouseItems.map((item) => (
                    <div
                      key={item.id}
                      className="flex items-center gap-3 p-3 bg-muted/30"
                    >
                      <Package className="h-4 w-4 text-muted-foreground" />
                      <div className="flex-1 min-w-0">
                        <div className="font-medium text-sm truncate">
                          {item.order_number}
                        </div>
                        <div className="text-xs text-muted-foreground truncate">
                          {item.product_name}
                        </div>
                      </div>
                      {item.color_code && (
                        <Badge
                          variant="secondary"
                          className={colorBadgeColors[item.color_code] || ""}
                        >
                          {item.color_code}
                        </Badge>
                      )}
                    </div>
                  ))}
                  {warehouseItems.length === 0 && (
                    <div className="p-8 text-center text-muted-foreground text-sm">
                      Brak elementów z magazynu
                    </div>
                  )}
                </div>
              </ScrollArea>
            </CardContent>
          </Card>
        </div>
      )}

      <Dialog open={autoAssignDialogOpen} onOpenChange={setAutoAssignDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Automatyczne przypisanie kolejności cięcia</DialogTitle>
            <DialogDescription>
              Ustaw kolejność kolorów w jakiej mają być cięte. ZLP będą pogrupowane według koloru i przypisane do sekwencji.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <p className="text-sm font-medium">Dostępne kolory:</p>
              <div className="flex flex-wrap gap-2">
                {uniqueColors.map((color) => (
                  <Button
                    key={color}
                    size="sm"
                    variant={selectedColors.includes(color) ? "default" : "outline"}
                    onClick={() => toggleColorSelection(color)}
                    className={selectedColors.includes(color) ? "" : colorBadgeColors[color]}
                  >
                    {color}
                  </Button>
                ))}
              </div>
            </div>

            {selectedColors.length > 0 && (
              <div className="space-y-2">
                <p className="text-sm font-medium">Kolejność cięcia:</p>
                <div className="space-y-1">
                  {selectedColors.map((color, index) => (
                    <div
                      key={color}
                      className="flex items-center gap-2 p-2 bg-muted rounded-md"
                    >
                      <span className="w-6 text-center font-medium">{index + 1}.</span>
                      <Badge className={colorBadgeColors[color] || ""}>{color}</Badge>
                      <div className="ml-auto flex gap-1">
                        <Button
                          size="icon"
                          variant="ghost"
                          className="h-6 w-6"
                          onClick={() => moveColorUp(color)}
                          disabled={index === 0}
                        >
                          <ChevronUp className="h-4 w-4" />
                        </Button>
                        <Button
                          size="icon"
                          variant="ghost"
                          className="h-6 w-6"
                          onClick={() => moveColorDown(color)}
                          disabled={index === selectedColors.length - 1}
                        >
                          <ChevronDown className="h-4 w-4" />
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setAutoAssignDialogOpen(false)}>
              Anuluj
            </Button>
            <Button
              onClick={() => autoAssignMutation.mutate(selectedColors)}
              disabled={selectedColors.length === 0 || autoAssignMutation.isPending}
              data-testid="button-confirm-auto-assign"
            >
              {autoAssignMutation.isPending && <Loader2 className="h-4 w-4 mr-2 animate-spin" />}
              Przypisz kolejność
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}

function SortableZlpItem({
  item,
  index,
  batches,
}: {
  item: ZlpCuttingInfo;
  index: number;
  batches: TransportBatch[];
}) {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ id: item.id });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.5 : 1,
  };

  return (
    <div
      ref={setNodeRef}
      style={style}
      className="flex items-center gap-3 p-3 hover:bg-muted/50"
    >
      <div
        {...attributes}
        {...listeners}
        className="cursor-grab active:cursor-grabbing touch-none"
      >
        <GripVertical className="h-4 w-4 text-muted-foreground" />
      </div>
      <div className="w-8 h-8 rounded-full bg-primary/10 flex items-center justify-center text-sm font-medium">
        {item.cutting_sequence || index + 1}
      </div>
      <div className="flex-1 min-w-0">
        <div className="font-medium text-sm truncate">{item.order_number}</div>
        <div className="text-xs text-muted-foreground truncate">
          {item.product_name}
        </div>
      </div>
      <div className="flex items-center gap-2">
        {item.color_code && (
          <Badge
            variant="secondary"
            className={colorBadgeColors[item.color_code] || ""}
          >
            {item.color_code}
          </Badge>
        )}
        {item.cutting_priority && item.cutting_priority !== "normal" && (
          <Badge className={priorityColors[item.cutting_priority]}>
            {priorityLabels[item.cutting_priority]}
          </Badge>
        )}
        {item.transport_batch_number && (
          <Badge variant="outline" className="text-xs">
            T:{item.transport_batch_number}
          </Badge>
        )}
      </div>
    </div>
  );
}
