import { useMemo } from "react";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { Plus, Package, AlertCircle, Image as ImageIcon, Edit, Warehouse } from "lucide-react";
import { cn } from "@/lib/utils";

interface CatalogProduct {
  id: number;
  sku: string;
  title: string;
  color: string | null;
  color_options: string[] | null;
  length: string | null;
  width: string | null;
  height: string | null;
  product_type: string | null;
  doors: string | null;
  legs: string | null;
  created_at: string;
  image_url: string | null;
  bom_component_count: number;
  product_group_id: number | null;
  product_group_name: string | null;
  product_group_color_hex: string | null;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_current_plan: boolean;
  warehouse_total_qty: number | null;
  warehouse_reserved_qty: number | null;
}

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  shortName: string | null;
  color: string | null;
  isActive: boolean;
}

interface CatalogProductsListProps {
  products: CatalogProduct[];
  colors: DictionaryItem[];
  onAddProduct: (product: CatalogProduct) => void;
  isAddingProduct?: boolean;
  recentlyAddedProducts?: Set<number>;
  isLocked?: boolean;
}

export function CatalogProductsList({ 
  products, 
  colors, 
  onAddProduct, 
  isAddingProduct = false,
  recentlyAddedProducts = new Set(),
  isLocked = false
}: CatalogProductsListProps) {
  const [, setLocation] = useLocation();

  const getColorHex = (colorCode: string | null): string | null => {
    if (!colorCode) return null;
    const colorItem = colors.find(c => c.code.toLowerCase() === colorCode.toLowerCase());
    return colorItem?.color || null;
  };

  const getTypeBackgroundColor = (type: string | null): string => {
    if (!type) return "#666666";
    const hash = type.split("").reduce((acc, char) => char.charCodeAt(0) + ((acc << 5) - acc), 0);
    const hue = Math.abs(hash) % 360;
    return `hsl(${hue}, 60%, 45%)`;
  };

  const getTextColor = (bgColor: string): string => {
    const hslMatch = bgColor.match(/hsl\((\d+),\s*(\d+)%,\s*(\d+)%\)/);
    if (hslMatch) {
      const lightness = parseInt(hslMatch[3]);
      return lightness > 50 ? "#000000" : "#FFFFFF";
    }
    return "#FFFFFF";
  };

  const handleEditProduct = (productId: number) => {
    setLocation(`/catalog/products/${productId}/edit`);
  };

  if (products.length === 0) {
    return (
      <div className="flex flex-col items-center justify-center py-12 text-center" data-testid="empty-catalog-products">
        <Package className="h-12 w-12 text-muted-foreground mb-3" />
        <p className="text-sm text-muted-foreground">
          Brak produktów katalogowych
        </p>
      </div>
    );
  }

  return (
    <div className="w-full overflow-x-auto" data-testid="catalog-products-list">
      <TooltipProvider>
        <Table className="w-full">
          <TableHeader className="sticky top-0 bg-background z-10 border-b">
            <TableRow className="h-8">
              <TableHead className="text-xs font-semibold py-1 w-[40px]">Zdj.</TableHead>
              <TableHead className="text-xs font-semibold py-1">SKU</TableHead>
              <TableHead className="text-xs font-semibold py-1">Typ</TableHead>
              <TableHead className="text-xs font-semibold py-1 w-[80px]">D×S×W</TableHead>
              <TableHead className="text-xs font-semibold py-1">Kolor</TableHead>
              <TableHead className="text-xs font-semibold py-1">Opcje koloru</TableHead>
              <TableHead className="text-xs font-semibold py-1 w-[70px]">Drzwi</TableHead>
              <TableHead className="text-xs font-semibold py-1 w-[70px]">Nogi</TableHead>
              <TableHead className="text-xs font-semibold py-1 w-[60px]">Mag.</TableHead>
              <TableHead className="text-xs font-semibold py-1 w-[50px]">Akcje</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {products.map((product) => {
              const colorHex = getColorHex(product.color);
              const isRecentlyAdded = recentlyAddedProducts.has(product.id);
              const bgColor = getTypeBackgroundColor(product.product_type);
              const textColorClass = getTextColor(bgColor) === "#000000" ? "text-black" : "text-white";
              
              return (
                <TableRow 
                  key={product.id}
                  className={cn(
                    "h-10 bg-muted/20 text-muted-foreground hover:bg-muted/30 transition-colors",
                    isRecentlyAdded && "bg-green-500/10",
                    product.in_current_plan && "opacity-50"
                  )}
                  data-testid={`catalog-product-${product.id}`}
                >
                  {/* Product Image */}
                  <TableCell className="py-1">
                    <div className="relative w-10 h-10 flex-shrink-0 group">
                      <div className="w-full h-full bg-muted/50 rounded flex items-center justify-center overflow-hidden">
                        {product.image_url ? (
                          <img
                            src={product.image_url}
                            alt={product.title}
                            className="h-full w-full object-cover opacity-70"
                            data-testid={`img-catalog-product-${product.id}`}
                          />
                        ) : (
                          <ImageIcon className="h-4 w-4 text-muted-foreground/50" />
                        )}
                      </div>
                      {/* Edit Icon Overlay */}
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          handleEditProduct(product.id);
                        }}
                        className="absolute inset-0 bg-black/60 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity rounded"
                        title="Edytuj produkt"
                        data-testid={`button-edit-product-${product.id}`}
                      >
                        <Edit className="h-4 w-4 text-white" />
                      </button>
                    </div>
                  </TableCell>

                  {/* SKU */}
                  <TableCell className="py-1 font-mono text-xs" data-testid={`text-sku-${product.id}`}>
                    {product.sku}
                  </TableCell>

                  {/* Product Type */}
                  <TableCell className="py-1">
                    {product.product_type ? (
                      <Badge
                        variant="default"
                        className={`text-xs h-5 ${textColorClass} border-transparent`}
                        style={{ backgroundColor: bgColor }}
                        data-testid={`badge-type-${product.id}`}
                      >
                        {product.product_type}
                      </Badge>
                    ) : (
                      <span className="text-muted-foreground">-</span>
                    )}
                  </TableCell>

                  {/* Dimensions */}
                  <TableCell className="py-1 text-xs text-center" data-testid={`text-dimensions-${product.id}`}>
                    {product.length && product.width && product.height ? (
                      <span className="font-mono">
                        {product.length}×{product.width}×{product.height}
                      </span>
                    ) : (
                      "-"
                    )}
                  </TableCell>

                  {/* Color - Tile with background */}
                  <TableCell className="py-1 text-xs" data-testid={`text-color-${product.id}`}>
                    {product.color ? (
                      <div
                        className="px-2 py-1 rounded text-center text-[10px] font-medium border"
                        style={{
                          backgroundColor: colorHex || '#cccccc',
                          color: colorHex ? (getTextColor(`hsl(0, 0%, ${parseInt(colorHex.slice(1, 3), 16) / 255 * 100}%)`) === "#000000" ? '#000' : '#fff') : '#000',
                          borderColor: colorHex || '#999'
                        }}
                        title={product.color}
                      >
                        {product.color}
                      </div>
                    ) : (
                      <span className="text-muted-foreground">-</span>
                    )}
                  </TableCell>

                  {/* Color Options - Tiles with background */}
                  <TableCell className="py-1" data-testid={`text-color-options-${product.id}`}>
                    {product.color_options && product.color_options.length > 0 ? (
                      <div className="flex flex-wrap gap-1">
                        {product.color_options.map((option, idx) => {
                          const optionHex = getColorHex(option);
                          const optionTextColor = optionHex ? 
                            (getTextColor(`hsl(0, 0%, ${parseInt(optionHex.slice(1, 3), 16) / 255 * 100}%)`) === "#000000" ? '#000' : '#fff') 
                            : '#000';
                          return (
                            <div
                              key={idx}
                              className="px-2 py-0.5 rounded text-center text-[10px] font-medium border"
                              style={{
                                backgroundColor: optionHex || '#e0e0e0',
                                color: optionTextColor,
                                borderColor: optionHex || '#999'
                              }}
                              title={option}
                            >
                              {option}
                            </div>
                          );
                        })}
                      </div>
                    ) : (
                      <span className="text-muted-foreground text-xs">-</span>
                    )}
                  </TableCell>

                  {/* Doors */}
                  <TableCell className="py-1 text-xs" data-testid={`text-doors-${product.id}`}>
                    {product.doors ? (
                      <Badge variant="outline" className="h-5 px-2 text-[10px]">
                        {product.doors}
                      </Badge>
                    ) : (
                      <span className="text-muted-foreground">-</span>
                    )}
                  </TableCell>

                  {/* Legs */}
                  <TableCell className="py-1 text-xs" data-testid={`text-legs-${product.id}`}>
                    {product.legs ? (
                      <Badge variant="outline" className="h-5 px-2 text-[10px]">
                        {product.legs}
                      </Badge>
                    ) : (
                      <span className="text-muted-foreground">-</span>
                    )}
                  </TableCell>

                  {/* Warehouse Stock */}
                  <TableCell className="py-1 text-xs" data-testid={`text-warehouse-${product.id}`}>
                    {product.warehouse_total_qty !== null ? (
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <div className="flex items-center gap-1">
                            <Warehouse className={cn(
                              "h-3 w-3",
                              (product.warehouse_total_qty - (product.warehouse_reserved_qty || 0)) > 0 
                                ? "text-green-500" 
                                : "text-red-500"
                            )} />
                            <span className={cn(
                              "font-mono text-[10px]",
                              (product.warehouse_total_qty - (product.warehouse_reserved_qty || 0)) > 0 
                                ? "text-green-500" 
                                : "text-red-500"
                            )}>
                              {product.warehouse_total_qty - (product.warehouse_reserved_qty || 0)}
                            </span>
                          </div>
                        </TooltipTrigger>
                        <TooltipContent>
                          <p className="text-xs">
                            Dostępne: {product.warehouse_total_qty - (product.warehouse_reserved_qty || 0)} szt.<br/>
                            Razem: {product.warehouse_total_qty} szt.<br/>
                            Zarezerwowane: {product.warehouse_reserved_qty || 0} szt.
                          </p>
                        </TooltipContent>
                      </Tooltip>
                    ) : (
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <span className="text-muted-foreground">-</span>
                        </TooltipTrigger>
                        <TooltipContent>
                          <p className="text-xs">Brak powiązania z magazynem</p>
                        </TooltipContent>
                      </Tooltip>
                    )}
                  </TableCell>

                  {/* Actions */}
                  <TableCell className="py-1">
                    {product.in_current_plan ? (
                      <Badge variant="secondary" className="h-6 text-[10px]">
                        W planie
                      </Badge>
                    ) : product.is_in_plan ? (
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <Button
                            size="sm"
                            variant="ghost"
                            onClick={() => onAddProduct(product)}
                            disabled={isAddingProduct || isLocked}
                            className="h-6 px-2 text-xs"
                            data-testid={`button-add-product-${product.id}`}
                          >
                            <Plus className="h-3 w-3 mr-1" />
                            Dodaj
                          </Button>
                        </TooltipTrigger>
                        <TooltipContent>
                          <p className="text-xs">Już w planie {product.in_plan_number}</p>
                        </TooltipContent>
                      </Tooltip>
                    ) : (
                      <Button
                        size="sm"
                        variant="ghost"
                        onClick={() => onAddProduct(product)}
                        disabled={isAddingProduct || isLocked}
                        className="h-6 px-2 text-xs"
                        data-testid={`button-add-product-${product.id}`}
                      >
                        <Plus className="h-3 w-3 mr-1" />
                        Dodaj
                      </Button>
                    )}
                  </TableCell>
                </TableRow>
              );
            })}
          </TableBody>
        </Table>
      </TooltipProvider>
    </div>
  );
}
