import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Link, useLocation, useRoute } from "wouter";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Input } from "@/components/ui/input";
import { Loader2, Layers, PlusSquare, Search, X, Copy } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Label } from "@/components/ui/label";
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip";

interface FurnitureType {
  furniture_type: string;
  template_count: number;
}

interface MatrixSubmenuProps {
  furnitureType: string | null;
}

export function MatrixSubmenu({ furnitureType }: MatrixSubmenuProps) {
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [filterQuery, setFilterQueryState] = useState(() => {
    try {
      return localStorage.getItem('matrix-filterQuery') || "";
    } catch {
      return "";
    }
  });
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [newFurnitureType, setNewFurnitureType] = useState("");
  const [duplicateSource, setDuplicateSource] = useState<string | null>(null);
  const [duplicateTarget, setDuplicateTarget] = useState("");

  // Helper to update filterQuery with localStorage persistence
  const setFilterQuery = (value: string) => {
    setFilterQueryState(value);
    try {
      localStorage.setItem('matrix-filterQuery', value);
    } catch (e) {
      console.error('Failed to save filterQuery:', e);
    }
  };

  const { data: furnitureTypes, isLoading } = useQuery<FurnitureType[]>({
    queryKey: ["/api/component-templates/furniture-types"],
  });

  // Filter furniture types based on search query
  const filteredTypes = furnitureTypes?.filter((ft) =>
    ft.furniture_type.toLowerCase().includes(filterQuery.toLowerCase())
  );

  // Mutation to duplicate furniture type templates
  const duplicateMutation = useMutation({
    mutationFn: async ({ source, target }: { source: string; target: string }) => {
      return apiRequest('POST', '/api/component-templates/duplicate-furniture-type', { 
        sourceFurnitureType: source, 
        targetFurnitureType: target 
      });
    },
    onSuccess: (data: any) => {
      toast({ 
        title: "Skopiowano rodzaj mebla", 
        description: `${data.duplicated} formatek skopiowano do ${duplicateTarget}` 
      });
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0]?.toString() || '';
          return key.startsWith('/api/component-templates');
        }
      });
      setDuplicateSource(null);
      setDuplicateTarget("");
      navigate(`/matrix/${duplicateTarget}`);
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd kopiowania", 
        description: error.message, 
        variant: "destructive" 
      });
    },
  });

  const handleAddFurnitureType = () => {
    // Just navigate to matrix page with new furniture type
    // User will add rows manually
    if (newFurnitureType.trim()) {
      navigate(`/matrix/${newFurnitureType.trim()}`);
      setIsAddDialogOpen(false);
      setNewFurnitureType("");
      // Refresh furniture types list after adding first template
      setTimeout(() => {
        queryClient.invalidateQueries({ queryKey: ["/api/component-templates/furniture-types"] });
      }, 1000);
      toast({ title: "Przejdź do nowego rodzaju", description: "Dodaj formatki używając przycisku 'Dodaj wiersz'" });
    }
  };

  const handleDuplicate = (source: string) => {
    setDuplicateSource(source);
    setDuplicateTarget(`${source}-KOPIA`);
  };

  const confirmDuplicate = () => {
    if (duplicateSource && duplicateTarget.trim()) {
      duplicateMutation.mutate({ source: duplicateSource, target: duplicateTarget.trim() });
    }
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-32">
        <Loader2 className="w-6 h-6 animate-spin text-muted-foreground" />
      </div>
    );
  }

  return (
    <div className="w-full p-4">
      <div className="mb-4">
        <h3 className="text-lg font-semibold mb-1">Matryca Formatek</h3>
        <p className="text-sm text-muted-foreground">Szablony komponentów według rodzaju mebla</p>
      </div>

      {/* Search and Add Button */}
      <div className="space-y-2 mb-4">
        <div className="relative">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
          <Input
            placeholder="Filtruj rodzaje..."
            value={filterQuery}
            onChange={(e) => setFilterQuery(e.target.value)}
            className="pl-10 pr-10"
            data-testid="input-filter-furniture-types"
          />
          {filterQuery && (
            <Button
              variant="ghost"
              size="icon"
              className="absolute right-1 top-1/2 -translate-y-1/2 h-7 w-7"
              onClick={() => setFilterQuery("")}
              data-testid="button-clear-filter"
            >
              <X className="w-4 h-4" />
            </Button>
          )}
        </div>

        <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
          <DialogTrigger asChild>
            <Button
              variant="outline"
              className="w-full"
              data-testid="button-add-furniture-type"
            >
              <PlusSquare className="w-4 h-4 mr-2" />
              Dodaj nowy rodzaj
            </Button>
          </DialogTrigger>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Dodaj nowy rodzaj mebla</DialogTitle>
              <DialogDescription>
                Wprowadź nazwę nowego rodzaju mebla (np. VB, ALTUS, SUPRA)
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4 py-4">
              <div className="space-y-2">
                <Label htmlFor="furniture-type">Nazwa rodzaju</Label>
                <Input
                  id="furniture-type"
                  placeholder="np. VB-PREMIUM"
                  value={newFurnitureType}
                  onChange={(e) => setNewFurnitureType(e.target.value.toUpperCase())}
                  data-testid="input-new-furniture-type"
                />
              </div>
            </div>
            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => setIsAddDialogOpen(false)}
                data-testid="button-cancel-add"
              >
                Anuluj
              </Button>
              <Button
                onClick={handleAddFurnitureType}
                disabled={!newFurnitureType.trim()}
                data-testid="button-confirm-add"
              >
                Dodaj
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>

      <ScrollArea className="h-[calc(100vh-320px)]">
        <div className="space-y-1">
          <Link href="/matrix">
            <Button
              variant={!furnitureType ? "secondary" : "ghost"}
              className="w-full justify-start min-w-0"
              data-testid="link-matrix-all"
            >
              <Layers className="w-4 h-4 mr-2 shrink-0" />
              <span className="truncate">Wszystkie</span>
              {furnitureTypes && furnitureTypes.length > 0 && (
                <Badge variant="outline" className="ml-auto shrink-0 text-xs">
                  {furnitureTypes.reduce((sum: number, ft) => sum + (Number(ft.template_count) || 0), 0)}
                </Badge>
              )}
            </Button>
          </Link>

          <div className="h-px bg-border my-2" />

          {filteredTypes && filteredTypes.length === 0 ? (
            <div className="text-center py-8 text-sm text-muted-foreground">
              Brak rodzajów pasujących do filtra
            </div>
          ) : (
            <TooltipProvider>
              {filteredTypes?.map((ft) => (
                <div key={ft.furniture_type} className="flex items-center gap-0.5">
                  <Link href={`/matrix/${ft.furniture_type}`} className="flex-1 min-w-0">
                    <Button
                      variant={furnitureType === ft.furniture_type ? "secondary" : "ghost"}
                      className="w-full justify-start text-left min-w-0"
                      data-testid={`link-matrix-${ft.furniture_type}`}
                    >
                      <span className="truncate flex-1">{ft.furniture_type}</span>
                      <Badge variant="outline" className="ml-2 shrink-0 text-xs">
                        {ft.template_count}
                      </Badge>
                    </Button>
                  </Link>
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <Button
                        variant="ghost"
                        size="icon"
                        className="h-7 w-7 shrink-0"
                        onClick={() => handleDuplicate(ft.furniture_type)}
                        data-testid={`button-duplicate-${ft.furniture_type}`}
                      >
                        <Copy className="w-3 h-3" />
                      </Button>
                    </TooltipTrigger>
                    <TooltipContent>Duplikuj rodzaj z danymi</TooltipContent>
                  </Tooltip>
                </div>
              ))}
            </TooltipProvider>
          )}
        </div>
      </ScrollArea>

      {/* Duplicate Dialog */}
      <AlertDialog open={!!duplicateSource} onOpenChange={(open) => !open && setDuplicateSource(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Duplikuj rodzaj mebla</AlertDialogTitle>
            <AlertDialogDescription>
              Skopiuj wszystkie formatki z <strong>{duplicateSource}</strong> do nowego rodzaju
            </AlertDialogDescription>
          </AlertDialogHeader>
          <div className="py-4">
            <Label htmlFor="target-type">Nazwa nowego rodzaju</Label>
            <Input
              id="target-type"
              value={duplicateTarget}
              onChange={(e) => setDuplicateTarget(e.target.value.toUpperCase())}
              placeholder="np. VB-PREMIUM"
              className="mt-2"
              data-testid="input-duplicate-target"
            />
          </div>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-duplicate">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={confirmDuplicate}
              disabled={!duplicateTarget.trim() || duplicateMutation.isPending}
              data-testid="button-confirm-duplicate"
            >
              {duplicateMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Kopiowanie...
                </>
              ) : (
                'Duplikuj'
              )}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
