import { createContext, useContext, useEffect, useState, useCallback, ReactNode } from 'react';
import { useLocation } from 'wouter';

interface NavigationContextType {
  goBack: (fallbackPath: string, navigate: (path: string) => void) => void;
  canGoBack: () => boolean;
}

const NavigationContext = createContext<NavigationContextType | null>(null);

const STORAGE_KEY = 'navigation_stack';
const MAX_STACK_SIZE = 50;

export function NavigationProvider({ children }: { children: ReactNode }) {
  const [location] = useLocation();
  const [stack, setStack] = useState<string[]>(() => {
    // Initialize from sessionStorage if available
    if (typeof window !== 'undefined') {
      try {
        const stored = sessionStorage.getItem(STORAGE_KEY);
        return stored ? JSON.parse(stored) : [];
      } catch {
        return [];
      }
    }
    return [];
  });

  // Track location changes and update stack
  useEffect(() => {
    setStack((prevStack) => {
      // Don't add if it's the same as the last entry (prevents duplicates on re-renders)
      if (prevStack.length > 0 && prevStack[prevStack.length - 1] === location) {
        return prevStack;
      }

      // Add current location to stack
      const newStack = [...prevStack, location];

      // Cap stack size
      const cappedStack = newStack.slice(-MAX_STACK_SIZE);

      // Persist to sessionStorage
      if (typeof window !== 'undefined') {
        try {
          sessionStorage.setItem(STORAGE_KEY, JSON.stringify(cappedStack));
        } catch {
          // Ignore storage errors
        }
      }

      return cappedStack;
    });
  }, [location]);

  const canGoBack = useCallback(() => {
    return stack.length > 1;
  }, [stack]);

  const goBack = useCallback((fallbackPath: string, navigate: (path: string) => void) => {
    setStack((prevStack) => {
      if (prevStack.length > 1) {
        // Pop current location and get previous
        const newStack = prevStack.slice(0, -1);
        const previousPath = newStack[newStack.length - 1];

        // Persist new stack
        if (typeof window !== 'undefined') {
          try {
            sessionStorage.setItem(STORAGE_KEY, JSON.stringify(newStack));
          } catch {
            // Ignore storage errors
          }
        }

        // Navigate to previous path
        navigate(previousPath);

        return newStack;
      } else {
        // No history, use fallback
        navigate(fallbackPath);
        return prevStack;
      }
    });
  }, []);

  return (
    <NavigationContext.Provider value={{ goBack, canGoBack }}>
      {children}
    </NavigationContext.Provider>
  );
}

export function useNavigation() {
  const context = useContext(NavigationContext);
  if (!context) {
    throw new Error('useNavigation must be used within NavigationProvider');
  }
  return context;
}
