import { useEffect, useRef, useState, useCallback } from 'react';
import { useLocation } from 'wouter';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Progress } from '@/components/ui/progress';
import { CheckCircle2, XCircle, Loader2, Factory, AlertTriangle, Route, GitBranch, Palette, ExternalLink } from 'lucide-react';
import { ScrollArea } from '@/components/ui/scroll-area';

interface ProgressEntry {
  timestamp: string;
  message: string;
  type: 'info' | 'success' | 'error' | 'warning';
  step: string;
  current: number;
  total: number;
  result?: any;
}

interface ZlpGenerationProgressDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  sessionId: string | null;
  planId?: string | number;
  onComplete?: (result: any) => void;
}

export function ZlpGenerationProgressDialog({
  open,
  onOpenChange,
  sessionId,
  planId,
  onComplete,
}: ZlpGenerationProgressDialogProps) {
  const [, setLocation] = useLocation();
  const [logs, setLogs] = useState<ProgressEntry[]>([]);
  const [isConnected, setIsConnected] = useState(false);
  const [progress, setProgress] = useState(0);
  const [currentStep, setCurrentStep] = useState('');
  const [isComplete, setIsComplete] = useState(false);
  const [hasError, setHasError] = useState(false);
  const [result, setResult] = useState<any>(null);
  const scrollRef = useRef<HTMLDivElement>(null);
  const eventSourceRef = useRef<EventSource | null>(null);

  useEffect(() => {
    if (scrollRef.current) {
      scrollRef.current.scrollTop = scrollRef.current.scrollHeight;
    }
  }, [logs]);

  // Track completion state in refs for use in callbacks
  const isCompleteRef = useRef(false);
  const isConnectedRef = useRef(false);
  const pollingIntervalRef = useRef<NodeJS.Timeout | null>(null);
  
  useEffect(() => {
    isCompleteRef.current = isComplete;
  }, [isComplete]);
  
  useEffect(() => {
    isConnectedRef.current = isConnected;
  }, [isConnected]);

  // Polling fallback when SSE fails
  const startPolling = useCallback((sid: string) => {
    console.log('[ZLP Progress] Starting polling fallback');
    setCurrentStep('Sprawdzanie statusu...');
    
    const poll = async () => {
      if (isCompleteRef.current) return;
      
      try {
        const response = await fetch(`/api/production/planning/zlp-status/${sid}`, {
          credentials: 'include',
        });
        
        if (!response.ok) {
          console.error('[ZLP Progress] Polling error:', response.status);
          return;
        }
        
        const data = await response.json();
        console.log('[ZLP Progress] Poll result:', data.status);
        
        if (data.status === 'completed' || data.status === 'failed') {
          isCompleteRef.current = true;
          setIsComplete(true);
          setProgress(100);
          
          if (data.status === 'completed' && data.result) {
            setResult(data.result);
            setHasError(!data.result.success);
            setCurrentStep(data.result.success 
              ? `Wygenerowano ${data.result.summary?.totalOrders || 0} zleceń ZLP`
              : `Błędy: ${data.result.errors?.join(', ') || 'Nieznany błąd'}`);
            if (onComplete && data.result) {
              onComplete(data.result);
            }
          } else if (data.status === 'failed') {
            setHasError(true);
            setCurrentStep(`Błąd: ${data.error || 'Nieznany błąd'}`);
          }
          
          if (pollingIntervalRef.current) {
            clearInterval(pollingIntervalRef.current);
            pollingIntervalRef.current = null;
          }
        } else if (data.status === 'running') {
          setCurrentStep('Generowanie w toku...');
          setProgress(50); // Show some progress while running
        }
      } catch (error) {
        console.error('[ZLP Progress] Polling fetch error:', error);
      }
    };

    // Poll immediately, then every 2 seconds
    poll();
    pollingIntervalRef.current = setInterval(poll, 2000);
  }, [onComplete]);

  useEffect(() => {
    if (!open || !sessionId) return;

    console.log('[ZLP Progress] Connecting to SSE stream:', sessionId);
    setIsComplete(false);
    setHasError(false);
    setProgress(0);
    setLogs([]);
    setResult(null);
    setIsConnected(false);
    isCompleteRef.current = false;
    isConnectedRef.current = false;
    
    const eventSource = new EventSource(`/api/production/planning/zlp-progress/${sessionId}`);
    eventSourceRef.current = eventSource;

    // Fallback to polling if SSE doesn't connect within 5 seconds
    const connectionTimeout = setTimeout(() => {
      if (!isConnectedRef.current && !isCompleteRef.current) {
        console.warn('[ZLP Progress] SSE connection timeout, falling back to polling');
        eventSource.close();
        startPolling(sessionId);
      }
    }, 5000);

    eventSource.onopen = () => {
      console.log('[ZLP Progress] SSE connection established');
      setIsConnected(true);
      isConnectedRef.current = true;
      clearTimeout(connectionTimeout);
    };

    eventSource.onmessage = (event) => {
      try {
        const data: ProgressEntry = JSON.parse(event.data);
        setLogs((prev) => [...prev, data]);
        setProgress(data.current);
        setCurrentStep(data.message);
        
        if (data.step === 'done') {
          isCompleteRef.current = true;
          setIsComplete(true);
          setIsConnected(false);
          isConnectedRef.current = false;
          setResult(data.result);
          setHasError(!data.result?.success);
          if (onComplete && data.result) {
            onComplete(data.result);
          }
          eventSource.close();
        } else if (data.step === 'error') {
          isCompleteRef.current = true;
          setHasError(true);
          setIsComplete(true);
          setIsConnected(false);
          isConnectedRef.current = false;
          eventSource.close();
        }
      } catch (error) {
        console.error('[ZLP Progress] Error parsing data:', error);
      }
    };

    eventSource.onerror = () => {
      console.warn('[ZLP Progress] SSE error, falling back to polling');
      eventSource.close();
      
      if (!isCompleteRef.current) {
        setIsConnected(false);
        isConnectedRef.current = false;
        // Fall back to polling instead of showing error immediately
        startPolling(sessionId);
      }
    };

    return () => {
      console.log('[ZLP Progress] Closing SSE connection');
      clearTimeout(connectionTimeout);
      eventSource.close();
      if (pollingIntervalRef.current) {
        clearInterval(pollingIntervalRef.current);
        pollingIntervalRef.current = null;
      }
      setIsConnected(false);
      isConnectedRef.current = false;
    };
  }, [open, sessionId, onComplete, startPolling]);

  useEffect(() => {
    if (!open) {
      setLogs([]);
      setProgress(0);
      setCurrentStep('');
      setIsComplete(false);
      setHasError(false);
      setResult(null);
    }
  }, [open]);

  const getLogColor = (type: ProgressEntry['type']) => {
    switch (type) {
      case 'success':
        return 'text-green-600 dark:text-green-400';
      case 'error':
        return 'text-red-600 dark:text-red-400';
      case 'warning':
        return 'text-yellow-600 dark:text-yellow-400';
      default:
        return 'text-foreground/80';
    }
  };

  const getLogIcon = (type: ProgressEntry['type']) => {
    switch (type) {
      case 'success':
        return '✓';
      case 'error':
        return '✗';
      case 'warning':
        return '!';
      default:
        return '→';
    }
  };

  const formatTimestamp = (timestamp: string) => {
    const date = new Date(timestamp);
    return date.toLocaleTimeString('pl-PL', {
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit',
    });
  };

  const handleClose = () => {
    if (isComplete || hasError) {
      onOpenChange(false);
    }
  };

  return (
    <Dialog open={open} onOpenChange={handleClose}>
      <DialogContent 
        className="max-w-2xl flex flex-col" 
        data-testid="dialog-zlp-progress"
        onPointerDownOutside={(e) => {
          if (!isComplete && !hasError) e.preventDefault();
        }}
        onEscapeKeyDown={(e) => {
          if (!isComplete && !hasError) e.preventDefault();
        }}
      >
        <DialogHeader className="flex-shrink-0">
          <div className="flex items-center gap-3">
            {isComplete ? (
              hasError ? (
                <XCircle className="h-6 w-6 text-red-500" />
              ) : (
                <CheckCircle2 className="h-6 w-6 text-green-500" />
              )
            ) : (
              <Factory className="h-6 w-6 text-primary animate-pulse" />
            )}
            <div>
              <DialogTitle className="text-lg font-semibold" data-testid="text-zlp-title">
                {isComplete 
                  ? hasError 
                    ? 'Generowanie nie powiodło się' 
                    : 'Generowanie zakończone'
                  : 'Generowanie zleceń ZLP'
                }
              </DialogTitle>
              <DialogDescription data-testid="text-zlp-description">
                {isComplete 
                  ? hasError
                    ? 'Wystąpił błąd podczas generowania zleceń produkcyjnych'
                    : result?.summary 
                      ? `Wygenerowano ${result.summary.totalOrders} zleceń z ${result.summary.totalComponents} komponentami`
                      : 'Proces zakończony'
                  : 'Proszę czekać, trwa przetwarzanie planu produkcyjnego...'
                }
              </DialogDescription>
            </div>
          </div>
        </DialogHeader>

        <div className="space-y-4 py-4">
          <div className="space-y-2">
            <div className="flex items-center justify-between text-sm">
              <span className="text-muted-foreground">Postęp</span>
              <span className="font-medium" data-testid="text-progress-percent">{progress}%</span>
            </div>
            <Progress value={progress} className="h-2" data-testid="progress-bar" />
            {currentStep && !isComplete && (
              <p className="text-sm text-muted-foreground flex items-center gap-2" data-testid="text-current-step">
                <Loader2 className="h-3 w-3 animate-spin" />
                {currentStep}
              </p>
            )}
          </div>

          <ScrollArea className="h-[200px] rounded-md border bg-muted/30 p-3">
            <div ref={scrollRef} className="space-y-1.5" data-testid="scroll-logs">
              {logs.length === 0 ? (
                <div className="text-center text-muted-foreground py-4 text-sm" data-testid="text-waiting">
                  Oczekiwanie na połączenie...
                </div>
              ) : (
                logs.map((log, index) => (
                  <div
                    key={index}
                    className="flex items-start gap-2 text-xs font-mono"
                    data-testid={`log-entry-${index}`}
                  >
                    <span className="text-muted-foreground flex-shrink-0 w-16">
                      {formatTimestamp(log.timestamp)}
                    </span>
                    <span className={`flex-shrink-0 w-4 text-center ${getLogColor(log.type)}`}>
                      {getLogIcon(log.type)}
                    </span>
                    <span className={getLogColor(log.type)}>{log.message}</span>
                  </div>
                ))
              )}
            </div>
          </ScrollArea>

          {result?.summary && isComplete && !hasError && (
            <div className="rounded-md border bg-green-500/10 border-green-500/30 p-3">
              <h4 className="text-sm font-medium text-green-700 dark:text-green-400 mb-2">
                Podsumowanie
              </h4>
              <div className="grid grid-cols-2 gap-2 text-sm">
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Zlecenia ZLP:</span>
                  <span className="font-medium" data-testid="text-total-orders">{result.summary.totalOrders}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Komponenty:</span>
                  <span className="font-medium" data-testid="text-total-components">{result.summary.totalComponents}</span>
                </div>
              </div>
              {result.summary.colorBreakdown && Object.keys(result.summary.colorBreakdown).length > 0 && (
                <div className="mt-2 pt-2 border-t border-green-500/20">
                  <span className="text-xs text-muted-foreground">Podział wg koloru:</span>
                  <div className="flex flex-wrap gap-1 mt-1">
                    {Object.entries(result.summary.colorBreakdown).map(([color, count]) => (
                      <span 
                        key={color} 
                        className="text-xs bg-muted px-2 py-0.5 rounded"
                        data-testid={`badge-color-${color}`}
                      >
                        {color}: {count as number}
                      </span>
                    ))}
                  </div>
                </div>
              )}
              
              {result.generatedOrders && result.generatedOrders.length > 0 && (
                <div className="mt-3 pt-3 border-t border-green-500/20">
                  <span className="text-xs text-muted-foreground mb-2 block">Przejdź do zlecenia:</span>
                  <div className="flex flex-wrap gap-2">
                    {result.generatedOrders.map((order: { orderId: number; orderNumber: string; colorCode: string; componentCount: number }) => (
                      <Button
                        key={order.orderId}
                        variant="outline"
                        size="sm"
                        className="text-xs h-7"
                        onClick={() => {
                          onOpenChange(false);
                          setLocation(`/production/orders/${order.orderId}`);
                        }}
                        data-testid={`button-goto-zlp-${order.colorCode}`}
                      >
                        <Palette className="h-3 w-3 mr-1" />
                        {order.colorCode}
                        <span className="ml-1 text-muted-foreground">({order.componentCount})</span>
                        <ExternalLink className="h-2.5 w-2.5 ml-1" />
                      </Button>
                    ))}
                  </div>
                </div>
              )}
            </div>
          )}

          {result?.errors && result.errors.length > 0 && (
            <div className="rounded-md border bg-red-500/10 border-red-500/30 p-3">
              <h4 className="text-sm font-medium text-red-700 dark:text-red-400 mb-2">
                Błędy ({result.errors.length})
              </h4>
              <ul className="text-xs text-red-600 dark:text-red-400 space-y-1">
                {result.errors.slice(0, 5).map((error: string, index: number) => (
                  <li key={index} data-testid={`error-${index}`}>• {error}</li>
                ))}
                {result.errors.length > 5 && (
                  <li className="text-muted-foreground">... i {result.errors.length - 5} więcej</li>
                )}
              </ul>
            </div>
          )}
        </div>

        <div className="flex justify-between items-center gap-2 pt-2 border-t">
          <div className="flex items-center gap-2">
            {planId && (isComplete || hasError) && (
              <>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => {
                    onOpenChange(false);
                    setLocation(`/production/plans/${planId}?panel=routing`);
                  }}
                  data-testid="button-goto-routing"
                >
                  <Route className="h-4 w-4 mr-1.5" />
                  Marszruty
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => {
                    onOpenChange(false);
                    setLocation(`/production/plans/${planId}/flow-tree`);
                  }}
                  data-testid="button-goto-flow-tree"
                >
                  <GitBranch className="h-4 w-4 mr-1.5" />
                  Drzewo przepływu
                </Button>
              </>
            )}
          </div>
          <Button
            onClick={handleClose}
            disabled={!isComplete && !hasError}
            data-testid="button-close-progress"
          >
            {isComplete || hasError ? 'Zamknij' : 'Przetwarzanie...'}
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
}
