import { useState, useEffect } from "react";
import { Link } from "wouter";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { 
  Dialog, 
  DialogContent, 
  DialogHeader, 
  DialogTitle, 
  DialogDescription,
  DialogFooter,
  DialogTrigger 
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Combobox } from "@/components/ui/combobox";
import { ScrollArea } from "@/components/ui/scroll-area";
import { useMutation, useQuery } from "@tanstack/react-query";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { 
  Play, 
  CheckCircle, 
  Clock, 
  Pause, 
  ChevronRight,
  ChevronDown,
  Factory, 
  Timer,
  User,
  Users,
  UserPlus,
  Loader2,
  AlertTriangle,
  XCircle,
  QrCode,
  Box,
  Boxes,
  Package,
  Plus,
  Eye,
  Layers,
  Warehouse,
  Pencil,
  Info,
  Trash2,
  Scissors,
  Grid3X3,
  Drill,
  Wrench,
  Armchair,
  Hammer,
  PackageCheck,
  Gift,
  Truck,
  CircleDot,
  Flag,
  ArrowRightCircle
} from "lucide-react";
import QRCode from 'qrcode';
import { format } from "date-fns";
import { pl } from "date-fns/locale";

// Live timer component for buffers - counts elapsed time in real-time
function LiveBufferTimer({ startTime }: { startTime: Date | null }) {
  const [elapsed, setElapsed] = useState(0);
  
  useEffect(() => {
    if (!startTime) {
      setElapsed(0);
      return;
    }
    
    const updateElapsed = () => {
      const now = Date.now();
      const start = new Date(startTime).getTime();
      setElapsed(Math.max(0, Math.floor((now - start) / 1000)));
    };
    
    updateElapsed();
    const interval = setInterval(updateElapsed, 1000);
    
    return () => clearInterval(interval);
  }, [startTime]);
  
  if (!startTime || elapsed === 0) return null;
  
  const hours = Math.floor(elapsed / 3600);
  const mins = Math.floor((elapsed % 3600) / 60);
  const secs = elapsed % 60;
  
  const timeStr = hours > 0 
    ? `${hours}h ${mins}m ${secs}s`
    : mins > 0 
      ? `${mins}m ${secs}s`
      : `${secs}s`;
  
  return (
    <div className="flex items-center gap-0.5 text-[9px] text-orange-600 dark:text-orange-400 font-mono bg-orange-100 dark:bg-orange-900/30 px-1.5 py-0.5 animate-pulse">
      <Clock className="h-2.5 w-2.5" />
      {timeStr}
    </div>
  );
}

interface Carrier {
  id: number;
  code: string;
  name: string;
  barcode: string | null;
  status: string;
  capacity: number | null;
  capacityUnit: string | null;
  currentLoad: number | null;
  groupName: string | null;
  groupCode: string | null;
}

interface PalletFlowData {
  buffers: Array<{
    workOrderId: number;
    workOrderNumber: string;
    sequence: number;
    status: string;
    operationCode: string;
    operationName: string;
    routingId: number;
    routingCode: string;
    bufferBeforeId: number | null;
    bufferBeforeCode: string | null;
    bufferBeforeName: string | null;
    bufferAfterId: number | null;
    bufferAfterCode: string | null;
    bufferAfterName: string | null;
    carriersInBufferBefore: Carrier[];
    carriersInBufferAfter: Carrier[];
  }>;
  carriers: Array<Carrier & {
    currentLocationId: number | null;
    locationName: string | null;
    locationCode: string | null;
  }>;
  availableCarriers: Array<{
    id: number;
    code: string;
    name: string;
    barcode: string | null;
    groupName: string | null;
    groupCode: string | null;
  }>;
  formatkiByPath: Record<string, number>;
}

interface Formatka {
  id: number;
  formatkaId: number;
  formatkaName: string;
  formatkaCode: string;
  width: number;
  length: number;
  thickness: number;
  colorCode: string;
  boardType: string;
  edgingTop: string;
  edgingBottom: string;
  edgingLeft: string;
  edgingRight: string;
  quantity: number;
  itemStatus: string;
  routingPath: string;
}

interface AssignedOperator {
  operatorId: number;
  operatorName: string;
  operatorCode: string | null;
  isPrimary: boolean;
}

interface OperationPallet {
  id: number;
  palletLabel: string;
  flowCode: string;
  carrierCode: string | null;
  carrierName: string | null;
  status: string;
  currentLoad: number;
  maxCapacity: number;
}

interface WorkOrder {
  id: number;
  workOrderNumber: string;
  sequence: number;
  status: string;
  quantityPlanned: string | number;
  quantityProduced: string | number;
  quantityScrap?: string | number;
  qualityCheckPassed?: boolean;
  qualityCheckNotes?: string;
  notes?: string;
  routingOperationId: number | null;
  operationCode: string | null;
  operationName: string | null;
  routingId?: number | null;
  routingCode?: string | null;
  routingName?: string | null;
  workCenterId: number | null;
  workCenterCode: string | null;
  workCenterName: string | null;
  operatorId?: number | null;
  operatorName?: string | null;
  operatorCode?: string | null;
  assignedOperators?: AssignedOperator[];
  scheduledStartTime: string | null;
  scheduledEndTime: string | null;
  actualStartTime: string | null;
  actualEndTime: string | null;
  bufferBeforeId?: number | null;
  bufferBeforeCode?: string | null;
  bufferBeforeName?: string | null;
  bufferAfterId?: number | null;
  bufferAfterCode?: string | null;
  bufferAfterName?: string | null;
  warehouseLocationId?: number | null;
  warehouseLocationCode?: string | null;
  warehouseLocationName?: string | null;
  estimatedDuration?: number | null;
  isWarehouseOperation?: boolean;
  damagedCount?: number;
  pallets?: OperationPallet[];
  carrierId?: number | null;
  carrierCode?: string | null;
  carrierName?: string | null;
}

interface RoutingPath {
  routingKey: string;
  routingId: number;
  routingCode: string;
  routingName: string;
  flowCode: string;
  color: string;
  textColor: string;
  accentColor: string;
  workOrders: WorkOrder[];
}

interface DamagedFormatka {
  id: number;
  generatedName: string;
  damageType?: string;
  damageNotes?: string;
}

interface WorkOrderFlowProps {
  workOrders: WorkOrder[];
  productionOrderId: number;
  orderStatus: string;
  damagedFormatki?: DamagedFormatka[];
  sourcePlanId?: number | null;
}

// Plan-level assembly types
interface PlanAssemblyWorkOrder {
  id: number;
  workOrderNumber: string;
  sequence: number;
  status: string;
  operationCode: string;
  operationName: string;
  zlpId: number;
  zlpOrderNumber: string;
  zlpColorCode: string;
  zlpStatus: string;
  quantityPlanned: number;
  quantityProduced: number;
  actualStartTime: string | null;
  actualEndTime: string | null;
}

interface PlanAssemblyOperationGroup {
  operationCode: string;
  operationName: string;
  sequence: number;
  workOrders: PlanAssemblyWorkOrder[];
  allDone: boolean;
  anyInProgress: boolean;
  allPrerequisitesDone: boolean;
  totalQuantityPlanned: number;
  totalQuantityProduced: number;
}

interface PlanZlpInfo {
  id: number;
  orderNumber: string;
  colorCode: string;
  status: string;
  quantityPlanned: number;
  allPreAssemblyDone: boolean;
}

interface PlanAssemblyOperations {
  planId: number;
  planNumber: string;
  planName: string;
  zlps: PlanZlpInfo[];
  operationGroups: PlanAssemblyOperationGroup[];
  canStartAssembly: boolean;
  canStartPacking: boolean;
}

const statusConfig: Record<string, { color: string; bgColor: string; label: string; icon: any }> = {
  pending: { 
    color: "text-gray-500", 
    bgColor: "bg-gray-100 dark:bg-gray-800", 
    label: "Oczekuje",
    icon: Clock 
  },
  ready: { 
    color: "text-blue-500", 
    bgColor: "bg-blue-100 dark:bg-blue-900/30", 
    label: "Gotowe",
    icon: Play 
  },
  in_progress: { 
    color: "text-yellow-500", 
    bgColor: "bg-yellow-100 dark:bg-yellow-900/30", 
    label: "W trakcie",
    icon: Loader2 
  },
  paused: { 
    color: "text-orange-500", 
    bgColor: "bg-orange-100 dark:bg-orange-900/30", 
    label: "Wstrzymane",
    icon: Pause 
  },
  done: { 
    color: "text-green-500", 
    bgColor: "bg-green-100 dark:bg-green-900/30", 
    label: "Zakończone",
    icon: CheckCircle 
  },
  cancelled: { 
    color: "text-red-500", 
    bgColor: "bg-red-100 dark:bg-red-900/30", 
    label: "Anulowane",
    icon: Clock 
  },
};

// Lucide icon components for operations
const operationIconComponents: Record<string, any> = {
  cutting: Scissors,
  edging: Grid3X3,
  drilling: Drill,
  drilling_holes: Drill,
  drilling_mount: Wrench,
  upholstering: Armchair,
  assembly: Hammer,
  packing: PackageCheck,
  strapping: Gift,
  magazynowanie: Warehouse,
  transport_samochodowy: Truck,
  default: CircleDot,
};

const getOperationIcon = (operationCode: string | null, isWarehouse: boolean = false): any => {
  if (isWarehouse) return Warehouse;
  if (!operationCode) return CircleDot;
  return operationIconComponents[operationCode.toLowerCase()] || CircleDot;
};

interface InsertWarehouseAfterProps {
  workOrderId: number;
  workOrderSequence: number;
  operationName: string;
  productionOrderId: number;
  onSuccess: () => void;
  nextOperationStatus?: string; // Status następnej operacji - jeśli "done", ukrywamy przycisk
  currentOperationStatus?: string; // Status bieżącej operacji - jeśli "done", ukrywamy przycisk
}

function InsertWarehouseAfterOperation({ workOrderId, workOrderSequence, operationName, productionOrderId, onSuccess, nextOperationStatus, currentOperationStatus }: InsertWarehouseAfterProps) {
  const { toast } = useToast();
  const [open, setOpen] = useState(false);
  const [selectedWarehouseId, setSelectedWarehouseId] = useState<string>("");
  const [estimatedDuration, setEstimatedDuration] = useState("60");
  const [notes, setNotes] = useState("");

  const { data: locations = [] } = useQuery<Array<{ id: number; name: string; code: string; allowsStorage?: boolean; status?: string }>>({
    queryKey: ['/api/production/locations'],
    enabled: open && currentOperationStatus !== 'done',
  });

  const warehouseLocations = locations.filter(l => 
    l.allowsStorage || 
    l.name?.toLowerCase().includes('magazyn') || 
    l.name?.toLowerCase().includes('bufor') ||
    l.name?.toLowerCase().includes('storage')
  );

  const insertMutation = useMutation({
    mutationFn: async (data: { afterWorkOrderId: number; warehouseId: number; warehouseName?: string; estimatedDuration?: number; notes?: string }) => {
      return apiRequest("POST", `/api/production/orders/${productionOrderId}/insert-warehouse-operation`, data);
    },
    onSuccess: () => {
      toast({ title: "Dodano operację magazynowania" });
      setOpen(false);
      setSelectedWarehouseId("");
      setEstimatedDuration("60");
      setNotes("");
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey;
          return Array.isArray(key) && 
            key[0] === '/api/production/orders' && 
            (key[1] === productionOrderId || key[1] === productionOrderId.toString());
        }
      });
      onSuccess();
    },
    onError: (error: any) => {
      console.error("Insert warehouse error:", error);
      toast({ title: "Błąd", description: error?.message || "Nie udało się wstawić operacji", variant: "destructive" });
    },
  });
  
  // Nie pokazuj przycisku dla zakończonych operacji
  const isOperationCompleted = currentOperationStatus === 'done';

  const handleInsert = () => {
    if (!selectedWarehouseId) {
      toast({ title: "Wybierz magazyn", variant: "destructive" });
      return;
    }
    const warehouse = locations.find(l => l.id === parseInt(selectedWarehouseId));
    insertMutation.mutate({
      afterWorkOrderId: workOrderId,
      warehouseId: parseInt(selectedWarehouseId),
      warehouseName: warehouse?.name,
      estimatedDuration: parseInt(estimatedDuration) || 60,
      notes: notes || undefined,
    });
  };

  // Nie renderuj nic dla zakończonych operacji
  if (isOperationCompleted) {
    return null;
  }
  
  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>
        <Button
          size="icon"
          variant="ghost"
          className="h-6 w-6 text-orange-500 hover:text-orange-600 hover:bg-orange-100 dark:hover:bg-orange-900/30"
          data-testid={`button-insert-warehouse-after-${workOrderId}`}
        >
          <Plus className="h-3.5 w-3.5" />
        </Button>
      </DialogTrigger>
      <DialogContent className="max-w-sm">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2 text-base">
            <Warehouse className="h-4 w-4 text-orange-500" />
            Wstaw magazynowanie
          </DialogTitle>
          <DialogDescription className="text-xs">
            Po operacji #{workOrderSequence}: {operationName}
          </DialogDescription>
        </DialogHeader>
        <div className="space-y-3 py-2">
          <div>
            <Label className="text-xs">Magazyn</Label>
            <Select value={selectedWarehouseId} onValueChange={setSelectedWarehouseId}>
              <SelectTrigger className="h-8 text-xs" data-testid={`select-warehouse-trigger-${workOrderId}`}>
                <SelectValue placeholder="Wybierz magazyn..." />
              </SelectTrigger>
              <SelectContent>
                {(warehouseLocations.length > 0 ? warehouseLocations : locations.slice(0, 10)).map(l => (
                  <SelectItem key={l.id} value={l.id.toString()} data-testid={`select-warehouse-option-${l.id}`}>
                    {l.name} {l.code ? `(${l.code})` : ''}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <div>
            <Label className="text-xs">Szacowany czas (min)</Label>
            <Input 
              type="number" 
              value={estimatedDuration} 
              onChange={(e) => setEstimatedDuration(e.target.value)}
              className="h-8 text-xs"
              data-testid={`input-duration-${workOrderId}`}
            />
          </div>
          <div>
            <Label className="text-xs">Notatki</Label>
            <Input 
              value={notes} 
              onChange={(e) => setNotes(e.target.value)}
              placeholder="Opcjonalnie..."
              className="h-8 text-xs"
              data-testid={`input-notes-${workOrderId}`}
            />
          </div>
        </div>
        <DialogFooter className="gap-2">
          <Button size="sm" variant="outline" onClick={() => setOpen(false)} data-testid={`button-cancel-insert-${workOrderId}`}>Anuluj</Button>
          <Button 
            size="sm" 
            onClick={handleInsert}
            disabled={!selectedWarehouseId || insertMutation.isPending}
            className="bg-orange-500 hover:bg-orange-600"
            data-testid={`button-confirm-insert-${workOrderId}`}
          >
            {insertMutation.isPending ? "Dodawanie..." : "Dodaj"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}

const getFlowCodeFromRouting = (routingCode: string | null | undefined): string => {
  if (!routingCode) return 'DEFAULT';
  const code = routingCode.toLowerCase();
  if (code.includes('cow')) return 'COW';
  if (code.includes('szuflad')) return 'SZUFLADY'; // Marszruta Szuflady (C+O+W)
  if (code.includes('_co')) return 'CO';
  if (code.includes('_cw')) return 'CW';
  if (code.includes('_c_')) return 'C';
  return 'DEFAULT';
};

const flowCodeColors: Record<string, { bg: string; text: string; accentColor: string }> = {
  CO: { bg: 'bg-blue-500 dark:bg-blue-600', text: 'text-white', accentColor: '#3b82f6' },
  COW: { bg: 'bg-purple-500 dark:bg-purple-600', text: 'text-white', accentColor: '#a855f7' },
  SZUFLADY: { bg: 'bg-pink-500 dark:bg-pink-600', text: 'text-white', accentColor: '#ec4899' },
  CW: { bg: 'bg-orange-500 dark:bg-orange-600', text: 'text-white', accentColor: '#f97316' },
  C: { bg: 'bg-teal-500 dark:bg-teal-600', text: 'text-white', accentColor: '#14b8a6' },
  DEFAULT: { bg: 'bg-gray-500 dark:bg-gray-600', text: 'text-white', accentColor: '#6b7280' },
};

const extractRoutingIdFromWorkOrderNumber = (woNumber: string): number | null => {
  const match = woNumber.match(/WO-\d+-R(\d+)-/);
  if (match) {
    return parseInt(match[1], 10);
  }
  return null;
};

const isSplitPointOperation = (opCode: string | null, opName: string | null): boolean => {
  if (!opCode && !opName) return false;
  const code = (opCode || '').toLowerCase();
  const name = (opName || '').toLowerCase();
  return code.includes('oklej') || code.includes('edging') || name.includes('oklejanie');
};

interface QRDialogState {
  open: boolean;
  qrDataUrl: string;
  operationCode: string;
  operationName: string;
  qrValue: string;
}

export function WorkOrderFlow({ workOrders, productionOrderId, orderStatus, damagedFormatki = [], sourcePlanId }: WorkOrderFlowProps) {
  const { toast } = useToast();
  const [completeDialogOpen, setCompleteDialogOpen] = useState(false);
  const [selectedWorkOrder, setSelectedWorkOrder] = useState<WorkOrder | null>(null);
  const [startingBatch, setStartingBatch] = useState(false);
  const [qrDialog, setQrDialog] = useState<QRDialogState>({
    open: false,
    qrDataUrl: '',
    operationCode: '',
    operationName: '',
    qrValue: '',
  });
  
  // Use sourcePlanId directly from the order detail
  const planId = sourcePlanId || null;
  
  const [formatkiModalOpen, setFormatkiModalOpen] = useState(false);
  const [formatkiModalPath, setFormatkiModalPath] = useState<string>('');
  const [palletModalOpen, setPalletModalOpen] = useState(false);
  const [newPalletType, setNewPalletType] = useState('COW');
  const [newPalletNotes, setNewPalletNotes] = useState('');
  const [startingWorkOrderId, setStartingWorkOrderId] = useState<number | null>(null);
  
  // Operation formatki modal with damage management
  const [operationFormatkiModalOpen, setOperationFormatkiModalOpen] = useState(false);
  const [selectedWorkOrderForFormatki, setSelectedWorkOrderForFormatki] = useState<WorkOrder | null>(null);
  const [selectedDamageFormatka, setSelectedDamageFormatka] = useState<{id: number; name: string} | null>(null);
  const [damageType, setDamageType] = useState<string>('');
  const [damageNotes, setDamageNotes] = useState<string>('');
  
  // Warehouse intake - zaznaczone formatki do dodania na magazyn
  const [selectedFormatkiIds, setSelectedFormatkiIds] = useState<Set<number>>(new Set());
  const [selectedWarehouseLocationId, setSelectedWarehouseLocationId] = useState<number | null>(null);
  
  // Buffer carrier assignment modal
  const [bufferCarrierModalOpen, setBufferCarrierModalOpen] = useState(false);
  const [selectedBufferLocationId, setSelectedBufferLocationId] = useState<number | null>(null);
  const [selectedBufferName, setSelectedBufferName] = useState<string>('');
  
  // Work order carrier assignment modal
  const [carrierModalOpen, setCarrierModalOpen] = useState(false);
  const [selectedWorkOrderForCarrier, setSelectedWorkOrderForCarrier] = useState<WorkOrder | null>(null);
  
  // Operator editing state
  const [operatorModalOpen, setOperatorModalOpen] = useState(false);
  const [editingWorkOrder, setEditingWorkOrder] = useState<WorkOrder | null>(null);
  const [selectedOperatorIds, setSelectedOperatorIds] = useState<number[]>([]);
  const [primaryOperatorId, setPrimaryOperatorId] = useState<number | null>(null);
  
  // Operation details modal state
  const [operationDetailsModalOpen, setOperationDetailsModalOpen] = useState(false);
  const [detailsWorkOrder, setDetailsWorkOrder] = useState<WorkOrder | null>(null);
  const [detailsWorkOrders, setDetailsWorkOrders] = useState<WorkOrder[]>([]);
  
  // Live timer for in-progress operations
  const [currentTime, setCurrentTime] = useState(Date.now());
  
  useEffect(() => {
    const interval = setInterval(() => {
      setCurrentTime(Date.now());
    }, 1000);
    return () => clearInterval(interval);
  }, []);
  
  // Formatki for details modal
  interface BomItem {
    id: number;
    formatkaName: string;
    formatkaCode: string;
    length: number;
    width: number;
    thickness: number;
    colorCode: string | null;
    quantity: number;
    itemStatus: string;
    isDamaged: boolean;
    damageType?: string | null;
    damageNotes?: string | null;
    routingCode: string | null;
    requiredOperations: string[];
    isDrilled?: boolean;
    drilledAt?: string | null;
    drilledByUsername?: string | null;
    lastWarehouseDocId?: number | null;
    warehouseDocNumber?: string | null;
  }
  
  const { data: allFormatkiData } = useQuery<BomItem[]>({
    queryKey: ['/api/production/orders', productionOrderId, 'items'],
    enabled: operationDetailsModalOpen || operationFormatkiModalOpen,
  });

  // Pobierz słownik powodów uszkodzeń
  const { data: scrapReasons } = useQuery<Array<{
    id: number;
    code: string;
    name: string;
    readableName: string;
    shortName?: string;
  }>>({
    queryKey: ['/api/dictionaries', { type: 'scrap_reason_cutting' }],
    enabled: operationFormatkiModalOpen,
  });
  
  const { data: palletFlowData } = useQuery<PalletFlowData>({
    queryKey: ['/api/production/orders', productionOrderId, 'pallet-flow'],
  });
  
  // Plan-level assembly operations query
  const { data: planAssemblyData, isLoading: planAssemblyLoading } = useQuery<PlanAssemblyOperations>({
    queryKey: ['/api/production/plans', planId, 'assembly-operations'],
    enabled: !!planId,
  });
  
  const { data: formatkiData, isLoading: formatkiLoading } = useQuery<Formatka[]>({
    queryKey: ['/api/production/orders', productionOrderId, 'path-formatki', formatkiModalPath],
    enabled: formatkiModalOpen && !!formatkiModalPath,
  });
  
  const createPalletMutation = useMutation({
    mutationFn: async (data: { palletType: string; notes?: string }) => {
      const res = await apiRequest("POST", `/api/production/orders/${productionOrderId}/pallets`, data);
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        queryKey: ['/api/production/orders', productionOrderId, 'pallet-flow'],
      });
      setPalletModalOpen(false);
      setNewPalletType('COW');
      setNewPalletNotes('');
      toast({
        title: "Sukces",
        description: "Utworzono nową paletę",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć palety",
      });
    },
  });
  
  // Mutation to assign carrier to a buffer location
  const assignCarrierToBufferMutation = useMutation({
    mutationFn: async ({ carrierId, locationId }: { carrierId: number; locationId: number | null }) => {
      const res = await apiRequest("PATCH", `/api/production/carriers/${carrierId}`, {
        currentLocationId: locationId,
        status: locationId ? 'in_use' : 'available',
      });
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        queryKey: ['/api/production/orders', productionOrderId, 'pallet-flow'],
      });
      toast({
        title: "Sukces",
        description: "Przypisano paletę do bufora",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się przypisać palety",
      });
    },
  });
  
  const openBufferCarrierModal = (locationId: number, bufferName: string) => {
    setSelectedBufferLocationId(locationId);
    setSelectedBufferName(bufferName);
    setBufferCarrierModalOpen(true);
  };
  
  const handleAssignCarrierToBuffer = (carrierId: number) => {
    if (!selectedBufferLocationId) return;
    assignCarrierToBufferMutation.mutate({ carrierId, locationId: selectedBufferLocationId });
  };
  
  const handleRemoveCarrierFromBuffer = (carrierId: number) => {
    assignCarrierToBufferMutation.mutate({ carrierId, locationId: null });
  };
  
  // Available carriers for work order assignment
  const { data: availableCarriers = [] } = useQuery<Array<{
    id: number;
    code: string;
    name: string;
    groupName: string | null;
  }>>({
    queryKey: ['/api/production/carriers/available'],
    enabled: carrierModalOpen,
  });
  
  // Open carrier modal for work order
  const openCarrierModal = (workOrder: WorkOrder) => {
    setSelectedWorkOrderForCarrier(workOrder);
    setCarrierModalOpen(true);
  };
  
  // Mutation to assign carrier to work order
  const assignCarrierToWorkOrderMutation = useMutation({
    mutationFn: async ({ workOrderId, carrierId }: { workOrderId: number; carrierId: number | null }) => {
      const res = await apiRequest("PATCH", `/api/production/work-orders/${workOrderId}/carrier`, {
        carrierId,
      });
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey;
          return Array.isArray(key) && key[0] === '/api/production/orders';
        }
      });
      setCarrierModalOpen(false);
      setSelectedWorkOrderForCarrier(null);
      toast({
        title: "Sukces",
        description: "Przypisano nośnik do operacji",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się przypisać nośnika",
      });
    },
  });
  
  // Open operation formatki modal
  const openOperationFormatkiModal = (workOrder: WorkOrder) => {
    setSelectedWorkOrderForFormatki(workOrder);
    setOperationFormatkiModalOpen(true);
  };
  
  // Mutation to mark formatka as damaged
  const markDamageMutation = useMutation({
    mutationFn: async ({ itemId, isDamaged, damageType, damageNotes }: { 
      itemId: number; 
      isDamaged: boolean; 
      damageType?: string; 
      damageNotes?: string;
    }) => {
      const res = await apiRequest("PATCH", `/api/production/order-bom-items/${itemId}/damage`, {
        isDamaged,
        damageType: isDamaged ? damageType : null,
        damageNotes: isDamaged ? damageNotes : null,
      });
      return await res.json();
    },
    onSuccess: () => {
      // Inwaliduj wszystkie powiązane query - items, pallet-flow i główne zlecenie
      queryClient.invalidateQueries({
        queryKey: ['/api/production/orders', productionOrderId, 'items'],
      });
      queryClient.invalidateQueries({
        queryKey: ['/api/production/orders', productionOrderId, 'pallet-flow'],
      });
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey;
          return Array.isArray(key) && 
            key[0] === '/api/production/orders' && 
            (key[1] === productionOrderId || key[1] === productionOrderId.toString());
        }
      });
      setSelectedDamageFormatka(null);
      setDamageType('');
      setDamageNotes('');
      toast({
        title: "Sukces",
        description: "Zaktualizowano status uszkodzenia",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się oznaczyć uszkodzenia",
      });
    },
  });
  
  const handleMarkDamage = () => {
    if (!selectedDamageFormatka || !damageType) return;
    markDamageMutation.mutate({
      itemId: selectedDamageFormatka.id,
      isDamaged: true,
      damageType,
      damageNotes,
    });
  };
  
  const handleRemoveDamage = (itemId: number) => {
    markDamageMutation.mutate({
      itemId,
      isDamaged: false,
    });
  };
  
  // Mutation to mark formatka as drilled
  const markDrilledMutation = useMutation({
    mutationFn: async ({ itemId, isDrilled }: { itemId: number; isDrilled: boolean }) => {
      const res = await apiRequest("PATCH", `/api/production/order-bom-items/${itemId}/drilling`, {
        isDrilled,
      });
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        queryKey: ['/api/production/orders', productionOrderId, 'items'],
      });
      toast({
        title: "Sukces",
        description: "Zaktualizowano status wiercenia",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się oznaczyć wiercenia",
      });
    },
  });
  
  // Fetch warehouse locations for formatki
  interface WarehouseLocation {
    id: number;
    name: string;
    code: string;
  }
  
  const { data: warehouseLocationsData } = useQuery<WarehouseLocation[]>({
    queryKey: ['/api/production/locations'],
    enabled: operationFormatkiModalOpen,
  });
  const warehouseLocations = warehouseLocationsData || [];
  
  // Mutation to add formatki to warehouse (PZ-PROD document)
  const warehouseIntakeMutation = useMutation({
    mutationFn: async ({ bomItemIds, workOrderId, targetLocationId }: { bomItemIds: number[]; workOrderId: number; targetLocationId?: number }) => {
      const res = await apiRequest("POST", `/api/production/orders/${productionOrderId}/formatki/warehouse-intake`, {
        bomItemIds,
        workOrderId,
        targetLocationId,
      });
      return await res.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({
        queryKey: ['/api/production/orders', productionOrderId, 'items'],
      });
      queryClient.invalidateQueries({
        queryKey: ['/api/production/orders', productionOrderId],
      });
      setSelectedFormatkiIds(new Set());
      toast({
        title: "Sukces",
        description: `Dodano ${data.itemsProcessed || 0} formatek na magazyn. Dokument: ${data.docNumber || 'PZ-PROD'}`,
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się dodać formatek na magazyn",
      });
    },
  });
  
  // Fetch operators for the modal
  interface ProductionOperator {
    id: number;
    full_name: string;
    short_code: string | null;
    is_active: boolean;
  }
  
  const { data: operatorsData } = useQuery<{ operators: ProductionOperator[] }>({
    queryKey: ["/api/production/operators", { is_active: "true" }],
    enabled: operatorModalOpen,
  });
  const operators = operatorsData?.operators || [];
  
  const updateOperatorsMutation = useMutation({
    mutationFn: async ({ workOrderId, operatorIds, primaryId }: { workOrderId: number; operatorIds: number[]; primaryId: number | null }) => {
      const res = await apiRequest("PUT", `/api/production/work-orders/${workOrderId}/operators`, {
        operatorIds,
        primaryOperatorId: primaryId,
      });
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      setOperatorModalOpen(false);
      setEditingWorkOrder(null);
      toast({
        title: "Sukces",
        description: "Zaktualizowano operatorów",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować operatorów",
      });
    },
  });
  
  const openOperatorModal = (wo: WorkOrder) => {
    setEditingWorkOrder(wo);
    const operators = wo.assignedOperators || [];
    setSelectedOperatorIds(operators.map(o => o.operatorId));
    const primary = operators.find(o => o.isPrimary);
    setPrimaryOperatorId(primary?.operatorId || null);
    setOperatorModalOpen(true);
  };
  
  const handleSaveOperators = () => {
    if (!editingWorkOrder) return;
    updateOperatorsMutation.mutate({
      workOrderId: editingWorkOrder.id,
      operatorIds: selectedOperatorIds,
      primaryId: primaryOperatorId,
    });
  };
  
  const toggleOperator = (operatorId: number) => {
    setSelectedOperatorIds(prev => {
      if (prev.includes(operatorId)) {
        if (primaryOperatorId === operatorId) {
          setPrimaryOperatorId(null);
        }
        return prev.filter(id => id !== operatorId);
      }
      return [...prev, operatorId];
    });
  };
  
  const openFormatkiModal = (pathCode: string) => {
    setFormatkiModalPath(pathCode);
    setFormatkiModalOpen(true);
  };
  
  const openOperationDetailsModal = (wo: WorkOrder, allWorkOrders?: WorkOrder[]) => {
    setDetailsWorkOrder(wo);
    setDetailsWorkOrders(allWorkOrders || [wo]);
    setOperationDetailsModalOpen(true);
  };
  
  const handleCreatePallet = () => {
    createPalletMutation.mutate({
      palletType: newPalletType,
      notes: newPalletNotes || undefined,
    });
  };
  
  const generateQRCode = async (operationCode: string, operationName: string) => {
    // Simple, compact format for barcode scanners: ZLP:993:magazynowanie
    const qrValue = `ZLP:${productionOrderId}:${operationCode}`;
    
    try {
      const dataUrl = await QRCode.toDataURL(qrValue, {
        width: 300,
        margin: 2,
        color: {
          dark: '#000000',
          light: '#ffffff',
        },
        errorCorrectionLevel: 'M',
      });
      
      setQrDialog({
        open: true,
        qrDataUrl: dataUrl,
        operationCode,
        operationName,
        qrValue,
      });
    } catch (err) {
      console.error('Error generating QR code:', err);
      toast({
        variant: "destructive",
        title: "Błąd",
        description: "Nie udało się wygenerować kodu QR",
      });
    }
  };
  
  const startWorkOrderMutation = useMutation({
    mutationFn: async (workOrderId: number) => {
      setStartingWorkOrderId(workOrderId);
      const res = await apiRequest("POST", `/api/production/work-orders/${workOrderId}/start`);
      return await res.json();
    },
    onSuccess: () => {
      setStartingWorkOrderId(null);
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Operacja rozpoczęta",
      });
    },
    onError: (error: Error) => {
      setStartingWorkOrderId(null);
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się rozpocząć operacji",
      });
    },
  });
  
  const startBatchWorkOrdersMutation = useMutation({
    mutationFn: async (workOrderIds: number[]) => {
      const results = await Promise.all(
        workOrderIds.map(id => 
          apiRequest("POST", `/api/production/work-orders/${id}/start`).then(r => r.json())
        )
      );
      return results;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      setStartingBatch(false);
      toast({
        title: "Sukces",
        description: "Wspólne operacje rozpoczęte dla wszystkich ścieżek",
      });
    },
    onError: (error: Error) => {
      setStartingBatch(false);
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się rozpocząć operacji",
      });
    },
  });
  
  const completeBatchWorkOrdersMutation = useMutation({
    mutationFn: async (workOrderIds: number[]) => {
      const results = await Promise.all(
        workOrderIds.map(id => 
          apiRequest("POST", `/api/production/work-orders/${id}/complete`, {}).then(r => r.json())
        )
      );
      return results;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      setCompleteDialogOpen(false);
      setSelectedWorkOrder(null);
      toast({
        title: "Sukces",
        description: "Wspólne operacje zakończone dla wszystkich ścieżek",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zakończyć operacji",
      });
    },
  });

  const completeWorkOrderMutation = useMutation({
    mutationFn: async (workOrderId: number) => {
      const res = await apiRequest("POST", `/api/production/work-orders/${workOrderId}/complete`, {});
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      setCompleteDialogOpen(false);
      setSelectedWorkOrder(null);
      toast({
        title: "Sukces",
        description: "Operacja zakończona",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zakończyć operacji",
      });
    },
  });

  const deleteWarehouseOpMutation = useMutation({
    mutationFn: async (workOrderId: number) => {
      const res = await apiRequest("DELETE", `/api/production/orders/${productionOrderId}/warehouse-operation/${workOrderId}`, {});
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Operacja magazynowania usunięta",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się usunąć operacji",
      });
    },
  });

  const handleCompleteClick = (wo: WorkOrder) => {
    setSelectedWorkOrder(wo);
    setCompleteDialogOpen(true);
  };

  const handleConfirmComplete = () => {
    if (!selectedWorkOrder) return;
    completeWorkOrderMutation.mutate(selectedWorkOrder.id);
  };

  const formatTime = (date: string | null) => {
    if (!date) return null;
    return format(new Date(date), "HH:mm", { locale: pl });
  };

  const formatDateTime = (date: string | null) => {
    if (!date) return "-";
    return format(new Date(date), "dd.MM HH:mm", { locale: pl });
  };

  const getCompletedCount = () => workOrders.filter(wo => wo.status === 'done').length;
  const getInProgressCount = () => workOrders.filter(wo => wo.status === 'in_progress').length;
  const getProgressPercentage = () => Math.round((getCompletedCount() / workOrders.length) * 100);

  const routingPaths: RoutingPath[] = (() => {
    const pathMap = new Map<string, RoutingPath>();
    
    workOrders.forEach((wo, idx) => {
      let routingId = wo.routingId;
      if (!routingId && wo.workOrderNumber) {
        routingId = extractRoutingIdFromWorkOrderNumber(wo.workOrderNumber);
      }
      
      const routingKey = routingId 
        ? `routing-${routingId}` 
        : (wo.routingCode ? `code-${wo.routingCode}` : `wo-${idx}`);
      
      if (!pathMap.has(routingKey)) {
        const flowCode = getFlowCodeFromRouting(wo.routingCode);
        const colors = flowCodeColors[flowCode] || flowCodeColors.DEFAULT;
        pathMap.set(routingKey, {
          routingKey,
          routingId: routingId || 0,
          routingCode: wo.routingCode || '',
          routingName: wo.routingName || '',
          flowCode,
          color: colors.bg,
          textColor: colors.text,
          accentColor: colors.accentColor,
          workOrders: [],
        });
      }
      pathMap.get(routingKey)!.workOrders.push(wo);
    });
    
    // Sort work orders within each path by sequence to ensure correct order
    const paths = Array.from(pathMap.values());
    paths.forEach(path => {
      path.workOrders.sort((a, b) => (a.sequence || 0) - (b.sequence || 0));
    });
    
    return paths.sort((a, b) => {
      const firstSeqA = a.workOrders[0]?.sequence || 0;
      const firstSeqB = b.workOrders[0]?.sequence || 0;
      return firstSeqA - firstSeqB;
    });
  })();

  const hasMultiplePaths = routingPaths.length > 1;
  
  interface SharedOperationGroup {
    sequence: number;
    operationCode: string | null;
    operationName: string | null;
    workOrders: WorkOrder[];
    isSplitPoint: boolean;
    isWarehouseOperation?: boolean;
    warehouseLocationName?: string | null;
  }
  
  const { sharedOperations, postSplitOperations, mergeOperations, splitPointSequence } = (() => {
    if (!hasMultiplePaths) {
      return { sharedOperations: [] as SharedOperationGroup[], postSplitOperations: routingPaths, mergeOperations: [] as SharedOperationGroup[], splitPointSequence: null };
    }
    
    // Standard shared operation codes - operations that CAN be shared at the BEGINNING
    const sharedOpCodes = new Set(['magazynowanie', 'cutting', 'transport_samochodowy', 'edging']);
    
    // Merge operation codes - operations that merge paths at the END
    const mergeOpCodes = new Set(['assembly', 'packing']);
    
    // Find the first path-specific operation sequence across all paths
    // Operations before this sequence are in the "shared zone"
    let firstPathSpecificSeq = Infinity;
    routingPaths.forEach(path => {
      path.workOrders.forEach(wo => {
        const code = wo.operationCode?.toLowerCase() || '';
        const name = wo.operationName?.toLowerCase() || '';
        // Only drilling/wiercenie is path-specific now, assembly/packing will be merge operations
        const isPathSpecific = code === 'drilling' || 
          code.includes('wierc') || name.includes('wiercenie');
        if (isPathSpecific && wo.sequence < firstPathSpecificSeq) {
          firstPathSpecificSeq = wo.sequence;
        }
      });
    });
    
    const isSharedOperationCode = (code: string | null, name: string | null, wo?: WorkOrder): boolean => {
      // Standard shared operations by code
      if (code && sharedOpCodes.has(code.toLowerCase())) return true;
      if (name) {
        const lname = name.toLowerCase();
        if (lname.includes('magazynowanie') || lname.includes('cięcie') || lname.includes('cutting') ||
            lname.includes('transport') || lname.includes('oklejanie') || lname.includes('edging')) return true;
      }
      // Dynamic warehouse operations are shared ONLY if their sequence is BEFORE first path-specific op
      if (wo?.isWarehouseOperation && !wo.operationCode && wo.sequence < firstPathSpecificSeq) {
        return true;
      }
      return false;
    };
    
    const isMergeOperationCode = (code: string | null, name: string | null): boolean => {
      if (code && mergeOpCodes.has(code.toLowerCase())) return true;
      if (name) {
        const lname = name.toLowerCase();
        if (lname.includes('kompletowanie') || lname.includes('assembly') ||
            lname.includes('pakowanie') || lname.includes('packing')) return true;
      }
      return false;
    };
    
    const sharedByOpCode = new Map<string, SharedOperationGroup>();
    const mergeByOpCode = new Map<string, SharedOperationGroup>();
    const postSplitByPath = new Map<string, WorkOrder[]>();
    
    routingPaths.forEach(path => {
      postSplitByPath.set(path.routingKey, []);
    });
    
    const opCodeOrder = ['magazynowanie', 'cutting', 'transport_samochodowy', 'edging'];
    
    routingPaths.forEach(path => {
      path.workOrders.forEach((wo) => {
        const opKey = wo.operationCode?.toLowerCase() || wo.operationName?.toLowerCase() || '';
        const isShared = isSharedOperationCode(wo.operationCode, wo.operationName, wo);
        const isMerge = isMergeOperationCode(wo.operationCode, wo.operationName);
        
        if (isShared) {
          // For dynamically inserted warehouse ops, use 'magazynowanie_dynamic' as key to group them separately
          let normalizedKey = wo.operationCode?.toLowerCase() || 
            (wo.operationName?.toLowerCase().includes('magazynowanie') ? 'magazynowanie' :
             wo.operationName?.toLowerCase().includes('cięcie') || wo.operationName?.toLowerCase().includes('cutting') ? 'cutting' :
             wo.operationName?.toLowerCase().includes('transport') ? 'transport_samochodowy' :
             wo.operationName?.toLowerCase().includes('oklejanie') || wo.operationName?.toLowerCase().includes('edging') ? 'edging' : opKey);
          
          // For dynamically inserted warehouse ops, determine their normalized key from location or notes
          if (wo.isWarehouseOperation && !wo.operationCode) {
            const locName = wo.warehouseLocationName?.toLowerCase() || wo.notes?.toLowerCase() || '';
            if (locName.includes('magazyn') || locName.includes('bufor')) {
              // Use sequence-based key to allow multiple warehouse stops in shared section
              normalizedKey = `magazynowanie_seq_${wo.sequence}`;
            }
          }
          
          if (!sharedByOpCode.has(normalizedKey)) {
            // Always use actual database sequence for correct ordering
            // This ensures dynamically inserted operations appear in their proper position
            sharedByOpCode.set(normalizedKey, {
              sequence: wo.sequence,
              operationCode: wo.operationCode,
              operationName: wo.operationName,
              workOrders: [],
              isSplitPoint: isSplitPointOperation(wo.operationCode, wo.operationName),
              isWarehouseOperation: wo.isWarehouseOperation,
              warehouseLocationName: wo.warehouseLocationName,
            });
          }
          sharedByOpCode.get(normalizedKey)!.workOrders.push(wo);
        } else if (isMerge) {
          // Merge operations (assembly, packing) - group by operation code
          const normalizedKey = wo.operationCode?.toLowerCase() || 
            (wo.operationName?.toLowerCase().includes('kompletowanie') || wo.operationName?.toLowerCase().includes('assembly') ? 'assembly' :
             wo.operationName?.toLowerCase().includes('pakowanie') || wo.operationName?.toLowerCase().includes('packing') ? 'packing' : opKey);
          
          if (!mergeByOpCode.has(normalizedKey)) {
            mergeByOpCode.set(normalizedKey, {
              sequence: wo.sequence,
              operationCode: wo.operationCode,
              operationName: wo.operationName,
              workOrders: [],
              isSplitPoint: false,
              isWarehouseOperation: wo.isWarehouseOperation,
              warehouseLocationName: wo.warehouseLocationName,
            });
          }
          mergeByOpCode.get(normalizedKey)!.workOrders.push(wo);
        } else {
          postSplitByPath.get(path.routingKey)!.push(wo);
        }
      });
    });
    
    const sharedArr = Array.from(sharedByOpCode.values()).sort((a, b) => a.sequence - b.sequence);
    const mergeArr = Array.from(mergeByOpCode.values()).sort((a, b) => a.sequence - b.sequence);
    
    const postSplitPaths: RoutingPath[] = routingPaths.map(path => ({
      ...path,
      workOrders: postSplitByPath.get(path.routingKey)?.sort((a, b) => a.sequence - b.sequence) || [],
    })).filter(p => p.workOrders.length > 0);
    
    const splitSeq = sharedArr.find(g => g.isSplitPoint)?.sequence || null;
    
    return { 
      sharedOperations: sharedArr, 
      postSplitOperations: postSplitPaths,
      mergeOperations: mergeArr,
      splitPointSequence: splitSeq 
    };
  })();

  const canStartWorkOrderInPath = (wo: WorkOrder, indexInPath: number, pathWorkOrders: WorkOrder[], pathRoutingKey?: string) => {
    if (orderStatus !== 'in_progress' && orderStatus !== 'confirmed') return false;
    if (wo.status !== 'pending' && wo.status !== 'ready') return false;
    
    if (indexInPath === 0) {
      if (sharedOperations.length > 0) {
        // Get the operation codes that THIS PATH actually has
        // This includes both the path-specific post-split operations AND any shared operations this path participates in
        const pathOperationCodes = new Set<string>();
        
        // Add operation codes from pathWorkOrders (post-split operations for this path)
        pathWorkOrders.forEach(pwo => {
          if (pwo.operationCode) {
            pathOperationCodes.add(pwo.operationCode.toLowerCase());
          }
        });
        
        // Also check which shared operations have work orders belonging to this path's routing
        sharedOperations.forEach(g => {
          g.workOrders.forEach(swo => {
            const swoRoutingKey = swo.routingCode ? `${swo.routingId || 0}-${swo.routingCode}` : null;
            if (swoRoutingKey === pathRoutingKey || 
                (pathWorkOrders.length > 0 && swo.routingId === pathWorkOrders[0]?.routingId)) {
              if (swo.operationCode) {
                pathOperationCodes.add(swo.operationCode.toLowerCase());
              }
            }
          });
        });
        
        // Filter shared operations to only those whose operation code is in this path's operations
        // This ensures CW path (which doesn't have edging/transport) won't wait for those operations
        const relevantSharedOps = sharedOperations.filter(g => {
          const opCode = (g.operationCode || '').toLowerCase();
          return pathOperationCodes.has(opCode);
        });
        
        // If no relevant shared ops found, fall back to checking if path has work orders in the group
        const opsToCheck = relevantSharedOps.length > 0 ? relevantSharedOps : sharedOperations.filter(g => 
          g.workOrders.some(swo => {
            const swoRoutingKey = swo.routingCode ? `${swo.routingId || 0}-${swo.routingCode}` : null;
            return swoRoutingKey === pathRoutingKey || 
              (pathWorkOrders.length > 0 && swo.routingId === pathWorkOrders[0]?.routingId);
          })
        );
        
        const allRelevantSharedDone = opsToCheck.every(g => 
          g.workOrders.every(swo => swo.status === 'done')
        );
        return allRelevantSharedDone;
      }
      return true;
    }
    return pathWorkOrders[indexInPath - 1]?.status === 'done';
  };
  
  const canStartSharedOperation = (group: SharedOperationGroup, groupIndex: number, allGroups: SharedOperationGroup[]) => {
    if (orderStatus !== 'in_progress' && orderStatus !== 'confirmed') return false;
    const hasPendingWorkOrders = group.workOrders.some(wo => wo.status === 'pending' || wo.status === 'ready');
    if (!hasPendingWorkOrders) return false;
    if (groupIndex === 0) return true;
    const prevGroup = allGroups[groupIndex - 1];
    return prevGroup.workOrders.every(wo => wo.status === 'done');
  };
  
  const canCompleteSharedOperation = (group: SharedOperationGroup) => {
    return group.workOrders.some(wo => wo.status === 'in_progress');
  };
  
  // Check if merge operations can start - all diverged paths must be complete
  const canStartMergeOperation = (group: SharedOperationGroup, groupIndex: number) => {
    if (orderStatus !== 'in_progress' && orderStatus !== 'confirmed') return false;
    const hasPendingWorkOrders = group.workOrders.some(wo => wo.status === 'pending' || wo.status === 'ready');
    if (!hasPendingWorkOrders) return false;
    
    // First merge operation requires all diverged paths to be complete
    if (groupIndex === 0) {
      // Check all postSplitOperations paths are done
      const allPathsDone = postSplitOperations.every(path => 
        path.workOrders.every(wo => wo.status === 'done')
      );
      return allPathsDone;
    }
    
    // Subsequent merge operations require previous merge operation to be done
    const prevGroup = mergeOperations[groupIndex - 1];
    return prevGroup.workOrders.every(wo => wo.status === 'done');
  };
  
  const canCompleteMergeOperation = (group: SharedOperationGroup) => {
    return group.workOrders.some(wo => wo.status === 'in_progress');
  };

  const canCompleteWorkOrder = (wo: WorkOrder) => {
    return wo.status === 'in_progress';
  };
  
  // Helper to get buffer name for a work order
  const getBufferAfterOperation = (workOrderId: number): string | null => {
    // First try to get from work order directly
    const wo = workOrders.find(w => w.id === workOrderId);
    if (wo?.bufferAfterName) return wo.bufferAfterName;
    
    // Fallback to palletFlowData for compatibility
    if (!palletFlowData?.buffers) return null;
    const bufferInfo = palletFlowData.buffers.find(b => b.workOrderId === workOrderId);
    return bufferInfo?.bufferAfterName || null;
  };
  
  // Helper to get carriers count in buffer after operation
  const getCarriersInBufferAfter = (workOrderId: number): Carrier[] => {
    if (!palletFlowData?.buffers) return [];
    const bufferInfo = palletFlowData.buffers.find(b => b.workOrderId === workOrderId);
    return bufferInfo?.carriersInBufferAfter || [];
  };
  
  // Helper to get buffer after ID for opening carrier modal
  const getBufferAfterId = (workOrderId: number): number | null => {
    if (!palletFlowData?.buffers) return null;
    const bufferInfo = palletFlowData.buffers.find(b => b.workOrderId === workOrderId);
    return bufferInfo?.bufferAfterId || null;
  };
  
  // Helper to calculate buffer wait time (time between previous operation end and current operation start)
  const getBufferWaitTime = (currentWo: WorkOrder, prevWo: WorkOrder | null): string | null => {
    if (!prevWo?.actualEndTime || !currentWo.actualStartTime) return null;
    
    const prevEnd = new Date(prevWo.actualEndTime).getTime();
    const currStart = new Date(currentWo.actualStartTime).getTime();
    const diffMs = Math.max(0, currStart - prevEnd);
    
    if (diffMs < 60000) return null; // Less than 1 minute, don't show
    
    const hours = Math.floor(diffMs / 3600000);
    const mins = Math.floor((diffMs % 3600000) / 60000);
    
    return hours > 0 ? `${hours}h ${mins}m` : `${mins}m`;
  };
  
  const handleStartSharedOperation = async (group: SharedOperationGroup) => {
    setStartingBatch(true);
    const pendingWorkOrders = group.workOrders.filter(wo => wo.status === 'pending' || wo.status === 'ready');
    if (pendingWorkOrders.length === 0) {
      setStartingBatch(false);
      return;
    }
    const ids = pendingWorkOrders.map(wo => wo.id);
    startBatchWorkOrdersMutation.mutate(ids);
  };
  
  const handleCompleteSharedOperation = async (group: SharedOperationGroup) => {
    const inProgressWorkOrders = group.workOrders.filter(wo => wo.status === 'in_progress');
    if (inProgressWorkOrders.length === 0) return;
    const ids = inProgressWorkOrders.map(wo => wo.id);
    completeBatchWorkOrdersMutation.mutate(ids);
  };

  if (!workOrders || workOrders.length === 0) {
    return null;
  }

  return (
    <Card className="lg:col-span-2">
      <CardHeader className="pb-3">
        <div className="flex items-center justify-between">
          <CardTitle className="flex items-center gap-2">
            <Factory className="h-5 w-5" />
            Przepływ Operacji
          </CardTitle>
          <div className="flex items-center gap-3 text-sm">
            <div className="flex items-center gap-1.5">
              <div className="h-2 w-2 rounded-full bg-green-500" />
              <span className="text-muted-foreground">
                {getCompletedCount()}/{workOrders.length} zakończone
              </span>
            </div>
            {damagedFormatki.length > 0 && (
              <>
                <div className="h-4 w-px bg-border" />
                <div className="flex items-center gap-1.5">
                  <AlertTriangle className="h-3.5 w-3.5 text-red-500" />
                  <span className="text-red-500 font-medium">{damagedFormatki.length} uszkodz.</span>
                </div>
              </>
            )}
            <div className="h-4 w-px bg-border" />
            <div className="flex items-center gap-1.5">
              <Timer className="h-3.5 w-3.5 text-muted-foreground" />
              <span className="text-muted-foreground">{getProgressPercentage()}%</span>
            </div>
          </div>
        </div>
        
        <div className="w-full h-2 bg-muted rounded-full overflow-hidden mt-2">
          <div 
            className="h-full bg-gradient-to-r from-green-500 to-green-400 transition-all duration-500"
            style={{ width: `${getProgressPercentage()}%` }}
          />
        </div>
      </CardHeader>
      
      <CardContent>
        {hasMultiplePaths && (
          <div className="flex flex-wrap items-center gap-2 mb-4">
            <span className="text-xs text-muted-foreground">Ścieżki:</span>
            {routingPaths.map((path) => {
              const completedInPath = path.workOrders.filter(wo => wo.status === 'done').length;
              return (
                <Badge 
                  key={path.routingKey} 
                  className={`${path.color} ${path.textColor} text-xs`}
                  data-testid={`badge-routing-path-${path.routingKey}`}
                >
                  {path.flowCode} ({completedInPath}/{path.workOrders.length})
                </Badge>
              );
            })}
          </div>
        )}
        
        <div className="space-y-4">
          {/* START indicator */}
          <div className="flex items-center gap-3 p-3 bg-green-100 dark:bg-green-900/30 border-l-4 border-green-500" data-testid="flow-start-indicator">
            <div className="p-2 bg-green-500">
              <Flag className="h-5 w-5 text-white" />
            </div>
            <div>
              <div className="font-semibold text-green-700 dark:text-green-300 text-sm">START PRODUKCJI</div>
              <div className="text-xs text-green-600 dark:text-green-400">Początek procesu produkcyjnego</div>
            </div>
          </div>
          
          {sharedOperations.length > 0 && (
            <div className="space-y-2" data-testid="shared-operations-section">
              <div className="flex items-center gap-2 mb-2 p-2 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800">
                <div className="h-3 w-3 rounded-full bg-green-500" />
                <span className="text-sm font-medium text-green-700 dark:text-green-300">
                  Wspólne operacje (wszystkie ścieżki razem)
                </span>
                <div className="flex gap-1 ml-auto">
                  {routingPaths.map(p => (
                    <Badge key={p.routingKey} className={`${p.color} ${p.textColor} text-[9px] px-1`}>
                      {p.flowCode}
                    </Badge>
                  ))}
                </div>
              </div>
              
              <div className="flex flex-col md:flex-row md:items-stretch gap-3 md:overflow-x-auto pt-2 md:pt-6 pb-2">
                {sharedOperations.map((group, groupIndex) => {
                  const firstWo = group.workOrders[0];
                  const allDone = group.workOrders.every(wo => wo.status === 'done');
                  const anyInProgress = group.workOrders.some(wo => wo.status === 'in_progress');
                  const OperationIconComponent = getOperationIcon(group.operationCode, !!group.isWarehouseOperation);
                  
                  const statusLabel = allDone ? 'Zakończone' : anyInProgress ? 'W trakcie' : 'Oczekuje';
                  const statusColor = allDone ? 'text-green-500' : anyInProgress ? 'text-yellow-500' : 'text-gray-500';
                  const bgColor = allDone ? 'bg-green-50/80 dark:bg-green-950/40' : anyInProgress ? 'bg-yellow-50/80 dark:bg-yellow-950/40' : 'bg-muted/50';
                  
                  return (
                    <div key={`shared-${group.sequence}`} className="flex flex-col md:flex-row md:items-stretch w-full md:w-auto gap-3" data-testid={`shared-operation-${group.sequence}`}>
                      <div 
                        className={`
                          relative flex flex-col justify-between w-full md:w-auto md:min-w-[200px] md:min-h-[220px] p-3 border border-border transition-all overflow-hidden
                          ${anyInProgress ? 'shadow-[inset_0_0_0_2px_#eab308]' : ''}
                          ${allDone ? 'opacity-70' : ''}
                          ${bgColor}
                          before:absolute before:top-0 before:left-0 before:right-0 before:h-1 before:bg-green-500
                        `}
                      >
                        <div>
                          <div className="flex items-center justify-between mb-2">
                            <div className="flex items-center gap-2">
                              <div className={`p-1.5 ${allDone ? 'bg-green-100 dark:bg-green-900/50' : anyInProgress ? 'bg-yellow-100 dark:bg-yellow-900/50' : 'bg-muted'}`}>
                                <OperationIconComponent className={`h-5 w-5 ${allDone ? 'text-green-600 dark:text-green-400' : anyInProgress ? 'text-yellow-600 dark:text-yellow-400' : 'text-muted-foreground'}`} />
                              </div>
                              {(() => {
                                const totalDamaged = group.workOrders.reduce((sum, wo) => sum + (wo.damagedCount || 0), 0);
                                return totalDamaged > 0 ? (
                                  <div className="flex items-center gap-0.5 px-1.5 py-0.5 bg-red-100 dark:bg-red-900/40 text-red-600 dark:text-red-400" title={`${totalDamaged} uszkodzonych`}>
                                    <AlertTriangle className="h-3 w-3" />
                                    <span className="text-[10px] font-bold">{totalDamaged}</span>
                                  </div>
                                ) : null;
                              })()}
                            </div>
                            <Badge variant="outline" className={`text-[10px] px-1.5 py-0 ${statusColor}`}>
                              {statusLabel}
                            </Badge>
                          </div>
                          
                          <div className="font-medium text-sm leading-tight">
                            {group.isWarehouseOperation 
                              ? (group.warehouseLocationName || 'Magazynowanie')
                              : (group.operationName || `Operacja ${group.sequence}`)}
                          </div>
                          <div className="text-[9px] text-muted-foreground font-mono mt-0.5">
                            {group.isWarehouseOperation ? (
                              <span className="text-amber-600 dark:text-amber-400">Magazyn</span>
                            ) : group.operationCode ? (
                              <span>{group.operationCode}</span>
                            ) : null}
                            {' '}
                            <span className="opacity-60">
                              [WO: {group.workOrders.map(wo => wo.id).join(', ')}]
                            </span>
                          </div>
                        </div>
                        
                        {group.isSplitPoint && (
                          <div className="flex items-center gap-1 mt-2 px-1.5 py-0.5 bg-amber-100 dark:bg-amber-900/30 text-[9px] text-amber-700 dark:text-amber-300">
                            <ChevronRight className="h-2.5 w-2.5" />
                            <span>Punkt podziału ścieżek</span>
                          </div>
                        )}
                        
                        {/* Czas operacji dla wspólnych operacji */}
                        {(() => {
                          const startedWos = group.workOrders.filter(wo => wo.actualStartTime);
                          const completedWos = group.workOrders.filter(wo => wo.actualEndTime);
                          
                          if (completedWos.length === group.workOrders.length && completedWos.length > 0) {
                            const starts = startedWos.map(wo => new Date(wo.actualStartTime!).getTime());
                            const ends = completedWos.map(wo => new Date(wo.actualEndTime!).getTime());
                            const minStart = Math.min(...starts);
                            const maxEnd = Math.max(...ends);
                            const diffMs = Math.max(0, maxEnd - minStart);
                            const hours = Math.floor(diffMs / 3600000);
                            const mins = Math.floor((diffMs % 3600000) / 60000);
                            const secs = Math.floor((diffMs % 60000) / 1000);
                            return (
                              <div className="mt-2 p-2 bg-green-100 dark:bg-green-900/30 text-center">
                                <div className="text-xs text-muted-foreground mb-0.5">Łączny czas</div>
                                <div className="text-lg font-bold text-green-700 dark:text-green-400 font-mono">
                                  {hours > 0 ? `${hours}h ${mins}m ${secs}s` : `${mins}m ${secs}s`}
                                </div>
                              </div>
                            );
                          } else if (startedWos.length > 0 && completedWos.length < group.workOrders.length) {
                            const starts = startedWos.map(wo => new Date(wo.actualStartTime!).getTime());
                            const minStart = Math.min(...starts);
                            const diffMs = Math.max(0, currentTime - minStart);
                            const hours = Math.floor(diffMs / 3600000);
                            const mins = Math.floor((diffMs % 3600000) / 60000);
                            const secs = Math.floor((diffMs % 60000) / 1000);
                            return (
                              <div className="mt-2 p-2 bg-yellow-100 dark:bg-yellow-900/30 text-center">
                                <div className="text-xs text-muted-foreground mb-0.5">W trakcie od</div>
                                <div className="text-lg font-bold text-yellow-700 dark:text-yellow-400 font-mono">
                                  {hours > 0 ? `${hours}h ${mins}m ${secs}s` : `${mins}m ${secs}s`}
                                </div>
                              </div>
                            );
                          }
                          return null;
                        })()}
                        
                        {/* Start/Koniec times for shared operations */}
                        {(() => {
                          const startedWos = group.workOrders.filter(wo => wo.actualStartTime);
                          const completedWos = group.workOrders.filter(wo => wo.actualEndTime);
                          const scheduledStartWos = group.workOrders.filter(wo => wo.scheduledStartTime);
                          const scheduledEndWos = group.workOrders.filter(wo => wo.scheduledEndTime);
                          
                          const minActualStart = startedWos.length > 0 
                            ? new Date(Math.min(...startedWos.map(wo => new Date(wo.actualStartTime!).getTime())))
                            : null;
                          const maxActualEnd = completedWos.length > 0 
                            ? new Date(Math.max(...completedWos.map(wo => new Date(wo.actualEndTime!).getTime())))
                            : null;
                          const minScheduledStart = scheduledStartWos.length > 0 
                            ? new Date(Math.min(...scheduledStartWos.map(wo => new Date(wo.scheduledStartTime!).getTime())))
                            : null;
                          const maxScheduledEnd = scheduledEndWos.length > 0 
                            ? new Date(Math.max(...scheduledEndWos.map(wo => new Date(wo.scheduledEndTime!).getTime())))
                            : null;
                          
                          return (
                            <div className="mt-2 pt-2 border-t border-border/50 space-y-1">
                              <div className="flex items-center justify-between text-[11px]">
                                <div className="flex items-center gap-1">
                                  <Play className="h-3 w-3 text-muted-foreground" />
                                  <span className="text-muted-foreground">Start:</span>
                                </div>
                                {minActualStart ? (
                                  <span className="font-medium text-green-600 dark:text-green-400">
                                    {format(minActualStart, 'dd.MM HH:mm', { locale: pl })}
                                  </span>
                                ) : minScheduledStart ? (
                                  <span className="text-muted-foreground">{format(minScheduledStart, 'dd.MM HH:mm', { locale: pl })}</span>
                                ) : (
                                  <span className="text-muted-foreground">-</span>
                                )}
                              </div>
                              <div className="flex items-center justify-between text-[11px]">
                                <div className="flex items-center gap-1">
                                  <CheckCircle className="h-3 w-3 text-muted-foreground" />
                                  <span className="text-muted-foreground">Koniec:</span>
                                </div>
                                {maxActualEnd ? (
                                  <span className="font-medium text-green-600 dark:text-green-400">
                                    {format(maxActualEnd, 'dd.MM HH:mm', { locale: pl })}
                                  </span>
                                ) : maxScheduledEnd ? (
                                  <span className="text-muted-foreground">{format(maxScheduledEnd, 'dd.MM HH:mm', { locale: pl })}</span>
                                ) : (
                                  <span className="text-muted-foreground">-</span>
                                )}
                              </div>
                            </div>
                          );
                        })()}
                        
                        <div className="flex items-center gap-1 mt-2 text-[10px] text-muted-foreground">
                          <span>{group.workOrders.length} work orders dla wszystkich ścieżek</span>
                        </div>
                        
                        {(() => {
                          const allOperators = group.workOrders.flatMap(wo => wo.assignedOperators || []);
                          const uniqueOperators = allOperators.filter((op, i, arr) => 
                            arr.findIndex(o => o.operatorId === op.operatorId) === i
                          );
                          const primaryOp = uniqueOperators.find(o => o.isPrimary);
                          const displayOps = primaryOp ? [primaryOp, ...uniqueOperators.filter(o => o.operatorId !== primaryOp.operatorId).slice(0, 1)] : uniqueOperators.slice(0, 2);
                          const moreCount = uniqueOperators.length - displayOps.length;
                          
                          return uniqueOperators.length > 0 ? (
                            <div 
                              className="flex items-center gap-1.5 mt-2 px-1.5 py-1 bg-blue-50 dark:bg-blue-900/20 text-[10px] cursor-pointer hover:bg-blue-100 dark:hover:bg-blue-900/40 transition-colors"
                              onClick={() => openOperatorModal(firstWo)}
                              data-testid={`button-edit-shared-operator-${group.sequence}`}
                            >
                              <Users className="h-3 w-3 text-blue-600 dark:text-blue-400 flex-shrink-0" />
                              <div className="flex flex-wrap gap-1 flex-1">
                                {displayOps.map(op => (
                                  <span key={op.operatorId} className={`${op.isPrimary ? 'font-medium text-blue-700 dark:text-blue-300' : 'text-muted-foreground'}`}>
                                    {op.operatorCode || op.operatorName.split(' ')[0]}
                                  </span>
                                ))}
                                {moreCount > 0 && <span className="text-muted-foreground">+{moreCount}</span>}
                              </div>
                              <Pencil className="h-2.5 w-2.5 text-blue-500" />
                            </div>
                          ) : (
                            <div 
                              className="flex items-center gap-1.5 mt-2 px-1.5 py-1 bg-gray-50 dark:bg-gray-800/50 text-[10px] text-muted-foreground cursor-pointer hover:bg-gray-100 dark:hover:bg-gray-700/50 transition-colors"
                              onClick={() => openOperatorModal(firstWo)}
                              data-testid={`button-add-shared-operator-${group.sequence}`}
                            >
                              <UserPlus className="h-3 w-3 flex-shrink-0" />
                              <span className="flex-1">Przypisz operatora</span>
                              <Pencil className="h-2.5 w-2.5" />
                            </div>
                          );
                        })()}
                        
                        {/* Carriers/Pallets visualization - always visible */}
                        {(() => {
                          const allCarriers = group.workOrders
                            .filter(wo => wo.carrierId && wo.carrierCode)
                            .map(wo => ({ id: wo.carrierId!, code: wo.carrierCode!, name: wo.carrierName }));
                          const uniqueCarriers = allCarriers.filter((c, i, arr) => 
                            arr.findIndex(x => x.id === c.id) === i
                          );
                          
                          return (
                            <div className="flex items-center gap-1 mt-2 px-1.5 py-1 bg-purple-50 dark:bg-purple-900/20">
                              <Package className="h-3 w-3 text-purple-600 dark:text-purple-400 flex-shrink-0" />
                              <div className="flex flex-wrap gap-1 flex-1 min-h-[18px]">
                                {uniqueCarriers.length > 0 ? (
                                  <>
                                    {uniqueCarriers.slice(0, 3).map(carrier => (
                                      <Tooltip key={carrier.id}>
                                        <TooltipTrigger asChild>
                                          <div className="flex items-center gap-0.5 px-1 py-0.5 bg-purple-100 dark:bg-purple-800/40 text-[9px] font-medium text-purple-700 dark:text-purple-300">
                                            <Box className="h-2.5 w-2.5" />
                                            {carrier.code}
                                          </div>
                                        </TooltipTrigger>
                                        <TooltipContent>
                                          <div className="text-xs font-medium">{carrier.name || carrier.code}</div>
                                        </TooltipContent>
                                      </Tooltip>
                                    ))}
                                    {uniqueCarriers.length > 3 && (
                                      <span className="text-[9px] text-purple-600 dark:text-purple-400">+{uniqueCarriers.length - 3}</span>
                                    )}
                                  </>
                                ) : (
                                  <span className="text-[9px] text-muted-foreground">Brak nośnika</span>
                                )}
                              </div>
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <Button
                                    size="icon"
                                    variant="ghost"
                                    className="h-5 w-5 text-purple-600 dark:text-purple-400"
                                    onClick={() => openCarrierModal(group.workOrders[0])}
                                    data-testid={`button-add-carrier-shared-${group.sequence}`}
                                  >
                                    <Plus className="h-3 w-3" />
                                  </Button>
                                </TooltipTrigger>
                                <TooltipContent>Dodaj nośnik</TooltipContent>
                              </Tooltip>
                            </div>
                          );
                        })()}
                        
                        <div className="mt-auto pt-3 space-y-2">
                          {(canStartSharedOperation(group, groupIndex, sharedOperations) || canCompleteSharedOperation(group)) && (
                            <div>
                              {canStartSharedOperation(group, groupIndex, sharedOperations) && (
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      size="sm"
                                      variant="default"
                                      className="w-full h-8 text-xs bg-green-600 hover:bg-green-700"
                                      onClick={() => handleStartSharedOperation(group)}
                                      disabled={startBatchWorkOrdersMutation.isPending || startingBatch}
                                      data-testid={`button-start-shared-${group.sequence}`}
                                    >
                                      {startBatchWorkOrdersMutation.isPending ? (
                                        <Loader2 className="h-3 w-3 animate-spin" />
                                      ) : (
                                        <>
                                          <Play className="h-3 w-3 mr-1" />
                                          Start
                                        </>
                                      )}
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Rozpoczyna {group.workOrders.filter(wo => wo.status === 'pending' || wo.status === 'ready').length} work orders</TooltipContent>
                                </Tooltip>
                              )}
                              
                              {canCompleteSharedOperation(group) && (
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      size="sm"
                                      variant="default"
                                      className="w-full h-8 text-xs bg-green-600 hover:bg-green-700"
                                      onClick={() => handleCompleteSharedOperation(group)}
                                      disabled={completeBatchWorkOrdersMutation.isPending}
                                      data-testid={`button-complete-shared-${group.sequence}`}
                                    >
                                      {completeBatchWorkOrdersMutation.isPending ? (
                                        <Loader2 className="h-3 w-3 animate-spin" />
                                      ) : (
                                        <>
                                          <CheckCircle className="h-3 w-3 mr-1" />
                                          Zakończ
                                        </>
                                      )}
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Kończy {group.workOrders.filter(wo => wo.status === 'in_progress').length} work orders</TooltipContent>
                                </Tooltip>
                              )}
                            </div>
                          )}
                          
                          <div className="flex gap-1">
                            {group.operationCode && !allDone && (
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <Button
                                    size="icon"
                                    variant="outline"
                                    className="h-7 w-7 flex-shrink-0"
                                    onClick={() => generateQRCode(group.operationCode!, group.operationName || '')}
                                    data-testid={`button-qr-shared-${group.sequence}`}
                                  >
                                    <QrCode className="h-3.5 w-3.5" />
                                  </Button>
                                </TooltipTrigger>
                                <TooltipContent>Pokaż kod QR do skanowania</TooltipContent>
                              </Tooltip>
                            )}
                            
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  size="icon"
                                  variant="outline"
                                  className="h-7 w-7 flex-shrink-0"
                                  onClick={() => openOperationFormatkiModal(firstWo)}
                                  data-testid={`button-formatki-shared-${group.sequence}`}
                                >
                                  <Boxes className="h-3.5 w-3.5" />
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>Formatki w operacji i uszkodzenia</TooltipContent>
                            </Tooltip>
                            
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  size="icon"
                                  variant="outline"
                                  className="h-7 w-7 flex-shrink-0"
                                  onClick={() => openOperationDetailsModal(firstWo, group.workOrders)}
                                  data-testid={`button-details-shared-${group.sequence}`}
                                >
                                  <Info className="h-3.5 w-3.5" />
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>Szczegóły operacji</TooltipContent>
                            </Tooltip>
                            
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <div>
                                  <InsertWarehouseAfterOperation
                                    workOrderId={firstWo.id}
                                    workOrderSequence={group.sequence}
                                    operationName={group.operationName || group.operationCode || `Operacja ${group.sequence}`}
                                    productionOrderId={productionOrderId}
                                    onSuccess={() => queryClient.invalidateQueries({ 
                                      predicate: (q) => Array.isArray(q.queryKey) && q.queryKey[0] === '/api/production/orders'
                                    })}
                                    currentOperationStatus={firstWo.status}
                                  />
                                </div>
                              </TooltipTrigger>
                              <TooltipContent>Wstaw magazynowanie po tej operacji</TooltipContent>
                            </Tooltip>
                          </div>
                        </div>
                        
                      </div>
                      
                      {(() => {
                        if (groupIndex >= sharedOperations.length - 1) return null;
                        const firstWo = group.workOrders[0];
                        const bufferName = firstWo?.bufferAfterName || (firstWo?.id ? getBufferAfterOperation(firstWo.id) : null);
                        if (!bufferName) return null;
                        const nextGroup = sharedOperations[groupIndex + 1];
                        const nextFirstWo = nextGroup?.workOrders[0];
                        const waitTime = nextFirstWo && firstWo ? getBufferWaitTime(nextFirstWo, firstWo) : null;
                        const isWaiting = firstWo?.actualEndTime && !nextFirstWo?.actualStartTime;
                        const carriersInBuffer = firstWo?.id ? getCarriersInBufferAfter(firstWo.id) : [];
                        const bufferAfterId = firstWo?.id ? getBufferAfterId(firstWo.id) : null;
                        
                        return (
                          <div 
                            className={`relative flex flex-col justify-between p-3 border border-border w-full md:w-auto md:min-w-[140px] md:min-h-[220px] cursor-pointer transition-all overflow-hidden ${
                              isWaiting 
                                ? 'bg-yellow-50/80 dark:bg-yellow-950/40' 
                                : 'bg-blue-50/80 dark:bg-blue-950/40'
                            }`}
                            onClick={() => bufferAfterId && openBufferCarrierModal(bufferAfterId, bufferName)}
                            data-testid={`buffer-shared-${group.sequence}`}
                          >
                            <div className="absolute top-0 left-0 right-0 h-1" style={{ backgroundColor: isWaiting ? '#eab308' : '#3b82f6' }} />
                            <div>
                              <div className="flex items-center gap-2 mb-2">
                                <div className={`p-1.5 ${isWaiting ? 'bg-yellow-100 dark:bg-yellow-900/50' : 'bg-blue-100 dark:bg-blue-900/50'}`}>
                                  <Boxes className={`h-4 w-4 ${isWaiting ? 'text-yellow-600 dark:text-yellow-400' : 'text-blue-600 dark:text-blue-400'}`} />
                                </div>
                                <Badge variant="outline" className={`text-[9px] ${isWaiting ? 'text-yellow-600 border-yellow-400' : 'text-blue-600 border-blue-400'}`}>
                                  Bufor
                                </Badge>
                              </div>
                              <div className={`text-xs font-medium leading-tight mb-1 ${isWaiting ? 'text-yellow-700 dark:text-yellow-300' : 'text-blue-700 dark:text-blue-300'}`}>
                                {bufferName}
                              </div>
                              <div className={`text-[10px] ${isWaiting ? 'text-yellow-600 dark:text-yellow-400' : 'text-blue-600 dark:text-blue-400'}`}>
                                {carriersInBuffer.length > 0 ? (
                                  <div className="space-y-0.5">
                                    {carriersInBuffer.slice(0, 3).map((c: any, i: number) => (
                                      <div key={i} className="flex items-center gap-1">
                                        <Package className="h-3 w-3" />
                                        <span className="font-mono">{c.name || c.code}</span>
                                      </div>
                                    ))}
                                    {carriersInBuffer.length > 3 && (
                                      <div className="text-muted-foreground">+{carriersInBuffer.length - 3} więcej</div>
                                    )}
                                  </div>
                                ) : (
                                  <div className="flex items-center gap-1 text-muted-foreground">
                                    <Package className="h-3 w-3" />
                                    <span>0 palet</span>
                                  </div>
                                )}
                              </div>
                            </div>
                            <div className="mt-auto pt-2">
                              {isWaiting ? (
                                <div className="flex items-center gap-1 text-[10px] text-yellow-600 dark:text-yellow-400 animate-pulse">
                                  <Clock className="h-3 w-3" />
                                  <LiveBufferTimer startTime={firstWo?.actualEndTime ? new Date(firstWo.actualEndTime) : null} />
                                </div>
                              ) : waitTime ? (
                                <div className="flex items-center gap-1 text-[10px] text-green-600 dark:text-green-400 font-mono">
                                  <CheckCircle className="h-3 w-3" />
                                  {waitTime}
                                </div>
                              ) : null}
                            </div>
                          </div>
                        );
                      })()}
                    </div>
                  );
                })}
              </div>
            </div>
          )}
          
          {postSplitOperations.length > 0 && sharedOperations.length > 0 && (
            <div className="flex items-center gap-2 my-4 px-2">
              <div className="flex-1 h-px bg-amber-300 dark:bg-amber-700" />
              <div className="flex items-center gap-2 px-3 py-1 bg-amber-100 dark:bg-amber-900/30 text-amber-700 dark:text-amber-300 text-xs">
                <ChevronRight className="h-3 w-3" />
                <span>Rozdzielenie ścieżek - osobne sterowanie</span>
              </div>
              <div className="flex-1 h-px bg-amber-300 dark:bg-amber-700" />
            </div>
          )}
          
          {postSplitOperations.map((path) => (
            <div key={path.routingKey} className="space-y-2" data-testid={`routing-path-${path.routingKey}`}>
              {hasMultiplePaths && (
                <div className="flex items-center gap-2 mb-2">
                  <Badge className={`${path.color} ${path.textColor} text-[10px]`}>
                    {path.flowCode}
                  </Badge>
                  <span className="text-xs text-muted-foreground">
                    {path.routingName || path.routingCode}
                  </span>
                  <span className="text-xs text-muted-foreground">
                    ({palletFlowData?.formatkiByPath?.[path.flowCode] || 0} formatek, {path.workOrders.length} operacji)
                  </span>
                </div>
              )}
              
              <div className="flex flex-col md:flex-row md:items-stretch gap-3 md:overflow-x-auto pt-2 md:pt-6 pb-2">
                {path.workOrders.map((wo, index) => {
                  const config = statusConfig[wo.status] || statusConfig.pending;
                  const isWarehouseOp = !!(wo.isWarehouseOperation || (!wo.routingOperationId && wo.warehouseLocationId));
                  const OperationIconComponent = getOperationIcon(wo.operationCode, isWarehouseOp);
                  const isActive = wo.status === 'in_progress';
                  const isCompleted = wo.status === 'done';
                  const isPending = wo.status === 'pending' || wo.status === 'ready';
                  
                  return (
                    <div key={wo.id} className="flex flex-col md:flex-row md:items-stretch w-full md:w-auto gap-3" data-testid={`work-order-flow-item-${wo.id}`}>
                      <div 
                        className={`
                          relative flex flex-col justify-between w-full md:w-auto md:min-w-[200px] md:min-h-[220px] p-3 border border-border transition-all overflow-hidden
                          ${isActive ? 'shadow-[inset_0_0_0_2px_#eab308]' : ''}
                          ${isCompleted ? 'opacity-70' : ''}
                          ${isWarehouseOp ? 'bg-amber-50/80 dark:bg-amber-950/40' : (isCompleted ? 'bg-green-50/60 dark:bg-green-950/30' : config.bgColor)}
                        `}
                      >
                        {(hasMultiplePaths || isWarehouseOp) && (
                          <div className="absolute top-0 left-0 right-0 h-1" style={{ backgroundColor: isWarehouseOp ? '#f59e0b' : path.accentColor }} />
                        )}
                        <div>
                          <div className="flex items-center justify-between mb-2">
                            <div className="flex items-center gap-2">
                              <div className={`p-1.5 ${isWarehouseOp ? 'bg-amber-100 dark:bg-amber-900/50' : isCompleted ? 'bg-green-100 dark:bg-green-900/50' : isActive ? 'bg-yellow-100 dark:bg-yellow-900/50' : 'bg-muted'}`}>
                                <OperationIconComponent className={`h-5 w-5 ${isWarehouseOp ? 'text-amber-600 dark:text-amber-400' : isCompleted ? 'text-green-600 dark:text-green-400' : isActive ? 'text-yellow-600 dark:text-yellow-400' : 'text-muted-foreground'}`} />
                              </div>
                              {wo.damagedCount && wo.damagedCount > 0 && (
                                <div className="flex items-center gap-0.5 px-1.5 py-0.5 bg-red-100 dark:bg-red-900/40 text-red-600 dark:text-red-400" title={`${wo.damagedCount} uszkodzonych`}>
                                  <AlertTriangle className="h-3 w-3" />
                                  <span className="text-[10px] font-bold">{wo.damagedCount}</span>
                                </div>
                              )}
                            </div>
                            {isWarehouseOp ? (
                              <div className="flex items-center gap-1">
                                <Badge className="text-[10px] px-1.5 py-0 bg-amber-500 text-white">
                                  Magazyn
                                </Badge>
                                {wo.status === 'pending' && (
                                  <Tooltip>
                                    <TooltipTrigger asChild>
                                      <Button
                                        size="icon"
                                        variant="ghost"
                                        className="h-5 w-5 text-red-500 hover:text-red-600 hover:bg-red-100 dark:hover:bg-red-900/30"
                                        onClick={(e) => {
                                          e.stopPropagation();
                                          if (confirm('Czy na pewno usunąć tę operację magazynowania?')) {
                                            deleteWarehouseOpMutation.mutate(wo.id);
                                          }
                                        }}
                                        disabled={deleteWarehouseOpMutation.isPending}
                                        data-testid={`button-delete-warehouse-${wo.id}`}
                                      >
                                        <Trash2 className="h-3 w-3" />
                                      </Button>
                                    </TooltipTrigger>
                                    <TooltipContent>Usuń zaplanowane magazynowanie</TooltipContent>
                                  </Tooltip>
                                )}
                              </div>
                            ) : (
                              <Badge 
                                variant="outline" 
                                className={`text-[10px] px-1.5 py-0 ${config.color}`}
                              >
                                {config.label}
                              </Badge>
                            )}
                          </div>
                          
                          <div className="font-medium text-sm leading-tight">
                            {isWarehouseOp 
                              ? (wo.warehouseLocationName || 'Magazynowanie')
                              : (wo.operationName || `Operacja ${wo.sequence}`)}
                          </div>
                          <div className="text-[9px] text-muted-foreground font-mono mt-0.5">
                            {isWarehouseOp ? (
                              <span className="text-amber-600 dark:text-amber-400">Postój w magazynie</span>
                            ) : wo.operationCode ? (
                              <span>{wo.operationCode}</span>
                            ) : null}
                            {' '}
                            <span className="opacity-60">[WO: {wo.id}]</span>
                          </div>
                        </div>
                        
                        {wo.workCenterName && (
                          <div className="flex items-center gap-1 mt-2 text-[10px] text-muted-foreground">
                            <Factory className="h-3 w-3" />
                            <span className="truncate">{wo.workCenterName}</span>
                          </div>
                        )}
                        
                        {(() => {
                          const ops = wo.assignedOperators || [];
                          const primaryOp = ops.find(o => o.isPrimary);
                          const displayName = primaryOp ? (primaryOp.operatorCode || primaryOp.operatorName) : wo.operatorName;
                          const moreCount = ops.length > 1 ? ops.length - 1 : 0;
                          
                          return displayName || ops.length > 0 ? (
                            <div 
                              className="flex items-center gap-1.5 mt-2 px-1.5 py-1 bg-blue-50 dark:bg-blue-900/20 text-[10px] cursor-pointer hover:bg-blue-100 dark:hover:bg-blue-900/40 transition-colors"
                              onClick={() => openOperatorModal(wo)}
                              data-testid={`button-edit-operator-${wo.id}`}
                            >
                              <User className="h-3 w-3 text-blue-600 dark:text-blue-400 flex-shrink-0" />
                              <span className="font-medium text-blue-700 dark:text-blue-300 truncate flex-1">
                                {displayName || ops[0]?.operatorName}
                              </span>
                              {moreCount > 0 && <span className="text-muted-foreground">+{moreCount}</span>}
                              <Pencil className="h-2.5 w-2.5 text-blue-500" />
                            </div>
                          ) : (
                            <div 
                              className="flex items-center gap-1.5 mt-2 px-1.5 py-1 bg-gray-50 dark:bg-gray-800/50 text-[10px] text-muted-foreground cursor-pointer hover:bg-gray-100 dark:hover:bg-gray-700/50 transition-colors"
                              onClick={() => openOperatorModal(wo)}
                              data-testid={`button-add-operator-${wo.id}`}
                            >
                              <UserPlus className="h-3 w-3 flex-shrink-0" />
                              <span className="flex-1">Przypisz operatora</span>
                              <Pencil className="h-2.5 w-2.5" />
                            </div>
                          );
                        })()}
                        
                        {/* Pallets visualization */}
                        {wo.pallets && wo.pallets.length > 0 && (
                          <div className="flex items-center gap-1 mt-2 px-1.5 py-1 bg-purple-50 dark:bg-purple-900/20">
                            <Package className="h-3 w-3 text-purple-600 dark:text-purple-400 flex-shrink-0" />
                            <div className="flex flex-wrap gap-1 flex-1">
                              {wo.pallets.slice(0, 2).map(pallet => (
                                <Tooltip key={pallet.id}>
                                  <TooltipTrigger asChild>
                                    <div className="flex items-center gap-0.5 px-1 py-0.5 bg-purple-100 dark:bg-purple-800/40 text-[9px] font-medium text-purple-700 dark:text-purple-300">
                                      <Box className="h-2.5 w-2.5" />
                                      {pallet.flowCode}
                                    </div>
                                  </TooltipTrigger>
                                  <TooltipContent>
                                    <div className="text-xs">
                                      <div className="font-medium">{pallet.palletLabel}</div>
                                      {pallet.carrierName && <div className="text-muted-foreground">{pallet.carrierName}</div>}
                                      <div className="text-muted-foreground">{pallet.currentLoad}/{pallet.maxCapacity} szt</div>
                                    </div>
                                  </TooltipContent>
                                </Tooltip>
                              ))}
                              {wo.pallets.length > 2 && (
                                <span className="text-[9px] text-purple-600 dark:text-purple-400">+{wo.pallets.length - 2}</span>
                              )}
                            </div>
                            <Button
                              size="icon"
                              variant="ghost"
                              className="h-5 w-5 text-purple-600 dark:text-purple-400"
                              data-testid={`button-add-pallet-${wo.id}`}
                            >
                              <Plus className="h-3 w-3" />
                            </Button>
                          </div>
                        )}
                        
                        {/* Czas trwania operacji - duża czcionka */}
                        {wo.actualStartTime && wo.actualEndTime && (
                          <div className="mt-2 p-2 bg-green-100 dark:bg-green-900/30 text-center">
                            <div className="text-xs text-muted-foreground mb-0.5">Czas operacji</div>
                            <div className="text-lg font-bold text-green-700 dark:text-green-400 font-mono">
                              {(() => {
                                const start = new Date(wo.actualStartTime).getTime();
                                const end = new Date(wo.actualEndTime).getTime();
                                const diffMs = Math.max(0, end - start);
                                const hours = Math.floor(diffMs / 3600000);
                                const mins = Math.floor((diffMs % 3600000) / 60000);
                                const secs = Math.floor((diffMs % 60000) / 1000);
                                return hours > 0 ? `${hours}h ${mins}m ${secs}s` : `${mins}m ${secs}s`;
                              })()}
                            </div>
                          </div>
                        )}
                        
                        {wo.actualStartTime && !wo.actualEndTime && (
                          <div className="mt-2 p-2 bg-yellow-100 dark:bg-yellow-900/30 text-center">
                            <div className="text-xs text-muted-foreground mb-0.5">W trakcie od</div>
                            <div className="text-lg font-bold text-yellow-700 dark:text-yellow-400 font-mono">
                              {(() => {
                                const start = new Date(wo.actualStartTime).getTime();
                                const diffMs = Math.max(0, currentTime - start);
                                const hours = Math.floor(diffMs / 3600000);
                                const mins = Math.floor((diffMs % 3600000) / 60000);
                                const secs = Math.floor((diffMs % 60000) / 1000);
                                return hours > 0 ? `${hours}h ${mins}m ${secs}s` : `${mins}m ${secs}s`;
                              })()}
                            </div>
                          </div>
                        )}
                        
                        <div className="mt-2 pt-2 border-t border-border/50 space-y-1">
                          <div className="flex items-center justify-between text-[11px]">
                            <div className="flex items-center gap-1">
                              <Play className="h-3 w-3 text-muted-foreground" />
                              <span className="text-muted-foreground">Start:</span>
                            </div>
                            {wo.actualStartTime ? (
                              <span className="font-medium text-green-600 dark:text-green-400">
                                {format(new Date(wo.actualStartTime), 'dd.MM HH:mm', { locale: pl })}
                              </span>
                            ) : wo.scheduledStartTime ? (
                              <span className="text-muted-foreground">{format(new Date(wo.scheduledStartTime), 'dd.MM HH:mm', { locale: pl })}</span>
                            ) : (
                              <span className="text-muted-foreground">-</span>
                            )}
                          </div>
                          <div className="flex items-center justify-between text-[11px]">
                            <div className="flex items-center gap-1">
                              <CheckCircle className="h-3 w-3 text-muted-foreground" />
                              <span className="text-muted-foreground">Koniec:</span>
                            </div>
                            {wo.actualEndTime ? (
                              <span className="font-medium text-green-600 dark:text-green-400">
                                {format(new Date(wo.actualEndTime), 'dd.MM HH:mm', { locale: pl })}
                              </span>
                            ) : wo.scheduledEndTime ? (
                              <span className="text-muted-foreground">{format(new Date(wo.scheduledEndTime), 'dd.MM HH:mm', { locale: pl })}</span>
                            ) : (
                              <span className="text-muted-foreground">-</span>
                            )}
                          </div>
                          <div className="flex items-center justify-between text-[11px] pt-1">
                            <span className="text-muted-foreground">Ilość:</span>
                            <span className="font-mono font-medium">
                              {wo.quantityProduced ?? 0}/{wo.quantityPlanned}
                            </span>
                          </div>
                        </div>
                        
                        <div className="mt-auto pt-3 space-y-2">
                          {(canStartWorkOrderInPath(wo, index, path.workOrders, path.routingKey) || canCompleteWorkOrder(wo)) && (
                            <div>
                              {canStartWorkOrderInPath(wo, index, path.workOrders, path.routingKey) && (
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      size="sm"
                                      variant="default"
                                      className="w-full h-8 text-xs"
                                      onClick={() => startWorkOrderMutation.mutate(wo.id)}
                                      disabled={startingWorkOrderId !== null}
                                      data-testid={`button-start-work-order-${wo.id}`}
                                    >
                                      {startingWorkOrderId === wo.id ? (
                                        <Loader2 className="h-3 w-3 animate-spin" />
                                      ) : (
                                        <>
                                          <Play className="h-3 w-3 mr-1" />
                                          Start
                                        </>
                                      )}
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Rozpocznij operację</TooltipContent>
                                </Tooltip>
                              )}
                              
                              {canCompleteWorkOrder(wo) && (
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      size="sm"
                                      variant="default"
                                      className="w-full h-8 text-xs bg-green-600 hover:bg-green-700"
                                      onClick={() => handleCompleteClick(wo)}
                                      disabled={completeWorkOrderMutation.isPending}
                                      data-testid={`button-complete-work-order-${wo.id}`}
                                    >
                                      {completeWorkOrderMutation.isPending ? (
                                        <Loader2 className="h-3 w-3 animate-spin" />
                                      ) : (
                                        <>
                                          <CheckCircle className="h-3 w-3 mr-1" />
                                          Zakończ
                                        </>
                                      )}
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Zakończ operację</TooltipContent>
                                </Tooltip>
                              )}
                            </div>
                          )}
                          
                          <div className="flex gap-1">
                            {wo.operationCode && (
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <Button
                                    size="icon"
                                    variant="outline"
                                    className="h-7 w-7 flex-shrink-0"
                                    onClick={() => generateQRCode(wo.operationCode!, wo.operationName || wo.operationCode!)}
                                    data-testid={`button-qr-path-${wo.id}`}
                                  >
                                    <QrCode className="h-3 w-3" />
                                  </Button>
                                </TooltipTrigger>
                                <TooltipContent>Pokaż kod QR</TooltipContent>
                              </Tooltip>
                            )}
                            
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  size="icon"
                                  variant="outline"
                                  className="h-7 w-7 flex-shrink-0"
                                  onClick={() => openOperationFormatkiModal(wo)}
                                  data-testid={`button-formatki-${wo.id}`}
                                >
                                  <Boxes className="h-3 w-3" />
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>Formatki i uszkodzenia</TooltipContent>
                            </Tooltip>
                            
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  size="icon"
                                  variant="outline"
                                  className="h-7 w-7 flex-shrink-0"
                                  onClick={() => openOperationDetailsModal(wo)}
                                  data-testid={`button-details-${wo.id}`}
                                >
                                  <Info className="h-3 w-3" />
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>Szczegóły operacji</TooltipContent>
                            </Tooltip>
                            
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <div>
                                  <InsertWarehouseAfterOperation
                                    workOrderId={wo.id}
                                    workOrderSequence={wo.sequence}
                                    operationName={wo.operationName || wo.operationCode || `Operacja ${wo.sequence}`}
                                    productionOrderId={productionOrderId}
                                    onSuccess={() => queryClient.invalidateQueries({ queryKey: ['/api/production/orders', productionOrderId] })}
                                    currentOperationStatus={wo.status}
                                  />
                                </div>
                              </TooltipTrigger>
                              <TooltipContent>Wstaw magazynowanie po tej operacji</TooltipContent>
                            </Tooltip>
                          </div>
                        </div>
                        

                      </div>
                      
                      {(() => {
                        if (index >= path.workOrders.length - 1) return null;
                        const bufferName = wo.bufferAfterName || getBufferAfterOperation(wo.id);
                        if (!bufferName) return null;
                        const nextWo = path.workOrders[index + 1];
                        const waitTime = getBufferWaitTime(nextWo, wo);
                        const isWaiting = wo.actualEndTime && !nextWo.actualStartTime;
                        const carriersInBuffer = getCarriersInBufferAfter(wo.id);
                        
                        return (
                          <div 
                            className={`relative flex flex-col justify-between p-3 border border-border w-full md:w-auto md:min-w-[140px] md:min-h-[220px] cursor-pointer transition-all overflow-hidden ${
                              isWaiting 
                                ? 'bg-yellow-50/80 dark:bg-yellow-950/40' 
                                : 'bg-blue-50/80 dark:bg-blue-950/40'
                            }`}
                            onClick={() => {
                              const bufferAfterId = getBufferAfterId(wo.id);
                              bufferAfterId && openBufferCarrierModal(bufferAfterId, bufferName);
                            }}
                            data-testid={`buffer-inline-${wo.id}`}
                          >
                            <div className="absolute top-0 left-0 right-0 h-1" style={{ backgroundColor: isWaiting ? '#eab308' : '#3b82f6' }} />
                            <div>
                              <div className="flex items-center gap-2 mb-2">
                                <div className={`p-1.5 ${isWaiting ? 'bg-yellow-100 dark:bg-yellow-900/50' : 'bg-blue-100 dark:bg-blue-900/50'}`}>
                                  <Boxes className={`h-4 w-4 ${isWaiting ? 'text-yellow-600 dark:text-yellow-400' : 'text-blue-600 dark:text-blue-400'}`} />
                                </div>
                                <Badge variant="outline" className={`text-[9px] ${isWaiting ? 'text-yellow-600 border-yellow-400' : 'text-blue-600 border-blue-400'}`}>
                                  Bufor
                                </Badge>
                              </div>
                              <div className={`text-xs font-medium leading-tight mb-1 ${isWaiting ? 'text-yellow-700 dark:text-yellow-300' : 'text-blue-700 dark:text-blue-300'}`}>
                                {bufferName}
                              </div>
                              <div className={`text-[10px] ${isWaiting ? 'text-yellow-600 dark:text-yellow-400' : 'text-blue-600 dark:text-blue-400'}`}>
                                {carriersInBuffer.length > 0 ? (
                                  <div className="space-y-0.5">
                                    {carriersInBuffer.slice(0, 3).map((c: any, i: number) => (
                                      <div key={i} className="flex items-center gap-1">
                                        <Package className="h-3 w-3" />
                                        <span className="font-mono">{c.name || c.code}</span>
                                      </div>
                                    ))}
                                    {carriersInBuffer.length > 3 && (
                                      <div className="text-muted-foreground">+{carriersInBuffer.length - 3} więcej</div>
                                    )}
                                  </div>
                                ) : (
                                  <div className="flex items-center gap-1 text-muted-foreground">
                                    <Package className="h-3 w-3" />
                                    <span>0 palet</span>
                                  </div>
                                )}
                              </div>
                            </div>
                            <div className="mt-auto pt-2">
                              {isWaiting ? (
                                <div className="flex items-center gap-1 text-[10px] text-yellow-600 dark:text-yellow-400 animate-pulse">
                                  <Clock className="h-3 w-3" />
                                  <LiveBufferTimer startTime={wo.actualEndTime ? new Date(wo.actualEndTime) : null} />
                                </div>
                              ) : waitTime ? (
                                <div className="flex items-center gap-1 text-[10px] text-green-600 dark:text-green-400 font-mono">
                                  <CheckCircle className="h-3 w-3" />
                                  {waitTime}
                                </div>
                              ) : null}
                            </div>
                          </div>
                        );
                      })()}
                    </div>
                  );
                })}
              </div>
            </div>
          ))}
          
          {/* Merge Operations Section - where paths come back together */}
          {mergeOperations.length > 0 && (
            <>
              <div className="flex items-center gap-2 my-4 px-2">
                <div className="flex-1 h-px bg-emerald-300 dark:bg-emerald-700" />
                <div className="flex items-center gap-2 px-3 py-1 bg-emerald-100 dark:bg-emerald-900/30 text-emerald-700 dark:text-emerald-300 text-xs">
                  <Flag className="h-3 w-3" />
                  <span>Punkt scalenia - wspólne operacje końcowe</span>
                </div>
                <div className="flex-1 h-px bg-emerald-300 dark:bg-emerald-700" />
              </div>
              
              <div className="flex flex-col md:flex-row md:items-stretch gap-3 md:overflow-x-auto pt-2 md:pt-6 pb-2">
                {mergeOperations.map((group, groupIndex) => {
                  const allDone = group.workOrders.every(wo => wo.status === 'done');
                  const anyInProgress = group.workOrders.some(wo => wo.status === 'in_progress');
                  const OperationIconComponent = getOperationIcon(group.operationCode, false);
                  const firstWo = group.workOrders[0];
                  
                  // Calculate aggregate times
                  const earliestStart = group.workOrders.reduce((min, wo) => {
                    if (wo.actualStartTime) {
                      const t = new Date(wo.actualStartTime).getTime();
                      return min ? Math.min(min, t) : t;
                    }
                    return min;
                  }, 0 as number);
                  
                  const latestEnd = group.workOrders.reduce((max, wo) => {
                    if (wo.actualEndTime) {
                      const t = new Date(wo.actualEndTime).getTime();
                      return max ? Math.max(max, t) : t;
                    }
                    return max;
                  }, 0 as number);
                  
                  return (
                    <div key={group.operationCode || groupIndex} className="flex flex-col md:flex-row md:items-stretch w-full md:w-auto gap-3" data-testid={`merge-operation-group-${group.operationCode}`}>
                      <div 
                        className={`
                          relative flex flex-col justify-between w-full md:w-auto md:min-w-[260px] md:min-h-[280px] p-4 border-2 transition-all overflow-hidden
                          ${anyInProgress ? 'border-yellow-400 shadow-[inset_0_0_0_2px_#eab308]' : allDone ? 'border-emerald-400' : 'border-border'}
                          ${allDone ? 'bg-emerald-50/80 dark:bg-emerald-950/40 opacity-80' : anyInProgress ? 'bg-yellow-50/80 dark:bg-yellow-950/40' : 'bg-muted/30'}
                        `}
                      >
                        <div className="absolute top-0 left-0 right-0 h-1.5 bg-emerald-500" />
                        <div>
                          <div className="flex items-center justify-between mb-3">
                            <div className="flex items-center gap-2">
                              <div className={`p-2 ${allDone ? 'bg-emerald-100 dark:bg-emerald-900/50' : anyInProgress ? 'bg-yellow-100 dark:bg-yellow-900/50' : 'bg-muted'}`}>
                                <OperationIconComponent className={`h-6 w-6 ${allDone ? 'text-emerald-600 dark:text-emerald-400' : anyInProgress ? 'text-yellow-600 dark:text-yellow-400' : 'text-muted-foreground'}`} />
                              </div>
                              <Badge className="text-[10px] px-2 py-0.5 bg-emerald-500 text-white">
                                Wspólna
                              </Badge>
                            </div>
                            <Badge 
                              variant="outline" 
                              className={`text-[10px] px-1.5 py-0 ${allDone ? 'text-emerald-600 border-emerald-400' : anyInProgress ? 'text-yellow-600 border-yellow-400' : 'text-muted-foreground'}`}
                            >
                              {allDone ? 'Zakończone' : anyInProgress ? 'W trakcie' : 'Oczekuje'}
                            </Badge>
                          </div>
                          
                          <div className="font-medium text-base leading-tight mb-1">
                            {group.operationName || group.operationCode || 'Operacja'}
                          </div>
                          <div className="text-[10px] text-muted-foreground font-mono mb-3">
                            {group.operationCode && <span className="uppercase">{group.operationCode}</span>}
                            {' '}
                            <span className="opacity-60">[{group.workOrders.length} ścieżek]</span>
                          </div>
                          
                          {/* Work orders list with individual controls */}
                          <div className="space-y-2 mb-3">
                            {group.workOrders.map(wo => {
                              const woConfig = statusConfig[wo.status] || statusConfig.pending;
                              const StatusIcon = woConfig.icon;
                              const routingCode = wo.routingCode || '';
                              const flowCode = getFlowCodeFromRouting(routingCode);
                              const flowColors = flowCodeColors[flowCode] || flowCodeColors.DEFAULT;
                              const canStartThis = (wo.status === 'pending' || wo.status === 'ready') && canStartMergeOperation(group, groupIndex);
                              const canCompleteThis = wo.status === 'in_progress';
                              
                              return (
                                <div 
                                  key={wo.id} 
                                  className={`p-2 border ${wo.status === 'done' ? 'bg-emerald-50/50 dark:bg-emerald-900/20 border-emerald-200 dark:border-emerald-800' : wo.status === 'in_progress' ? 'bg-yellow-50/50 dark:bg-yellow-900/20 border-yellow-200 dark:border-yellow-800' : 'bg-background border-border'}`}
                                  data-testid={`merge-wo-${wo.id}`}
                                >
                                  <div className="flex items-center gap-2 mb-1.5">
                                    <Badge className={`${flowColors.bg} ${flowColors.text} text-[9px] px-1.5 py-0`}>
                                      {flowCode}
                                    </Badge>
                                    <div className="flex-1 min-w-0">
                                      <div className="text-[10px] font-mono truncate text-muted-foreground">
                                        WO-{wo.id}
                                      </div>
                                    </div>
                                    <StatusIcon className={`h-3.5 w-3.5 ${woConfig.color} ${wo.status === 'in_progress' ? 'animate-spin' : ''}`} />
                                  </div>
                                  
                                  {/* Individual action buttons */}
                                  {(canStartThis || canCompleteThis) && (
                                    <div className="flex gap-1 mt-1.5">
                                      {canStartThis && (
                                        <Button
                                          size="sm"
                                          variant="outline"
                                          className="flex-1 h-6 text-[10px] border-emerald-400 text-emerald-600 hover:bg-emerald-50 dark:hover:bg-emerald-900/30"
                                          disabled={startWorkOrderMutation.isPending}
                                          onClick={() => startWorkOrderMutation.mutate(wo.id)}
                                          data-testid={`button-start-merge-single-${wo.id}`}
                                        >
                                          <Play className="h-2.5 w-2.5 mr-1" />
                                          Rozpocznij
                                        </Button>
                                      )}
                                      {canCompleteThis && (
                                        <Button
                                          size="sm"
                                          variant="outline"
                                          className="flex-1 h-6 text-[10px] border-emerald-400 text-emerald-600 hover:bg-emerald-50 dark:hover:bg-emerald-900/30"
                                          disabled={completeWorkOrderMutation.isPending}
                                          onClick={() => completeWorkOrderMutation.mutate(wo.id)}
                                          data-testid={`button-complete-merge-single-${wo.id}`}
                                        >
                                          <CheckCircle className="h-2.5 w-2.5 mr-1" />
                                          Zakończ
                                        </Button>
                                      )}
                                    </div>
                                  )}
                                </div>
                              );
                            })}
                          </div>
                          
                          {/* Aggregate timing */}
                          {earliestStart > 0 && latestEnd > 0 && (
                            <div className="p-2 bg-emerald-100 dark:bg-emerald-900/30 text-center mb-3">
                              <div className="text-xs text-muted-foreground mb-0.5">Łączny czas</div>
                              <div className="text-lg font-bold text-emerald-700 dark:text-emerald-400 font-mono">
                                {(() => {
                                  const diffMs = Math.max(0, latestEnd - earliestStart);
                                  const hours = Math.floor(diffMs / 3600000);
                                  const mins = Math.floor((diffMs % 3600000) / 60000);
                                  const secs = Math.floor((diffMs % 60000) / 1000);
                                  return hours > 0 ? `${hours}h ${mins}m ${secs}s` : `${mins}m ${secs}s`;
                                })()}
                              </div>
                            </div>
                          )}
                          
                          {anyInProgress && earliestStart > 0 && (
                            <div className="p-2 bg-yellow-100 dark:bg-yellow-900/30 text-center mb-3">
                              <div className="text-xs text-muted-foreground mb-0.5">W trakcie od</div>
                              <div className="text-lg font-bold text-yellow-700 dark:text-yellow-400 font-mono">
                                {(() => {
                                  const diffMs = Math.max(0, currentTime - earliestStart);
                                  const hours = Math.floor(diffMs / 3600000);
                                  const mins = Math.floor((diffMs % 3600000) / 60000);
                                  const secs = Math.floor((diffMs % 60000) / 1000);
                                  return hours > 0 ? `${hours}h ${mins}m ${secs}s` : `${mins}m ${secs}s`;
                                })()}
                              </div>
                            </div>
                          )}
                        </div>
                        
                        <div className="mt-auto pt-3 space-y-2">
                          {(canStartMergeOperation(group, groupIndex) || canCompleteMergeOperation(group)) && (
                            <div>
                              {canStartMergeOperation(group, groupIndex) && (
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      size="sm"
                                      variant="default"
                                      className="w-full h-8 text-xs"
                                      disabled={startingBatch}
                                      onClick={() => handleStartSharedOperation(group)}
                                      data-testid={`button-start-merge-${group.operationCode}`}
                                    >
                                      {startingBatch ? (
                                        <Loader2 className="h-3 w-3 mr-1 animate-spin" />
                                      ) : (
                                        <Play className="h-3 w-3 mr-1" />
                                      )}
                                      Rozpocznij wszystkie ({group.workOrders.filter(wo => wo.status === 'pending' || wo.status === 'ready').length})
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Rozpocznij operację dla wszystkich ścieżek</TooltipContent>
                                </Tooltip>
                              )}
                              {canCompleteMergeOperation(group) && (
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="w-full h-8 text-xs border-emerald-400 text-emerald-600 hover:bg-emerald-50 dark:hover:bg-emerald-900/30 mt-2"
                                  disabled={startingBatch}
                                  onClick={() => handleCompleteSharedOperation(group)}
                                  data-testid={`button-complete-merge-${group.operationCode}`}
                                >
                                  <CheckCircle className="h-3 w-3 mr-1" />
                                  Zakończ wszystkie ({group.workOrders.filter(wo => wo.status === 'in_progress').length})
                                </Button>
                              )}
                            </div>
                          )}
                        </div>
                      </div>
                      
                      {/* Arrow between merge operations */}
                      {groupIndex < mergeOperations.length - 1 && (
                        <div className="hidden md:flex items-center justify-center px-2">
                          <ArrowRightCircle className="h-6 w-6 text-emerald-400" />
                        </div>
                      )}
                    </div>
                  );
                })}
              </div>
            </>
          )}
          
          {sharedOperations.length === 0 && postSplitOperations.length === 0 && routingPaths.map((path) => (
            <div key={path.routingKey} className="space-y-2" data-testid={`routing-path-${path.routingKey}`}>
              {hasMultiplePaths && (
                <div className="flex items-center gap-2 mb-2">
                  <Badge className={`${path.color} ${path.textColor} text-[10px]`}>
                    {path.flowCode}
                  </Badge>
                  <span className="text-xs text-muted-foreground">
                    {path.routingName || path.routingCode}
                  </span>
                  <span className="text-xs text-muted-foreground">
                    ({path.workOrders.length} operacji)
                  </span>
                </div>
              )}
              
              <div className="flex flex-col md:flex-row md:items-stretch gap-3 md:overflow-x-auto pt-2 md:pt-6 pb-2">
                {path.workOrders.map((wo, index) => {
                  const config = statusConfig[wo.status] || statusConfig.pending;
                  const isWarehouseOp = !!(wo.isWarehouseOperation || (!wo.routingOperationId && wo.warehouseLocationId));
                  const OperationIconComponent = getOperationIcon(wo.operationCode, isWarehouseOp);
                  const isActive = wo.status === 'in_progress';
                  const isCompleted = wo.status === 'done';
                  const isPending = wo.status === 'pending' || wo.status === 'ready';
                  
                  return (
                    <div key={wo.id} className="flex flex-col md:flex-row md:items-stretch w-full md:w-auto gap-3" data-testid={`work-order-flow-item-${wo.id}`}>
                      <div 
                        className={`
                          relative flex flex-col justify-between w-full md:w-auto md:min-w-[200px] md:min-h-[220px] p-3 border border-border transition-all overflow-hidden
                          ${isActive ? 'shadow-[inset_0_0_0_2px_#eab308]' : ''}
                          ${isCompleted ? 'opacity-70' : ''}
                          ${isWarehouseOp ? 'bg-amber-50/80 dark:bg-amber-950/40' : (isCompleted ? 'bg-green-50/60 dark:bg-green-950/30' : config.bgColor)}
                        `}
                      >
                        {(hasMultiplePaths || isWarehouseOp) && (
                          <div className="absolute top-0 left-0 right-0 h-1" style={{ backgroundColor: isWarehouseOp ? '#f59e0b' : path.accentColor }} />
                        )}
                        <div>
                          <div className="flex items-center justify-between mb-2">
                            <div className="flex items-center gap-2">
                              <div className={`p-1.5 ${isWarehouseOp ? 'bg-amber-100 dark:bg-amber-900/50' : isCompleted ? 'bg-green-100 dark:bg-green-900/50' : isActive ? 'bg-yellow-100 dark:bg-yellow-900/50' : 'bg-muted'}`}>
                                <OperationIconComponent className={`h-5 w-5 ${isWarehouseOp ? 'text-amber-600 dark:text-amber-400' : isCompleted ? 'text-green-600 dark:text-green-400' : isActive ? 'text-yellow-600 dark:text-yellow-400' : 'text-muted-foreground'}`} />
                              </div>
                              {wo.damagedCount && wo.damagedCount > 0 && (
                                <div className="flex items-center gap-0.5 px-1.5 py-0.5 bg-red-100 dark:bg-red-900/40 text-red-600 dark:text-red-400" title={`${wo.damagedCount} uszkodzonych`}>
                                  <AlertTriangle className="h-3 w-3" />
                                  <span className="text-[10px] font-bold">{wo.damagedCount}</span>
                                </div>
                              )}
                            </div>
                            {isWarehouseOp ? (
                              <div className="flex items-center gap-1">
                                <Badge className="text-[10px] px-1.5 py-0 bg-amber-500 text-white">
                                  Magazyn
                                </Badge>
                                {wo.status === 'pending' && (
                                  <Tooltip>
                                    <TooltipTrigger asChild>
                                      <Button
                                        size="icon"
                                        variant="ghost"
                                        className="h-5 w-5 text-red-500 hover:text-red-600 hover:bg-red-100 dark:hover:bg-red-900/30"
                                        onClick={(e) => {
                                          e.stopPropagation();
                                          if (confirm('Czy na pewno usunąć tę operację magazynowania?')) {
                                            deleteWarehouseOpMutation.mutate(wo.id);
                                          }
                                        }}
                                        disabled={deleteWarehouseOpMutation.isPending}
                                        data-testid={`button-delete-warehouse-${wo.id}`}
                                      >
                                        <Trash2 className="h-3 w-3" />
                                      </Button>
                                    </TooltipTrigger>
                                    <TooltipContent>Usuń zaplanowane magazynowanie</TooltipContent>
                                  </Tooltip>
                                )}
                              </div>
                            ) : (
                              <Badge 
                                variant="outline" 
                                className={`text-[10px] px-1.5 py-0 ${config.color}`}
                              >
                                {config.label}
                              </Badge>
                            )}
                          </div>
                          
                          <div className="font-medium text-sm leading-tight">
                            {isWarehouseOp 
                              ? (wo.warehouseLocationName || 'Magazynowanie')
                              : (wo.operationName || `Operacja ${wo.sequence}`)}
                          </div>
                          <div className="text-[9px] text-muted-foreground font-mono mt-0.5">
                            {isWarehouseOp ? (
                              <span className="text-amber-600 dark:text-amber-400">Postój w magazynie</span>
                            ) : wo.operationCode ? (
                              <span>{wo.operationCode}</span>
                            ) : null}
                            {' '}
                            <span className="opacity-60">[WO: {wo.id}]</span>
                          </div>
                        </div>
                        
                        {wo.workCenterName && (
                          <div className="flex items-center gap-1 mt-2 text-[10px] text-muted-foreground">
                            <Factory className="h-3 w-3" />
                            <span className="truncate">{wo.workCenterName}</span>
                          </div>
                        )}
                        
                        {(() => {
                          const ops = wo.assignedOperators || [];
                          const primaryOp = ops.find(o => o.isPrimary);
                          const displayName = primaryOp ? (primaryOp.operatorCode || primaryOp.operatorName) : wo.operatorName;
                          const moreCount = ops.length > 1 ? ops.length - 1 : 0;
                          
                          return displayName || ops.length > 0 ? (
                            <div 
                              className="flex items-center gap-1.5 mt-2 px-1.5 py-1 bg-blue-50 dark:bg-blue-900/20 text-[10px] cursor-pointer hover:bg-blue-100 dark:hover:bg-blue-900/40 transition-colors"
                              onClick={() => openOperatorModal(wo)}
                              data-testid={`button-edit-operator-${wo.id}`}
                            >
                              <User className="h-3 w-3 text-blue-600 dark:text-blue-400 flex-shrink-0" />
                              <span className="font-medium text-blue-700 dark:text-blue-300 truncate flex-1">
                                {displayName || ops[0]?.operatorName}
                              </span>
                              {moreCount > 0 && <span className="text-muted-foreground">+{moreCount}</span>}
                              <Pencil className="h-2.5 w-2.5 text-blue-500" />
                            </div>
                          ) : (
                            <div 
                              className="flex items-center gap-1.5 mt-2 px-1.5 py-1 bg-gray-50 dark:bg-gray-800/50 text-[10px] text-muted-foreground cursor-pointer hover:bg-gray-100 dark:hover:bg-gray-700/50 transition-colors"
                              onClick={() => openOperatorModal(wo)}
                              data-testid={`button-add-operator-${wo.id}`}
                            >
                              <UserPlus className="h-3 w-3 flex-shrink-0" />
                              <span className="flex-1">Przypisz operatora</span>
                              <Pencil className="h-2.5 w-2.5" />
                            </div>
                          );
                        })()}
                        
                        {/* Pallets visualization */}
                        {wo.pallets && wo.pallets.length > 0 && (
                          <div className="flex items-center gap-1 mt-2 px-1.5 py-1 bg-purple-50 dark:bg-purple-900/20">
                            <Package className="h-3 w-3 text-purple-600 dark:text-purple-400 flex-shrink-0" />
                            <div className="flex flex-wrap gap-1 flex-1">
                              {wo.pallets.slice(0, 2).map(pallet => (
                                <Tooltip key={pallet.id}>
                                  <TooltipTrigger asChild>
                                    <div className="flex items-center gap-0.5 px-1 py-0.5 bg-purple-100 dark:bg-purple-800/40 text-[9px] font-medium text-purple-700 dark:text-purple-300">
                                      <Box className="h-2.5 w-2.5" />
                                      {pallet.flowCode}
                                    </div>
                                  </TooltipTrigger>
                                  <TooltipContent>
                                    <div className="text-xs">
                                      <div className="font-medium">{pallet.palletLabel}</div>
                                      {pallet.carrierName && <div className="text-muted-foreground">{pallet.carrierName}</div>}
                                      <div className="text-muted-foreground">{pallet.currentLoad}/{pallet.maxCapacity} szt</div>
                                    </div>
                                  </TooltipContent>
                                </Tooltip>
                              ))}
                              {wo.pallets.length > 2 && (
                                <span className="text-[9px] text-purple-600 dark:text-purple-400">+{wo.pallets.length - 2}</span>
                              )}
                            </div>
                            <Button
                              size="icon"
                              variant="ghost"
                              className="h-5 w-5 text-purple-600 dark:text-purple-400"
                              data-testid={`button-add-pallet-${wo.id}`}
                            >
                              <Plus className="h-3 w-3" />
                            </Button>
                          </div>
                        )}
                        
                        {/* Czas trwania operacji - duża czcionka */}
                        {wo.actualStartTime && wo.actualEndTime && (
                          <div className="mt-2 p-2 bg-green-100 dark:bg-green-900/30 text-center">
                            <div className="text-xs text-muted-foreground mb-0.5">Czas operacji</div>
                            <div className="text-lg font-bold text-green-700 dark:text-green-400 font-mono">
                              {(() => {
                                const start = new Date(wo.actualStartTime).getTime();
                                const end = new Date(wo.actualEndTime).getTime();
                                const diffMs = Math.max(0, end - start);
                                const hours = Math.floor(diffMs / 3600000);
                                const mins = Math.floor((diffMs % 3600000) / 60000);
                                const secs = Math.floor((diffMs % 60000) / 1000);
                                return hours > 0 ? `${hours}h ${mins}m ${secs}s` : `${mins}m ${secs}s`;
                              })()}
                            </div>
                          </div>
                        )}
                        
                        {wo.actualStartTime && !wo.actualEndTime && (
                          <div className="mt-2 p-2 bg-yellow-100 dark:bg-yellow-900/30 text-center">
                            <div className="text-xs text-muted-foreground mb-0.5">W trakcie od</div>
                            <div className="text-lg font-bold text-yellow-700 dark:text-yellow-400 font-mono">
                              {(() => {
                                const start = new Date(wo.actualStartTime).getTime();
                                const diffMs = Math.max(0, currentTime - start);
                                const hours = Math.floor(diffMs / 3600000);
                                const mins = Math.floor((diffMs % 3600000) / 60000);
                                const secs = Math.floor((diffMs % 60000) / 1000);
                                return hours > 0 ? `${hours}h ${mins}m ${secs}s` : `${mins}m ${secs}s`;
                              })()}
                            </div>
                          </div>
                        )}
                        
                        <div className="mt-2 pt-2 border-t border-border/50 space-y-1">
                          <div className="flex items-center justify-between text-[11px]">
                            <div className="flex items-center gap-1">
                              <Play className="h-3 w-3 text-muted-foreground" />
                              <span className="text-muted-foreground">Start:</span>
                            </div>
                            {wo.actualStartTime ? (
                              <span className="font-medium text-green-600 dark:text-green-400">
                                {format(new Date(wo.actualStartTime), 'dd.MM HH:mm', { locale: pl })}
                              </span>
                            ) : wo.scheduledStartTime ? (
                              <span className="text-muted-foreground">{format(new Date(wo.scheduledStartTime), 'dd.MM HH:mm', { locale: pl })}</span>
                            ) : (
                              <span className="text-muted-foreground">-</span>
                            )}
                          </div>
                          <div className="flex items-center justify-between text-[11px]">
                            <div className="flex items-center gap-1">
                              <CheckCircle className="h-3 w-3 text-muted-foreground" />
                              <span className="text-muted-foreground">Koniec:</span>
                            </div>
                            {wo.actualEndTime ? (
                              <span className="font-medium text-green-600 dark:text-green-400">
                                {format(new Date(wo.actualEndTime), 'dd.MM HH:mm', { locale: pl })}
                              </span>
                            ) : wo.scheduledEndTime ? (
                              <span className="text-muted-foreground">{format(new Date(wo.scheduledEndTime), 'dd.MM HH:mm', { locale: pl })}</span>
                            ) : (
                              <span className="text-muted-foreground">-</span>
                            )}
                          </div>
                          <div className="flex items-center justify-between text-[11px] pt-1">
                            <span className="text-muted-foreground">Ilość:</span>
                            <span className="font-mono font-medium">
                              {wo.quantityProduced ?? 0}/{wo.quantityPlanned}
                            </span>
                          </div>
                        </div>
                        
                        <div className="mt-auto pt-3 space-y-2">
                          {(canStartWorkOrderInPath(wo, index, path.workOrders, path.routingKey) || canCompleteWorkOrder(wo)) && (
                            <div>
                              {canStartWorkOrderInPath(wo, index, path.workOrders, path.routingKey) && (
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      size="sm"
                                      variant="default"
                                      className="w-full h-8 text-xs"
                                      onClick={() => startWorkOrderMutation.mutate(wo.id)}
                                      disabled={startingWorkOrderId !== null}
                                      data-testid={`button-start-work-order-${wo.id}`}
                                    >
                                      {startingWorkOrderId === wo.id ? (
                                        <Loader2 className="h-3 w-3 animate-spin" />
                                      ) : (
                                        <>
                                          <Play className="h-3 w-3 mr-1" />
                                          Start
                                        </>
                                      )}
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Rozpocznij operację</TooltipContent>
                                </Tooltip>
                              )}
                              
                              {canCompleteWorkOrder(wo) && (
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      size="sm"
                                      variant="default"
                                      className="w-full h-8 text-xs bg-green-600 hover:bg-green-700"
                                      onClick={() => handleCompleteClick(wo)}
                                      disabled={completeWorkOrderMutation.isPending}
                                      data-testid={`button-complete-work-order-${wo.id}`}
                                    >
                                      {completeWorkOrderMutation.isPending ? (
                                        <Loader2 className="h-3 w-3 animate-spin" />
                                      ) : (
                                        <>
                                          <CheckCircle className="h-3 w-3 mr-1" />
                                          Zakończ
                                        </>
                                      )}
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Zakończ operację</TooltipContent>
                                </Tooltip>
                              )}
                            </div>
                          )}
                          
                          <div className="flex gap-1">
                            {wo.operationCode && (
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <Button
                                    size="icon"
                                    variant="outline"
                                    className="h-7 w-7 flex-shrink-0"
                                    onClick={() => generateQRCode(wo.operationCode!, wo.operationName || wo.operationCode!)}
                                    data-testid={`button-qr-path-${wo.id}`}
                                  >
                                    <QrCode className="h-3 w-3" />
                                  </Button>
                                </TooltipTrigger>
                                <TooltipContent>Pokaż kod QR</TooltipContent>
                              </Tooltip>
                            )}
                            
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  size="icon"
                                  variant="outline"
                                  className="h-7 w-7 flex-shrink-0"
                                  onClick={() => openOperationFormatkiModal(wo)}
                                  data-testid={`button-formatki-${wo.id}`}
                                >
                                  <Boxes className="h-3 w-3" />
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>Formatki i uszkodzenia</TooltipContent>
                            </Tooltip>
                            
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  size="icon"
                                  variant="outline"
                                  className="h-7 w-7 flex-shrink-0"
                                  onClick={() => openOperationDetailsModal(wo)}
                                  data-testid={`button-details-${wo.id}`}
                                >
                                  <Info className="h-3 w-3" />
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>Szczegóły operacji</TooltipContent>
                            </Tooltip>
                            
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <div>
                                  <InsertWarehouseAfterOperation
                                    workOrderId={wo.id}
                                    workOrderSequence={wo.sequence}
                                    operationName={wo.operationName || wo.operationCode || `Operacja ${wo.sequence}`}
                                    productionOrderId={productionOrderId}
                                    onSuccess={() => queryClient.invalidateQueries({ queryKey: ['/api/production/orders', productionOrderId] })}
                                    currentOperationStatus={wo.status}
                                  />
                                </div>
                              </TooltipTrigger>
                              <TooltipContent>Wstaw magazynowanie po tej operacji</TooltipContent>
                            </Tooltip>
                          </div>
                        </div>
                        

                      </div>
                      
                      {(() => {
                        if (index >= path.workOrders.length - 1) return null;
                        const bufferName = wo.bufferAfterName || getBufferAfterOperation(wo.id);
                        if (!bufferName) return null;
                        const nextWo = path.workOrders[index + 1];
                        const waitTime = getBufferWaitTime(nextWo, wo);
                        const isWaiting = wo.actualEndTime && !nextWo.actualStartTime;
                        const carriersInBuffer = getCarriersInBufferAfter(wo.id);
                        
                        return (
                          <div 
                            className={`relative flex flex-col justify-between p-3 border border-border w-full md:w-auto md:min-w-[140px] md:min-h-[220px] cursor-pointer transition-all overflow-hidden ${
                              isWaiting 
                                ? 'bg-yellow-50/80 dark:bg-yellow-950/40' 
                                : 'bg-blue-50/80 dark:bg-blue-950/40'
                            }`}
                            onClick={() => {
                              const bufferAfterId = getBufferAfterId(wo.id);
                              bufferAfterId && openBufferCarrierModal(bufferAfterId, bufferName);
                            }}
                            data-testid={`buffer-inline-${wo.id}`}
                          >
                            <div className="absolute top-0 left-0 right-0 h-1" style={{ backgroundColor: isWaiting ? '#eab308' : '#3b82f6' }} />
                            <div>
                              <div className="flex items-center gap-2 mb-2">
                                <div className={`p-1.5 ${isWaiting ? 'bg-yellow-100 dark:bg-yellow-900/50' : 'bg-blue-100 dark:bg-blue-900/50'}`}>
                                  <Boxes className={`h-4 w-4 ${isWaiting ? 'text-yellow-600 dark:text-yellow-400' : 'text-blue-600 dark:text-blue-400'}`} />
                                </div>
                                <Badge variant="outline" className={`text-[9px] ${isWaiting ? 'text-yellow-600 border-yellow-400' : 'text-blue-600 border-blue-400'}`}>
                                  Bufor
                                </Badge>
                              </div>
                              <div className={`text-xs font-medium leading-tight mb-1 ${isWaiting ? 'text-yellow-700 dark:text-yellow-300' : 'text-blue-700 dark:text-blue-300'}`}>
                                {bufferName}
                              </div>
                              <div className={`text-[10px] ${isWaiting ? 'text-yellow-600 dark:text-yellow-400' : 'text-blue-600 dark:text-blue-400'}`}>
                                {carriersInBuffer.length > 0 ? (
                                  <div className="space-y-0.5">
                                    {carriersInBuffer.slice(0, 3).map((c: any, i: number) => (
                                      <div key={i} className="flex items-center gap-1">
                                        <Package className="h-3 w-3" />
                                        <span className="font-mono">{c.name || c.code}</span>
                                      </div>
                                    ))}
                                    {carriersInBuffer.length > 3 && (
                                      <div className="text-muted-foreground">+{carriersInBuffer.length - 3} więcej</div>
                                    )}
                                  </div>
                                ) : (
                                  <div className="flex items-center gap-1 text-muted-foreground">
                                    <Package className="h-3 w-3" />
                                    <span>0 palet</span>
                                  </div>
                                )}
                              </div>
                            </div>
                            <div className="mt-auto pt-2">
                              {isWaiting ? (
                                <div className="flex items-center gap-1 text-[10px] text-yellow-600 dark:text-yellow-400 animate-pulse">
                                  <Clock className="h-3 w-3" />
                                  <LiveBufferTimer startTime={wo.actualEndTime ? new Date(wo.actualEndTime) : null} />
                                </div>
                              ) : waitTime ? (
                                <div className="flex items-center gap-1 text-[10px] text-green-600 dark:text-green-400 font-mono">
                                  <CheckCircle className="h-3 w-3" />
                                  {waitTime}
                                </div>
                              ) : null}
                            </div>
                          </div>
                        );
                      })()}
                    </div>
                  );
                })}
              </div>
            </div>
          ))}
          
          {/* Plan-level Assembly/Packing Section */}
          {planAssemblyData && planAssemblyData.operationGroups.length > 0 && (
            <div className="space-y-3 mt-4 pt-4 border-t border-dashed border-purple-300 dark:border-purple-700" data-testid="plan-assembly-section">
              <div className="flex items-center gap-2 p-3 bg-purple-100 dark:bg-purple-900/30 border-l-4 border-purple-500">
                <Layers className="h-5 w-5 text-purple-600 dark:text-purple-400" />
                <div className="flex-1">
                  <div className="font-semibold text-purple-700 dark:text-purple-300 text-sm">
                    Montaż i Pakowanie na poziomie planu
                  </div>
                  <div className="text-xs text-purple-600 dark:text-purple-400">
                    {planAssemblyData.planNumber} - {planAssemblyData.planName || 'Plan produkcyjny'}
                    <span className="ml-2">({planAssemblyData.zlps.length} kolorów)</span>
                  </div>
                </div>
              </div>
              
              {/* ZLP Colors Status Grid */}
              <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-2 p-3 bg-muted/30 rounded-lg">
                {planAssemblyData.zlps.map((zlp) => (
                  <div 
                    key={zlp.id}
                    className={`p-2 rounded border ${zlp.allPreAssemblyDone 
                      ? 'bg-green-50 dark:bg-green-900/20 border-green-300 dark:border-green-700' 
                      : 'bg-yellow-50 dark:bg-yellow-900/20 border-yellow-300 dark:border-yellow-700'
                    }`}
                    data-testid={`plan-zlp-status-${zlp.id}`}
                  >
                    <div className="flex items-center gap-1.5">
                      {zlp.allPreAssemblyDone ? (
                        <CheckCircle className="h-3.5 w-3.5 text-green-500" />
                      ) : (
                        <Clock className="h-3.5 w-3.5 text-yellow-500" />
                      )}
                      <span className="text-xs font-medium truncate" title={zlp.colorCode}>
                        {zlp.colorCode || 'Brak koloru'}
                      </span>
                    </div>
                    <div className="text-[10px] text-muted-foreground mt-0.5">
                      {zlp.orderNumber}
                    </div>
                  </div>
                ))}
              </div>
              
              {/* Assembly/Packing Operation Groups */}
              <div className="flex flex-wrap gap-3">
                {planAssemblyData.operationGroups.map((group) => {
                  const OperationIcon = getOperationIcon(group.operationCode, false);
                  const bgClass = group.allDone 
                    ? 'bg-green-100 dark:bg-green-900/30 border-green-300 dark:border-green-700'
                    : group.anyInProgress
                    ? 'bg-yellow-100 dark:bg-yellow-900/30 border-yellow-300 dark:border-yellow-700'
                    : group.allPrerequisitesDone
                    ? 'bg-blue-100 dark:bg-blue-900/30 border-blue-300 dark:border-blue-700'
                    : 'bg-gray-100 dark:bg-gray-800 border-gray-300 dark:border-gray-600';
                  
                  const statusLabel = group.allDone 
                    ? 'Zakończone' 
                    : group.anyInProgress 
                    ? 'W trakcie' 
                    : group.allPrerequisitesDone 
                    ? 'Gotowe do startu'
                    : 'Oczekuje na poprzednie operacje';
                  
                  const textClass = group.allDone 
                    ? 'text-green-600 dark:text-green-400'
                    : group.anyInProgress
                    ? 'text-yellow-600 dark:text-yellow-400'
                    : group.allPrerequisitesDone
                    ? 'text-blue-600 dark:text-blue-400'
                    : 'text-gray-500 dark:text-gray-400';
                    
                  const canStart = group.allPrerequisitesDone && !group.allDone && !group.anyInProgress;
                  const canComplete = group.anyInProgress;
                  
                  return (
                    <div 
                      key={group.operationCode}
                      className={`flex-1 min-w-[200px] max-w-[300px] p-4 rounded-lg border ${bgClass}`}
                      data-testid={`plan-operation-group-${group.operationCode}`}
                    >
                      <div className="flex items-center gap-2 mb-2">
                        <div className={`p-2 rounded-full ${group.allDone ? 'bg-green-500' : group.anyInProgress ? 'bg-yellow-500' : 'bg-gray-400'}`}>
                          <OperationIcon className="h-4 w-4 text-white" />
                        </div>
                        <div>
                          <div className="font-medium text-sm">{group.operationName}</div>
                          <div className={`text-xs ${textClass}`}>{statusLabel}</div>
                        </div>
                      </div>
                      
                      <div className="text-xs text-muted-foreground mb-2">
                        {group.workOrders.filter(wo => wo.status === 'done').length}/{group.workOrders.length} kolorów zakończonych
                      </div>
                      
                      <div className="flex items-center gap-3 text-xs mb-3">
                        <div className="flex items-center gap-1">
                          <Package className="h-3 w-3" />
                          <span>{group.totalQuantityPlanned} szt. plan</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <CheckCircle className="h-3 w-3" />
                          <span>{group.totalQuantityProduced} szt. wyk.</span>
                        </div>
                      </div>
                      
                      {/* Per-ZLP Work Order Status */}
                      <div className="space-y-1 mb-3">
                        {group.workOrders.map((wo) => (
                          <div 
                            key={wo.id}
                            className={`flex items-center justify-between text-xs p-1.5 rounded ${
                              wo.status === 'done' 
                                ? 'bg-green-50/50 dark:bg-green-900/10' 
                                : wo.status === 'in_progress'
                                ? 'bg-yellow-50/50 dark:bg-yellow-900/10'
                                : 'bg-gray-50/50 dark:bg-gray-800/30'
                            }`}
                          >
                            <span className="font-mono truncate" title={wo.zlpColorCode}>
                              {wo.zlpColorCode || wo.zlpOrderNumber}
                            </span>
                            <Badge variant={wo.status === 'done' ? 'default' : wo.status === 'in_progress' ? 'secondary' : 'outline'} className="text-[9px] px-1">
                              {wo.status === 'done' ? 'OK' : wo.status === 'in_progress' ? 'W trakcie' : 'Oczekuje'}
                            </Badge>
                          </div>
                        ))}
                      </div>
                      
                      {/* Batch Action Buttons */}
                      <div className="flex gap-2">
                        {canStart && (
                          <Button
                            size="sm"
                            variant="default"
                            className="flex-1 h-8 text-xs"
                            onClick={async () => {
                              try {
                                await apiRequest("POST", `/api/production/plans/${planAssemblyData.planId}/assembly-operations/${group.operationCode}/start`, {});
                                queryClient.invalidateQueries({
                                  queryKey: ['/api/production/plans', planAssemblyData.planId, 'assembly-operations'],
                                });
                                queryClient.invalidateQueries({
                                  queryKey: ['/api/production/orders', productionOrderId],
                                });
                                toast({ title: 'Rozpoczęto operację', description: `${group.operationName} dla wszystkich kolorów` });
                              } catch (error) {
                                toast({ variant: 'destructive', title: 'Błąd', description: 'Nie udało się rozpocząć operacji' });
                              }
                            }}
                            data-testid={`button-start-plan-${group.operationCode}`}
                          >
                            <Play className="h-3 w-3 mr-1" />
                            Rozpocznij wszystkie
                          </Button>
                        )}
                        {canComplete && (
                          <Button
                            size="sm"
                            variant="outline"
                            className="flex-1 h-8 text-xs"
                            onClick={async () => {
                              try {
                                await apiRequest("POST", `/api/production/plans/${planAssemblyData.planId}/assembly-operations/${group.operationCode}/complete`, {});
                                queryClient.invalidateQueries({
                                  queryKey: ['/api/production/plans', planAssemblyData.planId, 'assembly-operations'],
                                });
                                queryClient.invalidateQueries({
                                  queryKey: ['/api/production/orders', productionOrderId],
                                });
                                toast({ title: 'Zakończono operację', description: `${group.operationName} dla wszystkich kolorów` });
                              } catch (error) {
                                toast({ variant: 'destructive', title: 'Błąd', description: 'Nie udało się zakończyć operacji' });
                              }
                            }}
                            data-testid={`button-complete-plan-${group.operationCode}`}
                          >
                            <CheckCircle className="h-3 w-3 mr-1" />
                            Zakończ wszystkie
                          </Button>
                        )}
                      </div>
                    </div>
                  );
                })}
              </div>
            </div>
          )}
          
          {/* FINISH indicator */}
          <div className="flex items-center gap-3 p-3 bg-blue-100 dark:bg-blue-900/30 border-l-4 border-blue-500" data-testid="flow-finish-indicator">
            <div className="p-2 bg-blue-500">
              <ArrowRightCircle className="h-5 w-5 text-white" />
            </div>
            <div>
              <div className="font-semibold text-blue-700 dark:text-blue-300 text-sm">KONIEC PRODUKCJI</div>
              <div className="text-xs text-blue-600 dark:text-blue-400">Zakończenie procesu produkcyjnego</div>
            </div>
          </div>
        </div>
      </CardContent>
      
      <Dialog open={completeDialogOpen} onOpenChange={setCompleteDialogOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Zakończ operację</DialogTitle>
            <DialogDescription>
              {selectedWorkOrder?.operationName || selectedWorkOrder?.workOrderNumber}
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4 text-sm">
              <div>
                <span className="text-muted-foreground">Ilość zaplanowana:</span>
                <span className="ml-2 font-medium">{selectedWorkOrder?.quantityPlanned} szt.</span>
              </div>
            </div>
            
            {damagedFormatki.length > 0 && (
              <div className="p-3 bg-yellow-100 dark:bg-yellow-900/30 rounded-md space-y-2">
                <div className="flex items-center gap-2">
                  <AlertTriangle className="h-4 w-4 text-yellow-600" />
                  <span className="text-sm font-medium text-yellow-700 dark:text-yellow-400">
                    {damagedFormatki.length} szt. oznaczonych jako uszkodzone:
                  </span>
                </div>
                <div className="max-h-32 overflow-y-auto space-y-1 pl-6">
                  {damagedFormatki.map((f) => (
                    <div key={f.id} className="text-xs text-yellow-700 dark:text-yellow-400 flex items-center gap-2">
                      <span className="font-medium">{f.generatedName}</span>
                      {f.damageType && (
                        <span className="text-yellow-600 dark:text-yellow-500">({f.damageType})</span>
                      )}
                    </div>
                  ))}
                </div>
              </div>
            )}
            
            <p className="text-sm text-muted-foreground">
              Czy na pewno chcesz zakończyć tę operację?
            </p>
          </div>
          
          <DialogFooter className="gap-2">
            <Button
              variant="outline"
              onClick={() => setCompleteDialogOpen(false)}
              data-testid="button-cancel-complete"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleConfirmComplete}
              disabled={completeWorkOrderMutation.isPending}
              data-testid="button-confirm-complete"
            >
              {completeWorkOrderMutation.isPending ? (
                <Loader2 className="h-4 w-4 animate-spin mr-2" />
              ) : (
                <CheckCircle className="h-4 w-4 mr-2" />
              )}
              Zakończ operację
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      <Dialog open={qrDialog.open} onOpenChange={(open) => setQrDialog(prev => ({ ...prev, open }))}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <QrCode className="h-5 w-5" />
              Kod QR - {qrDialog.operationName || qrDialog.operationCode}
            </DialogTitle>
            <DialogDescription>
              Zeskanuj kod aby rozpocząć/zakończyć operację
            </DialogDescription>
          </DialogHeader>
          
          <div className="flex flex-col items-center gap-4 py-4">
            {qrDialog.qrDataUrl && (
              <div className="bg-white p-4 rounded-lg shadow-sm">
                <img 
                  src={qrDialog.qrDataUrl} 
                  alt={`QR Code for ${qrDialog.operationCode}`}
                  className="w-[250px] h-[250px]"
                  data-testid="img-qr-code"
                />
              </div>
            )}
            
            <div className="text-center space-y-1">
              <p className="text-sm font-medium">
                ZLP-{productionOrderId}
              </p>
              <p className="text-xs text-muted-foreground">
                {qrDialog.operationName || qrDialog.operationCode}
              </p>
            </div>
            
            <div className="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 p-3 rounded-lg text-sm text-center w-full space-y-2">
              <p className="font-medium text-green-700 dark:text-green-300">Instrukcja skanowania:</p>
              <div className="grid grid-cols-2 gap-2 text-xs text-green-600 dark:text-green-400">
                <div className="bg-white dark:bg-green-900/30 p-2 rounded">
                  <p className="font-bold">1. skan</p>
                  <p>ROZPOCZNIJ</p>
                </div>
                <div className="bg-white dark:bg-green-900/30 p-2 rounded">
                  <p className="font-bold">2. skan</p>
                  <p>ZAKOŃCZ</p>
                </div>
              </div>
            </div>
            
            <div className="flex flex-col gap-1 w-full">
              <span className="text-[10px] text-muted-foreground text-center">Format kodu:</span>
              <code className="text-xs font-mono bg-muted p-2 rounded text-center">
                {qrDialog.qrValue}
              </code>
            </div>
          </div>
          
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setQrDialog(prev => ({ ...prev, open: false }))}
              data-testid="button-close-qr-dialog"
            >
              Zamknij
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      <Dialog open={formatkiModalOpen} onOpenChange={setFormatkiModalOpen}>
        <DialogContent className="max-w-3xl max-h-[80vh]">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Layers className="h-5 w-5" />
              Formatki - ścieżka {formatkiModalPath}
            </DialogTitle>
            <DialogDescription>
              Lista formatek przypisanych do ścieżki {formatkiModalPath}
            </DialogDescription>
          </DialogHeader>
          
          <ScrollArea className="h-[400px]">
            {formatkiLoading ? (
              <div className="flex items-center justify-center h-32">
                <Loader2 className="h-6 w-6 animate-spin" />
              </div>
            ) : formatkiData && formatkiData.length > 0 ? (
              <div className="space-y-2">
                <div className="grid grid-cols-7 gap-2 text-xs font-medium text-muted-foreground p-2 bg-muted rounded-t">
                  <span>Nazwa</span>
                  <span>Kod</span>
                  <span className="text-center">Wymiary</span>
                  <span className="text-center">Grubość</span>
                  <span className="text-center">Kolor</span>
                  <span className="text-center">Ilość</span>
                  <span className="text-center">Status</span>
                </div>
                {formatkiData.map((f) => (
                  <div key={f.id} className="grid grid-cols-7 gap-2 text-xs p-2 border-b hover:bg-muted/50" data-testid={`formatka-item-${f.id}`}>
                    <span className="font-medium truncate">{f.formatkaName}</span>
                    <span className="text-muted-foreground truncate">{f.formatkaCode}</span>
                    <span className="text-center">{f.length} x {f.width}</span>
                    <span className="text-center">{f.thickness}mm</span>
                    <span className="text-center">
                      <Badge variant="outline" className="text-[10px]">{f.colorCode || '-'}</Badge>
                    </span>
                    <span className="text-center font-medium">{f.quantity}</span>
                    <span className="text-center">
                      <Badge variant="outline" className="text-[10px]">{f.itemStatus}</Badge>
                    </span>
                  </div>
                ))}
              </div>
            ) : (
              <div className="flex items-center justify-center h-32 text-muted-foreground">
                Brak formatek dla tej ścieżki
              </div>
            )}
          </ScrollArea>
          
          <DialogFooter>
            <Button variant="outline" onClick={() => setFormatkiModalOpen(false)} data-testid="button-close-formatki-modal">
              Zamknij
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      <Dialog open={palletModalOpen} onOpenChange={setPalletModalOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Package className="h-5 w-5" />
              Nowa paleta
            </DialogTitle>
            <DialogDescription>
              Utwórz nową paletę dla tego zlecenia produkcyjnego
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="palletType">Typ palety (ścieżka)</Label>
              <Select value={newPalletType} onValueChange={setNewPalletType}>
                <SelectTrigger data-testid="select-pallet-type">
                  <SelectValue placeholder="Wybierz typ" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="COW">COW - Cięcie, Oklejanie, Wiercenie</SelectItem>
                  <SelectItem value="CO">CO - Cięcie, Oklejanie</SelectItem>
                  <SelectItem value="CW">CW - Cięcie, Wiercenie</SelectItem>
                  <SelectItem value="C">C - Tylko cięcie</SelectItem>
                </SelectContent>
              </Select>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="palletNotes">Notatki (opcjonalne)</Label>
              <Input
                id="palletNotes"
                value={newPalletNotes}
                onChange={(e) => setNewPalletNotes(e.target.value)}
                placeholder="Dodatkowe informacje..."
                data-testid="input-pallet-notes"
              />
            </div>
          </div>
          
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setPalletModalOpen(false)} data-testid="button-cancel-pallet">
              Anuluj
            </Button>
            <Button 
              onClick={handleCreatePallet}
              disabled={createPalletMutation.isPending}
              data-testid="button-create-pallet"
            >
              {createPalletMutation.isPending ? (
                <Loader2 className="h-4 w-4 animate-spin mr-2" />
              ) : (
                <Plus className="h-4 w-4 mr-2" />
              )}
              Utwórz paletę
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      <Dialog open={bufferCarrierModalOpen} onOpenChange={setBufferCarrierModalOpen}>
        <DialogContent className="sm:max-w-lg" data-testid="dialog-buffer-carriers">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Warehouse className="h-5 w-5" />
              Zarządzaj paletami w buforze
            </DialogTitle>
            <DialogDescription>
              Bufor: {selectedBufferName}
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-2">
            <div className="space-y-2">
              <Label className="text-sm font-medium">Palety w buforze</Label>
              {(() => {
                const carriersInBuffer = palletFlowData?.carriers?.filter(
                  c => c.currentLocationId === selectedBufferLocationId
                ) || [];
                
                if (carriersInBuffer.length === 0) {
                  return (
                    <div className="text-sm text-muted-foreground py-2 px-3 bg-muted rounded">
                      Brak palet w tym buforze
                    </div>
                  );
                }
                
                return (
                  <div className="space-y-1">
                    {carriersInBuffer.map(carrier => (
                      <div 
                        key={carrier.id}
                        className="flex items-center justify-between p-2 border rounded bg-card"
                      >
                        <div className="flex items-center gap-2">
                          <Package className="h-4 w-4 text-blue-500" />
                          <span className="font-medium">{carrier.code}</span>
                          <span className="text-sm text-muted-foreground">{carrier.name}</span>
                        </div>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => handleRemoveCarrierFromBuffer(carrier.id)}
                          disabled={assignCarrierToBufferMutation.isPending}
                          data-testid={`button-remove-carrier-${carrier.id}`}
                        >
                          <XCircle className="h-3 w-3 mr-1" />
                          Usuń
                        </Button>
                      </div>
                    ))}
                  </div>
                );
              })()}
            </div>
            
            <div className="border-t pt-4 space-y-2">
              <Label className="text-sm font-medium">Dostępne palety</Label>
              {(() => {
                const availableCarriers = palletFlowData?.availableCarriers || [];
                
                if (availableCarriers.length === 0) {
                  return (
                    <div className="text-sm text-muted-foreground py-2 px-3 bg-muted rounded">
                      Brak dostępnych palet do przypisania
                    </div>
                  );
                }
                
                return (
                  <ScrollArea className="max-h-[200px]">
                    <div className="space-y-1">
                      {availableCarriers.map(carrier => (
                        <div 
                          key={carrier.id}
                          className="flex items-center justify-between p-2 border rounded bg-card hover-elevate cursor-pointer"
                          onClick={() => handleAssignCarrierToBuffer(carrier.id)}
                          data-testid={`button-assign-carrier-${carrier.id}`}
                        >
                          <div className="flex items-center gap-2">
                            <Package className="h-4 w-4 text-muted-foreground" />
                            <span className="font-medium">{carrier.code}</span>
                            <span className="text-sm text-muted-foreground">{carrier.name}</span>
                            {carrier.groupName && (
                              <Badge variant="outline" className="text-[10px]">{carrier.groupName}</Badge>
                            )}
                          </div>
                          <Plus className="h-4 w-4 text-green-500" />
                        </div>
                      ))}
                    </div>
                  </ScrollArea>
                );
              })()}
            </div>
          </div>
          
          <DialogFooter>
            <Button variant="outline" onClick={() => setBufferCarrierModalOpen(false)} data-testid="button-close-buffer-modal">
              Zamknij
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      {/* Work Order Carrier Assignment Modal */}
      <Dialog open={carrierModalOpen} onOpenChange={setCarrierModalOpen}>
        <DialogContent className="sm:max-w-lg" data-testid="dialog-work-order-carrier">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Package className="h-5 w-5 text-purple-600" />
              Przypisz nośnik do operacji
            </DialogTitle>
            <DialogDescription>
              {selectedWorkOrderForCarrier?.operationName || selectedWorkOrderForCarrier?.workOrderNumber}
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-2">
            {/* Currently assigned carrier */}
            {selectedWorkOrderForCarrier?.carrierId && (
              <div className="space-y-2">
                <Label className="text-sm font-medium">Aktualnie przypisany</Label>
                <div className="flex items-center justify-between p-2 border rounded bg-purple-50 dark:bg-purple-900/20">
                  <div className="flex items-center gap-2">
                    <Box className="h-4 w-4 text-purple-600" />
                    <span className="font-medium">{selectedWorkOrderForCarrier.carrierCode}</span>
                    <span className="text-sm text-muted-foreground">{selectedWorkOrderForCarrier.carrierName}</span>
                  </div>
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={() => {
                      if (selectedWorkOrderForCarrier) {
                        assignCarrierToWorkOrderMutation.mutate({
                          workOrderId: selectedWorkOrderForCarrier.id,
                          carrierId: null,
                        });
                      }
                    }}
                    disabled={assignCarrierToWorkOrderMutation.isPending}
                    data-testid="button-remove-work-order-carrier"
                  >
                    <XCircle className="h-3 w-3 mr-1" />
                    Usuń
                  </Button>
                </div>
              </div>
            )}
            
            <div className={selectedWorkOrderForCarrier?.carrierId ? "border-t pt-4" : ""}>
              <Label className="text-sm font-medium">Dostępne nośniki</Label>
              {availableCarriers.length === 0 ? (
                <div className="text-sm text-muted-foreground py-2 px-3 bg-muted rounded mt-2">
                  Brak dostępnych nośników do przypisania
                </div>
              ) : (
                <ScrollArea className="max-h-[250px] mt-2">
                  <div className="space-y-1">
                    {availableCarriers.map(carrier => (
                      <div 
                        key={carrier.id}
                        className={`flex items-center justify-between p-2 border rounded bg-card hover-elevate cursor-pointer ${
                          selectedWorkOrderForCarrier?.carrierId === carrier.id 
                            ? 'border-purple-500 bg-purple-50 dark:bg-purple-900/20' 
                            : ''
                        }`}
                        onClick={() => {
                          if (selectedWorkOrderForCarrier) {
                            assignCarrierToWorkOrderMutation.mutate({
                              workOrderId: selectedWorkOrderForCarrier.id,
                              carrierId: carrier.id,
                            });
                          }
                        }}
                        data-testid={`button-assign-wo-carrier-${carrier.id}`}
                      >
                        <div className="flex items-center gap-2">
                          <Package className="h-4 w-4 text-muted-foreground" />
                          <span className="font-medium">{carrier.code}</span>
                          <span className="text-sm text-muted-foreground">{carrier.name}</span>
                          {carrier.groupName && (
                            <Badge variant="outline" className="text-[10px]">{carrier.groupName}</Badge>
                          )}
                        </div>
                        {assignCarrierToWorkOrderMutation.isPending ? (
                          <Loader2 className="h-4 w-4 animate-spin text-muted-foreground" />
                        ) : (
                          <Plus className="h-4 w-4 text-purple-500" />
                        )}
                      </div>
                    ))}
                  </div>
                </ScrollArea>
              )}
            </div>
          </div>
          
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setCarrierModalOpen(false);
              setSelectedWorkOrderForCarrier(null);
            }} data-testid="button-close-carrier-modal">
              Zamknij
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      <Dialog open={operationFormatkiModalOpen} onOpenChange={setOperationFormatkiModalOpen}>
        <DialogContent className="max-w-2xl max-h-[80vh] overflow-hidden flex flex-col" data-testid="dialog-operation-formatki">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Boxes className="h-5 w-5" />
              Formatki w operacji
            </DialogTitle>
            <DialogDescription>
              {selectedWorkOrderForFormatki?.operationName || selectedWorkOrderForFormatki?.workOrderNumber}
            </DialogDescription>
          </DialogHeader>
          
          <div className="flex-1 overflow-hidden">
            {allFormatkiData ? (
              <>
                {/* Podsumowanie */}
                {(() => {
                  const filteredFormatki = allFormatkiData.filter(f => {
                    const wo = selectedWorkOrderForFormatki;
                    if (!wo) return false;
                    
                    // Używaj operationCode (np. "assembly") zamiast operationName (np. "Kompletowanie")
                    const opCode = wo.operationCode?.toLowerCase() || '';
                    const opName = wo.operationName?.toLowerCase() || '';
                    
                    // Dla operacji magazynowania - pokaż WSZYSTKIE formatki (to jest shared operation)
                    if (wo.isWarehouseOperation || opCode === 'magazynowanie' || opName.includes('magazyn')) {
                      return true;
                    }
                    
                    // ZAWSZE najpierw sprawdź czy formatka należy do tej samej marszruty co work order
                    // Każda marszruta (CW, CO, COW, SZUFLADY) ma własne operacje assembly/packing
                    // Work order ma routingCode (np. "marsz_formatki_cw", "marsz_formatki_co", "marsz_szuflady")
                    // Formatka ma routingCode z wariantu (np. "CW", "CO", "COW", "BOCZKI-TYL-SZUFLAD")
                    if (wo.routingCode && f.routingCode) {
                      const woRoutingLower = wo.routingCode.toLowerCase();
                      const fRoutingLower = f.routingCode.toLowerCase();
                      
                      // Specjalne mapowania gdzie kod wariantu nie odpowiada sufixowi marszruty
                      const specialMappings: Record<string, string[]> = {
                        'marsz_szuflady': ['boczki-tyl-szuflad', 'szuflady'],
                        'marsz_siedziska_tapicerowane': ['cwtap'],
                        'marsz_panele': ['ctap'],
                      };
                      
                      // Sprawdź specjalne mapowania
                      const mappedVariants = specialMappings[woRoutingLower];
                      const specialMatch = mappedVariants && mappedVariants.includes(fRoutingLower);
                      
                      // Standardowe dopasowanie - routing kończy się na "_" + kod wariantu
                      const routingMatches = specialMatch ||
                                            woRoutingLower.endsWith(`_${fRoutingLower}`) ||
                                            fRoutingLower === woRoutingLower;
                      
                      if (!routingMatches) {
                        return false;
                      }
                    }
                    
                    // Dla operacji końcowych (assembly, packing) - po filtrze routing, pokaż wszystkie
                    if (opCode === 'assembly' || opCode === 'packing' || opCode === 'kompletowanie' || opCode === 'pakowanie') {
                      return true;
                    }
                    
                    // Dla innych operacji (cutting, drilling, edging itp.) filtruj po requiredOperations
                    if (!opCode && !opName) return false;
                    
                    // Mapowanie aliasów operacji (opCode -> możliwe warianty w requiredOperations)
                    const operationAliases: Record<string, string[]> = {
                      'drilling': ['drilling', 'drilling_holes', 'drilling_mount', 'wiercenie'],
                      'wiercenie 2': ['drilling', 'drilling_holes', 'drilling_mount', 'wiercenie'],
                      'cutting': ['cutting', 'ciecie', 'cięcie'],
                      'edging': ['edging', 'oklejanie', 'edgebanding'],
                      'upholstering': ['upholstering', 'tapicerowanie'],
                      'assembly': ['assembly', 'kompletowanie', 'montaz'],
                      'packing': ['packing', 'pakowanie'],
                    };
                    
                    const aliases = operationAliases[opCode] || [opCode];
                    
                    return f.requiredOperations?.some(op => {
                      const opLower = op.toLowerCase();
                      return aliases.includes(opLower) || 
                             opLower === opCode || 
                             opLower.includes(opName) ||
                             opName.includes(opLower);
                    }) ?? false;
                  });
                  const totalQuantity = filteredFormatki.reduce((sum, f) => sum + (f.quantity || 0), 0);
                  const damagedCount = filteredFormatki.filter(f => f.isDamaged || f.itemStatus === 'damaged').length;
                  
                  const selectableFormatki = filteredFormatki.filter(f => !f.isDamaged && f.itemStatus !== 'damaged' && !f.lastWarehouseDocId);
                  const allSelected = selectableFormatki.length > 0 && selectableFormatki.every(f => selectedFormatkiIds.has(f.id));
                  const someSelected = selectableFormatki.some(f => selectedFormatkiIds.has(f.id));
                  
                  return (
                    <div className="flex flex-wrap items-center gap-3 mb-3 p-2 bg-muted rounded-lg text-sm">
                      <div className="flex items-center gap-1.5">
                        <Boxes className="h-4 w-4 text-muted-foreground" />
                        <span className="font-medium">{filteredFormatki.length}</span>
                        <span className="text-muted-foreground">pozycji</span>
                      </div>
                      <div className="flex items-center gap-1.5">
                        <span className="text-muted-foreground">Łącznie:</span>
                        <span className="font-medium">{totalQuantity}</span>
                        <span className="text-muted-foreground">szt</span>
                      </div>
                      {damagedCount > 0 && (
                        <div className="flex items-center gap-1.5 text-red-600">
                          <AlertTriangle className="h-4 w-4" />
                          <span className="font-medium">{damagedCount}</span>
                          <span>uszkodzonych</span>
                        </div>
                      )}
                      
                      {/* Przyciski bulk zaznaczania */}
                      <div className="flex items-center gap-2 ml-auto">
                        {selectedFormatkiIds.size > 0 && (
                          <Badge variant="secondary" className="text-xs">
                            <Warehouse className="h-3 w-3 mr-1" />
                            {selectedFormatkiIds.size} zaznaczonych
                          </Badge>
                        )}
                        <Button
                          size="sm"
                          variant="outline"
                          className="h-7 text-xs"
                          onClick={() => {
                            if (allSelected) {
                              setSelectedFormatkiIds(new Set());
                            } else {
                              setSelectedFormatkiIds(new Set(selectableFormatki.map(f => f.id)));
                            }
                          }}
                          data-testid="button-toggle-select-all-formatki"
                        >
                          {allSelected ? 'Odznacz wszystkie' : 'Zaznacz wszystkie'}
                        </Button>
                      </div>
                    </div>
                  );
                })()}
                
                <ScrollArea className="h-[360px] pr-2">
                  <div className="space-y-2">
                    {allFormatkiData
                      .filter(f => {
                        const wo = selectedWorkOrderForFormatki;
                        if (!wo) return false;
                        
                        const opCode = wo.operationCode?.toLowerCase() || '';
                        const opName = wo.operationName?.toLowerCase() || '';
                        
                        // Dla operacji magazynowania - pokaż WSZYSTKIE formatki (to jest shared operation)
                        if (wo.isWarehouseOperation || opCode === 'magazynowanie' || opName.includes('magazyn')) {
                          return true;
                        }
                        
                        // ZAWSZE najpierw sprawdź czy formatka należy do tej samej marszruty co work order
                        // Każda marszruta (CW, CO, COW, SZUFLADY) ma własne operacje assembly/packing
                        if (wo.routingCode && f.routingCode) {
                          const woRoutingLower = wo.routingCode.toLowerCase();
                          const fRoutingLower = f.routingCode.toLowerCase();
                          
                          // Specjalne mapowania gdzie kod wariantu nie odpowiada sufixowi marszruty
                          const specialMappings: Record<string, string[]> = {
                            'marsz_szuflady': ['boczki-tyl-szuflad', 'szuflady'],
                            'marsz_siedziska_tapicerowane': ['cwtap'],
                            'marsz_panele': ['ctap'],
                          };
                          
                          const mappedVariants = specialMappings[woRoutingLower];
                          const specialMatch = mappedVariants && mappedVariants.includes(fRoutingLower);
                          
                          const routingMatches = specialMatch ||
                                                woRoutingLower.endsWith(`_${fRoutingLower}`) ||
                                                fRoutingLower === woRoutingLower;
                          
                          if (!routingMatches) {
                            return false;
                          }
                        }
                        
                        // Dla operacji końcowych (assembly, packing) - po filtrze routing, pokaż wszystkie
                        if (opCode === 'assembly' || opCode === 'packing' || opCode === 'kompletowanie' || opCode === 'pakowanie') {
                          return true;
                        }
                        
                        // Dla innych operacji filtruj po requiredOperations
                        if (!opCode && !opName) return false;
                        
                        // Mapowanie aliasów operacji
                        const operationAliases: Record<string, string[]> = {
                          'drilling': ['drilling', 'drilling_holes', 'drilling_mount', 'wiercenie'],
                          'wiercenie 2': ['drilling', 'drilling_holes', 'drilling_mount', 'wiercenie'],
                          'cutting': ['cutting', 'ciecie', 'cięcie'],
                          'edging': ['edging', 'oklejanie', 'edgebanding'],
                          'upholstering': ['upholstering', 'tapicerowanie'],
                          'assembly': ['assembly', 'kompletowanie', 'montaz'],
                          'packing': ['packing', 'pakowanie'],
                        };
                        
                        const aliases = operationAliases[opCode] || [opCode];
                        
                        return f.requiredOperations?.some(op => {
                          const opLower = op.toLowerCase();
                          return aliases.includes(opLower) || 
                                 opLower === opCode || 
                                 opLower.includes(opName) ||
                                 opName.includes(opLower);
                        }) ?? false;
                      })
                      .map((formatka, index) => {
                        const isOnWarehouse = !!formatka.lastWarehouseDocId;
                        const isDamaged = formatka.itemStatus === 'damaged' || formatka.isDamaged;
                        
                        return (
                        <div 
                          key={formatka.id} 
                          className={`p-3 border rounded-lg ${isDamaged ? 'border-red-300 bg-red-50 dark:bg-red-900/20 dark:border-red-700' : isOnWarehouse ? 'border-green-300 bg-green-50 dark:bg-green-900/20 dark:border-green-700' : 'bg-card'}`}
                        >
                          <div className="flex items-start justify-between gap-2">
                            <div className="flex items-start gap-3">
                              {/* Checkbox do zaznaczania formatki - ukryty dla uszkodzonych i już dodanych na magazyn */}
                              {!isDamaged && !isOnWarehouse && (
                                <Checkbox
                                  checked={selectedFormatkiIds.has(formatka.id)}
                                  onCheckedChange={(checked) => {
                                    setSelectedFormatkiIds(prev => {
                                      const newSet = new Set(prev);
                                      if (checked) {
                                        newSet.add(formatka.id);
                                      } else {
                                        newSet.delete(formatka.id);
                                      }
                                      return newSet;
                                    });
                                  }}
                                  className="mt-1.5 shrink-0"
                                  data-testid={`checkbox-formatka-${formatka.id}`}
                                />
                              )}
                              <div className="flex items-center justify-center w-7 h-7 rounded-full bg-primary/10 text-primary text-xs font-bold shrink-0">
                                {index + 1}
                              </div>
                              <div className="flex-1">
                                <div className="font-medium text-sm">{formatka.formatkaName}</div>
                                <div className="text-xs text-muted-foreground">
                                  {formatka.length} x {formatka.width} x {formatka.thickness}mm
                                </div>
                                <div className="flex items-center gap-2 mt-1 flex-wrap">
                                  {formatka.colorCode && (
                                    <Badge variant="outline" className="text-[10px]">{formatka.colorCode}</Badge>
                                  )}
                                  <span className="text-xs">Ilość: {formatka.quantity}</span>
                                  {formatka.routingCode && (
                                    <Badge variant="secondary" className="text-[10px]">{formatka.routingCode}</Badge>
                                  )}
                                  {isOnWarehouse && formatka.lastWarehouseDocId && (
                                    <Link 
                                      href={`/warehouse/documents/${formatka.lastWarehouseDocId}`}
                                      onClick={(e) => e.stopPropagation()}
                                    >
                                      <Badge variant="default" className="text-[10px] bg-green-600 hover:bg-green-700 cursor-pointer">
                                        <Package className="h-3 w-3 mr-1" />
                                        Na magazynie ({formatka.warehouseDocNumber})
                                      </Badge>
                                    </Link>
                                  )}
                                </div>
                              </div>
                            </div>
                          
                            <div className="flex items-center gap-1 shrink-0 flex-wrap justify-end">
                            {/* Drilling button - only for drilling operations */}
                            {(() => {
                              const wo = selectedWorkOrderForFormatki;
                              const opCode = wo?.operationCode?.toLowerCase() || '';
                              const opName = wo?.operationName?.toLowerCase() || '';
                              const isDrillingOp = opCode.includes('drill') || opCode.includes('wierc') || 
                                                   opName.includes('drill') || opName.includes('wierc');
                              if (!isDrillingOp) return null;
                              
                              return formatka.isDrilled ? (
                                <div className="flex flex-col items-end gap-0.5">
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    className="h-7 text-xs text-green-600 hover:text-green-700 hover:bg-green-50"
                                    onClick={() => markDrilledMutation.mutate({ itemId: formatka.id, isDrilled: false })}
                                    disabled={markDrilledMutation.isPending}
                                    data-testid={`button-unmark-drilled-${formatka.id}`}
                                  >
                                    {markDrilledMutation.isPending ? (
                                      <Loader2 className="h-3 w-3 animate-spin" />
                                    ) : (
                                      <>
                                        <Drill className="h-3 w-3 mr-1" />
                                        Nawiercona
                                      </>
                                    )}
                                  </Button>
                                  {(formatka.drilledAt || formatka.drilledByUsername) && (
                                    <div className="text-[10px] text-muted-foreground text-right">
                                      {formatka.drilledAt && (
                                        <span>{format(new Date(formatka.drilledAt), "dd.MM HH:mm", { locale: pl })}</span>
                                      )}
                                      {formatka.drilledByUsername && (
                                        <span className="ml-1">({formatka.drilledByUsername})</span>
                                      )}
                                    </div>
                                  )}
                                </div>
                              ) : (
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="h-7 text-xs text-blue-600 hover:text-blue-700 hover:bg-blue-50"
                                  onClick={() => markDrilledMutation.mutate({ itemId: formatka.id, isDrilled: true })}
                                  disabled={markDrilledMutation.isPending}
                                  data-testid={`button-mark-drilled-${formatka.id}`}
                                >
                                  {markDrilledMutation.isPending ? (
                                    <Loader2 className="h-3 w-3 animate-spin" />
                                  ) : (
                                    <>
                                      <Drill className="h-3 w-3 mr-1" />
                                      Nawierć
                                    </>
                                  )}
                                </Button>
                              );
                            })()}
                            
                            {/* Damage button - ukryty dla formatek już na magazynie */}
                            {!isOnWarehouse && (
                              isDamaged ? (
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="h-7 text-xs"
                                  onClick={() => handleRemoveDamage(formatka.id)}
                                  disabled={markDamageMutation.isPending}
                                  data-testid={`button-remove-damage-${formatka.id}`}
                                >
                                  {markDamageMutation.isPending ? (
                                    <Loader2 className="h-3 w-3 animate-spin" />
                                  ) : (
                                    <>
                                      <Trash2 className="h-3 w-3 mr-1" />
                                      Usuń uszkodzenie
                                    </>
                                  )}
                                </Button>
                              ) : (
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="h-7 text-xs text-orange-600 hover:text-orange-700 hover:bg-orange-50"
                                  onClick={() => setSelectedDamageFormatka({ id: formatka.id, name: formatka.formatkaName })}
                                  data-testid={`button-mark-damage-${formatka.id}`}
                                >
                                  <AlertTriangle className="h-3 w-3 mr-1" />
                                  Oznacz uszkodzenie
                                </Button>
                              )
                            )}
                          </div>
                        </div>
                      </div>
                    );
                    })}
                  
                  {allFormatkiData.filter(f => {
                    const wo = selectedWorkOrderForFormatki;
                    if (!wo) return false;
                    
                    const opCode = wo.operationCode?.toLowerCase() || '';
                    const opName = wo.operationName?.toLowerCase() || '';
                    
                    // Dla operacji magazynowania
                    if (wo.isWarehouseOperation || opCode === 'magazynowanie' || opName.includes('magazyn')) {
                      return true;
                    }
                    
                    // Dla operacji końcowych
                    if (opCode === 'assembly' || opCode === 'packing' || opCode === 'kompletowanie' || opCode === 'pakowanie') {
                      return true;
                    }
                    
                    if (!opCode && !opName) return false;
                    
                    const operationAliases: Record<string, string[]> = {
                      'drilling': ['drilling', 'drilling_holes', 'drilling_mount', 'wiercenie'],
                      'cutting': ['cutting', 'ciecie', 'cięcie'],
                      'edging': ['edging', 'oklejanie', 'edgebanding'],
                      'upholstering': ['upholstering', 'tapicerowanie'],
                      'assembly': ['assembly', 'kompletowanie', 'montaz'],
                      'packing': ['packing', 'pakowanie'],
                    };
                    
                    const aliases = operationAliases[opCode] || [opCode];
                    
                    return f.requiredOperations?.some(op => {
                      const opLower = op.toLowerCase();
                      return aliases.includes(opLower) || 
                             opLower === opCode || 
                             opLower.includes(opName) ||
                             opName.includes(opLower);
                    }) ?? false;
                  }).length === 0 && (
                    <div className="text-center text-muted-foreground py-8">
                      Brak formatek przypisanych do tej operacji
                    </div>
                  )}
                  </div>
                </ScrollArea>
              </>
            ) : (
              <div className="flex items-center justify-center h-32">
                <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
              </div>
            )}
          </div>
          
          {selectedDamageFormatka && (
            <div className="border-t pt-4 mt-4 space-y-3">
              <div className="font-medium text-sm flex items-center gap-2">
                <AlertTriangle className="h-4 w-4 text-orange-500" />
                Oznacz uszkodzenie: {selectedDamageFormatka.name}
              </div>
              
              <div className="space-y-2">
                <Label>Typ uszkodzenia</Label>
                <Select value={damageType} onValueChange={setDamageType}>
                  <SelectTrigger data-testid="select-damage-type">
                    <SelectValue placeholder="Wybierz typ uszkodzenia" />
                  </SelectTrigger>
                  <SelectContent>
                    {scrapReasons && scrapReasons.length > 0 ? (
                      scrapReasons.map((reason) => (
                        <SelectItem key={reason.code} value={reason.code}>
                          {reason.readableName || reason.name}
                        </SelectItem>
                      ))
                    ) : (
                      <>
                        <SelectItem value="BLAD_CIECIA">Błąd cięcia</SelectItem>
                        <SelectItem value="WADA_PLYTY">Wada płyty</SelectItem>
                        <SelectItem value="USZKODZENIE">Uszkodzenie mechaniczne</SelectItem>
                        <SelectItem value="ZLY_WYMIAR">Nieprawidłowy wymiar</SelectItem>
                        <SelectItem value="ODPRYSK">Odpryski na krawędzi</SelectItem>
                        <SelectItem value="PRZEBARWIENIE">Przebarwienie</SelectItem>
                        <SelectItem value="INNE">Inne</SelectItem>
                      </>
                    )}
                  </SelectContent>
                </Select>
              </div>
              
              <div className="space-y-2">
                <Label>Notatki (opcjonalne)</Label>
                <Input
                  value={damageNotes}
                  onChange={(e) => setDamageNotes(e.target.value)}
                  placeholder="Opisz uszkodzenie..."
                  data-testid="input-damage-notes"
                />
              </div>
              
              <div className="flex gap-2">
                <Button
                  variant="outline"
                  onClick={() => {
                    setSelectedDamageFormatka(null);
                    setDamageType('');
                    setDamageNotes('');
                  }}
                  data-testid="button-cancel-damage"
                >
                  Anuluj
                </Button>
                <Button
                  className="bg-orange-600 hover:bg-orange-700"
                  onClick={handleMarkDamage}
                  disabled={!damageType || markDamageMutation.isPending}
                  data-testid="button-confirm-damage"
                >
                  {markDamageMutation.isPending ? (
                    <Loader2 className="h-4 w-4 animate-spin mr-2" />
                  ) : (
                    <AlertTriangle className="h-4 w-4 mr-2" />
                  )}
                  Oznacz jako uszkodzony
                </Button>
              </div>
            </div>
          )}
          
          <DialogFooter className="gap-2 flex-wrap">
            <Button variant="outline" onClick={() => {
              setOperationFormatkiModalOpen(false);
              setSelectedDamageFormatka(null);
              setDamageType('');
              setDamageNotes('');
              setSelectedFormatkiIds(new Set());
            }} data-testid="button-close-operation-formatki">
              Zamknij
            </Button>
            
            {/* Wybór lokalizacji i przycisk dodawania na magazyn */}
            {selectedFormatkiIds.size > 0 && selectedWorkOrderForFormatki && (
              <div className="flex items-center gap-2">
                <Combobox
                  options={warehouseLocations.map((loc) => ({
                    value: loc.id.toString(),
                    label: loc.name,
                  }))}
                  value={selectedWarehouseLocationId?.toString() || ""}
                  onChange={(val) => setSelectedWarehouseLocationId(val ? parseInt(val) : null)}
                  placeholder="Lokalizacja..."
                  searchPlaceholder="Szukaj lokalizacji..."
                  emptyText="Brak lokalizacji"
                  className="w-[200px]"
                  side="top"
                />
                <Button
                  onClick={() => {
                    if (selectedWorkOrderForFormatki) {
                      warehouseIntakeMutation.mutate({
                        bomItemIds: Array.from(selectedFormatkiIds),
                        workOrderId: selectedWorkOrderForFormatki.id,
                        targetLocationId: selectedWarehouseLocationId || undefined,
                      });
                    }
                  }}
                  disabled={warehouseIntakeMutation.isPending}
                  className="bg-emerald-600 hover:bg-emerald-700"
                  data-testid="button-add-formatki-to-warehouse"
                >
                  {warehouseIntakeMutation.isPending ? (
                    <Loader2 className="h-4 w-4 animate-spin mr-2" />
                  ) : (
                    <Warehouse className="h-4 w-4 mr-2" />
                  )}
                  Dodaj na magazyn ({selectedFormatkiIds.size})
                </Button>
              </div>
            )}
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      <Dialog open={operatorModalOpen} onOpenChange={setOperatorModalOpen}>
        <DialogContent className="max-w-md" data-testid="dialog-edit-operators">
          <DialogHeader>
            <DialogTitle>Edytuj operatorów</DialogTitle>
            <DialogDescription>
              {editingWorkOrder?.operationName || editingWorkOrder?.workOrderNumber}
            </DialogDescription>
          </DialogHeader>
          
          <ScrollArea className="max-h-[300px] pr-3">
            <div className="space-y-2">
              {operators.length === 0 ? (
                <div className="text-sm text-muted-foreground text-center py-4">
                  Brak dostępnych operatorów
                </div>
              ) : (
                operators.map((operator) => {
                  const isSelected = selectedOperatorIds.includes(operator.id);
                  const isPrimary = primaryOperatorId === operator.id;
                  
                  return (
                    <div 
                      key={operator.id}
                      className={`flex items-center justify-between p-2 rounded-lg border transition-colors ${isSelected ? 'bg-blue-50 dark:bg-blue-900/20 border-blue-200 dark:border-blue-800' : 'bg-card'}`}
                    >
                      <div className="flex items-center gap-3">
                        <Checkbox
                          id={`operator-${operator.id}`}
                          checked={isSelected}
                          onCheckedChange={() => toggleOperator(operator.id)}
                          data-testid={`checkbox-operator-${operator.id}`}
                        />
                        <div>
                          <Label 
                            htmlFor={`operator-${operator.id}`}
                            className="font-medium cursor-pointer"
                          >
                            {operator.full_name}
                          </Label>
                          {operator.short_code && (
                            <div className="text-xs text-muted-foreground">{operator.short_code}</div>
                          )}
                        </div>
                      </div>
                      
                      {isSelected && (
                        <Button
                          size="sm"
                          variant={isPrimary ? "default" : "outline"}
                          className="h-6 text-xs"
                          onClick={() => setPrimaryOperatorId(isPrimary ? null : operator.id)}
                          data-testid={`button-primary-${operator.id}`}
                        >
                          {isPrimary ? "Główny" : "Ustaw głównym"}
                        </Button>
                      )}
                    </div>
                  );
                })
              )}
            </div>
          </ScrollArea>
          
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setOperatorModalOpen(false)} data-testid="button-cancel-operators">
              Anuluj
            </Button>
            <Button 
              onClick={handleSaveOperators}
              disabled={updateOperatorsMutation.isPending}
              data-testid="button-save-operators"
            >
              {updateOperatorsMutation.isPending ? (
                <Loader2 className="h-4 w-4 animate-spin mr-2" />
              ) : (
                <Users className="h-4 w-4 mr-2" />
              )}
              Zapisz
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      <Dialog open={operationDetailsModalOpen} onOpenChange={setOperationDetailsModalOpen}>
        <DialogContent className="max-w-2xl max-h-[85vh]" data-testid="dialog-operation-details">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Info className="h-5 w-5" />
              Szczegóły operacji: {detailsWorkOrder?.operationName || detailsWorkOrder?.workOrderNumber}
            </DialogTitle>
            <DialogDescription>
              {detailsWorkOrder?.operationCode} - {detailsWorkOrders.length > 1 ? `${detailsWorkOrders.length} work orders` : 'Pojedyncza operacja'}
            </DialogDescription>
          </DialogHeader>
          
          <ScrollArea className="max-h-[60vh] pr-3">
            <div className="space-y-4">
              {/* Podstawowe informacje */}
              <div className="grid grid-cols-2 gap-3">
                <div className="p-3 bg-muted rounded-lg">
                  <div className="text-xs text-muted-foreground mb-1">Status</div>
                  <Badge variant="outline" className={`
                    ${detailsWorkOrder?.status === 'done' ? 'bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400' : ''}
                    ${detailsWorkOrder?.status === 'in_progress' ? 'bg-yellow-100 text-yellow-700 dark:bg-yellow-900/30 dark:text-yellow-400' : ''}
                    ${detailsWorkOrder?.status === 'pending' || detailsWorkOrder?.status === 'ready' ? 'bg-gray-100 text-gray-700 dark:bg-gray-800 dark:text-gray-400' : ''}
                  `}>
                    {detailsWorkOrder?.status === 'done' ? 'Zakończone' :
                     detailsWorkOrder?.status === 'in_progress' ? 'W trakcie' :
                     detailsWorkOrder?.status === 'ready' ? 'Gotowe' : 'Oczekuje'}
                  </Badge>
                </div>
                <div className="p-3 bg-muted rounded-lg">
                  <div className="text-xs text-muted-foreground mb-1">Stanowisko</div>
                  <div className="font-medium text-sm">{detailsWorkOrder?.workCenterName || 'Nie przypisano'}</div>
                </div>
                <div className="p-3 bg-muted rounded-lg">
                  <div className="text-xs text-muted-foreground mb-1">Ilość zaplanowana</div>
                  <div className="font-medium text-sm">{detailsWorkOrder?.quantityPlanned || 0}</div>
                </div>
                <div className="p-3 bg-muted rounded-lg">
                  <div className="text-xs text-muted-foreground mb-1">Ilość wykonana</div>
                  <div className="font-medium text-sm">{detailsWorkOrder?.quantityProduced || 0}</div>
                </div>
              </div>
              
              {/* Operatorzy */}
              <div className="p-3 bg-muted rounded-lg">
                <div className="text-xs text-muted-foreground mb-2">Przypisani operatorzy</div>
                <div className="flex flex-wrap gap-2">
                  {detailsWorkOrders.flatMap(wo => wo.assignedOperators || [])
                    .filter((op, i, arr) => arr.findIndex(o => o.operatorId === op.operatorId) === i)
                    .map(op => (
                      <Badge key={op.operatorId} variant={op.isPrimary ? "default" : "outline"}>
                        {op.operatorName}
                        {op.isPrimary && <span className="ml-1 text-[10px]">(główny)</span>}
                      </Badge>
                    )) || <span className="text-sm text-muted-foreground">Brak operatorów</span>}
                </div>
              </div>
              
              {/* Czasy */}
              {(detailsWorkOrder?.actualStartTime || detailsWorkOrder?.actualEndTime) && (
                <div className="p-3 bg-muted rounded-lg">
                  <div className="text-xs text-muted-foreground mb-2">Czasy wykonania</div>
                  <div className="grid grid-cols-2 gap-2 text-sm">
                    {detailsWorkOrder?.actualStartTime && (
                      <div>
                        <span className="text-muted-foreground">Start: </span>
                        {format(new Date(detailsWorkOrder.actualStartTime), 'dd.MM.yyyy HH:mm', { locale: pl })}
                      </div>
                    )}
                    {detailsWorkOrder?.actualEndTime && (
                      <div>
                        <span className="text-muted-foreground">Koniec: </span>
                        {format(new Date(detailsWorkOrder.actualEndTime), 'dd.MM.yyyy HH:mm', { locale: pl })}
                      </div>
                    )}
                  </div>
                </div>
              )}
              
              {/* Notatki */}
              {detailsWorkOrder?.notes && (
                <div className="p-3 bg-muted rounded-lg">
                  <div className="text-xs text-muted-foreground mb-1">Notatki</div>
                  <div className="text-sm">{detailsWorkOrder.notes}</div>
                </div>
              )}
              
              {/* Lista work orders dla wspólnych operacji */}
              {detailsWorkOrders.length > 1 && (
                <div className="p-3 bg-muted rounded-lg">
                  <div className="text-xs text-muted-foreground mb-2">Work orders ({detailsWorkOrders.length})</div>
                  <div className="space-y-1.5">
                    {detailsWorkOrders.map(wo => (
                      <div key={wo.id} className="flex items-center justify-between text-sm p-2 bg-background rounded">
                        <span className="font-mono text-xs">{wo.workOrderNumber}</span>
                        <Badge variant="outline" className={`text-[10px] ${
                          wo.status === 'done' ? 'bg-green-100 text-green-700 dark:bg-green-900/30' :
                          wo.status === 'in_progress' ? 'bg-yellow-100 text-yellow-700 dark:bg-yellow-900/30' : ''
                        }`}>
                          {wo.routingCode || wo.routingName}
                        </Badge>
                      </div>
                    ))}
                  </div>
                </div>
              )}
              
              {/* Formatki biorące udział w operacji - filtrowane według operacji */}
              {allFormatkiData && allFormatkiData.length > 0 && (
                (() => {
                  const operationName = detailsWorkOrder?.operationName || '';
                  const isSharedOperation = detailsWorkOrders.length > 1;
                  
                  // Filter formatki that have this operation in their required operations
                  const filteredFormatki = allFormatkiData.filter(f => {
                    if (!f.requiredOperations || f.requiredOperations.length === 0) return true;
                    return f.requiredOperations.some(op => 
                      op.toLowerCase().includes(operationName.toLowerCase()) ||
                      operationName.toLowerCase().includes(op.toLowerCase())
                    );
                  });
                  
                  // For shared operations, group by routing code
                  const groupedByPath = isSharedOperation 
                    ? filteredFormatki.reduce((acc: Record<string, BomItem[]>, f) => {
                        const path = f.routingCode || 'unknown';
                        if (!acc[path]) acc[path] = [];
                        acc[path].push(f);
                        return acc;
                      }, {})
                    : null;
                  
                  // For individual operations, filter by routing code
                  const operationFormatki = isSharedOperation 
                    ? filteredFormatki 
                    : filteredFormatki.filter(f => {
                        if (!detailsWorkOrder?.routingCode) return true;
                        return f.routingCode === detailsWorkOrder.routingCode;
                      });
                  
                  if (operationFormatki.length === 0) return null;
                  
                  return (
                    <div className="p-3 bg-muted rounded-lg">
                      <div className="text-xs text-muted-foreground mb-2 flex items-center gap-2">
                        <Layers className="h-3.5 w-3.5" />
                        Formatki dla operacji ({operationFormatki.length})
                      </div>
                      
                      {isSharedOperation && groupedByPath ? (
                        // Grouped view for shared operations
                        <div className="space-y-3">
                          {Object.entries(groupedByPath).map(([pathCode, items]) => (
                            <div key={pathCode} className="space-y-1">
                              <div className="flex items-center gap-2 mb-1">
                                <Badge className={`text-[10px] ${flowCodeColors[pathCode]?.bg || 'bg-gray-500'}`}>
                                  {pathCode}
                                </Badge>
                                <span className="text-[10px] text-muted-foreground">{items.length} szt</span>
                              </div>
                              <div className="grid grid-cols-6 gap-1 text-[10px] font-medium text-muted-foreground p-1.5 bg-background rounded-t border-b">
                                <span>Nazwa</span>
                                <span>Kod</span>
                                <span className="text-center">Wymiary</span>
                                <span className="text-center">Grubość</span>
                                <span className="text-center">Kolor</span>
                                <span className="text-center">Ilość</span>
                              </div>
                              <ScrollArea className="max-h-[100px]">
                                {items.map((f) => (
                                  <div key={f.id} className="grid grid-cols-6 gap-1 text-[10px] p-1.5 hover:bg-background/50 border-b border-border/30">
                                    <span className="font-medium truncate" title={f.formatkaName}>{f.formatkaName}</span>
                                    <span className="text-muted-foreground font-mono truncate" title={f.formatkaCode}>{f.formatkaCode}</span>
                                    <span className="text-center">{f.length}x{f.width}</span>
                                    <span className="text-center">{f.thickness}mm</span>
                                    <span className="text-center">
                                      <Badge variant="outline" className="text-[9px] px-1">{f.colorCode || '-'}</Badge>
                                    </span>
                                    <span className="text-center font-medium">{f.quantity}</span>
                                  </div>
                                ))}
                              </ScrollArea>
                            </div>
                          ))}
                        </div>
                      ) : (
                        // Simple list for individual operations
                        <div className="space-y-1">
                          {detailsWorkOrder?.routingCode && (
                            <div className="flex items-center gap-2 mb-1">
                              <Badge className={`text-[10px] ${flowCodeColors[detailsWorkOrder.routingCode]?.bg || 'bg-gray-500'}`}>
                                {detailsWorkOrder.routingCode}
                              </Badge>
                            </div>
                          )}
                          <div className="grid grid-cols-6 gap-1 text-[10px] font-medium text-muted-foreground p-1.5 bg-background rounded-t border-b">
                            <span>Nazwa</span>
                            <span>Kod</span>
                            <span className="text-center">Wymiary</span>
                            <span className="text-center">Grubość</span>
                            <span className="text-center">Kolor</span>
                            <span className="text-center">Ilość</span>
                          </div>
                          <ScrollArea className="max-h-[150px]">
                            {operationFormatki.map((f) => (
                              <div key={f.id} className="grid grid-cols-6 gap-1 text-[10px] p-1.5 hover:bg-background/50 border-b border-border/30">
                                <span className="font-medium truncate" title={f.formatkaName}>{f.formatkaName}</span>
                                <span className="text-muted-foreground font-mono truncate" title={f.formatkaCode}>{f.formatkaCode}</span>
                                <span className="text-center">{f.length}x{f.width}</span>
                                <span className="text-center">{f.thickness}mm</span>
                                <span className="text-center">
                                  <Badge variant="outline" className="text-[9px] px-1">{f.colorCode || '-'}</Badge>
                                </span>
                                <span className="text-center font-medium">{f.quantity}</span>
                              </div>
                            ))}
                          </ScrollArea>
                        </div>
                      )}
                    </div>
                  );
                })()
              )}
            </div>
          </ScrollArea>
          
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => openOperatorModal(detailsWorkOrder!)} disabled={!detailsWorkOrder} data-testid="button-edit-operators-from-details">
              <Users className="h-4 w-4 mr-2" />
              Edytuj operatorów
            </Button>
            <Button variant="outline" onClick={() => setOperationDetailsModalOpen(false)} data-testid="button-close-details">
              Zamknij
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      {palletFlowData && ((palletFlowData.carriers?.length || 0) > 0 || Object.keys(palletFlowData.formatkiByPath || {}).length > 0) && (
        <CardContent className="border-t pt-4">
          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Package className="h-5 w-5 text-muted-foreground" />
                <span className="font-medium">Palety i przepływ</span>
              </div>
              <Button
                size="sm"
                variant="outline"
                onClick={() => setPalletModalOpen(true)}
                data-testid="button-add-pallet"
              >
                <Plus className="h-3 w-3 mr-1" />
                Nowa paleta
              </Button>
            </div>
            
            {Object.keys(palletFlowData.formatkiByPath).length > 0 && (
              <div className="flex flex-wrap gap-2">
                {Object.entries(palletFlowData.formatkiByPath).map(([pathCode, count]) => (
                  <div 
                    key={pathCode}
                    className="flex items-center gap-2 px-3 py-1.5 bg-muted rounded-lg cursor-pointer hover:bg-muted/80 transition-colors"
                    onClick={() => openFormatkiModal(pathCode)}
                    data-testid={`button-view-formatki-${pathCode}`}
                  >
                    <Badge className={flowCodeColors[pathCode]?.bg || 'bg-gray-500'}>
                      {pathCode}
                    </Badge>
                    <span className="text-sm">{count} formatek</span>
                    <Eye className="h-3 w-3 text-muted-foreground" />
                  </div>
                ))}
              </div>
            )}
            
            {(palletFlowData.carriers?.length || 0) > 0 && (
              <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-2">
                {palletFlowData.carriers?.map((carrier) => (
                  <div 
                    key={carrier.id}
                    className="flex flex-col gap-1 p-2 border rounded-lg bg-card"
                    data-testid={`carrier-card-${carrier.id}`}
                  >
                    <div className="flex items-center justify-between">
                      <span className="text-xs font-medium truncate">{carrier.code}</span>
                      <Badge variant="outline" className="text-[10px]">{carrier.status}</Badge>
                    </div>
                    <div className="flex items-center gap-1 text-[10px] text-muted-foreground">
                      <Box className="h-3 w-3" />
                      <span>{carrier.groupName || carrier.name}</span>
                    </div>
                    {carrier.locationName && (
                      <div className="text-[10px] text-muted-foreground truncate">
                        {carrier.locationName}
                      </div>
                    )}
                  </div>
                ))}
              </div>
            )}
          </div>
        </CardContent>
      )}
    </Card>
  );
}

export function WorkOrderTable({ workOrders, productionOrderId, orderStatus, damagedFormatki = [] }: WorkOrderFlowProps) {
  const { toast } = useToast();
  const [completeDialogOpen, setCompleteDialogOpen] = useState(false);
  const [selectedWorkOrder, setSelectedWorkOrder] = useState<WorkOrder | null>(null);

  const startWorkOrderMutation = useMutation({
    mutationFn: async (workOrderId: number) => {
      const res = await apiRequest("POST", `/api/production/work-orders/${workOrderId}/start`);
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Operacja rozpoczęta",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się rozpocząć operacji",
      });
    },
  });

  const completeWorkOrderMutation = useMutation({
    mutationFn: async (workOrderId: number) => {
      const res = await apiRequest("POST", `/api/production/work-orders/${workOrderId}/complete`, {});
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      setCompleteDialogOpen(false);
      setSelectedWorkOrder(null);
      toast({
        title: "Sukces",
        description: "Operacja zakończona",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zakończyć operacji",
      });
    },
  });

  const deleteWarehouseOpMutation = useMutation({
    mutationFn: async (workOrderId: number) => {
      const res = await apiRequest("DELETE", `/api/production/orders/${productionOrderId}/warehouse-operation/${workOrderId}`, {});
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Operacja magazynowania usunięta",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się usunąć operacji",
      });
    },
  });

  const handleCompleteClick = (wo: WorkOrder) => {
    setSelectedWorkOrder(wo);
    setCompleteDialogOpen(true);
  };

  const handleConfirmComplete = () => {
    if (!selectedWorkOrder) return;
    completeWorkOrderMutation.mutate(selectedWorkOrder.id);
  };

  const formatDateTime = (date: string | null) => {
    if (!date) return "-";
    return format(new Date(date), "dd.MM HH:mm", { locale: pl });
  };

  const canStartWorkOrder = (wo: WorkOrder, index: number) => {
    if (orderStatus !== 'in_progress' && orderStatus !== 'confirmed') return false;
    if (wo.status !== 'pending' && wo.status !== 'ready') return false;
    if (index === 0) return true;
    return workOrders[index - 1]?.status === 'done';
  };

  const canCompleteWorkOrder = (wo: WorkOrder) => {
    return wo.status === 'in_progress';
  };

  if (!workOrders || workOrders.length === 0) {
    return null;
  }

  return (
    <Card className="lg:col-span-2">
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Factory className="h-5 w-5" />
          Operacje - Szczegóły ({workOrders.length})
        </CardTitle>
      </CardHeader>
      <CardContent>
        <div className="overflow-x-auto">
          <table className="w-full text-sm">
            <thead>
              <tr className="border-b">
                <th className="text-left py-2 px-3 font-medium text-muted-foreground w-10">#</th>
                <th className="text-left py-2 px-3 font-medium text-muted-foreground">Operacja</th>
                <th className="text-left py-2 px-3 font-medium text-muted-foreground">Stanowisko</th>
                <th className="text-left py-2 px-3 font-medium text-muted-foreground">Status</th>
                <th className="text-center py-2 px-3 font-medium text-muted-foreground">Rzecz. Start</th>
                <th className="text-center py-2 px-3 font-medium text-muted-foreground">Rzecz. Koniec</th>
                <th className="text-right py-2 px-3 font-medium text-muted-foreground">Ilość</th>
                <th className="text-right py-2 px-3 font-medium text-muted-foreground">Uszkodz.</th>
                <th className="text-right py-2 px-3 font-medium text-muted-foreground w-24">Akcje</th>
              </tr>
            </thead>
            <tbody>
              {workOrders.map((wo, index) => {
                const config = statusConfig[wo.status] || statusConfig.pending;
                
                return (
                  <tr 
                    key={wo.id} 
                    className={`border-b hover:bg-muted/50 ${
                      wo.status === 'in_progress' ? 'bg-yellow-500/10' : ''
                    }`}
                    data-testid={`work-order-row-${wo.id}`}
                  >
                    <td className="py-2 px-3 font-mono text-muted-foreground">{wo.sequence}</td>
                    <td className="py-2 px-3">
                      <div className="font-medium">{wo.operationName || wo.workOrderNumber}</div>
                      {wo.operationCode && (
                        <div className="text-xs text-muted-foreground font-mono">{wo.operationCode}</div>
                      )}
                    </td>
                    <td className="py-2 px-3">
                      {wo.workCenterName ? (
                        <div>
                          <div className="font-medium">{wo.workCenterName}</div>
                          {wo.workCenterCode && (
                            <div className="text-xs text-muted-foreground font-mono">{wo.workCenterCode}</div>
                          )}
                        </div>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="py-2 px-3">
                      <Badge 
                        variant="outline" 
                        className={`${config.bgColor} ${config.color} border-current`}
                      >
                        {config.label}
                      </Badge>
                    </td>
                    <td className="py-2 px-3 text-center text-xs">
                      {wo.actualStartTime ? (
                        <span className="text-green-600 dark:text-green-400 font-medium">
                          {formatDateTime(wo.actualStartTime)}
                        </span>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="py-2 px-3 text-center text-xs">
                      {wo.actualEndTime ? (
                        <span className="text-green-600 dark:text-green-400 font-medium">
                          {formatDateTime(wo.actualEndTime)}
                        </span>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="py-2 px-3 text-right font-mono">
                      <span className={wo.quantityProduced && parseFloat(String(wo.quantityProduced)) > 0 ? 'text-green-600 dark:text-green-400' : ''}>
                        {wo.quantityProduced ?? 0}
                      </span>
                      <span className="text-muted-foreground">/{wo.quantityPlanned}</span>
                    </td>
                    <td className="py-2 px-3 text-right">
                      {wo.quantityScrap && parseFloat(String(wo.quantityScrap)) > 0 ? (
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <div className="flex items-center justify-end gap-1 cursor-help">
                              <XCircle className="h-3 w-3 text-red-500" />
                              <span className="text-red-500 font-medium">{wo.quantityScrap}</span>
                            </div>
                          </TooltipTrigger>
                          <TooltipContent>
                            <p className="font-medium">Uszkodzone elementy</p>
                            {wo.qualityCheckNotes && (
                              <p className="text-xs text-muted-foreground mt-1">{wo.qualityCheckNotes}</p>
                            )}
                          </TooltipContent>
                        </Tooltip>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="py-2 px-3 text-right">
                      <div className="flex justify-end gap-1">
                        {canStartWorkOrder(wo, index) && (
                          <Button
                            size="sm"
                            variant="default"
                            className="h-7 px-2"
                            onClick={() => startWorkOrderMutation.mutate(wo.id)}
                            disabled={startWorkOrderMutation.isPending}
                            data-testid={`button-start-wo-table-${wo.id}`}
                          >
                            {startWorkOrderMutation.isPending ? (
                              <Loader2 className="h-3 w-3 animate-spin" />
                            ) : (
                              <Play className="h-3 w-3" />
                            )}
                          </Button>
                        )}
                        {canCompleteWorkOrder(wo) && (
                          <Button
                            size="sm"
                            variant="default"
                            className="h-7 px-2 bg-green-600 hover:bg-green-700"
                            onClick={() => handleCompleteClick(wo)}
                            disabled={completeWorkOrderMutation.isPending}
                            data-testid={`button-complete-wo-table-${wo.id}`}
                          >
                            {completeWorkOrderMutation.isPending ? (
                              <Loader2 className="h-3 w-3 animate-spin" />
                            ) : (
                              <CheckCircle className="h-3 w-3" />
                            )}
                          </Button>
                        )}
                        {wo.status === 'done' && (
                          <CheckCircle className="h-4 w-4 text-green-500" />
                        )}
                      </div>
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>
      </CardContent>
      
      <Dialog open={completeDialogOpen} onOpenChange={setCompleteDialogOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Zakończ operację</DialogTitle>
            <DialogDescription>
              {selectedWorkOrder?.operationName || selectedWorkOrder?.workOrderNumber}
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4 text-sm">
              <div>
                <span className="text-muted-foreground">Ilość zaplanowana:</span>
                <span className="ml-2 font-medium">{selectedWorkOrder?.quantityPlanned} szt.</span>
              </div>
            </div>
            
            {damagedFormatki.length > 0 && (
              <div className="p-3 bg-yellow-100 dark:bg-yellow-900/30 rounded-md space-y-2">
                <div className="flex items-center gap-2">
                  <AlertTriangle className="h-4 w-4 text-yellow-600" />
                  <span className="text-sm font-medium text-yellow-700 dark:text-yellow-400">
                    {damagedFormatki.length} szt. oznaczonych jako uszkodzone:
                  </span>
                </div>
                <div className="max-h-32 overflow-y-auto space-y-1 pl-6">
                  {damagedFormatki.map((f) => (
                    <div key={f.id} className="text-xs text-yellow-700 dark:text-yellow-400 flex items-center gap-2">
                      <span className="font-medium">{f.generatedName}</span>
                      {f.damageType && (
                        <span className="text-yellow-600 dark:text-yellow-500">({f.damageType})</span>
                      )}
                    </div>
                  ))}
                </div>
              </div>
            )}
            
            <p className="text-sm text-muted-foreground">
              Czy na pewno chcesz zakończyć tę operację?
            </p>
          </div>
          
          <DialogFooter className="gap-2">
            <Button
              variant="outline"
              onClick={() => setCompleteDialogOpen(false)}
              data-testid="button-cancel-complete-table"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleConfirmComplete}
              disabled={completeWorkOrderMutation.isPending}
              data-testid="button-confirm-complete-table"
            >
              {completeWorkOrderMutation.isPending ? (
                <Loader2 className="h-4 w-4 animate-spin mr-2" />
              ) : (
                <CheckCircle className="h-4 w-4 mr-2" />
              )}
              Zakończ operację
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
    </Card>
  );
}
