import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { Save, Filter, Trash2, Check } from "lucide-react";
import type { FilterContext, UserSavedFilter } from "@shared/schema";

interface SavedFiltersManagerProps {
  context: FilterContext;
  currentFilters: Record<string, any>;
  onFiltersLoad: (filters: Record<string, any>) => void;
  onFiltersSave?: () => void;
}

export function SavedFiltersManager({
  context,
  currentFilters,
  onFiltersLoad,
  onFiltersSave,
}: SavedFiltersManagerProps) {
  const [saveDialogOpen, setSaveDialogOpen] = useState(false);
  const [filterName, setFilterName] = useState("");
  const { toast } = useToast();

  // Fetch saved filters for this context
  const { data: savedFilters = [], isLoading } = useQuery<UserSavedFilter[]>({
    queryKey: ["/api/saved-filters", context],
    queryFn: async () => {
      const response = await fetch(`/api/saved-filters?context=${context}`);
      if (!response.ok) throw new Error("Failed to fetch saved filters");
      return response.json();
    },
  });

  // Auto-load last used filter on mount
  useEffect(() => {
    if (savedFilters.length > 0) {
      const lastUsed = savedFilters.find((f) => f.isLastUsed);
      if (lastUsed) {
        onFiltersLoad(lastUsed.filterData as Record<string, any>);
      }
    }
  }, [savedFilters]);

  // Create new saved filter
  const createFilterMutation = useMutation({
    mutationFn: async (data: { name: string; filterData: Record<string, any> }) => {
      return apiRequest("POST", "/api/saved-filters", {
        context,
        name: data.name,
        filterData: data.filterData,
        isLastUsed: true, // Mark new filter as last used
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/saved-filters", context] });
      setSaveDialogOpen(false);
      setFilterName("");
      toast({
        title: "Filtr zapisany",
        description: "Twój filtr został pomyślnie zapisany.",
      });
      onFiltersSave?.();
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się zapisać filtru.",
        variant: "destructive",
      });
    },
  });

  // Delete saved filter
  const deleteFilterMutation = useMutation({
    mutationFn: async (filterId: number) => {
      return apiRequest("DELETE", `/api/saved-filters/${filterId}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/saved-filters", context] });
      toast({
        title: "Filtr usunięty",
        description: "Filtr został pomyślnie usunięty.",
      });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się usunąć filtru.",
        variant: "destructive",
      });
    },
  });

  // Set filter as last used (and load it)
  const setLastUsedMutation = useMutation({
    mutationFn: async (filterId: number) => {
      return apiRequest("PATCH", `/api/saved-filters/${filterId}/set-last-used`);
    },
    onSuccess: async () => {
      // Refetch to get updated last-used status
      await queryClient.invalidateQueries({ queryKey: ["/api/saved-filters", context] });
      await queryClient.refetchQueries({ queryKey: ["/api/saved-filters", context] });
    },
  });

  const handleSaveFilter = () => {
    if (!filterName.trim()) {
      toast({
        title: "Błąd",
        description: "Nazwa filtru jest wymagana.",
        variant: "destructive",
      });
      return;
    }

    createFilterMutation.mutate({
      name: filterName,
      filterData: currentFilters,
    });
  };

  const handleLoadFilter = (filter: UserSavedFilter) => {
    const filterData = filter.filterData as Record<string, any> | null;
    if (filterData) {
      onFiltersLoad(filterData);
    }
    setLastUsedMutation.mutate(filter.id);
  };

  const handleDeleteFilter = (filterId: number, e: React.MouseEvent) => {
    e.stopPropagation(); // Prevent triggering load when deleting
    deleteFilterMutation.mutate(filterId);
  };

  return (
    <div className="flex items-center gap-2">
      {/* Save current filters */}
      <Dialog open={saveDialogOpen} onOpenChange={setSaveDialogOpen}>
        <DialogTrigger asChild>
          <Button
            variant="outline"
            size="sm"
            data-testid="button-save-filter"
          >
            <Save className="h-4 w-4 mr-2" />
            Zapisz filtr
          </Button>
        </DialogTrigger>
        <DialogContent data-testid="dialog-save-filter">
          <DialogHeader>
            <DialogTitle>Zapisz aktualny filtr</DialogTitle>
            <DialogDescription>
              Nadaj nazwę aktualnemu filtrowi, aby móc go szybko załadować później.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="filter-name">Nazwa filtru</Label>
              <Input
                id="filter-name"
                data-testid="input-filter-name"
                placeholder="np. Meble drewniane"
                value={filterName}
                onChange={(e) => setFilterName(e.target.value)}
                onKeyDown={(e) => {
                  if (e.key === "Enter") {
                    handleSaveFilter();
                  }
                }}
              />
            </div>
            <Button
              onClick={handleSaveFilter}
              disabled={createFilterMutation.isPending}
              data-testid="button-confirm-save-filter"
              className="w-full"
            >
              {createFilterMutation.isPending ? "Zapisywanie..." : "Zapisz"}
            </Button>
          </div>
        </DialogContent>
      </Dialog>

      {/* Load saved filters */}
      <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button
            variant="outline"
            size="sm"
            disabled={isLoading || savedFilters.length === 0}
            data-testid="button-load-filter"
          >
            <Filter className="h-4 w-4 mr-2" />
            Załaduj filtr
            {savedFilters.length > 0 && (
              <span className="ml-2 text-xs text-muted-foreground">
                ({savedFilters.length})
              </span>
            )}
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent align="end" className="w-64">
          <DropdownMenuLabel>Zapisane filtry</DropdownMenuLabel>
          <DropdownMenuSeparator />
          {savedFilters.map((filter) => (
            <DropdownMenuItem
              key={filter.id}
              onClick={() => handleLoadFilter(filter)}
              className="flex items-center justify-between cursor-pointer"
              data-testid={`menu-item-filter-${filter.id}`}
            >
              <div className="flex items-center gap-2 flex-1">
                {filter.isLastUsed && (
                  <Check className="h-3 w-3 text-primary" />
                )}
                <span className="truncate">{filter.name}</span>
              </div>
              <Button
                variant="ghost"
                size="icon"
                className="h-6 w-6 hover:bg-destructive/10"
                onClick={(e) => handleDeleteFilter(filter.id, e)}
                data-testid={`button-delete-filter-${filter.id}`}
              >
                <Trash2 className="h-3 w-3 text-destructive" />
              </Button>
            </DropdownMenuItem>
          ))}
          {savedFilters.length === 0 && (
            <DropdownMenuItem disabled className="text-muted-foreground text-sm">
              Brak zapisanych filtrów
            </DropdownMenuItem>
          )}
        </DropdownMenuContent>
      </DropdownMenu>
    </div>
  );
}
