import { useEditor, EditorContent } from "@tiptap/react";
import StarterKit from "@tiptap/starter-kit";
import Image from "@tiptap/extension-image";
import Link from "@tiptap/extension-link";
import { Color } from "@tiptap/extension-color";
import { TextStyle } from "@tiptap/extension-text-style";
import Placeholder from "@tiptap/extension-placeholder";
import { Table } from "@tiptap/extension-table";
import { TableRow } from "@tiptap/extension-table-row";
import { TableHeader } from "@tiptap/extension-table-header";
import { TableCell } from "@tiptap/extension-table-cell";
import { mergeAttributes } from "@tiptap/core";
import {
  Bold,
  Italic,
  Strikethrough,
  Code,
  Heading1,
  Heading2,
  Heading3,
  List,
  ListOrdered,
  Quote,
  Undo,
  Redo,
  ImageIcon,
  Link2,
  Images,
  Package,
  Table as TableIcon,
  TableProperties,
  Columns,
  Rows,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { useState, useEffect } from "react";
import { ImageGalleryPicker } from "@/components/image-gallery-picker";
import { AddonPickerDialog } from "@/components/addon-picker-dialog";

// Custom Image extension with resizable support and style preservation
const ResizableImage = Image.extend({
  addAttributes() {
    const parentAttrs = (this as any).parent?.() || {};
    return {
      ...parentAttrs,
      width: {
        default: null,
        parseHTML: (element: HTMLElement) => element.getAttribute('width'),
      },
      height: {
        default: null,
        parseHTML: (element: HTMLElement) => element.getAttribute('height'),
      },
      style: {
        default: null,
        parseHTML: (element: HTMLElement) => element.getAttribute('style'),
        renderHTML: (attributes: any) => {
          if (!attributes.style) {
            return {};
          }
          return {
            style: attributes.style,
          };
        },
      },
    };
  },
  
  renderHTML({ HTMLAttributes }: { HTMLAttributes: Record<string, any> }) {
    return ['img', mergeAttributes(this.options.HTMLAttributes, HTMLAttributes)];
  },
});

interface RichTextEditorProps {
  content: string; // HTML string
  onContentChange: (html: string, json: any) => void;
  placeholder?: string;
  className?: string;
  enableGalleryPicker?: boolean; // Enable image gallery picker
  enableAddonPicker?: boolean; // Enable addon picker
  onEditorReady?: (editor: any) => void; // Callback when editor is ready
}

export function RichTextEditor({
  content,
  onContentChange,
  placeholder = "Zacznij pisać...",
  className = "",
  enableGalleryPicker = false,
  enableAddonPicker = false,
  onEditorReady,
}: RichTextEditorProps) {
  const [isImageDialogOpen, setIsImageDialogOpen] = useState(false);
  const [isGalleryPickerOpen, setIsGalleryPickerOpen] = useState(false);
  const [isAddonPickerOpen, setIsAddonPickerOpen] = useState(false);
  const [isLinkDialogOpen, setIsLinkDialogOpen] = useState(false);
  const [imageUrl, setImageUrl] = useState("");
  const [linkUrl, setLinkUrl] = useState("");
  const [isImageSelected, setIsImageSelected] = useState(false);

  const editor = useEditor({
    extensions: [
      StarterKit.configure({
        heading: {
          levels: [1, 2, 3],
        },
      }),
      ResizableImage.configure({
        inline: false, // Block-level to enable NodeSelection
        allowBase64: true,
        HTMLAttributes: {
          class: 'resizable-image',
        },
      }),
      Link.configure({
        openOnClick: false,
        autolink: false, // Disable autolink to prevent {{ai-safety}} from becoming links
        HTMLAttributes: {
          class: "text-primary underline",
        },
        validate: (href) => {
          // Block linking for text that looks like Handlebars tags
          // Pattern: {{anything}} or {imgN w=X}
          return !/^\{\{.*\}\}$/.test(href) && !/^\{img\d+\s+w=\d+\}$/.test(href);
        },
      }),
      Table.configure({
        resizable: true,
        HTMLAttributes: {
          class: "border-collapse table-auto w-full border border-border",
        },
      }),
      TableRow,
      TableHeader.configure({
        HTMLAttributes: {
          class: "border border-border bg-muted font-semibold p-2",
        },
      }),
      TableCell.configure({
        HTMLAttributes: {
          class: "border border-border p-2",
        },
      }),
      TextStyle,
      Color,
      Placeholder.configure({
        placeholder,
      }),
    ],
    content,
    editorProps: {
      attributes: {
        class:
          "prose prose-sm dark:prose-invert max-w-none focus:outline-none min-h-[200px] p-2 sm:p-4 overflow-x-auto",
      },
    },
    onUpdate: ({ editor }) => {
      const html = editor.getHTML();
      const json = editor.getJSON();
      onContentChange(html, json);
      
      // Also check selection on update
      checkImageSelection(editor);
    },
    onSelectionUpdate: ({ editor }) => {
      checkImageSelection(editor);
    },
    onTransaction: ({ editor }) => {
      checkImageSelection(editor);
    },
  });
  
  // Helper function to check if image is selected
  const checkImageSelection = (editor: any) => {
    try {
      const { selection } = editor.state;
      const { node } = selection as any;
      const isImage = node?.type?.name === "image";
      setIsImageSelected(isImage || false);
    } catch (error) {
      setIsImageSelected(false);
    }
  };

  // Sync editor content when prop changes (critical for template switching)
  useEffect(() => {
    if (editor && content !== editor.getHTML()) {
      editor.commands.setContent(content);
    }
  }, [content, editor]);

  // Notify parent when editor is ready
  useEffect(() => {
    if (editor && onEditorReady) {
      onEditorReady(editor);
    }
  }, [editor, onEditorReady]);

  if (!editor) {
    return null;
  }

  const addImage = () => {
    if (imageUrl) {
      editor.chain().focus().setImage({ src: imageUrl }).run();
      setImageUrl("");
      setIsImageDialogOpen(false);
    }
  };

  const addImageFromGallery = (url: string) => {
    editor.chain().focus().setImage({ src: url }).run();
  };

  const addAddonHtml = (addon: { htmlTemplate?: string | null; code?: string | null; name?: string; isGroup?: boolean }) => {
    if ('isGroup' in addon && addon.isGroup && addon.code) {
      // For accessory groups, insert the tag
      const tag = `{{akcesorium-${addon.code}}}`;
      editor.chain().focus().insertContent(tag).run();
    } else if (addon.htmlTemplate) {
      // For regular addons, insert HTML template
      editor.chain().focus().insertContent(addon.htmlTemplate).run();
    }
  };

  const addLink = () => {
    if (linkUrl) {
      editor
        .chain()
        .focus()
        .extendMarkRange("link")
        .setLink({ href: linkUrl })
        .run();
      setLinkUrl("");
      setIsLinkDialogOpen(false);
    }
  };

  return (
    <div className={`border rounded-md overflow-hidden ${className}`}>
      {/* Toolbar */}
      <div className="flex flex-wrap gap-0.5 sm:gap-1 p-1 sm:p-2 border-b bg-muted/30 overflow-x-auto">
        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().toggleBold().run()}
          data-testid="button-bold"
          className={editor.isActive("bold") ? "bg-accent" : ""}
        >
          <Bold className="h-4 w-4" />
        </Button>
        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().toggleItalic().run()}
          data-testid="button-italic"
          className={editor.isActive("italic") ? "bg-accent" : ""}
        >
          <Italic className="h-4 w-4" />
        </Button>
        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().toggleStrike().run()}
          data-testid="button-strike"
          className={editor.isActive("strike") ? "bg-accent" : ""}
        >
          <Strikethrough className="h-4 w-4" />
        </Button>
        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().toggleCode().run()}
          data-testid="button-code"
          className={editor.isActive("code") ? "bg-accent" : ""}
        >
          <Code className="h-4 w-4" />
        </Button>

        <div className="w-px h-6 bg-border mx-1" />

        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().toggleHeading({ level: 1 }).run()}
          data-testid="button-h1"
          className={editor.isActive("heading", { level: 1 }) ? "bg-accent" : ""}
        >
          <Heading1 className="h-4 w-4" />
        </Button>
        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().toggleHeading({ level: 2 }).run()}
          data-testid="button-h2"
          className={editor.isActive("heading", { level: 2 }) ? "bg-accent" : ""}
        >
          <Heading2 className="h-4 w-4" />
        </Button>
        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().toggleHeading({ level: 3 }).run()}
          data-testid="button-h3"
          className={editor.isActive("heading", { level: 3 }) ? "bg-accent" : ""}
        >
          <Heading3 className="h-4 w-4" />
        </Button>

        <div className="w-px h-6 bg-border mx-1" />

        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().toggleBulletList().run()}
          data-testid="button-bullet-list"
          className={editor.isActive("bulletList") ? "bg-accent" : ""}
        >
          <List className="h-4 w-4" />
        </Button>
        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().toggleOrderedList().run()}
          data-testid="button-ordered-list"
          className={editor.isActive("orderedList") ? "bg-accent" : ""}
        >
          <ListOrdered className="h-4 w-4" />
        </Button>
        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().toggleBlockquote().run()}
          data-testid="button-blockquote"
          className={editor.isActive("blockquote") ? "bg-accent" : ""}
        >
          <Quote className="h-4 w-4" />
        </Button>

        <div className="w-px h-6 bg-border mx-1" />

        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().insertTable({ rows: 3, cols: 3, withHeaderRow: true }).run()}
          data-testid="button-insert-table"
        >
          <TableIcon className="h-4 w-4" />
        </Button>
        {editor.isActive("table") && (
          <>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().addRowBefore().run()}
              data-testid="button-add-row-before"
            >
              <Rows className="h-3 w-3" />
              +↑
            </Button>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().addRowAfter().run()}
              data-testid="button-add-row-after"
            >
              <Rows className="h-3 w-3" />
              +↓
            </Button>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().addColumnBefore().run()}
              data-testid="button-add-col-before"
            >
              <Columns className="h-3 w-3" />
              +←
            </Button>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().addColumnAfter().run()}
              data-testid="button-add-col-after"
            >
              <Columns className="h-3 w-3" />
              +→
            </Button>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().deleteRow().run()}
              data-testid="button-delete-row"
            >
              <Rows className="h-3 w-3" />
              -
            </Button>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().deleteColumn().run()}
              data-testid="button-delete-col"
            >
              <Columns className="h-3 w-3" />
              -
            </Button>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().deleteTable().run()}
              data-testid="button-delete-table"
            >
              <TableIcon className="h-3 w-3" />
              ✕
            </Button>
          </>
        )}

        <div className="w-px h-6 bg-border mx-1" />

        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => setIsImageDialogOpen(true)}
          data-testid="button-add-image"
        >
          <ImageIcon className="h-4 w-4" />
        </Button>
        {enableGalleryPicker && (
          <Button
            type="button"
            size="sm"
            variant="ghost"
            onClick={() => setIsGalleryPickerOpen(true)}
            data-testid="button-gallery-picker"
          >
            <Images className="h-4 w-4" />
          </Button>
        )}
        {enableAddonPicker && (
          <Button
            type="button"
            size="sm"
            variant="ghost"
            onClick={() => setIsAddonPickerOpen(true)}
            data-testid="button-addon-picker"
          >
            <Package className="h-4 w-4" />
          </Button>
        )}
        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => setIsLinkDialogOpen(true)}
          data-testid="button-add-link"
        >
          <Link2 className="h-4 w-4" />
        </Button>

        {editor && ((editor.state.selection as any).node?.type?.name === "image") && (
          <>
            <div className="w-px h-6 bg-border mx-1" />
            <span className="text-xs text-muted-foreground px-2 flex items-center">Rozmiar:</span>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().updateAttributes("image", { width: "25%" }).run()}
              data-testid="button-image-25"
            >
              25%
            </Button>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().updateAttributes("image", { width: "50%" }).run()}
              data-testid="button-image-50"
            >
              50%
            </Button>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().updateAttributes("image", { width: "75%" }).run()}
              data-testid="button-image-75"
            >
              75%
            </Button>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().updateAttributes("image", { width: "100%" }).run()}
              data-testid="button-image-100"
            >
              100%
            </Button>
            <Button
              type="button"
              size="sm"
              variant="ghost"
              onClick={() => editor.chain().focus().updateAttributes("image", { width: null }).run()}
              data-testid="button-image-auto"
            >
              Auto
            </Button>
          </>
        )}

        <div className="w-px h-6 bg-border mx-1" />

        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().undo().run()}
          disabled={!editor.can().undo()}
          data-testid="button-undo"
        >
          <Undo className="h-4 w-4" />
        </Button>
        <Button
          type="button"
          size="sm"
          variant="ghost"
          onClick={() => editor.chain().focus().redo().run()}
          disabled={!editor.can().redo()}
          data-testid="button-redo"
        >
          <Redo className="h-4 w-4" />
        </Button>
      </div>

      {/* Editor Content */}
      <EditorContent editor={editor} />

      {/* Image Dialog */}
      <Dialog open={isImageDialogOpen} onOpenChange={setIsImageDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Dodaj obraz</DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <Input
              type="text"
              placeholder="URL obrazu"
              value={imageUrl}
              onChange={(e) => setImageUrl(e.target.value)}
              data-testid="input-image-url"
            />
          </div>
          <DialogFooter>
            <Button
              variant="ghost"
              onClick={() => setIsImageDialogOpen(false)}
              data-testid="button-cancel-image"
            >
              Anuluj
            </Button>
            <Button onClick={addImage} data-testid="button-insert-image">
              Wstaw
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Link Dialog */}
      <Dialog open={isLinkDialogOpen} onOpenChange={setIsLinkDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Dodaj link</DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <Input
              type="text"
              placeholder="URL linku"
              value={linkUrl}
              onChange={(e) => setLinkUrl(e.target.value)}
              data-testid="input-link-url"
            />
          </div>
          <DialogFooter>
            <Button
              variant="ghost"
              onClick={() => setIsLinkDialogOpen(false)}
              data-testid="button-cancel-link"
            >
              Anuluj
            </Button>
            <Button onClick={addLink} data-testid="button-insert-link">
              Wstaw
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Image Gallery Picker */}
      {enableGalleryPicker && (
        <ImageGalleryPicker
          open={isGalleryPickerOpen}
          onOpenChange={setIsGalleryPickerOpen}
          onSelect={addImageFromGallery}
        />
      )}

      {/* Addon Picker */}
      {enableAddonPicker && (
        <AddonPickerDialog
          open={isAddonPickerOpen}
          onOpenChange={setIsAddonPickerOpen}
          onSelect={addAddonHtml}
        />
      )}
    </div>
  );
}
